package App::orgsel;

our $DATE = '2016-04-01'; # DATE
our $VERSION = '0.001'; # VERSION

use 5.010001;
use strict;
use warnings;

use Scalar::Util qw(refaddr);

our %SPEC;

$SPEC{orgsel} = {
    v => 1.1,
    summary => 'Select Org document elements using CSel (CSS-like) syntax',
    args => {
        expr => {
            schema => 'str*',
            req => 1,
            pos => 0,
        },
        file => {
            schema => 'str*',
            'x.schema.entity' => 'filename',
            pos => 1,
            default => '-',
        },
        match_action => {
            schema => 'str*',
            default => 'print-as-string',
            cmdline_aliases => {
                count => { is_flag => 1, code => sub { $_[0]{match_action} = 'count' } },
                # dump
            },
        },
    },
};
sub orgsel {
    my %args = @_;

    # parse first so we can bail early on error without having to read the input
    require Data::CSel;
    my $expr = $args{expr};
    Data::CSel::parse_csel($expr)
          or return [400, "Invalid CSel expression '$expr'"];

    require Org::Parser;
    my $parser = Org::Parser->new;

    my $doc;
    if ($args{file} eq '-') {
        binmode STDIN, ":utf8";
        $doc = $parser->parse(join "", <>);
    } else {
        local $ENV{PERL_ORG_PARSER_CACHE} = $ENV{PERL_ORG_PARSER_CACHE} // 1;
        $doc = $parser->parse_file($args{file});
    }

    my @matches = Data::CSel::csel(
        {class_prefixes=>["Org::Element"]}, $expr, $doc);

    # skip root node itself
    @matches = grep { refaddr($_) ne refaddr($doc) } @matches;

    if ($args{match_action} eq 'count') {
        [200, "OK", ~~@matches];
    } else {
        [200, "OK", [map {$_->as_string} @matches]];
    }
}

1;
# ABSTRACT: Select Org document elements using CSel (CSS-like) syntax

__END__

=pod

=encoding UTF-8

=head1 NAME

App::orgsel - Select Org document elements using CSel (CSS-like) syntax

=head1 VERSION

This document describes version 0.001 of App::orgsel (from Perl distribution App-orgsel), released on 2016-04-01.

=head1 SYNOPSIS

=head1 FUNCTIONS


=head2 orgsel(%args) -> [status, msg, result, meta]

Select Org document elements using CSel (CSS-like) syntax.

This function is not exported.

Arguments ('*' denotes required arguments):

=over 4

=item * B<expr>* => I<str>

=item * B<file> => I<str> (default: "-")

=item * B<match_action> => I<str> (default: "print-as-string")

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/App-orgsel>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-App-orgsel>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=App-orgsel>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
