
package List::Tuples ;

use strict;
use warnings ;

BEGIN
{
use vars qw ($VERSION @ISA @EXPORT_OK %EXPORT_TAGS);

$VERSION     = 0.01;
@EXPORT_OK   = qw (all);
%EXPORT_TAGS = (all => [qw(ref_mesh tuples hash_tuples)]);
}

#-------------------------------------------------------------------------------

use Carp qw(carp croak confess) ;

#-------------------------------------------------------------------------------

=head1 NAME

 List::Tuples - Makes tuples from lists

=head1 SYNOPSIS

	tuples[2] => (1 .. 6) ;
		
	gives :
		tuples
		|- 0 
		|  |- 0 = 1 
		|  `- 1 = 2 
		|- 1 
		|  |- 0 = 3 
		|  `- 1 = 4 
		`- 2 
		   |- 0 = 5 
		   `- 1 = 6 
		   
	ref_mesh([1 .. 3], ['a' .. 'b'], ['*']) ;
		
	gives :
		ref_mesh
		|- 0 = 1 
		|- 1 = a 
		|- 2 = * 
		|- 3 = 2 
		|- 4 = b 
		|- 5 = undef 
		|- 6 = 3 
		|- 7 = undef 
		`- 8 = undef 
		
	hash_tuples ['key', 'other_key'] => (1 .. 5) ;
		
	gives :
		hash_tuples
		|- 0 
		|  |- key = 1 
		|  `- other_key = 2 
		|- 1 
		|  |- key = 3 
		|  `- other_key = 4 
		`- 2 
		   |- key = 5 
		   `- other_key = undef
		   
	# note: output generated by Data::TreeDumper

=head1 DESCRIPTION

This module defines subroutines that let you create tuples.

=head1 DOCUMENTATION

Ever got frustrated that you couldn't easily get tuples into a map{} or
create multiple hashes from an ordered list?

Jonathan Scott in In "Everyday Perl 6" L<http://www.perl.com/pub/a/2007/05/10/everyday-perl-6.html> writes:

    # Perl 6                        # Perl 5
    for @array -> $a     { ... }    for my $a (@array) { ... }
    for @array -> $a, $b { ... }    # too complex :)

The following subroutines will simplify your job. They could certainly be more effective implemented
directly in the language, IE in Perl6. If you have millions of tuples to handle, you may want monitor memory usage.

=head1 SUBROUTINES

=cut

#----------------------------------------------------------------------

sub tuples
{

=head2 tuples([$limit], \@size, @list)

B<tuples> will extract B<$size> elements from B<@lists> and group them in an array reference.
It will extract as many tuples as possible up to the, optional, B<$limit> you pass as argument.

	tuples 3 => [2] => (1 .. 14); # 3 tuples with 2 elements are returned
	tuples[2] => (1 .. 14); # as many tuples with 2 elements are returned
	
	for my $tuple (tuples[2], @array) 
		{
		print "[$tuple->[0], $tuple->[1]]\n" ;
		}
	

=head3 Arguments

=over 2

=item $limit - an optional maximum number of tuples to create

=item \@size - an array reference containing the number of elements in a tuple

=item @list - a list to be split into tuples

=back

=head3 Return

=over 2

=item * A list of tuples (array references)

=back

=head3 Input list with insufficient elements

	tuples[2] => (1 .. 3)) ; 
	
	gives :
	
	tuples
	|- 0 
	|  |- 0 = 1 
	|  `- 1 = 2 
	`- 1
	   `- 0 = 3

=cut

my ($limit, $size, @array) = @_ ;

if ('ARRAY' eq ref $limit)
	{
	# handle optional limit
	unshift @array, $size if defined $size ;
	$size = $limit ;
	$limit = undef ;
	}

my $number_of_tuples = 0 ;

if('ARRAY' eq ref $size)
	{
	$size = $size->[0] ;
	
	if(defined $size)
		{
		if($size > 0)
			{
			$number_of_tuples = @array / $size ;
			$number_of_tuples++ if @array % $size ;
			}
		else
			{
			confess 'Error: List::Tuples::tuples expects a positive tuple size!' ;
			}
		}
	else
		{
		confess 'Error: List::Tuples::tuples expects a positive tuple size!' ;
		}
		
	if(defined $limit)
		{
		if($limit > 0)
			{
			$number_of_tuples = $number_of_tuples > $limit ? $limit : $number_of_tuples ;
			}
		else
			{
			confess 'Error: List::Tuples::tuples expects a positive tuple limit!' ;
			}
		}
	}
else
	{
	confess 'Error: List::Tuples::tuples expects an array reference as size argument!' ;
	}
	
if(@array)
	{
	return
		(
		map{[splice(@array, 0, $size)] } (1 .. $number_of_tuples)
		) ;
	}
else
	{
	return ;
	}
}


#-------------------------------------------------------------------------------------------------------------

sub ref_mesh
{

=head2 ref_mesh(\@array1, \@array2, ...)

Mixes elements from arrays, one element at the time.

	ref_mesh
		['mum1', 'mum2', 'mum3'],
		['dad1', 'dad2'],
		[['child1_1', 'child1_2'], [], ['child3_1']] ;
	
	gives the folowing list:
	
	|- 0 = mum1 
	|- 1 = dad1 
	|- 2 
	|  |- 0 = child1_1 
	|  `- 1 = child1_2 
	|- 3 = mum2 
	|- 4 = dad2 
	|- 5 (no elements) 
	|- 6 = mum3 
	|- 7 = undef 
	`- 8 
	   `- 0 = child3_1 

This is equivalent to B<mesh> from L<List::MoreUtils> except the fact it takes arrays references instead for lists.
The implementation is directly taken from L<List::MoreUtils>.

=head3 Arguments

=over 2

=item * a list of array reference

=back

=head3 Return

=over 2

=item * a list consisting of the first elements of each array reference, then the second, then the third, etc, until all arrays are exhausted

=back

=cut

my $max = -1;
my $index = 0 ;

for my $array_ref (@_)
	{
	confess "element $index is not an array reference!\n" unless 'ARRAY' eq ref $array_ref  ;
	$max < $#{$array_ref}   &&  ($max = $#{$array_ref} )  ;
	
	$index++ ;
	}

return(map { my $ix = $_; map {$_->[$ix]} @_; } 0..$max) ;
}

#-------------------------------------------------------------------------------------------------------------

sub hash_tuples
{

=head2 hash_tuples([$limit], \@hash_keys, @input_array)

B<hash_tuples> uses elements from \@input_array and combine them with \@hash_keys to create hash references.
It will create as many hashes as possible up to the, optional,  $limit.

	hash_tuples
		['Mum',   'Dad',   'Children'] =>
		'Lena',   'Nadim', ['Yasmin', 'Miriam'],
		'Monika', 'ola',   ['astrid'] ;
	
	gives:
	
	|- 0 
	|  |- Children 
	|  |  |- 0 = Yasmin 
	|  |  `- 1 = Miriam 
	|  |- Dad = Nadim 
	|  `- Mum = Lena 
	`- 1 
	   |- Children 
	   |  `- 0 = astrid 
	   |- Dad = ola 
	   `- Mum = Monika 
	   
	for my $tuple (hash_tuples(['a', 'b'] => @array)) 
		{
		print $tuple->{a} . "\n" ;
		print $tuple->{b} . "\n" ;
		}

=head3 Arguments

=over 2

=item $limit - an optional maximum number of hashes to create

=item \@hash_keys - an array reference containing the list of keys apply to the input array

=item \@input_array- an array reference. the array contains the elements to extract 

=back

=head3 Return

=over 2

=item * A list of hashes

=back

=cut

my ($limit, $hash_keys, @input_array) = @_ ;

if ('ARRAY' eq ref $limit)
	{
	unshift @input_array, $hash_keys  if defined $hash_keys ;
	$hash_keys = $limit ;
	$limit = undef ;
	}
	
if('ARRAY' eq ref $hash_keys)
	{
	unless(@{$hash_keys})
		{
		confess 'Error: List::Tuples::hash_tuples expects a list of keys!' ;
		}
		
	if(defined $limit)
		{
		unless($limit > 0)
			{
			confess 'Error: List::Tuples::hash_tuples expects a positive tuple limit!' ;
			}
		}
	}
else
	{
	confess 'Error: List::Tuples::hash_tuples expects an array reference as size argument!' ;
	}
	
if(@input_array)
	{
	return
		(
		map	
			{
				{
				ref_mesh($hash_keys => $_)
				}
			}
			tuples $limit => [scalar(@{$hash_keys})] => @input_array
		) ;
	}
else
	{
	return ;
	}
}


#-------------------------------------------------------------------------------------------------------------

1 ;

=head1 BUGS AND LIMITATIONS

None so far.

=head1 AUTHOR

	Khemir Nadim ibn Hamouda
	CPAN ID: NKH
	mailto:nadim@khemir.net

=head1 LICENSE AND COPYRIGHT

This program is free software; you can redistribute
it and/or modify it under the same terms as Perl itself.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc List::Tuples

You can also look for information at:

=over 4

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/List-Tuples>

=item * RT: CPAN's request tracker

Please report any bugs or feature requests to  L <bug-list-tuples@rt.cpan.org>.

We will be notified, and then you'll automatically be notified of progress on
your bug as we make changes.

=item * Search CPAN

L<http://search.cpan.org/dist/List-Tuples>

=back

=head1 SEE ALSO


=cut
