#!/usr/local/bin/perl -w
use 5.008;
use strict;
use warnings;

use Test::More tests => 7;

# Before `make install' is performed this script should be runnable with
# `make test'. After `make install' it should work as `perl 1.t'
# Note:
# This test requires internet connection.
# And it will take several minutes to progress.

BEGIN {
	use_ok('Finance::YahooJPN::Quote')
};

# 9437.t: NTT DoCoMo at Tokyo Stock Exchange.

#######################################################
my $obj1 = Finance::YahooJPN::Quote->new('9437.t');
isa_ok($obj1, 'Finance::YahooJPN::Quote');

my(@noadjust, @adjust);
my $file = 'noadjust';
while (<DATA>) {
	my $row = $_;
	$row =~ s/\x0D\x0A|\x0D|\x0A/\n/;
	if ($row =~ /^cut$/) {
		$file = 'adjust';
		next;
	}
	
	if    ($file eq 'noadjust') {
		push @noadjust, $row;
	}
	elsif ($file eq 'adjust'  ) {
		push @adjust, $row;
	}
}
my $noadjust = join '', @noadjust;
my $adjust   = join '', @adjust;

$obj1->scan('start' => '1999-06-01');
my @tmp = $obj1->output('noadjust' => 1);
my @quote;
for (my $i = 0; $i <= $#tmp; $i++) {
	last if $tmp[$i] =~ /2003-09-29/;
	push @quote, $tmp[$i], "\n";
}
my $quote = join '', @quote;

is( $quote, $noadjust,
	'run scan() method w/o adjustment for the splits' );

#######################################################
my $obj2 = Finance::YahooJPN::Quote->new('9437.t');
$obj2->scan('start' => '1999-06-01');
@tmp = $obj2->output();
@quote = ();
for (my $i = 0; $i <= $#tmp; $i++) {
	last if $tmp[$i] =~ /2003-09-29/;
	push @quote, $tmp[$i], "\n";
}
$quote = join '', @quote;

is( $quote, $adjust,
	'run scan() method w/  adjustment for the splits' );

#######################################################
@tmp = Finance::YahooJPN::Quote->historical('9437.t', 'Start' => '1999-06-01');
@quote = ();
for (my $i = 0; $i <= $#tmp; $i++) {
	last if $tmp[$i] =~ /2003-09-29/;
	push @quote, $tmp[$i], "\n";
}
$quote = join '', @quote;
is( $quote, $adjust,
	'run historical() method (includes new(), scan() and output())' );

#######################################################
my $obj3 = Finance::YahooJPN::Quote->new('9437.t');
$obj3->scan('start' => '2002-03-26', 'last' => '2002-03-26');
@tmp = $obj3->output();
$quote = $tmp[0];
my $expected = '2002-03-26	341000	351000	329000	333000	208021';

is( $quote, $expected,
	'in a rare case split data is at the top row' );

#######################################################
my $obj4 = Finance::YahooJPN::Quote->new('0000.t');
$obj4->scan('start' => '2003-09-01', 'last' => '2003-09-30');
@tmp = $obj4->output();
$quote = join "\n", @tmp;
$expected = '';

is( $quote, $expected,
	'in a case the specified symbol is absent' );

__END__
1999-06-01	6850000	7030000	6820000	7030000	4074
1999-06-02	7030000	7030000	6860000	6880000	2041
1999-06-03	6900000	6900000	6740000	6820000	1531
1999-06-04	6860000	6920000	6780000	6830000	1486
1999-06-07	6930000	6980000	6870000	6900000	1549
1999-06-08	6940000	6940000	6780000	6820000	1826
1999-06-09	6820000	6890000	6780000	6830000	2311
1999-06-10	6860000	6960000	6860000	6940000	2420
1999-06-11	7000000	7150000	6970000	7110000	4840
1999-06-14	7130000	7210000	7070000	7070000	2265
1999-06-15	7170000	7200000	7090000	7200000	2432
1999-06-16	7240000	7310000	7220000	7250000	2624
1999-06-17	7350000	7350000	7280000	7340000	2464
1999-06-18	7340000	7340000	7200000	7220000	3716
1999-06-21	7220000	7400000	7220000	7340000	3677
1999-06-22	7390000	7400000	7200000	7250000	4680
1999-06-23	7180000	7250000	7130000	7160000	6497
1999-06-24	7200000	7440000	7190000	7400000	11595
1999-06-25	1680000	1680000	1640000	1680000	67063
1999-06-28	1710000	1720000	1650000	1670000	21182
1999-06-29	1680000	1690000	1630000	1640000	11974
1999-06-30	1650000	1690000	1640000	1640000	10846
1999-07-01	1700000	1770000	1680000	1760000	21033
1999-07-02	1790000	1810000	1760000	1810000	16657
1999-07-05	1820000	1870000	1810000	1850000	15305
1999-07-06	1840000	1840000	1770000	1820000	14070
1999-07-07	1800000	1810000	1760000	1780000	5742
1999-07-08	1770000	1780000	1750000	1760000	7017
1999-07-09	1740000	1750000	1720000	1740000	6236
1999-07-12	1740000	1790000	1730000	1770000	6274
1999-07-13	1760000	1790000	1740000	1740000	6386
1999-07-14	1770000	1770000	1740000	1760000	5987
1999-07-15	1770000	1830000	1750000	1830000	13650
1999-07-16	1860000	1880000	1810000	1820000	14740
1999-07-19	1800000	1850000	1770000	1850000	6059
1999-07-21	1820000	1850000	1800000	1840000	10939
1999-07-22	1820000	1820000	1760000	1770000	6742
1999-07-23	1740000	1820000	1730000	1750000	9962
1999-07-26	1780000	1790000	1750000	1750000	7844
1999-07-27	1740000	1770000	1730000	1750000	6816
1999-07-28	1780000	1790000	1760000	1780000	5704
1999-07-29	1780000	1790000	1760000	1780000	4089
1999-07-30	1790000	1800000	1770000	1780000	5963
1999-08-02	1780000	1780000	1750000	1770000	3102
1999-08-03	1760000	1770000	1740000	1770000	3390
1999-08-04	1760000	1760000	1720000	1720000	4973
1999-08-05	1710000	1730000	1670000	1670000	6340
1999-08-06	1670000	1690000	1650000	1650000	6802
1999-08-09	1670000	1700000	1670000	1680000	5185
1999-08-10	1680000	1700000	1670000	1700000	3487
1999-08-11	1700000	1720000	1680000	1710000	4003
1999-08-12	1730000	1730000	1710000	1720000	4180
1999-08-13	1740000	1750000	1720000	1730000	3741
1999-08-16	1750000	1780000	1740000	1760000	4501
1999-08-17	1800000	1830000	1780000	1830000	10631
1999-08-18	1830000	1860000	1810000	1840000	12837
1999-08-19	1840000	1840000	1800000	1810000	6416
1999-08-20	1780000	1830000	1780000	1810000	3580
1999-08-23	1820000	1820000	1750000	1760000	3868
1999-08-24	1790000	1800000	1760000	1760000	3365
1999-08-25	1780000	1780000	1730000	1760000	4118
1999-08-26	1770000	1790000	1750000	1760000	3307
1999-08-27	1770000	1790000	1750000	1780000	4441
1999-08-30	1800000	1830000	1790000	1820000	5702
1999-08-31	1840000	1850000	1820000	1820000	9899
1999-09-01	1830000	1830000	1790000	1810000	4924
1999-09-02	1820000	1830000	1800000	1820000	6875
1999-09-03	1830000	1860000	1830000	1850000	14897
1999-09-06	1880000	1960000	1880000	1940000	21607
1999-09-07	2010000	2020000	1970000	2000000	16221
1999-09-08	1990000	2000000	1960000	1990000	11647
1999-09-09	1980000	1990000	1930000	1930000	8485
1999-09-10	1960000	1990000	1950000	1990000	14461
1999-09-13	2030000	2050000	2000000	2030000	15660
1999-09-14	2040000	2050000	1950000	2000000	8806
1999-09-16	1870000	1910000	1870000	1910000	17867
1999-09-17	1890000	1890000	1850000	1860000	9707
1999-09-20	1950000	2040000	1940000	2010000	31319
1999-09-21	2040000	2050000	2020000	2050000	21629
1999-09-22	2010000	2100000	1990000	2050000	30980
1999-09-24	1970000	1990000	1880000	1940000	23074
1999-09-27	1960000	1970000	1890000	1930000	10774
1999-09-28	1960000	2000000	1950000	1990000	10365
1999-09-29	1990000	2000000	1960000	1990000	10851
1999-09-30	2020000	2100000	2010000	2100000	20957
1999-10-01	2100000	2230000	2090000	2170000	42238
1999-10-04	2180000	2190000	2160000	2180000	7533
1999-10-05	2190000	2200000	2130000	2150000	9936
1999-10-06	2130000	2150000	2100000	2120000	8500
1999-10-07	2180000	2230000	2160000	2160000	16574
1999-10-08	2190000	2200000	2160000	2180000	6947
1999-10-12	2240000	2300000	2240000	2300000	16775
1999-10-13	2280000	2310000	2260000	2270000	13390
1999-10-14	2300000	2340000	2280000	2320000	11994
1999-10-15	2300000	2320000	2250000	2250000	10919
1999-10-18	2090000	2140000	2080000	2120000	13582
1999-10-19	2140000	2150000	2070000	2110000	11875
1999-10-20	2190000	2270000	2180000	2270000	17695
1999-10-21	2310000	2320000	2220000	2230000	11587
1999-10-22	2290000	2360000	2280000	2340000	19663
1999-10-25	2420000	2510000	2410000	2470000	25046
1999-10-26	2460000	2680000	2440000	2600000	33341
1999-10-27	2590000	2650000	2510000	2550000	26753
1999-10-28	2620000	2700000	2590000	2680000	20988
1999-10-29	2820000	2860000	2750000	2770000	24760
1999-11-01	2810000	2910000	2790000	2860000	23432
1999-11-02	2880000	2910000	2830000	2870000	20673
1999-11-04	2910000	2980000	2900000	2940000	17603
1999-11-05	2930000	2970000	2870000	2940000	15370
1999-11-08	2900000	2940000	2810000	2810000	13113
1999-11-09	2770000	2920000	2710000	2730000	25528
1999-11-10	2730000	2830000	2730000	2830000	12431
1999-11-11	2820000	3000000	2780000	2890000	26888
1999-11-12	2870000	2940000	2850000	2940000	13459
1999-11-15	3020000	3100000	3010000	3080000	25625
1999-11-16	3030000	3040000	2920000	2920000	14796
1999-11-17	3050000	3070000	2800000	2850000	19677
1999-11-18	2810000	3000000	2790000	2950000	16183
1999-11-19	3030000	3050000	2960000	3020000	13305
1999-11-22	3090000	3280000	3050000	3260000	29824
1999-11-24	3230000	3760000	3200000	3510000	36064
1999-11-25	3760000	3930000	3450000	3560000	34664
1999-11-26	3630000	3700000	3470000	3690000	19177
1999-11-29	3710000	3890000	3680000	3890000	19987
1999-11-30	3740000	3800000	3580000	3580000	32426
1999-12-01	3480000	3530000	3220000	3220000	38589
1999-12-02	3260000	3270000	2990000	3190000	54400
1999-12-03	3300000	3380000	3270000	3290000	24330
1999-12-06	3490000	3540000	3430000	3480000	21192
1999-12-07	3530000	3580000	3410000	3450000	17240
1999-12-08	3400000	3490000	3340000	3490000	13636
1999-12-09	3480000	3480000	3320000	3410000	13219
1999-12-10	3420000	3470000	3390000	3450000	14157
1999-12-13	3420000	3480000	3390000	3450000	8932
1999-12-14	3420000	3450000	3360000	3370000	12447
1999-12-15	3350000	3500000	3340000	3450000	11791
1999-12-16	3500000	3580000	3490000	3550000	15011
1999-12-17	3600000	3610000	3460000	3520000	9222
1999-12-20	3510000	3510000	3360000	3410000	6615
1999-12-21	3360000	3480000	3350000	3430000	7206
1999-12-22	3490000	3530000	3460000	3500000	6958
1999-12-24	3600000	3700000	3530000	3700000	14065
1999-12-27	3710000	3720000	3610000	3650000	5667
1999-12-28	3630000	3960000	3620000	3890000	19194
1999-12-29	3990000	4070000	3910000	3950000	12723
1999-12-30	3940000	3990000	3920000	3930000	5328
2000-01-04	4000000	4100000	3930000	3950000	9461
2000-01-05	3550000	3660000	3450000	3570000	26522
2000-01-06	3610000	3630000	3380000	3400000	14490
2000-01-07	3100000	3270000	3050000	3160000	28815
2000-01-11	3610000	3640000	3530000	3640000	22527
2000-01-12	3500000	3650000	3490000	3500000	12168
2000-01-13	3450000	3500000	3380000	3430000	10193
2000-01-14	3500000	3510000	3250000	3260000	11193
2000-01-17	3360000	3450000	3330000	3360000	7453
2000-01-18	3360000	3410000	3310000	3380000	7169
2000-01-19	3330000	3350000	3200000	3210000	9656
2000-01-20	3220000	3300000	3160000	3250000	9603
2000-01-21	3230000	3260000	3170000	3210000	8598
2000-01-24	3170000	3290000	3140000	3290000	10731
2000-01-25	3240000	3370000	3230000	3250000	9908
2000-01-26	3390000	3500000	3290000	3480000	15989
2000-01-27	3550000	3750000	3530000	3730000	27052
2000-01-28	3800000	3820000	3660000	3660000	13440
2000-01-31	3640000	3800000	3610000	3640000	14291
2000-02-01	3690000	3720000	3620000	3700000	12827
2000-02-02	3830000	3850000	3720000	3720000	17272
2000-02-03	3740000	3800000	3700000	3800000	10158
2000-02-04	3830000	3830000	3670000	3760000	11227
2000-02-07	3850000	3880000	3780000	3840000	20746
2000-02-08	3850000	3860000	3770000	3800000	11479
2000-02-09	3850000	3860000	3800000	3850000	9734
2000-02-10	3820000	3830000	3740000	3800000	12763
2000-02-14	3700000	3760000	3690000	3720000	9448
2000-02-15	3770000	4000000	3770000	3940000	24752
2000-02-16	3940000	3970000	3750000	3850000	12710
2000-02-17	3890000	3910000	3760000	3840000	11195
2000-02-18	3890000	3900000	3820000	3850000	8720
2000-02-21	3810000	3820000	3710000	3710000	7864
2000-02-22	3730000	3760000	3570000	3610000	7850
2000-02-23	3740000	3810000	3690000	3810000	11468
2000-02-24	3910000	4020000	3880000	3930000	36846
2000-02-25	4080000	4250000	4040000	4230000	51984
2000-02-28	4320000	4570000	4300000	4520000	46011
2000-02-29	4470000	4470000	4380000	4430000	16340
2000-03-01	4460000	4470000	4350000	4400000	11963
2000-03-02	4410000	4510000	4360000	4410000	14225
2000-03-03	4310000	4320000	4160000	4170000	10566
2000-03-06	4320000	4330000	4010000	4080000	11065
2000-03-07	4050000	4100000	3920000	4100000	13141
2000-03-08	4060000	4200000	4030000	4140000	13822
2000-03-09	4130000	4190000	3980000	4080000	11211
2000-03-10	4030000	4060000	3820000	3880000	24621
2000-03-13	3680000	3710000	3380000	3390000	22109
2000-03-14	3640000	3890000	3630000	3890000	23635
2000-03-15	4040000	4380000	4010000	4250000	29479
2000-03-16	4210000	4330000	4120000	4300000	19401
2000-03-17	4370000	4410000	4230000	4400000	16867
2000-03-21	4300000	4390000	4270000	4350000	10672
2000-03-22	4320000	4380000	4250000	4380000	9901
2000-03-23	4360000	4450000	4300000	4340000	11997
2000-03-24	4350000	4390000	4120000	4180000	12867
2000-03-27	4180000	4250000	4010000	4040000	13272
2000-03-28	4140000	4310000	4120000	4300000	9992
2000-03-29	4350000	4500000	4310000	4370000	18117
2000-03-30	4450000	4550000	4310000	4310000	21395
2000-03-31	4210000	4360000	4170000	4210000	14546
2000-04-03	4150000	4350000	4120000	4240000	19560
2000-04-04	4190000	4220000	4060000	4070000	15107
2000-04-05	3880000	4020000	3800000	3930000	26146
2000-04-06	3950000	3970000	3730000	3750000	19827
2000-04-07	3850000	4020000	3820000	3990000	15406
2000-04-10	4160000	4250000	4150000	4160000	18018
2000-04-11	4140000	4190000	4010000	4010000	9292
2000-04-12	3920000	3980000	3880000	3980000	11154
2000-04-13	3830000	3890000	3620000	3710000	26913
2000-04-14	3700000	3860000	3660000	3750000	30519
2000-04-17	3450000	3520000	3250000	3370000	36538
2000-04-18	3660000	3710000	3490000	3610000	23805
2000-04-19	3760000	3820000	3670000	3700000	13457
2000-04-20	3400000	3580000	3390000	3490000	43299
2000-04-21	3500000	3660000	3490000	3550000	17698
2000-04-24	3550000	3690000	3520000	3520000	8972
2000-04-25	3520000	3610000	3500000	3590000	9188
2000-04-26	3640000	3710000	3610000	3660000	9980
2000-04-27	3680000	3690000	3500000	3500000	9619
2000-04-28	3600000	3720000	3520000	3610000	12409
2000-05-01	3710000	3810000	3660000	3800000	11546
2000-05-02	3860000	3920000	3840000	3840000	14266
2000-05-08	3940000	3960000	3860000	3860000	7932
2000-05-09	3760000	3810000	3650000	3690000	15131
2000-05-10	3490000	3590000	3440000	3580000	22995
2000-05-11	3380000	3430000	3290000	3340000	29793
2000-05-12	3420000	3460000	3320000	3340000	20158
2000-05-15	3340000	3370000	3310000	3340000	11695
2000-05-16	3390000	3500000	3330000	3460000	14634
2000-05-17	3600000	3620000	3490000	3530000	18894
2000-05-18	3380000	3430000	3350000	3350000	13818
2000-05-19	3220000	3260000	3040000	3120000	27443
2000-05-22	2890000	2960000	2800000	2840000	31367
2000-05-23	2870000	2930000	2720000	2800000	25517
2000-05-24	2690000	2790000	2590000	2670000	35880
2000-05-25	2870000	2940000	2830000	2830000	24513
2000-05-26	2870000	3010000	2860000	2970000	14663
2000-05-29	2930000	2960000	2820000	2850000	13321
2000-05-30	2830000	2840000	2660000	2700000	18998
2000-05-31	2780000	2800000	2700000	2780000	16655
2000-06-01	2730000	2790000	2650000	2790000	23816
2000-06-02	2870000	2940000	2840000	2880000	13751
2000-06-05	3050000	3090000	3010000	3040000	21124
2000-06-06	3040000	3090000	3000000	3080000	13298
2000-06-07	3050000	3130000	3050000	3120000	9696
2000-06-08	3130000	3170000	3060000	3090000	9798
2000-06-09	3060000	3180000	3060000	3160000	10028
2000-06-12	3190000	3250000	3170000	3210000	9179
2000-06-13	3110000	3120000	3040000	3070000	9205
2000-06-14	3020000	3050000	2950000	3020000	14094
2000-06-15	2930000	2960000	2850000	2850000	10818
2000-06-16	2810000	2960000	2790000	2820000	14065
2000-06-19	2830000	2890000	2820000	2870000	8754
2000-06-20	2900000	2990000	2900000	2980000	7239
2000-06-21	3020000	3100000	3010000	3080000	9864
2000-06-22	3130000	3150000	3040000	3040000	12758
2000-06-23	2990000	3060000	2970000	3010000	9833
2000-06-26	2970000	3000000	2890000	2900000	8066
2000-06-27	2930000	3040000	2900000	3020000	7488
2000-06-28	2970000	3000000	2930000	2950000	6739
2000-06-29	2990000	3030000	2970000	3010000	7349
2000-06-30	2880000	2910000	2840000	2870000	13946
2000-07-03	2920000	3010000	2910000	2930000	10491
2000-07-04	2970000	2970000	2920000	2920000	4068
2000-07-05	2960000	2970000	2920000	2970000	5408
2000-07-06	2930000	2980000	2920000	2980000	4248
2000-07-07	3000000	3000000	2960000	2980000	4587
2000-07-10	3070000	3090000	3050000	3070000	9509
2000-07-11	3100000	3130000	3070000	3120000	8916
2000-07-12	3150000	3180000	3120000	3170000	8691
2000-07-13	3170000	3190000	3050000	3050000	10939
2000-07-14	3100000	3170000	3080000	3160000	7084
2000-07-17	3160000	3170000	3080000	3100000	4249
2000-07-18	3060000	3070000	2980000	3010000	9315
2000-07-19	2960000	3020000	2950000	3000000	4963
2000-07-21	3050000	3050000	2920000	2940000	4918
2000-07-24	2900000	2920000	2860000	2910000	6143
2000-07-25	2880000	2900000	2820000	2880000	7941
2000-07-26	2920000	3010000	2910000	2970000	9329
2000-07-27	2930000	2960000	2900000	2930000	5181
2000-07-28	2860000	2900000	2820000	2840000	8620
2000-07-31	2750000	2770000	2720000	2750000	10771
2000-08-01	2780000	2850000	2770000	2820000	13690
2000-08-02	2830000	2850000	2800000	2820000	5049
2000-08-03	2810000	2810000	2750000	2770000	6220
2000-08-04	2780000	2860000	2760000	2800000	10423
2000-08-07	2780000	2810000	2720000	2810000	7742
2000-08-08	2840000	2850000	2780000	2800000	9729
2000-08-09	2840000	2850000	2760000	2800000	8627
2000-08-10	2800000	2810000	2770000	2800000	3998
2000-08-11	2800000	2820000	2770000	2800000	9292
2000-08-14	2800000	2800000	2750000	2780000	4389
2000-08-15	2780000	2790000	2730000	2760000	5110
2000-08-16	2750000	2750000	2710000	2710000	4858
2000-08-17	2670000	2680000	2610000	2640000	13258
2000-08-18	2650000	2700000	2630000	2690000	10979
2000-08-21	2650000	2660000	2570000	2590000	11063
2000-08-22	2580000	2630000	2510000	2630000	15523
2000-08-23	2630000	2720000	2630000	2700000	11898
2000-08-24	2720000	2750000	2680000	2690000	8585
2000-08-25	2740000	2800000	2710000	2770000	9372
2000-08-28	2800000	2870000	2750000	2860000	7518
2000-08-29	2900000	2950000	2870000	2920000	13795
2000-08-30	2940000	2950000	2860000	2870000	6164
2000-08-31	2830000	2830000	2790000	2820000	7771
2000-09-01	2840000	2850000	2780000	2800000	7482
2000-09-04	2900000	2910000	2860000	2890000	5870
2000-09-05	2870000	2940000	2850000	2920000	8949
2000-09-06	2920000	2990000	2910000	2980000	15006
2000-09-07	3050000	3100000	3030000	3080000	24251
2000-09-08	3170000	3250000	3160000	3250000	21391
2000-09-11	3220000	3240000	3150000	3150000	11897
2000-09-12	3190000	3270000	3160000	3230000	13657
2000-09-13	3250000	3330000	3240000	3310000	17499
2000-09-14	3330000	3380000	3320000	3330000	13300
2000-09-18	3280000	3300000	3210000	3270000	8618
2000-09-19	3170000	3280000	3150000	3280000	11704
2000-09-20	3290000	3400000	3280000	3390000	12932
2000-09-21	3360000	3380000	3310000	3350000	12535
2000-09-22	3220000	3240000	3050000	3070000	16974
2000-09-25	3120000	3180000	3010000	3040000	15389
2000-09-26	3110000	3150000	3060000	3070000	10472
2000-09-27	3050000	3170000	3020000	3130000	14249
2000-09-28	3080000	3100000	2990000	3010000	13239
2000-09-29	3060000	3130000	3050000	3100000	12401
2000-10-02	3070000	3200000	3020000	3200000	12597
2000-10-03	3200000	3270000	3160000	3240000	12670
2000-10-04	3240000	3310000	3200000	3310000	10754
2000-10-05	3310000	3310000	3250000	3280000	6914
2000-10-06	3270000	3350000	3260000	3330000	11697
2000-10-10	3310000	3370000	3200000	3220000	12611
2000-10-11	3170000	3180000	3110000	3160000	7647
2000-10-12	3080000	3130000	3080000	3080000	9262
2000-10-13	3030000	3090000	2990000	3090000	10155
2000-10-16	3150000	3220000	3130000	3170000	9641
2000-10-17	3170000	3180000	3110000	3180000	7854
2000-10-18	3100000	3140000	3080000	3120000	10459
2000-10-19	3100000	3130000	3000000	3010000	10212
2000-10-20	3110000	3180000	3100000	3120000	10886
2000-10-23	3100000	3100000	3020000	3030000	9209
2000-10-24	3030000	3050000	2910000	2930000	11493
2000-10-25	2970000	2970000	2930000	2970000	8130
2000-10-26	2930000	2950000	2860000	2950000	9979
2000-10-27	2950000	2990000	2860000	2860000	6009
2000-10-30	2860000	2870000	2770000	2810000	8582
2000-10-31	2770000	2790000	2650000	2690000	11676
2000-11-01	2810000	2950000	2800000	2920000	12798
2000-11-02	2890000	2970000	2850000	2960000	7216
2000-11-06	3000000	3040000	2970000	3030000	9336
2000-11-07	3050000	3100000	3030000	3100000	9719
2000-11-08	3060000	3100000	3030000	3080000	9022
2000-11-09	3050000	3050000	2970000	2990000	7680
2000-11-10	2950000	2960000	2880000	2900000	8388
2000-11-13	2780000	2890000	2750000	2860000	12560
2000-11-14	2870000	2940000	2820000	2840000	10141
2000-11-15	2950000	3050000	2940000	3010000	18521
2000-11-16	3030000	3040000	2940000	2950000	10745
2000-11-17	2900000	3000000	2900000	2990000	10200
2000-11-20	2960000	2980000	2920000	2940000	7473
2000-11-21	2900000	2900000	2850000	2900000	8157
2000-11-22	2910000	2920000	2770000	2770000	11751
2000-11-24	2690000	2730000	2610000	2630000	17560
2000-11-27	2670000	2710000	2630000	2660000	14851
2000-11-28	2650000	2890000	2610000	2820000	18003
2000-11-29	2760000	2780000	2730000	2740000	11615
2000-11-30	2680000	2870000	2680000	2830000	7162
2000-12-01	2790000	2920000	2770000	2850000	12112
2000-12-04	2850000	2900000	2770000	2900000	8724
2000-12-05	2940000	2940000	2850000	2860000	5138
2000-12-06	2980000	3000000	2950000	2980000	10733
2000-12-07	2950000	2950000	2900000	2930000	5565
2000-12-08	2890000	2990000	2880000	2940000	20988
2000-12-11	2900000	2940000	2850000	2850000	9770
2000-12-12	2850000	2880000	2800000	2840000	10132
2000-12-13	2780000	2790000	2710000	2750000	16368
2000-12-14	2680000	2710000	2630000	2630000	14219
2000-12-15	2580000	2620000	2560000	2580000	15622
2000-12-18	2580000	2620000	2560000	2600000	8623
2000-12-19	2590000	2590000	2510000	2510000	14395
2000-12-20	2350000	2360000	2240000	2250000	29049
2000-12-21	2090000	2150000	2000000	2100000	28569
2000-12-22	2140000	2190000	2120000	2180000	15947
2000-12-25	2270000	2300000	2200000	2240000	8706
2000-12-26	2240000	2250000	2200000	2220000	6926
2000-12-27	2180000	2190000	2100000	2130000	8384
2000-12-28	2050000	2070000	1960000	2010000	18963
2000-12-29	1970000	1990000	1950000	1970000	8143
2001-01-04	2070000	2110000	2030000	2060000	16471
2001-01-05	2060000	2200000	2060000	2200000	22175
2001-01-09	2160000	2160000	2060000	2090000	19134
2001-01-10	2050000	2080000	1990000	2020000	16825
2001-01-11	2000000	2020000	1910000	1950000	22566
2001-01-12	1990000	2020000	1950000	1970000	16060
2001-01-15	1930000	2000000	1910000	1960000	12981
2001-01-16	2010000	2050000	1990000	2050000	8193
2001-01-17	2130000	2200000	2100000	2190000	15759
2001-01-18	2390000	2570000	2360000	2430000	42762
2001-01-19	2380000	2460000	2310000	2380000	20456
2001-01-22	2380000	2460000	2350000	2460000	14527
2001-01-23	2390000	2400000	2340000	2360000	9123
2001-01-24	2370000	2380000	2300000	2300000	6753
2001-01-25	2300000	2340000	2250000	2310000	10219
2001-01-26	2300000	2330000	2300000	2300000	6722
2001-01-29	2270000	2310000	2260000	2270000	8537
2001-01-30	2280000	2290000	2250000	2260000	7825
2001-01-31	2280000	2280000	2230000	2240000	10802
2001-02-01	2230000	2230000	2170000	2220000	15306
2001-02-02	2160000	2190000	2150000	2160000	11055
2001-02-05	2120000	2150000	2100000	2130000	24844
2001-02-06	2130000	2190000	2110000	2170000	36370
2001-02-07	2200000	2300000	2190000	2280000	24284
2001-02-08	2260000	2260000	2160000	2160000	12801
2001-02-09	2120000	2180000	2110000	2170000	12840
2001-02-13	2160000	2160000	2120000	2140000	6948
2001-02-14	2130000	2130000	2080000	2130000	7394
2001-02-15	2090000	2110000	2080000	2110000	4968
2001-02-16	2100000	2110000	2070000	2070000	5284
2001-02-19	2020000	2030000	1980000	2000000	11019
2001-02-20	1990000	2060000	1980000	2040000	7246
2001-02-21	2070000	2100000	2020000	2040000	16714
2001-02-22	2010000	2060000	1980000	2060000	23338
2001-02-23	2030000	2060000	2020000	2030000	14856
2001-02-26	2050000	2070000	2040000	2060000	9371
2001-02-27	2050000	2060000	2000000	2020000	15658
2001-02-28	2000000	2060000	1990000	2030000	16038
2001-03-01	2010000	2040000	2010000	2030000	9914
2001-03-02	2010000	2020000	1930000	1940000	25505
2001-03-05	1870000	1880000	1800000	1850000	23158
2001-03-06	1880000	1940000	1880000	1940000	12572
2001-03-07	2030000	2030000	1970000	2010000	15225
2001-03-08	2000000	2000000	1950000	1980000	9589
2001-03-09	2020000	2030000	2000000	2020000	18956
2001-03-12	1980000	2010000	1970000	1980000	7922
2001-03-13	1890000	1920000	1830000	1880000	18861
2001-03-14	1910000	1920000	1860000	1870000	11155
2001-03-15	1820000	1930000	1810000	1900000	15210
2001-03-16	1950000	2020000	1930000	2000000	16113
2001-03-19	1990000	2040000	1980000	2000000	11023
2001-03-21	2000000	2180000	2000000	2180000	22889
2001-03-22	2140000	2190000	2090000	2190000	40734
2001-03-23	2180000	2230000	2160000	2220000	23941
2001-03-26	2290000	2330000	2240000	2330000	25544
2001-03-27	2310000	2380000	2270000	2350000	18739
2001-03-28	2390000	2430000	2370000	2390000	19598
2001-03-29	2310000	2320000	2230000	2230000	13190
2001-03-30	2270000	2340000	2180000	2180000	21075
2001-04-02	2190000	2260000	2110000	2210000	17699
2001-04-03	2220000	2260000	2180000	2220000	8689
2001-04-04	2180000	2290000	2170000	2290000	14322
2001-04-05	2340000	2350000	2300000	2300000	15342
2001-04-06	2380000	2380000	2300000	2330000	13471
2001-04-09	2330000	2360000	2270000	2330000	15330
2001-04-10	2320000	2350000	2290000	2300000	10562
2001-04-11	2360000	2400000	2350000	2390000	19172
2001-04-12	2410000	2420000	2370000	2410000	13909
2001-04-13	2430000	2490000	2410000	2440000	27149
2001-04-16	2410000	2430000	2390000	2390000	7422
2001-04-17	2360000	2380000	2300000	2310000	12862
2001-04-18	2340000	2440000	2340000	2430000	12225
2001-04-19	2520000	2580000	2500000	2550000	25879
2001-04-20	2510000	2540000	2440000	2460000	12996
2001-04-23	2470000	2540000	2470000	2520000	8118
2001-04-24	2340000	2420000	2320000	2400000	30443
2001-04-25	2420000	2500000	2410000	2500000	16070
2001-04-26	2540000	2550000	2500000	2520000	10048
2001-04-27	2540000	2550000	2500000	2540000	10347
2001-05-01	2580000	2630000	2570000	2630000	11607
2001-05-02	2670000	2700000	2640000	2690000	15060
2001-05-07	2730000	2900000	2730000	2890000	28640
2001-05-08	2780000	2800000	2710000	2730000	20663
2001-05-09	2650000	2660000	2610000	2640000	22712
2001-05-10	2560000	2620000	2530000	2560000	24315
2001-05-11	2520000	2540000	2500000	2500000	15123
2001-05-14	2480000	2520000	2460000	2480000	16438
2001-05-15	2480000	2590000	2470000	2590000	18624
2001-05-16	2550000	2570000	2490000	2520000	12714
2001-05-17	2560000	2590000	2510000	2580000	14065
2001-05-18	2570000	2580000	2490000	2490000	12916
2001-05-21	2510000	2540000	2490000	2500000	8209
2001-05-22	2540000	2550000	2490000	2490000	9508
2001-05-23	2480000	2530000	2430000	2440000	13656
2001-05-24	2400000	2450000	2400000	2430000	14595
2001-05-25	2450000	2460000	2410000	2420000	8413
2001-05-28	2380000	2420000	2380000	2390000	5281
2001-05-29	2400000	2440000	2390000	2400000	8415
2001-05-30	2390000	2400000	2320000	2330000	11711
2001-05-31	2280000	2330000	2260000	2280000	19282
2001-06-01	2330000	2360000	2280000	2280000	10399
2001-06-04	2320000	2320000	2280000	2290000	8284
2001-06-05	2300000	2320000	2280000	2280000	7781
2001-06-06	2290000	2310000	2250000	2250000	8599
2001-06-07	2150000	2250000	2150000	2250000	20909
2001-06-08	2290000	2330000	2250000	2250000	25972
2001-06-11	2250000	2270000	2190000	2190000	11710
2001-06-12	2150000	2180000	2050000	2060000	27400
2001-06-13	2080000	2090000	2030000	2050000	20835
2001-06-14	2040000	2110000	2030000	2090000	9786
2001-06-15	2050000	2060000	2010000	2060000	16857
2001-06-18	2020000	2030000	1980000	1990000	10561
2001-06-19	2020000	2060000	2000000	2000000	12713
2001-06-20	2020000	2030000	1960000	2020000	17659
2001-06-21	2040000	2090000	2020000	2090000	12780
2001-06-22	2100000	2160000	2080000	2160000	16671
2001-06-25	2160000	2160000	2070000	2110000	11946
2001-06-26	2110000	2150000	2090000	2130000	11361
2001-06-27	2130000	2140000	2100000	2100000	7332
2001-06-28	2060000	2100000	2020000	2060000	13039
2001-06-29	2140000	2170000	2110000	2170000	17845
2001-07-02	2160000	2180000	2090000	2100000	13289
2001-07-03	2150000	2190000	2130000	2150000	9148
2001-07-04	2140000	2150000	2060000	2100000	9472
2001-07-05	2080000	2100000	2050000	2070000	7521
2001-07-06	2040000	2100000	2030000	2090000	8812
2001-07-09	2050000	2060000	2010000	2040000	9779
2001-07-10	2030000	2090000	2010000	2070000	11591
2001-07-11	2030000	2060000	2020000	2030000	7395
2001-07-12	2070000	2090000	2040000	2070000	11339
2001-07-13	2070000	2080000	2010000	2040000	9003
2001-07-16	2040000	2040000	1990000	2000000	9463
2001-07-17	1960000	1970000	1920000	1950000	14223
2001-07-18	1900000	1910000	1800000	1840000	29045
2001-07-19	1810000	1840000	1810000	1810000	12892
2001-07-23	1840000	1840000	1700000	1710000	24258
2001-07-24	1720000	1770000	1670000	1770000	23491
2001-07-25	1750000	1860000	1740000	1810000	20725
2001-07-26	1820000	1830000	1770000	1790000	13304
2001-07-27	1760000	1790000	1680000	1710000	22469
2001-07-30	1740000	1740000	1670000	1680000	8356
2001-07-31	1710000	1750000	1700000	1730000	11585
2001-08-01	1760000	1850000	1740000	1840000	20461
2001-08-02	1890000	1900000	1850000	1890000	16333
2001-08-03	1880000	1890000	1830000	1840000	13347
2001-08-06	1810000	1840000	1790000	1820000	9820
2001-08-07	1790000	1820000	1760000	1810000	12662
2001-08-08	1820000	1890000	1820000	1850000	13106
2001-08-09	1780000	1790000	1720000	1720000	15458
2001-08-10	1730000	1790000	1720000	1720000	11687
2001-08-13	1710000	1730000	1660000	1690000	11576
2001-08-14	1700000	1800000	1700000	1780000	11806
2001-08-15	1770000	1780000	1730000	1760000	5190
2001-08-16	1700000	1730000	1690000	1700000	8115
2001-08-17	1700000	1700000	1660000	1670000	8070
2001-08-20	1630000	1660000	1620000	1630000	9252
2001-08-21	1640000	1650000	1600000	1620000	11613
2001-08-22	1590000	1620000	1580000	1600000	9465
2001-08-23	1580000	1590000	1480000	1490000	16879
2001-08-24	1540000	1560000	1500000	1550000	17318
2001-08-27	1630000	1630000	1570000	1600000	9575
2001-08-28	1610000	1610000	1510000	1520000	14031
2001-08-29	1490000	1500000	1450000	1460000	16833
2001-08-30	1450000	1480000	1430000	1450000	13218
2001-08-31	1420000	1470000	1400000	1460000	14094
2001-09-03	1440000	1450000	1370000	1380000	14266
2001-09-04	1390000	1450000	1330000	1450000	25367
2001-09-05	1390000	1420000	1350000	1380000	19062
2001-09-06	1380000	1400000	1320000	1350000	18730
2001-09-07	1290000	1310000	1260000	1290000	20370
2001-09-10	1260000	1270000	1210000	1220000	17745
2001-09-11	1240000	1260000	1210000	1240000	24246
2001-09-12	1140000	1160000	1140000	1140000	10036
2001-09-13	1180000	1240000	1170000	1240000	21380
2001-09-14	1310000	1340000	1290000	1340000	14967
2001-09-17	1330000	1410000	1270000	1380000	30742
2001-09-18	1480000	1550000	1440000	1470000	44183
2001-09-19	1540000	1620000	1530000	1550000	30389
2001-09-20	1480000	1530000	1470000	1500000	29284
2001-09-21	1430000	1450000	1390000	1430000	18325
2001-09-25	1540000	1570000	1510000	1530000	19531
2001-09-26	1500000	1520000	1460000	1480000	17463
2001-09-27	1500000	1550000	1490000	1550000	15623
2001-09-28	1610000	1630000	1570000	1610000	26140
2001-10-01	1640000	1720000	1610000	1720000	27732
2001-10-02	1700000	1720000	1670000	1720000	22981
2001-10-03	1710000	1720000	1600000	1610000	20980
2001-10-04	1690000	1700000	1650000	1690000	18759
2001-10-05	1690000	1790000	1660000	1760000	27376
2001-10-09	1700000	1720000	1630000	1630000	23060
2001-10-10	1600000	1620000	1530000	1540000	28326
2001-10-11	1600000	1670000	1570000	1670000	22401
2001-10-12	1760000	1770000	1710000	1740000	32852
2001-10-15	1690000	1700000	1660000	1660000	13284
2001-10-16	1660000	1690000	1640000	1670000	12090
2001-10-17	1700000	1710000	1680000	1710000	14760
2001-10-18	1660000	1680000	1650000	1670000	9803
2001-10-19	1640000	1700000	1630000	1680000	12181
2001-10-22	1680000	1700000	1650000	1660000	8859
2001-10-23	1700000	1720000	1680000	1700000	19075
2001-10-24	1710000	1760000	1710000	1740000	23319
2001-10-25	1760000	1820000	1680000	1710000	31549
2001-10-26	1740000	1770000	1730000	1740000	22016
2001-10-29	1740000	1750000	1720000	1740000	12515
2001-10-30	1710000	1730000	1680000	1700000	15766
2001-10-31	1670000	1690000	1650000	1660000	19643
2001-11-01	1670000	1690000	1660000	1680000	10974
2001-11-02	1740000	1750000	1640000	1660000	17013
2001-11-05	1650000	1660000	1580000	1590000	18601
2001-11-06	1600000	1630000	1580000	1590000	17187
2001-11-07	1590000	1600000	1530000	1530000	16427
2001-11-08	1500000	1510000	1460000	1490000	22096
2001-11-09	1470000	1490000	1450000	1460000	19471
2001-11-12	1460000	1500000	1440000	1490000	14161
2001-11-13	1490000	1560000	1490000	1560000	16304
2001-11-14	1590000	1610000	1580000	1580000	15560
2001-11-15	1590000	1660000	1580000	1650000	13715
2001-11-16	1690000	1740000	1670000	1710000	29519
2001-11-19	1740000	1770000	1720000	1750000	18014
2001-11-20	1740000	1750000	1670000	1680000	15992
2001-11-21	1660000	1680000	1590000	1620000	24719
2001-11-22	1620000	1680000	1610000	1660000	15645
2001-11-26	1720000	1770000	1700000	1770000	20572
2001-11-27	1740000	1760000	1720000	1720000	16080
2001-11-28	1690000	1700000	1640000	1670000	16850
2001-11-29	1640000	1690000	1600000	1610000	23639
2001-11-30	1640000	1650000	1580000	1610000	30238
2001-12-03	1630000	1630000	1570000	1580000	11710
2001-12-04	1560000	1610000	1540000	1600000	15778
2001-12-05	1630000	1660000	1620000	1660000	14975
2001-12-06	1720000	1730000	1680000	1710000	16796
2001-12-07	1680000	1680000	1640000	1660000	9763
2001-12-10	1640000	1640000	1580000	1580000	10751
2001-12-11	1550000	1610000	1540000	1570000	12922
2001-12-12	1540000	1590000	1540000	1560000	17008
2001-12-13	1520000	1530000	1470000	1490000	17518
2001-12-14	1450000	1470000	1430000	1440000	35649
2001-12-17	1440000	1510000	1390000	1470000	18839
2001-12-18	1500000	1510000	1410000	1440000	16323
2001-12-19	1420000	1480000	1400000	1460000	14651
2001-12-20	1490000	1550000	1470000	1550000	19354
2001-12-21	1490000	1510000	1480000	1510000	11786
2001-12-25	1520000	1520000	1470000	1480000	7916
2001-12-26	1460000	1470000	1430000	1430000	7976
2001-12-27	1450000	1510000	1450000	1490000	7455
2001-12-28	1520000	1540000	1510000	1540000	7766
2002-01-04	1590000	1600000	1570000	1600000	6012
2002-01-07	1600000	1610000	1580000	1600000	7652
2002-01-08	1590000	1600000	1530000	1550000	15216
2002-01-09	1530000	1530000	1490000	1500000	11380
2002-01-10	1480000	1500000	1450000	1460000	12485
2002-01-11	1450000	1470000	1420000	1420000	18226
2002-01-15	1380000	1390000	1370000	1370000	16359
2002-01-16	1370000	1420000	1370000	1410000	12645
2002-01-17	1390000	1410000	1370000	1380000	16070
2002-01-18	1400000	1430000	1400000	1430000	16853
2002-01-21	1410000	1420000	1380000	1400000	12004
2002-01-22	1400000	1410000	1360000	1360000	13252
2002-01-23	1350000	1360000	1310000	1320000	14806
2002-01-24	1340000	1370000	1320000	1350000	12090
2002-01-25	1390000	1390000	1340000	1380000	13921
2002-01-28	1480000	1490000	1450000	1480000	24367
2002-01-29	1460000	1460000	1420000	1430000	15270
2002-01-30	1390000	1420000	1380000	1380000	18443
2002-01-31	1410000	1430000	1400000	1400000	14755
2002-02-01	1400000	1410000	1370000	1390000	13043
2002-02-04	1410000	1420000	1390000	1410000	9778
2002-02-05	1370000	1380000	1350000	1360000	15057
2002-02-06	1340000	1360000	1310000	1320000	14275
2002-02-07	1330000	1390000	1330000	1390000	13976
2002-02-08	1370000	1390000	1340000	1360000	21932
2002-02-12	1400000	1430000	1400000	1410000	15836
2002-02-13	1390000	1440000	1380000	1440000	15873
2002-02-14	1450000	1480000	1430000	1440000	17093
2002-02-15	1440000	1450000	1400000	1420000	15864
2002-02-18	1400000	1400000	1380000	1390000	7879
2002-02-19	1400000	1410000	1360000	1390000	10316
2002-02-20	1350000	1390000	1350000	1370000	12135
2002-02-21	1390000	1420000	1380000	1420000	13024
2002-02-22	1400000	1410000	1370000	1390000	15242
2002-02-25	1410000	1420000	1390000	1390000	9463
2002-02-26	1430000	1450000	1400000	1410000	15890
2002-02-27	1390000	1420000	1380000	1390000	19056
2002-02-28	1410000	1430000	1380000	1390000	25333
2002-03-01	1420000	1430000	1390000	1420000	14889
2002-03-04	1470000	1530000	1460000	1520000	30809
2002-03-05	1560000	1580000	1540000	1540000	24870
2002-03-06	1540000	1600000	1530000	1590000	24886
2002-03-07	1650000	1720000	1640000	1720000	37718
2002-03-08	1750000	1770000	1710000	1760000	51455
2002-03-11	1780000	1790000	1720000	1760000	21357
2002-03-12	1740000	1740000	1670000	1670000	19941
2002-03-13	1640000	1680000	1630000	1630000	20772
2002-03-14	1620000	1670000	1620000	1670000	16778
2002-03-15	1710000	1720000	1690000	1720000	16415
2002-03-18	1720000	1740000	1670000	1670000	21176
2002-03-19	1700000	1710000	1680000	1710000	18253
2002-03-20	1720000	1740000	1700000	1710000	25769
2002-03-22	1700000	1700000	1660000	1670000	22447
2002-03-25	1670000	1710000	1660000	1680000	49377
2002-03-26	341000	351000	329000	333000	208021
2002-03-27	338000	351000	336000	347000	139550
2002-03-28	352000	364000	343000	360000	153194
2002-03-29	365000	369000	352000	352000	102695
2002-04-01	357000	360000	347000	347000	104449
2002-04-02	352000	358000	351000	357000	67572
2002-04-03	352000	369000	349000	367000	139874
2002-04-04	369000	373000	363000	372000	119671
2002-04-05	365000	366000	357000	358000	81233
2002-04-08	359000	369000	355000	367000	77402
2002-04-09	366000	366000	353000	353000	61930
2002-04-10	338000	342000	332000	334000	143115
2002-04-11	336000	338000	331000	332000	75891
2002-04-12	322000	324000	313000	316000	147905
2002-04-15	316000	321000	313000	320000	64452
2002-04-16	324000	329000	323000	329000	77327
2002-04-17	337000	338000	331000	334000	79001
2002-04-18	335000	342000	334000	338000	77379
2002-04-19	334000	335000	328000	335000	48452
2002-04-22	337000	343000	337000	342000	49006
2002-04-23	333000	340000	329000	335000	60427
2002-04-24	334000	338000	330000	331000	43539
2002-04-25	334000	338000	332000	334000	35147
2002-04-26	331000	334000	327000	328000	60590
2002-04-30	333000	333000	325000	325000	31619
2002-05-01	329000	333000	327000	328000	30663
2002-05-02	327000	329000	326000	327000	27050
2002-05-07	318000	320000	313000	318000	69210
2002-05-08	313000	319000	313000	315000	31006
2002-05-09	330000	332000	323000	325000	55601
2002-05-10	320000	321000	312000	315000	79678
2002-05-13	311000	312000	307000	309000	49562
2002-05-14	310000	311000	301000	302000	67461
2002-05-15	312000	314000	309000	310000	67685
2002-05-16	313000	323000	312000	322000	62508
2002-05-17	322000	331000	321000	321000	109239
2002-05-20	328000	334000	326000	332000	87381
2002-05-21	331000	339000	330000	338000	84689
2002-05-22	336000	347000	334000	344000	89789
2002-05-23	346000	348000	341000	344000	69812
2002-05-24	339000	340000	330000	331000	124111
2002-05-27	326000	328000	323000	323000	72543
2002-05-28	325000	325000	319000	321000	66100
2002-05-29	322000	325000	321000	324000	62711
2002-05-30	320000	323000	316000	323000	66784
2002-05-31	324000	336000	323000	336000	103775
2002-06-03	331000	337000	329000	331000	74375
2002-06-04	334000	335000	322000	325000	54688
2002-06-05	322000	329000	320000	323000	67074
2002-06-06	325000	333000	324000	326000	92034
2002-06-07	322000	323000	318000	321000	70145
2002-06-10	316000	317000	312000	313000	60149
2002-06-11	313000	316000	313000	314000	51505
2002-06-12	314000	314000	310000	314000	43029
2002-06-13	316000	318000	308000	308000	51263
2002-06-14	306000	310000	301000	301000	157819
2002-06-17	295000	296000	287000	290000	108038
2002-06-18	295000	301000	294000	300000	79465
2002-06-19	296000	298000	288000	294000	104970
2002-06-20	290000	295000	285000	294000	76657
2002-06-21	285000	287000	284000	284000	51372
2002-06-24	277000	286000	274000	285000	55004
2002-06-25	284000	295000	284000	292000	63279
2002-06-26	282000	284000	276000	277000	76560
2002-06-27	281000	285000	278000	281000	66571
2002-06-28	288000	295000	287000	295000	78797
2002-07-01	298000	303000	295000	297000	52600
2002-07-02	289000	292000	281000	287000	97870
2002-07-03	281000	287000	279000	284000	107602
2002-07-04	285000	287000	275000	276000	106416
2002-07-05	280000	283000	278000	281000	73217
2002-07-08	289000	290000	271000	271000	111197
2002-07-09	275000	277000	266000	277000	103039
2002-07-10	273000	279000	270000	271000	89375
2002-07-11	267000	268000	263000	264000	74613
2002-07-12	267000	270000	263000	264000	59947
2002-07-15	262000	262000	256000	256000	53962
2002-07-16	250000	254000	244000	244000	114328
2002-07-17	248000	255000	245000	253000	88181
2002-07-18	260000	268000	258000	264000	97778
2002-07-19	260000	262000	258000	260000	63692
2002-07-22	251000	264000	250000	264000	66447
2002-07-23	259000	262000	256000	260000	68751
2002-07-24	255000	257000	251000	251000	60366
2002-07-25	256000	258000	250000	250000	51177
2002-07-26	251000	252000	244000	247000	85129
2002-07-29	249000	270000	248000	269000	103195
2002-07-30	273000	275000	266000	273000	91838
2002-07-31	270000	271000	266000	267000	45321
2002-08-01	268000	269000	266000	267000	59351
2002-08-02	267000	279000	263000	275000	103012
2002-08-05	273000	285000	272000	281000	78932
2002-08-06	282000	283000	275000	276000	79213
2002-08-07	280000	283000	278000	280000	52100
2002-08-08	278000	281000	276000	276000	42991
2002-08-09	283000	285000	280000	284000	51567
2002-08-12	284000	286000	281000	281000	37219
2002-08-13	279000	279000	275000	276000	33062
2002-08-14	276000	276000	270000	272000	41767
2002-08-15	273000	276000	266000	267000	48342
2002-08-16	267000	270000	259000	264000	42914
2002-08-19	263000	263000	253000	253000	50380
2002-08-20	257000	259000	253000	256000	57359
2002-08-21	254000	258000	253000	258000	43669
2002-08-22	256000	261000	250000	259000	83342
2002-08-23	269000	269000	260000	262000	91104
2002-08-26	260000	272000	259000	269000	68709
2002-08-27	266000	266000	260000	260000	59583
2002-08-28	264000	264000	257000	257000	45580
2002-08-29	254000	256000	251000	251000	53128
2002-08-30	253000	255000	248000	252000	53268
2002-09-02	252000	253000	248000	251000	54575
2002-09-03	249000	250000	244000	244000	60554
2002-09-04	238000	240000	231000	232000	79289
2002-09-05	239000	245000	236000	241000	89158
2002-09-06	233000	235000	231000	232000	47401
2002-09-09	236000	239000	233000	233000	47153
2002-09-10	234000	238000	233000	234000	67125
2002-09-11	237000	238000	232000	233000	52660
2002-09-12	230000	233000	227000	230000	68399
2002-09-13	222000	226000	219000	220000	152092
2002-09-17	221000	227000	217000	226000	79283
2002-09-18	218000	225000	213000	222000	69930
2002-09-19	237000	239000	225000	225000	100221
2002-09-20	217000	225000	214000	215000	72822
2002-09-24	207000	209000	202000	208000	77359
2002-09-25	202000	209000	201000	206000	106090
2002-09-26	210000	213000	206000	206000	101066
2002-09-27	214000	219000	211000	216000	112065
2002-09-30	211000	214000	206000	208000	47058
2002-10-01	204000	210000	203000	206000	53825
2002-10-02	210000	211000	205000	205000	48091
2002-10-03	210000	213000	208000	212000	82887
2002-10-04	220000	235000	219000	235000	163229
2002-10-07	231000	233000	222000	225000	83931
2002-10-08	227000	232000	224000	229000	59993
2002-10-09	226000	227000	222000	224000	52887
2002-10-10	220000	222000	213000	218000	53741
2002-10-11	222000	227000	221000	225000	49135
2002-10-15	234000	240000	232000	238000	58671
2002-10-16	244000	245000	238000	242000	60232
2002-10-17	241000	245000	235000	236000	47424
2002-10-18	243000	243000	237000	237000	38226
2002-10-21	239000	241000	234000	235000	36283
2002-10-22	236000	236000	218000	218000	52237
2002-10-23	219000	227000	215000	226000	62081
2002-10-24	226000	227000	214000	219000	33429
2002-10-25	222000	231000	222000	230000	52673
2002-10-28	228000	231000	224000	231000	44791
2002-10-29	230000	235000	224000	224000	37296
2002-10-30	222000	231000	220000	228000	48338
2002-10-31	232000	232000	222000	226000	38900
2002-11-01	231000	237000	227000	236000	68250
2002-11-05	244000	251000	242000	245000	102262
2002-11-06	244000	251000	241000	241000	88949
2002-11-07	241000	245000	236000	241000	43581
2002-11-08	234000	237000	227000	227000	61568
2002-11-11	221000	223000	213000	213000	55815
2002-11-12	209000	223000	208000	221000	57525
2002-11-13	225000	228000	220000	223000	61560
2002-11-14	219000	229000	218000	221000	46946
2002-11-15	229000	232000	225000	227000	43474
2002-11-18	228000	228000	215000	218000	47301
2002-11-19	215000	218000	208000	212000	58999
2002-11-20	212000	222000	210000	216000	46572
2002-11-21	222000	226000	220000	224000	59336
2002-11-22	233000	237000	231000	233000	94085
2002-11-25	236000	244000	234000	244000	83124
2002-11-26	248000	250000	237000	240000	67701
2002-11-27	236000	241000	235000	237000	53582
2002-11-28	243000	254000	241000	252000	89654
2002-11-29	252000	257000	247000	248000	75882
2002-12-02	249000	251000	244000	244000	42532
2002-12-03	251000	256000	249000	253000	64356
2002-12-04	245000	250000	244000	246000	50263
2002-12-05	245000	247000	241000	242000	43824
2002-12-06	240000	242000	238000	240000	41917
2002-12-09	237000	238000	233000	235000	50246
2002-12-10	231000	239000	230000	234000	44623
2002-12-11	236000	237000	232000	234000	37429
2002-12-12	232000	235000	231000	231000	35327
2002-12-13	229000	231000	224000	224000	120229
2002-12-16	224000	227000	219000	221000	57627
2002-12-17	225000	227000	221000	222000	51476
2002-12-18	223000	225000	219000	219000	52470
2002-12-19	220000	225000	218000	223000	62431
2002-12-20	223000	226000	218000	218000	54986
2002-12-24	221000	223000	218000	222000	59959
2002-12-25	220000	222000	217000	217000	35158
2002-12-26	218000	221000	218000	220000	29514
2002-12-27	221000	224000	219000	221000	38524
2002-12-30	218000	220000	217000	219000	17513
2003-01-06	226000	232000	226000	229000	32374
2003-01-07	237000	238000	233000	233000	52812
2003-01-08	230000	235000	229000	234000	59691
2003-01-09	231000	239000	229000	239000	69653
2003-01-10	242000	242000	235000	237000	56362
2003-01-14	241000	252000	240000	248000	121115
2003-01-15	249000	256000	246000	256000	132971
2003-01-16	252000	258000	252000	256000	105361
2003-01-17	253000	258000	252000	254000	83098
2003-01-20	250000	251000	242000	247000	81658
2003-01-21	246000	254000	246000	250000	117044
2003-01-22	251000	251000	244000	246000	50332
2003-01-23	246000	249000	242000	249000	46785
2003-01-24	247000	251000	245000	246000	46639
2003-01-27	244000	247000	237000	237000	49186
2003-01-28	235000	238000	234000	234000	31819
2003-01-29	236000	237000	232000	233000	54663
2003-01-30	236000	240000	235000	238000	41238
2003-01-31	238000	239000	230000	231000	59554
2003-02-03	226000	230000	225000	228000	68685
2003-02-04	232000	237000	232000	234000	60929
2003-02-05	230000	243000	229000	233000	112672
2003-02-06	234000	235000	228000	230000	43147
2003-02-07	228000	233000	227000	232000	40853
2003-02-10	230000	236000	229000	236000	35789
2003-02-12	237000	244000	236000	244000	56853
2003-02-13	243000	247000	242000	246000	63013
2003-02-14	244000	249000	244000	246000	53835
2003-02-17	249000	250000	244000	247000	60496
2003-02-18	246000	250000	242000	242000	67061
2003-02-19	246000	246000	239000	239000	41425
2003-02-20	238000	244000	237000	240000	48110
2003-02-21	241000	245000	236000	239000	39191
2003-02-24	237000	239000	232000	233000	39503
2003-02-25	230000	232000	226000	228000	51876
2003-02-26	226000	230000	223000	227000	42507
2003-02-27	225000	227000	220000	227000	45441
2003-02-28	227000	228000	223000	223000	41961
2003-03-03	224000	228000	222000	228000	39611
2003-03-04	225000	231000	225000	229000	59159
2003-03-05	225000	228000	223000	227000	44911
2003-03-06	225000	231000	222000	223000	43028
2003-03-07	221000	223000	215000	215000	54464
2003-03-10	211000	212000	205000	208000	66018
2003-03-11	204000	211000	201000	201000	78338
2003-03-12	204000	215000	203000	215000	92639
2003-03-13	213000	216000	208000	210000	92497
2003-03-14	222000	223000	216000	219000	153409
2003-03-17	223000	224000	217000	222000	74606
2003-03-18	227000	229000	217000	217000	79382
2003-03-19	216000	221000	212000	220000	55451
2003-03-20	226000	232000	225000	232000	84467
2003-03-24	240000	243000	237000	241000	74995
2003-03-25	233000	236000	225000	227000	60457
2003-03-26	231000	234000	228000	231000	48880
2003-03-27	233000	240000	233000	236000	73287
2003-03-28	237000	240000	236000	238000	33726
2003-03-31	235000	236000	221000	221000	62657
2003-04-01	220000	227000	215000	225000	66782
2003-04-02	229000	230000	224000	230000	47723
2003-04-03	237000	238000	230000	232000	62684
2003-04-04	231000	234000	229000	230000	57433
2003-04-07	234000	242000	234000	240000	81923
2003-04-08	241000	246000	239000	245000	101402
2003-04-09	249000	250000	244000	246000	83410
2003-04-10	242000	246000	241000	243000	70915
2003-04-11	244000	245000	237000	238000	57244
2003-04-14	234000	237000	225000	229000	78101
2003-04-15	234000	242000	234000	237000	77763
2003-04-16	241000	246000	233000	243000	109455
2003-04-17	244000	248000	242000	246000	76947
2003-04-18	247000	248000	242000	242000	45768
2003-04-21	244000	250000	243000	249000	51250
2003-04-22	246000	248000	240000	240000	42472
2003-04-23	244000	250000	241000	246000	84851
2003-04-24	250000	254000	247000	251000	85988
2003-04-25	245000	248000	243000	247000	58662
2003-04-28	241000	246000	241000	241000	40900
2003-04-30	246000	248000	240000	246000	71448
2003-05-01	244000	246000	242000	245000	38574
2003-05-02	246000	251000	243000	250000	50720
2003-05-06	258000	267000	258000	266000	118845
2003-05-07	273000	277000	268000	269000	91199
2003-05-08	267000	268000	259000	262000	66944
2003-05-09	259000	260000	254000	258000	97197
2003-05-12	259000	264000	255000	263000	76299
2003-05-13	267000	270000	262000	265000	79969
2003-05-14	266000	269000	265000	265000	57641
2003-05-15	265000	266000	261000	263000	57446
2003-05-16	263000	264000	255000	256000	40283
2003-05-19	252000	262000	250000	251000	83139
2003-05-20	252000	259000	252000	256000	50406
2003-05-21	257000	258000	251000	254000	36737
2003-05-22	255000	260000	254000	257000	60373
2003-05-23	261000	264000	260000	261000	49431
2003-05-26	261000	264000	256000	256000	44640
2003-05-27	254000	254000	248000	249000	44174
2003-05-28	255000	257000	249000	249000	47566
2003-05-29	253000	257000	252000	256000	47120
2003-05-30	255000	257000	251000	251000	47674
2003-06-02	259000	262000	257000	260000	56140
2003-06-03	260000	268000	259000	265000	107949
2003-06-04	270000	271000	266000	266000	65404
2003-06-05	269000	270000	264000	265000	46873
2003-06-06	268000	272000	266000	272000	47356
2003-06-09	272000	272000	266000	266000	40471
2003-06-10	262000	264000	257000	262000	44457
2003-06-11	264000	267000	262000	263000	41078
2003-06-12	265000	266000	261000	261000	32561
2003-06-13	265000	269000	263000	267000	121204
2003-06-16	264000	265000	260000	262000	39798
2003-06-17	268000	269000	263000	265000	72321
2003-06-18	269000	272000	266000	267000	82372
2003-06-19	268000	272000	266000	271000	59482
2003-06-20	270000	277000	269000	276000	96030
2003-06-23	276000	277000	272000	273000	53321
2003-06-24	269000	270000	263000	264000	59755
2003-06-25	262000	267000	261000	262000	43308
2003-06-26	261000	262000	256000	259000	58537
2003-06-27	261000	262000	257000	262000	66368
2003-06-30	263000	264000	259000	260000	63812
2003-07-01	259000	263000	259000	260000	61747
2003-07-02	264000	278000	263000	277000	140009
2003-07-03	290000	296000	278000	283000	210518
2003-07-04	279000	285000	277000	284000	98697
2003-07-07	281000	291000	281000	289000	70813
2003-07-08	300000	306000	293000	300000	166381
2003-07-09	301000	304000	288000	296000	77688
2003-07-10	296000	303000	293000	293000	90180
2003-07-11	285000	288000	277000	277000	84890
2003-07-14	284000	288000	281000	283000	75285
2003-07-15	286000	290000	282000	284000	76077
2003-07-16	278000	281000	273000	274000	75270
2003-07-17	275000	277000	271000	272000	66499
2003-07-18	275000	279000	274000	275000	60975
2003-07-22	272000	273000	265000	265000	63494
2003-07-23	267000	269000	263000	266000	53531
2003-07-24	270000	271000	266000	267000	60840
2003-07-25	268000	276000	267000	272000	102616
2003-07-28	279000	281000	278000	279000	60632
2003-07-29	283000	284000	280000	280000	59464
2003-07-30	280000	281000	274000	274000	57720
2003-07-31	275000	281000	270000	276000	91782
2003-08-01	280000	281000	275000	275000	65455
2003-08-04	276000	281000	275000	278000	78893
2003-08-05	278000	279000	272000	273000	57271
2003-08-06	269000	280000	267000	278000	84873
2003-08-07	280000	282000	269000	269000	69699
2003-08-08	277000	284000	276000	280000	150809
2003-08-11	281000	287000	280000	286000	88924
2003-08-12	287000	287000	281000	282000	51319
2003-08-13	286000	289000	283000	287000	48312
2003-08-14	287000	294000	286000	293000	58297
2003-08-15	297000	298000	284000	286000	58353
2003-08-18	289000	293000	287000	287000	67297
2003-08-19	295000	304000	292000	303000	118011
2003-08-20	304000	311000	302000	307000	135653
2003-08-21	303000	315000	303000	315000	80059
2003-08-22	315000	320000	310000	311000	104586
2003-08-25	310000	315000	307000	308000	57857
2003-08-26	307000	315000	304000	312000	49954
2003-08-27	312000	318000	309000	310000	66623
2003-08-28	311000	312000	301000	302000	52570
2003-08-29	306000	306000	298000	300000	49123
2003-09-01	302000	307000	299000	307000	47903
2003-09-02	307000	308000	301000	301000	50299
2003-09-03	306000	307000	294000	297000	74457
2003-09-04	300000	300000	291000	292000	53719
2003-09-05	295000	297000	283000	288000	73468
2003-09-08	285000	288000	279000	281000	104640
2003-09-09	287000	291000	282000	288000	127038
2003-09-10	286000	294000	283000	290000	133052
2003-09-11	287000	288000	282000	282000	100536
2003-09-12	286000	291000	284000	288000	137276
2003-09-16	291000	294000	290000	291000	97449
2003-09-17	299000	308000	297000	302000	195042
2003-09-18	297000	302000	293000	299000	97095
2003-09-19	300000	302000	286000	286000	112699
2003-09-22	286000	289000	278000	282000	109986
2003-09-24	279000	287000	279000	285000	76709
2003-09-25	280000	282000	278000	278000	75973
2003-09-26	277000	281000	275000	278000	41642
cut
1999-06-01	274000	281200	272800	281200	101850
1999-06-02	281200	281200	274400	275200	51025
1999-06-03	276000	276000	269600	272800	38275
1999-06-04	274400	276800	271200	273200	37150
1999-06-07	277200	279200	274800	276000	38725
1999-06-08	277600	277600	271200	272800	45650
1999-06-09	272800	275600	271200	273200	57775
1999-06-10	274400	278400	274400	277600	60500
1999-06-11	280000	286000	278800	284400	121000
1999-06-14	285200	288400	282800	282800	56625
1999-06-15	286800	288000	283600	288000	60800
1999-06-16	289600	292400	288800	290000	65600
1999-06-17	294000	294000	291200	293600	61600
1999-06-18	293600	293600	288000	288800	92900
1999-06-21	288800	296000	288800	293600	91925
1999-06-22	295600	296000	288000	290000	117000
1999-06-23	287200	290000	285200	286400	162425
1999-06-24	288000	297600	287600	296000	289875
1999-06-25	336000	336000	328000	336000	335315
1999-06-28	342000	344000	330000	334000	105910
1999-06-29	336000	338000	326000	328000	59870
1999-06-30	330000	338000	328000	328000	54230
1999-07-01	340000	354000	336000	352000	105165
1999-07-02	358000	362000	352000	362000	83285
1999-07-05	364000	374000	362000	370000	76525
1999-07-06	368000	368000	354000	364000	70350
1999-07-07	360000	362000	352000	356000	28710
1999-07-08	354000	356000	350000	352000	35085
1999-07-09	348000	350000	344000	348000	31180
1999-07-12	348000	358000	346000	354000	31370
1999-07-13	352000	358000	348000	348000	31930
1999-07-14	354000	354000	348000	352000	29935
1999-07-15	354000	366000	350000	366000	68250
1999-07-16	372000	376000	362000	364000	73700
1999-07-19	360000	370000	354000	370000	30295
1999-07-21	364000	370000	360000	368000	54695
1999-07-22	364000	364000	352000	354000	33710
1999-07-23	348000	364000	346000	350000	49810
1999-07-26	356000	358000	350000	350000	39220
1999-07-27	348000	354000	346000	350000	34080
1999-07-28	356000	358000	352000	356000	28520
1999-07-29	356000	358000	352000	356000	20445
1999-07-30	358000	360000	354000	356000	29815
1999-08-02	356000	356000	350000	354000	15510
1999-08-03	352000	354000	348000	354000	16950
1999-08-04	352000	352000	344000	344000	24865
1999-08-05	342000	346000	334000	334000	31700
1999-08-06	334000	338000	330000	330000	34010
1999-08-09	334000	340000	334000	336000	25925
1999-08-10	336000	340000	334000	340000	17435
1999-08-11	340000	344000	336000	342000	20015
1999-08-12	346000	346000	342000	344000	20900
1999-08-13	348000	350000	344000	346000	18705
1999-08-16	350000	356000	348000	352000	22505
1999-08-17	360000	366000	356000	366000	53155
1999-08-18	366000	372000	362000	368000	64185
1999-08-19	368000	368000	360000	362000	32080
1999-08-20	356000	366000	356000	362000	17900
1999-08-23	364000	364000	350000	352000	19340
1999-08-24	358000	360000	352000	352000	16825
1999-08-25	356000	356000	346000	352000	20590
1999-08-26	354000	358000	350000	352000	16535
1999-08-27	354000	358000	350000	356000	22205
1999-08-30	360000	366000	358000	364000	28510
1999-08-31	368000	370000	364000	364000	49495
1999-09-01	366000	366000	358000	362000	24620
1999-09-02	364000	366000	360000	364000	34375
1999-09-03	366000	372000	366000	370000	74485
1999-09-06	376000	392000	376000	388000	108035
1999-09-07	402000	404000	394000	400000	81105
1999-09-08	398000	400000	392000	398000	58235
1999-09-09	396000	398000	386000	386000	42425
1999-09-10	392000	398000	390000	398000	72305
1999-09-13	406000	410000	400000	406000	78300
1999-09-14	408000	410000	390000	400000	44030
1999-09-16	374000	382000	374000	382000	89335
1999-09-17	378000	378000	370000	372000	48535
1999-09-20	390000	408000	388000	402000	156595
1999-09-21	408000	410000	404000	410000	108145
1999-09-22	402000	420000	398000	410000	154900
1999-09-24	394000	398000	376000	388000	115370
1999-09-27	392000	394000	378000	386000	53870
1999-09-28	392000	400000	390000	398000	51825
1999-09-29	398000	400000	392000	398000	54255
1999-09-30	404000	420000	402000	420000	104785
1999-10-01	420000	446000	418000	434000	211190
1999-10-04	436000	438000	432000	436000	37665
1999-10-05	438000	440000	426000	430000	49680
1999-10-06	426000	430000	420000	424000	42500
1999-10-07	436000	446000	432000	432000	82870
1999-10-08	438000	440000	432000	436000	34735
1999-10-12	448000	460000	448000	460000	83875
1999-10-13	456000	462000	452000	454000	66950
1999-10-14	460000	468000	456000	464000	59970
1999-10-15	460000	464000	450000	450000	54595
1999-10-18	418000	428000	416000	424000	67910
1999-10-19	428000	430000	414000	422000	59375
1999-10-20	438000	454000	436000	454000	88475
1999-10-21	462000	464000	444000	446000	57935
1999-10-22	458000	472000	456000	468000	98315
1999-10-25	484000	502000	482000	494000	125230
1999-10-26	492000	536000	488000	520000	166705
1999-10-27	518000	530000	502000	510000	133765
1999-10-28	524000	540000	518000	536000	104940
1999-10-29	564000	572000	550000	554000	123800
1999-11-01	562000	582000	558000	572000	117160
1999-11-02	576000	582000	566000	574000	103365
1999-11-04	582000	596000	580000	588000	88015
1999-11-05	586000	594000	574000	588000	76850
1999-11-08	580000	588000	562000	562000	65565
1999-11-09	554000	584000	542000	546000	127640
1999-11-10	546000	566000	546000	566000	62155
1999-11-11	564000	600000	556000	578000	134440
1999-11-12	574000	588000	570000	588000	67295
1999-11-15	604000	620000	602000	616000	128125
1999-11-16	606000	608000	584000	584000	73980
1999-11-17	610000	614000	560000	570000	98385
1999-11-18	562000	600000	558000	590000	80915
1999-11-19	606000	610000	592000	604000	66525
1999-11-22	618000	656000	610000	652000	149120
1999-11-24	646000	752000	640000	702000	180320
1999-11-25	752000	786000	690000	712000	173320
1999-11-26	726000	740000	694000	738000	95885
1999-11-29	742000	778000	736000	778000	99935
1999-11-30	748000	760000	716000	716000	162130
1999-12-01	696000	706000	644000	644000	192945
1999-12-02	652000	654000	598000	638000	272000
1999-12-03	660000	676000	654000	658000	121650
1999-12-06	698000	708000	686000	696000	105960
1999-12-07	706000	716000	682000	690000	86200
1999-12-08	680000	698000	668000	698000	68180
1999-12-09	696000	696000	664000	682000	66095
1999-12-10	684000	694000	678000	690000	70785
1999-12-13	684000	696000	678000	690000	44660
1999-12-14	684000	690000	672000	674000	62235
1999-12-15	670000	700000	668000	690000	58955
1999-12-16	700000	716000	698000	710000	75055
1999-12-17	720000	722000	692000	704000	46110
1999-12-20	702000	702000	672000	682000	33075
1999-12-21	672000	696000	670000	686000	36030
1999-12-22	698000	706000	692000	700000	34790
1999-12-24	720000	740000	706000	740000	70325
1999-12-27	742000	744000	722000	730000	28335
1999-12-28	726000	792000	724000	778000	95970
1999-12-29	798000	814000	782000	790000	63615
1999-12-30	788000	798000	784000	786000	26640
2000-01-04	800000	820000	786000	790000	47305
2000-01-05	710000	732000	690000	714000	132610
2000-01-06	722000	726000	676000	680000	72450
2000-01-07	620000	654000	610000	632000	144075
2000-01-11	722000	728000	706000	728000	112635
2000-01-12	700000	730000	698000	700000	60840
2000-01-13	690000	700000	676000	686000	50965
2000-01-14	700000	702000	650000	652000	55965
2000-01-17	672000	690000	666000	672000	37265
2000-01-18	672000	682000	662000	676000	35845
2000-01-19	666000	670000	640000	642000	48280
2000-01-20	644000	660000	632000	650000	48015
2000-01-21	646000	652000	634000	642000	42990
2000-01-24	634000	658000	628000	658000	53655
2000-01-25	648000	674000	646000	650000	49540
2000-01-26	678000	700000	658000	696000	79945
2000-01-27	710000	750000	706000	746000	135260
2000-01-28	760000	764000	732000	732000	67200
2000-01-31	728000	760000	722000	728000	71455
2000-02-01	738000	744000	724000	740000	64135
2000-02-02	766000	770000	744000	744000	86360
2000-02-03	748000	760000	740000	760000	50790
2000-02-04	766000	766000	734000	752000	56135
2000-02-07	770000	776000	756000	768000	103730
2000-02-08	770000	772000	754000	760000	57395
2000-02-09	770000	772000	760000	770000	48670
2000-02-10	764000	766000	748000	760000	63815
2000-02-14	740000	752000	738000	744000	47240
2000-02-15	754000	800000	754000	788000	123760
2000-02-16	788000	794000	750000	770000	63550
2000-02-17	778000	782000	752000	768000	55975
2000-02-18	778000	780000	764000	770000	43600
2000-02-21	762000	764000	742000	742000	39320
2000-02-22	746000	752000	714000	722000	39250
2000-02-23	748000	762000	738000	762000	57340
2000-02-24	782000	804000	776000	786000	184230
2000-02-25	816000	850000	808000	846000	259920
2000-02-28	864000	914000	860000	904000	230055
2000-02-29	894000	894000	876000	886000	81700
2000-03-01	892000	894000	870000	880000	59815
2000-03-02	882000	902000	872000	882000	71125
2000-03-03	862000	864000	832000	834000	52830
2000-03-06	864000	866000	802000	816000	55325
2000-03-07	810000	820000	784000	820000	65705
2000-03-08	812000	840000	806000	828000	69110
2000-03-09	826000	838000	796000	816000	56055
2000-03-10	806000	812000	764000	776000	123105
2000-03-13	736000	742000	676000	678000	110545
2000-03-14	728000	778000	726000	778000	118175
2000-03-15	808000	876000	802000	850000	147395
2000-03-16	842000	866000	824000	860000	97005
2000-03-17	874000	882000	846000	880000	84335
2000-03-21	860000	878000	854000	870000	53360
2000-03-22	864000	876000	850000	876000	49505
2000-03-23	872000	890000	860000	868000	59985
2000-03-24	870000	878000	824000	836000	64335
2000-03-27	836000	850000	802000	808000	66360
2000-03-28	828000	862000	824000	860000	49960
2000-03-29	870000	900000	862000	874000	90585
2000-03-30	890000	910000	862000	862000	106975
2000-03-31	842000	872000	834000	842000	72730
2000-04-03	830000	870000	824000	848000	97800
2000-04-04	838000	844000	812000	814000	75535
2000-04-05	776000	804000	760000	786000	130730
2000-04-06	790000	794000	746000	750000	99135
2000-04-07	770000	804000	764000	798000	77030
2000-04-10	832000	850000	830000	832000	90090
2000-04-11	828000	838000	802000	802000	46460
2000-04-12	784000	796000	776000	796000	55770
2000-04-13	766000	778000	724000	742000	134565
2000-04-14	740000	772000	732000	750000	152595
2000-04-17	690000	704000	650000	674000	182690
2000-04-18	732000	742000	698000	722000	119025
2000-04-19	752000	764000	734000	740000	67285
2000-04-20	680000	716000	678000	698000	216495
2000-04-21	700000	732000	698000	710000	88490
2000-04-24	710000	738000	704000	704000	44860
2000-04-25	704000	722000	700000	718000	45940
2000-04-26	728000	742000	722000	732000	49900
2000-04-27	736000	738000	700000	700000	48095
2000-04-28	720000	744000	704000	722000	62045
2000-05-01	742000	762000	732000	760000	57730
2000-05-02	772000	784000	768000	768000	71330
2000-05-08	788000	792000	772000	772000	39660
2000-05-09	752000	762000	730000	738000	75655
2000-05-10	698000	718000	688000	716000	114975
2000-05-11	676000	686000	658000	668000	148965
2000-05-12	684000	692000	664000	668000	100790
2000-05-15	668000	674000	662000	668000	58475
2000-05-16	678000	700000	666000	692000	73170
2000-05-17	720000	724000	698000	706000	94470
2000-05-18	676000	686000	670000	670000	69090
2000-05-19	644000	652000	608000	624000	137215
2000-05-22	578000	592000	560000	568000	156835
2000-05-23	574000	586000	544000	560000	127585
2000-05-24	538000	558000	518000	534000	179400
2000-05-25	574000	588000	566000	566000	122565
2000-05-26	574000	602000	572000	594000	73315
2000-05-29	586000	592000	564000	570000	66605
2000-05-30	566000	568000	532000	540000	94990
2000-05-31	556000	560000	540000	556000	83275
2000-06-01	546000	558000	530000	558000	119080
2000-06-02	574000	588000	568000	576000	68755
2000-06-05	610000	618000	602000	608000	105620
2000-06-06	608000	618000	600000	616000	66490
2000-06-07	610000	626000	610000	624000	48480
2000-06-08	626000	634000	612000	618000	48990
2000-06-09	612000	636000	612000	632000	50140
2000-06-12	638000	650000	634000	642000	45895
2000-06-13	622000	624000	608000	614000	46025
2000-06-14	604000	610000	590000	604000	70470
2000-06-15	586000	592000	570000	570000	54090
2000-06-16	562000	592000	558000	564000	70325
2000-06-19	566000	578000	564000	574000	43770
2000-06-20	580000	598000	580000	596000	36195
2000-06-21	604000	620000	602000	616000	49320
2000-06-22	626000	630000	608000	608000	63790
2000-06-23	598000	612000	594000	602000	49165
2000-06-26	594000	600000	578000	580000	40330
2000-06-27	586000	608000	580000	604000	37440
2000-06-28	594000	600000	586000	590000	33695
2000-06-29	598000	606000	594000	602000	36745
2000-06-30	576000	582000	568000	574000	69730
2000-07-03	584000	602000	582000	586000	52455
2000-07-04	594000	594000	584000	584000	20340
2000-07-05	592000	594000	584000	594000	27040
2000-07-06	586000	596000	584000	596000	21240
2000-07-07	600000	600000	592000	596000	22935
2000-07-10	614000	618000	610000	614000	47545
2000-07-11	620000	626000	614000	624000	44580
2000-07-12	630000	636000	624000	634000	43455
2000-07-13	634000	638000	610000	610000	54695
2000-07-14	620000	634000	616000	632000	35420
2000-07-17	632000	634000	616000	620000	21245
2000-07-18	612000	614000	596000	602000	46575
2000-07-19	592000	604000	590000	600000	24815
2000-07-21	610000	610000	584000	588000	24590
2000-07-24	580000	584000	572000	582000	30715
2000-07-25	576000	580000	564000	576000	39705
2000-07-26	584000	602000	582000	594000	46645
2000-07-27	586000	592000	580000	586000	25905
2000-07-28	572000	580000	564000	568000	43100
2000-07-31	550000	554000	544000	550000	53855
2000-08-01	556000	570000	554000	564000	68450
2000-08-02	566000	570000	560000	564000	25245
2000-08-03	562000	562000	550000	554000	31100
2000-08-04	556000	572000	552000	560000	52115
2000-08-07	556000	562000	544000	562000	38710
2000-08-08	568000	570000	556000	560000	48645
2000-08-09	568000	570000	552000	560000	43135
2000-08-10	560000	562000	554000	560000	19990
2000-08-11	560000	564000	554000	560000	46460
2000-08-14	560000	560000	550000	556000	21945
2000-08-15	556000	558000	546000	552000	25550
2000-08-16	550000	550000	542000	542000	24290
2000-08-17	534000	536000	522000	528000	66290
2000-08-18	530000	540000	526000	538000	54895
2000-08-21	530000	532000	514000	518000	55315
2000-08-22	516000	526000	502000	526000	77615
2000-08-23	526000	544000	526000	540000	59490
2000-08-24	544000	550000	536000	538000	42925
2000-08-25	548000	560000	542000	554000	46860
2000-08-28	560000	574000	550000	572000	37590
2000-08-29	580000	590000	574000	584000	68975
2000-08-30	588000	590000	572000	574000	30820
2000-08-31	566000	566000	558000	564000	38855
2000-09-01	568000	570000	556000	560000	37410
2000-09-04	580000	582000	572000	578000	29350
2000-09-05	574000	588000	570000	584000	44745
2000-09-06	584000	598000	582000	596000	75030
2000-09-07	610000	620000	606000	616000	121255
2000-09-08	634000	650000	632000	650000	106955
2000-09-11	644000	648000	630000	630000	59485
2000-09-12	638000	654000	632000	646000	68285
2000-09-13	650000	666000	648000	662000	87495
2000-09-14	666000	676000	664000	666000	66500
2000-09-18	656000	660000	642000	654000	43090
2000-09-19	634000	656000	630000	656000	58520
2000-09-20	658000	680000	656000	678000	64660
2000-09-21	672000	676000	662000	670000	62675
2000-09-22	644000	648000	610000	614000	84870
2000-09-25	624000	636000	602000	608000	76945
2000-09-26	622000	630000	612000	614000	52360
2000-09-27	610000	634000	604000	626000	71245
2000-09-28	616000	620000	598000	602000	66195
2000-09-29	612000	626000	610000	620000	62005
2000-10-02	614000	640000	604000	640000	62985
2000-10-03	640000	654000	632000	648000	63350
2000-10-04	648000	662000	640000	662000	53770
2000-10-05	662000	662000	650000	656000	34570
2000-10-06	654000	670000	652000	666000	58485
2000-10-10	662000	674000	640000	644000	63055
2000-10-11	634000	636000	622000	632000	38235
2000-10-12	616000	626000	616000	616000	46310
2000-10-13	606000	618000	598000	618000	50775
2000-10-16	630000	644000	626000	634000	48205
2000-10-17	634000	636000	622000	636000	39270
2000-10-18	620000	628000	616000	624000	52295
2000-10-19	620000	626000	600000	602000	51060
2000-10-20	622000	636000	620000	624000	54430
2000-10-23	620000	620000	604000	606000	46045
2000-10-24	606000	610000	582000	586000	57465
2000-10-25	594000	594000	586000	594000	40650
2000-10-26	586000	590000	572000	590000	49895
2000-10-27	590000	598000	572000	572000	30045
2000-10-30	572000	574000	554000	562000	42910
2000-10-31	554000	558000	530000	538000	58380
2000-11-01	562000	590000	560000	584000	63990
2000-11-02	578000	594000	570000	592000	36080
2000-11-06	600000	608000	594000	606000	46680
2000-11-07	610000	620000	606000	620000	48595
2000-11-08	612000	620000	606000	616000	45110
2000-11-09	610000	610000	594000	598000	38400
2000-11-10	590000	592000	576000	580000	41940
2000-11-13	556000	578000	550000	572000	62800
2000-11-14	574000	588000	564000	568000	50705
2000-11-15	590000	610000	588000	602000	92605
2000-11-16	606000	608000	588000	590000	53725
2000-11-17	580000	600000	580000	598000	51000
2000-11-20	592000	596000	584000	588000	37365
2000-11-21	580000	580000	570000	580000	40785
2000-11-22	582000	584000	554000	554000	58755
2000-11-24	538000	546000	522000	526000	87800
2000-11-27	534000	542000	526000	532000	74255
2000-11-28	530000	578000	522000	564000	90015
2000-11-29	552000	556000	546000	548000	58075
2000-11-30	536000	574000	536000	566000	35810
2000-12-01	558000	584000	554000	570000	60560
2000-12-04	570000	580000	554000	580000	43620
2000-12-05	588000	588000	570000	572000	25690
2000-12-06	596000	600000	590000	596000	53665
2000-12-07	590000	590000	580000	586000	27825
2000-12-08	578000	598000	576000	588000	104940
2000-12-11	580000	588000	570000	570000	48850
2000-12-12	570000	576000	560000	568000	50660
2000-12-13	556000	558000	542000	550000	81840
2000-12-14	536000	542000	526000	526000	71095
2000-12-15	516000	524000	512000	516000	78110
2000-12-18	516000	524000	512000	520000	43115
2000-12-19	518000	518000	502000	502000	71975
2000-12-20	470000	472000	448000	450000	145245
2000-12-21	418000	430000	400000	420000	142845
2000-12-22	428000	438000	424000	436000	79735
2000-12-25	454000	460000	440000	448000	43530
2000-12-26	448000	450000	440000	444000	34630
2000-12-27	436000	438000	420000	426000	41920
2000-12-28	410000	414000	392000	402000	94815
2000-12-29	394000	398000	390000	394000	40715
2001-01-04	414000	422000	406000	412000	82355
2001-01-05	412000	440000	412000	440000	110875
2001-01-09	432000	432000	412000	418000	95670
2001-01-10	410000	416000	398000	404000	84125
2001-01-11	400000	404000	382000	390000	112830
2001-01-12	398000	404000	390000	394000	80300
2001-01-15	386000	400000	382000	392000	64905
2001-01-16	402000	410000	398000	410000	40965
2001-01-17	426000	440000	420000	438000	78795
2001-01-18	478000	514000	472000	486000	213810
2001-01-19	476000	492000	462000	476000	102280
2001-01-22	476000	492000	470000	492000	72635
2001-01-23	478000	480000	468000	472000	45615
2001-01-24	474000	476000	460000	460000	33765
2001-01-25	460000	468000	450000	462000	51095
2001-01-26	460000	466000	460000	460000	33610
2001-01-29	454000	462000	452000	454000	42685
2001-01-30	456000	458000	450000	452000	39125
2001-01-31	456000	456000	446000	448000	54010
2001-02-01	446000	446000	434000	444000	76530
2001-02-02	432000	438000	430000	432000	55275
2001-02-05	424000	430000	420000	426000	124220
2001-02-06	426000	438000	422000	434000	181850
2001-02-07	440000	460000	438000	456000	121420
2001-02-08	452000	452000	432000	432000	64005
2001-02-09	424000	436000	422000	434000	64200
2001-02-13	432000	432000	424000	428000	34740
2001-02-14	426000	426000	416000	426000	36970
2001-02-15	418000	422000	416000	422000	24840
2001-02-16	420000	422000	414000	414000	26420
2001-02-19	404000	406000	396000	400000	55095
2001-02-20	398000	412000	396000	408000	36230
2001-02-21	414000	420000	404000	408000	83570
2001-02-22	402000	412000	396000	412000	116690
2001-02-23	406000	412000	404000	406000	74280
2001-02-26	410000	414000	408000	412000	46855
2001-02-27	410000	412000	400000	404000	78290
2001-02-28	400000	412000	398000	406000	80190
2001-03-01	402000	408000	402000	406000	49570
2001-03-02	402000	404000	386000	388000	127525
2001-03-05	374000	376000	360000	370000	115790
2001-03-06	376000	388000	376000	388000	62860
2001-03-07	406000	406000	394000	402000	76125
2001-03-08	400000	400000	390000	396000	47945
2001-03-09	404000	406000	400000	404000	94780
2001-03-12	396000	402000	394000	396000	39610
2001-03-13	378000	384000	366000	376000	94305
2001-03-14	382000	384000	372000	374000	55775
2001-03-15	364000	386000	362000	380000	76050
2001-03-16	390000	404000	386000	400000	80565
2001-03-19	398000	408000	396000	400000	55115
2001-03-21	400000	436000	400000	436000	114445
2001-03-22	428000	438000	418000	438000	203670
2001-03-23	436000	446000	432000	444000	119705
2001-03-26	458000	466000	448000	466000	127720
2001-03-27	462000	476000	454000	470000	93695
2001-03-28	478000	486000	474000	478000	97990
2001-03-29	462000	464000	446000	446000	65950
2001-03-30	454000	468000	436000	436000	105375
2001-04-02	438000	452000	422000	442000	88495
2001-04-03	444000	452000	436000	444000	43445
2001-04-04	436000	458000	434000	458000	71610
2001-04-05	468000	470000	460000	460000	76710
2001-04-06	476000	476000	460000	466000	67355
2001-04-09	466000	472000	454000	466000	76650
2001-04-10	464000	470000	458000	460000	52810
2001-04-11	472000	480000	470000	478000	95860
2001-04-12	482000	484000	474000	482000	69545
2001-04-13	486000	498000	482000	488000	135745
2001-04-16	482000	486000	478000	478000	37110
2001-04-17	472000	476000	460000	462000	64310
2001-04-18	468000	488000	468000	486000	61125
2001-04-19	504000	516000	500000	510000	129395
2001-04-20	502000	508000	488000	492000	64980
2001-04-23	494000	508000	494000	504000	40590
2001-04-24	468000	484000	464000	480000	152215
2001-04-25	484000	500000	482000	500000	80350
2001-04-26	508000	510000	500000	504000	50240
2001-04-27	508000	510000	500000	508000	51735
2001-05-01	516000	526000	514000	526000	58035
2001-05-02	534000	540000	528000	538000	75300
2001-05-07	546000	580000	546000	578000	143200
2001-05-08	556000	560000	542000	546000	103315
2001-05-09	530000	532000	522000	528000	113560
2001-05-10	512000	524000	506000	512000	121575
2001-05-11	504000	508000	500000	500000	75615
2001-05-14	496000	504000	492000	496000	82190
2001-05-15	496000	518000	494000	518000	93120
2001-05-16	510000	514000	498000	504000	63570
2001-05-17	512000	518000	502000	516000	70325
2001-05-18	514000	516000	498000	498000	64580
2001-05-21	502000	508000	498000	500000	41045
2001-05-22	508000	510000	498000	498000	47540
2001-05-23	496000	506000	486000	488000	68280
2001-05-24	480000	490000	480000	486000	72975
2001-05-25	490000	492000	482000	484000	42065
2001-05-28	476000	484000	476000	478000	26405
2001-05-29	480000	488000	478000	480000	42075
2001-05-30	478000	480000	464000	466000	58555
2001-05-31	456000	466000	452000	456000	96410
2001-06-01	466000	472000	456000	456000	51995
2001-06-04	464000	464000	456000	458000	41420
2001-06-05	460000	464000	456000	456000	38905
2001-06-06	458000	462000	450000	450000	42995
2001-06-07	430000	450000	430000	450000	104545
2001-06-08	458000	466000	450000	450000	129860
2001-06-11	450000	454000	438000	438000	58550
2001-06-12	430000	436000	410000	412000	137000
2001-06-13	416000	418000	406000	410000	104175
2001-06-14	408000	422000	406000	418000	48930
2001-06-15	410000	412000	402000	412000	84285
2001-06-18	404000	406000	396000	398000	52805
2001-06-19	404000	412000	400000	400000	63565
2001-06-20	404000	406000	392000	404000	88295
2001-06-21	408000	418000	404000	418000	63900
2001-06-22	420000	432000	416000	432000	83355
2001-06-25	432000	432000	414000	422000	59730
2001-06-26	422000	430000	418000	426000	56805
2001-06-27	426000	428000	420000	420000	36660
2001-06-28	412000	420000	404000	412000	65195
2001-06-29	428000	434000	422000	434000	89225
2001-07-02	432000	436000	418000	420000	66445
2001-07-03	430000	438000	426000	430000	45740
2001-07-04	428000	430000	412000	420000	47360
2001-07-05	416000	420000	410000	414000	37605
2001-07-06	408000	420000	406000	418000	44060
2001-07-09	410000	412000	402000	408000	48895
2001-07-10	406000	418000	402000	414000	57955
2001-07-11	406000	412000	404000	406000	36975
2001-07-12	414000	418000	408000	414000	56695
2001-07-13	414000	416000	402000	408000	45015
2001-07-16	408000	408000	398000	400000	47315
2001-07-17	392000	394000	384000	390000	71115
2001-07-18	380000	382000	360000	368000	145225
2001-07-19	362000	368000	362000	362000	64460
2001-07-23	368000	368000	340000	342000	121290
2001-07-24	344000	354000	334000	354000	117455
2001-07-25	350000	372000	348000	362000	103625
2001-07-26	364000	366000	354000	358000	66520
2001-07-27	352000	358000	336000	342000	112345
2001-07-30	348000	348000	334000	336000	41780
2001-07-31	342000	350000	340000	346000	57925
2001-08-01	352000	370000	348000	368000	102305
2001-08-02	378000	380000	370000	378000	81665
2001-08-03	376000	378000	366000	368000	66735
2001-08-06	362000	368000	358000	364000	49100
2001-08-07	358000	364000	352000	362000	63310
2001-08-08	364000	378000	364000	370000	65530
2001-08-09	356000	358000	344000	344000	77290
2001-08-10	346000	358000	344000	344000	58435
2001-08-13	342000	346000	332000	338000	57880
2001-08-14	340000	360000	340000	356000	59030
2001-08-15	354000	356000	346000	352000	25950
2001-08-16	340000	346000	338000	340000	40575
2001-08-17	340000	340000	332000	334000	40350
2001-08-20	326000	332000	324000	326000	46260
2001-08-21	328000	330000	320000	324000	58065
2001-08-22	318000	324000	316000	320000	47325
2001-08-23	316000	318000	296000	298000	84395
2001-08-24	308000	312000	300000	310000	86590
2001-08-27	326000	326000	314000	320000	47875
2001-08-28	322000	322000	302000	304000	70155
2001-08-29	298000	300000	290000	292000	84165
2001-08-30	290000	296000	286000	290000	66090
2001-08-31	284000	294000	280000	292000	70470
2001-09-03	288000	290000	274000	276000	71330
2001-09-04	278000	290000	266000	290000	126835
2001-09-05	278000	284000	270000	276000	95310
2001-09-06	276000	280000	264000	270000	93650
2001-09-07	258000	262000	252000	258000	101850
2001-09-10	252000	254000	242000	244000	88725
2001-09-11	248000	252000	242000	248000	121230
2001-09-12	228000	232000	228000	228000	50180
2001-09-13	236000	248000	234000	248000	106900
2001-09-14	262000	268000	258000	268000	74835
2001-09-17	266000	282000	254000	276000	153710
2001-09-18	296000	310000	288000	294000	220915
2001-09-19	308000	324000	306000	310000	151945
2001-09-20	296000	306000	294000	300000	146420
2001-09-21	286000	290000	278000	286000	91625
2001-09-25	308000	314000	302000	306000	97655
2001-09-26	300000	304000	292000	296000	87315
2001-09-27	300000	310000	298000	310000	78115
2001-09-28	322000	326000	314000	322000	130700
2001-10-01	328000	344000	322000	344000	138660
2001-10-02	340000	344000	334000	344000	114905
2001-10-03	342000	344000	320000	322000	104900
2001-10-04	338000	340000	330000	338000	93795
2001-10-05	338000	358000	332000	352000	136880
2001-10-09	340000	344000	326000	326000	115300
2001-10-10	320000	324000	306000	308000	141630
2001-10-11	320000	334000	314000	334000	112005
2001-10-12	352000	354000	342000	348000	164260
2001-10-15	338000	340000	332000	332000	66420
2001-10-16	332000	338000	328000	334000	60450
2001-10-17	340000	342000	336000	342000	73800
2001-10-18	332000	336000	330000	334000	49015
2001-10-19	328000	340000	326000	336000	60905
2001-10-22	336000	340000	330000	332000	44295
2001-10-23	340000	344000	336000	340000	95375
2001-10-24	342000	352000	342000	348000	116595
2001-10-25	352000	364000	336000	342000	157745
2001-10-26	348000	354000	346000	348000	110080
2001-10-29	348000	350000	344000	348000	62575
2001-10-30	342000	346000	336000	340000	78830
2001-10-31	334000	338000	330000	332000	98215
2001-11-01	334000	338000	332000	336000	54870
2001-11-02	348000	350000	328000	332000	85065
2001-11-05	330000	332000	316000	318000	93005
2001-11-06	320000	326000	316000	318000	85935
2001-11-07	318000	320000	306000	306000	82135
2001-11-08	300000	302000	292000	298000	110480
2001-11-09	294000	298000	290000	292000	97355
2001-11-12	292000	300000	288000	298000	70805
2001-11-13	298000	312000	298000	312000	81520
2001-11-14	318000	322000	316000	316000	77800
2001-11-15	318000	332000	316000	330000	68575
2001-11-16	338000	348000	334000	342000	147595
2001-11-19	348000	354000	344000	350000	90070
2001-11-20	348000	350000	334000	336000	79960
2001-11-21	332000	336000	318000	324000	123595
2001-11-22	324000	336000	322000	332000	78225
2001-11-26	344000	354000	340000	354000	102860
2001-11-27	348000	352000	344000	344000	80400
2001-11-28	338000	340000	328000	334000	84250
2001-11-29	328000	338000	320000	322000	118195
2001-11-30	328000	330000	316000	322000	151190
2001-12-03	326000	326000	314000	316000	58550
2001-12-04	312000	322000	308000	320000	78890
2001-12-05	326000	332000	324000	332000	74875
2001-12-06	344000	346000	336000	342000	83980
2001-12-07	336000	336000	328000	332000	48815
2001-12-10	328000	328000	316000	316000	53755
2001-12-11	310000	322000	308000	314000	64610
2001-12-12	308000	318000	308000	312000	85040
2001-12-13	304000	306000	294000	298000	87590
2001-12-14	290000	294000	286000	288000	178245
2001-12-17	288000	302000	278000	294000	94195
2001-12-18	300000	302000	282000	288000	81615
2001-12-19	284000	296000	280000	292000	73255
2001-12-20	298000	310000	294000	310000	96770
2001-12-21	298000	302000	296000	302000	58930
2001-12-25	304000	304000	294000	296000	39580
2001-12-26	292000	294000	286000	286000	39880
2001-12-27	290000	302000	290000	298000	37275
2001-12-28	304000	308000	302000	308000	38830
2002-01-04	318000	320000	314000	320000	30060
2002-01-07	320000	322000	316000	320000	38260
2002-01-08	318000	320000	306000	310000	76080
2002-01-09	306000	306000	298000	300000	56900
2002-01-10	296000	300000	290000	292000	62425
2002-01-11	290000	294000	284000	284000	91130
2002-01-15	276000	278000	274000	274000	81795
2002-01-16	274000	284000	274000	282000	63225
2002-01-17	278000	282000	274000	276000	80350
2002-01-18	280000	286000	280000	286000	84265
2002-01-21	282000	284000	276000	280000	60020
2002-01-22	280000	282000	272000	272000	66260
2002-01-23	270000	272000	262000	264000	74030
2002-01-24	268000	274000	264000	270000	60450
2002-01-25	278000	278000	268000	276000	69605
2002-01-28	296000	298000	290000	296000	121835
2002-01-29	292000	292000	284000	286000	76350
2002-01-30	278000	284000	276000	276000	92215
2002-01-31	282000	286000	280000	280000	73775
2002-02-01	280000	282000	274000	278000	65215
2002-02-04	282000	284000	278000	282000	48890
2002-02-05	274000	276000	270000	272000	75285
2002-02-06	268000	272000	262000	264000	71375
2002-02-07	266000	278000	266000	278000	69880
2002-02-08	274000	278000	268000	272000	109660
2002-02-12	280000	286000	280000	282000	79180
2002-02-13	278000	288000	276000	288000	79365
2002-02-14	290000	296000	286000	288000	85465
2002-02-15	288000	290000	280000	284000	79320
2002-02-18	280000	280000	276000	278000	39395
2002-02-19	280000	282000	272000	278000	51580
2002-02-20	270000	278000	270000	274000	60675
2002-02-21	278000	284000	276000	284000	65120
2002-02-22	280000	282000	274000	278000	76210
2002-02-25	282000	284000	278000	278000	47315
2002-02-26	286000	290000	280000	282000	79450
2002-02-27	278000	284000	276000	278000	95280
2002-02-28	282000	286000	276000	278000	126665
2002-03-01	284000	286000	278000	284000	74445
2002-03-04	294000	306000	292000	304000	154045
2002-03-05	312000	316000	308000	308000	124350
2002-03-06	308000	320000	306000	318000	124430
2002-03-07	330000	344000	328000	344000	188590
2002-03-08	350000	354000	342000	352000	257275
2002-03-11	356000	358000	344000	352000	106785
2002-03-12	348000	348000	334000	334000	99705
2002-03-13	328000	336000	326000	326000	103860
2002-03-14	324000	334000	324000	334000	83890
2002-03-15	342000	344000	338000	344000	82075
2002-03-18	344000	348000	334000	334000	105880
2002-03-19	340000	342000	336000	342000	91265
2002-03-20	344000	348000	340000	342000	128845
2002-03-22	340000	340000	332000	334000	112235
2002-03-25	334000	342000	332000	336000	246885
2002-03-26	341000	351000	329000	333000	208021
2002-03-27	338000	351000	336000	347000	139550
2002-03-28	352000	364000	343000	360000	153194
2002-03-29	365000	369000	352000	352000	102695
2002-04-01	357000	360000	347000	347000	104449
2002-04-02	352000	358000	351000	357000	67572
2002-04-03	352000	369000	349000	367000	139874
2002-04-04	369000	373000	363000	372000	119671
2002-04-05	365000	366000	357000	358000	81233
2002-04-08	359000	369000	355000	367000	77402
2002-04-09	366000	366000	353000	353000	61930
2002-04-10	338000	342000	332000	334000	143115
2002-04-11	336000	338000	331000	332000	75891
2002-04-12	322000	324000	313000	316000	147905
2002-04-15	316000	321000	313000	320000	64452
2002-04-16	324000	329000	323000	329000	77327
2002-04-17	337000	338000	331000	334000	79001
2002-04-18	335000	342000	334000	338000	77379
2002-04-19	334000	335000	328000	335000	48452
2002-04-22	337000	343000	337000	342000	49006
2002-04-23	333000	340000	329000	335000	60427
2002-04-24	334000	338000	330000	331000	43539
2002-04-25	334000	338000	332000	334000	35147
2002-04-26	331000	334000	327000	328000	60590
2002-04-30	333000	333000	325000	325000	31619
2002-05-01	329000	333000	327000	328000	30663
2002-05-02	327000	329000	326000	327000	27050
2002-05-07	318000	320000	313000	318000	69210
2002-05-08	313000	319000	313000	315000	31006
2002-05-09	330000	332000	323000	325000	55601
2002-05-10	320000	321000	312000	315000	79678
2002-05-13	311000	312000	307000	309000	49562
2002-05-14	310000	311000	301000	302000	67461
2002-05-15	312000	314000	309000	310000	67685
2002-05-16	313000	323000	312000	322000	62508
2002-05-17	322000	331000	321000	321000	109239
2002-05-20	328000	334000	326000	332000	87381
2002-05-21	331000	339000	330000	338000	84689
2002-05-22	336000	347000	334000	344000	89789
2002-05-23	346000	348000	341000	344000	69812
2002-05-24	339000	340000	330000	331000	124111
2002-05-27	326000	328000	323000	323000	72543
2002-05-28	325000	325000	319000	321000	66100
2002-05-29	322000	325000	321000	324000	62711
2002-05-30	320000	323000	316000	323000	66784
2002-05-31	324000	336000	323000	336000	103775
2002-06-03	331000	337000	329000	331000	74375
2002-06-04	334000	335000	322000	325000	54688
2002-06-05	322000	329000	320000	323000	67074
2002-06-06	325000	333000	324000	326000	92034
2002-06-07	322000	323000	318000	321000	70145
2002-06-10	316000	317000	312000	313000	60149
2002-06-11	313000	316000	313000	314000	51505
2002-06-12	314000	314000	310000	314000	43029
2002-06-13	316000	318000	308000	308000	51263
2002-06-14	306000	310000	301000	301000	157819
2002-06-17	295000	296000	287000	290000	108038
2002-06-18	295000	301000	294000	300000	79465
2002-06-19	296000	298000	288000	294000	104970
2002-06-20	290000	295000	285000	294000	76657
2002-06-21	285000	287000	284000	284000	51372
2002-06-24	277000	286000	274000	285000	55004
2002-06-25	284000	295000	284000	292000	63279
2002-06-26	282000	284000	276000	277000	76560
2002-06-27	281000	285000	278000	281000	66571
2002-06-28	288000	295000	287000	295000	78797
2002-07-01	298000	303000	295000	297000	52600
2002-07-02	289000	292000	281000	287000	97870
2002-07-03	281000	287000	279000	284000	107602
2002-07-04	285000	287000	275000	276000	106416
2002-07-05	280000	283000	278000	281000	73217
2002-07-08	289000	290000	271000	271000	111197
2002-07-09	275000	277000	266000	277000	103039
2002-07-10	273000	279000	270000	271000	89375
2002-07-11	267000	268000	263000	264000	74613
2002-07-12	267000	270000	263000	264000	59947
2002-07-15	262000	262000	256000	256000	53962
2002-07-16	250000	254000	244000	244000	114328
2002-07-17	248000	255000	245000	253000	88181
2002-07-18	260000	268000	258000	264000	97778
2002-07-19	260000	262000	258000	260000	63692
2002-07-22	251000	264000	250000	264000	66447
2002-07-23	259000	262000	256000	260000	68751
2002-07-24	255000	257000	251000	251000	60366
2002-07-25	256000	258000	250000	250000	51177
2002-07-26	251000	252000	244000	247000	85129
2002-07-29	249000	270000	248000	269000	103195
2002-07-30	273000	275000	266000	273000	91838
2002-07-31	270000	271000	266000	267000	45321
2002-08-01	268000	269000	266000	267000	59351
2002-08-02	267000	279000	263000	275000	103012
2002-08-05	273000	285000	272000	281000	78932
2002-08-06	282000	283000	275000	276000	79213
2002-08-07	280000	283000	278000	280000	52100
2002-08-08	278000	281000	276000	276000	42991
2002-08-09	283000	285000	280000	284000	51567
2002-08-12	284000	286000	281000	281000	37219
2002-08-13	279000	279000	275000	276000	33062
2002-08-14	276000	276000	270000	272000	41767
2002-08-15	273000	276000	266000	267000	48342
2002-08-16	267000	270000	259000	264000	42914
2002-08-19	263000	263000	253000	253000	50380
2002-08-20	257000	259000	253000	256000	57359
2002-08-21	254000	258000	253000	258000	43669
2002-08-22	256000	261000	250000	259000	83342
2002-08-23	269000	269000	260000	262000	91104
2002-08-26	260000	272000	259000	269000	68709
2002-08-27	266000	266000	260000	260000	59583
2002-08-28	264000	264000	257000	257000	45580
2002-08-29	254000	256000	251000	251000	53128
2002-08-30	253000	255000	248000	252000	53268
2002-09-02	252000	253000	248000	251000	54575
2002-09-03	249000	250000	244000	244000	60554
2002-09-04	238000	240000	231000	232000	79289
2002-09-05	239000	245000	236000	241000	89158
2002-09-06	233000	235000	231000	232000	47401
2002-09-09	236000	239000	233000	233000	47153
2002-09-10	234000	238000	233000	234000	67125
2002-09-11	237000	238000	232000	233000	52660
2002-09-12	230000	233000	227000	230000	68399
2002-09-13	222000	226000	219000	220000	152092
2002-09-17	221000	227000	217000	226000	79283
2002-09-18	218000	225000	213000	222000	69930
2002-09-19	237000	239000	225000	225000	100221
2002-09-20	217000	225000	214000	215000	72822
2002-09-24	207000	209000	202000	208000	77359
2002-09-25	202000	209000	201000	206000	106090
2002-09-26	210000	213000	206000	206000	101066
2002-09-27	214000	219000	211000	216000	112065
2002-09-30	211000	214000	206000	208000	47058
2002-10-01	204000	210000	203000	206000	53825
2002-10-02	210000	211000	205000	205000	48091
2002-10-03	210000	213000	208000	212000	82887
2002-10-04	220000	235000	219000	235000	163229
2002-10-07	231000	233000	222000	225000	83931
2002-10-08	227000	232000	224000	229000	59993
2002-10-09	226000	227000	222000	224000	52887
2002-10-10	220000	222000	213000	218000	53741
2002-10-11	222000	227000	221000	225000	49135
2002-10-15	234000	240000	232000	238000	58671
2002-10-16	244000	245000	238000	242000	60232
2002-10-17	241000	245000	235000	236000	47424
2002-10-18	243000	243000	237000	237000	38226
2002-10-21	239000	241000	234000	235000	36283
2002-10-22	236000	236000	218000	218000	52237
2002-10-23	219000	227000	215000	226000	62081
2002-10-24	226000	227000	214000	219000	33429
2002-10-25	222000	231000	222000	230000	52673
2002-10-28	228000	231000	224000	231000	44791
2002-10-29	230000	235000	224000	224000	37296
2002-10-30	222000	231000	220000	228000	48338
2002-10-31	232000	232000	222000	226000	38900
2002-11-01	231000	237000	227000	236000	68250
2002-11-05	244000	251000	242000	245000	102262
2002-11-06	244000	251000	241000	241000	88949
2002-11-07	241000	245000	236000	241000	43581
2002-11-08	234000	237000	227000	227000	61568
2002-11-11	221000	223000	213000	213000	55815
2002-11-12	209000	223000	208000	221000	57525
2002-11-13	225000	228000	220000	223000	61560
2002-11-14	219000	229000	218000	221000	46946
2002-11-15	229000	232000	225000	227000	43474
2002-11-18	228000	228000	215000	218000	47301
2002-11-19	215000	218000	208000	212000	58999
2002-11-20	212000	222000	210000	216000	46572
2002-11-21	222000	226000	220000	224000	59336
2002-11-22	233000	237000	231000	233000	94085
2002-11-25	236000	244000	234000	244000	83124
2002-11-26	248000	250000	237000	240000	67701
2002-11-27	236000	241000	235000	237000	53582
2002-11-28	243000	254000	241000	252000	89654
2002-11-29	252000	257000	247000	248000	75882
2002-12-02	249000	251000	244000	244000	42532
2002-12-03	251000	256000	249000	253000	64356
2002-12-04	245000	250000	244000	246000	50263
2002-12-05	245000	247000	241000	242000	43824
2002-12-06	240000	242000	238000	240000	41917
2002-12-09	237000	238000	233000	235000	50246
2002-12-10	231000	239000	230000	234000	44623
2002-12-11	236000	237000	232000	234000	37429
2002-12-12	232000	235000	231000	231000	35327
2002-12-13	229000	231000	224000	224000	120229
2002-12-16	224000	227000	219000	221000	57627
2002-12-17	225000	227000	221000	222000	51476
2002-12-18	223000	225000	219000	219000	52470
2002-12-19	220000	225000	218000	223000	62431
2002-12-20	223000	226000	218000	218000	54986
2002-12-24	221000	223000	218000	222000	59959
2002-12-25	220000	222000	217000	217000	35158
2002-12-26	218000	221000	218000	220000	29514
2002-12-27	221000	224000	219000	221000	38524
2002-12-30	218000	220000	217000	219000	17513
2003-01-06	226000	232000	226000	229000	32374
2003-01-07	237000	238000	233000	233000	52812
2003-01-08	230000	235000	229000	234000	59691
2003-01-09	231000	239000	229000	239000	69653
2003-01-10	242000	242000	235000	237000	56362
2003-01-14	241000	252000	240000	248000	121115
2003-01-15	249000	256000	246000	256000	132971
2003-01-16	252000	258000	252000	256000	105361
2003-01-17	253000	258000	252000	254000	83098
2003-01-20	250000	251000	242000	247000	81658
2003-01-21	246000	254000	246000	250000	117044
2003-01-22	251000	251000	244000	246000	50332
2003-01-23	246000	249000	242000	249000	46785
2003-01-24	247000	251000	245000	246000	46639
2003-01-27	244000	247000	237000	237000	49186
2003-01-28	235000	238000	234000	234000	31819
2003-01-29	236000	237000	232000	233000	54663
2003-01-30	236000	240000	235000	238000	41238
2003-01-31	238000	239000	230000	231000	59554
2003-02-03	226000	230000	225000	228000	68685
2003-02-04	232000	237000	232000	234000	60929
2003-02-05	230000	243000	229000	233000	112672
2003-02-06	234000	235000	228000	230000	43147
2003-02-07	228000	233000	227000	232000	40853
2003-02-10	230000	236000	229000	236000	35789
2003-02-12	237000	244000	236000	244000	56853
2003-02-13	243000	247000	242000	246000	63013
2003-02-14	244000	249000	244000	246000	53835
2003-02-17	249000	250000	244000	247000	60496
2003-02-18	246000	250000	242000	242000	67061
2003-02-19	246000	246000	239000	239000	41425
2003-02-20	238000	244000	237000	240000	48110
2003-02-21	241000	245000	236000	239000	39191
2003-02-24	237000	239000	232000	233000	39503
2003-02-25	230000	232000	226000	228000	51876
2003-02-26	226000	230000	223000	227000	42507
2003-02-27	225000	227000	220000	227000	45441
2003-02-28	227000	228000	223000	223000	41961
2003-03-03	224000	228000	222000	228000	39611
2003-03-04	225000	231000	225000	229000	59159
2003-03-05	225000	228000	223000	227000	44911
2003-03-06	225000	231000	222000	223000	43028
2003-03-07	221000	223000	215000	215000	54464
2003-03-10	211000	212000	205000	208000	66018
2003-03-11	204000	211000	201000	201000	78338
2003-03-12	204000	215000	203000	215000	92639
2003-03-13	213000	216000	208000	210000	92497
2003-03-14	222000	223000	216000	219000	153409
2003-03-17	223000	224000	217000	222000	74606
2003-03-18	227000	229000	217000	217000	79382
2003-03-19	216000	221000	212000	220000	55451
2003-03-20	226000	232000	225000	232000	84467
2003-03-24	240000	243000	237000	241000	74995
2003-03-25	233000	236000	225000	227000	60457
2003-03-26	231000	234000	228000	231000	48880
2003-03-27	233000	240000	233000	236000	73287
2003-03-28	237000	240000	236000	238000	33726
2003-03-31	235000	236000	221000	221000	62657
2003-04-01	220000	227000	215000	225000	66782
2003-04-02	229000	230000	224000	230000	47723
2003-04-03	237000	238000	230000	232000	62684
2003-04-04	231000	234000	229000	230000	57433
2003-04-07	234000	242000	234000	240000	81923
2003-04-08	241000	246000	239000	245000	101402
2003-04-09	249000	250000	244000	246000	83410
2003-04-10	242000	246000	241000	243000	70915
2003-04-11	244000	245000	237000	238000	57244
2003-04-14	234000	237000	225000	229000	78101
2003-04-15	234000	242000	234000	237000	77763
2003-04-16	241000	246000	233000	243000	109455
2003-04-17	244000	248000	242000	246000	76947
2003-04-18	247000	248000	242000	242000	45768
2003-04-21	244000	250000	243000	249000	51250
2003-04-22	246000	248000	240000	240000	42472
2003-04-23	244000	250000	241000	246000	84851
2003-04-24	250000	254000	247000	251000	85988
2003-04-25	245000	248000	243000	247000	58662
2003-04-28	241000	246000	241000	241000	40900
2003-04-30	246000	248000	240000	246000	71448
2003-05-01	244000	246000	242000	245000	38574
2003-05-02	246000	251000	243000	250000	50720
2003-05-06	258000	267000	258000	266000	118845
2003-05-07	273000	277000	268000	269000	91199
2003-05-08	267000	268000	259000	262000	66944
2003-05-09	259000	260000	254000	258000	97197
2003-05-12	259000	264000	255000	263000	76299
2003-05-13	267000	270000	262000	265000	79969
2003-05-14	266000	269000	265000	265000	57641
2003-05-15	265000	266000	261000	263000	57446
2003-05-16	263000	264000	255000	256000	40283
2003-05-19	252000	262000	250000	251000	83139
2003-05-20	252000	259000	252000	256000	50406
2003-05-21	257000	258000	251000	254000	36737
2003-05-22	255000	260000	254000	257000	60373
2003-05-23	261000	264000	260000	261000	49431
2003-05-26	261000	264000	256000	256000	44640
2003-05-27	254000	254000	248000	249000	44174
2003-05-28	255000	257000	249000	249000	47566
2003-05-29	253000	257000	252000	256000	47120
2003-05-30	255000	257000	251000	251000	47674
2003-06-02	259000	262000	257000	260000	56140
2003-06-03	260000	268000	259000	265000	107949
2003-06-04	270000	271000	266000	266000	65404
2003-06-05	269000	270000	264000	265000	46873
2003-06-06	268000	272000	266000	272000	47356
2003-06-09	272000	272000	266000	266000	40471
2003-06-10	262000	264000	257000	262000	44457
2003-06-11	264000	267000	262000	263000	41078
2003-06-12	265000	266000	261000	261000	32561
2003-06-13	265000	269000	263000	267000	121204
2003-06-16	264000	265000	260000	262000	39798
2003-06-17	268000	269000	263000	265000	72321
2003-06-18	269000	272000	266000	267000	82372
2003-06-19	268000	272000	266000	271000	59482
2003-06-20	270000	277000	269000	276000	96030
2003-06-23	276000	277000	272000	273000	53321
2003-06-24	269000	270000	263000	264000	59755
2003-06-25	262000	267000	261000	262000	43308
2003-06-26	261000	262000	256000	259000	58537
2003-06-27	261000	262000	257000	262000	66368
2003-06-30	263000	264000	259000	260000	63812
2003-07-01	259000	263000	259000	260000	61747
2003-07-02	264000	278000	263000	277000	140009
2003-07-03	290000	296000	278000	283000	210518
2003-07-04	279000	285000	277000	284000	98697
2003-07-07	281000	291000	281000	289000	70813
2003-07-08	300000	306000	293000	300000	166381
2003-07-09	301000	304000	288000	296000	77688
2003-07-10	296000	303000	293000	293000	90180
2003-07-11	285000	288000	277000	277000	84890
2003-07-14	284000	288000	281000	283000	75285
2003-07-15	286000	290000	282000	284000	76077
2003-07-16	278000	281000	273000	274000	75270
2003-07-17	275000	277000	271000	272000	66499
2003-07-18	275000	279000	274000	275000	60975
2003-07-22	272000	273000	265000	265000	63494
2003-07-23	267000	269000	263000	266000	53531
2003-07-24	270000	271000	266000	267000	60840
2003-07-25	268000	276000	267000	272000	102616
2003-07-28	279000	281000	278000	279000	60632
2003-07-29	283000	284000	280000	280000	59464
2003-07-30	280000	281000	274000	274000	57720
2003-07-31	275000	281000	270000	276000	91782
2003-08-01	280000	281000	275000	275000	65455
2003-08-04	276000	281000	275000	278000	78893
2003-08-05	278000	279000	272000	273000	57271
2003-08-06	269000	280000	267000	278000	84873
2003-08-07	280000	282000	269000	269000	69699
2003-08-08	277000	284000	276000	280000	150809
2003-08-11	281000	287000	280000	286000	88924
2003-08-12	287000	287000	281000	282000	51319
2003-08-13	286000	289000	283000	287000	48312
2003-08-14	287000	294000	286000	293000	58297
2003-08-15	297000	298000	284000	286000	58353
2003-08-18	289000	293000	287000	287000	67297
2003-08-19	295000	304000	292000	303000	118011
2003-08-20	304000	311000	302000	307000	135653
2003-08-21	303000	315000	303000	315000	80059
2003-08-22	315000	320000	310000	311000	104586
2003-08-25	310000	315000	307000	308000	57857
2003-08-26	307000	315000	304000	312000	49954
2003-08-27	312000	318000	309000	310000	66623
2003-08-28	311000	312000	301000	302000	52570
2003-08-29	306000	306000	298000	300000	49123
2003-09-01	302000	307000	299000	307000	47903
2003-09-02	307000	308000	301000	301000	50299
2003-09-03	306000	307000	294000	297000	74457
2003-09-04	300000	300000	291000	292000	53719
2003-09-05	295000	297000	283000	288000	73468
2003-09-08	285000	288000	279000	281000	104640
2003-09-09	287000	291000	282000	288000	127038
2003-09-10	286000	294000	283000	290000	133052
2003-09-11	287000	288000	282000	282000	100536
2003-09-12	286000	291000	284000	288000	137276
2003-09-16	291000	294000	290000	291000	97449
2003-09-17	299000	308000	297000	302000	195042
2003-09-18	297000	302000	293000	299000	97095
2003-09-19	300000	302000	286000	286000	112699
2003-09-22	286000	289000	278000	282000	109986
2003-09-24	279000	287000	279000	285000	76709
2003-09-25	280000	282000	278000	278000	75973
2003-09-26	277000	281000	275000	278000	41642
