package SVN::Hooks::AllowLogChange;

use strict;
use warnings;
use Carp;
use SVN::Hooks;

use Exporter qw/import/;
my $HOOK = 'ALLOW_LOG_CHANGE';
our @EXPORT = ($HOOK);

our $VERSION = $SVN::Hooks::VERSION;

=head1 NAME

SVN::Hooks::AllowLogChange - Allow changes in revision log messages.

=head1 SYNOPSIS

This SVN::Hooks plugin is used to allow revision log changes by some
users.

It's deprecated. You should use SVN::Hooks::AllowPropChange instead.

It's active in the C<pre-revprop-change> hook.

It's configured by the following directive.

=head2 ALLOW_LOG_CHANGE(WHO, ...)

This directive enables the change of revision log messages, which are
mantained in the C<svn:log> revision property.

The optional WHO argument specifies the users that are allowed to make
those changes. If absent, any user can change a log
message. Otherwise, it specifies the allowed users depending on its
type.

=over

=item STRING

Specify a single user by name.

=item REGEXP

Specify the class of users whose names are matched by the
Regexp.

=back

	ALLOW_LOG_CHANGE();
	ALLOW_LOG_CHANGE('jsilva');
	ALLOW_LOG_CHANGE(qr/silva$/);

=cut

my @Valid_Users;

sub ALLOW_LOG_CHANGE {
    my @args = @_;

    foreach my $who (@args) {
	if (not ref $who or ref $who eq 'Regexp') {
	    push @Valid_Users, $who;
	}
	else {
	    croak "$HOOK: invalid argument '$who'\n";
	}
    }

    PRE_REVPROP_CHANGE(\&pre_revprop_change);

    return 1;
}

sub pre_revprop_change {
    my ($svnlook, $rev, $author, $propname, $action) = @_;

    $propname eq 'svn:log'
	or croak "$HOOK: the revision property $propname cannot be changed.\n";

    $action eq 'M'
	or croak "$HOOK: a revision log can only be modified, not added or deleted.\n";

    # If no users are specified, anyone can do it.
    return unless @Valid_Users;

    for my $user (@Valid_Users) {
	return if not ref $user and $author eq $user or $author =~ $user;
    }

    croak "$HOOK: you are not allowed to change a revision log.\n";
}

=head1 AUTHOR

Gustavo Chaves, C<< <gnustavo@cpan.org> >>

=head1 BUGS

Please report any bugs or feature requests to
C<bug-svn-hooks-checkproperty at rt.cpan.org>, or through the web
interface at
L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=SVN-Hooks>.  I will
be notified, and then you'll automatically be notified of progress on
your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc SVN::Hooks

You can also look for information at:

=over 4

=item * RT: CPAN's request tracker

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=SVN-Hooks>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/SVN-Hooks>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/SVN-Hooks>

=item * Search CPAN

L<http://search.cpan.org/dist/SVN-Hooks>

=back

=head1 COPYRIGHT & LICENSE

Copyright 2008-2009 CPqD, all rights reserved.

This program is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut

1; # End of SVN::Hooks::AllowLogChange
