#!/usr/bin/env perl
use strict;
use warnings;
use Test::More;
use Test::Exception;
use lib 'lib';
use Settlers::Game;
use Data::Dumper;
use List::Util 'sum';
use Settlers::Map;
use Mojo::JSON 'encode_json';

my $tiles = Settlers::Map->_starter;
my %sndr = (player => 'A');

ok my $game = Settlers::Game->new(), 'constructor';
is $game->{max_players}, 4, '4 max players by default';
ok $game->start_time, 'game has start time';
ok $game->uuid, 'game has uuid';

# setup
ok $game->event({%sndr, event => 'PA', value => {}}), 'add player 1';
ok $game->event({%sndr, event => 'PA', value => {}}), 'add player 2';
ok $game->event({%sndr, event => 'PA', value => {}}), 'add player 3';
ok $game->event({%sndr, event => 'PA', value => {}}), 'add player 4';

# test adding an extra player and the return values
ok my $no_PA_msgs = $game->event({%sndr, event => 'PA', value => {}}), 'cannot exceed 4 players';
ok scalar @$no_PA_msgs == 1, 'just 1 msg returned';
ok $no_PA_msgs->[0]{event} eq 'CH', 'msg returned was a chat message';
ok $game->event({%sndr, event => 'MD', value => {type=>'random' }})->[0]{event} eq 'MD', 'define random map';
ok $game->event({%sndr, event => 'MD', value => {type=>'starter'}}), 'define starter map';
ok $game->event({%sndr, event => 'MD', value => {type=>'custom', schema=>$tiles}}), 'define custom map';
ok $game->event({%sndr, event => 'PE', value => {}}), 'end setup phase';
ok $game->event({%sndr, event => 'XxX',value => {}})->[0]{event} eq 'CH', 'incorrect action code msg';

# deployment
is $game->robber->location, "-2,0", 'robber has been deployed to desert tile';
ok $game->event({%sndr, event => 'BS', value => { player => 1, intersection => [[0,-2],[-1,-1],[-1,-2]]}})->[0]{event} eq 'BS', 'deploy settlement';
ok $game->event({%sndr, event => 'BR', value => { player => 1, path => [[[0,-2],[-1,-1],[-1,-2]],[[0,-2],[0,-1],[-1,-1]]]}})->[0]{event} eq 'BR', 'deploy road';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'BS', value => { player => 2, intersection => [[0,2],[-1,3],[-1,2]]}})->[0]{event} eq 'BS', 'deploy settlement';
ok $game->event({%sndr, event => 'BR', value => { player => 2, path => [[[0,2],[-1,3],[-1,2]],[[0,1],[0,2],[-1,2]]]}})->[0]{event} eq 'BR', 'deploy road';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'BS', value => { player => 3, intersection => [[-2,1],[-2,2],[-3,2]]}})->[0]{event} eq 'BS', 'deploy settlement';
ok $game->event({%sndr, event => 'BR', value => { player => 3, path => [[[-2,1],[-2,2],[-3,2]],[[-2,2],[-3,3],[-3,2]]]}})->[0]{event} eq 'BR', 'deploy road';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'BS', value => { player => 4, intersection => [[2,-2],[2,-1],[1,-1]]}})->[0]{event} eq 'BS', 'deploy settlement';
ok $game->event({%sndr, event => 'BR', value => { player => 4, path => [[[2,-2],[2,-1],[1,-1]],[[2,-1],[1,-1],[1,0]]]}})->[0]{event} eq 'BR', 'deploy road';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'BS', value => { player => 4, intersection => [[2,0],[2,-1],[1,0]]}})->[0]{event} eq 'BS', 'deploy settlement';
ok $game->event({%sndr, event => 'BR', value => { player => 4, path => [[[2,0],[2,-1],[1,0]],[[2,-1],[1,-1],[1,0]]]}})->[0]{event} eq 'BR', 'deploy road';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'BS', value => { player => 3, intersection => [[-1,0],[-1,1],[-2,1]]}})->[0]{event} eq 'BS', 'deploy settlement';
ok $game->event({%sndr, event => 'BS', value => { player => 3, intersection => [[-1,0],[-1,1],[-2,1]]}})->[0]{event} eq 'CH', 'deploy settlement in same spot msg';
ok $game->event({%sndr, event => 'BR', value => { player => 3, path => [[[-1,0],[-1,1],[-2,1]],[[0,0],[-1,1],[-1,0]]]}})->[0]{event} eq 'BR', 'deploy road';
ok $game->event({%sndr, event => 'BS', value => { player => 2, intersection => [[1,1],[0,2],[0,1]]}})->[0]{event} eq 'CH', 'deploy settlement out of turn msg';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'BS', value => { player => 2, intersection => [[1,1],[0,2],[0,1]]}})->[0]{event} eq 'BS', 'deploy settlement';
ok $game->event({%sndr, event => 'BR', value => { player => 2, path => [[[1,1],[0,2],[0,1]],[[0,1],[0,2],[-1,2]]]}})->[0]{event} eq 'BR', 'deploy road';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'CH', 'end turn unfinished business msg';
ok $game->event({%sndr, event => 'BS', value => { player => 1, intersection => [[1,-2],[0,-1],[0,-2]]}})->[0]{event} eq 'BS', 'deploy settlement';
ok $game->event({%sndr, event => 'BR', value => { player => 1, path => [[[1,-2],[0,-1],[0,-2]],[[0,-2],[0,-1],[-1,-1]]]}})->[0]{event} eq 'BR', 'deploy road';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
cmp_ok grep(2 == $_->victory_points_count, @{$game->players}),'==', 4, 'all players have 2 victory points at deployment end';

# play
# round 1
ok $game->event({%sndr, event => 'DR', value => { player => 1, result => 4}})->[0]{event} eq 'DR', 'roll dice';
my $p1_total_resources = sum values %{$game->player->resources};
my $bank_total_resources = sum values %{$game->bank->resources};
ok $game->event({%sndr, event => 'BR', value => { player => 1, path => [[[0,-2],[0,-1],[-1,-1]],[[0,-1],[-1,0],[-1,-1]]]}})->[0]{event} eq 'BR', 'build road';
ok sum(values %{$game->player->resources})- $p1_total_resources == -2, 'building a road removed 2 resources';
ok sum(values %{$game->bank->resources})- $bank_total_resources == 2, 'the bank gained 2 resources';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 2, result => 6}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TB', value => { player => 2, resources => {2 => {L=>-2, B=>1}} }})->[0]{event} eq 'TB', 'trade 2 lumber for a brick';
ok $game->event({%sndr, event => 'BR', value => { player => 2, path => [[[1,0],[1,1],[0,1]],[[1,1],[0,2],[0,1]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 3, result => 8}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'BR', value => { player => 3, path => [[[0,-1],[0,0],[-1,0]],[[0,0],[-1,1],[-1,0]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'TO', value => { player => 3, resources=> {3=>{B=>-1,L=>1},4=>{B=>1,L=>-1}}, uuid => 1}})->[0]{event} eq 'TO', 'trade offer';
ok $game->event({%sndr, event => 'TA', value => { player => 4, uuid => 1}})->[0]{event} eq 'TA', 'trade accept';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 4, result => 4}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'BR', value => { player => 4, path => [[[2,-1],[2,0],[1,0]],[[2,0],[1,1],[1,0]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';

# round 2
ok $game->event({%sndr, event => 'DR', value => { player => 1, result => 11}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 2, result => 7}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'RM', value => { player => 2, tile => [1,0]}})->[0]{event} eq 'RM', 'move robber';
ok $game->event({%sndr, event => 'RR', value => { player => 2, target_player => 4, code => 'G'}})->[0]{event} eq 'RR', 'robber rob';
ok $game->event({%sndr, event => 'BD', value => { player => 2, type => 'YP'}})->[0]{event} eq 'BD', 'build development card';
ok $game->event({%sndr, event => 'PD', value => { player => 2, type => 'YP'}})->[0]{event} eq 'CH', 'cant play card same turn as buying it';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 3, result => 5}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 4, result => 9}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'BR', value => { player => 4, path => [[[1,0],[1,1],[0,1]],[[2,0],[1,1],[1,0]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';

# round 3
ok $game->event({%sndr, event => 'DR', value => { player => 1, result => 6}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TO', value => { player => 1, resources=> {1=>{L=>-1,G=>1},4=>{L=>1,G=>-1}}, uuid => 2}})->[0]{event} eq 'TO', 'trade offer';
ok $game->event({%sndr, event => 'TA', value => { player => 4, uuid => 2}})->[0]{event} eq 'TA', 'trade accept';
ok $game->event({%sndr, event => 'BS', value => { player => 1, intersection => [[0,-1],[-1,0],[-1,-1]]}})->[0]{event} eq 'BS', 'build settlement';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'PD', value => { player => 2, type => 'YP' }})->[0]{event} eq 'PD', 'play year of plenty';
ok $game->event({%sndr, event => 'YP', value => { player => 2, resources => {2 =>{G=>1, B=>1}} }})->[0]{event} eq 'YP', 'collect year of plenty resources';
ok $game->event({%sndr, event => 'DR', value => { player => 2, result => 11}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 3, result => 12}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'BR', value => { player => 3, path => [[[0,0],[-1,1],[-1,0]],[[0,0],[0,1],[-1,1]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 4, result => 5}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'BR', value => { player => 4, path => [[[1,0],[1,1],[0,1]],[[1,0],[0,1],[0,0]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';

# round 4
ok $game->event({%sndr, event => 'DR', value => { player => 1, result => 4}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TO', value => { player => 1, resources=> {1=>{L=>1,B=>-1},2=>{L=>-1,B=>1}}, uuid => 3}})->[0]{event} eq 'TO', 'trade offer';
ok $game->event({%sndr, event => 'TC', value => { player => 1, uuid => 3}})->[0]{event} eq 'TC', 'trade cancel';
ok $game->event({%sndr, event => 'TO', value => { player => 1, resources=> {1=>{L=>1,B=>-1,O=>-1},2=>{O=>1,L=>-1,B=>1}}, uuid => 4}})->[0]{event} eq 'TO', 'trade offer';
ok $game->event({%sndr, event => 'TA', value => { player => 2, uuid => 4}})->[0]{event} eq 'TA', 'trade accept';
ok $game->event({%sndr, event => 'BR', value => { player => 1, path => [[[0,-1],[0,0],[-1,0]],[[0,-1],[-1,0],[-1,-1]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 2, result => 7}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'RM', value => { player => 2, tile => [1,-1]}})->[0]{event} eq 'RM', 'move robber';
ok $game->event({%sndr, event => 'RR', value => { player => 2, target_player => 4, code => 'B'}})->[0]{event} eq 'CH', 'robber steal resource they dont have';
ok $game->event({%sndr, event => 'RR', value => { player => 2, target_player => 4, code => 'W'}})->[0]{event} eq 'RR', 'robber steal';
ok $game->event({%sndr, event => 'BR', value => { player => 2, path => [[[0,1],[0,2],[-1,2]],[[0,1],[-1,2],[-1,1]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'BD', value => { player => 2, type => 'MO'}})->[0]{event} eq 'BD', 'build development card';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 3, result => 10}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 4, result => 5}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';

# round 5
ok $game->event({%sndr, event => 'DR', value => { player => 1, result => 6}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TO', value => { player => 1, resources=> {1=>{O=>-1,G=>1},4=>{G=>-1,O=>1}}, uuid => 5}})->[0]{event} eq 'TO', 'trade offer';
ok $game->event({%sndr, event => 'TA', value => { player => 4, uuid => 5}})->[0]{event} eq 'TA', 'trade accept';
ok $game->event({%sndr, event => 'BC', value => { player => 1, intersection => [[0,-1],[-1,0],[-1,-1]]}})->[0]{event} eq 'CH', 'build city too few resources';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 2, result => 2}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'BS', value => { player => 2, intersection => [[0,1],[-1,2],[-1,1]]}})->[0]{event} eq 'BS', 'build settlement';
ok $game->event({%sndr, event => 'PD', value => { player => 2, type => 'MO'}})->[0]{event} eq 'PD', 'play monopoly';
ok $game->event({%sndr, event => 'MO', value => { player => 2, resource_code => 'O'}})->[0]{event} eq 'MO', 'declare monopoly on Ore';
ok $game->event({%sndr, event => 'TB', value => { player => 2, resources => {2 =>{O=>-4, B=>1}}}})->[0]{event} eq 'TB', 'trade 4 ore to the bank';
ok $game->event({%sndr, event => 'BR', value => { player => 2, path => [[[0,1],[-1,2],[-1,1]],[[-1,1],[-1,2],[-2,2]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 3, result => 8}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 4, result => 5}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TO', value => { player => 4, resources=> {4=>{O=>1,W=>-1},3=>{W=>1,O=>-1}}, uuid => 6}})->[0]{event} eq 'TO', 'trade offer';
ok $game->event({%sndr, event => 'TA', value => { player => 3, uuid => 6}})->[0]{event} eq 'TA', 'trade accept';
ok $game->event({%sndr, event => 'BD', value => { player => 4, type => 'RB'}})->[0]{event} eq 'BD', 'build development card';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';

# round 6
ok $game->event({%sndr, event => 'DR', value => { player => 1, result => 6}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 2, result => 2}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'BC', value => { player => 2, intersection => [[0,1],[-1,2],[-1,1]]}})->[0]{event} eq 'BC', 'build city';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 3, result => 3}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'BR', value => { player => 3, path => [[[-2,2],[-3,3],[-3,2]],[[-2,2],[-2,3],[-3,3]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';

my $p4_total_resources = sum values %{$game->player->resources};
ok $game->event({%sndr, event => 'PD', value => { player => 4, type => 'RB' }})->[0]{event} eq 'PD', 'play road building';
ok $game->event({%sndr, event => 'BR', value => { player => 4, path => [[[1,-1],[1,0],[0,0]],[[1,0],[0,1],[0,0]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'BR', value => { player => 4, path => [[[2,-1],[1,0],[1,-1]],[[1,-1],[1,0],[0,0]]]}})->[0]{event} eq 'BR', 'build road';
ok sum(values %{$game->player->resources}) - $p4_total_resources == 0, 'building a road with roadbuilding did not use resources';

ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'CH', 'end turn before rolling dice msg';
ok $game->event({%sndr, event => 'DR', value => { player => 4, result => 7}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'RM', value => { player => 4, tile => [0,2]}})->[0]{event} eq 'RM', 'move robber';
ok $game->event({%sndr, event => 'RR', value => { player => 4, target_player => 2, code => 'L'}})->[0]{event} eq 'RR', 'robber steal';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';

# round 7
ok $game->event({%sndr, event => 'DR', value => { player => 1, result => 4}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'BR', value => { player => 1, path => [[[1,-1],[0,0],[0,-1]],[[0,-1],[0,0],[-1,0]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'BS', value => { player => 1, intersection => [[1,-1],[0,0],[0,-1]]}})->[0]{event} eq 'BS', 'build settlement';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 2, result => 2}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TO', value => { player => 2, resources=> {2=>{B=>1,W=>1,G=>-2},3=>{B=>-1,G=>2,W=>-1}}, uuid => 7}})->[0]{event} eq 'TO', 'trade offer';
ok $game->event({%sndr, event => 'TA', value => { player => 3, uuid => 7}})->[0]{event} eq 'TA', 'trade accept';
ok $game->event({%sndr, event => 'BR', value => { player => 2, path => [[[-1,2],[-2,3],[-2,2]],[[-1,1],[-1,2],[-2,2]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'BD', value => { player => 2, type => 'RB'}})->[0]{event} eq 'BD', 'build development card';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 3, result => 7}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'RM', value => { player => 3, tile => [0,-1]}})->[0]{event} eq 'RM', 'move robber';
ok $game->event({%sndr, event => 'RR', value => { player => 3, target_player => 1, code => 'L'}})->[0]{event} eq 'RR', 'robber steal';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 4, result => 5}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';

# round 8
ok $game->event({%sndr, event => 'DR', value => { player => 1, result => 11}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TO', value => { player => 1, resources=> {1=>{L=>-1,O=>1,G=>1},3=>{L=>1,G=>-1,O=>-1}}, uuid => 8}})->[0]{event} eq 'TO', 'trade offer';
ok $game->event({%sndr, event => 'TA', value => { player => 3, uuid => 8}})->[0]{event} eq 'TA', 'trade accept';
ok $game->event({%sndr, event => 'BC', value => { player => 1, intersection => [[1,-1],[0,0],[0,-1]]}})->[0]{event} eq 'BC', 'build city';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 2, result => 6}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'PD', value => { player => 2, type => 'RB' }})->[0]{event} eq 'PD', 'play road building';
ok $game->event({%sndr, event => 'BR', value => { player => 2, path => [[[-1,2],[-2,3],[-2,2]],[[-1,2],[-1,3],[-2,3]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'BR', value => { player => 2, path => [[[0,2],[-1,3],[-1,2]],[[-1,2],[-1,3],[-2,3]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 3, result => 8}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'BS', value => { player => 3, intersection => [[-2,2],[-2,3],[-3,3]]}})->[0]{event} eq 'BS', 'build settlement';
ok $game->event({%sndr, event => 'BR', value => { player => 3, path => [[[-2,2],[-2,3],[-3,3]],[[-1,2],[-2,3],[-2,2]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 4, result => 5}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TO', value => { player => 4, resources=> {3=>{B=>1,W=>-1},4=>{B=>-1,W=>1}}, uuid => 8}})->[0]{event} eq 'TO', 'trade offer';
ok $game->event({%sndr, event => 'TA', value => { player => 3, uuid => 8}})->[0]{event} eq 'TA', 'trade accept';
ok $game->event({%sndr, event => 'BS', value => { player => 4, intersection => [[1,0],[0,1],[0,0]]}})->[0]{event} eq 'BS', 'build settlement';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';

# round 9
ok $game->event({%sndr, event => 'DR', value => { player => 1, result => 7}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'RM', value => { player => 1, tile => [-2,0]}})->[0]{event} eq 'RM', 'move robber';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 2, result => 6}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 3, result => 11}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 4, result => 5}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'BR', value => { player => 4, path => [[[2,-2],[1,-1],[1,-2]],[[2,-2],[2,-1],[1,-1]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';

# round 10
ok $game->event({%sndr, event => 'DR', value => { player => 1, result => 11}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'BC', value => { player => 1, intersection => [[0,-1],[-1,0],[-1,-1]]}})->[0]{event} eq 'BC', 'build city';
ok $game->event({%sndr, event => 'TB', value => { player => 1, resources => {1 => {G=>1, B=>-4}} }})->[0]{event} eq 'TB', 'trade 4 brick for a grain';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 2, result => 2}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TB', value => { player => 2, resources => {2 => {B=>1, L=>-2}} }})->[0]{event} eq 'TB', 'trade 2 lumber for a brick';
ok $game->event({%sndr, event => 'BR', value => { player => 2, path => [[[0,2],[-1,3],[-1,2]],[[0,2],[0,3],[-1,3]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 3, result => 11}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TO', value => { player => 3, resources=> {1=>{L=>-1,B=>-1,O=>4},3=>{L=>1,B=>1,O=>-4}}, uuid => 2}})->[0]{event} eq 'TO', 'trade offer';
ok $game->event({%sndr, event => 'TA', value => { player => 1, uuid => 2}})->[0]{event} eq 'TA', 'trade accept';
ok $game->event({%sndr, event => 'BR', value => { player => 3, path => [[[-1,0],[-1,1],[-2,1]],[[-1,0],[-2,1],[-2,0]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 4, result => 6}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';

# round 11
ok $game->event({%sndr, event => 'DR', value => { player => 1, result => 12}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'BC', value => { player => 1, intersection => [[1,-2],[0,-1],[0,-2]]}})->[0]{event} eq 'BC', 'build city';
ok $game->event({%sndr, event => 'BC', value => { player => 1, intersection => [[0,-2],[-1,-1],[-1,-2]]}})->[0]{event} eq 'BC', 'build city';
ok $game->event({%sndr, event => 'BD', value => { player => 1, type => 'KN'}})->[0]{event} eq 'BD', 'build development card';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'TB', value => { player => 2, resources => {2 => {B=>1, L=>-2}} }})->[0]{event} eq 'TB', 'trade 2 lumber for a brick';
ok $game->event({%sndr, event => 'DR', value => { player => 2, result => 12}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'BR', value => { player => 2, path => [[[1,2],[0,3],[0,2]],[[0,2],[0,3],[-1,3]]]}})->[0]{event} eq 'BR', 'build road';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 3, result => 12}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 4, result => 6}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';

# round 12
ok $game->event({%sndr, event => 'DR', value => { player => 1, result => 11}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'PD', value => { player => 1, type => 'KN'}})->[0]{event} eq 'PD', 'play knight';
ok $game->event({%sndr, event => 'RM', value => { player => 1, tile => [-1,2]}})->[0]{event} eq 'RM', 'move robber';
ok $game->event({%sndr, event => 'RR', value => { player => 1, target_player => 2, code => 'G'}})->[0]{event} eq 'RR', 'robber rob';
ok $game->event({%sndr, event => 'BD', value => { player => 1, type => 'KN'}})->[0]{event} eq 'BD', 'build development card';
ok $game->event({%sndr, event => 'BD', value => { player => 1, type => 'KN'}})->[0]{event} eq 'BD', 'build development card';
ok $game->event({%sndr, event => 'BD', value => { player => 1, type => 'VP'}})->[0]{event} eq 'BD', 'build development card';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 2, result => 9}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 3, result => 11}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 4, result => 2}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';

# round 13
ok $game->event({%sndr, event => 'DR', value => { player => 1, result => 2}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 2, result => 2}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 3, result => 2}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 4, result => 2}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';

# round 14
ok $game->event({%sndr, event => 'PD', value => { player => 1, type => 'KN'}})->[0]{event} eq 'PD', 'play knight';
ok $game->event({%sndr, event => 'RM', value => { player => 1, tile => [-2,0]}})->[0]{event} eq 'RM', 'move robber';
ok $game->event({%sndr, event => 'DR', value => { player => 1, result => 12}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 1}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 2, result => 12}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 2}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 3, result => 12}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 3}})->[0]{event} eq 'TE', 'end turn';
ok $game->event({%sndr, event => 'DR', value => { player => 4, result => 12}})->[0]{event} eq 'DR', 'roll dice';
ok $game->event({%sndr, event => 'TE', value => { player => 4}})->[0]{event} eq 'TE', 'end turn';

# round 15, end
ok my $endgame = $game->event({%sndr, event => 'PD', value => { player => 1, type => 'KN'}}), 'play knight';
ok $endgame->[0]{event} eq 'PD', 'play knight';
ok $endgame->[1]{event} eq 'RA', 'activate robber';
ok $endgame->[2]{event} eq 'LA', 'award largest army';
ok $endgame->[3]{event} eq 'GO', 'game over';
ok $endgame->[4]{event} eq 'PE', 'phase end play';
ok $endgame->[5]{event} eq 'PS', 'phase start end';

# dump the log
open my $fh, '>', 'data/example.json' or die 'unable to open log file';
print $fh encode_json ($game->log);
done_testing;
