package Yandex::Translate;

use strict;
use warnings;
use HTTP::Tiny;
use JSON;

$Yandex::Translate::VERSION = '1.0';

sub new
{
    my $class = shift;
    my $self = {
	_key_        => shift,
        _text_       => shift,
        _from_lang_  => shift,
        _to_lang_    => shift,
	_base_       => "https://translate.yandex.net/api/v1.5/tr.json",
	_post_       => undef
    };

    bless $self, $class;
    return $self
}

sub set_key
{
    my ( $self, $key ) = @_;
    $self->{_key_} = $key if defined($key);
}

sub get_langs_list
{
    my $self = shift;
    my $query = '/getLangs?';
    $self->{_post_} = 'ui=en&key='.$self->{_key_};
    my $response = HTTP::Tiny->new->get($self->{_base_} . $query  . $self->{_post_});
    die "Invalid Api\n" if $response->{status} eq '403';
    die "Blocked Api\n" if $response->{status} eq '402';
    die "Failed!\n" unless $response->{success};
    my $result = undef;
    if (length $response->{content}) {
        my $json_respond =  JSON->new->utf8->decode($response->{content});
	my $index = 0;
	while ($index < 10) {
	    $result .= ' | '.$json_respond->{dirs}[$index];
	    $index++;
	}
    }

    return $result.'... for more info visit https://translate.yandex.com\n' if defined($result);
}

sub set_text 
{
    my ( $self, $text ) = @_;
    $self->{_text_} = $text if defined($text);
}

sub set_from_lang
{
    my ($self, $from_lang) = @_;
    $self->{_from_lang_} = $from_lang if defined($from_lang);
}

sub set_to_lang
{
    my ($self, $to_lang) = @_;
    $self->{_to_lang_} = $to_lang if defined($to_lang);
}

sub detect_lang
{
    my $self = shift;
    my $query = '/detect?';
    $self->{_post_} = 'text='.$self->{_text_}.'&key='.$self->{_key_};
    my $response = HTTP::Tiny->new->get($self->{_base_} . $query  . $self->{_post_});
    
    die "Failed!\n" unless $response->{success};
   
    if (length $response->{content}) {
        my $json_respond =  JSON->new->utf8->decode($response->{content});
	return $json_respond->{lang}."\n";
    }
}

sub translate
{
    my $self = shift;
    my $query = '/translate?';
    $self->{_text_} =~ s/ /+/ig;
    $self->{_post_} = 'lang='.$self->{_from_lang_}.'-'.$self->{_to_lang_}.'&key='.$self->{_key_}.'&text='.$self->{_text_};
    my $response = HTTP::Tiny->new->get($self->{_base_} . $query  . $self->{_post_});
    die "Invalid Api\n" if $response->{status} eq '401';
    die "Invalid Api\n" if $response->{status} eq '403';
    die "Exceeded the daily limit on the amount of translated text\n" if $response->{status} eq '403';
    die "Exceeded the maximum text size\n" if $response->{status} eq '413';
    die "The text cannot be translated\n" if $response->{status} eq '422';
    die "The specified translation direction is not supported\n" if $response->{status} eq '501';
    die "Failed!\n" unless $response->{success};

    if (length $response->{content}) {
        my $json_respond =  JSON->new->utf8->decode($response->{content});
        return $json_respond->{text}[0]."\n";
    }
}

1;
__END__

=pod
 
=head1 NAME

Yandex::Translate - An Api, simple, for Yandex translate
 
=head1 VERSION

    version 1.0
 
=head1 SYNOPSIS

    use Yandex::Translate;

    my $tr = Yandex::Translate->new;

    #Set your yandex api key that you can get from https://tech.yandex.com/
    $tr->set_key("yandex_key"); 

    #Set the text that you want to translate
    $tr->set_text('In the name of God');
    
    #set the source lang
    $tr->set_from_lang('en');
    
    #set the dist lang (xx means whater you want)
    $tr->set_to_lang('xx');
    
    #Get the result
    print $tr->translate();
 
=head1 DESCRIPTION

Yandex.Translate (previously Yandex.Translation) is a web service
provided by Yandex intended for the translation of text or web pages
into another language.

The service uses a self-learning statistical machine translation,
developed by Yandex.
The system constructs the dictionary of correspondences based
on the analysis of millions of translated texts.
    
 
=head1 METHODS

=over 4

=item  new

    $tr = Yandex->Translate->new( %attributes );

This constructor return a new Yandex::Translate object, optionel attributes (see examples directory).
include:

*-key  -- Key Api, you should get it from yandex website after register, string format.

*-text -- Text that you want to translate, string format.

*-from_lang -- The source lang, string format.

*-to_lang -- the distination lang, string format.

=item  set_key

    $tr->set_key( 'yandex_key' ); 
    
This method is used to pass an api key (string), instead of pass it in 'new' 
you can define an object without any attributes (I like methods way).

=item  set_text

    $tr->set_text( 'text_to_translate' );

This method is used to pass your text to translate (string).

=item  set_from_lang

    $tr->set_from_lang( 'en' );
    
This method is used to pass your source lang (string), example english => 'en'.

=item  set_to_lang

    $tr->set_to_lang( 'xx' );

This method is used to pass your destination lang (string) for example arabic => 'ar'.

=item  translate

    print  $tr->translate();

This method is used to get the result, It returns a string.

=item  detect_lang

    print  $tr->detect_lang();

This method is used to detect the language of the text, It returns a string.

=item  get_langs_list

    print  $tr->get_langs_list();

This method is used to get the list of supported languages, It returns a string.

=back

=head1 SEE ALSO

For more info please visit Yandex https://translate.yandex.com/developers

=head1 SUPPORT

Bugs / Feature Requests
Please report any bugs or feature requests on my email.

Source Code
This is open source software. The code repository is available for
public review and contribution under the terms of the license.

<https://fossil.falseking.site/ticket>

=head1 AUTHORS

*    James Axl <axlrose112@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is copyleft (c) 2017 by James Axl.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut

