package Venus::Run;

use 5.018;

use strict;
use warnings;

use Venus::Class 'base';

base 'Venus::Task';

require Venus;

our $NAME = __PACKAGE__;

# METHODS

state $args = {
  'command' => {
    help => 'Command to run',
    required => 1,
  }
};

sub args {

  return $args;
}

state $cmds = {
  'help' => {
    help => 'Display help and usages',
    arg => 'command',
  },
  'init' => {
    help => 'Initialize the configuration file',
    arg => 'command',
  },
};

sub cmds {

  return $cmds;
}

sub conf {
  my ($self) = @_;

  require Venus::Config;

  return Venus::Config->read_file($self->file)->value;
}

sub file {

  return $ENV{VENUS_FILE} || (grep -f, map ".vns.$_", qw(yaml yml json js perl pl))[0]
}

state $footer = <<"EOF";
Config:

Here is an example configuration in YAML (e.g. in .vns.yaml).

  ---
  data:
    ECHO: true
  exec:
    okay: \$PERL -c
    cpan: cpanm -llocal -qn
    deps: cpan --installdeps .
    each: \$PERL -MVenus=log -nE
    exec: \$PERL -MVenus=log -E
    repl: \$PERL -dE0
    says: exec "map log(\$_), map eval, \@ARGV"
    test: \$PROVE
  libs:
  - -Ilib
  - -Ilocal/lib/perl5
  load:
  - -MVenus=true,false
  path:
  - ./bin
  - ./dev
  - -Ilocal/bin
  perl:
    perl: perl
    prove: prove
  vars:
    PERL: perl
    PROVE: prove

Examples:

Here are examples usages using the example YAML configuration.

  # Mint a new configuration file
  vns init

  # Install a distribution
  vns cpan \$DIST

  # Upgrade to the latest version of Venus
  vns deps

  # Check that a package can be compiled
  vns okay \$FILE

  # Use the Perl debugger as a REPL
  vns repl

  # Evaluate arbitrary Perl expressions
  vns exec ...

  # Test the Perl project in the CWD
  vns test t

Copyright 2022-2023, Vesion $Venus::VERSION, The Venus "AUTHOR" and "CONTRIBUTORS"

More information on "vns" and/or the "Venus" standard library, visit
https://p3rl.org/Venus.
EOF

sub footer {

  return $footer;
}

sub handler {
  my ($self, $data) = @_;

  my $help = $data->{help};
  my $next = $data->{command};

  return $self->okay if !$help && !$next;

  return $self->handler_for_help($data) if (!$next && $help) || (lc($next) eq 'help');

  return $self->handler_for_init($data) if lc($next) eq 'init';

  return $self->handler_for_exec($data);
}

sub handler_for_exec {
  my ($self, $data) = @_;

  my $code = sub {
    my $command = join ' ', @_;

    $self->log_info('Using:', $command) if $ENV{ECHO};

    my $error = $self->catch('system', $command);

    $self->log_error("Error running command! $command") if $error;
  };

  return $self->okay(sub{_exec($code, $self->conf, @{$self->data})});
}

sub handler_for_help {
  my ($self, $data) = @_;

  return $self->fail(sub{$self->log_info($self->help)});
}

sub handler_for_init {
  my ($self, $data) = @_;

  my $file = $self->file;

  return $self->fail('log_error', "Already using $file") if $file && -f $file;

  $file ||= '.vns.pl';

  require Venus::Config;

  my $init = $self->init;

  Venus::Config->new($self->init)->write_file($file);

  return $self->okay('log_info', "Initialized with generated file $file");
}

state $init = {
  data => {
    ECHO => 1,
  },
  exec => {
    brew => 'perlbrew',
    cpan => 'cpanm -llocal -qn',
    deps => 'cpan --installdeps .',
    each => '$PERL -MVenus=true,false,log -nE',
    eval => '$PERL -MVenus=true,false,log -E',
    exec => '$PERL',
    info => '$PERL -V',
    okay => '$PERL -c',
    repl => '$PERL -dE0',
    says => 'eval "map log($_), map eval, @ARGV"',
    test => '$PROVE'
  },
  find => {
  },
  libs => [
    '-Ilib',
    '-Ilocal/lib/perl5',
  ],
  load => [
  ],
  path => [
    './bin',
    './dev',
    './local/bin',
  ],
  perl => {
    perl => 'perl',
    prove => 'prove',
    'perl-5.18.0' => 'perlbrew exec --with perl-5.18.0 perl',
    'prove-5.18.0' => 'perlbrew exec --with perl-5.18.0 prove'
  },
  task => {
  },
  vars => {
    PERL => 'perl',
    PROVE => 'prove'
  },
};

sub init {

  return $init;
}

sub name {

  return $ENV{VENUS_RUN_NAME} || $NAME;
}

state $opts = {
  'help' => {
    help => 'Show help information',
  }
};

sub opts {

  return $opts;
}

# ROUTINES

sub _exec {
  my ($code, $conf, @data) = @_;

  return () if !@data;

  my %ORIG_ENV = %ENV;

  _set_vars($conf);

  _set_path($conf);

  _set_libs($conf);

  my @args = grep defined, _make($conf, shift(@data)), @data;

  my $prog = shift @args;

  require Venus::Os;

  @args = map +(/^\w+$/ ? $_ : /^\$[A-Z]\w+$/ ? $_ : Venus::Os->quote($_)), @args;

  return () if !$prog;

  $code->($prog, @args);

  %ENV = %ORIG_ENV;

  return ($prog, @args);
}

sub _find {
  my ($seen, $conf, @data) = @_;

  return () if !@data;

  @data = map _split($_), @data;

  my @item = map {s/^\$//r} shift @data;

  @item = (_find_in_exec($seen, $conf, @item));

  if (@item > 1) {
    unshift @data, @item; @item = shift @data;
  }

  @item = (_find_in_find($seen, $conf, @item));

  if (@item > 1) {
    unshift @data, @item; @item = shift @data;
  }

  @item = (_find_in_perl($seen, $conf, @item));

  if (@item > 1) {
    unshift @data, @item; @item = shift @data;
  }

  @item = (_find_in_task($seen, $conf, @item));

  if (@item > 1) {
    unshift @data, @item; @item = shift @data;
  }

  @item = (_find_in_vars($seen, $conf, @item));

  if (@item > 1) {
    unshift @data, @item; @item = shift @data;
  }

  return (@item, @data);
}

sub _find_in_exec {
  my ($seen, $conf, $item) = @_;

  return $conf->{exec}
    ? (
    $conf->{exec}{$item}
    ? (do {
      my $value = $conf->{exec}{$item};
      ($value eq $item)
        ? ($value)
        : (_find_in_seen($seen, $conf, 'exec', $item, $value));
    })
    : ($item)
    )
    : ($item);
}

sub _find_in_find {
  my ($seen, $conf, $item) = @_;

  return $conf->{find}
    ? (
    $conf->{find}{$item}
    ? (do {
      my $value = $conf->{find}{$item};
      ($value eq $item)
        ? ($value)
        : (_find_in_seen($seen, $conf, 'find', $item, $value));
    })
    : ($item)
    )
    : ($item);
}

sub _find_in_perl {
  my ($seen, $conf, $item) = @_;

  return $conf->{perl}
    ? (
    $conf->{perl}{$item}
    ? (do {
      my $value = $conf->{perl}{$item};
      ($value eq $item)
        ? ($value)
        : (_find_in_seen($seen, $conf, 'perl', $item, $value));
    },
      _load_from_libs($seen, $conf),
      _load_from_load($seen, $conf),
    )
    : ($item)
    )
    : ($item);
}

sub _find_in_task {
  my ($seen, $conf, $item) = @_;

  return $conf->{task}
    ? (
    $conf->{task}{$item}
    ? (do {
      $ENV{VENUS_TASK_RUN} = 1;
      my $value = $conf->{task}{$item};
      ($value eq $item)
        ? ($value)
        : (_find_in_seen($seen, $conf, 'task', $item, $value));
    })
    : ($item)
    )
    : ($item);
}

sub _find_in_vars {
  my ($seen, $conf, $item) = @_;

  return $conf->{vars}
    ? (
    $conf->{vars}{$item}
    ? (do {
      my $value = $conf->{vars}{$item};
      ($value eq $item)
        ? ($value)
        : (_find_in_seen($seen, $conf, 'vars', $item, $value));
    })
    : ($item)
    )
    : ($item);
}

sub _find_in_seen {
  my ($seen, $conf, $item, $name, $value) = @_;

  return $seen->{$item}{$name}++
    ? ((_split($value))[0])
    : (_find($seen, $conf, $value));
}

sub _libs {
  my ($conf) = @_;

  return (map /^-I\w*?(.*)$/, _load_from_libs({}, $conf));
}

sub _load_from_libs {
  my ($seen, $conf) = @_;

  return !$seen->{libs}++ ? ($conf->{libs} ? (@{$conf->{libs}}) : ()) : ();
}

sub _load_from_load {
  my ($seen, $conf) = @_;

  return !$seen->{load}++ ? ($conf->{load} ? (@{$conf->{load}}) : ()) : ();
}

sub _make {
  my ($conf, $name) = @_;

  my ($item, @data) = _find({}, $conf, $name);

  require Venus::Os;

  my $path = Venus::Os->which($item);

  return (($path ? $path : $item), @data);
}

sub _split {
  my ($text) = @_;

  return (grep length, ($text // '') =~ /(?x)(?:"([^"]*)"|([^\s]*))\s?/g);
}

sub _set_path {
  my ($conf) = @_;

  require Venus::Os;
  require Venus::Path;

  if (my $path = $conf->{path}) {
    $ENV{PATH} = join((Venus::Os->is_win ? ';' : ':'),
      (map Venus::Path->new($_)->absolute, @{$conf->{path}}), $ENV{PATH});
  }

  return $conf;
}

sub _set_libs {
  my ($conf) = @_;

  require Venus::Os;
  require Venus::Path;

  my %seen;
  $ENV{PERL5LIB} = join((Venus::Os->is_win ? ';' : ':'),
    (grep !$seen{$_}++, map Venus::Path->new($_)->absolute, _libs($conf)));

  return $conf;
}

sub _set_vars {
  my ($conf) = @_;

  if (my $vars = $conf->{data}) {
    $ENV{$_} = join(' ', grep defined, $vars->{$_}) for keys %{$vars};
  }

  if (my $vars = $conf->{vars}) {
    $ENV{$_} = join(' ', grep defined, _find({}, $conf, $_)) for keys %{$vars};
  }

  return $conf;
}

# AUTORUN

run Venus::Run;

1;



=head1 NAME

Venus::Run - Task Runner Class

=cut

=head1 ABSTRACT

Task Runner Class for Perl 5

=cut

=head1 SYNOPSIS

  package main;

  use Venus::Run;

  my $run = Venus::Run->new;

  # bless({...}, 'Venus::Run')

=cut

=head1 DESCRIPTION

This package is a L<Venus::Task> which provides generic task running
capabilities. A simple CLI using this package has been made available in
L<vns>.

=head1 USAGES

Here is an example configuration file in YAML (e.g. in C<.vns.yaml>).

  ---
  data:
    ECHO: true
  exec:
    okay: \$PERL -c
    cpan: cpanm -llocal -qn
    deps: cpan --installdeps .
    each: \$PERL -MVenus=log -nE
    exec: \$PERL -MVenus=log -E
    repl: \$PERL -dE0
    says: exec "map log(\$_), map eval, \@ARGV"
    test: \$PROVE
  libs:
  - -Ilib
  - -Ilocal/lib/perl5
  load:
  - -MVenus=true,false
  path:
  - ./bin
  - ./dev
  - -Ilocal/bin
  perl:
    perl: perl
    prove: prove
  vars:
    PERL: perl
    PROVE: prove

The following describes the configuration file sections and how they're used:

=over 4

=item *

The C<data> section provides a non-dynamic list of key/value pairs that will
be used as environment variables.

=item *

The C<exec> section provides the main dynamic tasks which can be recursively
resolved and expanded.

=item *

The C<find> section provides aliases which can be recursively resolved and
expanded for use in other tasks.

=item *

The C<libs> section provides a list of C<-I/path/to/lib> "include" statements
that will be automatically added to tasks expanded from the C<perl> section.

=item *

The C<load> section provides a list of C<-MPackage> "import" statements
that will be automatically added to tasks expanded from the C<perl> section.

=item *

The C<path> section provides a list of paths to be prepended to the C<PATH>
environment variable which allows programs to be found.

=item *

The C<perl> section provides the dynamic perl tasks which can serve as tasks
with default commands (with options) and which can be recursively resolved and
expanded.

=item *

The C<task> section provides the dynamic perl tasks which "load" L<Venus::Task>
derived packages, and which can be recursively resolved and expanded. These
tasks will typically take the form of C<perl -Ilib -MMyApp::Task -E0 --> and
will be automatically executed as a CLI.

=item *

The C<vars> section provides a list of dynamic key/value pairs that can be
recursively resolved and expanded and will be used as environment variables.

=back

Here are examples usages using the example YAML configuration file and the
L<vns> CLI.

  # Mint a new configuration file
  vns init

  ...

  # Install a distribution
  vns cpan $DIST

  ... same as $(which cpanm) --llocal -qn $DIST

  # Upgrade to the latest version of Venus
  vns deps

  ... same as $(which cpanm) --llocal -qn --installdeps .

  # Check that a package can be compiled
  vns okay $FILE

  ... same as $(which perl) -Ilib -Ilocal/lib/perl5 -c $FILE

  # Use the Perl debugger as a REPL
  vns repl

  ... same as $(which perl) -Ilib -Ilocal/lib/perl5 -dE0

  # Evaluate arbitrary Perl expressions
  vns exec ...

  ... same as $(which perl) -Ilib -Ilocal/lib/perl5 -MVenus=log -E $@

  # Test the Perl project in the CWD
  vns test t

  ... same as $(which prove) -Ilib -Ilocal/lib/perl5

This package and CLI allows you to define task definitions for any application,
which you can run using the name of the task. You can reuse existing task
definitions in new tasks which will be recursively resolved when needed. You
can define static and dynamic environment variables, and also pre-define
"includes" and the order in which they're declared.

=cut

=head1 INHERITS

This package inherits behaviors from:

L<Venus::Task>

=cut

=head1 METHODS

This package provides the following methods:

=cut

=head2 args

  args() (HashRef)

The args method returns the task argument declarations.

I<Since C<2.91>>

=over 4

=item args example 1

  # given: synopsis

  package main;

  my $args = $run->args;

  # {
  #   'command' => {
  #     help => 'Command to run',
  #     required => 1,
  #   }
  # }

=back

=cut

=head2 cmds

  cmds() (HashRef)

The cmds method returns the task command declarations.

I<Since C<2.91>>

=over 4

=item cmds example 1

  # given: synopsis

  package main;

  my $cmds = $run->cmds;

  # {
  #   'help' => {
  #     help => 'Display help and usages',
  #     arg => 'command',
  #   },
  #   'init' => {
  #     help => 'Initialize the configuration file',
  #     arg => 'command',
  #   },
  # }

=back

=cut

=head2 conf

  conf() (HashRef)

The conf method loads the configuration file returned by L</file>, then decodes
and returns the information as a hashref.

I<Since C<2.91>>

=over 4

=item conf example 1

  # given: synopsis

  package main;

  my $conf = $run->conf;

  # {}

=back

=over 4

=item conf example 2

  # given: synopsis

  package main;

  local $ENV{VENUS_FILE} = 't/conf/.vns.pl';

  my $conf = $run->conf;

  # {...}

=back

=over 4

=item conf example 3

  # given: synopsis

  package main;

  # e.g. current directory has only a .vns.yaml file

  my $conf = $run->conf;

  # {...}

=back

=over 4

=item conf example 4

  # given: synopsis

  package main;

  # e.g. current directory has only a .vns.yml file

  my $conf = $run->conf;

  # {...}

=back

=over 4

=item conf example 5

  # given: synopsis

  package main;

  # e.g. current directory has only a .vns.json file

  my $conf = $run->conf;

  # {...}

=back

=over 4

=item conf example 6

  # given: synopsis

  package main;

  # e.g. current directory has only a .vns.js file

  my $conf = $run->conf;

  # {...}

=back

=over 4

=item conf example 7

  # given: synopsis

  package main;

  # e.g. current directory has only a .vns.perl file

  my $conf = $run->conf;

  # {...}

=back

=over 4

=item conf example 8

  # given: synopsis

  package main;

  # e.g. current directory has only a .vns.pl file

  my $conf = $run->conf;

  # {...}

=back

=cut

=head2 file

  file() (Str)

The file method returns the configuration file specified in the C<VENUS_FILE>
environment variable, or the discovered configuration file in the current
directory. The default name for a configuration file is in the form of
C<.vns.*>. Configuration files will be decoded based on their file extensions.
Valid file extensions are C<yaml>, C<yml>, C<json>, C<js>, C<perl>, and C<pl>.

I<Since C<2.91>>

=over 4

=item file example 1

  # given: synopsis

  package main;

  my $file = $run->file;

  # undef

=back

=over 4

=item file example 2

  # given: synopsis

  package main;

  local $ENV{VENUS_FILE} = 't/conf/.vns.pl';

  my $file = $run->file;

  # "t/conf/.vns.pl"

=back

=over 4

=item file example 3

  # given: synopsis

  package main;

  # e.g. current directory has only a .vns.yaml file

  my $file = $run->file;

  # ".vns.yaml"

=back

=over 4

=item file example 4

  # given: synopsis

  package main;

  # e.g. current directory has only a .vns.yml file

  my $file = $run->file;

  # ".vns.yml"

=back

=over 4

=item file example 5

  # given: synopsis

  package main;

  # e.g. current directory has only a .vns.json file

  my $file = $run->file;

  # ".vns.json"

=back

=over 4

=item file example 6

  # given: synopsis

  package main;

  # e.g. current directory has only a .vns.js file

  my $file = $run->file;

  # ".vns.js"

=back

=over 4

=item file example 7

  # given: synopsis

  package main;

  # e.g. current directory has only a .vns.perl file

  my $file = $run->file;

  # ".vns.perl"

=back

=over 4

=item file example 8

  # given: synopsis

  package main;

  # e.g. current directory has only a .vns.pl file

  my $file = $run->file;

  # ".vns.pl"

=back

=cut

=head2 footer

  footer() (Str)

The footer method returns examples and usage information used in usage text.

I<Since C<2.91>>

=over 4

=item footer example 1

  # given: synopsis

  package main;

  my $footer = $run->footer;

  # "..."

=back

=cut

=head2 handler

  handler(HashRef $data) (Any)

The handler method processes the data provided and executes the request then
returns the invocant unless the program is exited.

I<Since C<2.91>>

=over 4

=item handler example 1

  package main;

  use Venus::Run;

  my $run = Venus::Run->new;

  $run->execute;

  # ()

=back

=over 4

=item handler example 2

  package main;

  use Venus::Run;

  my $run = Venus::Run->new(['help']);

  $run->execute;

  # ()

=back

=over 4

=item handler example 3

  package main;

  use Venus::Run;

  my $run = Venus::Run->new(['--help']);

  $run->execute;

  # ()

=back

=over 4

=item handler example 4

  package main;

  use Venus::Run;

  my $run = Venus::Run->new(['init']);

  $run->execute;

  # ()

=back

=over 4

=item handler example 5

  package main;

  use Venus::Run;

  # on linux

  my $run = Venus::Run->new(['echo']);

  $run->execute;

  # ()

  # i.e. ['echo']

=back

=over 4

=item handler example 6

  package main;

  use Venus::Run;

  # on linux

  my $run = Venus::Run->new(['cpan', 'Venus']);

  $run->execute;

  # ()

  # i.e. cpanm '-llocal' '-qn' Venus

=back

=over 4

=item handler example 7

  package main;

  use Venus::Run;

  # on linux

  my $run = Venus::Run->new(['deps']);

  $run->execute;

  # ()

  # i.e. cpanm '-llocal' '-qn' '--installdeps' '.'

=back

=over 4

=item handler example 8

  package main;

  use Venus::Run;

  # on linux

  my $run = Venus::Run->new(['okay', 'lib/Venus.pm']);

  $run->execute;

  # ()

  # i.e. perl '-Ilib' '-Ilocal/lib/perl5' '-c'

=back

=over 4

=item handler example 9

  package main;

  use Venus::Run;

  # on linux

  my $run = Venus::Run->new(['repl']);

  $run->execute;

  # ()

  # i.e. perl '-Ilib' '-Ilocal/lib/perl5' '-dE0'

=back

=over 4

=item handler example 10

  package main;

  use Venus::Run;

  # on linux

  my $run = Venus::Run->new(['exec', '-MVenus=date', 'say date']);

  $run->execute;

  # ()

  # i.e. perl '-Ilib' '-Ilocal/lib/perl5' '-MVenus=date' 'say date'

=back

=over 4

=item handler example 11

  package main;

  use Venus::Run;

  # on linux

  my $run = Venus::Run->new(['test', 't']);

  $run->execute;

  # ()

  # i.e. prove '-Ilib' '-Ilocal/lib/perl5' t

=back

=cut

=head2 init

  init() (HashRef)

The init method returns the default configuration to be used when initializing
the system with a new configuration file.

I<Since C<2.91>>

=over 4

=item init example 1

  # given: synopsis

  package main;

  my $init = $run->init;

  # {
  #   data => {
  #     ECHO => 1,
  #   },
  #   exec => {
  #     brew => 'perlbrew',
  #     cpan => 'cpanm -llocal -qn',
  #     deps => 'cpan --installdeps .',
  #     each => '$PERL -MVenus=true,false,log -nE',
  #     eval => '$PERL -MVenus=true,false,log -E',
  #     exec => '$PERL',
  #     info => '$PERL -V',
  #     okay => '$PERL -c',
  #     repl => '$PERL -dE0',
  #     says => 'eval "map log($_), map eval, @ARGV"',
  #     test => '$PROVE'
  #   },
  #   find => {
  #   },
  #   libs => [
  #     '-Ilib',
  #     '-Ilocal/lib/perl5',
  #   ],
  #   load => [
  #   ],
  #   path => [
  #     './bin',
  #     './dev',
  #     './local/bin',
  #   ],
  #   perl => {
  #     perl => 'perl',
  #     prove => 'prove',
  #     'perl-5.18.0' => 'perlbrew exec --with perl-5.18.0 perl',
  #     'prove-5.18.0' => 'perlbrew exec --with perl-5.18.0 prove'
  #   },
  #   task => {
  #   },
  #   vars => {
  #     PERL => 'perl',
  #     PROVE => 'prove'
  #   },
  # }

=back

=cut

=head2 name

  name() (Str)

The name method returns the default name for the task. This is used in usage
text and can be controlled via the C<VENUS_RUN_NAME> environment variable, or
the C<NAME> package variable.

I<Since C<2.91>>

=over 4

=item name example 1

  # given: synopsis

  package main;

  my $name = $run->name;

  # "Venus::Run"

=back

=over 4

=item name example 2

  # given: synopsis

  package main;

  local $ENV{VENUS_RUN_NAME} = 'venus-runner';

  my $name = $run->name;

  # "venus-runner"

=back

=over 4

=item name example 3

  # given: synopsis

  package main;

  local $Venus::Run::NAME = 'venus-runner';

  my $name = $run->name;

  # "venus-runner"

=back

=cut

=head2 opts

  opts() (HashRef)

The opts method returns the task options declarations.

I<Since C<2.91>>

=over 4

=item opts example 1

  # given: synopsis

  package main;

  my $opts = $run->opts;

  # {
  #   'help' => {
  #     help => 'Show help information',
  #   }
  # }

=back

=cut

=head1 AUTHORS

Awncorp, C<awncorp@cpan.org>

=cut

=head1 LICENSE

Copyright (C) 2000, Al Newkirk.

This program is free software, you can redistribute it and/or modify it under
the terms of the Apache license version 2.0.

=cut