/*
Copyright: 2001-2003 The Perl Foundation.  All Rights Reserved.
$Id: unicode.c,v 1.21 2004/02/04 15:45:14 mikescott Exp $

=head1 NAME

chartypes/unicode.c - Unicode Chartype

=head1 DESCRIPTION

This file implements the Unicode character set functionality in Parrot.

Note that calls to the charclass functions are done via macros of the
form C<Parrot_char_is_digit(chartype, value)> which are generated by
F<config/gen/parrot_include.pl> during configuration to point to either
a dedicated slot or the generic function, as appropriate.

=head2 Functions

=over 4

=cut

*/

#include "parrot/parrot.h"

/* This list is based on Unicode 3.2 */
struct chartype_digit_map_t unicode_digit_map[] = {
   { 0x30, 0x39, 0 },       /* DIGIT ZERO..NINE */
   { 0x660, 0x669, 0 },     /* ARABIC-INDIC DIGIT ZERO..NINE */
   { 0x6F0, 0x6F9, 0 },     /* EXTENDED ARABIC-INDIC DIGIT ZERO..NINE */
   { 0x966, 0x96F, 0 },     /* DEVANAGARI DIGIT ZERO..NINE */
   { 0x9E6, 0x9EF, 0 },     /* BENGALI DIGIT ZERO..NINE */
   { 0xA66, 0xA6F, 0 },     /* GURMUKHI DIGIT ZERO..NINE */
   { 0xAE6, 0xAEF, 0 },     /* GUJARATI DIGIT ZERO..NINE */
   { 0xB66, 0xB6F, 0 },     /* ORIYA DIGIT ZERO..NINE */
   { 0xBE7, 0xBEF, 1 },     /* TAMIL DIGIT ONE..NINE */
   { 0xC66, 0xC6F, 0 },     /* TELUGU DIGIT ZERO..NINE */
   { 0xCE6, 0xCEF, 0 },     /* KANNADA DIGIT ZERO..NINE */
   { 0xD66, 0xD6F, 0 },     /* MALAYALAM DIGIT ZERO..NINE */
   { 0xE50, 0xE59, 0 },     /* THAI DIGIT ZERO..NINE */
   { 0xED0, 0xED9, 0 },     /* LAO DIGIT ZERO..NINE */
   { 0xF20, 0xF29, 0 },     /* TIBETAN DIGIT ZERO..NINE */
   { 0x1040, 0x1049, 0 },   /* MYANMAR DIGIT ZERO..NINE */
   { 0x1369, 0x1371, 1 },   /* ETHIOPIC DIGIT ONE..NINE */
   { 0x17E0, 0x17E9, 0 },   /* KHMER DIGIT ZERO..NINE */
   { 0x1810, 0x1819, 0 },   /* MONGOLIAN DIGIT ZERO..NINE */
   { 0xFF10, 0xFF19, 0 },   /* FULLWIDTH DIGIT ZERO..NINE */
   { 0x1D7CE, 0x1D7D7, 0 }, /* MATHEMATICAL BOLD DIGIT ZERO..NINE */
   { 0x1D7D8, 0x1D7E1, 0 }, /* MATHEMATICAL DOUBLE-STRUCK DIGIT ZERO..NINE */
   { 0x1D7E2, 0x1D7EB, 0 }, /* MATHEMATICAL SANS-SERIF DIGIT ZERO..NINE */
   { 0x1D7EC, 0x1D7F5, 0 }, /* MATHEMATICAL SANS-SERIF BOLD DIGIT ZERO..NINE */
   { 0x1D7F6, 0x1D7FF, 0 }, /* MATHEMATICAL MONOSPACE DIGIT ZERO..NINE */
   { 0, 0, -1 }
};

/* Inversions lists based on Unicode 4.0 */
static Parrot_UInt unicode_upper[] = {
    65,26,101,23,1,7,33,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,2,1,1,1,1,3,2,1,1,1,2,1,3,2,4,1,2,1,3,3,2,1,2,1,1,1,
    1,1,2,1,1,2,1,1,2,1,3,1,1,1,2,3,1,7,1,2,1,2,1,2,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,2,1,1,
    3,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,339,1,
    1,3,1,1,1,2,1,17,1,9,38,3,3,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,5,1,2,1,1,2,5,48,48,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,9,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    3,1,7,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,34,38,2889,38,3386,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,11,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,15,8,8,6,10,8,8,8,8,6,
    11,1,1,1,1,1,1,1,8,8,72,4,12,4,12,4,12,5,11,4,262,1,4,1,3,3,2,
    3,2,1,3,5,6,1,1,1,1,1,1,4,2,2,1,1,10,2,5,1,56795,26,1221,40,5,
    3208,26,26,26,26,26,26,1,1,2,2,1,2,2,2,4,1,8,26,26,26,2,1,4,2,
    8,1,7,27,2,1,4,1,5,1,1,3,7,27,26,26,26,26,26,26,26,26,26,26,2,
    6,30,25,33,25,33,25,33,25,33,25,993367
};

static Parrot_UInt unicode_lower[] = {
    97,26,47,1,10,1,4,1,36,24,1,8,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,3,2,1,1,1,2,1,3,2,4,1,2,1,3,3,2,
    1,2,1,1,1,1,1,2,1,1,2,1,1,2,1,3,1,1,1,2,2,2,3,6,1,2,1,2,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    2,2,1,1,1,3,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,4,25,96,224,1,27,35,1,2,3,3,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,5,1,1,2,1,2,1,52,48,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,9,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,1,1,2,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,3,1,7,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,81,39,6008,44,54,10,
    149,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,7,5,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,
    1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,6,8,8,6,10,8,
    8,8,8,6,10,8,8,8,8,14,2,8,8,8,8,8,8,5,1,2,6,1,3,3,1,2,8,4,2,2,
    8,8,10,3,1,2,121,1,13,1,138,1,3,2,3,1,27,1,4,1,4,1,3,1,8,4,55,
    734,7,12,5,1065,26,1229,40,53194,26,26,7,1,18,26,26,26,4,1,1,
    1,7,1,11,26,26,26,26,26,26,26,26,26,26,26,26,26,26,26,26,26,2,
    6,30,25,1,6,26,25,1,6,26,25,1,6,26,25,1,6,26,25,1,6,993334
};

static Parrot_UInt unicode_alpha[] = {
    65,26,6,26,47,1,10,1,4,1,5,23,1,31,1,195,1,4,4,1,1,2,1,2,1,3,
    8,1,68,25,96,214,1,1,3,1,1,1,20,1,44,1,38,1,5,4,130,8,69,1,38,
    2,2,6,16,33,38,10,39,2840,38,3130,44,54,10,148,156,4,90,6,22,
    2,6,2,38,2,6,2,8,1,1,1,1,1,1,1,31,2,8,8,8,8,8,8,5,1,6,2,1,3,3,
    1,6,4,4,2,6,4,13,5,3,1,6,117,1,13,1,130,1,4,1,2,10,1,1,3,5,6,
    1,1,1,1,1,1,4,1,3,1,2,4,1,3,3,5,5,55734,7,12,5,1033,26,6,26,1,
    189,80,53168,85,1,71,1,2,2,1,2,2,2,4,1,12,1,1,1,7,1,65,1,4,2,
    8,1,7,1,28,1,4,1,5,1,1,3,7,1,338,4,25,1,25,1,31,1,25,1,31,1,2,
    5,1,31,1,25,1,31,1,25,1,6,993334
};

static Parrot_UInt unicode_alnum[] = {
    48,10,7,26,6,26,47,1,10,1,4,1,5,23,1,31,1,195,1,4,4,1,1,2,1,
    2,1,38,1,68,25,96,214,1,1,3,1,1,1,20,1,44,1,38,1,5,4,130,8,69,
    1,38,2,2,6,16,33,38,10,39,216,10,134,10,620,10,118,10,118,10,
    118,10,118,10,119,9,118,10,118,10,118,10,224,10,118,10,70,10,
    278,10,86,38,675,9,1134,10,38,10,300,10,944,44,54,10,148,156,
    4,90,6,22,2,6,2,38,2,6,2,8,1,1,1,1,1,1,1,31,2,8,8,8,8,8,8,5,1,
    6,2,1,3,3,1,6,4,4,2,6,4,13,5,3,1,6,117,1,13,1,130,1,4,1,2,10,
    1,1,3,5,6,1,1,1,1,1,1,4,1,3,1,2,4,1,3,3,5,5,55734,7,12,5,1016,
    10,7,26,6,26,1189,80,80,10,53078,85,1,71,1,2,2,1,2,2,2,4,1,12,
    1,1,1,7,1,65,1,4,2,8,1,7,1,28,1,4,1,5,1,1,3,7,1,338,4,25,1,25,
    1,31,1,25,1,31,1,25,1,31,1,25,1,31,1,25,1,6,4,50,993280
};

static struct chartype_transcoder_entry_t unicode_transcoders[] = {
    { NULL, NULL, NULL }
};

/*

=item C<static int
inversion_list(Parrot_UInt *list, const Parrot_UInt c)>

This is an internal function called by C<unicode_is_charclass()> to
invert character C<c> using array C<*list>. The available invertions
currently are from/to upper/lower case and alpha/numeric.

=cut

*/

static int
inversion_list(Parrot_UInt *list, const Parrot_UInt c)
{
    int i;
    Parrot_UInt t = 0;

    for (i=0; c>=t; i++) {
        t += list[i];
    }
    return !(i & 1);
}

/*

=item C<static INTVAL
unicode_is_charclass(const struct parrot_chartype_t *type, const Parrot_UInt c, 
                     const unsigned int class)>

Returns whether C<c> is a member of character class C<class>.

This raises an exception of C<class> is and unknown character class.

=cut

*/

static INTVAL
unicode_is_charclass(const struct parrot_chartype_t *type, const Parrot_UInt c, 
                     const unsigned int class)
{
    switch (class) {
        case enum_charclass_digit:
            return chartype_is_digit_mapn(type, c, class);
        case enum_charclass_upper:
            return inversion_list(unicode_upper, c);
        case enum_charclass_lower:
            return inversion_list(unicode_lower, c);
        case enum_charclass_alpha:
            return inversion_list(unicode_alpha, c);
        case enum_charclass_alnum:
            return inversion_list(unicode_alnum, c);
    }
    internal_exception(INVALID_CHARCLASS, "Unknown character class <%d>\n", 
                       class);
    return 0;
}

CHARTYPE unicode_chartype = {
    enum_chartype_unicode,
    "unicode",
    "utf32",
    { 
        chartype_is_digit_mapn,      /* is_digit() */
        unicode_is_charclass         /* is_charclass() - slow version */
    },
    chartype_get_digit_mapn,         /* get_digit() */
    unicode_digit_map,               /* digit_map */
    NULL,                            /* unicode_map */
    &chartype_transcode_nop,         /* from_unicode() */
    &chartype_transcode_nop,         /* to_unicode() */
    NULL                             /* other transcoders */
};

/*

=back

=head1 SEE ALSO

F<chartypes/usacsii.c>,
F<src/chartype.c>,
F<include/parrot/chartype.h>,
F<config/gen/parrot_include.pl>,
F<docs/strings.pod>.

=cut

*/

/*
 * Local variables:
 * c-indentation-style: bsd
 * c-basic-offset: 4
 * indent-tabs-mode: nil
 * End:
 *
 * vim: expandtab shiftwidth=4:
*/
