# App::hopen::Gen::Make - generator for a generic make(1).
package App::hopen::Gen::Make;
use strict; use warnings;
use Data::Hopen::Base;

our $VERSION = '0.000014'; # TRIAL

use parent 'App::hopen::Gen';
use Class::Tiny;

use App::hopen::BuildSystemGlobals;
use App::hopen::Phases qw(is_last_phase);
use Data::Hopen qw(:default getparameters *QUIET);
use Data::Hopen::Scope::Hash;
use Data::Hopen::Util::Data qw(forward_opts);
use File::Which;
use Quote::Code;

use App::hopen::Gen::Make::AssetGraphNode;     # for $OUTPUT

# Docs {{{1

=head1 NAME

App::hopen::Gen::Make - hopen generator for simple Makefiles

=head1 SYNOPSIS

This generator makes a Makefile that does its best to run on cmd.exe or sh(1).

=head1 FUNCTIONS

=cut

# }}}1

=head2 finalize

Write out the Makefile.  Usage:

    $Generator->finalize($dag);     # $data parameter unused

C<$dag> is the build graph.

=cut

sub finalize {
    my ($self, %args) = getparameters('self', [qw(dag; data)], @_);
    hlog { Finalizing => __PACKAGE__ , '- phase', $Phase };
    return unless is_last_phase $Phase;

    hlog { __PACKAGE__, 'Asset graph', '' . $self->_assets->_graph } 3;

    # During the Gen phase, create the Makefile
    open my $fh, '>', $self->dest_dir->file('Makefile') or die "Couldn't create Makefile";
    print $fh <<EOT;
# Makefile generated by hopen (https://github.com/hopenbuild/App-hopen)
# at @{[scalar gmtime]} GMT
# From ``@{[$self->proj_dir->absolute]}'' into ``@{[$self->dest_dir->absolute]}''

.PHONY: first__goal__
EOT

    # Make sure the first goal is 'all' regardless of order.
    say $fh qc'first__goal__: {$args{dag}->default_goal->name}\n';

    my $context = Data::Hopen::Scope::Hash->new;
    $context->put(App::hopen::Gen::Make::AssetGraphNode::OUTPUT, $fh);

    # Write the Makefile.  TODO flip the order.

    $self->_assets->run(-context => $context);

    close $fh;
} #finalize()

=head2 default_toolset

Returns the package name of the default toolset for this generator,
which is C<Gnu> (i.e., L<App::hopen::T::Gnu>).

=cut

sub default_toolset { 'Gnu' }

=head2 _assetop_class

The class of asset-graph operations, which in this case is
L<App::hopen::Gen::Make::AssetGraphNode>.

=cut

sub _assetop_class { 'App::hopen::Gen::Make::AssetGraphNode' }

=head2 _run_build

Implementation of L<App::hopen::Gen/run_build>.

=cut

sub _run_build {
    # Look for the make(1) executable.  Listing make before gmake since a
    # system with both Cygwin and Strawberry Perl installed has cygwin's
    # make(1) and Strawberry's gmake(1).
    foreach my $candidate (qw[make gmake mingw32-make dmake]) {
        my $path = File::Which::which($candidate);
        next unless defined $path;
        hlog { Running => $path };
        system $path, ();
        return;
    }
    warn "Could not find a 'make' program to run";
} #_run_build()

1;
__END__
# vi: set fdm=marker: #
