/* cstate.c
 *
 * Export and import the state of the ML system for sml2c programs
 * Created: 11/5/90 by David Tarditi
 *
 *
 * The format of our export files:
 *        magic number (octal 123)
 *        time stamp associated with the time of creation of the C code
 *        MLState vector
 *        signal status vector
 *        current signal handler (sighandler0[1])
 *        heapsize 
 *        heap
 *
 */

#include <stdio.h>
#include <sys/file.h>
#include <errno.h>
extern int errno;   /* some header files are missing these declarations */
extern char *sys_errlist[];
extern int sys_nerr;

#include "ml_state.h"
#include "ml_types.h"

extern int old_high,arenabase,arstart;
extern MLState_ptr Exporters_State;
extern char *sml2c_timestamp;
extern int sighandler0[];
/* copied from signals.c */

extern struct siginfo_t {	/* Info about the ML signals */
    char	    unix_code;	    /* the unix signal code of this signal */
    char	    state;	    /* the state of this signal. */
    char            default_action; /* what to do when disabled */
} siginfo[NUM_ML_SIGS];


extern void restart_ml(),restart_gc(),init_externlist();
static void import();
static int bulletproofWrite();
static void bulletproofRead();
static char *errorMsg();
static int completeRead();

#define HEAPCMAGIC 0123
static int magicNum = HEAPCMAGIC;

#ifdef DEBUG
static void gc_info()
{  MLState_ptr msp = Exporters_State;
  fprintf(stderr,"Arena base = %d\n",arenabase);
  fprintf(stderr,"Arena start = %d\n",arstart);
  fprintf(stderr,"Old high = %d\n",old_high);
  fprintf(stderr,"Alloc ptr = %d\n",msp->ml_allocptr);
}
#endif

export(filid)
    int filid;
{ int size = old_high - arenabase;
#ifdef DEBUG
  gc_info();
#endif
  if (bulletproofWrite(filid,&magicNum,sizeof(magicNum)))
    if (bulletproofWrite(filid,sml2c_timestamp,strlen(sml2c_timestamp)))
     if (bulletproofWrite(filid,&Exporters_State,sizeof(MLState_ptr)))
       if (bulletproofWrite(filid,Exporters_State,sizeof(MLState_t)))
         if (bulletproofWrite(filid,siginfo,sizeof(siginfo)))
           if (bulletproofWrite(filid,&(sighandler0[1]),sizeof(sighandler0[1])))
             if (bulletproofWrite(filid,&size,sizeof(size)))
               return(!bulletproofWrite(filid,arenabase,size));
  return(1);
}

load_heap(s)
char *s;
{  int fd = -1;
   fd = open(s, O_RDONLY, 0666);
   if (fd < 0) 
     die("[Can't open heap file %s: %s]\n",s,errorMsg());
   import(fd,s);
   close(fd);
#ifdef DEBUG
   gc_info();
#endif
   init_externlist();
   restart_ml();
 }

static void import(filid,s)
    int filid;
    char *s;
{ int heapsize,heapMagic;
  char time[128];
  int len = strlen(sml2c_timestamp);
/*
  printf("sizeof(siginfo)=%d\n",sizeof(siginfo));
  printf("siginfo = %d\n",siginfo);
*/
  bulletproofRead(filid,&heapMagic,sizeof(heapMagic));
  if (heapMagic != magicNum)
       die("[%s is not a valid heap file]\n",s);
  bulletproofRead(filid,time,len);
  if (strncmp(time,sml2c_timestamp,len))
       die("[%s is not a valid heap file for this binary]\n",s);
  bulletproofRead(filid,&Exporters_State,sizeof(MLState_ptr));
  bulletproofRead(filid,Exporters_State,sizeof(MLState_t));
  bulletproofRead(filid,siginfo,sizeof(siginfo));
  bulletproofRead(filid,&(sighandler0[1]),sizeof(sighandler0[1]));
  bulletproofRead(filid,&heapsize,sizeof(heapsize));

  /* initialize heap */

  restart_c_gc(Exporters_State,heapsize);
  bulletproofRead(filid,arenabase,heapsize);
  Exporters_State->ml_pc = CODE_ADDR(Exporters_State->ml_cont);
}

/* bulletproofRead:
 * Read nbytes from buffer to the file fd, taking care of incomplete reads.
 *
 * This is to be used *only* for loading heaps, since it dies if an error
 * occurs.
 */

static void bulletproofRead (fd, buf, nbytes)
    int		fd;
    char	*buf;
    int		nbytes;
{
    int		bytesRead = 0, i;
#ifdef DEBUG
    fprintf(stderr,"Reading %d bytes into %d\n",nbytes,buf);
#endif
    do {
	if ((i = read(fd, buf, nbytes-bytesRead)) <= 0)
	      if (i == 0)
		  die("[Unexpected eof in loading heap\n");
	      else
		  die("[Error loading heap: %s]\n",errorMsg);
	bytesRead += i;
	buf += i;
	if (bytesRead < nbytes)
	    chatting("[Read incomplete (%d/%d), retrying]\n", bytesRead, nbytes);
    } while (bytesRead < nbytes);

} /* end of bulletproofRead */

static int completeRead(fd)
     int fd;
{ char buf[8];
  return (!read(fd,buf,1));
}
    
/* bulletproofWrite:
 * Write nbytes from buffer to the file fd, taking care of incomplete writes.
 *
 * Returns 1 if successful, 0 if an error occurred.
 */

static int bulletproofWrite (fd, buf, nbytes)
    int		fd;
    char	*buf;
    int		nbytes;
{
    int		bytesWritten = 0, i;
#ifdef DEBUG
    fprintf(stderr,"Writing %d bytes out from %d\n",nbytes,buf);
#endif
    do {
	if ((i = write(fd, buf, nbytes-bytesWritten)) < 0)
	   { fprintf(stderr,"[Error while exporting heap: %s]\n",errorMsg());
	     goto exit;
           }
	bytesWritten += i;
	buf += i;
	if (bytesWritten < nbytes)
	    chatting("[Write incomplete (%d/%d), retrying]\n", bytesWritten, nbytes);
    } while (bytesWritten < nbytes);
    return(1);
exit: return(0);

} /* end of bulletproofWrite */

static char *errorMsg()
{ static char number[128];
  if (errno > sys_nerr) {
          sprintf(number,"errno: %d",errno);
	  return number;
       }
  else return (sys_errlist[errno]);
}

