(* Copyright (c) 1991 by Carnegie Mellon University *)

(* machine.sml
 *
 *  A C code target machine for CPS programs
 *
 * Author: David Tarditi
 *         Carnegie Mellon University
 *         tarditi@cs.cmu.edu
 *
 * This program provides a set of functions for creating instructions for
 * a C code machine.  The semantics of these functions are described in
 * machine.sig.  The functions are used in cpstoc.sml.
 *)
            
functor CMachineFun(structure Control : CONTROL
		    structure C : C
		    sharing C.Bignum = Control.Bignum) : ANSICMACHINE =
   struct
      open Array   (* aargh. this stinks *)
      infix 9 sub

      open C Control

      val tailcalls = ref 0
      val totalcalls = ref 0
      val arithopers = ref 0
      val arithopts = ref 0
      val zerochecks = ref 0
      val totalchecks = ref 0

      val statistics = 
	  let val say = outputc std_out
	      val pct = fn (label,a,b) =>
		   if b<>0 then
		     say ("\n" ^label^(makestring ((100 * a) div b)))
		   else ()
	       val sayint = say o (makestring : int -> string)
          in fn _ =>
	      (say "function call sites = ";
	       sayint (!totalcalls);
	       say "\nimmediate tail call sites = ";
	       sayint (!tailcalls);
	       pct("% immediate tail calls",!tailcalls,!totalcalls);
	       say "\narithmetic operations = ";
	       sayint (!arithopers);
	       say "\noptimized arithmetic operations = ";
	       sayint (!arithopts);
               pct("\n% optimized arithmetic operations = ",
		   !arithopts,!arithopers);
	       say "\npotential heap checks = ";
	       sayint (!totalchecks);
	       say "\neliminated heap checks = ";
	       sayint (!zerochecks);
	       pct ("\n%optimized heap checks = ",!zerochecks,!totalchecks);
	       say "\n")
            end

      val inc = fn a => (a := (!a)+1)
 
      (* dist: create a list of integer between start and finish *)

      fun dist start finish =
	 if start <= finish then start :: dist (start+1) finish
	 else nil

      (* sublist: make a new list of elements satisfying pred *)

      fun sublist (pred : 'a->bool) =
	   let fun f nil = nil
		 | f (h::t) = if pred h then h :: f t else f t
	    in f
	    end

      (* registers *)

      val numRegs = 31 (* 9 dedicated registers, 22 misc + standard calling
                          convention registers *)
      val limitRegister = 2
      val signalLimitRegister = 3
      val dataRegister = 5
      val PCregister = 7

      val arithTemp = R 0
      val tempEA = R 1
      val limitPtr = R limitRegister
      val signalLimitPtr = R signalLimitRegister
      val storePtr = R 4
      val dataPtr = R dataRegister
      val exnPtr = R 6
      val varPtr = R 8
      val standardClosure = R 9
      val standardArg = R 10
      val standardCont = R 11


      val miscRegs = map R (dist 12 (numRegs - 1))

      val allRegs = map R (dist 0 (numRegs-1))

      val localRegs = dataPtr :: standardClosure :: standardArg ::
	              standardCont :: tempEA :: arithTemp :: miscRegs

      val tagInt = fn i =>  i*2+1
      val unTagInteger : EA -> int option =
	  fn (IMMED i) => SOME (Bits.rshift(i,1))
           | _ => NONE

      val mkLabel =
	 let val count = ref 0
         in fn () => (count := !count + 1; !count)
         end

      val immedEA = IMMED
      val functionEA = N o mkLabel
      val namedFunctionEA = NAMED
      val stringEA = S o mkLabel
      val realEA = REAL o mkLabel

      (* global variables for dynamic profiling *)

      val jct = NAMED "_s2c_jc"   (* count of jumps *)
      val gct = NAMED "_s2c_gc"   (* count of gotos *)
      val ict = NAMED "_s2c_ic"   (* count of instructions *)
      val wct = NAMED "_s2c_wc"   (* writes to pseudo-registers *)
      val rct = NAMED "_s2c_rc"   (* reads from pseudo-registers *)
      val hct = NAMED "_s2c_hc"   (* count of heap checks *)

      val immed1 = IMMED 1

      val instruction_count : stmt -> stmt = 
	 let val aplength : CPS.accesspath -> int =
		 let open CPS
		     fun f (OFFp 0,i) = i
                       | f (OFFp _,i) = i+1
                       | f (SELp (_,b),i) = f(b,i+1)
                 in fn x => f(x,0)
                 end
                      
             fun ic (ASSIGN(a,b,s),ct) = ASSIGN (a,b,ic(s,ct+1))
               | ic (ALLOC(a,b,c),ct) =
                   (* cost of alloc = reads (r) + writes (w), where
		          r = sum of length of access paths for all
			      elements
                          w = # of elements *)
                     
		      ALLOC(a,b,
			      ic(c,fold (fn ((addr,path),count) =>
			            (aplength path+1)+count) a ct))
               | ic (CALL(a,b,c,d),ct) = CALL(a,b,c,ic(d,ct+1))
               | ic (CASE(a,branches),ct) =
		        CASE(a,map (fn (i,s) => (i,ic(s,ct+1))) branches)
               | ic (COND(a,b,c),ct) = COND(a,ic(b,ct+1),ic(c,ct+1))
               | ic (COMMENT(a,b),ct) = COMMENT(a,ic(b,ct))
               | ic (FETCH(a,b,c,d),ct) = FETCH(a,b,c,ic(d,ct+1))
               | ic (FETCHB(a,b,c,d),ct) = FETCHB(a,b,c,ic(d,ct+1))
               | ic (FLOAT(a,b,c,d,e),ct) = FLOAT(a,b,c,d,ic(e,ct+1))
               | ic (s as GOTO l,ct) =  INT2OP(ADD,ict,IMMED (ct+1),ict,s)
               | ic (INT2OP(a,b,c,d,e),ct) = INT2OP(a,b,c,d,ic(e,ct+1))
               | ic (INT1OP(a,b,c,d),ct) = INT1OP(a,b,c,ic(d,ct+1))
               | ic (s as JMP a,ct) =
		       INT2OP(ADD,jct,immed1,jct,
			         INT2OP(ADD,ict,IMMED (ct+1),ict,s))
	       | ic (LABEL(a,b),ct) = LABEL(a,ic(b,ct))
               | ic (SEQ s,ct) = SEQ (map (fn a=>ic(a,ct)) s)
               | ic (SET(a,b,c,d),ct) = SET(a,b,c,ic(d,ct+1))
               | ic (SETB(a,b,c,d),ct) = SETB(a,b,c,ic(d,ct+1))
       in fn s => if !instrum then ic(s,0) else s
       end


        (* count_ma: insert code to count memory loads/stores for
           pseudo-registers *)

        val count_ma : stmt -> stmt =
	    let val cost = fn R _ => 1 | _ => 0
                val costlist = 
		      let fun f(nil,i) = i
                            | f(h::t,i) = f(t,cost h+i)
                      in fn l=>f(l,0)
                      end

                (* expression cost *)

		fun ec (VAL ea) = cost ea
	          | ec (IB (oper,a,b)) = ec a + ec b
		  | ec (IU (oper,a)) = ec a
                  | ec (ICOND (c,a,b)) = ec a + ec b
		  | ec (FCOND (c,a,b)) = cost a + cost b
		  | ec (CAND (a,b)) = ec a + ec b

	        fun f (ASSIGN (a,b,c),{r,w}) = 
		             ASSIGN(a,b,f(c,{r=r+cost a,w=w+cost b}))
		  | f (ALLOC (elems,dest,c),{r,w}) =
		         let val reads =
			        fold (fn ((ea,_),r) => r+cost ea) elems 0
                         in ALLOC(elems,dest,f(c,{r=r+reads,w=w+cost dest}))
			 end
                  | f (CASE (r,cases),cost) =
		             CASE(r,map (fn (i,a) => (i,f(a,cost))) cases)
		  | f (CALL (a as (_,spill,load),b,c,d),{r,w}) =
		        CALL(a,b,c,f(d,{r=r+costlist load+costlist b,
					w=costlist spill+
					  (case c
					   of NONE=>w
			                    | SOME i=>w+cost i)}))
                  | f (COND (test,a,b),{r,w}) = 
		              let val reads = ec test
				  val cost = {r=r+reads,w=w}
			      in COND(test,f(a,cost),f(b,cost))
			      end
                  | f (COMMENT (c,s),cost) = COMMENT(c,f(s,cost))
		  | f (FETCH(a,b,c,d),{r,w}) =
		           FETCH(a,b,c,f(d,{r=cost a+cost b+r,w=cost c+w}))
		  | f (FETCHB(a,b,c,d),{r,w}) =
		           FETCHB(a,b,c,f(d,{r=cost a+cost b+r,w=cost c+w}))
                  | f (FLOAT (a,b,c,d,e),{r,w}) = 
		           FLOAT(a,b,c,d,f(e,{r=cost b+cost c+r,w=cost d+w}))
		  | f (INT2OP (i,a,b,c,d),{r,w}) =
		           INT2OP(i,a,b,c,f(d,{r=cost a+cost b+r,w=cost c+w}))
		  | f (INT1OP (i,a,b,c),{r,w}) =
		           INT1OP(i,a,b,f(c,{r=cost a,w=cost b+w}))
                  | f (s as GOTO _,{r,w}) =
		         let val s' = if w=0
				        then s
				        else INT2OP(ADD,wct,IMMED w,wct,s)
			     val s'' = if r=0
				        then s
				        else INT2OP(ADD,rct,IMMED r,rct,s')
                         in INT2OP(ADD,gct,immed1,gct,s'')
                         end
                  | f (s as (JMP (dest :: args)),{r,w}) =
		         let val r=r+cost dest
			     val s' = if w=0
				        then s
				        else INT2OP(ADD,wct,IMMED w,wct,s)
			     val s'' = if r=0
				        then s
				        else INT2OP(ADD,rct,IMMED r,rct,s')
                         in s''
                         end
		  | f (LABEL (l,s),cost) = LABEL(l,f(s,cost))
		  | f (SET (a,b,c,d),{r,w}) = 
		         SET(a,b,c,f(d,{r=cost a+cost b+cost c+r,w=w}))
		  | f (SETB (a,b,c,d),{r,w}) = 
		         SETB(a,b,c,f(d,{r=cost a+cost b+cost c+r,w=w}))
		  | f (SEQ l,cost) = SEQ(map (fn a=>f(a,cost)) l)
                  | f (a,_) = a
           in fn s => if !instrum then f(s,{r=0,w=0}) else s
           end

      local
         val substMap = arrayoflist allRegs
         val _ = app (fn (R i) => update(substMap,i,LOCAL i)) localRegs
         val shadowMap = array(numRegs,false)
         val _ = app (fn (R i) => update(shadowMap,i,true)) localRegs
      in
         val subst = fn R i => substMap sub i | a => a
         val shadow = fn R i => shadowMap sub i | _ => false
      end

        (* rewrite a statement, replacing variables with shadow variables
           and replacing jumps with gotos when possible *)

        fun rewrite (s:stmt, subst : EA -> EA, localvar : EA -> bool,
		     gotosubst : EA -> label option) : stmt =
	    let val subst = fn e =>
 		      if localvar e then subst e else e
		val islocalDataPtr = localvar dataPtr
		val localDataPtr = subst dataPtr
		fun e (VAL ea) = VAL (subst ea)
	          | e (IB (oper,a,b)) = IB (oper,e a,e b)
		  | e (IU (oper,a)) = IU (oper,e a)
                  | e (ICOND (c,a,b)) = ICOND(c,e a,e b)
		  | e (FCOND (c,a,b)) = FCOND(c,subst a,subst b)

		  | e (CAND (a,b)) = CAND(e a,e b)

	        fun f (ASSIGN (a,b,c)) = ASSIGN(subst a,subst b,f c)
		  | f (ALLOC (elems,dest,c)) =
		       ALLOC (map (fn (a,b)=>(subst a,b)) elems,subst dest,f c)
                  | f (CASE (r,cases)) =
		             CASE (subst r,map (fn (a,b) => (a,f b)) cases)
		  | f (CALL (a as (_,spill,load),b,c,d)) =
		       fold (fn (arg,rest) =>
		             if localvar arg
				then ASSIGN (subst arg,arg,rest)
		                else rest) spill
			(CALL (a,map subst b,
			       case c
			       of NONE => NONE
			        | SOME s => SOME (subst s),
			    fold (fn (arg,rest)=> ASSIGN(arg,subst arg,rest))
			         (sublist localvar load) (f d)))
                  | f (COND (test,a,b)) = COND(e test,f a,f b)
                  | f (COMMENT (c,s)) = COMMENT(c,f s)
		  | f (FETCH(a,b,c,d)) = FETCH(subst a,subst b,subst c,f d)
                  | f (FETCHB (a,b,c,d)) = FETCHB(subst a,subst b,subst c,f d)
                  | f (FLOAT (a,b,c,d,e)) = 
		          if islocalDataPtr then
			       ASSIGN(localDataPtr,dataPtr,
				    FLOAT(a,subst b,subst c,subst d,f e))
			  else
			       FLOAT(a,subst b, subst c, subst d, f e)
		  | f (INT2OP  (DIV,a,b,c,d)) =
		          if islocalDataPtr then
			        ASSIGN(localDataPtr,dataPtr,
				     INT2OP(DIV, subst a,subst b, subst c,f d))
			  else INT2OP(DIV, subst a,subst b, subst c,f d)
		  | f (INT2OP (i,a,b,c,d)) =
		             INT2OP(i,subst a,subst b,subst c,f d)
		  | f (INT1OP (i,a,b,c)) =
                              INT1OP(i,subst a,subst b,f c)
                  | f (JMP (dest :: args)) =
		        (case gotosubst dest
			 of NONE =>
	                      fold (fn (arg,rest) =>
		                   if localvar arg
				     then ASSIGN (subst arg,arg,rest)
		                     else rest)
			           (dataPtr :: args) 
				   (JMP [subst dest])
			 | SOME i => (inc tailcalls; GOTO i))
		  | f (LABEL (l,s)) = LABEL(l,f s)
		  | f (SET (a,b,c,d)) = SET(subst a,subst b,subst c,f d)
		  | f (SETB (a,b,c,d)) = SETB(subst a,subst b,subst c, f d)
		  | f (SEQ l) = SEQ(map f l)
		  | f arg = arg
           in f s
           end

        (* cache: compute which registers to cache in a statement *)

        fun cache (s : stmt) : (EA -> bool) =
           let val memo = array(numRegs,0)
	       val inc = fn (R i,memo) => update(memo,i,(memo sub i)+1)
		          | _ => ()
	       val dec = fn (R i,memo) => update(memo,i,(memo sub i)-1)
		          | _ => ()
	       val max = fn (a,b) =>
		   let fun f i =
		       if i<numRegs then (update(a,i,max(a sub i,b sub i));
					  f (i+1))
		       else ()
		    in f 0
		    end
		val add = fn (a,b) =>
		    let fun f i =
			if i<numRegs then (update(a,i,(a sub i) + (b sub i));
					   f (i+1))
			else ()
		    in f 0
		    end
	      fun branches (parent,cases) =
			let val memos = map (fn a=>(a,array(numRegs,0))) cases
			in app f memos;
			   case memos
			   of nil => ()
			 | (_,h) :: t => (app (fn (_,a) => max(h,a)) t;
				      add (parent,h))
			end             
	      and e (VAL a,m) = inc(a,m)
	        | e (IB (_,a,b),m) = (e(a,m); e(b,m))
                | e (IU (_,a),m) = e(a,m)
		| e (ICOND (_,a,b),m) = (e(a,m); e(b,m))
		| e (FCOND (_,a,b),m) = (inc(a,m); inc(b,m))
	        | e (CAND  (a,b),m) = (e(a,m); e(b,m))
              and f (ASSIGN (a,b,c),m) = (inc(a,m); inc(b,m); f(c,m))
		| f (ALLOC (elems,b,c),m) = 
		               (app (fn (a,_)=>inc(a,m)) elems;
			        inc (b,m); f(c,m))
		| f (CASE (r,cases),m) =
		       (inc(r,m);
			branches(m,map #2 cases))
	        | f (CALL (_,b,NONE,d),m) = (app (fn a=> inc(a,m)) b; f(d,m))
	        | f (CALL (_,b,SOME c,d),m) = (app (fn a=> inc(a,m)) b;
					     inc (c,m); f(d,m))
		| f (COND (test,a,b),m) =
		      (e(test,m);
		       branches(m,[a,b]))
                | f (COMMENT (_,s),m) = f(s,m)
	        | f (FETCH (a,b,c,d),m) = (inc(a,m);inc(b,m);inc(c,m);f(d,m))
		| f (FETCHB (a,b,c,d),m) =
		      (inc(a,m); inc(b,m); inc(c,m); f(d,m))
	        | f (FLOAT (a,b,c,d,e),m) =
		      (inc(b,m); inc(c,m);  inc(d,m); f(e,m))
	        | f (INT2OP (_,a,b,c,d),m) =
		      (inc(a,m); inc(b,m); inc(c,m); f(d,m))
	        | f (INT1OP (_,a,b,c),m) = (inc(a,m); inc(b,m); f(c,m))
		| f (JMP (a :: b),m) = (inc(a,m); app (fn a=>dec(a,m)) b)
		| f (LABEL (_,s),m) = f(s,m)
		| f (SET (a,b,c,d),m) = (inc(a,m); inc(b,m); inc(c,m); f(d,m))
	        | f (SETB (a,b,c,d),m) = (inc(a,m); inc(b,m); inc(c,m); f(d,m))
                | f (SEQ l,m) = branches(m,l)
		| f _ = ()
	   in f(s,memo);
	      fn (R i) => (memo sub i) > !usageCount
	       | _ => false
          end
                
        (* hascall: return true if a statement has a call to name in it *)

       fun hascall (s : stmt,name : EA) =
	 let fun checklist (h::t) = f h orelse checklist t
	       | checklist nil = false
             and checkcases ((_,stmt)::t) = f stmt orelse checkcases t
	       | checkcases nil = false
	     and f (ASSIGN (_,_,a)) = f a
	       | f (ALLOC  (_,_,a)) = f a
	       | f (CASE (r,cases)) = checkcases cases
	       | f (CALL (_,_,_,a)) = f a
	       | f (COND (_,a,b)) = checklist[a,b]
               | f (COMMENT (_,a)) = f a
	       | f (FETCH (_,_,_,a)) = f a
	       | f (FETCHB (_,_,_,a)) = f a
	       | f (FLOAT (_,_,_,_,a)) = f a
	       | f (INT2OP (_,_,_,_,a)) = f a
	       | f (INT1OP (_,_,_,a)) = f a
               | f (GOTO l) = (case name
		               of N l' => l=l'
			        | _ => false)
	       | f (JMP (a::_)) = if a=name then true else false
	       | f (LABEL (_,a)) = f s
	       | f (SET (_,_,_,a)) = f a
	       | f (SETB (_,_,_,a)) = f a
	       | f (SEQ l) = checklist l
	       | f _ = false
         in f s
	 end

        (* constructing declarations *)

	val mkString = STRING_DECL
	val mkReal = REAL_DECL

        val immed0 = IMMED 0
        val immed1 = IMMED 1
        val immed3 = IMMED 3

        (* construct statements *)

	val makeFloatOp : floatOp -> EA * EA * EA * stmt -> stmt =
	   let val floattag = IMMED (System.Tags.make_desc(floatSize,
				       System.Tags.tag_string))
               val immedFloatSize = IMMED(floatSize div 4+1)
	   in fn floatOp =>
		fn (ea1,ea2,dest,cont) => 
		    ALLOC ([(floattag,CPS.OFFp 0)],dataPtr,
		      FLOAT(floatOp,dataPtr,ea1,ea2,
			INT2OP(PTRADD,dataPtr,immed1,dest,
			  INT2OP(PTRADD,dataPtr,immedFloatSize,dataPtr,cont))))
           end

	val ashl = fn (a,b,dest,cont) => INT2OP(ASHL,b,a,dest,cont)
	val ashr = fn (a,b,dest,cont) => INT2OP(ASHR,b,a,dest,cont)
	val orb = fn (a,b,dest,cont) => INT2OP(ORB,a,b,dest,cont)
	val andb = fn (a,b,dest,cont) => INT2OP(ANDB,a,b,dest,cont)
	val xorb = fn (a,b,dest,cont) => INT2OP(XORB,a,b,dest,cont)
	val notb = fn (a,dest,cont) => INT1OP(NOTB,a,dest,cont)

	val addi = fn (a,b,dest,cont) => INT2OP(ADD,a,b,dest,cont)
	val subi = fn (a,b,dest,cont) => INT2OP(SUB,b,a,dest,cont)

	val mulf = makeFloatOp FMUL
	val divf = makeFloatOp FDIV
	val addf = makeFloatOp FADD
	val subf = makeFloatOp FSUB

        val mkCase = CASE
	val comment = COMMENT

        val fetchIndex = 
	       fn (from,to,index,cont) => FETCH(from,index,to,cont)

        val fetchIndexB = 
	   fn (from,to,index,cont) => FETCHB(from,index,to,cont)

        val bbs =
	    fn (n,from,stmt1,stmt2) =>
 	        COND(IB(ANDB,VAL from,VAL (IMMED (Bits.lshift (1,n)))),
		     stmt2,stmt1)

	val gbranch =
	   fn (args as (cond,ea1,ea2,stmt1,stmt2)) =>
		   COND(FCOND(cond,ea1,ea2),stmt1,stmt2)

	val ibranch =
	   fn (args as (cond,ea1,ea2,stmt1,stmt2)) =>
		   COND(ICOND(cond,VAL ea1,VAL ea2),stmt1,stmt2)


	val jmp=  fn a => (inc totalcalls; JMP a)

	val offset : int * EA * EA * stmt -> stmt =
	   fn (i,ea as R r,dest,cont) => INT2OP(PTRADD,ea,IMMED i,dest,cont)
	    | _ => ErrorMsg.impossible "c/machine.sml: 844"

        (* test whether 0 <= a < b.  Used unsigned integer comparison *)

        val rangeChk =
	    fn (args as (ea1,ea2,stmt1,stmt2)) =>
		 COND(ICOND(ULT,VAL ea1,VAL ea2),stmt1,stmt2)

        val move = ASSIGN

	val record : (EA * CPS.accesspath) list * EA * stmt -> stmt =
	  fn (vl,dest as (R _),cont) =>
	     ALLOC(vl,dataPtr,
		if !autoIncrement then
		    INT2OP(PTRADD,dataPtr,IMMED(~(List.length vl-1)),dest,cont)
		else
	         INT2OP(PTRADD,dataPtr,immed1,dest,
		    INT2OP(PTRADD,dataPtr,
				  IMMED (List.length vl),
				  dataPtr,cont)))
           | _ => ErrorMsg.impossible "c/machine.sml: 864"

	val select : int * EA * EA * stmt -> stmt =
	    fn (offset,src,dest,cont) =>
	        FETCH(src,(IMMED o tagInt) offset,dest,cont)


        val overflowexn = EXTERN_REC overflow
	val multFunc = (NAMED mult,[dataPtr],nil)
        val csp = NAMED "r"

        val overflowLabel = mkLabel()
        val overflowCode =
	    LABEL(overflowLabel,
	       move(exnPtr,standardCont,
		     move(overflowexn,standardArg,
		       select(0,standardCont,tempEA,
			   jmp[tempEA,standardArg,standardCont]))))

        val overflowCont = GOTO overflowLabel

        val val0 = VAL immed0

      (* Lack of overflow is r <- (a+b) if (eor(a,b)) | !(eor(a,r)) *)

        val addti = 
	 let open Bignum
	     val check = fn (c,ea,cont) =>
	       (inc totalcalls; inc arithopers; inc arithopts;
	        ((if Integer.<(c,0) then 
		  COND(ICOND(GEQ,VAL ea,VAL(BIGNUM(minint - inttobignum c))),
                     cont,overflowCont)
	        else if Integer.>(c,0) then
		  COND(ICOND(LEQ,VAL ea,VAL(BIGNUM(maxint - inttobignum c))),
		     cont,overflowCont)
		 else cont)
	          handle Overflow =>
                  ErrorMsg.impossible "c/machine.sml: addti: 1: overflow"))
	 in fn arg =>
	    if !unsafeArith
	     then case arg
		  of (ea1,ea2,dest,cont) => INT2OP(ADD,ea1,ea2,dest,cont)
	     else
	       case (arg,!arithOpt)
               of ((arg as (ea1 as (IMMED i),ea2,dest,cont)),true) =>
	            check(i,ea2,INT2OP(ADD,ea1,ea2,dest,cont))
	        | ((arg as (ea1,ea2 as (IMMED i),dest,cont)),true) =>
		    check(i,ea1,INT2OP(ADD,ea1,ea2,dest,cont))
	        | ((ea1,ea2,dest,cont),_) =>
	           (inc totalcalls; inc arithopers;
	            INT2OP(ADD,ea1,ea2,tempEA,
		      COND(ICOND(LT,IB(ORB,IB(XORB,VAL ea1,VAL ea2),
		          IU(NOTB,IB(XORB,VAL ea1,VAL tempEA))),
		        val0),
		     ASSIGN(tempEA,dest,cont),overflowCont)))
        end 

        (* subti(a,b,c): c <-(b-a) *)

        local open Bignum
        in val subti = 
	 fn args =>
	   if !unsafeArith then
	     case args
	      of (ea1,ea2,dest,cont) =>   INT2OP(SUB,ea2,ea1,dest,cont)
	   else
	    (inc totalcalls; inc arithopers;
	     case (args,!arithOpt)
	      of ((ea1 as IMMED i,ea2,dest,cont),true) =>
	       (inc arithopts;
	        if Integer.>(i,0) then
		   (COND(ICOND(GEQ,VAL ea2,VAL(BIGNUM(minint+inttobignum i))),
		         INT2OP(SUB,ea2,ea1,dest,cont),overflowCont))
	        else
		   (COND(ICOND(LEQ,VAL ea2,VAL(BIGNUM(maxint+inttobignum i))),
		    INT2OP(SUB,ea2,ea1,dest,cont),overflowCont)))
               | ((ea1,ea2 as IMMED i,dest,cont),_) =>
	          (inc arithopts;
	           if Integer.>=(i,0) then
		     COND(ICOND(GEQ,VAL ea1,VAL(BIGNUM(inttobignum i-maxint))),
		          INT2OP(SUB,ea2,ea1,dest,cont),overflowCont)
 	           else
	             COND(ICOND(LEQ,VAL ea1,VAL(BIGNUM(inttobignum i-minint))),
		         INT2OP(SUB,ea2,ea1,dest,cont),overflowCont))
               | ((ea1,ea2,dest,cont),_) =>
	            INT2OP(SUB,ea2,ea1,tempEA,
		       COND(ICOND(GEQ,IB(ANDB,IB(XORB,VAL ea1,VAL ea2),
		                          IB(XORB,VAL ea2,VAL tempEA)),
			         val0),
			   ASSIGN(tempEA,dest,cont),overflowCont))
           )
        end

        local open Bignum
	in
        fun multi arg =
	 if !unsafeArith then
	     case arg
	     of (ea1,ea2,dest,cont) => INT2OP(MUL,ea1,ea2,dest,cont)
         else
	    case (arg,!arithOpt)
            of ((ea1 as (IMMED i),ea2,dest,cont),true) =>
	      if i=0 then ASSIGN(immed0,dest,cont)
              else let open Bignum
		       val c = inttobignum i
		       val (min,max) =
		         if Integer.>(i,0) then (minint div c,maxint div c)
			 else (maxint div c,minint div c)
		       val arg2 = VAL ea2
		       val _ =  (inc totalcalls; inc arithopers;
				 inc arithopts)
		   in COND(CAND(ICOND(GEQ,arg2,VAL(BIGNUM min)),
				ICOND(LEQ,arg2,VAL(BIGNUM max))),
			INT2OP(MUL,ea1,ea2,dest,cont),overflowCont)
		   end
	  | ((ea1,ea2 as (IMMED _),dest,cont),true) => multi(ea2,ea1,dest,cont)
	  | ((ea1,ea2,dest,cont),_) =>
	       (inc totalcalls; inc arithopers;
		CALL(multFunc,[ea1,ea2,csp],SOME dest,cont))
        end

        (* overflow for division occurs only when we divide MININT by
	   -1.  Thus we need to check for overflow only when (a) dividing
	   two unknown numbers (b) dividing MININT by an unknown number
	   (c) dividing MAXINT by an unknown number

               divti computes dest <- b/a 
	*)
        
        fun divti (arg  as (a,b,dest,cont)) =
	   let val divop = INT2OP(DIV,b,a,dest,cont)
	   in if !unsafeArith then divop
	      else case (arg,!arithOpt)
	          of ((IMMED ~1,_,_,_),true) =>
		     COND(ICOND(NEQ,VAL b,VAL (BIGNUM minint)),divop,
			  overflowCont)
		   | ((IMMED i,_,_,_),true) => divop
		   | ((_,IMMED i,_,_),_) =>
		       if Bignum.inttobignum i=minint then
			  COND(ICOND(NEQ,VAL a,VAL (IMMED ~1)),divop,overflowCont)
		       else divop
	           | _ => COND(CAND(ICOND(EQL,VAL a,VAL (IMMED ~1)),
				 ICOND(EQL,VAL b,VAL (BIGNUM minint))),
			    overflowCont,divop)
           end

	val storeIndex = SET
        val storeIndexB = SETB

      (* Bodies of generated C functions have the following format:
                <declare local variables>
	   top: <load local variables>
	   label: <gc test> --true --> <stmt>
	            |
		   false  <invoke g.c. and goto top>
      *)

      local 
	  val heapValue = VAL dataPtr
          val limitValue = VAL limitPtr
          val signalLimitValue = VAL signalLimitPtr
 	  val invokeGC = NAMED "invoke_gc"
          val inlinedGC = NAMED "inlined_gc"
          val csp = NAMED "r"
          val overflowCall = N overflowLabel
	  val overflowCode' = [overflowCode]
      in val mkFunction =
	   fn nil => ErrorMsg.impossible "c/machine.sml: 1016"
	    | (funs as ({args,name,body,alloc,global,regmask}::localfuns)) =>
	    let val labels = map (fn {name,...} => (name,mkLabel())) funs
		val bodies = SEQ (map (fn {body,...}=> body) funs)

                (* find used variables *)

	        val iscached = cache bodies

                (* a variable should be made local if it is used enough
		    and it can be shadowed *)

                val islocalvar = 
		    if !regOpt then fn i => iscached i andalso shadow i
		    else fn i => false	

		  (* find list of local variables for this function *)

		val localvars =
		     if !regOpt then sublist iscached localRegs else nil

		val locals = 
		     case localvars of nil=>false | _ => true

		val top = if locals then mkLabel() else #2 (hd labels)

		val needOverflow = hascall(bodies,overflowCall)

		val mkstmt =
		  fn (label,istop,{args,name,body,alloc,global,regmask}) =>
		    let val body = instruction_count body
                    in

                 (* add heap check: there are 3 cases.  A function allocates
		    nothing, it allocates < 4096 words, it allocates > 4096
                    words *)

		     if alloc=0 andalso not (!zeroCheck) then
			  (inc totalchecks; inc zerochecks; LABEL(label,body))
		     else
		      let val gcspill = if istop andalso
			                   not (hascall(bodies,name))
			                then  nil else dataPtr :: args
			    val _ = inc totalchecks
			    val s =

	               (* the first labelled statement (the "top" statement)
			  is not integrated.  It can use the pseudo-limit
			  pointer *)
	      
			  if istop then
			    let val callgc =
			       CALL((invokeGC,gcspill,nil),
			          [immedEA regmask,csp,name],NONE,GOTO top)

                         (* integrated functions must use the real limit
			    pointer.*)

		            in if alloc<4096 then
			        COND(ICOND(LT,heapValue,signalLimitValue),
				     body,callgc)
			       else 
				 COND(ICOND(LT,IB(ADD,heapValue,
						    VAL(IMMED(alloc-4096))),
					        signalLimitValue),
				      body,callgc) 
			    end
			  else
			     let val callgc =
			        CALL((inlinedGC,gcspill,gcspill),
				     [immedEA regmask,csp],NONE,GOTO label)
			     in if alloc<4096 then
			           COND(ICOND(LT,heapValue,limitValue),
					body,callgc)
				else
				   COND(ICOND(LT,IB(ADD,heapValue,
						     VAL(IMMED(alloc-4096))),
					      limitValue),body,callgc)
			      end

			   val s = if !instrum then
			              INT2OP(ADD,hct,immed1,hct,s)
				   else s
		      in
		         LABEL(label,s)
		      end
                     end (* mkstmt *)

                val stmts = 
		  let fun f ((_,label)::r,stmt :: s,istop) =
			   mkstmt(label,istop,stmt)
                           :: f (r,s,false)
			| f (nil,nil,_) = if needOverflow then overflowCode'
					  else nil
                        | f _ = ErrorMsg.impossible "c/machine: 1019"
		  in SEQ(f(labels,funs,true))
		  end
		      
                (* gotosubst: goto substitution.  Returns label option to
		   substitute for a jmp destination *)

                val gotosubst = 
		    if (!tailOpt) orelse (!integOpt) then
		      fn a =>
		        let fun f ((name,label)::rest) =
			        if a=name then SOME label else f rest
			      | f nil = NONE
		         in f labels
		         end
		    else fn a => NONE

                (* rewrite statement, replacing pseudo-registers
		   with local variables and turning tail-recursive jumps 
		   into gotos *)

                val stmt' = 
		    if locals orelse !tailOpt orelse !integOpt then
			rewrite(stmts,subst,islocalvar,gotosubst)
                    else stmts

                (* add code to load locals *)

		val stmt' =
		    if locals
		    then LABEL(top,
			 let val localvars = sublist islocalvar
			                             (dataPtr :: args)
			 in fold (fn (r,rest) => move(r,subst r,rest))
			         localvars stmt'
			 end)
		    else stmt'
	    in FUNC(name,localvars,count_ma stmt',global)
	    end
      end

	val mkProg = PROG
end
