/*

Copyright 1994 Silicon Graphics, Inc. -- All Rights Reserved

If the Software is acquired by or on behalf of an entity of government
of  the  United States of America, the following provision applies: U.
S.  GOVERNMENT  RESTRICTED  RIGHTS  LEGEND:    Use,   duplication   or
disclosure of Software by the Government is subject to restrictions as
set forth in FAR 52.227-19(c)(2) or  subparagraph  (c)(1)(ii)  of  the
Rights  in  Technical  Data  and  Computer  Software  clause  at DFARS
252.227-7013 and/or in similar or successor clauses in the FAR, or the
DOD  or  NASA  FAR Supplement. Unpub-lished- rights reserved under the
Copyright  Laws  of  the  United  States.  Contractor/manufacturer  is
SILICON  GRAPHICS,  INC.,  2011  N. Shoreline Blvd., Mountain View, CA
94039- 7311.

Silicon Graphics, Inc. hereby grants  to  you  a  non-exclusive,  non-
transferable,  personal, paid-up license to use, modify and distribute
the Software solely with SGI computer products.  You must include,  in
all  copies  of  the  Software  and  any associated documentation, the
copyright notice and restricted rights legend set forth above.

THE SOFTWARE IS PROVIDED  TO  YOU  "AS-IS"  AND  WITHOUT  ANY  SUPPORT
OBLIGATION  OR  WARRANTY  OF  ANY KIND, EXPRESS, IMPLIED OR OTHERWISE,
INCLUDING WITHOUT  LIMITATION,  ANY  WARRANTY  OF  MERCHANTABILITY  OR
FITNESS  FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL SGI BE LIABLE FOR
SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF  LIABILITY,
ARISING  OUT  OF  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

You agree that you will not export or re-export the Software, directly
or  indirectly,  unless  (a)  the  Export  Administration of the U. S.
Department of Commerce explicitly permits the export or  re-export  of
the  Software  or  (b)  the  Office  of  Export Licensing of the U. S.
Department of Commerce has granted au-thorization to  you  in  writing
for the  export or re- export the Software.

If you fail to fulfill any  of  the  foregoing  obligations,  SGI  may
pursue  all  available  legal  remedies  to  enforce  these  terms and
conditions, and SGI may,  at  any  time  after  your  default  hereof,
terminate  the  license  and  rights  granted  to  you hereunder.  You
further agree that, if SGI terminates this license for  your  default,
you  will, within ten (10) days after any such termination, deliver to
SGI or  render  unusable  all  Software  originally  provided  to  you
hereunder and any copies thereof embodied in any medium.
*/

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "misInvDist.h"

#define Xdir 0
#define Ydir 1
#define Zdir 2

static int False = 0;
static int One   = 1;

/*
 * This routine computes an interpolating surface to N arbitrary points. It uses
 * an inverse weighting scheme such that any point on the surface is defines by:
 *
 *                N-1
 *                ----
 *                \        dist((x_i,y_j), (x_k, y_k)) * f_k
 *                / 
 *                ----
 *                k = 0
 *
 *  f(x_i,y_j) =  -----------------------------------------------
 *
 *                N-1
 *                ----
 *                \        dist (x_i,y_j), (x_k, y_k))
 *                / 
 *                ----
 *                k = 0
 *
 * Where the scattered points are defined as (x_k, y_k, f_k)
 */

void Interpolate2Ddata( coord *Data, int Npoints, int IndependentVarDirection,
                        coord *InterpolatedData, int xRes, int yRes )
{
   int      i, j, k;
   int     *f, *Df;

   float   *xCoords, *yCoords;

   float   *Datax, *Datay, *Dataf, *Datad;

   float    xMin, xMax, xRange;
   float    yMin, yMax, yRange;
   float    fMin, fMax, fRange;

   float    dx, dy;

   int      UpDirection;

   float   *Eval;

   float    dist, s1, s2;


   if ( IndependentVarDirection < 0 ||  IndependentVarDirection  > Zdir )
   {
      fprintf( stderr, "Bogus independent variable direction: %d. Using Z axis\n",
	       IndependentVarDirection );
      UpDirection = Zdir;
   }
   else
   {
      UpDirection = IndependentVarDirection;
   }

   f = (int *)malloc( sizeof(int) * Npoints );
   Df = (int *)malloc( sizeof(int) * Npoints );

   if ( f == NULL || Df == NULL )
   {
      fprintf( stderr, "Unable to allocate work array for interpolation routine\n" );
      exit( EXIT_FAILURE );
   }

   xCoords = (float *)malloc( sizeof(float)*xRes );
   yCoords = (float *)malloc( sizeof(float)*yRes );

   if ( xCoords == NULL || yCoords == NULL )
   {
      fprintf( stderr, "Unable to allocate memory for dependent variable coordinates\n" );
      exit( EXIT_FAILURE );
   }

   Datax = (float *)malloc( sizeof(float) * Npoints );
   Datay = (float *)malloc( sizeof(float) * Npoints );
   Dataf = (float *)malloc( sizeof(float) * Npoints );
   Datad = (float *)malloc( sizeof(float) * Npoints );

   if ( Datax == NULL || Datay == NULL || Dataf == NULL  || Datad == NULL )
   {
      fprintf( stderr, "Unable to allocate memory for data coordinates\n" );
      exit( EXIT_FAILURE );
   }

   for ( i = 0; i < Npoints; i++ )
   {
      switch ( UpDirection )
      {
	 case Xdir: Datax[i] = Data[i].z;
                    Datay[i] = Data[i].y;
                    Dataf[i] = Data[i].x;
                    break;

	 case Ydir: Datax[i] = Data[i].x;
                    Datay[i] = Data[i].z;
                    Dataf[i] = Data[i].y;
                    break;

	 case Zdir: Datax[i] = Data[i].x;
                    Datay[i] = Data[i].y;
                    Dataf[i] = Data[i].z;
                    break;
      }
   }

   xMin = yMin = fMin =  HUGE;
   xMax = yMax = fMax = -HUGE;

   for ( i = 0; i < Npoints; i++ )
   {
      if ( xMin > Datax[i] ) xMin = Datax[i];
      if ( yMin > Datay[i] ) yMin = Datay[i];
      if ( fMin > Dataf[i] ) fMin = Dataf[i];

      if ( xMax < Datax[i] ) xMax = Datax[i];
      if ( yMax < Datay[i] ) yMax = Datay[i];
      if ( fMax < Dataf[i] ) fMax = Dataf[i];
   }

   xRange = xMax - xMin;
   yRange = yMax - yMin;
   fRange = fMax - fMin;

   if ( xRange == 0 ) xRange = 0.01;
   if ( yRange == 0 ) yRange = 0.01;
   if ( fRange == 0 ) fRange = 0.00001;

   /* Pad the spatial ranges by +/- 2% */
   xMax += xRange * 0.02;
   yMax += yRange * 0.02;

   xMin -= xRange * 0.02;
   yMin -= yRange * 0.02;

   /* Recompute the range */
   xRange = xMax - xMin;
   yRange = yMax - yMin;

   dx = xRange / (float)(xRes - 1);
   dy = yRange / (float)(yRes - 1);

   for ( i = 0; i < xRes; i++ )
      xCoords[i] = i * dx + xMin;

   for ( i = 0; i < yRes; i++ )
      yCoords[i] = i * dy + yMin;

   Eval = (float *)malloc( sizeof(float) * xRes * yRes );

   if ( Eval == NULL )
   {
      fprintf( stderr, "Unable to allocate temporary surface data.\n" );
      exit( EXIT_FAILURE );
   }

   dx = xRange / (float)(Npoints - 1);
   dy = yRange / (float)(Npoints - 1);

   for ( j = 0; j < xRes; j++ )
   {
      for ( i = 0; i < yRes; i++ )
      {
	 s1 = 0; s2 = 0;

	 for ( k = 0; k < Npoints; k++ )
	 {
	    dist = pow( SQUARE(Datax[k] - xCoords[j]) + SQUARE(Datay[k] - yCoords[i]), -1.5 );

	    s1 += dist * Dataf[k];
	    s2 += dist;
	 }
	 
	 if (s2 == 0) s2 = 1;

	 switch ( UpDirection )
	 {
	    case Xdir: InterpolatedData[i*xRes +j].z = xCoords[j];
	               InterpolatedData[i*xRes +j].y = yCoords[i];
	               InterpolatedData[i*xRes +j].x = s1/s2;
                       break;

	    case Ydir: InterpolatedData[i*xRes +j].z = yCoords[i];
   	               InterpolatedData[i*xRes +j].y = s1/s2;
	               InterpolatedData[i*xRes +j].x = xCoords[j];
                       break;

	    case Zdir: InterpolatedData[i*xRes +j].z = s1/s2;
   	               InterpolatedData[i*xRes +j].y = yCoords[i];
	               InterpolatedData[i*xRes +j].x = xCoords[j];
                       break;
	 }
      }
   }

   free( f );
   free( Df );
   free( xCoords );
   free( yCoords );
   free( Datax );
   free( Datay );
   free( Dataf );
   free( Datad );
   free( Eval );
}
