;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; emp-shell.el -- Shell interaction support for Gnu Emacs Empire Tool (GEET)
;; 
;; Copyright (c) 1990 Lynn Randolph Slater, Jr
;; 
;; Author          : Darryl Okahata (darrylo%hpnmd@hpcea.hp.com), and 
;;                   Lynn Slater (lrs@indetech.com)
;; Created On      : Fri Oct 26 13:50:14 1990
;; Last Modified By: Lynn Slater x2048
;; Last Modified On: Thu Feb 14 19:16:58 1991
;; Update Count    : 88
;; Status          : GEET General Release 2d Patch 0
;; 
;; PURPOSE
;; 	Provides empire shell, its mode, and interaction command support
;; Started with stuff by Lynn but Darryl is by now responsible for the bulk
;; of the file.
;; HISTORY
;; 31-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Fri Jan 25 19:00:51 1991 #81 (Lynn Slater x2048)
;;    split out some stuff to emp-buffers
;; 25-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Fri Jan 25 11:45:23 1991 #78 (Lynn Slater x2048)
;;    added R Foresman's alternative define-prompt stuff.
;; 7-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Fri Dec  7 13:17:10 1990 #72 (Lynn Slater x2048)
;;    is quieter in batch mode
;; 30-Nov-1990		Lynn Slater x2048	
;;    Last Modified: Wed Nov 28 09:19:57 1990 #63 (Lynn Slater x2048)
;;    modified to throw client-hosed more often in batch mode
;; 27-Nov-1990		Lynn Slater x2048	
;;    Last Modified: Tue Nov 27 13:23:30 1990 #46 (Lynn Slater x2048)
;;    added a timeout to shell waits
;;    extra accept-process-output hung batch processes
;; 25-Nov-1990		Lynn Slater x2048	
;;    Last Modified: Sun Nov 25 16:06:38 1990 #43 (Lynn Slater x2048)
;;    made empire-last-prompt work
;; 11-Nov-1990		Lynn Slater	
;;    Last Modified: Fri Nov  9 18:57:52 1990 #30 (Lynn Slater)
;;    added empire-message-client-replies
;; 8-Nov-1990		Lynn Slater	
;;    Last Modified: Sun Nov  4 16:11:55 1990 #20 (Lynn Slater)
;;    empire-map-des-interactively is turned off for large commands if asked
;;    and map is refreshed once at the end
;; 4-Nov-1990		Lynn Slater	
;;    Last Modified: Fri Nov  2 19:32:59 1990 #19 (Lynn Slater)
;;    changed aborts message in wait-for-empire-prompt to not send c-g to
;;    batch output.
;; 2-Nov-1990		Lynn Slater	
;;    Last Modified: Wed Oct 31 08:57:13 1990 #13 (Lynn Slater)
;;    added darryl's fixes dealing with multiple empire-prompts.
;; 30-Oct-1990		Lynn Slater	
;;    Last Modified: Mon Oct 29 12:50:41 1990 #11 (Lynn Slater)
;;    added wait-for-empire-prompt-if-necessary
;; TABLE OF CONTENTS
;;   shell -- Run an inferior shell, with I/O through buffer *shell*.
;;   empire-shell -- Makes an empire shell for command execution
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; The contents of this file ARE copyrighted but permission to use, modify,
;; and distribute this code is granted as described in the file
;; emp-install.el which should have been distributed with this file. These
;; terms constitute what the Free Software Foundation calls a COPYLEFT.
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'emp-shell)
(require 'shell)
(require 'cl)
(require 'emp-db)


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; User variables -- do not edit here, use empire-edit-options
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar empire-move-shell-cursor t
  "*If non-nil, move the empire shell cursor to the end of the buffer
when output is received from the empire client.  Moving the cursor insures
that the cursor is next to a prompt, but it also means that Emacs always has
to update the display while output is being received."
)

(defvar empire-favor-left-coordinate t
  "*If non-nil, illegal map coordinates will be replaced with the legal
coordinate on the left (only for those functions that support
correcting illegal coordinates).  If nil, illegal map coordinates will
be replaced with the legal coordinate on the right."
)

(defvar empire-local-tmp-directory "/tmp"
  "*The directory from which temporary files are read (e.g., the output
from a `dump' command).  The idea is that an empire client, running
remotely, will write to the directory defined by
`empire-remote-tmp-directory' (on the remote machine), and that
directory will be accessed by the local machine using NFS (or something
similar).  This variable determines the name of the directory from the
local machine's point of view.  For example, if the remote empire
client, running on machine `foobar', writes to a file `/usr/tmp/XXX'
(which implies that `empire-remote-tmp-directory' is \"/usr/tmp\"), this
variable may have to be set to something like \"/nfs/foobar/usr/tmp\".

If the empire client is being run locally, `empire-local-tmp-directory'
and `empire-remote-tmp-directory' must be the same.

Paranoid users may wish to set this variable to their favorite
\"protected\" directory.  This variable is saved as part of the empire
data file.")

(defvar empire-remote-tmp-directory "/tmp"
  "*The directory in which a remotely running empire client creates
temporary files (e.g., the output from a `dump' command).  The idea is
that an empire client, running remotely, will write to files in this
directory (from the remote machine's point of view) and that directory
will be accessed by the local machine using NFS (or something similar).
For example, if the remote empire client, running on machine `foobar',
is supposed to write to files in the directory `/usr/tmp', this variable
should be set to \"/usr/tmp\".  In this case the
`empire-local-tmp-directory' variable should probably be set to
something like \"/nfs/foobar/usr/tmp\".

If the empire client is being run locally, `empire-local-tmp-directory'
and `empire-remote-tmp-directory' must be the same.

Paranoid users may wish to set this variable to their favorite
\"protected\" directory.  This variable is saved as part of the empire
data file.")

(defvar empire-shell-window-upper nil
  "*If non-nil, the uppermost window displaying the empire shell
buffer is chosen when deciding which window should display the empire
shell output.  If `nil', the lowermost window is chosen.  This
variable has an effect only if there is more than one window
displaying the empire shell buffer.  This variable is saved as part of
the empire data file.")

(defvar empire-shell-window-right nil
  "*If non-nil, the rightmost window displaying the empire shell
buffer is chosen when deciding which window should display the empire
shell output.  If `nil', the leftmost window is chosen.  This variable
has an effect only if there is more than one window displaying the
empire shell buffer.  This variable is saved as part of the empire data
file.")

(defvar empire-btu-left 0
  "The amount of BTUs left, after executing a command.  This variable is
set ONLY after an empire prompt is found."
  )

(defconst empire-temp-buffer "*Temporary Empire*"
  "The name of the temporary work buffer.  The contents of this buffer
will not normally be visible to the user."
  )

(defvar empire-old-process-filter nil
  "If a process filter already exists in the empire shell buffer, it is
saved here and a new one is installed (which will call the old one)."
  )

(defvar empire-wait-for-semaphore nil
  "A semaphore used while waiting for the empire command prompt to appear.
While non-nil, the empire command prompt has not yet been received, and the
process filter should wait some more."
  )

(defvar empire-output nil
  "A temporary place to hold text as it is received from the empire client."
  )

(defvar empire-send-input-command 'shell-send-input
  "Lisp function to execute to send input to the empire client."
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Support for recognition of empire prompts
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar empire-prompt-regexp nil
  "Calculated `on-the-fly'.")

(defvar empire-symbol-prompt-alist nil
  "An alist of (symbol . empire-prompt-regexp)."
  )

(defvar empire-misc-prompt-regexp nil
  "A regexp to match all empire prompts *BUT* the main command prompt.
The regexp for the main command prompt is given by
`shell-prompt-pattern'.  Each prompt regexp pattern in this regexp
must be surrounded by \"\\\\(\" and \"\\\\)\" and be separated by a
\"\\\\|\""
  )

(defmacro local-empire-prompts-within (&rest body)
  "makes local copies of the empire prompt variables so that
any prompts added will not be searched for outside the scope."
  (` (let ( (empire-symbol-prompt-alist empire-symbol-prompt-alist)
	    (empire-misc-prompt-regexp empire-misc-prompt-regexp)
	    )
       (,@ body)
       ))
  )

(defun add-empire-prompt-regexp (regexp symbol)
  "This function adds a new regexp to the empire-symbol-prompt-alist.
This consists of updating the alist and recalculating the master
empire-misc-prompt-regexp"
  (let (item)
    ;;
    ;; Update empire-symbol-prompt-alist
    ;;
    (if (and empire-symbol-prompt-alist
	     (setq item (assoc symbol empire-symbol-prompt-alist))
	     )
	;; entry already exists
	(setcdr item regexp)
      ;; entry does not exist
      (setq empire-symbol-prompt-alist (cons (cons symbol regexp)
					     empire-symbol-prompt-alist))
      )
    ;;
    ;; Recalculate empire-misc-prompt-regexp
    ;;
    (setq empire-misc-prompt-regexp
	  (concat "\\("
		  (mapconcat 'cdr empire-symbol-prompt-alist "\\)\\|\\(")
		  "\\)" ) )
    )
  )

(defmacro define-empire-prompt-regexp (name doc regexp symbol)
  "Define a regexp for an empire prompt.  NAME is the lisp name given to
the lisp variable containing the regexp, DOC is the doc for the variable,
REGEXP is the regexp, and SYMBOL is the symbol associated with the regexp.
SYMBOL is returned by `wait-for-empire-prompt' to tell the caller what
prompt was found.

Regexps are associated with SYMBOL.  If there are two regexps with the
same SYMBOL, the later occurring of the two will take precedence and
will override the old one.  This is done so that the user can use this
macro to change regexps without changing this file (after this file is
loaded, he loads his own file, which uses this macro to redefine the
prompt regexps).

Note that SYMBOL *cannot* be `t', as `t' is reserved for the main
empire client prompt.
"
  (add-empire-prompt-regexp regexp symbol)
  ;;
  ;; Define the variable
  ;;
  (` (defvar (, name) (, regexp) (, doc)) )
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Empire prompts
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(define-empire-prompt-regexp
  empire-move-explore-prompt-regexp
  "The regexp matching the prompt displayed when moving or exploring.
The following match-data positions are defined:
     1 -- The designation of the sector
     2 -- The X-coordinate
     3 -- The Y-coordinate"
  "^<[0-9]+\\.[0-9]:[ \t]+\\(.\\)[ \t]+\\(-?[0-9]+\\),\\(-?[0-9]+\\)>[ \t]+"
  move-explore-prompt
  )

(define-empire-prompt-regexp
  empire-telegram-regexp
  "The regexp matching the prompt that appears when a telegram is being
entered."
  "^[ \t]*[0-9]+[ \t]+left:[ \t]+"
  telegram-prompt
  )

(define-empire-prompt-regexp
  empire-in-use-regexp
  "The regexp matching the prompt that appears when the country is being used."
  "^[0-9]+ country in use by"
  country-in-use-prompt
  )

(define-empire-prompt-regexp
  empire-no-connect-regexp
  "The regexp matching the prompt that appears when the server is not there."
  "^connect: "
  connect-refused-prompt
  )

(define-empire-prompt-regexp
  empire-bad-move
  "The regexp matching the prompt that appears when you move where you
cannot go."
  "^You can't go there"
  bad-move-prompt
  )

(define-empire-prompt-regexp
  empire-bad-mobility
  "The regexp matching the prompt that appears when you move where you
cannot go."
  "^Not enough mobility.  You can't go there"
  bad-mobility-prompt
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar empire-last-prompt ""
  "The text of the last recognized empire shell prompt."
  )


(defvar empire-last-command "")

(defun shell (&optional arg name)
  "Run an inferior shell, with I/O through buffer *shell*.
If buffer exists but shell process is not running, make new shell.
Program used comes from variable explicit-shell-file-name,
 or (if that is nil) from the ESHELL environment variable,
 or else from SHELL if there is no ESHELL.
If a file ~/.emacs_SHELLNAME exists, it is given as initial input
 (Note that this may lose due to a timing error if the shell
  discards input when it starts up.)
The buffer is put in shell-mode, giving commands for sending input
and controlling the subjobs of the shell.  See shell-mode.
See also variable shell-prompt-pattern.

The shell file name (sans directories) is used to make a symbol name
such as `explicit-csh-arguments'.  If that symbol is a variable,
its value is used as a list of arguments when invoking the shell.
Otherwise, one argument `-i' is passed to the shell.

Note that many people's .cshrc files unconditionally clear the prompt.
If yours does, you will probably want to change it."
  (interactive "P")
  (let* ((prog (or explicit-shell-file-name
		   (getenv "ESHELL")
		   (getenv "SHELL")
		   "/bin/sh"))
	 (buffer-name (or name
			  (if arg
			      (read-string "Shell Buffer Name: " "shell")
			    "shell")))
	 (name (file-name-nondirectory prog)))
    (switch-to-buffer
     (apply 'make-shell buffer-name prog
	    (if (file-exists-p (concat "~/.emacs_" name))
		(concat "~/.emacs_" name))
	    (let ((symbol (intern-soft (concat "explicit-" name "-args"))))
	      (if (and symbol (boundp symbol))
		  (symbol-value symbol)
		'("-i")))))))

(defun empire-shell ()
  "Makes an empire shell for command execution"
  (interactive)
  (shell nil "Empire")
  (require 'emp-modes)
  (empire-shell-mode))

(defun show-shell ()
  "Shows the shell"
  (let (
	(current-buf (current-buffer))
	(current-window (selected-window))
	)
    ;; We can't use save-excursion, etc. here because we do not want any
    ;; values of point being saved.  We just want the current buffer/window
    ;; information saved.
    (unwind-protect
	(progn
	  (pop-to-buffer empire-shell-buffer)
	  (end-of-buffer))
      (progn
	(select-window current-window)
	(set-buffer current-buf)
	)
      )
    )
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Shell communication support by Darryl Okahata
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-send-input ()
  "Wrapper function to send input to the empire client.  Some people
will use shell-mode, in which case `empire-send-input-command' will be
set to `shell-send-input'.  Others will use cmushell, in which case
`empire-send-input-command' will be set to `comint-send-input'.  This
function makes everything work with any shell mode."
  (let ( (moving (empire-point-should-be-moved empire-shell-buffer
					       (point-max)))
	 )
    (funcall empire-send-input-command)
    (if moving
	(move-empire-buffer-point (point-max)))
    )
  )

(defvar empire-message-client-replies nil
  "If true, replies from the empire client will be sent to stdout.")

(defun empire-process-prompt-filter (process output)
  "The empire client process filter.  This function is used in conjunction
with `wait-for-empire-prompt' to wait for the empire client to output
the empire command prompt."
  (let* ((old-match (match-data))
	 (proc-marker (process-mark process))
	 (insert-location (marker-position proc-marker))
	 pos line)
    (unwind-protect
	(progn
	  (if empire-message-client-replies
	      (message "%s" output))
	  (goto-char insert-location)
	  (setq empire-output (concat empire-output output))
	  (while (setq pos (string-match "\n" empire-output))
	    (setq line (substring empire-output 0 (1+ pos)))
	    (insert line)
	    (setq empire-output (substring empire-output (1+ pos) nil))
	    (if (string-match empire-prompt-regexp line)
		(progn
		  (setq empire-wait-for-semaphore nil)
		  (setq empire-last-prompt line)
		  (insert empire-output)
		  )
	      )
	    (set-marker proc-marker (point))
	    )
	  (if (and empire-wait-for-semaphore
		   (string-match empire-prompt-regexp empire-output))
	      (progn
		(setq empire-wait-for-semaphore nil)
		(setq empire-last-prompt empire-output)
		(insert empire-output)
		(set-marker proc-marker (point))
		)
	    )
	  )
      (store-match-data old-match)
      )
    )
  )

(defvar empire-sleep-limit 120
  "*This is the number of seconds of sleeps which the game will permit
before concluding that the interaction with the client is hosed. The time
of the sleeps in seconds is 1,2,4,8,16,32,...")

(defun wait-for-empire-prompt ()
  "Wait for an empire prompt to appear in the empire-shell-buffer.  If
the main empire command prompt appeared, `t' is returned; otherwise,
the symbol associated with the regexp is returned (this association is
made using the `define-empire-prompt-regexp' macro).  The variable
`empire-last-prompt' contains the text of the prompt that caused this
routine to return."
  (let ( (cursor-in-echo-area t) (oldmatch (match-data))
	 (cnt 1)
	 old process old-filter)
    (if (not empire-batch-play)
	(message "Waiting for empire `%s' ... (Control-g aborts)" empire-last-command))
    (switch-to-empire-buffer-if-necessary
     (setq process (get-buffer-process (current-buffer)))
     (if (not process)
	 (if empire-batch-play
	     (throw 'client-hosed "Empire process not running")
	   (error "Empire process is not running."))
       )
     (setq old-filter (process-filter process))
     (unwind-protect
	 (progn
	   (goto-char (point-max))
	   (setq empire-prompt-regexp (concat "\\("
					      shell-prompt-pattern
					      "\\)\\|"
					      empire-misc-prompt-regexp
					      ))
	   (setq empire-output nil)
	   (setq empire-wait-for-semaphore t)
	   (set-process-filter process 'empire-process-prompt-filter)
	   (while (and empire-wait-for-semaphore
		       (<= cnt empire-sleep-limit))
	     (if (input-pending-p)
		 (progn
		   ;;
		   ;; Handle the case where the user has pressed a key.
		   ;; For this case, we want to process all subprocesses,
		   ;; but we do not want to process any keystrokes.  Because
		   ;; of this, we cannot use `sit-for', as the pending
		   ;; keystrokes will cause `sit-for' to immediately return
		   ;; without processing any subprocess output.  Purging the
		   ;; input buffer is a harsh solution that we want to avoid,
		   ;; and so we settle for the following.
		   ;;
		   (sleep-for cnt)
		   (accept-process-output)
		   )
	       (progn
		 (sit-for cnt)
		 ;; the next line hangs batch processes
		 ;;(accept-process-output);; unnecessary, but it soothes
		 ;; our paranoia
		 )
	       )
	     ;; wait a bit longer next time
	     (if empire-batch-play (setq cnt (+ cnt cnt)))
	     )
	   (goto-char (point-max))
	   (set-marker (process-mark process) (point))
	   )
       (progn
	 (set-process-filter process old-filter)
	 (store-match-data oldmatch)
	 )
       )
     )
    (if (and empire-wait-for-semaphore	; would like to sleep more
	     (> cnt empire-sleep-limit) ; but limit was reached
	     )
	(progn
	  (message "\nClient interaction is Hosed! %s seconds with no command line"
		   cnt)
	  (throw 'client-hosed (format "Client stall! %s seconds with no command line"
				       cnt))
	  )
      ;; else

      (if (not empire-batch-play)
	  (message "Done."))
      ;;
      ;; Return `t' if the prompt that was matched was the main empire
      ;; command prompt; else search the list of empire prompt regexps.
      ;;
      (if (string-match shell-prompt-pattern empire-last-prompt)
	  (progn
	    (setq empire-btu-left (string-to-int (substring empire-last-prompt
							    (match-beginning 1)
							    (match-end 1))))
	    t
	    )
	(catch 'found-prompt
	  (dolist (item empire-symbol-prompt-alist)
	    (if (string-match (cdr item) empire-last-prompt)
		(throw 'found-prompt (car item)))
	    )
	  nil
	  )
	)
      ))
  )

(defun wait-for-empire-prompt-if-necessary ()
  "Wait for the empire command prompt to appear in the empire-shell-buffer
   if you are not already at the prompt."
  (save-excursion
    (switch-to-empire-buffer-if-necessary
     (require 'emp-shell)
     (show-shell)
     ;; see if we must wait
     (end-of-buffer)
     (let ((here (point))
	   prompt
	   start )
       (beginning-of-line 1)
       (setq prompt (buffer-substring (point) here))
       (if (not (setq start (string-match
			     (concat shell-prompt-pattern "[ \t\n]*")
			     prompt)))
	   ;; we must wait
	   (wait-for-empire-prompt)
	 ;; else
	 (setq empire-btu-left (string-to-int (substring prompt
							 (match-beginning 1)
							 (match-end 1))))
	 (setq start (+ (point) (match-end 0) start))
	 (if (not (string-match "^[ \t\n]*$"
				(buffer-substring start (point-max))))
	     (if empire-batch-play
		 (throw 'client-hosed "Command started, not completed")
	       (error "An empire command has been started but not completed."))
	   )))
     )))

(defun empire-point-should-be-moved (buffer-being-modified
				     insert-location
				     &optional window-being-modified)
  "Return non-nil if `point' should be moved."
  (let ( (current-buf (current-buffer))
	 (current-window (selected-window))
	 (current-point (point))
	 )
    (if (not window-being-modified)
	(setq window-being-modified (empire-get-window-to-change
				     buffer-being-modified)))
    ;;
    ;; Cases in which the cursor should be moved:
    ;;
    ;; * The current window is displaying the process buffer, and point is at
    ;;   the process mark.
    ;;
    ;; * The current window is displaying the process buffer, point is not at
    ;;   the process mark, but the current window is not the window whose
    ;;   cursor would be moved.
    ;;
    ;; * The current window is not displaying the process buffer, and the
    ;;   point of the window whose cursor would be moved is at the process
    ;;   mark.
    ;;
    (or (and (eq current-buf (get-buffer buffer-being-modified))
	     (or (= current-point insert-location)
		 (not (eq current-window window-being-modified))))
	(and (not (eq current-buf (get-buffer buffer-being-modified)))
	     (= (window-point window-being-modified)
		insert-location)))
    )
  )

(defun empire-output-filter (process output)
  "Filter to update the window point of the empire shell buffer
whenever the empire client writes to the buffer."
  (let* ( moving
	  (old-buffer (current-buffer))
	  (proc-mark (process-mark process))
	  (insert-location (marker-position proc-mark))
	  (empire-window (empire-get-window-to-change empire-shell-buffer))
	  (old-match (match-data));; just in case
	  )
    (set-buffer empire-shell-buffer)
    (setq moving (empire-point-should-be-moved empire-shell-buffer
					       insert-location
					       empire-window))
    (unwind-protect
	(save-excursion
	  (goto-char insert-location)
	  (insert output)
	  (set-marker proc-mark (point))
	  )
      (progn
	;; Are we supposed to update the window point?
	(if moving
	    (progn
	      (goto-char (setq insert-location (marker-position proc-mark)))
	      (if empire-window
		  (set-window-point empire-window insert-location))
	      )
	  )
	(set-buffer old-buffer)
	(store-match-data old-match)
	)
      )
    )
  )

(defun empire-output-postfilter (process output)
  "This filter is used if a process filter already exists in the empire
shell buffer."
  (let ( moving
	 (old-buffer (current-buffer))
	 (proc-mark (process-mark process))
	 (insert-location (marker-position proc-mark))
	 (empire-window (empire-get-window-to-change empire-shell-buffer))
	 (old-match (match-data))
	 )
    (setq moving (empire-point-should-be-moved empire-shell-buffer
					       insert-location
					       empire-window))
    (unwind-protect
	(progn
	  ;; call the old process filter
	  (funcall empire-old-process-filter process output)
	  ;; update the window point if we have to
	  (if moving
	      (switch-to-empire-buffer-if-necessary
	       (goto-char (setq insert-location (marker-position proc-mark)))
	       (if empire-window
		   (set-window-point empire-window insert-location))
	       )
	    )
	  )
      (store-match old-match)
      )
    )
  )


(defun send-empire-command (command)
  "Send COMMAND to the empire shell and execute it."
  (setq empire-last-command command)
  (let (here prompt-pos moving)
    (switch-to-empire-buffer-if-necessary
     (move-to-empire-shell-prompt) ;; Can we execute a command now?
     (delete-region (point) (point-max));; insure that its clear
     (insert command)
     (empire-send-input);; execute the command
     )
    )
  ;;(setq empire-last-command "")
  )

(defun send-empire-command-wait (command)
  "Send COMMAND to the empire shell and execute it."
  (send-empire-command command)
  (wait-for-empire-prompt))


(defun send-empire-command-and-parse-reply (command &optional fcn narrow
						    &rest args)
  "Sends COMMAND to the empire shell and calls FCN, with optional buffer
narrowing if NARROW is non-nil, with ARGS once the corresponding reply
is sent.  Before the fcn call, the buffer is narrowed to contain only
the reply if NARROW is non-nil.  FCN can be nil, in which case nothing
will be done when the command finishes executing (Emacs will still pause
while the command is executing)."
  (let (start end)
    (switch-to-empire-buffer-if-necessary
     (move-to-empire-shell-prompt)
     (delete-region (point) (point-max));; insure that its clear
     (setq start (point))
     (insert command)
     (setq empire-last-command command)
     (empire-send-input);; execute the command
     (wait-for-empire-prompt);; wait for it to finish
     (if (not empire-batch-play) (message "Processing"))
     (setq empire-last-command "")
     (move-to-empire-shell-prompt)
     (if fcn
	 (progn
	   (setq end (point))
	   (if narrow
	       (narrow-to-region start end)
	     )
	   (goto-char start)
	   (apply fcn args)
	   (widen)
	   (move-to-empire-shell-prompt)
	   )
       )
     )
    )
  )

(defun empire-check-prompt ()
  "Check to see if `point' is at a command prompt (and only a command
prompt).  If it is not, raise an error.  If it is, do nothing.  A
valid prompt must match the shell-prompt-pattern regexp, and it must
be followed only by whitespace to the end of the buffer (newlines are
acceptable).  If is not followed by whitespace, it is assumed that
`point' is either at the start of an old prompt (an error), or an
empire command has been started but not completed (empire is waiting
for more input)."
  (let ( (here (point)) start )
    (save-excursion
      (beginning-of-line 1)
      (setq prompt (buffer-substring (point) here))
      (if (not (setq start (string-match shell-prompt-pattern prompt)))
	  (if empire-batch-play
	      (throw 'client-hosed "Point not at prompt")
	    (error "Point is not at a prompt."))
	)
      (setq empire-btu-left (string-to-int (substring prompt
						      (match-beginning 1)
						      (match-end 1))))
      (setq start (+ (point) (match-end 0) start))
      (if (not (string-match "^[ \t\n]*$"
			     (buffer-substring start (point-max))))
	  (if empire-batch-play
	      (throw 'client-hosed "Command started, not completed")
	    (error "An empire command has been started but not completed."))
	)
      )
    )
  )

(defun move-to-empire-shell-prompt ()
  "Move point to the start of where a new command can be entered.
Note that this can be different from the end of a command prompt, as the
actual location is defined by the process-mark.  If an empire command is
in progress, an error is raised."
  (let (start end)
    (switch-to-empire-buffer-if-necessary
     (setq end (marker-position (process-mark
				 (get-buffer-process (current-buffer)))))
     (goto-char end)
     (if (re-search-backward shell-prompt-pattern nil t)
	 (progn
	   ;; found a prompt
	   (goto-char (match-end 0))
	   )
       (progn
	 ;; did not find a prompt
	 (if empire-batch-play
	     (throw 'client-hosed "Cannot find empire prompt")
	   (error "Cannot find empire prompt!"))
	 )
       )
     (empire-check-prompt)
     (move-empire-buffer-point end)
     )
    )
  )

