;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; emp-nsc.el -- Expressions for Gnu Emacs Empire Tool (GEET)
;; 
;; Copyright Robert Forsman (GNU General Public License)
;;
;; AFSID           : $__Header$
;; Author          : Robert Forsman <thoth@manatee.cis.ufl.edu>
;; Created On      : Tue Jan 22 10:17:45 1991
;; Last Modified By: Lynn Slater x2048
;; Last Modified On: Thu Feb 14 19:16:40 1991
;; Update Count    : 9
;; Status          : GEET General Release 2d Patch 0
;; 
;; HISTORY
;; PURPOSE
;;   This file was developed by looking at the C code in the
;;   empire 1.1 distribution (I've seen almost all patchlevels).
;;   The files was player/subs/nstr.c which contains the code
;;   to parse an empire condition expression.  Since he was using
;;   C and I'm using lisp there wasn't much code I could duplicate.
;;   Therefore I can only reverse engineer the intent.
;;   
;;   As with all my software this is bound to have some bugs.  Break
;;   it and mail me or gnurus-bugs the pieces.
;; TABLE OF CONTENTS
;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar cond-compare-alist
  (list (cons "<" '<)
	(cons ">" '>)
	(cons "=" 'equal)
	(cons "#" 'nequal) ; is there a better to do this one?
	)
  "An associative list that links an empire comparator [<>=#] with the
lisp equivalent"
  )

(defun nequal (a b)
  "returns nil if the two arguments are equal.  This is a little bit
slower than the equal function.  It only exists to simplify
cond-compare-alist."
  (not (equal a b))
  )

(defun retrieval-form-for (attribute)
  "returns a lisp form that will evaluate to an empire condition lhs or
rhs.  This handles both numbers and commodities but assumes anything that
is not a valid sector attribute or number is a literal string."
  (let (sect-attr)
    ( cond
      ( (string-match "^-?[0-9]+$" attribute)
	(string-to-int attribute) ; we have a simple number
	)
      ( (string-match "^(.+)$" attribute)
	(read attribute) )
      ( t
	(setq sect-attr (dynamic-position-of (read attribute)))
	(if (not sect-attr)
	    ;; assume it's something like des=g, return the string value
	    attribute
	  (` (recall x y (, sect-attr)))
	  )
	)
      )
    )
  )

(defun empire-parse-condition (cond &optional accept-cons)
  "This function takes an empire condition like \"?des=g&gold<5\" and
returns a lambda that SHOULD return true if the condition is true.
The lambda form will accept two parameters, the X and Y coordinates of
the sector to be checked.

  empire-parse-condition also allows lisp code (surrounded by parens
of course) to be inserted inline.  See the ideal-civ-fcn example below.

  If the optional ACCEPT-CONS is non-nil then the generated lambda
form will instead accept one argument that is (cons x y).  It inserts
a let to extract the x and y coordinates.  See the third example.

  Usage:
	(funcall (empire-parse-condition \"?des=g&gold=62\") 0 6)
	(funcall (empire-parse-condition
			\"?des=+&(funcall empire-ideal-civ-fcn x y)<civ\")
		 0 6)
	(funcall (empire-parse-condition \"?des=g&gold=62\" t) (cons 0 6))
	(empire-calc-survey-all (position-of 'gold) 10
			(empire-parse-condition \"?des=+&eff>98&sdes=_\"))
	"
  (let ( commodities-used
	 and-clauses scan-clauses

	 sub-cond
	 comparator lhs rhs)
    (if (not (equal ?? (aref cond 0)))
	(error "invalid condition %s" cond)
      )
    (setq scan-clauses (setq and-clauses '( (oursp x y) ) ))
    (setq cond (substring cond 1))

    (while (or (string-match "^\\([^&]+\\)&" cond)
	       (string-match "^\\([^&]+\\)$" cond))
      (setq sub-cond (substring cond (match-beginning 1) (match-end 1)))
      (setq cond (substring cond (match-end 0)))
      ( cond
	( (string-match "^\\([a-z_A-Z]+\\|-?[0-9]+\\|(.+)\\)\\([<>=#]\\)\\([a-z_A-Z]+\\|-?[0-9]+\\|(.+)\\|.\\)$" sub-cond)
	  (setq lhs (substring sub-cond (match-beginning 1) (match-end 1))
		comparator (substring sub-cond (match-beginning 2)
				      (match-end 2))
		rhs (substring sub-cond (match-beginning 3) (match-end 3))
		)
	  (setcdr scan-clauses
		  (list (list (cdr (assoc comparator cond-compare-alist))
			      (retrieval-form-for lhs)
			      (retrieval-form-for rhs))))
	  )
	( (string-match "^(.*)$" sub-cond) ; lisp expression
	  (setcdr scan-clauses (list (read sub-cond)))
	  )
	( t
	  (error "invalid subcondition %s (at %d)" sub-cond (match-end 0))
	  )
	)
      (setq scan-clauses (cdr scan-clauses))
      )
    (if accept-cons
	(` (lambda (sect)
	     (let ((x (car sect))
		   (y (cdr sect)))
	       (and (,@ and-clauses))
	       )))
      (` (lambda (x y)
	   (and (,@ and-clauses))
	   ))
      )
    )
  )
