/*
 * sel.c
 *
 * arrange to block on read/write file descriptors
 * using lwp yield and the null process
 *
 */

#include <malloc.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <sys/file.h>
#include <varargs.h>
#include <sys/time.h>

#include "bit.h"
#include "lwp.h"
#include "lwpint.h"

struct lwpSelect {
	int	maxfd;
	int	nfile;
	bit_fdmask readmask;
	bit_fdmask writemask;
	struct lwpProc **wait;
	struct lwpQueue delayq;
	struct lwpProc *proc;
};

struct lwpSelect LwpSelect;

void lwpInitSelect(proc)
	struct lwpProc *proc;
{
	LwpSelect.maxfd = 0;
	LwpSelect.nfile = getdtablesize();
	LwpSelect.readmask = bit_newfdmask();
	LwpSelect.writemask = bit_newfdmask();
	LwpSelect.wait = (struct lwpProc **)
		calloc(sizeof(char *), LwpSelect.nfile);
	LwpSelect.delayq.head = 0;
	LwpSelect.delayq.tail = 0;
	LwpSelect.proc = proc;
}

void lwpSleepFd(fd, mask)
	int	fd;
	int	mask;
{
	extern struct lwpProc *LwpCurrent;

	if (LwpSelect.wait[fd] != 0) {
		logerror("multiple sleeps attempted on file descriptor %d", fd);
		return;
	}
	if (mask & LWP_FD_READ)
		BIT_SETB(fd, LwpSelect.readmask);
	if (mask & LWP_FD_WRITE)
		BIT_SETB(fd, LwpSelect.writemask);
	if (LwpSelect.maxfd == 0 && LwpSelect.delayq.head == 0) {
		/* select process is sleeping until first waiter arrives */
		lwpReady(LwpSelect.proc);
	}
	if (fd > LwpSelect.maxfd)
		LwpSelect.maxfd = fd;
	LwpSelect.wait[fd] = LwpCurrent;
	LwpCurrent->fd = fd;
	lwpReschedule();
}

void lwpWakeupFd(proc)
	struct lwpProc *proc;
{
	if (proc->fd < 0)
		return;
	BIT_CLRB(proc->fd, LwpSelect.readmask);
	BIT_CLRB(proc->fd, LwpSelect.writemask);
	LwpSelect.wait[proc->fd] = 0;
	proc->fd = -1;
	lwpReady(proc);
}

void lwpSleepUntil(until)
	long	until;
{
	extern struct lwpProc *LwpCurrent;

	LwpCurrent->runtime = until;
	if (LwpSelect.maxfd == 0 && LwpSelect.delayq.head == 0) {
		/* select process is sleeping until first waiter arrives */
		lwpReady(LwpSelect.proc);
	}
	lwpAddTail(&LwpSelect.delayq, LwpCurrent);
	lwpReschedule();
}

/*ARGSUSED*/
void
lwpSelect(argc, argv)
	int	argc;
	char	**argv;
{
	extern struct lwpProc *LwpCurrent;
	struct lwpProc *us = LwpCurrent;
	bit_fdmask readmask;
	bit_fdmask writemask;
	int	n;
	int	fd;
	long	now;
	long	delta;
	struct lwpProc *proc;
	struct timeval tv;
	struct lwpQueue save;

	readmask = bit_newfdmask();
	writemask = bit_newfdmask();
	while (1) {
		while (1) {
			bit_copy(LwpSelect.readmask, readmask);
			bit_copy(LwpSelect.writemask, writemask);
			if (bit_fd(readmask) >= 0 || bit_fd(writemask) >= 0)
				break;
			if (LwpSelect.delayq.head)
				break;
			/* wait for someone to lwpSleepFd or lwpSleepUntil */
			LwpSelect.maxfd = 0;
			lwpReschedule();
		}
		tv.tv_sec = 1000000;
		tv.tv_usec = 0;
		if (LwpSelect.delayq.head) {
			time(&now);
			proc = LwpSelect.delayq.head;
			for ( ; proc != 0; proc = proc->next) {
				delta = proc->runtime - now;
				if (delta < tv.tv_sec)
					tv.tv_sec = delta;
			}
			if (tv.tv_sec < 0)
				tv.tv_sec = 0;
		}
		n = select(LwpSelect.maxfd + 1, readmask, writemask, 0, &tv);
		if (n < 0) {
			if (errno == EINTR) {
				/* go handle the signal */
				lwpReady(us);
				lwpReschedule();
				continue;
			}
			logerror("select failed (bad file descriptor?)");
			exit(-1);
		}
		if (LwpSelect.delayq.head) {
			/* sleeping proecss activity */
			time(&now);
			save.tail = save.head = 0;
			while (proc = lwpGetFirst(&LwpSelect.delayq)) {
				if (now > proc->runtime)
					lwpReady(proc);
				else
					lwpAddTail(&save, proc);
			}
			LwpSelect.delayq = save;
		}
		if (n > 0) {
			/* file descriptor activity */
			while ((fd = bit_fd(readmask)) >= 0) {
				BIT_CLRB(fd, readmask);
				if (LwpSelect.wait[fd] == 0)
					continue;
				lwpWakeupFd(LwpSelect.wait[fd]);
			}
			while ((fd = bit_fd(writemask)) >= 0) {
				BIT_CLRB(fd, writemask);
				if (LwpSelect.wait[fd] == 0)
					continue;
				lwpWakeupFd(LwpSelect.wait[fd]);
			}
		}
		lwpReady(LwpCurrent);
		lwpReschedule();
	}
	/*NOTREACHED*/
}
