;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; emp-buffers.el -- Support for buffer manips for Gnu Emacs Empire Tool (GEET)
;; 
;; Copyright (c) 1990 Lynn Randolph Slater, Jr
;; 
;; Author          : Lynn Slater  (lrs@indetech.com), and
;;                   Darryl Okahata (darrylo%hpnmd@hpcea.hp.com)
;; Created On      : Thu Jan 31 09:03:39 1991
;; Last Modified By: Lynn Slater x2048
;; Last Modified On: Thu Feb 14 19:16:56 1991
;; Update Count    : 10
;; Status          : GEET General Release 2d Patch 0
;; 
;; PURPOSE
;; 	This file supplies buffer manipulation functions and utility lisp
;; functions. Most of this was taken from emp-modes and some came from
;; emp-shell. 
;; HISTORY
;; 31-Jan-1991		Lynn Slater x2048	
;;    Split out from emp-modes
;; TABLE OF CONTENTS
;;   empire-cmd-other-buf-and-exec -- Execute an empire COMMAND, redirect its output into a temporary
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; The contents of this file ARE copyrighted but permission to use, modify,
;; and distribute this code is granted as described in the file
;; emp-install.el which should have been distributed with this file. These
;; terms constitute what the Free Software Foundation calls a COPYLEFT.
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'emp-buffers)
(require 'emp-shell)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Commands outoutto Buffers (Mostly by Darryl)
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun execute-empire-command-with-buffer-output (command buffer)
  "Execute the empire COMMAND, and append results to BUFFER. Use and delete
temp file."
  (let ( (remote-tempfile (concat empire-remote-tmp-directory "/"
				  (make-temp-name "GnuEmp")))
	 (local-tempfile (concat empire-local-tmp-directory "/"
				 (make-temp-name "GnuEmp")))
       )
    (if (file-exists-p local-tempfile)
	(delete-file local-tempfile)
      )
    (switch-to-empire-buffer-if-necessary
     (move-to-empire-shell-prompt)
     (delete-region (point) (point-max))
     (insert command " >" remote-tempfile)
     (setq empire-last-command command)
     (empire-send-input)
     (wait-for-empire-prompt)
     (setq empire-last-command "")
     (move-to-empire-shell-prompt)
     (set-buffer buffer)
     (goto-char (point-max))
     (insert-file local-tempfile)
     (goto-char (point-min))
     )
    (delete-file local-tempfile)
    )
  )

(defun interactive-execute-empire-commands (fcns)
  "Interactively execute the command(s) given by FCNS.  FCNS can be
either a single symbol or a list of symbols (each symbol must natually
have a valid function cell).  This is similar to `run-hooks', except
that functions are called interactively (and can therefore prompt the
user for their arguments)"
  (let (fcn)
    (if (listp fcns)
	(progn
	  (dolist (fcn fcns)
	    (call-interactively fcn)
	    )
	  )
      (progn
	(call-interactively fcns)
	)
      )
    )
  )

(defun empire-cmd-other-buf-and-exec (command fcn &optional
					      show-buffer interactive-disable)
  "Execute an empire COMMAND, redirect its output into a temporary
file, read the file into a temporary buffer, move to the beginning of
the buffer, and execute function FCN.  This can be much faster than
executing the command manually in the empire shell buffer, waiting for
the command to finish, and then executing a function (Emacs does not
have to update the buffer display).  If the optional SHOW-BUFFER is
non-nil, the temporary buffer will be displayed just before FCN is
executed (the old window configuration will be restored after FCN has
finished executing).  FCN can also be a list of functions to execute.
This will work only if the empire client is running on the same
machine as Emacs.

If INTERACTIVE-DISABLE is non-nil, `empire-map-des-interactively' is
temporarily bound to `nil' while this function is being executed."
  (let ((empire-map-des-interactively empire-map-des-interactively))
    (if interactive-disable
	(setq empire-map-des-interactively 0)
      )
    (unwind-protect
	(save-excursion
	  (set-buffer (get-buffer-create empire-temp-buffer))
	  (delete-region (point-min) (point-max))
	  (execute-empire-command-with-buffer-output command
						     empire-temp-buffer)
	  (if show-buffer
	      (progn
		(empire-save-window-excursion
		 (delete-other-windows (display-buffer empire-temp-buffer))
		 (interactive-execute-empire-commands fcn)
		 )
		)
	    (progn
	      (interactive-execute-empire-commands fcn)
	      )
	    )
	  (bury-buffer empire-temp-buffer)
	  )
      (if interactive-disable (refresh-map))
      )
    )
  (conditionally-refresh-map)		; update if needed
  )

(defun get-sectors-and-execute-other-buffer (prompt cmd fcn
						    &optional show-buffer
						    interactive-disable)
  "Use PROMPT to prompt the user for a sector, sectors, or realm,
execute the empire command CMD, place the output in a hidden buffer, and
then execute the function FCN.  In CMD, a `%s' is replaced with the
sector, sectors, or realm entered by the user.  FCN can also be a list
of functions to execute."
  (empire-cmd-other-buf-and-exec
   (format cmd
	   (empire-read-sector-or-realm prompt))
   fcn show-buffer interactive-disable)
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Command insertion (Mostly by Darryl)
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun insert-in-empire-buffer (str)
  "Insert `str' into the empire shell buffer.  This isn't the world's most
efficient routine, but it works. Note that this routine does not make the
shell buffer visible; use `visible-insert-in-empire-buffer' for that."
  (let ( (this-buf (buffer-name)) )
    (switch-to-empire-buffer-if-necessary
     (goto-char (point-max))
     (insert str)
     (move-empire-buffer-point (point))
     )
    )
  )

(defun visible-insert-in-empire-buffer (str &optional
					    insert-space
					    returnp)
  "Make the empire shell buffer visible, and insert STR into it.  If there
is more than one window displaying the empire shell buffer, the appropriate
window is selected.

If returnp is true, do not change the current buffer"
  (let ( (shell-window (empire-get-window-to-change empire-shell-buffer))
	 (cb (current-buffer))
	 (cw (selected-window))
	 )
    (if shell-window
	(select-window shell-window)
      (pop-to-buffer empire-shell-buffer)
      )
    (point-wysiwyg)
    (if insert-space
	(insert-space-in-empire-buffer-if-necessary))
    (insert-in-empire-buffer str)
    (if returnp (select-window cw))
    )
  )

(defun insert-space-in-empire-buffer-if-necessary ()
  "Insert a space into the empire-shell-buffer if the previous character,
relative to the point, is not a space."
  (switch-to-empire-buffer-if-necessary
   (let ( (previous-char (preceding-char)) )
     (if (and (not (= previous-char 9));; does empire recognize tabs?
	      (not (= previous-char 32)))
	 (progn
	   (insert-in-empire-buffer " ")
	   )
       )
     )
   )
  )

(defun backspace-in-empire-buffer ()
  "Delete the last-character from the empire shell buffer."
  (switch-to-empire-buffer-if-necessary
   (backward-delete-char-untabify 1 t)
   (move-empire-buffer-point (point))
   )
  )

(defun empire-get-window-to-change (buffer)
  "If there is more than one window displaying BUFFER, return the appropriate
window to manipulate (move point, etc.).  If there is only one window, return
it.  If there are no windows displaying BUFFER, return `nil'.  This function
uses the variables `empire-shell-window-upper' and `empire-shell-window-right'
to determine which window to return."
  (let* (
	 (start-window (get-buffer-window buffer))
	 current-window
	 edges
	 result-window
	 result-x result-y
	 new-x new-y
	 )
    (if start-window		;;; if a window exists ...
	(progn
	  ;; Insure that we're dealing with a buffer type, and not a buffer
	  ;; name.
	  (setq buffer (get-buffer buffer))
	  (setq result-window start-window)
	  (setq edges (window-edges result-window))
	  (setq result-x (car edges))
	  (setq result-y (car (cdr edges)))
	  (setq current-window (next-window start-window))
	  (while (not (eq current-window start-window))
	    (if (eq buffer (window-buffer current-window))
		(progn
		  ;; We've found another window with the same buffer.
		  (setq edges (window-edges current-window))
		  (setq new-x (car edges))
		  (setq new-y (car (cdr edges)))
		  (if (or (and empire-shell-window-upper
			       (< new-y result-y))
			  (and (not empire-shell-window-upper)
			       (> new-y result-y))
			  (and empire-shell-window-right
			       (> new-x result-x))
			  (and (not empire-shell-window-right)
			       (< new-x result-x))
			  )
		      (progn
			(setq result-window current-window)
			(setq result-x new-x)
			(setq result-y new-y)
			)
		    )
		  )
	      )
	    (setq current-window (next-window current-window))
	    )
	  result-window
	  )
      nil			;;; there is no window -- just return nil
      )
    )
  )

;;
;; This silly function should really be made into a macro.
;; All it does is insure that the point is properly updated so that the
;; cursor moves to the correct position when switching windows.
;;
(defun move-empire-buffer-point (here)
  "Move point and the window point to `HERE'.  This function assumes that
the current buffer is the empire shell buffer."
  (let (
	(empire-window (empire-get-window-to-change empire-shell-buffer))
	)
    (goto-char here)
    ;;(point-wysiwyg)
    (if empire-window
	(set-window-point empire-window here)
      )
    )
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Map access
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun get-map-sector ()
  "Get the x-y location of the sector under the cursor as a cons cell,
  (X . Y)."
  (switch-to-map-buffer-if-necessary
   (visually-normalize-sect (cons (map-x) (map-y))))
  )

(defun get-map-sector-string ()
  "Get the x-y location of the sector under the cursor as a string,
\"X,Y\"."
  (let (sect x y)
    (setq sect (get-map-sector))
    (setq x (car sect))
    (setq y (cdr sect))
    (format "%s,%s" x y)
    )
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Game Sensitive Prompts
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; This could use some sprucing up.
;;
(defun empire-prompt-direction (&optional prompt initial default allow-dash)
  "Prompt for a single character direction letter, and return it as a string."
  (let (dir)
    (if (not prompt)
	(progn
	  (if default
	      (setq prompt
		    (format "Direction (a single letter: one of `juygbn') [%s]? "
			    default
			    ))
	    (setq prompt "Direction (a single letter: one of `juygbn')? ")
	    )
	  )
      )
    (setq dir (empire-drop-spaces (read-from-minibuffer prompt initial)))
    (if (string= dir "")
	(progn
	  (if (not default)
	      (error "Error: there is not default direction -- you must specify a direction")
	    )
	  (setq dir default)
	  )
      (progn
	(if (and (not (string-match "^[juygbnh]$" dir))
		 (or (not allow-dash)
		     (and allow-dash
			  (not (string= dir "-")))
		     )
		 )
	    (error "Error: `%s' is not a valid direction.")
	  )
	)
      )
    dir
    )
  )

(defun prompt-for-item (items &optional default)
  "Prompt for a commodity"
  (let (item)
    (setq item (completing-read "Item? " items nil t default))
    (if (string= item "")
	(error "Error: No commodity specified.")
      )
    item
    )
  )

(defun empire-read-number-from-minibuffer (prompt &optional
						  initial default-value)
  "Read a number from the minibuffer."
  (let (value)
    (setq value (empire-drop-spaces (read-from-minibuffer prompt initial)))
    (if (string= value "")
	(progn
	  (if (not default-value)
	      (error "Error: there is no default value -- you must specify a number.")
	    )
	  (setq value default-value)
	  )
      (progn
	(if (not (string-match "^[-+]?[0-9]+$" value))
	    (error (format "Error: `%s' is not a number." value))
	  )
	(setq value (string-to-int value))
	)
      )
    value
    )
  )

(defun empire-prompting-read-items (x y action items)
  "Prompt and read for people/commodity to move, explore, etc."
  (let (item amount left)
    (setq item (prompt-for-item items))
    (setq left (recall x y (dynamic-position-of (read item))))
    (setq amount (empire-read-number-from-minibuffer
		  (format "Amount to %s (%s)? " action (or left "?")) nil))
    (list item amount)
    )
  )

(defun empire-read-sector-or-realm (prompt &optional initial)
  "Read a sector coordinate, a range of sector coordinates, or a realm
# from the minibuffer.  The string \".\" is special-cased to mean the
current map sector."
  (if (not initial) (setq initial empire-last-sectors))
  (let (value sect x y)
    (setq value (empire-drop-spaces
		 (read-from-minibuffer
		 (format "%s (default \"%s\"): " prompt initial) )))
    (if (string-match "^[ \t]*$" value) (setq value initial))
    (if (string= value ".")
	(progn
	  (setq sect (get-map-sector))
	  (setq x (car sect))
	  (setq y (cdr sect))
	  (setq value (format "%s,%s" x y))
	  )
      )
    (if (and (not (string-match "," value))
	     (not (string-match "^#" value)))
	(progn
	  (setq value (concat "#" value))
	  ;;(setq empire-last-sectors value) ; ?
	  )
      )
    (if (string-match
	 "\\(^\\.$\\)\\|\\(^#[0-9]*$\\)\\|\\(^-?[0-9]+,-?[0-9]+$\\)\\|\\(^-?[0-9]+:-?[0-9]+,-?[0-9]+:-?[0-9]+$\\)"
	 value)
	(setq empire-last-sectors value)
      (error "`%s' is not a valid sector, sector area, or realm." value)
      )
    value
    )
  )

(defun empire-prompt-read-condition ()
  "Prompt the user for a sector condition expression (e.g., \"?civ>400\")
only if a prefix arg was given"
  (if current-prefix-arg
      (let (condition)
	(setq condition (read-from-minibuffer
			 (format "Condition (default \"%s\"): "
				 empire-last-condition)
			 ""))
	(if (string-match "^$" condition)
	    (setq condition empire-last-condition)
	  (setq empire-last-condition
		(setq condition (empire-drop-spaces condition))))
	;; Remove any embedded spaces.
	;; Gads, what a mess.
	(while (string-match "^\\([^ \t]*\\)\\([ \t]+\\)\\(.*\\)" condition)
	  (setq condition (concat (substring condition (match-beginning 1)
					     (match-end 1))
				  (substring condition (match-beginning 3)
					     (match-end 3))
				  )
		)
	  )
	;; Remove any question marks.
	(while (string-match "^\\([^?]*\\)\\(\\?+\\)\\(.*\\)" condition)
	  (setq condition (concat (substring condition (match-beginning 1)
					     (match-end 1))
				  (substring condition (match-beginning 3)
					     (match-end 3))
				  )
		)
	  )
	;;
	;; We really need to remove any spaces from the condition string.
	;; Also we should remove any "?".
	;;
	(if (string= condition "")
	    condition
	  (concat "?" condition)
	  )
	)
    ;; no arg, no condition
    "")
  )

(defun empire-prompt-read-designation (&optional prompt)
  "Read a sector designation, and return it as a string."
  (let (des)
    (setq des (empire-drop-spaces (read-string (or prompt "Des? "))))
    (if (not (string-match (concat empire-sector-types-regexp "$") des))
	(error (format "Unknown sector designation `%s'" des)))
    des
    )
  )

(defun empire-prompt-threshold (&optional prompt initial default)
  "Prompt for a threshold.  No error checking is done, aside from syntax
checking.  The threshold is returned as a string.  Note that the empty
string is considered a valid threshold (it means, \"there is no
 threshold\".).
"
  (let (threshold)
    (if (not prompt)
	(progn
	  (if default
	      (setq prompt (format "Threshold [%s]? " default))
	    (setq prompt "Threshold? ")
	    )
	  )
      )
    (if (not (string-match "\\(^([0-9]+)$\\)\\|\\(^[9-0]+$\\)\\|\\(^$\\)"
			   (setq threshold
				 (empire-drop-spaces
				  (read-from-minibuffer prompt initial)))))
	(error "Error: `%s' is not a valid threshold.")
      )
    threshold
    )
  )

;;
;; The following function exists because we do not like the way `y-or-n-p'
;; and `yes-or-no-p' operate.
;;
(defun empire-prompt-yes-or-no (prompt &optional initial default)
  "Get a yes-or-no answer from the user.  The user can type any of:
	y
	yes
	n
	no
Case is insensitive.
"
  (let (answer)
    (catch 'exit
      (while t
	(setq answer (empire-drop-spaces
		      (downcase (read-from-minibuffer prompt initial))))
	(if (not (assoc answer '( ("y") ("yes") ("n") ("no"))))
	    (progn
	      (if (and (string= answer "") default)
		  (throw 'exit default)
		)
	      )
	  (throw 'exit answer)
	  )
	(message "Warning: `%s' is not a valid yes/no answer.  Try again.")
	(sit-for 2)			; Sit for 2 seconds or until the user
					; presses a key.
	)
      )
    )
  )


;;
;; This will have to be fixed when ship info is kept.
;; We would like things such as ship number (name?) completion, such as
;; completions based upon a string list such as:
;;
;;	( "42 (100% eff cargo boat at -12,44 (at sea))"
;;	  "45 (98% eff fishing boat at -45,89 (45% eff harbor))"
;;	  "58 (100% battleship at 92,2 (at sea))"
;;	)
;;
;; We might also want the completion list to contain stuff such as mobility,
;; petrol (for those games that require petrol for ships), etc.  How far
;; should this go?  The more ships there are, the SLOWER things will go.
;;
(defun empire-prompt-read-ship ()
  "Prompt the user for a fleet or ship number."
  (let (ship ship-list)
    (setq ship-list '("*"))		; FIX ME!  Use list of valid
					; ships and fleets.
    (setq ship (completing-read "Ship, fleet, or sectors? " ship-list
				nil nil "*"))
    ;; Extensive error checking will have to be added later
    ship
  )
)

