#ifdef BSD
#include <sys/ioctl.h>
#else
#ifdef HPUX
#include <sys/termio.h>
#else
#include <sys/termios.h>
#endif
#endif

#include <signal.h>
#include "type.h"
#include "main.h"

char intrchar, erasechar, eraseword, eraseline;
unsigned con_timeout = 0;
bool silent_con = False;

static void SignalHandler ();

int
getfdtablesize()
{
#ifdef hpux
        return _NFILE;
#else
        return getdtablesize();
#endif
}
	
#ifdef HPUX
_noecho(fd)
        int     fd;
{
        struct  termio io;

        echomode = 0;
        (void) ioctl(fd, TCGETA, &io);
        io.c_line |= ECHO;
        (void) ioctl(fd, TCSETA, &io);
}

_echo(fd)
        int     fd;
{
        struct  termio io;

        if (echomode)
                return;
        (void) ioctl(fd, TCGETA, &io);
        io.c_line &= ~ECHO;
        (void) ioctl(fd, TCSETA, &io);
        echomode++;
}
#endif

#ifdef BSD
void GetSpecialChars ()
{
	struct tchars tc;
	struct ltchars ltc;
	struct sgttyb mode;

	if ((ioctl(1, TIOCGETP, &mode) < 0) || 
	    (ioctl(0, TIOCGLTC, &ltc) < 0) || 
	    (ioctl(0, TIOCGETC, &tc) < 0 )) 
	{   
		intrchar = CINTR;
		erasechar = CERASE;
		eraseword = CWERASE;
		eraseline = CKILL;
	}
	else
	{
		intrchar = tc.t_intrc;
		erasechar = mode.sg_erase;
		eraseword = ltc.t_werasc;
		eraseline = mode.sg_kill;
	}
}

#else

void GetSpecialChars ()
{
#ifdef HPUX
        struct termio tt;
#else
	struct termios tt;
#endif
	
#ifdef __ultrix
	if (ioctl (0, TCGETP, &tt) < 0)
#else
	    
#ifdef HPUX
	if (ioctl (0, TCGETA, &tt) < 0)
#else
	if (ioctl (0, TCGETS, &tt) < 0)
#endif /* HPUX */
	    
#endif
	{
		intrchar = '\003';	/* ^C */
		erasechar = '\008';	/* ^H */
		eraseword = '\027';	/* ^W */
		eraseline = '\026';	/* ^X */
	}
	else
	{
		intrchar = tt.c_cc[VINTR];
		erasechar = tt.c_cc[VERASE];
		eraseword = tt.c_cc[VWERASE];
		eraseline = tt.c_cc[VKILL];
	}
}

#endif

void Forceleave ()
{
	static int sig = 0;

	if (sig++ != 0) {
		if (EmpireStatus () != E_DEAD) {
			fprintf (stderr, "Aborting connection\n");
			sigsetmask (0);
			leave0 ();
		}
		else  {
			fprintf (stderr, "Closing Down be patient\n");
			return;
		}
	}

	fprintf (stderr, "Closing connection\n");
	sigsetmask (0);
	leave ();
}

void SetSignalsOn ()
{
	void Disconnect ();

	signal (SIGHUP,  SignalHandler);
	signal (SIGILL,  SignalHandler);
	signal (SIGFPE,  SignalHandler);
	signal (SIGBUS,  SignalHandler);
	signal (SIGSEGV, SignalHandler);
	signal (SIGQUIT, SignalHandler);
	signal (SIGTERM, SignalHandler);
	signal (SIGINT, Forceleave);

	if (con_timeout > MIN_CONTIMEOUT)
		signal (SIGALRM, Disconnect);
	else
		signal (SIGALRM, SIG_IGN);
}

void SetSignalsOff ()
{
	signal (SIGINT, SIG_IGN);
	signal (SIGQUIT, SIG_IGN);

	if (con_timeout > MIN_CONTIMEOUT)
		signal (SIGALRM, SIG_IGN);
}

static char * SignalName (i)
int i;
{
	switch (i)
	{

	case SIGHUP:
		return "Hang Up Signal (SIGHUP)";

	case SIGILL:
		return "Illegal instruction (SIGILL)";

	case SIGFPE:
		return "Floating point exception (SIGFPE)";
	
	case SIGBUS:
		return "Bus error (SIGBUS)";

	case SIGSEGV:
		return "Segmentation violation (SIGSEGV)";

	case SIGQUIT:
		return "Quit Signal";
	
	case SIGTERM:
		return "Software signal";

	default:
		return Fmt ("Unexpected signal (%d)", i);
	
	}
}

bool dump_core = False;

static void SaveDataDumpCore (win, starty)
WinInfo win;
int starty;
{
	char * ans;

	do {
		interrupt = False;
		ans = GetQuest (win, starty, "Save Data [y]: ", 1, "yn");
	}
	while (ans == (char *) 0 || interrupt);

	if (* ans == 'n')
		command_state = FASTQUIT_STATE;

	do {
		interrupt = False;
		ans = GetQuest (win, starty + 1, "Dump Core [n]: ", 1, "yn");
	}
	while (ans == (char *) 0 || interrupt);

	dump_core = * ans == 'y' ? True : False;
	
	sigsetmask (0);
	leave ();
}

void Panic (funcname, filename, mes)
char * funcname, * filename, * mes;
{
	WinInfo win;
	char * ans;

	win = OpenRelToRoot (map_win, 2, 2, 80, 13, CHARS, 0);

	PrintN (win, 1, 1, "Panic: Internal Error !");
	PrintN (win, 1, 3, Fmt ("Function: %s", funcname));
	PrintN (win, 1, 4, Fmt ("File: %s", filename));
	PrintB (win, 1, 6, mes);

	do {
		interrupt = False;
		ans = GetQuest (win, 8, "Continue [y]: ", 1, GS_YN);
	}
	while (ans == (char *) 0 || interrupt);
	
	if (* ans == '\0' || * ans == 'y' || * ans == 'Y')
	{
		DestroyWindow (win);
		return;
	}

	SaveDataDumpCore (win, 10);
}

static void SignalHandler (i)
int i;
{
	WinInfo win;
	static int sig = 0;

	if (sig++ != 0) {
		if (i == SIGQUIT) {
			if (EmpireStatus () != E_DEAD) {
				fprintf (stderr, "Aborting connection\n");
				sigsetmask (0);
				leave0 ();
			}
			fprintf (stderr, "Closing Down be patient\n");
			return;
		}

		CloseConnection ();
		exit (-1);
	}

#ifdef TERMC_VERSION
	win = OpenRelToRoot (map_win, 2, 2, 60, 7, FULL_BOX, 0);
#else
	win = OpenRelToRoot (map_win, 2, 2, 80, 7, CHARS, 0);
#endif /* TERMC_VERSION */

	PrintN (win, 1, 1, "-=[ SIGNAL CAUGHT ]=-");

	PrintN (win, 1, 3, Fmt ("---> %s", SignalName (i)));

	if (i == SIGHUP)
		leave ();

	SaveDataDumpCore (win, 4);
}

void ResetConTimeout ()
{
	if (con_timeout > MIN_CONTIMEOUT) alarm (con_timeout);
}

void NoConTimeout ()
{
	if (con_timeout > MIN_CONTIMEOUT) alarm (0);
}

bool disconnecting = False;

void Disconnect (i)
int i;
{

	NoConTimeout ();

	if (EmpireStatus () != E_COMMAND) {
		alarm (con_timeout);
		return;
	}

	if (! silent_con && i == SIGALRM)
		Message ("Timesaver timeout, disconnecting....");

	disconnecting = True;

	AbortClient (silent_con ? DONT_PRINT : PRINT);

	disconnecting = False;
}

void FastQuit ()
{
	command_state = FASTQUIT_STATE;
	leave ();
}

void DoQuit ()
{
	if (BufferdCommands ())
		if (! Confirm (
		  	    "You still have buffered commands, really quit ?",
			    False))
			return;
	
	leave ();
}
