# tips.tcl --
#
#	The Tips Window and related commands.
#

namespace eval NSTips {

variable Priv

# NSTips::InitModule --
#
#	One-time-only-ever initialization.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc InitModule {} {

	variable Priv

	set Priv(current) [Value tip,current]
	set Priv(showTips) [Value tip,show]
#	NSValueManager::AddClient showTips \
#		"set NSTips::Priv(showTips) [Value showTips]"

	ReadTipsFile

	NSObject::New NSTips
}

# NSTips::CloseModule --
#
#	One-time-only-ever cleanup.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc CloseModule {} {
}

# NSTips::ReadTipsFile --
#
#	Read the tips.txt file.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc ReadTipsFile {} {

	global Angband
	variable Priv

	if {[catch {open [file join $Angband(dirTK) doc tips.txt]} fileId]} {
		set msg "The following error occurred while attempting to open "
		append msg "the \"tips.txt\" file for reading:\n$fileId\nQuit now?"
		set answer [tk_messageBox -title "$Angband(name) Error" \
			-message $msg -type yesno -icon question]
		if {$answer == "yes"} [angband game abort]
		return
	}

	set Priv(count) 0
	while 1 {
		if {[gets $fileId lineBuf] <= 0} break
		lappend Priv(tips) $lineBuf
		incr Priv(count)
	}

	close $fileId
}

# NSTips::NSTips --
#
#	Object constructor called by NSObject::New().
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc NSTips oop {

	global Windows

	InitWindow $oop

	DisplayNextTip $oop

	#
	# Global list of application windows
	#

	set Windows(tip,win) [Info $oop win]
	set Windows(tip,class) NSTips
	set Windows(tip,oop) $oop
}

# NSAssign::Info --
#
#	Query and modify info.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc Info {oop info args} {

	global NSTips

	# Set info
	if {[llength $args]} {
		switch -- $info {
			default {
				set NSTips($oop,$info) [lindex $args 0]
			}
		}

	# Get info
	} else {
		switch -- $info {
			default {
				return $NSTips($oop,$info)
			}
		}
	}
}

# NSTips::InitWindow --
#
#	Initialize the Tips Window.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc InitWindow oop {

	global Angband
	global Windows
	variable Priv

	set win .tips$oop
	toplevel $win
	wm title $win "$Angband(name) Tips"

	wm transient $win $Windows(main,win)

	# Do stuff when window closes
	wm protocol $win WM_DELETE_WINDOW "NSTips::Close $oop"

	Info $oop win $win

	# Header
	label $win.header \
		-text "Welcome to $Angband(name)" -font "Times 18 bold"

	# Tip display
	frame $win.tip \
		-borderwidth 1 -relief sunken
	text $win.tip.text \
		 -font {Courier 9} -width 46 -height 10 -cursor {} \
		-background #FFFFE0 -borderwidth 0 -highlightthickness 0 \
		-wrap word
	bindtags $win.tip.text [list $win.tip.text $win all]
	pack $win.tip.text \
		-fill both

	frame $win.misc \
		-borderwidth 0

	# Checkbutton
	checkbutton $win.misc.checkshow \
		-text "Show this window every time the game starts" \
		-variable NSTips::Priv(showTips) \
		-command "Value tip,show \$NSTips::Priv(showTips)"

	# Tip number
	label $win.misc.tipnumber \
		-anchor e

	pack $win.misc.checkshow \
		-side left -expand no
	pack $win.misc.tipnumber \
		-side right -expand no

	# Divider
	frame $win.divider1 \
		-borderwidth 1 -relief sunken -height 2

	# Buttons
	frame $win.buttons \
		-borderwidth 0
	button $win.buttons.next \
		-text "Next >" -command "NSTips::DisplayNextTip $oop" -width 11 \
		-underline 0
	button $win.buttons.back \
		-text "< Back" -command "NSTips::DisplayPreviousTip $oop" -width 11 \
		-underline 0
	button $win.buttons.close \
		-text "Close" -command "NSTips::Close $oop" -width 11
	pack $win.buttons.close \
		-side right -pady 5 -padx 5
	pack $win.buttons.next \
		-side right -pady 5 -padx 5
	pack $win.buttons.back \
		-side right -pady 5 -padx 5

	# Geometry
	grid columnconfigure $win 0 -weight 0
	grid rowconfigure $win 0 -weight 0
	grid rowconfigure $win 1 -weight 0
	grid rowconfigure $win 2 -weight 0
	grid rowconfigure $win 3 -weight 0
	grid rowconfigure $win 4 -weight 0

	grid $win.header \
		-row 0 -column 0 -rowspan 1 -columnspan 2 -sticky w
	grid $win.tip \
		-row 1 -column 0 -rowspan 1 -columnspan 1 -padx 5 -pady 5
	grid $win.misc \
		-row 2 -column 0 -rowspan 1 -columnspan 2 -padx 5 -sticky ew
	grid $win.divider1 \
		-row 3 -column 0 -rowspan 1 -columnspan 1 -sticky ew -padx 5
	grid $win.buttons \
		-row 4 -column 0 -rowspan 1 -columnspan 1 -sticky e

	bind $win <KeyPress-Left> "tkButtonInvoke $win.buttons.back"
	bind $win <KeyPress-b> "tkButtonInvoke $win.buttons.back"
	bind $win <KeyPress-Right> "tkButtonInvoke $win.buttons.next"
	bind $win <KeyPress-n> "tkButtonInvoke $win.buttons.next"
	bind $win <KeyPress-Escape> "tkButtonInvoke $win.buttons.close"

	WindowPosition $win 2 3
	focus $win
}

# NSTips::Close --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc Close oop {

	variable Priv

	wm withdraw [Info $oop win]
	Value tip,current $Priv(current)
}

# NSTips::DisplayTip --
#
#	Display a tip.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc DisplayTip {oop tip} {

	variable Priv

	set win [Info $oop win]
	set text $win.tip.text

	$text delete 1.0 end

	$text insert end "Did you know...\n" tag1
	$text tag configure tag1 -font "helvetica 10 bold" \
		-lmargin1 20 -spacing1 20 -spacing3 15

	$text insert end $tip tag2
	$text tag configure tag2 -font "helvetica 10" \
		-lmargin1 20 -lmargin2 20 -rmargin 20

	$win.misc.tipnumber configure -text "[expr $Priv(current) + 1]/$Priv(count)"
}

# NSTips::DisplayNextTip --
#
#	Display the next tip, wrapping if required.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc DisplayNextTip oop {

	variable Priv

	set current $Priv(current)
	incr current
	if {$current >= $Priv(count)} {
		set current 0
	}
	set Priv(current) $current
	DisplayTip $oop [lindex $Priv(tips) $current]
}

# NSTips::DisplayPreviousTip --
#
#	Display the previous tip, wrapping if required.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc DisplayPreviousTip oop {

	variable Priv

	set current $Priv(current)
	incr current -1
	if {$current < 0} {
		set current [expr $Priv(count) - 1]
	}
	set Priv(current) $current
	DisplayTip $oop [lindex $Priv(tips) $current]
}

# namespace eval NSTips
}
