/***********************************************************************
 *
 * PROJECT:	  PMake
 * MODULE:	  Customs -- BSD UNIX dependencies
 * FILE:	  os-bsd.c
 *
 * AUTHOR:  	  Adam de Boor: Sep  9, 1989
 *
 * ROUTINES:
 *	Name	  	    Description
 *	----	  	    -----------
 *	OS_Init	    	    Initialize module and return mask of criteria
 *	    	    	    to be considered in determination.
 *	OS_Exit	    	    Undo any initialization.
 *	OS_Idle	    	    Return machine idle time, in seconds
 *	OS_Load	    	    Return machine load factor
 *	OS_Swap	    	    Return free swap space
 *	OS_Proc	    	    Return number of free processes
 *
 * REVISION HISTORY:
 *	Date	  Name	    Description
 *	----	  ----	    -----------
 *	9/ 9/89	  ardeb	    Initial version
 *	7/91      stolcke   Extension for NeXT/Mach and Mips/Ultrix
 *	9/24/91   stolcke   Free processes support added
 *	2/19/92   stolcke   HP-UX port
 *	2/92      stolcke   OS_Exit added
 *
 * DESCRIPTION:
 *	OS-dependent functions for BSD-related systems. This includes:
 *	    SunOS 3.x
 *	    Ultrix
 *	    BSD 4.2 and 4.3
 *	    Mach on the NeXT
 *          HP-UX (believe it or not)
 *
 *	These functions are responsible for the determination of the
 *	current state of the system, returning it as a set of numbers
 *	in a customs-standard form, whatever form they may be in in the
 *	system itself.
 *
 *	The format used is the same as that transmitted for the AVAIL_SET
 *	RPC.
 *
 * 	Copyright (c) Berkeley Softworks 1989
 * 	Copyright (c) Adam de Boor 1989
 *
 * 	Permission to use, copy, modify, and distribute this
 * 	software and its documentation for any non-commercial purpose
 *	and without fee is hereby granted, provided that the above copyright
 * 	notice appears in all copies.  Neither Berkeley Softworks nor
 * 	Adam de Boor makes any representations about the suitability of this
 * 	software for any purpose.  It is provided "as is" without
 * 	express or implied warranty.
 *
 *
 ***********************************************************************/
#ifndef lint
static char *rcsid =
"$Id: os-bsd.c,v 1.14 1992/07/31 00:12:31 stolcke Exp $ ICSI (Berkeley)";
#endif lint

#include    <sys/param.h>
#ifdef mips
#include    <sys/fixpoint.h>
#endif
#include    <sys/stat.h>
#include    <sys/conf.h>
#include    <nlist.h>
#include    <sys/file.h>
#include    <time.h>
#include    <sys/proc.h>

#include    "customsInt.h"

#ifdef __MACH__		/* Mach has slighly different struct nlist */
#define n_symbol		n_un.n_name
#define NLIST_AVENRUN		{"_avenrun"}
#define NLIST_PROC		{"_proc"}
#define NLIST_NPROC		{"_nproc"}
#define NLIST_END		{NULL}
#define KERNEL			"/mach"
#define NO_IDLE			/* console access time is meaningless */
#define NO_SWAP			/* doesn't have fixed swap space */
#endif
#ifdef hpux
#define NLIST_AVENRUN		"avenrun"
#define NLIST_PROC		"proc"
#define NLIST_NPROC		"nproc"
#define KERNEL			"/hp-ux"
#define KEYBOARD		"/dev/hil1"
#define KEYBOARD2		"/dev/hil2"
#define NO_SWAP			/* doesn't have BSD kernel structures */
#endif
#ifdef sun
#define KEYBOARD		"/dev/kbd"
#define KBD_OPEN		/* can keep device open */
#endif

				/* defaults for most BSD systems */
#ifndef n_symbol
#define n_symbol		n_name
#endif
#ifndef KERNEL
#define KERNEL			"/vmunix"
#endif
#ifndef KEYBOARD
#define KEYBOARD		"/dev/console"
#endif
#ifndef KMEM
#define KMEM			"/dev/kmem"
#endif

#ifndef NLIST_AVENRUN
#define NLIST_AVENRUN		"_avenrun"
#endif
#ifndef NLIST_PROC
#define NLIST_PROC		"_proc"
#endif
#ifndef NLIST_NPROC
#define NLIST_NPROC		"_nproc"
#endif
#ifndef NLIST_END
#define NLIST_END		NULL
#endif

#ifndef NO_SWAP
#include    <sys/map.h>
#ifndef mapstart
/*
 * Apparently, Ultrix doesn't have these macros...
 */
#define mapstart(X)	(struct mapent *)((X)+1)
#define mapfree(X)	(X)->m_free
#define mapwant(X)	(X)->m_want
#define mapname(X)	((struct maplast *)(X))->m_nam
#endif /* mapstart */
#endif /* !NO_SWAP */

static struct nlist kAddrs[] = {
{	NLIST_AVENRUN	},   	/* Load averages    	    	*/
#define AVENRUN	  	0
{	NLIST_PROC	},   	/* Process table base  	    	*/
#define PROCBASE	1
{	NLIST_NPROC	},   	/* Process table size  	    	*/
#define NPROC		2
#ifndef NO_SWAP
{	"_nswapmap"	},   	/* Number of swap resource maps */
#define NSWAPMAP  	3
{	"_nswdev" 	},   	/* Number of swap devices   	*/
#define NSWDEV	  	4
{	"_swapmap"	},   	/* The swap resource maps   	*/
#define SWAPMAP	  	5
{	"_swdevt" 	},   	/* The swap devices 	    	*/
#define SWDEVT	  	6
#endif /* !NO_SWAP */
{	NLIST_END   	}
};

static int  	  	kmem;	    	/* Descriptor to /dev/kmem */
#ifndef NO_IDLE
#ifdef KBD_OPEN
static int  	  	kbd;	    	/* Descriptor to /dev/kbd */
#endif
#endif

#ifndef NO_SWAP
static int  	  	swblocks; 	/* Total swap space available */
static int  	  	nSwapMap; 	/* Number of entries in the swap map */
static off_t	  	swapMapAddr;	/* Address in sysspace of the map */
static struct map 	*swapMap; 	/* Space for swap map */
static int  	  	swapMapSize;	/* Size of the swap map (bytes) */
#endif /* NO_SWAP */


/*
 * read data from kernel memory
 */
static int
kmem_read(addr, buffer, nbytes)
	off_t addr;
	char *buffer;
	unsigned nbytes;
{
    if (lseek (kmem, addr, L_SET) != addr ||
        read (kmem, buffer, nbytes) != nbytes)
	return (-1);
    else
	return (0);
}


/***********************************************************************
 *				OS_Init
 ***********************************************************************
 * SYNOPSIS:	    Initialize this module
 * CALLED BY:	    Avail_Init
 * RETURN:	    Mask of AVAIL_* bits indicating what criteria are
 *	    	    to be examined.
 * SIDE EFFECTS:
 *
 * STRATEGY:
 *
 * REVISION HISTORY:
 *	Name	Date		Description
 *	----	----		-----------
 *	ardeb	9/ 9/89		Initial Revision
 *	stolcke	7/31/91		more robustness against unavailable info
 *	stolcke	9/24/91		Rewritten in terms of kmem_read()
 *
 ***********************************************************************/
int
OS_Init()
{
    struct swdevt   *swapDevices,
		    *sw;
    int	    	    numSwapDevices;
    int	    	    retMask;
    struct stat	    kbStat;
    
    /*
     * Default to everything.
     */
    retMask = AVAIL_EVERYTHING;

#ifndef NO_IDLE
    /*
     * Try for a keyboard device. It's ok if we can't open this thing. It
     * just means we're not on a workstation and so can't determine idle time.
     */
#ifdef KBD_OPEN
    if ((kbd = open (KEYBOARD, O_RDONLY, 0)) < 0) {
#else
    if (stat (KEYBOARD, &kbStat) < 0) {
#endif
	/*
	 * If couldn't open/stat keyboard, we can't tell how long the machine's
	 * been idle.
	 */
	xlog (XLOG_WARNING,
		"OS_Init: cannot open or stat %s: disabling idle time check", 
		KEYBOARD);
	retMask &= ~AVAIL_IDLE;
    }
#else
    retMask &= ~AVAIL_IDLE;
#endif

    /*
     * Extract the addresses for the various data structures that we examine.
     * XXX: Perhaps this thing should allow some other name than /vmunix?
     */
    if (nlist (KERNEL, kAddrs) < 0) {
        xlog (XLOG_WARNING,
		"OS_Init: cannot locate %s symbols: disabling load/swap/proc checks",
		KERNEL);
	return (retMask & ~(AVAIL_LOAD | AVAIL_SWAP | AVAIL_PROC));
    }

    /*
     * Open a stream to the kernel's memory so we can actually look at the
     * data structures.
     */
    if ((kmem = open (KMEM, O_RDONLY, 0)) < 0) {
        xlog (XLOG_WARNING,
		"OS_Init: cannot open %s: disabling load/swap/proc checks",
		KMEM);
	return (retMask & ~(AVAIL_LOAD | AVAIL_SWAP | AVAIL_PROC));
    }

    /*
     * Check for presence of avenrun symbol
     */
    if (!kAddrs[AVENRUN].n_type) {
        xlog (XLOG_WARNING,
		"OS_Init: cannot locate %s: disabling load check",
		kAddrs[AVENRUN].n_symbol);
	retMask &= ~AVAIL_LOAD;
    }

    /*
     * Check for presence of process table info
     */
    if (!kAddrs[PROCBASE].n_type || !kAddrs[NPROC].n_type) {
	xlog (XLOG_WARNING,
		"OS_Init: cannot locate %s: disabling proc check",
		(!kAddrs[PROCBASE].n_type ? kAddrs[PROCBASE].n_symbol :
		                            kAddrs[NPROC].n_symbol));
	retMask &= ~AVAIL_PROC;
    }

#ifndef NO_SWAP
    /*
     * Check for presence of swap device symbols
     */
    if (!kAddrs[NSWAPMAP].n_type || !kAddrs[NSWDEV].n_type ||
        !kAddrs[SWAPMAP].n_type || !kAddrs[SWDEVT].n_type ) {
	xlog (XLOG_WARNING,
		"OS_Init: cannot locate %s: disabling swap check",
		(!kAddrs[NSWAPMAP].n_type ? kAddrs[NSWAPMAP].n_symbol :
		 (!kAddrs[NSWDEV].n_type ? kAddrs[NSWDEV].n_symbol :
		  (!kAddrs[SWAPMAP].n_type ? kAddrs[SWAPMAP].n_symbol :
		                             kAddrs[SWDEVT].n_symbol))));
	retMask &= ~AVAIL_SWAP;
    }
    else {
	/*
	 * Find the total number of swap blocks available to the machine
	 * by summing the amounts in the swdevt descriptors
	 */
	if (kmem_read((off_t)kAddrs[NSWDEV].n_value,
	              (char *)&numSwapDevices, sizeof(numSwapDevices)) < 0) {
		xlog (XLOG_WARNING,
			"OS_Init: cannot read %s: disabling swap check",
			kAddrs[NSWDEV].n_symbol);
		return (retMask & ~AVAIL_SWAP);
	}

	swapDevices =
	    (struct swdevt *)malloc (numSwapDevices * sizeof (struct swdevt));

	if (kmem_read((off_t)kAddrs[SWDEVT].n_value,
	      (char *)swapDevices, numSwapDevices*sizeof(struct swdevt)) < 0) {
		free ((Address) swapDevices);
		xlog (XLOG_WARNING,
			"OS_Init: cannot read %s: disabling swap check",
			kAddrs[SWDEVT].n_symbol);
		return (retMask & ~AVAIL_SWAP);
	}

	for (swblocks=0, sw=swapDevices; numSwapDevices!=0; sw++, numSwapDevices--)
	{
	    if (sw->sw_freed) {
		swblocks += sw->sw_nblks;
	    }
	}
	free ((Address) swapDevices);

	/*
	 * Find and save the number and location of the swap maps for
	 * the local machine, then allocate enough room to hold them
	 * all, pointing 'swapMap' to the space.
	 */
	if (kmem_read((off_t)kAddrs[NSWAPMAP].n_value,
	              (char *)&nSwapMap, sizeof(nSwapMap)) < 0 ||
	    kmem_read((off_t) kAddrs[SWAPMAP].n_value,
	              (char *)&swapMapAddr, sizeof(swapMapAddr)) < 0) {
		xlog (XLOG_WARNING,
			"OS_Init: cannot read %s/%s: disabling swap check",
			kAddrs[NSWAPMAP].n_symbol, kAddrs[SWAPMAP].n_symbol);
		return (retMask & ~AVAIL_SWAP);
	}
	
	if (verbose)
	    xlog (XLOG_DEBUG,
		    "OS_Init: %d swap blocks total allocated among %d maps at 0x%08x",
		    swblocks, nSwapMap, swapMapAddr);
	swapMapSize = nSwapMap * sizeof (struct map);

	swapMap = (struct map *) malloc (swapMapSize);
    }
#else
    retMask &= ~AVAIL_SWAP;
#endif /* NO_SWAP */

    return (retMask);
}

/***********************************************************************
 *				OS_Exit
 ***********************************************************************
 * SYNOPSIS:	    Deinitialize this module
 * CALLED BY:	    CustomsRestart()
 * RETURN:	    0
 * SIDE EFFECTS:    kmem and kbd are closed.
 *
 * STRATEGY:
 *
 * REVISION HISTORY:
 *	Name	Date		Description
 *	----	----		-----------
 *	stolcke	2/20/92		Initial Revision
 *
 ***********************************************************************/
int
OS_Exit()
{
    if (kmem)
	close(kmem);
#ifndef NO_IDLE
#ifdef KBD_OPEN
    if (kbd >= 0)
	close(kbd);
#endif
#endif
    return (0);
}

/***********************************************************************
 *				OS_Idle
 ***********************************************************************
 * SYNOPSIS:	    Find the idle time of the machine
 * CALLED BY:	    Avail_Local
 * RETURN:	    The number of seconds for which the machine has been
 *	    	    idle.
 * SIDE EFFECTS:    None
 *
 * STRATEGY:
 *	Locate the access time for the keyboard device and subtract it
 *	from the current time to obtain the number of seconds the
 *	keyboard has been idle. The assumption is that the keyboard
 *	device's idle time reflects that of the machine. This does not
 *	take into account rlogin connections, or non-Sun systems that
 *	are workstations but don't have a keyboard device.
 *
 *
 * REVISION HISTORY:
 *	Name	Date		Description
 *	----	----		-----------
 *	ardeb	9/10/89		Initial Revision
 *	stolcke	9/24/91		Rewritten in terms of kmem_read()
 *	stolcke	2/19/92		Allow for two input devices (HP-UX)
 *
 ***********************************************************************/
long
OS_Idle()
{
#ifdef NO_IDLE
    return (0);
#else
    struct stat	    kbStat;
    struct timeval  now;
    long            atime;

#ifdef KBD_OPEN
    if (fstat (kbd, &kbStat) < 0) {
#else
    if (stat (KEYBOARD, &kbStat) < 0) {
#endif
	xlog (XLOG_ERROR, "OS_Idle: can no longer stat %s", KEYBOARD);
	return (0);
    }

#ifdef KEYBOARD2
    /*
     * With two input devices, take the most recent access time.
     */
    atime = kbStat.st_atime;
    if (stat (KEYBOARD2, &kbStat) != -1 &&
        kbStat.st_atime < atime)
	kbStat.st_atime = atime;
#endif /* KEYBOARD2 */

    gettimeofday (&now, (struct timezone *)0);
	
    atime = now.tv_sec - kbStat.st_atime;
    if (verbose)
	xlog (XLOG_DEBUG, "OS_Idle: last access = %.8s, idle = %ld:%02ld",
	        ctime(&kbStat.st_atime) + 11, atime/60, atime%60);

    return (atime);
#endif /* NO_IDLE */
}


/***********************************************************************
 *				OS_Swap
 ***********************************************************************
 * SYNOPSIS:	    Find the percentage of the system's swap space that
 *	    	    isn't being used.
 * CALLED BY:	    Avail_Local
 * RETURN:	    The percentage of free swap space
 * SIDE EFFECTS:    None
 *
 * STRATEGY:
 *	The number of free blocks is simple the number of blocks described
 *	by the system's swap maps, whose address we've got in swapMapAddr.
 *
 * REVISION HISTORY:
 *	Name	Date		Description
 *	----	----		-----------
 *	ardeb	9/10/89		Initial Revision
 *	stolcke	9/24/91		Rewritten in terms of kmem_read()
 *
 ***********************************************************************/
int
OS_Swap()
{
#ifdef NO_SWAP
    return (0);
#else
    int	    		free;		/* Number of free blocks so far */
    struct mapent	*mapEnd;	/* End of swap maps */
    struct mapent	*mapEntry;	/* Current map */
    
    if (kmem_read ((off_t)swapMapAddr,
	           (char *)swapMap, swapMapSize) < 0)
    {
	xlog (XLOG_ERROR, "OS_Swap: can no longer read %s",
		kAddrs[SWAPMAP].n_symbol);
	return (0);
    }

    mapEnd = (struct mapent *) &swapMap[nSwapMap];
    free = 0;
    for (mapEntry = mapstart(swapMap); mapEntry < mapEnd; mapEntry++) {
	free += mapEntry->m_size;
    }

    if (verbose)
	xlog (XLOG_DEBUG, "OS_Swap: max swap = %d, free blocks = %d (%d%%)",
		swblocks, free, (free * 100) / swblocks);

    return ((free * 100) / swblocks);
#endif /* NO_SWAP */
}


/***********************************************************************
 *				OS_Proc
 ***********************************************************************
 * SYNOPSIS:	    Find the number of additional processes that can be
 *	    	    created.
 * CALLED BY:	    Avail_Local
 * RETURN:	    Number of free process slots
 * SIDE EFFECTS:    None
 *
 * STRATEGY:
 *	Count current no. of processes by scanning process list,
 *	then subtract from max. no. of processes.
 *
 * REVISION HISTORY:
 *	Name	Date		Description
 *	----	----		-----------
 *	stolcke	9/21/91		Initial Revision (dummy version)
 *	stolcke	9/24/91		Working version
 *
 ***********************************************************************/
int
OS_Proc()
{
    long procbase;	/* address of process table */
    int cproc = 0;
    int nproc, i;
    struct proc p;

    if (kmem_read((off_t)kAddrs[PROCBASE].n_value,
	          (char *)&procbase, sizeof(procbase)) < 0 ||
	kmem_read((off_t)kAddrs[NPROC].n_value,
		  (char *)&nproc, sizeof(nproc)) < 0)
    {
	xlog (XLOG_ERROR, "OS_Proc: can no longer read %s/%s",
		kAddrs[PROCBASE].n_symbol, kAddrs[NPROC].n_symbol);
	return (0);
    }

    for (i = 0; i < nproc; i++) {
	if (kmem_read((off_t)procbase + i*sizeof(p),
	              (char *)&p, sizeof(p)) < 0)
		break;
	if (p.p_stat != 0 && p.p_stat != SIDL)
		cproc++;
    }

    if (verbose)
	xlog (XLOG_DEBUG, "OS_Proc: nproc = %d, #proc = %d, free = %d",
		nproc, cproc, nproc - cproc);

    return (nproc - cproc);
}


/***********************************************************************
 *				OS_Load
 ***********************************************************************
 * SYNOPSIS:	    Return the current load average in standard form
 * CALLED BY:	    Avail_Local
 * RETURN:	    The current load as a 32-bit fixed-point number
 * SIDE EFFECTS:    None
 *
 * STRATEGY:
 *
 * REVISION HISTORY:
 *	Name	Date		Description
 *	----	----		-----------
 *	ardeb	9/10/89		Initial Revision
 *	stolcke	7/31/91		MIPS load format support added
 *	stolcke	9/24/91		Rewritten in terms of kmem_read()
 *	stolcke	2/19/92		Some trivial changes for HP-UX
 *
 ***********************************************************************/
unsigned long
OS_Load()
{
/*
 * The existence of the constant FSCALE is used to determine in what format
 * the system's load average is stored. If FSCALE is defined, it indicates
 * a 32-bit fixed-point number is being used. The number is divided by
 * FSCALE using floating point arithmetic to obtain the actual load
 * average.
 *
 * If FSCALE isn't defined (standard BSD), the calculations are performed using
 * double floating point math.
 */
#ifdef FSCALE
    long 	  	avenrun[3];	/* Load averages */
#define FLOAT(v)	((double)(v)/FSCALE)
#else
#ifdef FIX_TO_DBL
    fix 	  	avenrun[3];	/* Load averages */
#define FLOAT(v)	FIX_TO_DBL(v)
#else
    double		avenrun[3];	/* Load averages */
#define FLOAT(v)	(v)
#endif /* FIX_TO_DBL */
#endif /* FSCALE */
    double              result;

    if (kmem_read((off_t)kAddrs[AVENRUN].n_value,
	          (char *)avenrun, sizeof (avenrun)) < 0 )
    {
	xlog (XLOG_ERROR, "OS_Load: can no longer read %s",
		kAddrs[AVENRUN].n_symbol);
	return (0);
    }
    
#ifdef ALL_LOAD
    /*
     * Find largest of the three averages and return that
     */
    if (avenrun[0] > avenrun[1]) {
	if (avenrun[0] > avenrun[2]) {
	    result = FLOAT(avenrun[0]);
	} else {
	    result = FLOAT(avenrun[2]);
	}
    } else if (avenrun[1] > avenrun[2]) {
	result = FLOAT(avenrun[1]);
    } else {
	result = FLOAT(avenrun[2]);
    }
#else
    /*
     * Just return the 1-minute average.
     */
    result = FLOAT(avenrun[0]);
#endif
    if (verbose)
	xlog (XLOG_DEBUG, "OS_Load: avenrun = %.2lf, %.2lf, %.2lf",
	       FLOAT(avenrun[0]), FLOAT(avenrun[1]), FLOAT(avenrun[2]));

    return (result * LOADSCALE);
}
