/*-
 * import.c --
 *	Functions to import processes.
 *
 * Copyright (c) 1988, 1989 by the Regents of the University of California
 * Copyright (c) 1988, 1989 by Adam de Boor
 * Copyright (c) 1989 by Berkeley Softworks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any non-commercial purpose
 * and without fee is hereby granted, provided that the above copyright
 * notice appears in all copies.  The University of California,
 * Berkeley Softworks and Adam de Boor make no representations about
 * the suitability of this software for any purpose.  It is provided
 * "as is" without express or implied warranty.
 */
#ifndef lint
static char *rcsid =
"$Id: import.c,v 1.41 1992/10/05 04:43:11 stolcke Exp $ ICSI (Berkeley)";
#endif lint

#include    <sys/types.h>
#include    <signal.h>
#include    <sys/wait.h>
#include    <sys/time.h>
#include    <sys/file.h>
#include    <stdio.h>
#include    <strings.h>
#include    <varargs.h>
#include    <pwd.h>
#include    <errno.h>

extern int   errno;
extern char *getusershell();

#ifdef NO_VFORK
#define vfork fork
#else

#ifdef sparc
#include    <vfork.h>
#endif

#endif /* NO_VFORK */

#ifdef SYSV
#ifndef SIGCHLD
#define SIGCHLD			SIGCLD
#endif
#endif

#ifdef SYSV
#define KILLPG(pgrp,signo)	kill(-(pgrp),signo)
#else
#define KILLPG(pgrp,signo)	killpg(pgrp,signo)
#endif

#ifdef SYSV
#define SETPGRP()		setpgrp()
#else
#define SETPGRP()		setpgrp(0, getpid())
#endif

#ifdef hpux
#define setreuid(ruid,euid)	setresuid(ruid,euid,-1)
#define setregid(rgid,egid)	setresgid(rgid,egid,-1)
#endif

#include    "customsInt.h"
#include    "lst.h"

/*
 * Permits published by the MCA are kept in ImportPermit structures on the
 * permits list until a process actually arrives for them. If one doesn't
 * arrive before the expiration timer goes off, the ImportPermit (and hence
 * the ExportPermit) is discarded.
 */
typedef struct {
    ExportPermit  permit;   	/* Actual permit from MCA */
    Rpc_Event	  expire;   	/* Expiration timer. If this goes off,
				 * the permit gets nuked */
} ImportPermit;

/*
 * A Process structure is assigned to each imported process so it can be
 * killed, its status returned, etc. The ExportPermit for the job is also
 * saved to allow the log server to match a job finish with a start.
 */
typedef struct {
    int	    	  	pid;	    	/* ID of child */
    int	    	  	pgrp;	    	/* Process group of job */
    char 	    	command[MAX_CMD_LEN];    	
					/* Command executed */
    long		start;		/* Start time of child */
    ImportPermit  	*permit;  	/* Import permit for job */
    struct sockaddr_in 	retAddr;  	/* Address of socket on which to
					 * return the exit status */
} Process;

static Lst  	  	permits;    /* List of permits awaiting processes */
static Lst  	  	imports;    /* List of active processes */

/*
 * This is the time for which an ImportPermit may remain active without a
 * process.
 */
static struct timeval 	expirationDate = {
    30, 0,
};

/*
 * Private call number for import-internal Rpc
 */
#define CUSTOMS_PGRP	((int)CUSTOMS_PING + 1000)
/*
 * This structure is passed from the child daemon to the main daemon to
 * notify of a new process group created for the imported job.  The extra
 * overhead will allow the main daemon to send SIGSTOP and SIGKILL to the
 * import while leaving the the child daemon alone.
 */
typedef struct {
    int			pid;		/* Child pid */
    int			pgrp;		/* Job process group */
} Pgrp_Data;

static Boolean ImportCheckAll();


/*-
 *-----------------------------------------------------------------------
 * ImportCmpPid --
 *	Callback function for ImportCheckAll to find a Process record
 *	with the given pid.
 *
 * Results:
 *	0 if it matches, non-0 if it doesn't.
 *
 * Side Effects:
 *	None.
 *
 *-----------------------------------------------------------------------
 */
static int
ImportCmpPid(procPtr, pid)
    Process 	  	*procPtr;   	/* Process to examine */
    int	    	  	pid;	    	/* PID desired. */
{
    return (procPtr->pid - pid);
}
	
/*-
 *-----------------------------------------------------------------------
 * Import_Kill --
 *	Kill imported jobs.
 *
 * Results:
 *	0 if all went well, a positive errno value if kill failed,
 *	-1 if no jobs were found.
 *
 * Side Effects:
 *	The signal is sent to child process associated with job id.
 *	Job id = 0 means kill all imported jobs.
 *
 *-----------------------------------------------------------------------
 */
int
Import_Kill(jobid, signo, from)
    int			    jobid;	/* Job to kill */
    int			    signo;	/* Signal to send */
    struct sockaddr_in	    *from;	/* Caller id */
{
    register Process   	    *procPtr;
    LstNode 	  	    ln;
    Boolean		    found = False;
    int		    	    error = 0;

#ifdef SYSV
    /*
     * Some system don't seem to like terminal-related stop signals sent
     * to jobs -- aborts them with SIGHUP.  So turn them into SIGSTOP's.
     * This has some drawbacks, like sub-makes cannot catch them and pass
     * them on, but it is better than nothing.
     */
    if ((signo == SIGTSTP) || (signo == SIGTTIN) || (signo == SIGTTOU)) {
	signo = SIGSTOP;
    }
#endif
    /*
     * Find the Process structure corresponding to the id number
     * given, then kill the process.
     */
    if (Lst_Open(imports) == SUCCESS) {
	while ((ln = Lst_Next(imports)) != NILLNODE) {
	    procPtr = (Process *)Lst_Datum(ln);
	    if (jobid == 0 ||
		procPtr->permit->permit.id == jobid)
	    {
		found = True;

		/*
		 * Killing is a job is allowed if
		 * - the call is internal (from = 0), or
		 * - it comes from a priviledged port, or
		 * - it comes from the return socket for that job.
		 */
		if (from &&
#ifdef USE_RESERVED_PORTS
		    (ntohs(from->sin_port) >= IPPORT_RESERVED) &&
#endif
		    (from->sin_addr.s_addr != procPtr->retAddr.sin_addr.s_addr
		     || from->sin_port != procPtr->retAddr.sin_port))
		{
		    xlog (XLOG_WARNING,
			    "Import_Kill: killpg(%d, %d) from %d@%s denied",
			    procPtr->pgrp, signo,
			    ntohs((from)->sin_port),
			    InetNtoA((from)->sin_addr));

		    Log_Send(LOG_ACCESS, 1, xdr_sockaddr_in, from);
		    error = EPERM;
		    break;
		}

		if (verbose) {
		    xlog (XLOG_DEBUG, "Import_Kill: killpg(%d, %d) from %d@%s",
			    procPtr->pgrp, signo,
			    from ? ntohs((from)->sin_port) : 0,
			    from ? InetNtoA((from)->sin_addr) : "(internal)");
		}

		if (KILLPG(procPtr->pgrp, signo) < 0)
		{
		    error = errno;
		    xlog (XLOG_WARNING, "Couldn't send signal: %s",
			    strerror(errno));
		}

		Log_Send(LOG_KILL, 2,
			 xdr_exportpermit, &procPtr->permit->permit,
			 xdr_long, &signo);

		if (jobid != 0) {
		    break;	/* jobs ids are unique */
		}
	    }
	}
	Lst_Close(imports);
    }

    if (!found)
	return (-1);
    else
	return (error);
}
	
/*-
 *-----------------------------------------------------------------------
 * ImportHandleKill --
 *	Handle the killing of an imported process.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The signal number in the message given to the child.
 *	Job id = 0 means kill all imported jobs.
 *
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
ImportHandleKill (from, msg, len, data)
    struct sockaddr_in	    *from;
    Rpc_Message	  	    msg;
    int	    	  	    len;
    Rpc_Opaque 	  	    data;
{
    Kill_Data	  	    *packet = (Kill_Data *)data;
    
    switch (Import_Kill(packet->id, packet->signo, from)) {
    case 0:
	Rpc_Return(msg, 0, (Rpc_Opaque)0);
	break;
    case -1:
	Rpc_Error(msg, RPC_BADARGS);	/* job not found */
	break;
    /*
     * We map errno values one-to-one to rpc error codes to be
     * able to give more informative error message at the other end.
     * This matters for cctrl -kill and rmt.c.
     */
    case ESRCH:	/* process not there */
	/*
	 * Most likely just a child that exited but hasn't been noticed
	 * by ImportCheckAll yet.  Ignore.
	 */
	Rpc_Return(msg, 0, (Rpc_Opaque)0);
	break;
    case EPERM:	/* permission denied */
	Rpc_Error(msg, RPC_ACCESS);
	break;
    case EINVAL:	/* bad signal number */
    default:	
	Rpc_Error(msg, RPC_SYSTEMERR);
	break;
    }
}

/*-
 *-----------------------------------------------------------------------
 * ImportPgrp --
 *	Receive info from child about a new process group allocated for
 *	imported job.
 *
 * Results:
 *	None..
 *
 * Side Effects:
 *	The pgrp info in the process table is updated.
 *
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
ImportPgrp (from, msg, len, data)
    struct sockaddr_in	    *from;
    Rpc_Message	  	    msg;
    int	    	  	    len;
    Rpc_Opaque 	  	    data;
{
    Pgrp_Data	  	    *packet = (Pgrp_Data *)data;
    LstNode 	  	    ln;
    Process 	  	    *procPtr;

    ln = Lst_Find (imports, (ClientData)packet->pid, ImportCmpPid);
    if (ln == NILLNODE) {
	xlog (XLOG_WARNING,
		"Process group for unknown child received (pid %d)",
		packet->pid);
	Rpc_Error(msg, RPC_BADARGS);
	return;
    }

    Rpc_Return(msg, 0, (Rpc_Opaque)0);

    procPtr = (Process *)Lst_Datum(ln);
    procPtr->pgrp = packet->pgrp;

    if (verbose) {
	xlog (XLOG_DEBUG, "ImportPgrp: pgrp for job %d set to %d",
		procPtr->permit->permit.id, procPtr->pgrp);
    }
}

/*-
 *-----------------------------------------------------------------------
 * ImportCatchChild --
 *	Catches SIGCHLD and schedules a call to ImportCheckAll to get
 *	its exit status.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Rpc event is created.
 *
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
static SIGRET
ImportCatchChild(signo)
    int		signo;		/* Always SIGCHLD */
{
    struct timeval timeout;

    timeout.tv_sec = 0;
    timeout.tv_usec = 0;

    (void)Rpc_EventOnce(&timeout, ImportCheckAll, (Rpc_Opaque)0);
#ifndef SYSV
    /*
     * Don't bother with any more SIGCHLD's until the event goes off.
     * SYSV disarms signals by default, so this is redundant.
     */
    (void)signal (SIGCHLD, SIG_DFL);
#endif
}

/*-
 *-----------------------------------------------------------------------
 * ImportCheckAll --
 *	Check on all the jobs. This is kinda gross. It just does a wait3
 *	to see if anyone wants to say anything and finishes the job out
 *	if it does.
 *
 * Results:
 *	FALSE.
 *
 * Side Effects:
 *	Jobs will be removed from the imports list, if they actually
 *	finish. If there are no imported jobs left, the event that caused
 *	the invocation of this function is deleted.
 *
 *-----------------------------------------------------------------------
 */
static Boolean
ImportCheckAll()
{
    union wait	  status;   	/* Status of child */
    int	    	  pid;	    	/* ID of reporting child */
    LstNode 	  ln;
    Process 	  *procPtr;

    while ((pid=wait3(&status,WNOHANG,(struct rusage *)0)) > 0) {
	ln = Lst_Find (imports, (ClientData)pid, ImportCmpPid);
	if (ln == NILLNODE) {
	    xlog (XLOG_WARNING,
		    "Wait returned status from unknown child (pid %d)", pid);
	}
	else {
	    procPtr = (Process *)Lst_Datum(ln);

	    if (verbose) {
		int job = procPtr->permit->permit.id;

		if (WIFSIGNALED(status)) {
		    xlog (XLOG_DEBUG,
			    "ImportCheckAll: Job %d (pid %d): killed by %d",
			    job, pid, status.w_termsig);
		} else {
		    xlog (XLOG_DEBUG,
			    "ImportCheckAll: Job %d (pid %d): exited with %d",
		            job, pid, status.w_retcode);
		}
	    }

	    (void)Lst_Remove(imports, ln);
	    Avail_Send();
	    free((char *)procPtr->permit);
	    free((char *)procPtr);
	}
    }

    (void)signal (SIGCHLD, ImportCatchChild);

    return(FALSE);
}

/*-
 *-----------------------------------------------------------------------
 * ImportExpire --
 *	Delete an expired ImportPermit.
 *
 * Results:
 *	False.
 *
 * Side Effects:
 *	The permit is removed from the permits list and the event that
 *	caused this call back is deleted. An availability packet is
 *	sent to the mca, too, since this expiration could have made the
 *	host available again.
 *
 *-----------------------------------------------------------------------
 */
static Boolean
ImportExpire(ln)
    LstNode 	  	    ln;	    	/* Node of permit to nuke */
{
    ImportPermit  	    *permit;	/* The actual permit */

    permit = (ImportPermit *)Lst_Datum(ln);
    Rpc_EventDelete(permit->expire);
    Lst_Remove(permits, ln);
    free((char *)permit);
    Avail_Send();
}

/*-
 *-----------------------------------------------------------------------
 * ImportCheckUser --
 *	Check import permission for a user.
 *
 * Results:
 *	True iff access allowed.
 *
 * Side Effects:
 *	None.
 *
 *-----------------------------------------------------------------------
 */
static Boolean
ImportCheckUser(uid)
    int 	  	    uid;	 /* User to check */
{
    struct passwd *pwd;
    
    /*
     * XXX: All we have is the user id.  With more information in the
     * export permit we could check that user names are the same,
     * do password authentication, etc.
     */
    if (checkUser >= PWD_CHECK) {
        if (!(pwd = getpwuid(uid)) ||
	    strcmp(pwd->pw_passwd, "*") == 0)
	    return False;
    }

    if (checkUser >= SHELL_CHECK) {
	char *ashell;

	setusershell();
	while (ashell = getusershell()) {
	    if (strcmp(pwd->pw_shell, ashell) == 0)
		break;
	}
	endusershell();

	if (!ashell)
	    return False;
    }

    return True;
}

/*-
 *-----------------------------------------------------------------------
 * ImportAllocated --
 *	Notice that we've been allocated. This call is only allowed to
 *	come from the udp customs port on the master machine (or 127.1
 * 	[localhost] if we are the master).
 *	If user access control is enabled we also check that the user
 *	is allowed to use this machine.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	None.
 *
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
ImportAllocated(from, msg, len, permit)
    struct sockaddr_in	*from;
    Rpc_Message	  	msg;
    int	    	  	len;
    ExportPermit  	*permit;
{
    ImportPermit  	*newPermit;
    AllocReply	    	reply;

    if (len != sizeof(ExportPermit)) {
	Rpc_Error (msg, RPC_BADARGS);
    } else if ((from->sin_port == htons(udpPort)) &&
	       ((from->sin_addr.s_addr == masterAddr.sin_addr.s_addr) ||
		(amMaster && Local(from))))
    {
	int uid = (permit->id >> 16) & 0xffff;		/* user id recovered */

	if (!ImportCheckUser(uid)) {
	    if (verbose || checkUser >= LOG_CHECK) {
		xlog (XLOG_WARNING, "Allocation for user %d from %d@%s denied",
			uid, ntohs(from->sin_port), InetNtoA(from->sin_addr));
	    }
	    if (checkUser >= LOG_CHECK) {
		Log_Send(LOG_ACCESS, 1, xdr_sockaddr_in, from);
	    }
	    Rpc_Error(msg, RPC_ACCESS);
	    return;
	}

	if (verbose) {
	    xlog (XLOG_DEBUG,
		    "ImportAllocated: Incoming process from %s (id %u)",
		    InetNtoA(permit->addr), permit->id);
	}
	
	newPermit = (ImportPermit *)malloc (sizeof (ImportPermit));
	newPermit->permit = *permit;
	Lst_AtEnd (permits, (ClientData)newPermit);
	newPermit->expire =
	    Rpc_EventCreate(&expirationDate, ImportExpire,
			    (Rpc_Opaque)Lst_Last(permits));
	reply.avail = Avail_Local(AVAIL_EVERYTHING, &reply.rating);
	Rpc_Return(msg, sizeof(reply), (Rpc_Opaque)&reply);
    } else {
	xlog (XLOG_WARNING,
		"Attempted Allocation from non-master %d@%s",
		ntohs(from->sin_port), InetNtoA(from->sin_addr));
	Rpc_Error(msg, RPC_ACCESS);
	Log_Send(LOG_ACCESS, 1, xdr_sockaddr_in, from);
    }
}

/*-
 *-----------------------------------------------------------------------
 * ImportFindID --
 *	Look for a permit with the given id number. Callback procedure
 *	for ImportProcess.
 *
 * Results:
 *	0 if the current permit matches. non-zero otherwise.
 *
 * Side Effects:
 *	None
 *
 *-----------------------------------------------------------------------
 */
static int
ImportFindID (permit, id)
    ImportPermit  *permit;
    u_long  	  id;
{
    return (permit->permit.id - id);
}

/*-
 *-----------------------------------------------------------------------
 * ImportPrintPermit --
 *	Print out a permit...Used by ImportProcess in verbose mode.
 *
 * Results:
 *	Always returns 0.
 *
 * Side Effects:
 *	None.
 *
 *-----------------------------------------------------------------------
 */
static int
ImportPrintPermit (permitPtr)
    ImportPermit  *permitPtr;
{
    xlog (XLOG_DEBUG, "   #%u to %s,", permitPtr->permit.id,
	    InetNtoA(permitPtr->permit.addr));
    return (0);
}

/*-
 *-----------------------------------------------------------------------
 * ImportExtractVector --
 *	Extract an array of strings from a buffer and return a vector of
 *	char pointers. Used by ImportProcess to get the argv and envp for
 *	the new process.
 *
 * Results:
 *	A dynamically-allocated vector of char *'s.
 *
 * Side Effects:
 *	*strPtrPtr is set to point beyond the extracted strings.
 *
 *-----------------------------------------------------------------------
 */
static char **
ImportExtractVector(strPtrPtr)
    char    	  **strPtrPtr;
{
    register char *cp;
    register char **vec;
    register int  numStrings;
    char    	  **vecPtr;

    cp = *strPtrPtr;
    numStrings = *(int *)cp;
    vecPtr = (char **)malloc((unsigned)((numStrings + 1) * sizeof(char *)));
    cp += sizeof(int);
    for (vec = vecPtr; numStrings != 0; vec++, numStrings--) {
	*vec = cp;
	cp += strlen(cp) + 1;
    }
    *vec = (char *)0;
    cp = Customs_Align(cp, char *);
    *strPtrPtr = cp;
    return(vecPtr);
}

/*-
 *-----------------------------------------------------------------------
 * ImportProcess --
 *	Import a process from another machine. Requires an unique ID
 *	which is communicated by the MCA in the CustomsAlloc call.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
ImportProcess (from, msg, len, wbPtr)
    struct sockaddr_in	*from;
    Rpc_Message	  	msg;
    int	    	  	len;
    WayBill 	  	*wbPtr;
{
    LstNode 	  	ln; 	    	/* Node of published permit */
    ImportPermit  	*permit;    	/* The permit itself */
    Process 	  	*proc;	    	/* Structure to track job */
    int	    	  	sock;	    	/* I/O socket for job */
    char    	  	*cp;	    	/* General char pointer */
    char    	  	*cwd;	    	/* Directory for job */
    char    	  	*file;	    	/* File to exec */
    char    	  	**argv;	    	/* Arguments for it */
    char    	  	**envp;	    	/* Environment for job */
    extern char	  	**environ;  	/* Current environment */
    int	    	  	permituid;  	/* UID in permit's ID */

#define ERROR(str) Rpc_Return(msg, sizeof(str), (Rpc_Opaque)str); \
                   (void)close(sock);\
                   Rpc_Ignore(sock);\
		   return;

    sock = Rpc_MessageSocket(msg);
    
    ln = Lst_Find (permits, (ClientData)wbPtr->id, ImportFindID);
    if (ln == NILLNODE) {
	if (verbose) {
	    if (Lst_IsEmpty (permits)) {
		xlog (XLOG_DEBUG, "ImportProcess: No permits issued");
	    } else {
		xlog (XLOG_DEBUG, "ImportProcess: No permit for %u:",
			wbPtr->id);
		Lst_ForEach (permits, ImportPrintPermit, (ClientData)0);
	    }
	}
	ERROR("No permit issued to you");
    } else {
	permit = (ImportPermit *)Lst_Datum (ln);
	Rpc_EventDelete(permit->expire);
	(void) Lst_Remove (permits, ln);
	if (permit->permit.addr.s_addr != from->sin_addr.s_addr) {
	    ERROR("Invalid address");
	}
	if (verbose) {
	    xlog (XLOG_DEBUG, "ImportProcess: Received IMPORT from %s",
		    InetNtoA(permit->permit.addr));
	}
	
#ifndef INSECURE
	/*
	 * Make sure the person's not trying to execute as root...
	 */
	if (wbPtr->ruid == 0 || wbPtr->euid == 0) {
	    xlog (XLOG_WARNING, "Attempted execution as ROOT from %d@%s",
		    ntohs(from->sin_port), InetNtoA(from->sin_addr));
	    /*
	     * We don't care if this RPC times out...
	     */
	    Log_Send(LOG_ACCESS, 1, xdr_sockaddr_in, from);
	    ERROR("Root execution not allowed");
	}
#endif /* INSECURE */
	/*
	 * The effective uid of the caller is encoded in the high word of
	 * permit id. We make sure it matches the id in the WayBill
	 * to prevent one source of fraud
	 */
	permituid = (wbPtr->id >> 16) & 0xffff;
	
	if (wbPtr->euid != permituid) {
	    xlog (XLOG_WARNING,
		    "Mismatched uid's (permit = %d, waybill=%d) from %d@%s",
		    permituid, wbPtr->euid,
		    ntohs(from->sin_port), InetNtoA(from->sin_addr));
	    Log_Send(LOG_ACCESS, 1, xdr_sockaddr_in, from);
	    ERROR("Mismatched user IDs");
	}

	cp = (char *)&wbPtr[1];
	cwd = cp;
	cp += strlen(cwd) + 1;
	file = cp;
	cp += strlen(file) + 1;
	cp = Customs_Align(cp, char *);
	argv = ImportExtractVector(&cp);
	envp = ImportExtractVector(&cp);
	
	proc = (Process *) malloc (sizeof (Process));
	proc->permit = permit;
	proc->retAddr = *from;
	proc->retAddr.sin_port = wbPtr->port;

	/*
	 * Record start time and command file of this job for later reference.
	 */
	strncpy(proc->command, file, sizeof(proc->command));
	time(&proc->start);

#ifdef DOUBLECHECK_TIMEOUT
	/*
	 * Check for export deadline expiration.  If we weren't able to get
	 * to this point in the time alotted, we give up.
	 */
	if (wbPtr->deadline && proc->start > wbPtr->deadline) {
	    xlog (XLOG_WARNING,
		    "Import from %d@%s took %ld secs too long",
		    ntohs(from->sin_port), InetNtoA(from->sin_addr),
		    proc->start - wbPtr->deadline);
	    ERROR("Import took too long");
	}
#endif /* DOUBLECHECK_TIMEOUT */

	Rpc_Return(msg, sizeof("Ok"), (Rpc_Opaque)"Ok");
	
	fflush(stdout);

	proc->pid = fork();
	proc->pgrp = proc->pid;
	if (proc->pid == 0) {
	    /*
	     * Child process:
	     * Set std{in,out,err} to send things to and receive things from
	     * the remote machine. Files opened for other jobs will close when
	     * we exec... Once that is done, attempt to set up the running
	     * environment:
	     *	  1) set both gids
	     *	  2) install all the groups the caller is in
	     *	  3) set both uids
	     *	  4) chdir to the working directory
	     *	  5) set the umask correctly.
	     * Then fork and execute the given command using the passed
	     * environment instead of our own. If any of these steps fails,
	     * we print an error message for pen pal to read and return a
	     * non-zero exit status.
	     */
	    union wait	status;
	    int	  	cpid;
	    int	  	oldstdout;
	    Boolean	jobbyowner = False;
	    char	procTitle[12];

	    /*
	     * For easy identification in ps(1) output, set our argv to
	     * the job id.
	     */
	    sprintf(procTitle, "%-11lu", wbPtr->id);
	    Customs_SetProcTitle(procTitle);

	    /*
	     * Check whether we are running as non-root and the customs
	     * owner is just testing his/her own network.
	     */
	    if (getuid() != 0 &&
		wbPtr->ruid == getuid() &&
	        wbPtr->euid == geteuid()) {
		jobbyowner = True;
	        if (verbose) {
		    xlog (XLOG_DEBUG,
			    "ImportProcess: assuming job is from customs owner (%d/%d)",
			    wbPtr->ruid, wbPtr->euid);
		}
	    }
#ifndef NO_PRIORITY
	    /*
	     * Reset our priority to 0 since we're just another job now.
	     */
	    if (!jobbyowner) {
#ifdef SYSV
		nice(-40); nice(20); nice(0);
#else
		setpriority (PRIO_PROCESS, 0, 0);
#endif
	    }
#endif
	    /*
	     * Make sure all signals handled on the client side are not
	     * just swept under the rug by the server side.
	     */
	    signal (SIGHUP, SIG_DFL);
	    signal (SIGQUIT, SIG_DFL);
	    signal (SIGINT, SIG_DFL);
	    signal (SIGTERM, SIG_DFL);
	    signal (SIGTSTP, SIG_DFL);
	    signal (SIGCONT, SIG_DFL);
	    signal (SIGTTOU, SIG_DFL);
	    signal (SIGTTIN, SIG_DFL);
#ifdef SIGWINCH
	    signal (SIGWINCH, SIG_DFL);
#endif
#ifdef SIGWINDOW
	    signal (SIGWINDOW, SIG_DFL);
#endif
	    signal (SIGUSR1, SIG_DFL);
	    signal (SIGUSR2, SIG_DFL);

	    /*
	     * Ignore eviction notification by default.  Allows regular
	     * jobs to terminated withing grace period.
	     */
	    signal (EVICT_NOTIFY, SIG_IGN);
	    signal (EVICT_SIGNAL, SIG_DFL);

	    /*
	     * Make sure user process doesn't inherit files opened by OS
	     * module.
	     */
	    OS_Exit();

	    /*
	     * Redirect error messages to user.
	     */
	    xlog_set(NULL, stdout);

	    oldstdout = dup(1);
	    
	    if (sock != 1) {
		dup2 (sock, 1);
	    }

	    status.w_status = 0;
	    
	    environ = envp;
	    
	    /*
	     * Restore priority and resource limits of exported job.
	     * If we are running our own jobs, at least try but don't
	     * worry about permission denied errors.
	     */
#define max(a,b) ((a)>(b)?(a):(b))
	    /*
	     * Make imports nicer ...
	     */
	    if (niceLevel) {
		wbPtr->priority = max(wbPtr->priority, niceLevel);
	    }
#ifdef SYSV
	    /*
	     * We are at nice level 0 after the normalization above.
	     */
	    if (nice(wbPtr->priority) == -1)
#else
	    if (setpriority(PRIO_PROCESS, 0, wbPtr->priority) < 0)
#endif
	    {
		if (!jobbyowner) {
		    perror("Couldn't set priority");
		    status.w_retcode = 5;	/* unused code */
		}
	    }
#define min(a,b) ((a)<(b)?(a):(b))
	    /*
	     * Prevent runaway or abusive imports ...
	     */
	    if (cpuLimit) {
		wbPtr->rlimits[0].rlim_cur =
		    min(wbPtr->rlimits[0].rlim_cur, cpuLimit);
		wbPtr->rlimits[0].rlim_max =
		    min(wbPtr->rlimits[0].rlim_max, cpuLimit + GRACE_CPU);
	    }

#ifdef RLIMIT_FSIZE
#ifdef FSIZE_SCALE
	    /*
	     * On HP-UX RLIMIT_FSIZE is scaled by the same blocksize as
	     * arguments to ulimit(2).
	     */
	    if (wbPtr->rlimits[1].rlim_cur != RLIM_INFINITY)
		wbPtr->rlimits[1].rlim_cur /= FSIZE_SCALE;
	    if (wbPtr->rlimits[1].rlim_max != RLIM_INFINITY)
		wbPtr->rlimits[1].rlim_max /= FSIZE_SCALE;
#endif /* FSIZE_SCALE */
#endif /* RLIMIT_FSIZE */

	    /*
	     * The order in which rlimits are set here must match the
	     * one used for getrlimit() calls in customslib.c.
	     */
	    if (0
#ifdef RLIMIT_CPU
		|| setrlimit(RLIMIT_CPU,    &wbPtr->rlimits[0]) < 0
#endif
#ifdef RLIMIT_FSIZE
		|| setrlimit(RLIMIT_FSIZE,  &wbPtr->rlimits[1]) < 0
#else /* !RLIMIT_FSIZE */
	        || (wbPtr->rlimits[1].rlim_max != RLIM_INFINITY &&
		    ulimit(2, (wbPtr->rlimits[1].rlim_max - 1)/512 + 1) < 0)
#endif
#ifdef RLIMIT_DATA
		|| setrlimit(RLIMIT_DATA,   &wbPtr->rlimits[2]) < 0
#endif
#ifdef RLIMIT_STAT
		|| setrlimit(RLIMIT_STACK,  &wbPtr->rlimits[3]) < 0
#endif
#ifdef RLIMIT_CORE
		|| setrlimit(RLIMIT_CORE,   &wbPtr->rlimits[4]) < 0
#endif
#ifdef RLIMIT_RSS
		|| setrlimit(RLIMIT_RSS,    &wbPtr->rlimits[5]) < 0
#endif
#ifdef RLIMIT_NOFILE
		|| setrlimit(RLIMIT_NOFILE, &wbPtr->rlimits[6]) < 0
#endif
	       )
	    {
		if (!jobbyowner) {
		    perror("Couldn't set resource limits");
		    status.w_retcode = 7;	/* unused code */
		}
	    }
	
	    /* 
	     * Now change our identity, but only if running as root
	     * on behalf of another user.
	     */
	    if (!jobbyowner) {
		if ( setregid (wbPtr->rgid, wbPtr->egid) < 0) {
		    perror("Couldn't set real/effective group ids");
		    status.w_retcode = 1;
		}
		if (setgroups (min(wbPtr->ngroups, NGROUPS), wbPtr->groups) < 0){
		    perror("Couldn't set groups");
		    status.w_retcode = 2;
		}
		if (setreuid (wbPtr->ruid, wbPtr->euid) < 0) {
		    perror("Couldn't set real/effective user ids");
		    status.w_retcode = 3;
		}
	    }

	    if (chdir (cwd) < 0) {
		perror(cwd);
		status.w_retcode = 4;
	    }
	    umask (wbPtr->umask);
	    signal (SIGPIPE, SIG_DFL);

	    /*
	     * Don't want to do anything our parent is doing, so reset
	     * the RPC system.
	     */
	    Rpc_Reset();

	    SETPGRP();

	    if (status.w_status == 0) {
		Pgrp_Data	pgrp;

		pgrp.pid = getpid();

		/*
		 * If we're still ok, fork and exec the program to
		 * be run, then wait for it to finish. We do a bare
		 * wait since we will suspend when it suspends, etc.
		 * We be a dedicated process...
		 */
 		cpid = vfork();
		if (cpid == 0) {
		    Rpc_Stat    rstat;
		    int		udpSocket;

		    /*
		     * Tell our master about the new process group the
		     * job will be running under.  This will allow
		     * SIGSTOP and SIGKILL to be sent without affecting
		     * the parent.
		     */
		    pgrp.pgrp = getpid();

		    udpSocket = Rpc_UdpCreate(FALSE, 0);
		    rstat = Rpc_Call(udpSocket, &localAddr,
				     (Rpc_Proc)CUSTOMS_PGRP,
				     sizeof(pgrp), (Rpc_Opaque)&pgrp,
				     0, (Rpc_Opaque)0,
				     CUSTOMSINT_NRETRY, &retryTimeOut);
		    if (rstat != RPC_SUCCESS) {
			xlog_set(NULL, XLOG_PREVIOUS);
			xlog (XLOG_ERROR, "PGRP call failed: %s",
			        Rpc_ErrorMessage(rstat));
			xlog_set(NULL, XLOG_PREVIOUS);
		    }
		    else {
			SETPGRP();
		    }
		    close(udpSocket);

		    close(oldstdout);

		    if (sock > 2) {
			(void)close(sock);
		    }
		    /*
		     * Hook all stdio streams up to the remote socket.
		     * stdout has already been dealt with above.
		     * The only way to not lose stdin and stderr seems
		     * to be to do the dup-ing here rather than together
		     * with stdout above.
		     */
		    dup2(1, 0);
		    dup2(1, 2);

		    execvp (file, argv);

		    perror(file);
		    _exit(5);
		} else if (cpid < 0) {
		    perror("Couldn't fork");
		    status.w_retcode = 6;
		}
	    }

	    /*
	     * Block all signals we can. Anything we get, our
	     * child will get too, and we will exit when it does,
	     * so there's no point in our dying without sending
	     * a status back, is there?
	     */
	    sigblock(~0);

	    /*
	     * Substitute new socket for sending log messages and exit
	     * statuses.
	     */
	    udpSocket = Rpc_UdpCreate(FALSE, 0);

	    /*
	     * No need for us to keep the socket open. Also want to print
	     * our messages to the log file, so redup the old stdout back to
	     * stream 1.
	     */
	    if (sock != 1) {
		close(1);
	    }
	    dup2(oldstdout, 1);
	    close(oldstdout);
	    
	    /*
	     * Redirect errors back to system log.
	     */
	    xlog_set(NULL, XLOG_PREVIOUS);

	    while (1) {
		Rpc_Stat    rstat;
		Exit_Data   retVal;
		
		if (status.w_status == 0) {
		    int pid;

		    /*
		     * Haven't got an exit status yet, so wait for one.
		     * We block on the wait since we've got nothing better
		     * to do.
		     */
		    do {
			pid = wait3(&status, WUNTRACED, (struct rusage *)0);
		    } while ((pid >= 0) && (pid != cpid) ||
			     (pid < 0) && (errno == EINTR));
		    
		    if (pid < 0) {
			perror("Unexpected error from wait");
		        status.w_retcode = 8;
		    }
		}

		/*
		 * Force an EOF-equivalent on the socket on the remote side
		 */
		if (sock >= 0) {
		    if (send(sock, "", 1, MSG_OOB) < 0) {
			xlog (XLOG_WARNING, "send OOB error: %s",
				strerror(errno));
		    }
		    close(sock);
		    sock = -1;
		}
		
		/*
		 * Return exit status. We don't really care if the
		 * other side receives it. We're very optimistic.
		 * They'll find out either by the RPC or by the socket
		 * going away...
		 */
		if (verbose) {
		    xlog (XLOG_DEBUG,
			    "ImportProcess: Returning exit status to %d@%s",
			    ntohs(proc->retAddr.sin_port),
			    InetNtoA(proc->retAddr.sin_addr));
		}

		retVal.id = permit->permit.id;
		retVal.status = status.w_status;
		
		rstat = Rpc_Call(udpSocket, &proc->retAddr,
				 (Rpc_Proc)CUSTOMS_EXIT,
				 sizeof(retVal), (Rpc_Opaque)&retVal,
				 0, (Rpc_Opaque)0,
				 CUSTOMSINT_NRETRY, &retryTimeOut);
		if (rstat != RPC_SUCCESS) {
		    Log_Send(LOG_EXITFAIL, 1,
			     xdr_exportpermit, &permit->permit);
		    xlog (XLOG_DEBUG, "Couldn't return exit status: %s",
			   Rpc_ErrorMessage(rstat));
		}
		
		if (verbose) {
		    if (WIFSIGNALED(status)) {
			xlog (XLOG_DEBUG, "ImportProcess: Pid %d: signal %d",
				cpid, status.w_termsig);
		    } else if (WIFSTOPPED(status)) {
			xlog (XLOG_DEBUG, "ImportProcess: Pid %d: stopped(%d)",
				cpid, status.w_stopsig);
		    } else {
			xlog (XLOG_DEBUG, "ImportProcess: Pid %d: exit(%d)",
				cpid, status.w_retcode);
		    }
		}
		if (!WIFSTOPPED(status)) {
		    /*
		     * The process is actually done, so break out of this
		     * loop after telling the logger that the job is
		     * finished.
		     */
		    Log_Send(LOG_FINISH, 2,
			     xdr_exportpermit, &permit->permit,
			     xdr_int, &status);

		    /*
		     * We don't allow detached children of the imported job
		     * to continue running.  Kill them, but be nice doing so.
		     * Also, the event is logged since this could mean
		     * someone is trying to evade the eviction mechanism.
		     */
		    if (KILLPG(cpid, SIGHUP) < 0) {
			if (errno != ESRCH) {
			    xlog (XLOG_WARNING,
				    "Error killing orphans of %d: %s",
				    cpid, strerror(errno));
			}
		    } else {
			xlog (XLOG_WARNING,
				"Killing orphans of process %d, uid %d",
				cpid, getuid());
			sleep(1);
			(void)KILLPG(cpid, SIGKILL);
		    }

		    break;
		} else {
		    /*
		     * Tell logger the job is just stopped and loop.
		     */
		    Log_Send(LOG_STOPPED, 1,
			     xdr_exportpermit, &permit->permit);
		    status.w_status = 0;
		}
	    }
	    exit(status.w_retcode);
	} else if (proc->pid == -1) {
	    /*
	     * Couldn't fork:
	     * close everything we just opened and return an error.
	     */
	    free ((Address) proc);
	    free((char *)argv);
	    free((char *)envp);
	    ERROR("Could not fork");
	} else {
	    /*
	     * Parent process:
	     * Close the socket and start up the child reaper to catch dead
	     * children if it isn't going already (it won't be if there were
	     * no jobs running before this one).
	     */

	    Rpc_Ignore(sock);
	    (void)close(sock);
	    Lst_AtEnd (imports, (ClientData)proc);

	    Log_Send(LOG_START, 4,
		       xdr_exportpermit, &permit->permit,
		       xdr_short, &wbPtr->euid,
		       xdr_short, &wbPtr->ruid,
		       xdr_strvec, &argv);
	    free((char *)argv);
	    free((char *)envp);
	}
    }
}

/*-
 *-----------------------------------------------------------------------
 * Import_NJobs --
 *	Return the number of imported jobs. This includes not only
 *	currently-running jobs, but potential jobs as well. This is to
 *	keep the quota from being overflowed by requesting enough hosts
 *	to cause this machine to be overallocated before it can send
 *	of an availability packet...Better to overestimate the number
 *	of jobs and have this machine unavailable than to overload this
 *	machine...
 *
 * Results:
 *	The number of jobs.
 *
 * Side Effects:
 *	None.
 *
 *-----------------------------------------------------------------------
 */
Import_NJobs()
{
    return (Lst_Length (imports) + Lst_Length(permits));
}
/*-
 *-----------------------------------------------------------------------
 * ImportInfo --
 *	Provide information about all currently imported jobs.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The info is sent as a reply.
 *
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
ImportInfo (from, msg, len, data)
    struct sockaddr_in	*from;
    Rpc_Message	  	msg;
    int	    	  	len;
    Rpc_Opaque 	  	data;
{
    LstNode 	     	ln;
    register Process	*procPtr;
    Job_Info    	info[MAX_INFO_SIZE/sizeof(Job_Info)];
    register int	j;
    time_t		now;

    /*
     * The information about all the jobs is stored in the 'info' buffer
     * as an array of Job_Info structures.  The list is terminated by a
     * structure with pid = 0.
     */
    if (Lst_Open(imports) == FAILURE) {
	Rpc_Error(msg, RPC_SYSTEMERR);
	return;
    }

    if (Lst_Length(imports) > sizeof(info)/sizeof(Job_Info)) {
	Rpc_Error(msg, RPC_TOOBIG);
	return;
    }

    time(&now);

    for (j = 0, ln=Lst_Next(imports);
	 !Lst_IsAtEnd(imports);
	 j++, ln=Lst_Next(imports))
    {
	procPtr = (Process *)Lst_Datum(ln);
	info[j].pid = procPtr->pgrp;
	info[j].id = procPtr->permit->permit.id;
	info[j].time = now - procPtr->start;
	info[j].uid = (info[j].id >> 16) & 0xffff;
	info[j].from_port = procPtr->retAddr.sin_port;
	info[j].from_addr = procPtr->retAddr.sin_addr.s_addr;
	strncpy(info[j].command, procPtr->command, sizeof(info[j].command));
	info[j].command[sizeof(info[j].command)-1] = '\0';
    }

    if (j < sizeof(info)/sizeof(Job_Info))
	info[j++].pid = 0;

    Lst_Close(imports);
    Rpc_Return(msg, ((char *)&info[j])-((char *)info), (Rpc_Opaque)info);
}

/*-
 *-----------------------------------------------------------------------
 * Import_Init --
 *	Initialize this module.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The imports list is initialized.
 *
 *-----------------------------------------------------------------------
 */
void
Import_Init()
{
    imports = Lst_Init (FALSE);
    permits = Lst_Init (FALSE);
    (void)signal (SIGPIPE, SIG_IGN);
    (void)signal (SIGCHLD, ImportCatchChild);

    Rpc_ServerCreate(tcpSocket, (Rpc_Proc)CUSTOMS_IMPORT, ImportProcess,
		     Swap_WayBill, Rpc_SwapNull, (Rpc_Opaque)0);
    Rpc_ServerCreate(udpSocket, (Rpc_Proc)CUSTOMS_ALLOC, ImportAllocated,
		     Swap_ExportPermit, Rpc_SwapLong, (Rpc_Opaque)0);
    Rpc_ServerCreate(udpSocket, (Rpc_Proc)CUSTOMS_KILL, ImportHandleKill,
		     Swap_Kill, Rpc_SwapNull, (Rpc_Opaque)0);
    Rpc_ServerCreate(udpSocket, (Rpc_Proc)CUSTOMS_JOBS, ImportInfo,
		     Rpc_SwapNull, Swap_Jobs, (Rpc_Opaque)0);
    /*
     * Import-internal RPC for comunicating new process groups.
     * No need for byte-swapping since always local.
     */
    Rpc_ServerCreate(udpSocket, (Rpc_Proc)CUSTOMS_PGRP, ImportPgrp,
		     Rpc_SwapNull, Rpc_SwapNull, (Rpc_Opaque)0);
}
