#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <kvm.h>
#include <net/if.h>
#include <net/if_dl.h>
#include <net/if_types.h>
#include <net/bpf.h>
#include <net/if_llc.h>
#include <netinet/in.h>
#include <netinet/in_var.h>
#include <netinet/if_ether.h>
#include <netdb.h>
#include <ctype.h>

#include <netat/appletalk.h>
#include "../uab/proto_intf.h"

union ether_union {
  u_char ether_addr[6];
  struct {
    u_int       high;
    u_int       low:16;
  } split_addr;
};

struct addr_list {
  union ether_union     address;
  struct addr_list      *next;
  int n;
};

typedef struct ephandle {       /* ethernet protocol driver handle */
  int inuse;                    /* true if inuse */
  int fd;                       /* file descriptor of socket */
  struct ifreq ifr;
  int protocol;                 /* ethernet protocol */
  int socket;                   /* ddp socket */
  u_char my_ether_addr[6];
  struct addr_list addrs;
  int bufsize;
  char *buffer, *bp;
  int cc;
} EPHANDLE;

private inited = FALSE;

private EPHANDLE ephlist[MAXOPENPROT];

extern char interface[50];
static struct addr_list mcast_addrs;

char *ether_sprint();

static u_char e_broad[6] = {0x09, 0x00, 0x07, 0xff, 0xff, 0xff};
/*
 * setup for particular device devno
 * all pi_open's will go this device
*/
export
pi_setup()
{
  int i;

  if (!inited) {
    for (i = 0 ; i < MAXOPENPROT; i++)
      ephlist[i].inuse = FALSE;
    (void)init_fdlistening();
    inited = TRUE;              /* don't forget now */
    add_address(e_broad, &mcast_addrs);
  }
  return(TRUE);
}

static int
fd2edx(fd)
{
  int i;

  for (i = 0; i < MAXOPENPROT; i++) {
    if (ephlist[i].inuse && (ephlist[i].fd == fd)) {
      return(i+1);
    }
  }
  return(-1);
}

static int
bpf_open()
{
  int fd;
  int n = 0;
  char device[sizeof "/dev/bpf000"];

  /*
   * Go through all the minors and find one that isn't in use.
   */
  do {
    (void)sprintf(device, "/dev/bpf%d", n++);
    fd = open(device, O_RDWR);
  } while (fd < 0 && errno == EBUSY);

  if (dbug.db_lap) {
    fprintf(stderr, "LAP bpf: open fd = %d\n", fd);
  }

  return fd;
}

dump_pf(i, n)
struct bpf_insn *i;
int n;
{
  char buffer[200], *cp;

  while (n--) {
    switch (BPF_CLASS(i->code)) {
    case BPF_LD:
      sprintf(cp = buffer, "{BPF_LD");
      cp += strlen(cp);
      if (BPF_SIZE(i->code) == BPF_H) {
        sprintf(cp, "+BPF_H");
        cp += strlen(cp);
      } else if (BPF_SIZE(i->code) == BPF_B) {
        sprintf(cp, "+BPF_B");
        cp += strlen(cp);
      } else {
        sprintf(cp, "+BPF_W");
        cp += strlen(cp);
      }
      if (BPF_MODE(i->code) == BPF_ABS) {
        sprintf(cp, "+BPF_ABS, %d}\n", i->k);
        cp += strlen(cp);
      } else {
        sprintf(cp, "+BPF_K, %d}\n", i->k);
        cp += strlen(cp);
      }
      break;
    case BPF_JMP:
      sprintf(cp = buffer, "{BPF_JMP");
      cp += strlen(cp);
      if (BPF_OP(i->code) == BPF_JA) {
        sprintf(cp, "+BPF_JA");
        cp += strlen(cp);
      } else if (BPF_OP(i->code) == BPF_JGT) {
        sprintf(cp, "+BPF_JGT");
        cp += strlen(cp);
      } else if (BPF_OP(i->code) == BPF_JGE) {
        sprintf(cp, "+BPF_JGE");
        cp += strlen(cp);
      } else if (BPF_OP(i->code) == BPF_JEQ) {
        sprintf(cp, "+BPF_JEQ");
        cp += strlen(cp);
      } else if (BPF_OP(i->code) == BPF_JSET) {
        sprintf(cp, "+BPF_JSET");
        cp += strlen(cp);
      }
      if (BPF_SRC(i->code) == BPF_X) {
        sprintf(cp, "+BPF_X");
        cp += strlen(cp);
      } else {
        sprintf(cp, "+BPF_K");
        cp += strlen(cp);
      }
      sprintf(cp, ", %d, %d, %d (%x)}\n", i->jt, i->jf, i->k, i->k);
      break;
    case BPF_RET:
      sprintf(cp = buffer, "{BPF_RET");
      cp += strlen(cp);
      if (BPF_RVAL(i->code) == BPF_A) {
        sprintf(cp, "+BPF_A, 0}\n");
        cp += strlen(cp);
      } else {
        sprintf(cp, "+BPF_K, %d}\n", i->k);
        cp += strlen(cp);
      }
      break;
    default:
      sprintf(buffer, "{%x, %d, %d, %d (%x)}\n", i->code, i->jt, i->jf, i->k,
              i->k);
    }
    fprintf(stderr, buffer);
    i++;
  }
}

/*
 * establish protocol filter
 *
*/

private struct bpf_insn address_filter[] = {
  BPF_STMT(BPF_LD+BPF_W+BPF_ABS, 0),
  BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, /* high order */ 0, 0, 2),
  BPF_STMT(BPF_LD+BPF_H+BPF_ABS, 4),
  BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, /* low order */ 0, /* ours */ 0, 0)
};

#define N_ADDR_INSN     4
#define I_HIGH_INSN     1
#define I_LOW_INSN      3

private struct bpf_insn header_filter[] = {
  BPF_STMT(BPF_RET+BPF_K, 0),
  BPF_STMT(BPF_LD+BPF_H+BPF_ABS, 12),
  BPF_JUMP(BPF_JMP+BPF_JGT+BPF_K, ETHERMTU, 0, 1),
  BPF_STMT(BPF_RET+BPF_K, 0),
  BPF_STMT(BPF_LD+BPF_B+BPF_ABS, 16),
  BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, LLC_UI, 1, 0),
  BPF_STMT(BPF_RET+BPF_K, 0),
  BPF_STMT(BPF_LD+BPF_H+BPF_ABS, 14),
  BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_ABS, 0xaaaa, 1, 0),
  BPF_STMT(BPF_RET+BPF_K, 0),
  BPF_STMT(BPF_LD+BPF_H+BPF_ABS, 20),
  BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, /* proto */ 0, 1, 0),
  BPF_STMT(BPF_RET+BPF_K, 0)
};

#define N_HDR_INSN      13
#define I_PROT_INSN     11

private struct bpf_insn sock_filter[] = {
  BPF_STMT(BPF_LD+BPF_B+BPF_ABS, 32),
  BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, /* sock */ 0, 1, 0),
  BPF_STMT(BPF_RET+BPF_K, 0)
};

#define N_SOCK_INSN     3
#define I_SOCK_INSN     1

private struct bpf_insn ret_filter[] = {
  BPF_STMT(BPF_LD+BPF_W+BPF_LEN, 0),
  BPF_STMT(BPF_RET+BPF_A, 0)
};

#define N_RET_INSN      2

private int
setup_pf(s, prot, sock, addrs)
int s;
u_short prot;
int sock;
struct addr_list *addrs;
{
  struct bpf_insn filter[200];
  struct bpf_insn *insn = filter;
  int n, i;
  struct bpf_insn *prog = address_filter;
  struct bpf_insn *base;
  struct bpf_program bpf_prog;
  int naddrs = addrs->n;

  while (naddrs--) {
    i = N_ADDR_INSN;
    prog = address_filter;
    base = insn;

    while (i--) {
      *insn++ = *prog++;
    }

    base[I_HIGH_INSN].k = ntohl(addrs->address.split_addr.high);
    base[I_LOW_INSN].k = ntohs(addrs->address.split_addr.low);
    base[I_LOW_INSN].jt = (naddrs * N_ADDR_INSN) + 1;

    addrs = addrs->next;
  }

  prog = header_filter;
  i = N_HDR_INSN;

  base = insn;
  while (i--) {
    *insn++ = *prog++;
  }
  base[I_PROT_INSN].k = prot;

  if (sock >= 0) {
    prog = sock_filter;
    i = N_SOCK_INSN;

    base = insn;
    while (i--) {
      *insn++ = *prog++;
    }
    base[I_SOCK_INSN].k = sock;
  }

  prog = ret_filter;
  i = N_RET_INSN;

  while (i--) {
    *insn++ = *prog++;
  }

  bpf_prog.bf_len = insn - filter;
  bpf_prog.bf_insns = filter;

  if (dbug.db_lap) {
    dump_pf(filter, insn-filter);
  }

  if (ioctl(s, BIOCSETF, (char *)&bpf_prog) < 0) {
    perror("BIOCSETF");
    return(-1);
  }

  return(0);
}

private int
init_bpf(eph)
EPHANDLE *eph;
{
  u_short protocol = eph->protocol;
  int socket = eph->socket;
  struct ifreq *ifr = &eph->ifr;
  int s;
  int one = 1;
  struct bpf_version bpf_vers;

  if ((s = bpf_open()) < 0) {
    perror("bpf");
    return(-1);
  }

  if (ioctl(s, BIOCVERSION, (char *)&bpf_vers) < 0) {
    perror("BIOCVERSION");
    close(s);
    return(-1);
  }

  if ((bpf_vers.bv_major != BPF_MAJOR_VERSION) ||
      (bpf_vers.bv_minor < BPF_MINOR_VERSION)) {
    fprintf(stderr, "bpf version mismatch:  need %d.%d, got %d.%d.\n",
            BPF_MAJOR_VERSION, BPF_MINOR_VERSION,
            bpf_vers.bv_major, bpf_vers.bv_minor);
    close(s);
    return(-1);
  }

  if (ioctl(s, BIOCSETIF, (char *)ifr) < 0){
    perror("BIOCSETIF");
    close(s);
    return(-1);
  }

  if (ioctl(s, BIOCPROMISC, (char *)&one) < 0) {
    perror("BIOCPROMISC");
    close(s);
    return(-1);
  }

  if (ioctl(s, BIOCGBLEN, (char *)&eph->bufsize) < 0) {
    perror("BIOCGBLEN");
    close(s);
    return(-1);
  }

  eph->buffer = (char *)malloc(eph->bufsize);
  eph->cc = 0;

  if (dbug.db_lap) {
    fprintf(stderr, "LAP bpf: bufsize = %d\n", eph->bufsize);
  }

  if (ioctl(s, BIOCIMMEDIATE, (char *)&one) < 0) {
    perror("BIOCIMMEDIATE");
    close(s);
    return(-1);
  }

  return(s);
}

struct addr_list *
add_address(ea, alist)
u_char *ea;
struct addr_list *alist;
{
  struct addr_list *addr =
    (struct addr_list *)malloc(sizeof(struct addr_list));

  bcopy((char *)alist, (char *)addr, sizeof(struct addr_list));

  bcopy((char *)ea, (char *)&alist->address, 6);
  alist->next = addr;
  alist->n = addr->n + 1;
}

/*
 * Open up a protocol handle:
 *   user level data:
 *      file descriptor
 *      protocol
 * 
 *   returns -1 and ephandle == NULL if memory allocation problems
 *   returns -1 for other errors
 *   return edx > 0 for okay
*/
export int
pi_open(protocol, socket, dev, devno)
int protocol;
int socket;
char *dev;
int devno;
{
  int s, i;
  struct ephandle *eph;

  for (i = 0; i < MAXOPENPROT; i++) {
    if (!ephlist[i].inuse)
      break;
  }
  if (i == MAXOPENPROT)
    return(0);                  /* nothing */
  eph = &ephlist[i];            /* find handle */

  strncpy(eph->ifr.ifr_name, interface, sizeof eph->ifr.ifr_name);
  eph->ifr.ifr_name[sizeof eph->ifr.ifr_name - 1] = ' ';

  eph->protocol = protocol;
  eph->socket = socket;

  if ((s = init_bpf(eph)) < 0) {
    return(-1);
  }

  eph->inuse = TRUE;
  eph->fd = s;

  pi_get_ethernet_address(i+1, eph->my_ether_addr);
  bcopy(eph->my_ether_addr, eph->addrs.address.ether_addr, 6);

  eph->addrs.n = mcast_addrs.n + 1;
  eph->addrs.next = &mcast_addrs;

  setup_pf(eph->fd, eph->protocol, eph->socket, &eph->addrs);

  return(i+1);                  /* skip zero */
}

/*
 * Convert Ethernet address to printable (loggable) representation.
 */
static char digits[] = "0123456789abcdef";
char *
ether_sprintf(ap)
register u_char *ap;
{
  register i;
  static char etherbuf[18];
  register char *cp = etherbuf;

  for (i = 0; i < 6; i++) {
    *cp++ = digits[*ap >> 4];
    *cp++ = digits[*ap++ & 0xf];
    *cp++ = ':';
  }
  *--cp = 0;
  return (etherbuf);
}

/* returns TRUE if machine will see own broadcasts */
export int
pi_delivers_self_broadcasts()
{
  return(TRUE);
}

export int
pi_addmulti(multi, ifr)
u_char multi[6];
struct ifreq *ifr;
{
  int i;

  add_address(multi, &mcast_addrs);
  for (i = 0; i < MAXOPENPROT; i++) {
    if (ephlist[i].inuse) {
      ephlist[i].addrs.next = &mcast_addrs;
      ephlist[i].addrs.n = mcast_addrs.n + 1;
      setup_pf(ephlist[i].fd, ephlist[i].protocol, ephlist[i].socket,
               &ephlist[i].addrs);
    }
  }

  return(0);
}

export int
pi_close(edx)
int edx;
{
  if (edx < 1 || edx > MAXOPENPROT || !ephlist[edx-1].inuse)
    return(-1);
  fdunlisten(ephlist[edx-1].fd); /* toss listener */
  close(ephlist[edx-1].fd);
  ephlist[edx-1].inuse = 0;

  return(0);
}

#ifdef __bsdi__
/* kvm stuff taken from netstat */
/*    
 * Read kernel memory, return 0 on success.
 */	
int
kread(addr, buf, size, kvmd)
	off_t addr;
	char *buf;
	int size;
	kvm_t *kvmd;
{
	if (kvm_read(kvmd, addr, buf, size) != size) {
		/* XXX this duplicates kvm_read's error printout */
		(void)fprintf(stderr, "netstat: kvm_read %s\n",
		    kvm_geterr(kvmd));
		return (-1); 
	}
	return (0);
}

struct nlist nl[] = {
        { "_ifnet" },   
	{ 0 }
};
#endif /* __bsdi__ */

export int
pi_get_ethernet_address(edx,ea)
int edx;
u_char *ea;
{
  struct sockaddr *sa;
  struct ephandle *eph;

#ifdef __bsdi__
	kvm_t *kvmd;
	off_t ifnetaddr;
        struct ifnet ifnet;
        union { 
                struct ifaddr ifa;
                struct in_ifaddr in;
        } ifaddr;       
        off_t ifaddraddr, ifaddrfound, ifnetfound;
        char name[32];          
        struct arpcom ac;
#endif /* __bsdi__ */

  if (edx < 1 || edx > MAXOPENPROT || !ephlist[edx-1].inuse)
    return(-1);

  eph = &ephlist[edx-1];

#ifdef __bsdi__
	if ((kvmd = kvm_open(NULL, NULL, NULL, O_RDONLY, NULL)) == NULL) {
		fprintf(stderr, "netstat: kvm_open: can't get namelist\n");
		return -1;
	}
	if (kvm_nlist(kvmd, nl) < 0 || nl[0].n_type == 0) {
		fprintf(stderr, "netstat: no namelist\n");
		goto errexit;
	}

	ifnetaddr = nl[0].n_value;
	do {
		if (kread(ifnetaddr, (char *)&ifnetaddr, sizeof ifnetaddr,
			   kvmd))
			goto errexit;
		if (kread(ifnetaddr, (char *)&ifnet, sizeof ifnet, kvmd) ||
		    kread((off_t)ifnet.if_name, name, 16, kvmd))
			goto errexit;
		name[15] = '\0';
		sprintf(name + strlen(name), "%d", ifnet.if_unit);
		ifnetfound = ifnetaddr;
		ifnetaddr = (off_t) ifnet.if_next;
		if (strcmp(name, eph->ifr.ifr_name) != 0)
			continue;
		ifaddraddr = (off_t)ifnet.if_addrlist;
		do {
			struct sockaddr_dl *sdl;
			char *cp;
			int n;

			if (kread(ifaddraddr, (char *)&ifaddr, sizeof ifaddr,
				  kvmd)) {
				goto errexit;
			}	
#define CP(x) ((char *)(x))	    
			cp = (CP(ifaddr.ifa.ifa_addr) - CP(ifaddraddr)) +
				CP(&ifaddr); sa = (struct sockaddr *)cp;
			ifaddraddr = (off_t)ifaddr.ifa.ifa_next;
			if (sa->sa_family != AF_LINK) {
				continue;
			}
			sdl = (struct sockaddr_dl *)sa;
			cp = (char *)LLADDR(sdl);
			n = sdl->sdl_alen;
			if (ifnet.if_type != IFT_ETHER) {
				goto errexit;
			}
			/*
			 * This is gross, but will suffice 
			 * until drivers are fixed to set the
			 * link addr.
			 */
			if (n == 0) {
				kread(ifnetfound, (char *)&ac, sizeof(ac),
				      kvmd);
				cp = (char *)ac.ac_enaddr;
				n = 6;
			}
			if (n != 6) {
				goto errexit;
			}
			bcopy(cp, ea, n);
			goto gotit;
		} while (ifaddraddr != 0);
		goto errexit;
	} while (ifnetaddr != 0);

errexit:
	kvm_close(kvmd);
	return -1;
#else /* __bsdi__ */
  if (ioctl(eph->fd, SIOCGIFADDR, &eph->ifr) < 0) {
    perror("ioctl: SIOCGIFADDR");
    return(-1);
  }
  sa = (struct sockaddr *)eph->ifr.ifr_data;
  bcopy(sa->sa_data, ea, 6);
#endif

gotit:
  if (dbug.db_lap) {
    fprintf(stderr, "LAP bpf: pi_get_ethernet_address: %s\n",
            ether_sprintf(ea));
  }
#ifdef __bsdi__
  kvm_close(kvmd);
#endif
  return(0);
}

export
pi_listener(edx, listener, arg)
int edx;
int (*listener)();
caddr_t arg;
{
  if (edx < 1 || edx > MAXOPENPROT || !ephlist[edx-1].inuse)
    return(-1);

  fdlistener(ephlist[edx-1].fd, listener, arg, edx);
}

export
pi_listener_2(edx, listener, arg1, arg2)
int edx;
int (*listener)();
caddr_t arg1;
int arg2;
{
  if (edx < 1 || edx > MAXOPENPROT || !ephlist[edx-1].inuse)
    return(-1);

  fdlistener(ephlist[edx-1].fd, listener, arg1, arg2);
}

extern int got_more;

/*
 * cheat - iov[0] == struct etherheader
 *
*/
export int
pi_readv(edx, iov, iovlen)
int edx;
struct iovec *iov;
int iovlen;
{
  struct ephandle *eph;
  char *cp;
  struct bpf_hdr *bpf_hdr;
  int pktlen, len;

  if (edx < 0) {
    edx = fd2edx(-edx);
  }
  if (edx < 1 || edx > MAXOPENPROT)
    return(-1);
  eph = &ephlist[edx-1];

  if (!eph->inuse)
    return(-1);

  if (eph->cc <= 0) {
    if ((eph->cc = read(eph->fd, eph->buffer, eph->bufsize)) < 0) {
      perror("abread");
      return(eph->cc);
    }
    eph->bp = eph->buffer;
  }
  if (dbug.db_lap) {
    fprintf(stderr, "LAP bpf: pi_readv: got %d\n", eph->cc);
  }

  bpf_hdr = (struct bpf_hdr *)eph->bp;
  cp = eph->bp;

  cp += bpf_hdr->bh_hdrlen;

  pktlen = len = bpf_hdr->bh_caplen;
  while (iovlen--) {
    int n = (len > iov->iov_len) ? iov->iov_len : len;

    if (n > 0) {
      bcopy(cp, iov->iov_base, n);
      len -= n;
      cp += n;
    }

    iov->iov_len = n;
    iov++;
  }

  eph->cc -= BPF_WORDALIGN(bpf_hdr->bh_hdrlen + bpf_hdr->bh_caplen);
  eph->bp += BPF_WORDALIGN(bpf_hdr->bh_hdrlen + bpf_hdr->bh_caplen);

  if (eph->cc > 0) {
    if (dbug.db_lap) {
      fprintf(stderr, "LAP bpf: pi_readv: %d bytes left\n", eph->cc);
    }
    got_more = 1;
  }

  if (dbug.db_lap) {
    fprintf(stderr, "LAP bpf: pi_readv = %d\n", pktlen);
  }
  return(pktlen);
}

export int
pi_read(edx, buf, bufsiz)
int edx;
caddr_t buf;
int bufsiz;
{
  struct iovec iov[3];
  struct ether_header ea;
  int iovlen;
#if PHASE2
  char header[8];
#endif
  int cc;

  iov[0].iov_base = (caddr_t)&ea;
  iov[0].iov_len = 14;
#if PHASE2
  iov[1].iov_base = (caddr_t)header;
  iov[1].iov_len = sizeof(header);
  iovlen = 3;
#else /* PHASE2 */
  iovlen = 2;
#endif /* PHASE2 */

  iov[iovlen - 1].iov_base = (caddr_t)buf;
  iov[iovlen - 1].iov_len = bufsiz;

  cc = pi_readv(edx, iov, iovlen);

#if PHASE2
  return(cc - sizeof(header) - sizeof(ea));
#else /* PHASE2 */
  return(cc - sizeof(ea));
#endif /* PHASE2 */
}

pi_reada(fd, buf, bufsiz, eaddr)
int fd;
caddr_t buf;
int bufsiz;
char *eaddr;
{
  struct iovec iov[3];
  int iovlen;
  struct bpf_hdr bpf_hdr;
#if PHASE2
  char header[5];       /* must be 5! */
#endif
  int cc;

  iov[0].iov_base = (caddr_t)eaddr;
  iov[0].iov_len = 14;
#if PHASE2
  iov[1].iov_base = (caddr_t)header;
  iov[1].iov_len = sizeof(header);
  iovlen = 3;
#else /* PHASE2 */
  iovlen = 2;
#endif /* PHASE2 */

  iov[iovlen - 1].iov_base = (caddr_t)buf;
  iov[iovlen - 1].iov_len = bufsiz;

  if ((cc = pi_readv(-fd, iov, iovlen)) < 0) {
    perror("abread - reada");
    return(cc);
  }

#if PHASE2
  /* construct a fake LAP header */
  buf[0] = buf[11];
  buf[1] = buf[12];
  buf[2] = 0x02;

  return(cc - sizeof(header) - 14);
#else
  return(cc - 14);
#endif
}

export int
pi_write(edx, buf, buflen, eaddr)
int edx;
caddr_t buf;
int buflen;
char *eaddr;
{
  struct ephandle *eph;
  struct ether_header *eh;
  struct sockaddr sa;
  struct iovec iov;

  iov.iov_base = buf + 8;
  iov.iov_len = buflen - 8;

  return(pi_writev(edx, &iov, 1, eaddr));
}

static char ether_pad[ETHERMIN];

export int
pi_writev(edx, xiov, xiovlen, eaddr)
int edx;
struct iovec *xiov;
int xiovlen;
unsigned char eaddr[6];
{
  EPHANDLE *eph;
  int i;
  char *p;
  int len;
  struct msghdr msghdr;
  struct iovec iov[10];
  int iovlen;
  struct ether_header ether_header;
  struct llc llc;
  
  if (edx < 1 || edx > MAXOPENPROT || eaddr == NULL)
    return(-1);

  eph = ephlist + edx - 1;
  if (!eph->inuse)
    return(-1);

#if PHASE2
  i = 2;
#else /* PHASE2 */
  i = 1;
#endif /* PHASE2 */

  len = 0;
  while (xiovlen--) {
    iov[i].iov_base = xiov->iov_base;
    len += (iov[i].iov_len = xiov->iov_len);
    i++;
    xiov++;
  }

#if PHASE2
  len += 8; /* llc */
#endif
  if (len < ETHERMIN) {
    if (dbug.db_lap) {
      fprintf(stderr, "pi_writev: padding %d bytes\n", len);
    }
    iov[i].iov_len = ETHERMIN - len;
    iov[i].iov_base = ether_pad;
    i++;
  }

  iov[0].iov_base = (char *)&ether_header;
  iov[0].iov_len = sizeof(ether_header);

  bcopy(eaddr, &ether_header.ether_dhost, 6);
  bcopy(eph->my_ether_addr, &ether_header.ether_shost, 6);

#if PHASE2
  ether_header.ether_type = htons(len);

  llc.llc_dsap = LLC_SNAP_LSAP;
  llc.llc_ssap = LLC_SNAP_LSAP;
  llc.llc_control = LLC_UI;
  llc.llc_un.type_snap.org_code[0] = (eph->protocol == 0x809b) ? 0x08 : 0x00;
  llc.llc_un.type_snap.org_code[1] = 0;
  llc.llc_un.type_snap.org_code[2] = (eph->protocol == 0x809b) ? 0x07 : 0x00;
  llc.llc_un.type_snap.ether_type = htons(eph->protocol);

  iov[1].iov_base = (char *)&llc;
  iov[1].iov_len = 8;
#else
  ether_header.ether_type = htons(eph->protocol);
#endif

  return writev(eph->fd, iov, i);
}
