/*
   OSTIME.C
   Platform dependent timing routines.

   $Id$
 */
/*    Copyright (c) 1994.  The Regents of the University of California.
                    All rights reserved.  */

/* The platform dependence is of two types:

   1. Strong- the System V timing routine times is not present (or not
              functional) in all BSD systems.  This is true despite the
	      fact that times is a POSIX standard.  For example, under
	      SunOS 4.1.3, the times() function exists, but its return
	      value is not the elapsed wall time.
   2. Weak- the units of "clock ticks" differ among different operating
              systems, and unless you can find the correct header file,
	      there is no way to disentangle them.  Furthermore, the
	      POSIX standard warns that the ANSI C clock() function may
	      return different "ticks" units (CLOCKS_PER_SECOND) than the
	      times() function (CLK_TCK).

   
 */

/* The following two routines require no special knowledge on the part of
   the caller (e.g.- macros like CLK_TCK) in order to function usefully.  */

/* os_clock returns the wall time elapsed since the first call to os_clock
   or os_timer made by this program.  The return value is floating point
   in units of seconds.
   If the system clock rolls over, the wall time will lose the time between
   the rollover time and the last call to os_clock or os_timer before the
   rollover time.  This is a very rare occurrence.  In no event will the
   returned wall time be negative, or decrease between successive calls.  */
extern double os_clock(void);

/* os_timer provides a way to obtain CPU time as well as wall clock time-
   its return values are floating point seconds, with the wall time
   being measured from the first call to os_timer or os_clock.
   If the system clock rolls over, the wall time will lose the time between
   the rollover time and the last call to os_clock or os_timer before the
   rollover time.  This is a very rare occurrence.  In no event will the
   returned wall time be negative, or decrease between successive calls.  */
extern void os_timer(double *cpu, double *sys, double *wall);

/* os_datetime returns 26 characters representing the current time, e.g.-
   "Sun Jan  3 15:14:13 1988\n\0".  The returned pointer points to statically
   allocated space, and the string is only good until the next call to
   os_datetime.  */
extern char *os_datetime(void);

/* ------------------------------------------------------------------------ */

static double initWall= 1.0;
static double lastWall= 0.0;

#ifndef ANSI_C_TIMER
#ifndef BSD_TIMER
/* Assume POSIX 1003.1-1990 standard timing interface.
   However-- CLK_TCK is noted as "obsolescent" there...  */
#include <time.h>
#include <sys/times.h>
/* Try to handle modest deviations from POSIX standard (e.g.- Sun).  */
#ifndef CLK_TCK
#include <unistd.h>
#ifndef CLK_TCK
#define CLK_TCK sysconf(_SC_CLK_TCK)
#endif
#endif
static double ticksPerSecond= 0.0;
void os_timer(double *cpu, double *sys, double *wall)
{
  struct tms cpuTime;
  long wallTicks= times(&cpuTime);
  if (ticksPerSecond==0.0) ticksPerSecond= CLK_TCK;
  *cpu= cpuTime.tms_utime/(double)ticksPerSecond;
  *sys= cpuTime.tms_stime/(double)ticksPerSecond;
  *wall= wallTicks/(double)ticksPerSecond;
  if (initWall>lastWall) initWall= lastWall= *wall;
  else if (*wall<lastWall) initWall-= lastWall;
  lastWall= *wall;
  *wall-= initWall;
}

#else
/* This will work on our Suns -- other BSD machines may differ?
    sys/time.h includes time.h on Sun */
#ifdef TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
/* autoconf recommends HAVE_SYS_TIME_H test here,
   and just including time.h if sys/time.h is not present */
#include <sys/time.h>
#endif
#include <sys/resource.h>
void os_timer(double *cpu, double *sys, double *wall)
{
  struct rusage cpuTime;
  struct timeval wallTime;
  struct timezone zone;
  getrusage(RUSAGE_SELF, &cpuTime);
  *cpu= cpuTime.ru_utime.tv_sec + 1.0e-6*cpuTime.ru_utime.tv_usec;
  *sys= cpuTime.ru_stime.tv_sec + 1.0e-6*cpuTime.ru_stime.tv_usec;
  gettimeofday(&wallTime, &zone);
  *wall= wallTime.tv_sec + 1.0e-6*wallTime.tv_usec;
  if (initWall>lastWall) initWall= lastWall= *wall;
  else if (*wall<lastWall) initWall-= lastWall;
  lastWall= *wall;
  *wall-= initWall;
}

#endif
#else
/* Use ANSI C standard clock() function for both wall and CPU time --
   it really only measures the latter, but that might be OK on a single
   user OS.  */
#include <time.h>

void os_timer(double *cpu, double *sys, double *wall)
{
  *cpu= *wall= ((double)clock()) / ((double)CLOCKS_PER_SEC);
  *sys= 0.0;
  if (initWall>lastWall) initWall= lastWall= *wall;
  else if (*wall<lastWall) initWall-= lastWall;
  lastWall= *wall;
  *wall-= initWall;
}
#endif

/* ------------------------------------------------------------------------ */

double os_clock(void)
{
  double cpu, sys, wall;
  os_timer(&cpu, &sys, &wall);
  return wall;
}

/* ------------------------------------------------------------------------ */

/* ctime and time are from ANSI <time.h> -- should have been
   included above... */
static char *os_string[26];
char *os_datetime(void)
{
  time_t now= time((void *)0);
  char *string= ctime(&now);
  int i;
  for (i=0 ; i<25 && string[i] ; i++) os_string[i]= string[i];
  os_string[i]= '\0';
  return os_string;
}

/* ------------------------------------------------------------------------ */
