/*
   OSPOLL.C
   Platform dependent part of event handling.

   $Id$
 */
/*    Copyright (c) 1994.  The Regents of the University of California.
                    All rights reserved.  */

/* For UNIX platforms, the poll() and select() functions require
   the file descriptor (fd) for the I/O channel.

   Each type of event channel (e.g.- streams, X windows, etc.) must
   register the routine used to return the file descriptor given
   the channel context.  The registration routine is OSevRegister,
   which returns the type number for that type of channel.  This
   type number should be used by the high level routine which
   registers a handler of this type.

   At most 8 different types of event channels are allowed, and at
   most 16 simultaneously non-busy event channels are allowed.

   Two different event channels may not share the same file
   descriptor -- OSPoll can only detect the presence of input on
   the channel, not the intended recipient.

   The type of the keyboard handler is automatically registered
   when EVKeyHandler is called.  This type may be used by any event
   handler whose context is a FILE* (the keyboard context is stdin).
   Use the OSKeyType routine defined in osterm.c to retrieve the
   appropriate type number.

   If OSevRegister returns < 0, too many handler types have been
   registered, and this file must be recompiled to allow for more.
 */
extern int os_ev_register(int (*fd)(void *context));

/* Functions defined in this file -- used only in events.c.  */
extern int os_poll(EVChannel *list, long timeout, int clear);

/* Function defined in osterm.c for use here -- determines if stdin
   read would halt the program with SIGTTIN signal.  */
extern int os_background(void);

/* Function defined in ostime.c.  */
extern double os_clock(void);

#include "events.h"

/* Functions defined in events.c for private use here.  */
extern int ev_type(EVChannel *channel);
extern void *ev_context(EVChannel *channel);
extern EVChannel *ev_next_channel(EVChannel *channel);
extern int ev_ready(EVChannel *channel);
extern void ev_set_ready(EVChannel *channel, int ready);

/* ------------------------------------------------------------------------ */

#ifndef OS_MAX_EVTYPES
#define OS_MAX_EVTYPES 8
#endif
static int (*get_fd[OS_MAX_EVTYPES])(void *context);
static int fd_get(int type, void *context);

int os_ev_register(int (*fd)(void *context))
{
  int n;
  for (n=0 ; n<OS_MAX_EVTYPES ; n++) {
    if (!get_fd[n]) {
      get_fd[n]= fd;
      return n;
    }
  }
  return -1;
}

static int fd_get(int type, void *context)
{
  if (type<0 || type>=OS_MAX_EVTYPES) return -1;
  return get_fd[type](context);
}

/* ------------------------------------------------------------------------ */

/* The events.c handler needs to be able to block waiting for input on any
   of several I/O channels (e.g.- an X window socket or stdin).  It also
   needs to be able to check whether input is available.

   In BSD UNIX, this is the select function.
   POSIX allows select as an extension, but does not require that this
   functionality even exist.  Most SYSV operating systems seem to
   supply a working select function.
   As of March 1993, the select on SGI workstations frequently fails
   to detect input from an X socket.
   An alternative to select is the poll function, which is part of the
   streamio(4) package.

   A problem arises when a program is running in the background --
   select reports input on stdin, but any attempt to read it will cause
   SIGTTIN and stop the program.  The fix here is rather crude:
   If running in background, OSPoll removes the keyboard from the
   channel list, and tries again with a maximum 1 second wait.  If no
   non-keyboard input is available, it again checks to see if it is in
   the background; if so, it waits another second.  Because this timing
   is rather crude, it won't wait for less than 1/5 second.  Thus, if
   the program is in background and no input from other channels is
   available, the program will continue to bother UNIX once a second.
 */

/* errno tells if a signal caused premature return from poll or select */
#include <errno.h>

/* To make the implementations independent of memory management, limit
   the maximum allowed number of active channels to OS_CHANNELS.  */
#define OS_CHANNELS 16

/* comment #1:
   (1) The keyboard channel is active (not busy).
   (2) There are active input channels besides the keyboard,
       OR we are just checking to see whether any input is immediately
       available.
   (3) Input has arrived on the keyboard channel.
   (4) This process is in the background -- i.e.-, the keyboard
       message is intended for another process, and this process
       would stop with SIGTTIN if it actually attempted to read
       the keyboard message.

       The "expected" response of a background process which wants
       keyboard input is to stop with a SIGTTIN signal.  UNIX shells
       provide this behavior as a feature of their job control, and
       we will get into trouble if we mess with it.  If we are just
       peeking to see if any input has arrived (wait==0), however,
       then it is correct to report that none has, since the message
       which has arrived is intended for the foreground process.  The
       case of wait>0 is more problematic, since it is unclear why
       anyone would ask a job with no graphics windows to pause.

       On the other hand, if there are other input channels, then we
       must ignore the keyboard message, since it is intended for the
       foreground process, and we want to keep listening to our other
       channels.  Even an X window which is "output only" actually
       must listen for expose events and WM_DELETE messages in order
       to avoid disastrous misbehavior.
 */

/* comment #2:
   Ignoring the keyboard input is very difficult.
   The basic idea is to poll for input on the other channels
   (i.e.- excluding the keyboard).  The problem is that there will
   be no notification if this process comes back into the foreground
   while we are waiting for input on the other channels.  The only
   easy solution is to break the wait down into, say, one second
   waits for channels other than the keyboard, followed by a wait
   for the full remaining time on every channel including the
   keyboard.  A further complication, if we aren't waiting forever,
   is that we must call a separate timing routine to see how long
   we waited if background input appears on the keyboard.  */

#ifdef USE_POLL
/* implementation using poll() function */
/* ------------------------------------------------------------------------ */

#include <poll.h>

int os_poll(EVChannel *list, long timeout, int clear)
{
  EVChannel *chan, *keybd= ev_find_key();
  struct pollfd fds[OS_CHANNELS];  /* at most OS_CHANNELS file descriptors */
  EVChannel *chs[OS_CHANNELS];
  int ready, nfds, keyfd;
  double current_time;

  if (timeout<0) timeout= -1;

  keyfd= -1;
  nfds= 0;
  for (chan=list ; chan ; chan=ev_next_channel(chan)) {
    if (!ev_busy(chan)) {
      fd= fd_get(ev_type(chan), ev_context(chan));
      if (fd>=0) {
	if (nfds>=OS_CHANNELS) break;
	if (chan!=keybd) {
	  fds[nfds].fd= fd;
	  chs[nfds]= chan;
	  nfds++;
	} else {
	  keyfd= fd;  /* normally, stdin is fd 0 */
	}
      }
    }
  }
  if (keyfd>=0) {
    if (nfds>=OS_CHANNELS) nfds= OS_CHANNELS-1;
    fds[nfds].fd= fd;
    chs[nfds]= keybd;
    nfds++;
    if (timeout>0) current_time= os_clock();
  } else {
    if (!nfds) return 0;
    keybd= 0;
  }

  for (;;) {
    for (i=0 ; i<nfds ; i++) {
      fds[i].events= POLLIN;
      fds[i].revents= 0;
    }
    ready= poll(fds, nfds, (int)timeout);
    if (keybd /*(1)*/ && (nfds>1 || !timeout) /*(2)*/ &&
	(fds[nfds-1].revents&(POLLIN|POLLPRI|POLLERR)) /*(3)*/ &&
	OSBackground() /*(4)*/) {
      /* see comment #1 above */
      fds[nfds-1].revents= 0;
      ready--;
      if (!ready && timeout) {
	/* see comment #2 above */
	int partial= 1000;   /* 1 second peeks */
	if (timeout>0) {
	  double elapsed= current_time;
	  current_time= os_clock();
	  elapsed= 1000.*(current_time-elapsed);
	  if (elapsed>1.e9) elapsed= 1.e9;
	  if (timeout<elapsed) timeout-= (long)elapsed;
	  else timeout= 0;
	  if (timeout<partial) partial= timeout;
	  timeout-= partial;
	}
	for (i=0 ; i<nfds-1 ; i++) {
	  fds[i].events= POLLIN;
	  fds[i].revents= 0;
	}
	ready= poll(fds, nfds, partial);
	if (!ready) continue;
      }
    }
    break;
  }

  if (ready>0) {
    while (nfds--) {
      if (fds[nfds].revents&(POLLIN|POLLPRI|POLLERR))
	ev_set_ready(chs[nfds], 1);
    }
  } else if (ready<0) {
    /* An error has occurred.  This is serious unless the "error"
       is merely that a signal has arrived before any of the selected
       events.  */
    if (errno==EINTR) ready= 0;
  }
  return ready;
}

/* ------------------------------------------------------------------------ */

#else
/* implementation using select() function */
/* ------------------------------------------------------------------------ */

/* Most UNIX variants define struct timeval here.  <time.h> is the
   other possibility, but that loads in a bunch of unnecessary junk.  */
#ifndef NO_SYS_TIME_H
#include <sys/time.h>
#else
#ifndef NO_TIME_H
#include <time.h>
#else
struct timeval {
  long	tv_sec;		/* seconds */
  long	tv_usec;	/* and microseconds */
};
#endif
#endif

/* copy mask macros from Xlibos.h (MIT R4 distribution for Suns) */

/* May need to define "MIT_R4_SYSV" (was "att") appropriately... */
#ifdef MIT_R4_SYSV
/*
 * UNIX System V Release 3.2
 */
#include <sys/param.h>
#define MSKBITCNT NOFILES_MAX

#else
/*
 * 4.2BSD-based systems
 */
#include <sys/param.h> /* needed for XConnDis.c */
#ifdef NOFILE
#define MSKBITCNT NOFILE
#else
/* go ahead and try to get it with the other convention */
#define MSKBITCNT NOFILES_MAX
#endif
#endif

/* Does this require sizeof(unsigned long)==32?  Probably... */
#define MSKCNT ((MSKBITCNT+31) / 32)

#if (MSKCNT==1)
#define BITMASK(i) (1 << (i))
#define MASKIDX(i) 0
#endif
#if (MSKCNT>1)
/* (ditto above comment)... */
#define BITMASK(i) (1 << ((i) & 31))
#define MASKIDX(i) ((i) >> 5)
#endif

#define MASKWORD(buf, i) buf[MASKIDX(i)]
#define BITSET(buf, i) MASKWORD(buf, i) |= BITMASK(i)
#define BITCLEAR(buf, i) MASKWORD(buf, i) &= ~BITMASK(i)
#define GETBIT(buf, i) (MASKWORD(buf, i) & BITMASK(i))

int os_poll(EVChannel *list, long timeout, int clear)
{
  EVChannel *chan, *keybd= ev_find_key();
  unsigned long mask[MSKCNT], msk[MSKCNT];
  int nfds, fds[OS_CHANNELS];  /* at most OS_CHANNELS file descriptors */
  EVChannel *chs[OS_CHANNELS];
  int ready, i, fd, maxfd, keyfd;
  struct timeval *delay, delay_;
  double current_time;

  nfds= 0;
  maxfd= keyfd= -1;
  for (i=0 ; i<MSKCNT ; i++) mask[i]= 0;
  for (chan=list ; chan ; chan=ev_next_channel(chan)) {
    if (!ev_busy(chan) && chan!=keybd) {
      fd= fd_get(ev_type(chan), ev_context(chan));
      if (fd>=0) {
	BITSET(mask, fd);
	if (fd>maxfd) maxfd= fd;
	chs[nfds]= chan;
	fds[nfds++]= fd;
	if (chan==keybd) keyfd= fd;
      }
    }
  }
  if (!nfds) return 0;
  if (keyfd>=0 && timeout>0) current_time= os_clock();

  for (;;) {
    for (i=0 ; i<MSKCNT ; i++) msk[i]= mask[i];
    if (timeout>=0) {
      delay_.tv_sec= timeout/1000;
      delay_.tv_usec= 1000*(timeout%1000);
      delay= &delay_;
    } else {
      delay= 0;
    }
    ready= select(maxfd+1, msk, (void *)0, (void *)0, delay);
    if (keyfd>=0 /*(1)*/ && (nfds>1 || !timeout) /*(2)*/ &&
	GETBIT(mask, keyfd) /*(3)*/ && OSBackground() /*(4)*/) {
      /* see comment #1 above */
      BITCLEAR(mask, keyfd);
      ready--;
      if (!ready && timeout) {
	/* see comment #2 above */
	long partial= 1000;   /* 1 second peeks */
	if (timeout>0) {
	  double elapsed= current_time;
	  current_time= os_clock();
	  elapsed= 1000.*(current_time-elapsed);
	  if (elapsed>1.e9) elapsed= 1.e9;
	  if (timeout<elapsed) timeout-= (long)elapsed;
	  else timeout= 0;
	  timeout-= (long)elapsed;
	  if (timeout<partial) partial= timeout;
	  timeout-= partial;
	}
	for (i=0 ; i<MSKCNT ; i++) msk[i]= mask[i];
	BITCLEAR(msk, keyfd);
	delay_.tv_sec= partial/1000;
	delay_.tv_usec= 1000*(partial%1000);
	ready= select(maxfd+1, msk, (void *)0, (void *)0, &delay_);
	if (!ready) continue;
      }
    }
    break;
  }

  if (ready>0) {
    while (nfds--) {
      if (GETBIT(mask, fds[nfds])) ev_set_ready(chs[nfds], 1);
    }
  } else if (ready<0) {
    /* An error has occurred.  This is serious unless the "error"
       is merely that a signal has arrived before any of the selected
       events.  */
    if (errno==EINTR) ready= 0;
  }
  return ready;
}

/* ------------------------------------------------------------------------ */

#endif
