/*
 * java.lang.System.c
 *
 * Copyright (c) 1996 Systems Architecture Research Centre,
 *		   City University, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@sarc.city.ac.uk>, February 1996.
 */

#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include <sys/utsname.h>
#include <sys/time.h>
#include <sys/param.h>
#include <pwd.h>
#include <unistd.h>
#include <native.h>
#include "system.h"
#include "java.util/Properties.h"
#include "java.lang/System.h"
#include "../../kaffevm/itypes.h"
#include "../../kaffevm/classMethod.h"

/*
 * Copy one part of an array to another.
 */
void
java_lang_System_arraycopy(struct Hjava_lang_Object* src, long srcpos, struct Hjava_lang_Object* dst, long dstpos, long len)
{
	char* in;
	char* out;
	int elemsz;

	/* Must be some array type */
	assert(dst->mtable->class->sig[0] == '[');
	elemsz = TYPE_SIZE_C(dst->mtable->class->sig[1]);

	len *= elemsz;
	srcpos *= elemsz;
	dstpos *= elemsz;

	in = &src->data[srcpos];
	out = &dst->data[dstpos];
	for (; len > 0; len--) {
		*out++ = *in++;
	}
}

/*
 * Initialise system properties to their defaults.
 */
struct Hjava_util_Properties*
java_lang_System_initProperties(struct Hjava_util_Properties* p)
{
	int r;
	struct utsname system;
	struct passwd* pw;
	char* jhome;
	char* cpath;
	char* dir;

	/* Define PUT to be a call to the put method taking two strings
	 * as arguments.  call_java_method does all the majic looking up
	 * here.
	 */
#define	PUT(k,s) \
	do_execute_java_method(0, &p->obj, "put", \
	  "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;", \
	  0, false, \
	  makeJavaString(k, strlen(k)), \
	  makeJavaString(s, strlen(s)))

	/* Now use it to add the default properties:
	 * java.version		Java version number
	 * java.vendor          Java vendor specific string
	 * java.vendor.url      Java vendor URL
	 * java.home            Java installation directory
	 * java.class.version   Java class version number
	 * java.class.path      Java classpath
	 * os.name              Operating System Name
	 * os.arch              Operating System Architecture
	 * os.version           Operating System Version
	 * file.separator       File separator ("/" on Unix)
	 * path.separator       Path separator (":" on Unix)
	 * line.separator       Line separator ("\n" on Unix)
	 * user.name            User account name
	 * user.home            User home directory
	 * user.dir             User's current working directory
	 */

	PUT("java.version", kaffe_version);
	PUT("java.vendor", kaffe_vendor);
	PUT("java.vendor.url", kaffe_vendor_url);

	jhome = getenv(KAFFEHOME);
	if (jhome == 0) {
		jhome = ".";
	}
	PUT("java.home", jhome);

	PUT("java.class.version", kaffe_class_version);

	cpath = getenv(KAFFECLASSPATH);
	if (cpath == 0) {
		cpath = ".";
	}
	PUT("java.class.path", cpath);

	/* Setup system properties */
	r = uname(&system);
	assert(r >= 0);
	PUT("os.name", system.sysname);
	PUT("os.arch", system.machine);
	PUT("os.version", system.version);

	PUT("file.separator", file_seperator);
	PUT("path.separator", path_seperator);
	PUT("line.separator", line_seperator);

	/* Setup user properties */
	pw = getpwuid(getuid());
	assert(pw != 0);
	dir = getcwd(0, MAXPATHLEN);
	if (dir == 0) {
		dir = ".";
	}
	PUT("user.name", pw->pw_name);
	PUT("user.home", pw->pw_dir);
	PUT("user.dir", dir);
	free(dir);

	return (p);
}

/*
 * Return current time.
 */
long long
java_lang_System_currentTimeMillis(void)
{
	struct timeval tm;
	gettimeofday(&tm, 0);
	return (((long long)tm.tv_sec * 1000LL) + ((long long)tm.tv_usec / 1000LL));
}
