/*
 * i386-asm.h
 * Hand crafted inline voids to turn semi-understandable i386 instructions
 * into machine code.
 *
 * Copyright (c) 1996 Systems Architecture Research Centre,
 *		   City University, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@sarc.city.ac.uk>, February 1996.
 */

#include <assert.h>
#include "gtypes.h"

extern nativecode* pc;
#define	OUT	(*pc++)

#define	EXTERNAL_LINK(j)	addLink(j, (int)pc, 4+(int)pc, false)
#define	INTERNAL_LINK(j)	addLink(j, (int)pc, 4+(int)pc, true)

inline void asm__adcl_RxR(int r1, int r2)
{
	OUT = 0x11;
	OUT = 0xC0|(r1<<3)|r2;
}

inline void asm__adcl_VxR(int v1, int r1)
{
	OUT = 0x81;
	OUT = 0xD0|r1;
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__addl_RxR(int r1, int r2)
{
	OUT = 0x01;
	OUT = 0xC0|(r1<<3)|r2;
}

inline void asm__addl_VxVoRo(int v1, int v2, int r1)
{
	/* addl v1,v2(r1) */
	OUT = 0x81;
	OUT = 0x85;
	OUT = v2 & 0xff;
	OUT = (v2 >> 8) & 0xff;
	OUT = (v2 >> 16) & 0xff;
	OUT = (v2 >> 24) & 0xff;
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__addl_VxR(int v1, int r1)
{
	OUT = 0x81;
	OUT = 0xC0|r1;
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__andl_RxR(int r1, int r2)
{
	OUT = 0x21;
	OUT = 0xC0|(r1<<3)|r2;
}

inline void asm__andl_VxR(int v1, int r1)
{
	OUT = 0x81;
	OUT = 0xE0|r1;
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__brkpt(void)
{
	OUT = 0xCC;
}

inline void asm__call_J(int j1)
{
	OUT = 0xE8;
	INTERNAL_LINK(j1);
	OUT = j1 & 0xff;
	OUT = (j1 >> 8) & 0xff;
	OUT = (j1 >> 16) & 0xff;
	OUT = (j1 >> 24) & 0xff;
}

inline void asm__call_X(int j1)
{
	OUT = 0xE8;
	EXTERNAL_LINK(j1);
	OUT = j1 & 0xff;
	OUT = (j1 >> 8) & 0xff;
	OUT = (j1 >> 16) & 0xff;
	OUT = (j1 >> 24) & 0xff;
}

inline void asm__call_sR(int r1)
{
	OUT = 0xFF;
	OUT = 0xD0|r1;
}

inline void asm__cmpl_RxR(int r1, int r2)
{
	OUT = 0x39;
	OUT = 0xC0|(r1<<3)|r2;
}

inline void asm__cmpl_VxR(int v1, int r1)
{
	OUT = 0x81;
	OUT = 0xF8|r1;
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__cmpl_VxVoRo (int v1, int v2, int r1)
{
	/* cmpl v1,v2(r1) */
	OUT = 0x81;
	OUT = 0xB8|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v2 & 0xff;
	OUT = (v2 >> 8) & 0xff;
	OUT = (v2 >> 16) & 0xff;
	OUT = (v2 >> 24) & 0xff;
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__faddl_VoRo(int v1, int r1)
{
	/* faddl v1(r1) */
	OUT = 0xDC;
	OUT = 0x80|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__fadds_VoRo(int v1, int r1)
{
	/* fadds v1(r1) */
	OUT = 0xD8;
	OUT = 0x80|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__fdivl_VoRo(int v1, int r1)
{
	/* fdivl v1(r1) */
	OUT = 0xDC;
	OUT = 0xB0|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__fdivs_VoRo(int v1, int r1)
{
	/* fdivs v1(r1) */
	OUT = 0xD8;
	OUT = 0xB0|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__filds_VoRo(int v1, int r1)
{
	/* filds v1(r1) */
	OUT = 0xDB;
	OUT = 0x80|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__fildd_VoRo(int v1, int r1)
{
	/* fildd v1(r1) */
	OUT = 0xDF;
	OUT = 0x80|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__fistpl_VoRo(int v1, int r1)
{
	/* fistpl v1(r1) */
	OUT = 0xDF;
	OUT = 0xB8|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__fistps_VoRo(int v1, int r1)
{
	/* fistps v1(r1) */
	OUT = 0xDB;
	OUT = 0x98|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__fldl_VoRo(int v1, int r1)
{
	/* fldl v1(r1) */
	OUT = 0xDD;
	OUT = 0x80|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__flds_VoRo(int v1, int r1)
{
	/* flds v1(r1) */
	OUT = 0xD9;
	OUT = 0x80|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__fldz(void)
{
	OUT = 0xD9;
	OUT = 0xEE;
}

inline void asm__fmull_VoRo(int v1, int r1)
{
	/* fmull v1(r1) */
	OUT = 0xDC;
	OUT = 0x88|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__fmuls_VoRo(int v1, int r1)
{
	/* fmuls v1(r1) */
	OUT = 0xD8;
	OUT = 0x88|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__fstpl_VoRo(int v1, int r1)
{
	/* fstpl v1(r1) */
	OUT = 0xDD;
	OUT = 0x98|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__fstps_VoRo(int v1, int r1)
{
	/* fstps v1(r1) */
	OUT = 0xD9;
	OUT = 0x98|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__fsubl_VoRo(int v1, int r1)
{
	/* fsubl v1(r1) */
	OUT = 0xDC;
	OUT = 0xA0|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__fsubs_VoRo(int v1, int r1)
{
	/* fsubs v1(r1) */
	OUT = 0xD8;
	OUT = 0xA0|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__idivl_RxR(int r1, int r2)
{
	assert(r1 == REG_eax);
	OUT = 0xF7;
	OUT = 0xF8|r2;
}

inline void asm__imull_RxR(int r1, int r2)
{
	OUT = 0x0F;
	OUT = 0xAF;
	OUT = 0xC0|(r2<<3)|r1;
}

inline void asm__je_J(int j1)
{
	OUT = 0x0f;
	OUT = 0x84;
	INTERNAL_LINK(j1);
	OUT = j1 & 0xff;
	OUT = (j1 >> 8) & 0xff;
	OUT = (j1 >> 16) & 0xff;
	OUT = (j1 >> 24) & 0xff;
}

inline void asm__jne_J(int j1)
{
	OUT = 0x0f;
	OUT = 0x85;
	INTERNAL_LINK(j1);
	OUT = j1 & 0xff;
	OUT = (j1 >> 8) & 0xff;
	OUT = (j1 >> 16) & 0xff;
	OUT = (j1 >> 24) & 0xff;
}

inline void asm__jge_J(int j1)
{
	OUT = 0x0f;
	OUT = 0x8d;
	INTERNAL_LINK(j1);
	OUT = j1 & 0xff;
	OUT = (j1 >> 8) & 0xff;
	OUT = (j1 >> 16) & 0xff;
	OUT = (j1 >> 24) & 0xff;
}

inline void asm__jgt_J(int j1)
{
	OUT = 0x0f;
	OUT = 0x8f;
	INTERNAL_LINK(j1);
	OUT = j1 & 0xff;
	OUT = (j1 >> 8) & 0xff;
	OUT = (j1 >> 16) & 0xff;
	OUT = (j1 >> 24) & 0xff;
}

inline void asm__jle_J(int j1)
{
	OUT = 0x0f;
	OUT = 0x8e;
	INTERNAL_LINK(j1);
	OUT = j1 & 0xff;
	OUT = (j1 >> 8) & 0xff;
	OUT = (j1 >> 16) & 0xff;
	OUT = (j1 >> 24) & 0xff;
}

inline void asm__jlt_J(int j1)
{
	OUT = 0x0f;
	OUT = 0x8c;
	INTERNAL_LINK(j1);
	OUT = j1 & 0xff;
	OUT = (j1 >> 8) & 0xff;
	OUT = (j1 >> 16) & 0xff;
	OUT = (j1 >> 24) & 0xff;
}

inline void asm__jmpl_J(int j1)
{
	OUT = 0xE9;
	INTERNAL_LINK(j1);
	OUT = j1 & 0xff;
	OUT = (j1 >> 8) & 0xff;
	OUT = (j1 >> 16) & 0xff;
	OUT = (j1 >> 24) & 0xff;
}

inline void asm__jmpl_sR(int r1)
{
	OUT = 0xFF;
	OUT = 0xE0|r1;
}

inline void asm__leal_VoRxRxVoxR(int v1, int r1, int r2, int v2, int r3)
{
	/* leal v1(r1,r2,v2),r3 */
	assert(v2 == 1 || v2 == 2 || v2 == 4 || v2 == 8);
	assert(r2 != REG_esp);
	OUT = 0x8D;
	OUT = 0x84|(r3<<3);
	if (v2 == 1) {
		OUT = 0x00|(r2<<3)|r1;
	}
	else if (v2 == 2) {
		OUT = 0x40|(r2<<3)|r1;
	}
	else if (v2 == 4) {
		OUT = 0x80|(r2<<3)|r1;
	}
	else {
		OUT = 0xC0|(r2<<3)|r1;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__movb_RxVoRxRxVo(int r1, int v1, int r2, int r3, int v2)
{
	/* movb r1,v1(r2,r3,v2) */
	assert(v2 == 1 || v2 == 2 || v2 == 4 || v2 == 8);
	assert(r3 != REG_esp);
	OUT = 0x88;
	OUT = 0x84|(r1<<3);
	if (v2 == 1) {
		OUT = 0x00|(r3<<3)|r2;
	}
	else if (v2 == 2) {
		OUT = 0x40|(r3<<3)|r2;
	}
	else if (v2 == 4) {
		OUT = 0x80|(r3<<3)|r2;
	}
	else {
		OUT = 0xC0|(r3<<3)|r2;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__movw_RxVoRxRxVo(int r1, int v1, int r2, int r3, int v2)
{
	/* movw r1,v1(r2,r3,v2) */
	assert(v2 == 1 || v2 == 2 || v2 == 4 || v2 == 8);
	assert(r3 != REG_esp);
	OUT = 0x66;
	OUT = 0x89;
	OUT = 0x84|(r1<<3);
	if (v2 == 1) {
		OUT = 0x00|(r3<<3)|r2;
	}
	else if (v2 == 2) {
		OUT = 0x40|(r3<<3)|r2;
	}
	else if (v2 == 4) {
		OUT = 0x80|(r3<<3)|r2;
	}
	else {
		OUT = 0xC0|(r3<<3)|r2;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__movl_oRoxR(int r1, int r2)
{
	/* movl (r1),r2 */
	OUT = 0x8B;
	OUT = 0x00|(r2<<3)|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
}

inline void asm__movl_VoRoxR(int v1, int r1, int r2)
{
	/* movl v1(r1),r2 */
	OUT = 0x8B;
	OUT = 0x80|(r2<<3)|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__movl_VoRxRxVoxR(int v1, int r1, int r2, int v2, int r3)
{
	/* movl v1(r1,r2,v2),r3 */
	assert(v2 == 1 || v2 == 2 || v2 == 4 || v2 == 8);
	assert(r2 != REG_esp);
	OUT = 0x8B;
	OUT = 0x84|(r3<<3);
	if (v2 == 1) {
		OUT = 0x00|(r2<<3)|r1;
	}
	else if (v2 == 2) {
		OUT = 0x40|(r2<<3)|r1;
	}
	else if (v2 == 4) {
		OUT = 0x80|(r2<<3)|r1;
	}
	else {
		OUT = 0xC0|(r2<<3)|r1;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__movl_RxVoRo(int r1, int v1, int r2)
{
	/* movl r1,v1(r2) */
	OUT = 0x89;
	OUT = 0x80|(r1<<3)|r2;
	if (r2 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__movl_RxVoRxRxVo(int r1, int v1, int r2, int r3, int v2)
{
	/* movl r1,v1(r2,r3,v2) */
	assert(v2 == 1 || v2 == 2 || v2 == 4 || v2 == 8);
	assert(r3 != REG_esp);
	OUT = 0x89;
	OUT = 0x84|(r1<<3);
	if (v2 == 1) {
		OUT = 0x00|(r3<<3)|r2;
	}
	else if (v2 == 2) {
		OUT = 0x40|(r3<<3)|r2;
	}
	else if (v2 == 4) {
		OUT = 0x80|(r3<<3)|r2;
	}
	else {
		OUT = 0xC0|(r3<<3)|r2;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__movl_RxoVo(int r1, int v1)
{
	OUT = 0x89;
	OUT = 0x05|(r1<<3);
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__movl_oVoxR(int v1, int r1)
{
	OUT = 0x8B;
	OUT = 0x05|(r1<<3);
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__movl_RxR(int r1, int r2)
{
	OUT = 0x89;
	OUT = 0xC0|(r1<<3)|r2;
}

inline void asm__movl_VxR(int v1, int r1)
{
	OUT = 0xB8|r1;
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__movsbl_VoRxRxVoxR(int v1, int r1, int r2, int v2, int r3)
{
	/* movsbl v1(r1,r2,v2),r3 */
	assert(v2 == 1 || v2 == 2 || v2 == 4 || v2 == 8);
	assert(r3 != REG_esp);
	OUT = 0x0F;
	OUT = 0xBE;
	OUT = 0x84|(r1<<3);
	if (v2 == 1) {
		OUT = 0x00|(r3<<3)|r2;
	}
	else if (v2 == 2) {
		OUT = 0x40|(r3<<3)|r2;
	}
	else if (v2 == 4) {
		OUT = 0x80|(r3<<3)|r2;
	}
	else {
		OUT = 0xC0|(r3<<3)|r2;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__movsbl_RxR(int r1, int r2)
{
	/* movsbl r1,r2 */
	OUT = 0x0F;
	OUT = 0xBE;
	OUT = 0xC0|(r2<<3)|r1;
}

inline void asm__movswl_VoRxRxVoxR(int v1, int r1, int r2, int v2, int r3)
{
	/* movswl v1(r1,r2,v2),r3 */
	assert(v2 == 1 || v2 == 2 || v2 == 4 || v2 == 8);
	assert(r2 != REG_esp);
	OUT = 0x0F;
	OUT = 0xBF;
	OUT = 0x84|(r3<<3);
	if (v2 == 1) {
		OUT = 0x00|(r2<<3)|r1;
	}
	else if (v2 == 2) {
		OUT = 0x40|(r2<<3)|r1;
	}
	else if (v2 == 4) {
		OUT = 0x80|(r2<<3)|r1;
	}
	else {
		OUT = 0xC0|(r2<<3)|r1;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__movswl_RxR(int r1, int r2)
{
	OUT = 0x0F;
	OUT = 0xBF;
	OUT = 0xC0|(r2<<3)|r1;
}

inline void asm__movzbl_VoRxRxVoxR(int v1, int r1, int r2, int v2, int r3)
{
	/* movzbl v1(r1,r2,v2),r3 */
	assert(v2 == 1 || v2 == 2 || v2 == 4 || v2 == 8);
	assert(r2 != REG_esp);
	OUT = 0x0F;
	OUT = 0xB6;
	OUT = 0x84|(r3<<3);
	if (v2 == 1) {
		OUT = 0x00|(r2<<3)|r1;
	}
	else if (v2 == 2) {
		OUT = 0x40|(r2<<3)|r1;
	}
	else if (v2 == 4) {
		OUT = 0x80|(r2<<3)|r1;
	}
	else {
		OUT = 0xC0|(r2<<3)|r1;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__movzwl_RxR(int r1, int r2)
{
	OUT = 0x0F;
	OUT = 0xB7;
	OUT = 0xC0|(r2<<3)|r1;
}

inline void asm__negl_R(int r1)
{
	OUT = 0xF7;
	OUT = 0xD8|r1;
}

inline void asm__orl_RxR(int r1, int r2)
{
	OUT = 0x09;
	OUT = 0xC0|(r1<<3)|r2;
}

inline void asm__popl_R(int r1)
{
	OUT = 0x58|r1;
}

inline void asm__pushl_R(int r1)
{
	OUT = 0x50|r1;
}

inline void asm__pushl_VoRo(int v1, int r1)
{
	OUT = 0xFF;
	OUT = 0xB0|r1;
	if (r1 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__pushl_V(int v1)
{
	OUT = 0x68;
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__ret(void)
{
	OUT = 0xC3;
}

inline void asm__sall_RxVoRo(int r1, int v1, int r2)
{
	/* sall r1,v1(r2) */
	assert(r1 == REG_ecx);
	OUT = 0xD3;
	OUT = 0xA0|r2;
	if (r2 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__sarl_RxVoRo(int r1, int v1, int r2)
{
	/* sarl r1,v1(r2) */
	assert(r1 == REG_ecx);
	OUT = 0xD3;
	OUT = 0xB8|r2;
	if (r2 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__sarl_VxR(int v1, int r1)
{
	assert(v1 >= 0 && v1 <= 255);
	OUT = 0xC1;
	OUT = 0xF8|r1;
	OUT = v1;
}

inline void asm__sbbl_RxR(int r1, int r2)
{
	OUT = 0x19;
	OUT = 0xC0|(r1<<3)|r2;
}

inline void asm__shrl_RxVoRo(int r1, int v1, int r2)
{
	/* shrl r1,v1(r2) */
	assert(r1 == REG_ecx);
	OUT = 0xD3;
	OUT = 0xA8|r2;
	if (r2 == REG_esp) {
		OUT = 0x20|REG_esp;
	}
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__subl_RxR(int r1, int r2)
{
	OUT = 0x29;
	OUT = 0xC0|(r1<<3)|r2;
}

inline void asm__subl_VxR(int v1, int r1)
{
	OUT = 0x81;
	OUT = 0xE8|r1;
	OUT = v1 & 0xff;
	OUT = (v1 >> 8) & 0xff;
	OUT = (v1 >> 16) & 0xff;
	OUT = (v1 >> 24) & 0xff;
}

inline void asm__xchgl_RxR(int r1, int r2)
{
	OUT = 0x87;
	OUT = 0xC0|(r1<<3)|r2;
}

inline void asm__xorl_RxR(int r1, int r2)
{
	OUT = 0x31;
	OUT = 0xC0|(r1<<3)|r2;
}

/*
 * Magic instruction used in INVOKE....
 */
inline void asm__skipeq5(void)
{
	OUT = 0x74;
	OUT = 18;
}

/*
 * Magic instruction used in TABLESWITCH
 */
inline void asm__skipb1(void)
{
	OUT = 0x72;
	OUT = 5;
}

/*
 * Magic instructions used in LOOKUPSWITCH
 */
inline void asm__skipeq3(void)
{
	OUT = 0x74;
	OUT = 10;
}

inline void asm__skipneback5(void)
{
	OUT = 0x75;
	OUT = -16;
}

/*
 * Magic instructions used for array bounds checks
 */
inline void asm__skiplt1(void)
{
	OUT = 0x72;
	OUT = 5;
}
