/*
  program: ntu2pmc.c
  version: $Id: ntu2pmc.c,v 1.2 1997/06/14 07:24:28 d791013 Exp $
  author: Chun-Yu Lee <d791013@ce.ntu.edu.tw>
  function: Similar to ntu2cjk except that the generated Type 1 font
            each font plane forms a font file.
  usage:
        ntu2pmc mingli.ttf 0xa4 MingTi 5001 > b5mia4.ps
   will produces b5mia4.ps font file (needs advanced processing
   to be PFA or PFB fonts, e.g. by t1asm and t1binary)
   with fontname MingTiA4 and UniqueID 5001003.
*/

/*
  The original source documentation (shortened):

  software: ttf2ps
  version : 1.0, 29 May 1994.
  function: Chinese True Type Font to Postscript Translator
            Reads Chinese TTF file data. Output to stdout in
            Postscript format curve data.
  ftp site: ifcss.org[129.107.1.155]:/software/fonts/util
  filename: ttf2ps.zip
            Chinese TTF file can be found in ifcss.org:/software/fonts
  author  : Liangsheng Wu <lsn@mail.soton.ac.uk>

  This is the first 'crude' implementation of ttf2ps.

  The output is in Postscript format. Redirect the output from stdout to a
  file or your postscript printer.

  Tested on SunOS Unix using ntu_li_m.ttf TTF font file.

  Special thanks to Lin YawJen who provides the Chinese TTF file loader
  module.
*/

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <memory.h>
#include <time.h>

#ifdef MSDOS
#define READ_BIN    "rb"
#else
#define READ_BIN    "r"
#endif

#define UGC

#ifdef UGC
#define __fastcall
#define __max(a,b) ( a >= b ? a : b )
#define _getch getchar
#define getch getchar
#define SEEK_SET 0
#endif

#define BOOL short
#define FALSE 0
#define TRUE 1
/*
#define EM 1024 */ /* It is better to get em from file */
unsigned short EM;

struct SUBHD
       {
       unsigned short fstcode, ecnt, ioff;
       short idlt;
       } *subheader = NULL;

typedef short fixp; /* May use long to prevent overflow */
/* Fix point : S_############.## bit, range +-4096 with 1/4 precision */

struct SCANLINE
     {
     short max, cur;
     fixp *grid;
     } *rowarray, *colarray;

/* drawdot and hline mask */
unsigned char mask_pixel[8] = {0x80,0x40,0x20,0x10,8,4,2,1};
unsigned char mask_from[8] = {0xff,0x7f,0x3f,0x1f,0xf,7,3,1};
unsigned char mask_to[8] = {0x80,0xc0,0xe0,0xf0,0xf8,0xfc,0xfe,0xff};

unsigned long cmap, loca, glyf, head, loca_size;
unsigned short subheadkey[256], *glyfarray, *epts_ctr, num_pts;
short num_ctr;
fixp *xcoor, *ycoor;
fixp lastx, lasty;
fixp lastpsx, lastpsy;
unsigned char *flag;

void load_table(FILE *fn);
void get_upem(FILE *fn);
void load_cmap(FILE *fn);
short font_index(unsigned char *str);
BOOL load_font(FILE *fn,short idx);
void init_row();
void convert();
void clear_data();

unsigned short get16(FILE *fn)
     {
     unsigned char c1 = getc(fn);
     unsigned char c2 = getc(fn);
     return ((unsigned short)c1)*256+c2;
     }

unsigned long get32(FILE *fn)
     {
     unsigned short s1 = get16(fn);
     unsigned short s2 = get16(fn);
     return ((unsigned long)s1)*65536L+s2;
     }

unsigned char startc[3] = {0xa4,0x40,0};
short ymin, ymax, xmax, xmin, xsize;
#define ysize ymax
short res;
unsigned char *font_buf;

#define PSLastPt(x,y) lastpsx = x; lastpsy = y
#define PSMoveto(x,y) printf("%d %d rmoveto\n",x-lastpsx,y-lastpsy); PSLastPt(x,y)
#define PSLineto(x,y) printf("%d %d rlineto\n",x-lastpsx,y-lastpsy); PSLastPt(x,y)
void PSCurveto(fixp x, fixp y, int s, int t);
void PSHead();
void PSEnd();
void Beg();
void End();

int main(int argc, char *argv[]) {
     FILE *fn;
     short cidx, i;
     int j, fst, start_offpt, end_offpt=0;
     int plan1, plan2, iplan;

     if (argc < 5) {
       fprintf(stderr,"Usage: ntu2pmc ntu_ttf_file font_plane font_prefix UID_prefix\n");
       fprintf(stderr,"  where, 0xa1 <= font_plane <= 0xfe\n");
       fprintf(stderr,"         fontname_prefix: e.g. KaiSu\n");
       fprintf(stderr,"         4000 <= UID_prefix   <= 16777\n");
       fprintf(stderr,"  and Fontname = fontname_prefix+font_plane(%%X)\n");
       return 1;
     }

     plan1 = (int) strtol(argv[2],NULL,0);
     plan2 = plan1;

     res = 1000;  /* for Type 1 Font */

     fn = fopen(argv[1],READ_BIN);
     if (fn == NULL) { fprintf(stderr, "Can't open ttffile\n"); return -1; }
     load_table(fn);
     get_upem(fn);
     load_cmap(fn);
     init_row();
     /* cidx = font_index(startc); */

     for (iplan=plan1;iplan<=plan2;iplan++) {
       int code_st, code_en, ich, iloop=0;

       PSHead(iplan, argv[3], atoi(argv[4]));
       startc[0] = iplan;
       code_st = 0x40; code_en = 0x7e;
       do {
         for (ich=code_st;ich<=code_en;ich++) {
           startc[1] = ich;
           cidx = font_index (startc);

           if (!load_font(fn,cidx)) {
             fprintf(stderr,"Font error or composite font\n");
             /* break; */
           }

           Beg(ich);

           /* convert(); */

           for (i = 0, j = 0; i < num_ctr; i++) {
             fst = j;
             if (i > 0) puts("closepath");
             PSMoveto(xcoor[j],ycoor[j]);
             start_offpt = 0; /*start at least 1*/

/* data pts for all contours stored in one array.
   each round j init at last j + 1 */

/* start_offpt means start of off points.
   0 means no off points in record.
   N means the position of the off point.
   end_offpt means the ending off point.
   lastx, lasty is the last ON point from which Curve and Line
   shall start.
*/

/* start with j=0. into loop, j=1.
   if pt[1] off, if start_offpt == 0, toggle start_offpt
   next j=2. if on, now start_off != 0, run Curveto.
   if pt[1] on, start_off == 0, will run Lineto.
*/
             for (j++; j <= epts_ctr[i]; j++) {
               if (!(flag[j]&1))
                  { /*Off curve*/
                  if (!start_offpt) { start_offpt = end_offpt = j; }
                  else end_offpt++;
                  }
               else { /*On Curve*/
                  if (start_offpt)
                    {
/* start_offpt stuck at j, end_offpt++.
   end_offpt - start_offpt gives no of off pts.
   start_offpt gives start of sequence.
   why need xcoor[j] ycoor[j]?
*/
                     PSCurveto(xcoor[j],ycoor[j],start_offpt,end_offpt);
                     start_offpt = 0;

/* also use start_offpt as indicator to save one variable!! */
/* after curveto, reset condition. */

                     }
                   else PSLineto(xcoor[j], ycoor[j]);
/* uses lastx, lasty */
/*Lineto(xcoor[j],ycoor[j]);*/
               }
             }
/* looks like closepath
   fst = first, i.e. go back to first */
             if (start_offpt) PSCurveto(xcoor[fst],ycoor[fst],start_offpt,end_offpt);
             else PSLineto(xcoor[fst],ycoor[fst]);
           }

           /* clear_data(); */
           End();
         }
         code_st = 0xa1; code_en = 0xfe;
       } while (++iloop<2);
       PSEnd();
     } /* loops from plan1 to plan2 */
     fclose(fn);
     return 0;
}

void PSHead(int iplan, char* font, int UID) {
int cjk;
printf("%%!FontType1-1.0: CJK-%s 001.001\n",font);
puts("%%CreationDate: Jun 20, 1997");
puts("%%VMusage: 100000 100000");
puts("11 dict begin");
puts("/FontInfo 8 dict dup begin");
puts("/version (001.001) readonly def");
printf("/FullName (%s%X) readonly def\n",font,iplan);
printf("/FamilyName (%s) readonly def\n",font);
puts("/Weight (Regular) readonly def");
puts("/ItalicAngle 0 def");
puts("/isFixedPitch true def");
puts("end readonly def");
printf("/FontName /%s%X def\n",font,iplan);
puts("/PaintType 0 def");
puts("/FontType 1 def");
puts("/FontMatrix [0.001 0 0 0.001 0 -0.25] readonly def");
puts("/hex2 {(c0 ) dup 2 index 16 2 string cvrs");
puts("dup length 3 exch sub exch putinterval cvn} def");
puts("/Encoding 256 array");
puts("0 1 255 {1 index exch /.notdef put} for");
for(cjk=0x40;cjk<=0x7e;cjk++)
  printf("dup %d /cjk%x put\n",cjk-1,cjk);
for(cjk=0xa1;cjk<=0xfe;cjk++)
  printf("dup %d /cjk%x put\n",cjk-1,cjk);
puts("readonly def");
puts("/FontBBox [0 -100 1000 900] readonly def");
printf("/UniqueID %d%03d def\n",UID,iplan);
puts("currentdict end");
puts("currentfile eexec");
puts("dup /Private 14 dict dup begin");
puts("/-| { string currentfile exch readstring pop } executeonly def");
puts("/|- { noaccess def } executeonly def");
puts("/| { noaccess put } executeonly def");
puts("/BlueValues [ ] |-");
puts("/MinFeature{16 16} |-");
puts("/StdHW [ 24 ] |-");
puts("/StdVW [ 48 ] |-");
puts("/ForceBold false def");
puts("/password 5839 def");
printf("/UniqueID %d%03d def\n",UID,iplan);
/*
puts("/OtherSubrs ");
puts("[ {} {} {} ");
puts("{systemdict /internaldict known not");
puts("{pop 3}");
puts("{1183615869 systemdict /internaldict get exec");
puts("dup /startlock known");
puts("{/startlock get exec}");
puts("{dup /strtlck known");
puts("{/strtlck get exec}");
puts("{pop 3}");
puts("ifelse}");
puts("ifelse}");
puts("ifelse");
puts("} executeonly");
puts("]|-");
*/
puts("/Subrs 4 array");
puts("dup 0 {");
puts("3 0 callothersubr");
puts("pop");
puts("pop");
puts("setcurrentpoint");
puts("return");
puts("} |");
puts("dup 1 {");
puts("0 1 callothersubr");
puts("return");
puts("} |");
puts("dup 2 {");
puts("0 2 callothersubr");
puts("return");
puts("} |");
puts("dup 3 {");
puts("return");
puts("} |");
puts("|-");
puts("2 index /CharStrings 158 dict dup begin");
}

void PSEnd() {
puts("/.notdef { 0 250 hsbw endchar } |-");
puts("end end readonly put noaccess put");
puts("dup /FontName get exch definefont pop");
puts("mark currentfile closefile");
}

void Beg(int iglyph) {
  printf("/c%2X {\n", iglyph);
  puts("0 1000 hsbw");
  lastpsx = lastpsy = 0;
}

void End() {
  puts("closepath\nendchar } |-");
}

void load_table(FILE *fn)
     {
     char buf[5];
     short i, cnt;

     fseek(fn,4L,SEEK_SET);
     cnt = get16(fn);
     fseek(fn,12L,SEEK_SET);
     for (i = 0; i < cnt; i++)
         {
         fread(buf,1,4,fn);
         buf[4] = 0;
         if (!strcmp(buf,"cmap")) { get32(fn); cmap = get32(fn); get32(fn); }
         else if (!strcmp(buf,"loca"))
              { get32(fn); loca = get32(fn); loca_size = get32(fn); }
         else if (!strcmp(buf,"glyf"))
              { get32(fn); glyf = get32(fn); get32(fn); }
         else if (!strcmp(buf,"head"))
              { get32(fn); head = get32(fn); get32(fn); }
         else { get32(fn); get32(fn); get32(fn); } /*skip*/
         }
     }

void get_upem (FILE *fn) {
     fseek(fn,head+18L,SEEK_SET);
     EM = (short)get16(fn);
}

void load_cmap(FILE *fn)
     {
     unsigned short i, j, cnt, fmt, len, num_var = 0;
     unsigned long *off;
     fseek(fn,cmap+2,SEEK_SET);
     cnt = get16(fn);
     off = (unsigned long *)calloc(sizeof(unsigned long),cnt);
     for (i = 0; i < cnt; i++) { get16(fn); get16(fn); off[i] = get32(fn); }
     for (i = 0; i < cnt; i++)
         {
         fseek(fn,cmap+off[i],SEEK_SET);
         fmt = get16(fn);
         if (fmt != 2) continue;
         len = get16(fn); get16(fn);
         for (j = 0; j < 256; j++)
             {
             subheadkey[j] = get16(fn)/8;
             if (subheadkey[j] > num_var) num_var = subheadkey[j];
             }
         subheader = (struct SUBHD *)calloc(sizeof(struct SUBHD),++num_var);
         for (j = 0; j < num_var; j++)
             {
             subheader[j].fstcode = get16(fn);
             subheader[j].ecnt = get16(fn);
             subheader[j].idlt = (short)get16(fn);
             /* Modified by Chun-Yu Lee
             subheader[j].ioff = get16(fn);
             */
	     /* ttf2ps NTU version */
             subheader[j].ioff = get16(fn) - (num_var-j-1)*8-2;
             }
         len -= 6+2*256+num_var*8;
         glyfarray = (unsigned short *)calloc(sizeof(unsigned short), len/2);
         for (j = 0; j < len/2; j++) glyfarray[j] = get16(fn);
         break;
         }
     free(off);
     if (!subheader) { fprintf(stderr,"Not found format 2 mapping\n"); exit(-1); }

     /* Get ymax ymin from head */
     fseek(fn,head+36L,SEEK_SET);
     xmin = (short)get16(fn);  /* usual negtive or 0 */
     ymin = (short)get16(fn);
     xmax = (short)get16(fn) - xmin; /* shift to original (0,0) */
     ymax = (short)get16(fn) - ymin;
     xmax = (short)(((long)xmax)*res/EM);
     ymax = (short)(((long)ymax)*res/EM);
     }

void init_row() /* Initial the data space */
     {
     short i;
     xsize = (xmax+7)/8;       /* in byte */
     rowarray = (struct SCANLINE *)calloc(sizeof(struct SCANLINE),ysize);
     if (rowarray == NULL)
        { fprintf(stderr,"Memory error when initial.\n"); exit(1); }
     colarray = (struct SCANLINE *)calloc(sizeof(struct SCANLINE),xmax);
     if (colarray == NULL)
        { fprintf(stderr,"Memory error when initial.\n"); exit(1); }
     for (i = 0; i < ysize; i++)
         {
         rowarray[i].max = 32;  /* from max of NTU_MM */
         rowarray[i].cur = 0;
         rowarray[i].grid = (fixp *)calloc(sizeof(fixp),32);
         if (rowarray[i].grid == NULL)
            { fprintf(stderr,"Memory overflow when init rowarray.\n"); exit(2); }
         }
     for (i = 0; i < xmax; i++)
         {
         colarray[i].max = 32;
         colarray[i].cur = 0;
         colarray[i].grid = (fixp *)calloc(sizeof(fixp),32);
         if (colarray[i].grid == NULL)
            { fprintf(stderr,"Memory overflow when init colarray.\n"); exit(2); }
         }
     font_buf = (unsigned char *)calloc(xsize,ysize+1);
     if (font_buf == NULL)
        { fprintf(stderr,"Memory overflow when init font_buf\n"); exit(3); }
     }
/* Must free all and allocate new ones when resize the resolution. */

void clear_data()
     {
     short i;
     for (i = 0; i < ysize; i++) rowarray[i].cur = 0;
     for (i = 0; i < xmax; i++) colarray[i].cur = 0;
     memset(font_buf,0,xsize*ysize);
     free(epts_ctr);
     free(flag);
     free(xcoor);
     free(ycoor);
     }

short font_index(unsigned char *str)
      {

      /* Modified by Chun-Yu Lee */
      /*
      unsigned short i = subheadkey[str[0]];
      unsigned short index;
      if (subheader[i].ioff==0) {
        index = str[0]*0xff+str[1] + subheader[i].idlt;
      } else {
        index = *( subheader[i].ioff/2+
                   (str[1]-subheader[i].fstcode)+
                   &subheader[i].ioff );
        if (index != 0)
          index += subheader[i].idlt;
      }
      return index;
      */

      /* Original version */
      unsigned short idx = subheadkey[str[0]];
      short ridx = subheader[idx].idlt;
      unsigned short code = str[1] - subheader[idx].fstcode;

      if (code >= subheader[idx].ecnt) return 0;
      ridx += glyfarray[subheader[idx].ioff/2+code];
      return ridx;

      }

BOOL load_font(FILE *fn, short idx)
     {
     long off;
     short i;
     unsigned short j;
     unsigned char c, ct;

     fseek(fn,loca+(long)idx*4L,SEEK_SET);
     off = get32(fn);
     fseek(fn,glyf+off,SEEK_SET);
     num_ctr = (short)get16(fn);
     if (num_ctr < 0) return FALSE;
     get16(fn); get16(fn); get16(fn); get16(fn);
     epts_ctr = (unsigned short *)calloc(sizeof(unsigned short),num_ctr);
     for (i = 0; i < num_ctr; i++) epts_ctr[i] = get16(fn);
     num_pts = epts_ctr[num_ctr-1]+1; /*zero base*/
     j = get16(fn);
     while (j--) getc(fn); /*skip instruction*/
     flag = (unsigned char *)calloc(1,num_pts);
     xcoor = (fixp *)calloc(sizeof(fixp),num_pts);
     ycoor = (fixp *)calloc(sizeof(fixp),num_pts);
     for (j = 0; j < num_pts;)
         {
         flag[j++] = c = getc(fn);
         if ( c&8) { ct = getc(fn); while (ct--) flag[j++] = c; }
         }
     for (j = 0; j < num_pts; j++)
         {
         if (flag[j] & 2)
            {
            c = getc(fn);
            xcoor[j] = (flag[j] & 0x10)?((short)c):(-1*(short)c);
            }
         else if (flag[j] & 0x10) xcoor[j] = 0;
         else xcoor[j] = (short)get16(fn);
         }
     for (j = 0; j < num_pts; j++)
         {
         if (flag[j] & 4)
            {
            c = getc(fn);
            ycoor[j] =(flag[j] & 0x20)?((short)c):(-1*(short)c);
            }
         else if (flag[j] & 0x20) ycoor[j] = 0;
         else ycoor[j] = (short)get16(fn);
         }
     for (j = 1; j < num_pts; j++)
         { xcoor[j] += xcoor[j-1]; ycoor[j] += ycoor[j-1]; }
     for (j = 0; j < num_pts; j++) {
         /* Scale maybe loose precision in old version */
         /*
         xcoor[j] = (fixp)(((long)(xcoor[j]-xmin))*res/(EM/4))-2;
         ycoor[j] = (fixp)(((long)(ycoor[j]-ymin))*res/(EM/4))-2;
         */
         /*res x res, 1/4 precision, ** must shift coor before scale */
         xcoor[j] = (fixp)(((long)(xcoor[j]-xmin))*res/EM); /* Modify */
         ycoor[j] = (fixp)(((long)(ycoor[j]-ymin))*res/EM);
         }
     return TRUE;
     }

void PSCurveto(fixp x, fixp y, int s, int t)
{
     int N, i;
     double sx[3], sy[3], cx[4], cy[4];

     N = t-s+2;
     for(i=0; i<N-1; i++) {
     sx[0] = i==0?xcoor[s-1]:(xcoor[i+s]+xcoor[i+s-1])/2;
     sy[0] = i==0?ycoor[s-1]:(ycoor[i+s]+ycoor[i+s-1])/2;
     sx[1] = xcoor[s+i];
     sy[1] = ycoor[s+i];
     sx[2] = i==N-2?x:(xcoor[s+i]+xcoor[s+i+1])/2;
     sy[2] = i==N-2?y:(ycoor[s+i]+ycoor[s+i+1])/2;
     cx[3] = sx[2];
     cy[3] = sy[2];
     cx[1] = (2*sx[1]+sx[0])/3;
     cy[1] = (2*sy[1]+sy[0])/3;
     cx[2] = (sx[2]+2*sx[1])/3;
     cy[2] = (sy[2]+2*sy[1])/3;
     printf("%d %d %d %d %d %d rrcurveto\n",
       (int)(cx[1]-lastpsx),(int)(cy[1]-lastpsy),
       (int)(cx[2]-cx[1]),(int)(cy[2]-cy[1]),
       (int)(cx[3]-cx[2]),(int)(cy[3]-cy[2]));
     PSLastPt(cx[3],cy[3]);
     }
}


#define Moveto(x, y)  {lastx = (x); lasty = (y);}
void Lineto(fixp x, fixp y);
void Curveto(fixp x, fixp y,unsigned short ctrl_fst,unsigned short ctrl_last);
void SplitCurve(fixp x1, fixp y1, fixp x2, fixp y2); /* Modify */
void Draw();

void convert()
     {
     short i;
     unsigned short j, start_offpt, fst, end_offpt = 0;

     for (i = 0, j = 0; i < num_ctr; i++)
         {
         fst = j;
         Moveto(xcoor[j],ycoor[j]);
         start_offpt = 0; /*start at least 1*/

/* data pts for all contours stored in one array.
   each round j init at last j + 1 */

         for (j++; j <= epts_ctr[i]; j++)
             {
             if (!(flag[j]&1))
                { /*Off curve*/
                if (!start_offpt) { start_offpt = end_offpt = j; }
                else end_offpt++;
                }
             else { /*On Curve*/
                  if (start_offpt)
                     {
                     Curveto(xcoor[j],ycoor[j],start_offpt,end_offpt);
                     start_offpt = 0;
                     }
                  else Lineto(xcoor[j],ycoor[j]);
                  }

             }
         if (start_offpt) Curveto(xcoor[fst],ycoor[fst],start_offpt,end_offpt);
         else Lineto(xcoor[fst],ycoor[fst]);
         }
     Draw();
     }

void Lineto(fixp x, fixp y)
     {
     short start, end;
     long delta, sy, sx; /* 10 bit point */
     fixp *tmp;

     if (x != lastx) /* usual apply under 96X96 */
        {
        delta = ((long)(y-lasty))*1024/((long)(x-lastx));
        if (x < lastx)
           {
           start = (x+3)/4;
           end = (lastx+3)/4;
           sy = ((long)y)*256 + (((long)start)*4-((long)x))*delta/4;
           }
        else {
             start = (lastx+3)/4;
             end = (x+3)/4;
             sy = ((long)lasty)*256 + ((long)(start*4-lastx))*delta/4;
             }
        if (start < 0) { sy += delta*(-start); start = 0;}
        if (end >= xmax) end = xmax-1;
        for (; start < end; start++, sy += delta)
            { /* xadd(start,(fixp)(sy/256));*/
            if (colarray[start].cur == colarray[start].max)
               { /* Growth 8 each time */
               tmp = (fixp *)realloc(colarray[start].grid,
                                  (colarray[start].max+8)*sizeof(fixp));
               if (tmp == NULL) { fprintf(stderr,"Memory over.\n"); continue; }
               colarray[start].grid = tmp;
               colarray[start].max += 8;
               }
            colarray[start].grid[colarray[start].cur++] = (fixp)(sy/256);
            }
        }
     if (y != lasty)
        {
        delta = ((long)(x-lastx))*1024/((long)(y-lasty));
        if (y < lasty)
           {
           start = (y+3)/4;
           end = (lasty+3)/4;
           sx = ((long)x)*256 + (((long)start)*4-((long)y))*delta/4;
           }
        else {
             start = (lasty+3)/4;
             end = (y+3)/4;
             sx = ((long)lastx)*256 + ((long)(start*4-lasty))*delta/4;
             }
        if (start < 0) { sx += delta*(-start); start = 0;}
        if (end >= ymax) end = ymax-1;
        for (; start < end; start++, sx += delta)
            { /* yadd(start,(fixp)(sx/256));*/
            if (rowarray[start].cur == rowarray[start].max)
               { /* Growth 8 each time */
               tmp = (fixp *)realloc(rowarray[start].grid,
                                  (rowarray[start].max+8)*sizeof(fixp));
               if (tmp == NULL) { fprintf(stderr,"Memory over.\n"); continue; }
               rowarray[start].grid = tmp;
               rowarray[start].max += 8;
               }
            rowarray[start].grid[rowarray[start].cur++] = (fixp)(sx/256);
            }
        }
     lastx = x; lasty = y;
     }

void Curveto(fixp x, fixp y,unsigned short ctrl_fst,unsigned short ctrl_last)
     {
     unsigned short ctrl_next = ctrl_fst+1;
     fixp _x = xcoor[ctrl_fst], _y = ycoor[ctrl_fst];
     fixp tx, ty;

     Lineto((lastx+_x)/2, (lasty+_y)/2); /*Uniform B-spline*/
     for (; ctrl_fst <= ctrl_last; ctrl_fst++, ctrl_next++)
         {
         if (ctrl_next <= ctrl_last)
            { tx = xcoor[ctrl_next]; ty = ycoor[ctrl_next]; }
         else { tx = x; ty = y; }
         SplitCurve(_x, _y, (_x+tx)/2, (_y+ty)/2);
         _x = tx;
         _y = ty;
         }
     Lineto(x, y);
     }

void SplitCurve(fixp x1, fixp y1, fixp x2, fixp y2) /* Great Modify */
     { /*Delta routine for split nonuniform part of B-spline*/
     short n, i, m;
     long p1x = ((long)lastx-2*(long)x1+(long)x2) << 8;
     long p1y = ((long)lasty-2*(long)y1+(long)y2) << 8;
     long ddx, ddy, p2x, p2y, fx, fy, dx, dy;
     long ratio = __max(labs(p1x),labs(p1y)); /* 10 bit point */

     for (n = 0; ratio >= 512L; n++) ratio >>= 2; /* Compute coef. of spline */
     if (n)
        {                                       /* delta = 1/2^n */
        p2x = ((long)x1 - (long)lastx) << 9;
        p2y = ((long)y1 - (long)lasty) << 9;
        ddx = ((p1x*2) >> (2*n)); /* 2 order */
        ddy = ((p1y*2) >> (2*n));
        dx = (p2x >> n)+(p1x >> (2*n));
        dy = (p2y >> n)+(p1y >> (2*n));
        fx = dx + ((long)lastx << 8);
        fy = dy + ((long)lasty << 8);
        for (m = (1 << n), i = 1; i < m; i++)
            {
            Lineto((fixp)(fx/256),(fixp)(fy/256));
            dx += ddx; fx += dx;
            dy += ddy; fy += dy;
            }
        }
     Lineto(x2,y2);
     }

short ycur;

/* These two functions use data coordinates */
void Pixel(short x, short y);
void Hline(short x1, short x2);  /* Hline use ycur as y */

void Draw()
     {
     short i, j, k, cnt, len; /* Modify */
     fixp *tmp, t;

     for (i = 0; i < ysize; i++)
         {
         if (!rowarray[i].cur) continue;
         ycur = (ysize-i-1)*xsize;
         tmp = rowarray[i].grid;
         cnt = rowarray[i].cur;
         for (j = 1; j < cnt; j++) /* Insert sort for average length 6*/
             {
             for (t = tmp[j], k = j-1; (k >= 0) && (tmp[k] > t); k--)
                 tmp[k+1] = tmp[k];
             if (k < j-1) tmp[k+1] = t;
             }
         cnt &= 0xfffe; /* must be paired */
         for (j = 0; j < cnt; j+=2)
             {   /* Scan convert rule 3, usual appling under 96X96 */
             len = (tmp[j+1]-tmp[j]+2)/4; /* 0.5 */
             if (len < 2) Pixel(tmp[j]/4,ysize-i-1); /* Modify */
             else Hline((tmp[j]+3)/4,(tmp[j]+3)/4+len-1);
             /* Modify use another way do that like old version */
             }    /* Scan convert rule 0 & 1 */
         }
     for (i = 0; i < xmax; i++)
        { /* Only check rule 3, can be skipped when res > 96 */
        if (!colarray[i].cur) continue;
        tmp = colarray[i].grid;
        cnt = colarray[i].cur;
        for (j = 1; j < cnt; j++) /* Insert sort for average length 6*/
            {
            for (t = tmp[j], k = j-1; (k >= 0) && (tmp[k] > t); k--)
                tmp[k+1] = tmp[k];
            if (k < j-1) tmp[k+1] = t;
            }
        cnt &= 0xfffe; /* must be paired */
        for (j = 0; j < cnt; j+=2) /* Modify */
            if (tmp[j+1]-tmp[j] < 6) Pixel(i,ysize-1-(tmp[j]+3)/4);
            /* Scan convert rule 3, usual appling under 96X96 */
        }
     }

void __fastcall Pixel(short x, short y)
     {
     if ((y < 0) || (y >= ysize) || (x < 0) || (x >= xmax)) return;
     font_buf[y*xsize+x/8] |= mask_pixel[x&7];
     }

void Hline(short x1, short x2)
     {
     short _x1, _x2;
     unsigned char *ptr;

     if (x1 < 0)
        {
        if (x2 < 0) return;
        else if (x2 >= xmax) { x1 = 0; x2 = xmax;}
        else x1 = 0;
        }
     else if (x1 >= xmax) return;
     else if (x2 >= xmax) x2 = xmax;
     _x1 = x1/8; _x2 = x2/8;
     ptr = font_buf+ycur+_x1;
     if (_x1 == _x2) *ptr |= (mask_from[x1&7] & mask_to[x2&7]);
     else {
          *ptr++ |= mask_from[x1&7];
          for (_x1++; _x1 < _x2; _x1++) *ptr++ = 0xff;
          *ptr |= mask_to[x2&7];
          }
     }

