/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

#include <stdio.h>
#include <ctype.h>

/*
 * PrintMag
 * ********
 * Print magnification mag in a "TeX readable way". If
 * it's one of the standard magnification print besides the
 * value also \magstep... If it's not then print a warning.
 *
 * mag: magnification value, as an integer, in TeX terms,
 *     i.e. 1000 times the magnification.
 * file: file to use for outputing the message generated below such
 *     as stderr.
 */
void
PrintMag (mag, file)
     int mag;
     int file;
{
  if (mag==1000) {
    fprintf (file, "1000 [\\magstep 0]");
    return;
  }
  if (1094<=mag && mag<=1095) {
    fprintf (file, "%d [\\magstephalf]", mag);
    return;
  }
  if (1199<=mag && mag<=1201) {
    fprintf (file, "%d [\\magstep 1]", mag);
    return;
  }
  if (1439<=mag && mag<=1441) {
    fprintf (file, "%d [\\magstep 2]", mag);
    return;
  }
  if (1727<=mag && mag<=1729) {
    fprintf (file, "%d [\\magstep 3]", mag);
    return;
  }
  if (2072<=mag && mag<=2074) {
    fprintf (file, "%d [\\magstep 4]", mag);
    return;
  }
  if (2487<=mag && mag<=2489) {
    fprintf (file, "%d [\\magstep 5]", mag);
    return;
  }
  fprintf (file, "%d [no standard magnification]", mag);
}

/*
 * ReadDimensionReturnSp
 * *********************
 * Convert a string containing a TeX dimension specification and return it
 * as scaled points.
 *
 * str: a string, consisting of a number like 12 or 12.4 followed
 *      by one of TeX's dimension units such as pt, in and so on.
 * RET: this dimension as an integer in scaled points.
 */
int
ReadDimensionReturnSp(str)
     char *str;
{
  double val;
  double factor; /* conversion factor to inches. */
  int ret; /* Return value, scaled points */

  if (Strlen(str) == 0)
    Fatal ("ReadDimensionReturnSp(): empty string.");
  if (sscanf(str, "%lf", &val) != 1)
    Fatal ("ReadDimensionReturnSp(): can't read dimension value.");

  /* Now skip the number to access the dimension unit. */
  while (*str && (isdigit(*str) || *str=='.' || *str=='-' || *str=='+' || *str == ' '))
    str++;
  if (Strlen(str) != 2)
    Fatal ("ReadDimensionReturnSp(): dimension unit missing or wrong.");

  /* Now find the dimension unit. factor is the multiplication factor by which
     the input must be multiplied to 'normalize' the dimension to inches. factor = 0
     means that no legal dimension unit was found. */
  factor = 0.0;
  if (Strcmp(str, "pt") == 0) factor = 1.0/72.27;
  if (Strcmp(str, "pc") == 0) {factor= 1.0/72.27; factor *= 12;}
  if (Strcmp(str, "bp") == 0) factor = 1.0/72.00;
  if (Strcmp(str, "in") == 0) factor = 1.0;
  if (Strcmp(str, "cm") == 0) factor = 1.0/2.54;
  if (Strcmp(str, "mm") == 0) factor = 1.0/10.0/2.54;
  if (Strcmp(str, "dd") == 0) {factor = 1238.0 / 1157.0; factor /= 72.27;}
  if (Strcmp(str, "cc") == 0) {factor = 1238.0 / 1157.0; factor /= 72.27; factor *= 12.0;}
  if (Strcmp(str, "sp") == 0) {factor = 1.0/72.27; factor /= 65536.0;}
  if (factor == 0.0)
    Fatal2 ("ReadDimensionReturnSp(): illegal dimension unit \"%s\"", str);

  val *= factor; /* val is now in inches. */
  val *= 72.27; /* val is now in points. */
  val *= 65536; /* val is now in scaled points. */
  ret = val; /* Convert to integer and return. */
  return (ret);
}

/*
 * ReadDimensionReturnInches
 * *************************
 * Convert a string containing a TeX dimension specification and return it
 * converted into inches.
 *
 * str: a string, consisting of a number like 12 or 12.4 followed
 *      by one of TeX's dimension units such as pt, in, and so on.
 * RET: this dimension as a floating point number, in scaled points.
 */
double
ReadDimensionReturnInches(str)
     char *str;
{
  return (ReadDimensionReturnSp(str) / 65536.0 / 72.27);
}
