/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

#include <stdio.h>
#include "extfil.h"

void SwitchPsOutput();

EX_FILES Ex_PsOutput; /* Where does the PostScript output go? This variable is
			 the only global variable accessed by all the various
			 procedures, functions defined in emit.h etc. */

/* The following three variables are three output files which are
   reserved for the three different output types: prologue files,
   pass 0 output and pass 1 output. Splitting the output up
   into these three file types was doen for statistical purposes in order
   to measure the lenght of the three outputs. If three separate files
   are generated then these three files can NOT be just concatenated together
   and printed. */
EX_FILES Ex_PsOutputPro; /* Prologue output goes here. */
EX_FILES Ex_PsOutputPass0; /* Output from pass0 goes here. */
EX_FILES Ex_PsOutputPass1; /* Output from pass1 goes here. */

int PsOutputSwitch; /* 0, 1, 2, for prologue, pass0 and pass1 output. */

/*
 * InitPsOutputFiles
 * *****************
 * Initialize PostScript output files. A call to this routine occurs
 * only once, from DecodeArgs().
 * 
 * control: 0: output goes to specified file, which can be stdout,
 *             which is the normal case.
 *          1: output goes to three separate files.
 *             Useful for collecting some statistics.
 * fn: file name
 *   control == 0: file name used ("-" for stdout)
 *              1: base name for the three output files to be opened.
 */
void
InitPsOutputFiles(control, fn)
     int control;
     char *fn;
{
  if (Strlen(fn) == 0)
    Fatal ("InitPsOutputFiles(): empty file name.");

  switch (control) {
    case 0:
      FExOpen(&Ex_PsOutputPro, EFT_WRITE, 0, fn, NULL);
      DuplicateExF(&Ex_PsOutputPass0, &Ex_PsOutputPro);
      DuplicateExF(&Ex_PsOutputPass1, &Ex_PsOutputPro);
      break;
    case 1:
      FExOpen(&Ex_PsOutputPro,   EFT_WRITE, EFQ_NO_STDIN, fn, "pro");
      FExOpen(&Ex_PsOutputPass0, EFT_WRITE, EFQ_NO_STDIN, fn, "pass0");
      FExOpen(&Ex_PsOutputPass1, EFT_WRITE, EFQ_NO_STDIN, fn, "pass1");
      break;
    default:
      Fatal ("InitPsOutputFiles(): default.");
  }
  SwitchPsOutput(0);
}

/*
 * ClosePsOutput
 * *************
 * Close the PostScript output.
 */
void
ClosePsOutput()
{
  FExClose(&Ex_PsOutputPro);
  FExClose(&Ex_PsOutputPass0);
  FExClose(&Ex_PsOutputPass1);
}

/*
 * SwitchPsOutput
 * **************
 * Switch PostScript output to one of the three given files. Actually
 * as explained in the beginning of this section this switching is only
 * relevant if statistics should be computed.
 *
 * control: 0: switch to output for prologue files
 *          1: switch to output for pass0 output
 *          2: switch to output for pass1 output
 */
void
SwitchPsOutput(control)
     int control;
{
  switch(control) {
    case 0:
      Ex_PsOutput = Ex_PsOutputPro;
      PsOutputSwitch = 0;
      break;
    case 1:
      Ex_PsOutput = Ex_PsOutputPass0;
      PsOutputSwitch = 1;
      break;
    case 2:
      Ex_PsOutput = Ex_PsOutputPass1;
      PsOutputSwitch = 2;
      break;
    default:
      Fatal ("SwitchPsOutput(): default.");
  }
}

/* -1: it is not loaded, otherwise 0, 1, or 2, depending on what
   the switch was turned on. */
int PsOutputSave = -1;

/*
 * PushAndSwitchPsOutput
 * *********************
 * Save where the current PostScript output goes and
 * switch to a new output file. Again, it's the same deal
 * than with the previous procudure: only relevant for
 * the case where statistics is collected.
 *
 * control: like above.
 */
void
PushAndSwitchPsOutput(control)
{
  if (PsOutputSave != -1)
    Fatal ("PushAndSwitchPsOutput(): double push.");
  PsOutputSave = PsOutputSwitch;
  SwitchPsOutput(control);
}

/*
 * PopPsOutputSwitch
 * *****************
 * Restore the current output switch.
 */
void
PopPsOutputSwitch()
{
  if (PsOutputSave == -1)
    Fatal ("PopPsOutputSwitch(): empty.");
  SwitchPsOutput(PsOutputSave);
  PsOutputSave = -1;
}
