/*
 *   BITPXL converts
 *
 *      -   the command-stream of a HP-LJ Laserprinter
 *      -   Microsoft Paint files
 *
 *   into a (TeX)-PXL file (format PXL 1001) and additionally
 *   a PL file which can be converted by PLTOTF into a TFM file.
 *   The intended usage is for including pictures into TeX-documentes.
 *   Text and Figures can be previewed using standard previewing
 *   programs (eg. CDVI).
 *
 *   LJTOPXL has been tested on LJ-outputs from the following
 *   programs:  HP-DRAWING GALLERY, FRAMEWORK-II, OA-II
 *
 *                                               Gustaf Neumann
 */

#include <stdio.h>
#include <string.h>

#define HPLJ (unsigned char) 0
#define MSP  (unsigned char) 1

#define FILENAMLEN 100     /* max length of names of the used files */
#define PXLLINELEN 1000    /* max length of a pixelline (in bytes) */

#define CR  '\015'
#define FF  '\014'
#define LF  '\012'
#define ESC '\033'
#define TRUE      1
#define FALSE     0
#define TEMPNAME "$TEMP$.$$$"

#define SKIP_LEADING_TRAILING TRUE
#define STRIP_LEFT TRUE

#define TO_FULL_WORD(n) ((n+3)>>2)<<2


struct dir_entry {
   unsigned int width, height;
   int x_off, y_off;
   long pix_ptr;
   long tfm;
};
struct dir_entry dir[128];
int    pxl_width=0,
       pxl_width_bytes=0,
       corr_pxl_width_bytes=0,
       pxl_height=0,
       leading=0,
       trailing=0,
       left_skip=0;
long
       char_start_dirloc=0,
       pxlpos=0;

FILE *charfile, *tempfile, *pxlfile, *plfile, *masterfile;

char *formatname[] = {"HP-LJ output", "Microsoft Paint", "undef"};

char
     MasterFileName[FILENAMLEN],
       CharFileName[FILENAMLEN],
       TempFileName[FILENAMLEN],
        PxlFileName[FILENAMLEN],
         PlFileName[FILENAMLEN];


unsigned char read_char(f)
FILE *f;
{
   if (!feof(f)) return(fgetc(f));
   else return(EOF);
}

int
read_int(f,c)
FILE *f;
char *c;
{
char s[80];
int i;
   i=0; *c=read_char(f);
   while (' '== *c) {
      *c=read_char(f);
   }
   while (*c>='0' && *c<='9') {
      s[i] = *c;
      *c=read_char(f);
      i++;
      }
   s[i]='\0';
/*   printf("numstring is <%s>\n",s); */
   return(atoi(s));
}


put4(f, num )
FILE *f;
long num;
        {
        char c;
        register int i;

        for( i=0; i<4; i++ )
                {
                c = num >> (8*(3-i));
                put(f, &c, 1);
                }

        }

put2(f, num )
FILE *f;
unsigned int num;
        {
        char c;
        register int i;

        for( i=0; i<2; i++ )
                {
                c = (num >> (8*(1-i))) & 255;
                put(f, &c, 1);
                }

        }


put(f, buf, n)
FILE* f;
char *buf;
int n;
{
        int i;
        for (i=0;i<n;i++)
            fprintf(f,"%c",*buf+i);
}


void
Fatal(fmt, a, b, c)      /* issue a fatal error message */
char *fmt;               /* format                      */
char *a, *b, *c;         /* arguments                   */
{
    fprintf(stderr,"\n");
    fprintf(stderr, "FATAL--");
    fprintf(stderr, fmt, a, b, c);
    fprintf(stderr, "\n\n");
    exit(13);
}

FATALU(s,c)
char *s;
unsigned char c;
{
   Fatal("unexpected character <%c> (%d) after %s\n",c,c,s);
}

int                                  /* compare strings, ignore case */
IsSame(a, b)
char *a, *b;
{       char *x, *y;
        x=a; y=b;
        for( ; *a != '\0'; )
                if( tolower(*a++) != tolower(*b++) ) return( -1 );
        return( *x == *y ? 1 : -1 );
}

/**********************************************************************/
/*******************   read_ljfile  ***********************************/
/**********************************************************************/
void
read_ljfile(ljfile)
FILE* ljfile;
{
   int i;
   int raster_length,
       start_at_cursor=-1,
       h_cursor_pos=-1,
       v_cursor_pos=-1,
       orientation=-1,
       copies=-1,
       resolution=-1,
       perforation_skip=-1,        /* fw2 */
       top_margin=-1,              /* fw2 */
       text_length=-1,             /* fw2 */
       stroke_weight=-1,           /* oa2 */
       upright_character_style=-1, /* oa2 */
       primary_font_pitch=-1,      /* oa2 */
       primary_point_size=-1,      /* oa2 */
       temp_skip,
       first=TRUE,
       read_leading=TRUE,
       skipping=TRUE ;
   unsigned char c;
   char pixelline[PXLLINELEN];

   pxl_width = pxl_width_bytes = pxl_height = leading= trailing = 0;
   first = read_leading = skipping = TRUE ;

   if (!(tempfile = fopen(TempFileName,"wb")))
      Fatal("could not open temp file <%s> to write\n",TempFileName);

   if (STRIP_LEFT) left_skip=9999;
   else            left_skip=0;

   while (!feof(ljfile)) {
      switch (c=read_char(ljfile)) {
         case CR: break;
         case FF: break;
         case LF: break;
         case ' ': break;
         case ESC:
            switch (c=read_char(ljfile)) {
               case '*':
                  switch (c=read_char(ljfile)) {
                     case 'b':
                        i = read_int(ljfile,&c);
                        switch (c) { char *ptr;
                           case 'W':
                              raster_length=i;
                              if (raster_length>PXLLINELEN)
                                 Fatal("length of Pixelline to big (max=%d)\n",
                                       PXLLINELEN);

                              for(i=1;i<=raster_length;i++)
                                 pixelline[i]=read_char(ljfile);
                              for (i=raster_length;
                                   i>0 && pixelline[i]=='\0';
                                   i--);
                              raster_length=i;

                              if (SKIP_LEADING_TRAILING) {
                                 if (raster_length==0) {
                                    if (read_leading) leading ++;
                                    trailing ++;
                                 }
                                 else {
                                    read_leading=FALSE;
                                    trailing=0;
                                 }
                              }

                              if (pxl_width_bytes<raster_length)
                                               pxl_width_bytes = raster_length;
                              pxl_height++;
                              fprintf(tempfile,"%d:",raster_length);

                              skipping=TRUE;
                              temp_skip=0;

                              for(i=1;i<=raster_length;i++) {
                                 fprintf(tempfile,"%c",pixelline[i]);
                                 if (STRIP_LEFT) {
                                     if (skipping) {
                                        if (pixelline[i]=='\0') temp_skip++;
                                        else {
                                           skipping=FALSE;
                                           if (temp_skip<left_skip)
                                               left_skip=temp_skip;
                                        }
                                     }
                                 }
                              }
                              break;
                           default: FATALU("ESC *b#",c);
                        }
                        break;
                     case 'r':
                        c=read_char(ljfile);
                        if (c=='B') break;
                        if (c==')') { /* funnies from OA-II */
                            for (;ESC!=(c=read_char(ljfile)););
                            ungetc(c,ljfile);
                           }
                        else {
                           ungetc(c,ljfile);
                           i = read_int(ljfile,&c);
                           switch (c) {
                              case 'A':
                                 if (first) {
                                    start_at_cursor=i;
                                    /* printf("reset\n"); */
                                    pxl_width_bytes=0;
                                    pxl_height=0;
                                    first=FALSE;
                                 }
                                 break;
                              default: FATALU("ESC *r#",c);
                           }
                           break;
                        }
                        break;
                     case 't':
                        i = read_int(ljfile,&c);
                        switch (c) {
                           case 'R': resolution=i; break;
                           default: FATALU("ESC *t#",c);
                        }
                        break;
                     default:
                        FATALU("ESC*",c);
                  }
                  break;
               case '&':
                  switch (c=read_char(ljfile)) {
                     case 'a':
                        i = read_int(ljfile,&c);
                        switch (c) {
                           case 'H': h_cursor_pos =i; break;
                           case 'V': v_cursor_pos =i; break;
                           default: FATALU("ESC &a#",c);
                        }
                        break;
                     case 'l':
                        i = read_int(ljfile,&c);
                        switch (c) {
                           case 'O': orientation=i; break;
                           case 'X': copies=i; break;
                           case 'L': perforation_skip=i; break;
                           case 'E': top_margin=i; break;
                           case 'F': text_length=i; break;
                           default: FATALU("ESC &l#",c);
                        }
                        break;
                     default:
                        FATALU("ESC&",c);
                     }
                  break;
               case '(':
                  switch (c=read_char(ljfile)) {
                     case 's':
                        i = read_int(ljfile,&c);
                        switch (c) {
                           case 'h': primary_font_pitch=i;
                               i=read_int(ljfile,&c);
                               primary_point_size=i;
                               break;
                           case 'B': stroke_weight=i; break;
                           case 'S': upright_character_style =i; break;
                           default: FATALU("ESC &a#",c);
                        }
                        break;
                     default:
                        FATALU("ESC(",c);
                     }
                  break;
               case 'E': break; /* reset printer */
               default :
                  FATALU("ESC",c);
            }
            break;
         case 255:
            /* Fatal("EOF encountered on top level\n"); */
            break;
         default :
            Fatal("unexpected character <%c> (%d)\n",c,c);
      }
   }

   printf("I figured out:\n");
   printf("\tOrientation:\t\t%d\tResolution:\t\t%d\n",
             orientation,        resolution);
   printf("\tperforation_skip:\t%d\ttop_margin:\t\t%d\n",
             perforation_skip,      top_margin);
   printf("\thorizontal cursor pos:\t%d\tvertical cursor pos:\t%d\n",
             h_cursor_pos,               v_cursor_pos);
   printf("\ttext_length:\t\t%d\tstart_at_cursor:\t%d\n",
             text_length,        start_at_cursor);
   printf("\tcopies:\t\t\t%d\n",copies);

   fclose(tempfile);
}

/**********************************************************************/
/*******************   ReadTempFile  **********************************/
/**********************************************************************/
void                        /* modifies pxlpos, pxl_height, pxl_width */
ReadTempFile(left_skip,leading,trailing)
int
       leading,
       trailing,
       left_skip;
{
   int  i, remain;
   int length, to_strip;
   unsigned char c;


    if (SKIP_LEADING_TRAILING)
       pxl_height      -= (leading+trailing);
    if (STRIP_LEFT)
       pxl_width_bytes -= left_skip;

    pxl_width= pxl_width_bytes*8;
    corr_pxl_width_bytes = TO_FULL_WORD(pxl_width_bytes);


    if (!(tempfile = fopen(TempFileName,"rb")))
       Fatal("could not open temp file <%s> to read\n",TempFileName);

    /* pxl_width_bytes=corr_pxl_width_bytes; */
    for(remain=pxl_height+(leading+trailing); remain>0; remain--) {
       to_strip = left_skip;
       length   = read_int(tempfile,&c);
       if (leading>0) leading--;
       else if (remain==trailing) trailing --;
       else {
          for (i=1;i<=length;i++) {
             c=fgetc(tempfile);
             if (to_strip==0) {
                fputc(c,pxlfile);
                pxlpos++;
             }
             else to_strip--;
          }
          for (i=length; i<(left_skip+corr_pxl_width_bytes); i++) {
             if (to_strip==0) {
                fputc('\0',pxlfile);
                pxlpos++;
             }
             else to_strip--;
          }
       }
    }

   fclose(tempfile);
}


/**********************************************************************/
/*******************   ReadMSP  ***************************************/
/**********************************************************************/
void
ReadMSP(charfile)
FILE* charfile;
{
    unsigned char cc;
    int reduct=15,j,i,l;
    unsigned short lastshort, stripping, ltemp_skip, rtemp_skip, right_skip;
    short only_ff,look_for_leading, lskipping;

    pxl_width= pxl_height= leading=  trailing= left_skip=0;
    fseek(charfile,4l,0);
    cc=fgetc(charfile);
    pxl_width=(int)cc ;
    cc=fgetc(charfile);
    pxl_width |= (int)cc << 8 ;
    printf("pxl_width =%d (%X)\n", pxl_width,pxl_width);

    cc=fgetc(charfile);
    pxl_height=(int)cc ;
    cc=fgetc(charfile);
    pxl_height |= (int)cc << 8;
    printf("pxl_height=%d (%X)\n", pxl_height,pxl_height);
    fseek(charfile,32l,0);

    pxl_width_bytes = pxl_width /8;
    /* corr_pxl_width_bytes = TO_FULL_WORD(pxl_width_bytes); */


    leading=trailing=0;
    look_for_leading=1;
    if (STRIP_LEFT) {
       left_skip=99999;
       right_skip=99999;
    }

    for(l=1; l<=pxl_height; l++) {
       only_ff=1;
       lskipping=1;
       ltemp_skip=0;
       rtemp_skip=0;

       for (i=1;i<=pxl_width_bytes;i++) {short b;
          if ((cc=fgetc(charfile))==0xff) {
             if (lskipping) ltemp_skip++;
             rtemp_skip++;
          }
          else {
             only_ff=0;
             lskipping=0;
             rtemp_skip=0;
          }
       }
       if (only_ff) {
          if (look_for_leading) leading++;
          trailing++;
       }
       else {
           look_for_leading=0;
           trailing=0;
       }
       if (ltemp_skip<left_skip)  left_skip =ltemp_skip;
       if (rtemp_skip<right_skip) right_skip=rtemp_skip;

/*       printf("%.4d: leading=%d, trailing=%d (%d), ls=%d, rs=%d\n",
               l,leading,trailing,only_ff, left_skip, right_skip); */

    }

    /* if (right_skip>1) printf("right_skip(%d) !\n",right_skip);*/

    fseek(charfile,32l,0);

    pxl_height      -= (leading+trailing);
    pxl_width_bytes -= (left_skip+right_skip);

    pxl_width= pxl_width_bytes*8;
    corr_pxl_width_bytes = TO_FULL_WORD(pxl_width_bytes);


    for(l=1; l<=(pxl_height+leading+trailing); l++) {
       lastshort=0;

       if ( (l>leading) &&
            (l<=(leading+pxl_height))
          ) {
          for (i=1;i<=left_skip;i++) fgetc(charfile);
          for (i=1;i<=pxl_width_bytes;i++) {
             cc=fgetc(charfile)^0xff;
             fputc(cc,pxlfile); pxlpos++;
             lastshort <<= 8;
             lastshort |= (short)cc;
          }
          for (;i<=corr_pxl_width_bytes;i++) {
             fputc('\0',pxlfile); pxlpos++;
          }
          for (i=1;i<=right_skip;i++) fgetc(charfile);

          for (j=0; (j<=15) && ((lastshort & (1<<j))==0); j++);
          if (j<reduct) reduct=j;
       }
       else
          for (i=1;i<=(pxl_width_bytes+left_skip+right_skip);i++)
              fgetc(charfile);

    }
    printf("\tbit reduct:\t\t%d\n",reduct);
    pxl_width -= reduct;
    fclose(charfile);

}



main ( argc , argv )
  int argc ;
  char *argv[] ;
{
   int  i;
   char *s;
   char TheCharacter;
   long checksum, magnification, designsize;
   double
      pl_width, pl_height;
   char *tcp;       /* temporary character pointer */
   unsigned char CharFileType;
   long rasterloc;

   if (argv[1]==NULL) {
      printf("\nThis is BITPXL Version 0.01 stored in %s\n",argv[0]);
      printf("\nUsage:\tBITPXL filename\n");
      exit(1);
   }
   strcpy(MasterFileName,argv[1]);


   tcp = strchr(MasterFileName,'.');
   if (tcp == NULL) {
       strcpy(PlFileName,  MasterFileName);
       strcpy(PxlFileName, MasterFileName);
       strcat(MasterFileName, ".chr");
       }
   else {
       *tcp = '\0';
       strcpy(PlFileName,  MasterFileName);
       strcpy(PxlFileName, MasterFileName);
       *tcp = '.';
       }

   strcat(PlFileName,  ".pl" );
   strcat(PxlFileName, ".pxl");
   strcpy(TempFileName,  TEMPNAME);


   printf("reading from master file <%s>\n",MasterFileName);
   if (!(masterfile = fopen(MasterFileName,"r")))
      Fatal("could not open master file <%s>\n",MasterFileName);

   printf("writing to PL file <%s> ...\n",PlFileName);
   if (!(plfile = fopen(PlFileName,"w")))
      Fatal("could not open PL file <%s>\n",PlFileName);

   printf("writing to PXL file <%s> ...\n",PxlFileName);
   if (!(pxlfile = fopen(PxlFileName,"wb")))
      Fatal("could not open PXL file <%s>\n",PxlFileName);


   fprintf(plfile,"(DESIGNSIZE R 100.0)\n");
   fprintf(plfile,"(CHECKSUM O 377)\n");

   for (i=0; i<=127; i++) {
      dir[i].width=0;
      dir[i].height=0;
      dir[i].x_off=0;
      dir[i].y_off=0;
      dir[i].pix_ptr=0;
      dir[i].tfm=0;
   }
   pxlpos=4;
   checksum=255;
   magnification=23772;
   designsize=104857600l;
   put4(pxlfile,1001l);


   while (!feof(masterfile)) {

       fscanf(masterfile,"%c %s\n",&TheCharacter,CharFileName);

       tcp = strchr(CharFileName,'.');
       if (tcp == NULL)                 CharFileType = HPLJ;
       else {
           if (IsSame(tcp+1,"msp")==1)  CharFileType = MSP;
           else                         CharFileType = HPLJ;
           }

       if (!(charfile = fopen(CharFileName,"rb")))
          Fatal("could not open character-file <%s>\n", CharFileName);

       printf("Reading character <%c> from file <%s> format: <%s>\n",
          TheCharacter, CharFileName, formatname[CharFileType]);

       char_start_dirloc=pxlpos/4;
       rasterloc=pxlpos;

       switch (CharFileType) {
       case HPLJ:
                  read_ljfile(charfile);
                  fclose(charfile);
                  ReadTempFile(left_skip,leading,trailing);
                  break;
       case MSP:
                  ReadMSP(charfile);
                  break;
       default :
                  Fatal("That should never happen\n");
       }

       pl_width  = pxl_width *72.27/30000.0;
       pl_height = pxl_height*72.27/30000.0;

       printf("\tpxl_width:\t\t%d\tpxl_width (bytes):\t%d\n",
                 pxl_width,        pxl_width_bytes);
       printf("\tpxl_height:\t\t%d\tleft_skip:\t\t%d\n",
                 pxl_height,        left_skip);
       printf("\tskip leading 0-lines:\t%d\tskip trailing 0-lines:\t%d\n",
                 leading,              trailing);
       printf("\tpxl_width (PL):\t\t%.4f\tpxl_height (PL):\t%.4f\n",
                 pl_width,               pl_height);
       printf("\tcorr_pxl_width_bytes:\t%d\n",corr_pxl_width_bytes);
       printf("\trasterloc:\t\t%ld",rasterloc);
       printf("\tdirloc:\t\t%ld\n",char_start_dirloc);

       dir[TheCharacter].width=pxl_width;
       dir[TheCharacter].height=pxl_height;
       dir[TheCharacter].x_off=0;
       dir[TheCharacter].y_off=pxl_height;
       dir[TheCharacter].pix_ptr=char_start_dirloc;
       dir[TheCharacter].tfm=pxl_width*2526.0192;


       fprintf(plfile,"(CHARACTER C %c\n",TheCharacter);
       fprintf(plfile,"   (CHARWD R %f)\n",pl_width);
       fprintf(plfile,"   (CHARHT R %f)\n",pl_height);
       fprintf(plfile,"   )\n");

   } /* end of reading through master_file */



   for (i=0; i<=127; i++) {
      put2(pxlfile, dir[i].width  );
      put2(pxlfile, dir[i].height );
      put2(pxlfile, dir[i].x_off  );
      put2(pxlfile, dir[i].y_off  );
      put4(pxlfile, dir[i].pix_ptr);
      put4(pxlfile, dir[i].tfm    );
   }

   pxlpos=pxlpos / 4;
   put4(pxlfile, checksum);
   put4(pxlfile, magnification);
   put4(pxlfile, designsize);
   put4(pxlfile, pxlpos);
   put4(pxlfile,1001l);

   fclose(plfile);
   fclose(pxlfile);
}
