// $Id: TeXtfm.h,v 1.1 2000/12/13 00:12:27 yotam Exp $
// TeX's TFM file class
#ifndef _tExtfm
#define _tExtfm

#include <iostream>
#include <string>
#include <vector>
#include "baseint.h"

class TeXtfm
{
 public:

   enum verbose
   {
      errors    =   0x1,
      warnings  =   0x2,
      fclopen   =   0x4, // file open or close
      memory    =   0x8, // allocation and free
      preamble  =  0x10,
      eofsync   =  0x20,
      all       =  0xff
   };

   // constructors / destructors
   TeXtfm();
   TeXtfm(const char* fn, US32 verbFlags = 0, ostream& verboseStream = cerr);
   TeXtfm(const TeXtfm&);
   virtual ~TeXtfm();
   TeXtfm&  operator=(const TeXtfm&);

   // open also closes if necessary
   bool  open(const char* fn, US32 verbFlags = 0,
                              ostream& verboseStream = cerr);

   bool active() const;

   enum {lf, lh, bc, ec, nw, nh, nd, ni, nl, nk, ne, np, nLen}; // nLen==12
   static const char*  lengthNames[nLen];

   // preamble
   US16         length(US8 i) const;  // indexed by the above enum

   US32         header(US16 i) const;
   US32         checkSum() const {return header(0);};
   US32         hDesignSize() const {return header(1);};
   double       designSize() const;
   std::string  extraHeader(char  unprintReplace = '\0') const;

   class FixWord
   {
    public:
     FixWord(US32 w) : _w(w) {};
     operator  Int32()  const {return _w;};
     operator  US32()   const {return _w;};
     operator  float()  const {return (float)_w  / (float)(1<<20);};
     operator  double() const {return (double)_w / (double)(1<<20);};
    private:
     Int32  _w;
   };


   class CharInfo
   {
    public:
      CharInfo(US32 w);
      enum Tag {no_tag, lig_tag, list_tag, ext_tag};
      static const char*  tagNames[4];
      bool       valid() const        {return (_width_index != 0);};
      US8        width_index() const  {return _width_index;};
      US8        height_index() const {return _height_index;};
      US8        depth_index() const  {return _depth_index;};
      US8        italic_index() const {return _italic_index;};
      Tag        tag() const          {return _tag;};
      US8        reminder() const     {return _reminder;};
    private:
      US8        _width_index;
      US8        _height_index;
      US8        _depth_index;
      US8        _italic_index;
      Tag        _tag;
      US8        _reminder;
   }; // CharInfo
   CharInfo     char_info(US8 c) const; //  bc <= c <= ec

   Int32         width(US16 i) const;
   Int32         height(US16 i) const;
   Int32         depth(US16 i) const;
   Int32         italic(US16 i) const;
        
   Int32         cWidth(US8 c) const;
   Int32         cHeight(US8 c) const;
   Int32         cDepth(US8 c) const;
   Int32         cItalic(US8 c) const;

   // Max of the above
   Int32         maxWidth() const;
   Int32         maxHeight() const;
   Int32         maxDepth() const;
   Int32         maxItalic() const;

   // next-ascend char if tag(c)==list_tag
   Int16        listAscend(US8 c) const;

   class FourBytes
   {
    public:
      FourBytes(US32 w);
    protected:
      US8  _byte[4];
   };

   class LigKern : FourBytes
   {
    public:
      LigKern(US32 w) : FourBytes(w) {};
      enum {stop_bit, next_char, op_bit, rem_byte};  // byte indices
      US8  byte(US8 bi) const {return _byte[bi];};
      bool stop_flag() const  {return(_byte[stop_bit] & 0x80);};
      bool kern_flag() const  {return(_byte[op_bit] & 0x80);};
      US8  next()      const  {return(_byte[next_char]);};
      US8  reminder() const   {return(_byte[rem_byte]);};
   }; // LigKern
   LigKern      lig_kern(US16 i) const;

   // The characters owning the Lig-Kern entry.
   // return an STL-range of iterators
   typedef vector<US8>::const_iterator  OwnerIter;
   bool    ligKernOwners(OwnerIter& first, OwnerIter& last, US16 i) const;

   Int32   kern(US16 i) const;

   class ExtensibleRecipe : FourBytes
   {
    public:
     ExtensibleRecipe(US32 w) : FourBytes(w) {};
     enum TMRB {top, mid, bot, rep};
     US8  byte(US8 bi) const {return _byte[bi];}
   }; // ExtensibleRecipe
   ExtensibleRecipe   exten(US16 i) const;

   enum // param indices
   {
      slant_code = 1,
      space_code,
      space_stretch_code,
      space_shrink_code,
      x_height_code,
      quad_code,
      extra_space_code
   };
   US32         param(US16 i) const;

   bool         sizeOk() const;

 private:
   class _TeXtfmImplement* _data;
}; // TeXtfm


extern std::ostream&
operator<<(std::ostream& , const TeXtfm::FixWord&);

extern std::ostream&
operator<<(std::ostream& , const TeXtfm&);

#endif /* _tExtfm */
