"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDslRoute = registerDslRoute;
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
var _metrics = require("../metrics");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

// eslint-disable-next-line @typescript-eslint/no-var-requires
const performance = require('perf_hooks').performance;
function registerDslRoute(router, dataSourceEnabled) {
  router.post({
    path: _common.ServiceEndpoints.GetSearchResults,
    validate: {
      body: _configSchema.schema.any()
    }
  }, async (context, request, response) => {
    const {
      query1,
      dataSourceId1,
      query2,
      dataSourceId2
    } = request.body;
    const actionName = query1 && query2 ? _metrics.METRIC_ACTION.COMPARISON_SEARCH : _metrics.METRIC_ACTION.SINGLE_SEARCH;
    const resBody = {};
    if (query1) {
      const {
        index,
        pipeline,
        size,
        ...rest
      } = query1;
      const params = pipeline !== '' ? {
        index,
        size,
        body: rest,
        search_pipeline: pipeline
      } : {
        index,
        size,
        body: rest
      };
      const start = performance.now();
      try {
        let resp;
        const invalidCharactersPattern = /[\s,:\"*+\/\\|?#><]/;
        if (index !== index.toLowerCase() || index.startsWith('_') || index.startsWith('-') || invalidCharactersPattern.test(index)) {
          resBody.errorMessage1 = {
            statusCode: 400,
            body: 'Invalid Index or missing'
          };
        }
        if (pipeline !== '*' && pipeline !== '_none' && pipeline !== '' && !/^[a-zA-Z0-9_\-*]+(,[a-zA-Z0-9_\-*]+)*$/.test(pipeline)) {
          resBody.errorMessage1 = {
            statusCode: 400,
            body: 'Invalid Pipepline'
          };
        }
        if (dataSourceEnabled && dataSourceId1) {
          const client = context.dataSource.opensearch.legacy.getClient(dataSourceId1);
          resp = await client.callAPI('search', params);
        } else {
          resp = await context.core.opensearch.legacy.client.callAsCurrentUser('search', params);
        }
        const end = performance.now();
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, 200, end - start);
        resBody.result1 = resp;
      } catch (error) {
        var _error$body, _error$body2;
        const end = performance.now();
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, error.statusCode, end - start);
        if (error.statusCode !== 404) console.error(error);

        // Template: Error: {{Error.type}} – {{Error.reason}}
        const errorMessage = `Error: ${(_error$body = error.body) === null || _error$body === void 0 || (_error$body = _error$body.error) === null || _error$body === void 0 ? void 0 : _error$body.type} - ${(_error$body2 = error.body) === null || _error$body2 === void 0 || (_error$body2 = _error$body2.error) === null || _error$body2 === void 0 ? void 0 : _error$body2.reason}`;
        resBody.errorMessage1 = {
          statusCode: error.statusCode || 500,
          body: errorMessage
        };
      }
    }
    if (query2) {
      const {
        index,
        pipeline,
        size,
        ...rest
      } = query2;
      const params = pipeline !== '' ? {
        index,
        size,
        body: rest,
        search_pipeline: pipeline
      } : {
        index,
        size,
        body: rest
      };
      const start = performance.now();
      try {
        let resp;
        const invalidCharactersPattern = /[\s,\"*+\/\\|?#><]/;
        if (index !== index.toLowerCase() || index.startsWith('_') || index.startsWith('-') || invalidCharactersPattern.test(index)) {
          throw new Error("Index invalid or missing.");
        }
        if (pipeline !== '*' && pipeline !== '_none' && pipeline !== '' && !/^[a-zA-Z0-9_\-*]+(,[a-zA-Z0-9_\-*]+)*$/.test(pipeline)) {
          resBody.errorMessage1 = {
            statusCode: 400,
            body: 'Invalid Pipepline'
          };
        }
        if (dataSourceEnabled && dataSourceId2) {
          const client = context.dataSource.opensearch.legacy.getClient(dataSourceId2);
          resp = await client.callAPI('search', params);
        } else {
          resp = await context.core.opensearch.legacy.client.callAsCurrentUser('search', params);
        }
        const end = performance.now();
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, 200, end - start);
        resBody.result2 = resp;
      } catch (error) {
        var _error$body3, _error$body4;
        const end = performance.now();
        if (error.statusCode !== 404) console.error(error);
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, error.statusCode, end - start);

        // Template: Error: {{Error.type}} – {{Error.reason}}
        const errorMessage = `Error: ${(_error$body3 = error.body) === null || _error$body3 === void 0 || (_error$body3 = _error$body3.error) === null || _error$body3 === void 0 ? void 0 : _error$body3.type} - ${(_error$body4 = error.body) === null || _error$body4 === void 0 || (_error$body4 = _error$body4.error) === null || _error$body4 === void 0 ? void 0 : _error$body4.reason}`;
        resBody.errorMessage2 = {
          statusCode: error.statusCode || 500,
          body: errorMessage
        };
      }
    }
    return response.ok({
      body: resBody
    });
  });

  // Get Indices
  router.get({
    path: `${_common.ServiceEndpoints.GetIndexes}/{dataSourceId?}`,
    validate: {
      params: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const params = {
      format: 'json'
    };
    const start = performance.now();
    try {
      const dataSourceId = request.params.dataSourceId;
      let callApi;
      if (dataSourceEnabled && dataSourceId) {
        callApi = context.dataSource.opensearch.legacy.getClient(dataSourceId).callAPI;
      } else {
        callApi = context.core.opensearch.legacy.client.callAsCurrentUser;
      }
      let resp = await callApi('cat.indices', params);
      const remoteConnections = await callApi('transport.request', {
        method: 'GET',
        path: "/_remote/info"
      });
      if (Object.keys(remoteConnections).length > 0) {
        // fetch remote indices if remote clusters exist
        const remoteClusters = Object.keys(remoteConnections).map(key => `${key}:*`).join(',');
        const resolveResponse = await callApi('transport.request', {
          method: 'GET',
          path: `/_resolve/index/${remoteClusters}`
        });
        let remoteIndices = resolveResponse.indices.map(item => ({
          index: item.name,
          format: 'json'
        }));
        resp = resp.concat(remoteIndices);
      }
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_INDEX, 200, end - start);
      return response.ok({
        body: resp
      });
    } catch (error) {
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_INDEX, error.statusCode, end - start);
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });

  // Get Pipelines
  router.get({
    path: `${_common.ServiceEndpoints.GetPipelines}/{dataSourceId?}`,
    validate: {
      params: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const params = {
      format: 'json'
    };
    const start = performance.now();
    let resBody = {};
    let resp;
    try {
      const dataSourceId = request.params.dataSourceId;
      if (dataSourceEnabled && dataSourceId) {
        resp = (await context.dataSource.opensearch.getClient(dataSourceId)).transport;
        resp = await resp.request({
          method: 'GET',
          path: `${_common.SEARCH_API}/pipeline`
        });
      } else {
        resp = await context.core.opensearch.client.asCurrentUser.transport.request({
          method: 'GET',
          path: `${_common.SEARCH_API}/pipeline`
        });
      }
      resBody = resp.body;
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_PIPELINE, 200, end - start);
      return response.ok({
        body: resBody
      });
    } catch (error) {
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_PIPELINE, error.statusCode, end - start);
      if (error.statusCode !== 404) console.error(error);
      return response.customError({
        statusCode: 404,
        body: error
      });
    }
  });
  router.post({
    path: _common.ServiceEndpoints.GetSingleSearchResults,
    validate: {
      body: _configSchema.schema.any()
    }
  }, async (context, request, response) => {
    const {
      query,
      dataSourceId
    } = request.body;
    const resBody = {};
    const {
      index,
      size,
      search_pipeline,
      ...rest
    } = query;
    const params = {
      index,
      size,
      body: rest
    };
    if (typeof search_pipeline === 'string' && search_pipeline.trim() !== '') {
      params.search_pipeline = search_pipeline;
    }
    try {
      // Execute search
      let resp;
      if (dataSourceEnabled && dataSourceId) {
        const client = context.dataSource.opensearch.legacy.getClient(dataSourceId);
        resp = await client.callAPI('search', params);
      } else {
        resp = await context.core.opensearch.legacy.client.callAsCurrentUser('search', params);
      }
      resBody.result = resp;
    } catch (error) {
      var _error$body5, _error$body6;
      if (error.statusCode !== 404) console.error(error);
      const errorMessage = `Error: ${(_error$body5 = error.body) === null || _error$body5 === void 0 || (_error$body5 = _error$body5.error) === null || _error$body5 === void 0 ? void 0 : _error$body5.type} - ${(_error$body6 = error.body) === null || _error$body6 === void 0 || (_error$body6 = _error$body6.error) === null || _error$body6 === void 0 ? void 0 : _error$body6.reason}`;
      resBody.errorMsg = {
        statusCode: error.statusCode || 500,
        body: errorMessage
      };
    }
    return response.ok({
      body: resBody
    });
  });
  router.get({
    path: _common.ServiceEndpoints.GetClusterSettings,
    validate: {}
  }, async (context, request, response) => {
    const params = {
      include_defaults: true
    };
    const start = performance.now();
    try {
      let callApi;
      if (dataSourceEnabled && request.params.dataSourceId) {
        callApi = context.dataSource.opensearch.legacy.getClient(request.params.dataSourceId).callAPI;
      } else {
        callApi = context.core.opensearch.legacy.client.callAsCurrentUser;
      }
      const resp = await callApi('cluster.getSettings', params);
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_CLUSTER_SETTINGS, 200, end - start);
      return response.ok({
        body: resp
      });
    } catch (error) {
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_CLUSTER_SETTINGS, error.statusCode, end - start);
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 400,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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