"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationsManager = void 0;
var _path = _interopRequireDefault(require("path"));
var _metrics_helper = require("../../common/metrics/metrics_helper");
var _integrations_builder = require("./integrations_builder");
var _repository = require("./repository/repository");
var _fs_data_adaptor = require("./repository/fs_data_adaptor");
var _index_data_adaptor = require("./repository/index_data_adaptor");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class IntegrationsManager {
  constructor(client, repository) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "instanceBuilder", void 0);
    _defineProperty(this, "repository", void 0);
    _defineProperty(this, "deleteIntegrationInstance", async id => {
      let children;
      try {
        children = await this.client.get('integration-instance', id);
      } catch (err) {
        var _err$output;
        return ((_err$output = err.output) === null || _err$output === void 0 ? void 0 : _err$output.statusCode) === 404 ? Promise.resolve([id]) : Promise.reject(err);
      }
      const toDelete = children.attributes.assets.filter(i => i.assetId).map(i => {
        return {
          id: i.assetId,
          type: i.assetType
        };
      });
      toDelete.push({
        id,
        type: 'integration-instance'
      });
      const result = Promise.all(toDelete.map(async asset => {
        try {
          await this.client.delete(asset.type, asset.id);
          return Promise.resolve(asset.id);
        } catch (err) {
          var _err$output2;
          (0, _metrics_helper.addRequestToMetric)('integrations', 'delete', err);
          return ((_err$output2 = err.output) === null || _err$output2 === void 0 ? void 0 : _err$output2.statusCode) === 404 ? Promise.resolve(asset.id) : Promise.reject(err);
        }
      }));
      (0, _metrics_helper.addRequestToMetric)('integrations', 'delete', 'count');
      return result;
    });
    // Internal; use getIntegrationTemplates.
    _defineProperty(this, "_getAllIntegrationTemplates", async () => {
      const integrationList = await this.repository.getIntegrationList();
      const configResults = await Promise.all(integrationList.map(x => x.getConfig()));
      const configs = configResults.filter(cfg => cfg.ok);
      return Promise.resolve({
        hits: configs.map(cfg => cfg.value)
      });
    });
    // Internal; use getIntegrationTemplates.
    _defineProperty(this, "_getIntegrationTemplatesByName", async name => {
      const integration = await this.repository.getIntegration(name);
      const config = await (integration === null || integration === void 0 ? void 0 : integration.getConfig());
      if (!config || !config.ok) {
        return Promise.resolve({
          hits: []
        });
      }
      return Promise.resolve({
        hits: [config.value]
      });
    });
    _defineProperty(this, "getIntegrationTemplates", async query => {
      if (query !== null && query !== void 0 && query.name) {
        return this._getIntegrationTemplatesByName(query.name);
      }
      return this._getAllIntegrationTemplates();
    });
    _defineProperty(this, "getIntegrationInstances", async _query => {
      var _result$saved_objects;
      (0, _metrics_helper.addRequestToMetric)('integrations', 'get', 'count');
      const result = await this.client.find({
        type: 'integration-instance',
        perPage: 1000
      });
      return Promise.resolve({
        total: result.total,
        hits: (_result$saved_objects = result.saved_objects) === null || _result$saved_objects === void 0 ? void 0 : _result$saved_objects.map(x => ({
          ...x.attributes,
          id: x.id
        }))
      });
    });
    _defineProperty(this, "getIntegrationInstance", async query => {
      (0, _metrics_helper.addRequestToMetric)('integrations', 'get', 'count');
      const result = await this.client.get('integration-instance', `${query.id}`);
      return Promise.resolve(this.buildInstanceResponse(result));
    });
    _defineProperty(this, "buildInstanceResponse", async savedObj => {
      const assets = savedObj.attributes.assets;
      const status = assets ? await this.getAssetStatus(assets) : 'available';
      return {
        id: savedObj.id,
        status,
        ...savedObj.attributes
      };
    });
    _defineProperty(this, "getAssetStatus", async assets => {
      const statuses = await Promise.all(assets.map(async asset => {
        try {
          await this.client.get(asset.assetType, asset.assetId);
          return {
            id: asset.assetId,
            status: 'available'
          };
        } catch (err) {
          var _err$output3;
          const statusCode = (_err$output3 = err.output) === null || _err$output3 === void 0 ? void 0 : _err$output3.statusCode;
          if (statusCode && 400 <= statusCode && statusCode < 500) {
            return {
              id: asset.assetId,
              status: 'unavailable'
            };
          }
          console.error('Failed to get asset status', err);
          return {
            id: asset.assetId,
            status: 'unknown'
          };
        }
      }));
      const [available, unavailable, unknown] = [statuses.filter(x => x.status === 'available').length, statuses.filter(x => x.status === 'unavailable').length, statuses.filter(x => x.status === 'unknown').length];
      if (unknown > 0) return 'unknown';
      if (unavailable > 0 && available > 0) return 'partially-available';
      if (unavailable > 0) return 'unavailable';
      return 'available';
    });
    _defineProperty(this, "loadIntegrationInstance", async (templateName, name, indexPattern, workflows, dataSource, tableName) => {
      const template = await this.repository.getIntegration(templateName);
      if (template === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      try {
        (0, _metrics_helper.addRequestToMetric)('integrations', 'create', 'count');
        const result = await this.instanceBuilder.build(template, {
          name,
          indexPattern,
          workflows,
          dataSource,
          tableName
        });
        const test = await this.client.create('integration-instance', result);
        return Promise.resolve({
          ...result,
          id: test.id
        });
      } catch (err) {
        (0, _metrics_helper.addRequestToMetric)('integrations', 'create', err);
        return Promise.reject({
          message: err.message,
          statusCode: 500
        });
      }
    });
    _defineProperty(this, "getStatic", async (templateName, staticPath) => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const data = await integration.getStatic(staticPath);
      if (data.ok) {
        return data.value;
      }
      const is404 = data.error.code === 'ENOENT';
      return Promise.reject({
        message: data.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    _defineProperty(this, "getSchemas", async templateName => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const result = await integration.getSchemas();
      if (result.ok) {
        return result.value;
      }
      const is404 = result.error.code === 'ENOENT';
      return Promise.reject({
        message: result.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    _defineProperty(this, "getAssets", async templateName => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const assets = await integration.getAssets();
      if (assets.ok) {
        return assets.value;
      }
      const is404 = assets.error.code === 'ENOENT';
      return Promise.reject({
        message: assets.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    _defineProperty(this, "getSampleData", async templateName => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const sampleData = await integration.getSampleData();
      if (sampleData.ok) {
        return sampleData.value;
      }
      const is404 = sampleData.error.code === 'ENOENT';
      return Promise.reject({
        message: sampleData.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    this.client = client;
    this.repository = repository !== null && repository !== void 0 ? repository : new _repository.TemplateManager([new _index_data_adaptor.IndexDataAdaptor(this.client), new _fs_data_adaptor.FileSystemDataAdaptor(_path.default.join(__dirname, '__data__/repository'))]);
    this.instanceBuilder = new _integrations_builder.IntegrationInstanceBuilder(this.client);
  }
}
exports.IntegrationsManager = IntegrationsManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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