"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _constants = require("../../utils/constants");
var _helpers = require("./utils/helpers");
var _MDSEnabledClientService = require("./MDSEnabledClientService");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class MonitorService extends _MDSEnabledClientService.MDSEnabledClientService {
  constructor(...args) {
    super(...args);
    _defineProperty(this, "createMonitor", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const createResponse = await client('alerting.createMonitor', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - createMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "createWorkflow", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const createResponse = await client('alerting.createWorkflow', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - createWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "deleteMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const response = await client('alerting.deleteMonitor', params);
        return res.ok({
          body: {
            ok: response.result === 'deleted' || response.result === undefined
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - deleteMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "deleteWorkflow", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          workflowId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const response = await client('alerting.deleteWorkflow', params);
        return res.ok({
          body: {
            ok: response.result === 'deleted' || response.result === undefined
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - deleteWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getMonitor', params);
        let monitor = _lodash.default.get(getResponse, 'monitor', null);
        const version = _lodash.default.get(getResponse, '_version', null);
        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);
        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);
        const associated_workflows = _lodash.default.get(getResponse, 'associated_workflows', null);
        if (monitor) {
          const client = this.getClientBasedOnDataSource(context, req);
          const aggsParams = {
            index: _constants.INDEX.ALL_ALERTS,
            body: {
              size: 0,
              query: {
                bool: {
                  must: {
                    term: {
                      monitor_id: id
                    }
                  }
                }
              },
              aggs: {
                active_count: {
                  terms: {
                    field: 'state'
                  }
                },
                '24_hour_count': {
                  date_range: {
                    field: 'start_time',
                    ranges: [{
                      from: 'now-24h/h'
                    }]
                  }
                }
              }
            }
          };
          const searchResponse = await client('alerting.getMonitors', aggsParams);
          const dayCount = _lodash.default.get(searchResponse, 'aggregations.24_hour_count.buckets.0.doc_count', 0);
          const activeBuckets = _lodash.default.get(searchResponse, 'aggregations.active_count.buckets', []);
          const activeCount = activeBuckets.reduce((acc, curr) => curr.key === 'ACTIVE' ? curr.doc_count : acc, 0);
          if (associated_workflows) {
            monitor = {
              ...monitor,
              associated_workflows,
              associatedCompositeMonitorCnt: associated_workflows.length
            };
          }
          monitor = {
            ...monitor,
            item_type: monitor.workflow_type || monitor.monitor_type,
            id,
            version
          };
          return res.ok({
            body: {
              ok: true,
              resp: monitor,
              activeCount,
              dayCount,
              version,
              ifSeqNo,
              ifPrimaryTerm
            }
          });
        } else {
          return res.ok({
            body: {
              ok: false
            }
          });
        }
      } catch (err) {
        console.error('Alerting - MonitorService - getMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getWorkflow", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getWorkflow', params);
        let workflow = _lodash.default.get(getResponse, 'workflow', null);
        const version = _lodash.default.get(getResponse, '_version', null);
        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);
        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);
        workflow.monitor_type = workflow.workflow_type;
        workflow = {
          ...workflow,
          item_type: workflow.workflow_type,
          id,
          version
        };
        return res.ok({
          body: {
            ok: true,
            resp: workflow,
            activeCount: 0,
            dayCount: 0,
            version,
            ifSeqNo,
            ifPrimaryTerm
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - getWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "updateMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          body: req.body,
          refresh: 'wait_for'
        };
        const {
          type
        } = req.body;

        // TODO DRAFT: Are we sure we need to include ifSeqNo and ifPrimaryTerm from the UI side when updating monitors?
        const {
          ifSeqNo,
          ifPrimaryTerm
        } = req.query;
        if (ifSeqNo && ifPrimaryTerm) {
          params.if_seq_no = ifSeqNo;
          params.if_primary_term = ifPrimaryTerm;
        }
        const client = this.getClientBasedOnDataSource(context, req);
        const updateResponse = await client(`alerting.${type === 'workflow' ? 'updateWorkflow' : 'updateMonitor'}`, params);
        const {
          _version,
          _id
        } = updateResponse;
        return res.ok({
          body: {
            ok: true,
            version: _version,
            id: _id
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - updateMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getMonitors", async (context, req, res) => {
      try {
        const {
          from,
          size,
          search,
          sortDirection,
          sortField,
          state,
          monitorIds
        } = req.query;
        let must = {
          match_all: {}
        };
        if (search.trim()) {
          // This is an expensive wildcard query to match monitor names such as: "This is a long monitor name"
          // search query => "long monit"
          // This is acceptable because we will never allow more than 1,000 monitors
          must = {
            query_string: {
              default_field: 'monitor.name',
              default_operator: 'AND',
              query: `*${search.trim().split(' ').join('* *')}*`
            }
          };
        }
        const should = [];
        const mustList = [must];
        if (monitorIds !== undefined) {
          mustList.push({
            terms: {
              _id: Array.isArray(monitorIds) ? monitorIds : [monitorIds]
            }
          });
        } else if (monitorIds === 'empty') {
          mustList.push({
            terms: {
              _id: []
            }
          });
        }
        if (state !== 'all') {
          const enabled = state === 'enabled';
          should.push({
            term: {
              'monitor.enabled': enabled
            }
          });
          should.push({
            term: {
              'workflow.enabled': enabled
            }
          });
        }
        const monitorSorts = {
          name: 'monitor.name.keyword'
        };
        const monitorSortPageData = {
          size: 1000
        };
        if (monitorSorts[sortField]) {
          monitorSortPageData.sort = [{
            [monitorSorts[sortField]]: sortDirection
          }];
          monitorSortPageData.size = _lodash.default.defaultTo(size, 1000);
          monitorSortPageData.from = _lodash.default.defaultTo(from, 0);
        }
        const params = {
          body: {
            seq_no_primary_term: true,
            version: true,
            ...monitorSortPageData,
            query: {
              bool: {
                should,
                minimum_should_match: state !== 'all' ? 1 : 0,
                must: mustList
              }
            },
            aggregations: {
              associated_composite_monitors: {
                nested: {
                  path: 'workflow.inputs.composite_input.sequence.delegates'
                },
                aggs: {
                  monitor_ids: {
                    terms: {
                      field: 'workflow.inputs.composite_input.sequence.delegates.monitor_id'
                    }
                  }
                }
              }
            }
          }
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getMonitors', params);
        const totalMonitors = _lodash.default.get(getResponse, 'hits.total.value', 0);
        const monitorKeyValueTuples = _lodash.default.get(getResponse, 'hits.hits', []).map(result => {
          const {
            _id: id,
            _version: version,
            _seq_no: ifSeqNo,
            _primary_term: ifPrimaryTerm,
            _source
          } = result;
          const monitor = _source.monitor ? _source.monitor : _source;
          monitor['item_type'] = monitor.workflow_type || monitor.monitor_type;
          const {
            name,
            enabled,
            item_type
          } = monitor;
          return [id, {
            id,
            version,
            ifSeqNo,
            ifPrimaryTerm,
            name,
            enabled,
            item_type,
            monitor
          }];
        }, {});
        const monitorMap = new Map(monitorKeyValueTuples);
        const associatedCompositeMonitorCountMap = {};
        _lodash.default.get(getResponse, 'aggregations.associated_composite_monitors.monitor_ids.buckets', []).forEach(({
          key,
          doc_count
        }) => {
          associatedCompositeMonitorCountMap[key] = doc_count;
        });
        const monitorIdsOutput = [...monitorMap.keys()];
        const aggsOrderData = {};
        const aggsSorts = {
          active: 'active',
          acknowledged: 'acknowledged',
          errors: 'errors',
          ignored: 'ignored',
          lastNotificationTime: 'last_notification_time'
        };
        if (aggsSorts[sortField]) {
          aggsOrderData.order = {
            [aggsSorts[sortField]]: sortDirection
          };
        }
        const aggsParams = {
          index: _constants.INDEX.ALL_ALERTS,
          body: {
            size: 0,
            query: {
              terms: {
                monitor_id: monitorIdsOutput
              }
            },
            aggregations: {
              uniq_monitor_ids: {
                terms: {
                  field: 'monitor_id',
                  ...aggsOrderData,
                  size: from + size
                },
                aggregations: {
                  active: {
                    filter: {
                      term: {
                        state: 'ACTIVE'
                      }
                    }
                  },
                  acknowledged: {
                    filter: {
                      term: {
                        state: 'ACKNOWLEDGED'
                      }
                    }
                  },
                  errors: {
                    filter: {
                      term: {
                        state: 'ERROR'
                      }
                    }
                  },
                  ignored: {
                    filter: {
                      bool: {
                        filter: {
                          term: {
                            state: 'COMPLETED'
                          }
                        },
                        must_not: {
                          exists: {
                            field: 'acknowledged_time'
                          }
                        }
                      }
                    }
                  },
                  last_notification_time: {
                    max: {
                      field: 'last_notification_time'
                    }
                  },
                  latest_alert: {
                    top_hits: {
                      size: 1,
                      sort: [{
                        start_time: {
                          order: 'desc'
                        }
                      }],
                      _source: {
                        includes: ['last_notification_time', 'trigger_name']
                      }
                    }
                  }
                }
              }
            }
          }
        };
        const esAggsResponse = await client('alerting.getMonitors', aggsParams);
        const buckets = _lodash.default.get(esAggsResponse, 'aggregations.uniq_monitor_ids.buckets', []).map(bucket => {
          const {
            key: id,
            last_notification_time: {
              value: lastNotificationTime
            },
            ignored: {
              doc_count: ignored
            },
            acknowledged: {
              doc_count: acknowledged
            },
            active: {
              doc_count: active
            },
            errors: {
              doc_count: errors
            },
            latest_alert: {
              hits: {
                hits: [{
                  _source: {
                    trigger_name: latestAlert
                  }
                }]
              }
            }
          } = bucket;
          const monitor = monitorMap.get(id);
          monitorMap.delete(id);
          return {
            ...monitor,
            id,
            lastNotificationTime,
            ignored,
            latestAlert,
            acknowledged,
            active,
            errors,
            currentTime: Date.now(),
            associatedCompositeMonitorCnt: associatedCompositeMonitorCountMap[id] || 0
          };
        });
        const unusedMonitors = [...monitorMap.values()].map(monitor => ({
          ...monitor,
          lastNotificationTime: null,
          ignored: 0,
          active: 0,
          acknowledged: 0,
          errors: 0,
          latestAlert: '--',
          currentTime: Date.now(),
          associatedCompositeMonitorCnt: associatedCompositeMonitorCountMap[monitor.id] || 0
        }));
        let results = _lodash.default.orderBy(buckets.concat(unusedMonitors), [sortField], [sortDirection]);
        // If we sorted on monitor name then we already applied from/size to the first query to limit what we're aggregating over
        // Therefore we do not need to apply from/size to this result set
        // If we sorted on aggregations, then this is our in memory pagination
        if (!monitorSorts[sortField]) {
          results = results.slice(from, from + size);
        }
        return res.ok({
          body: {
            ok: true,
            monitors: results,
            totalMonitors
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - getMonitors', err);
        if ((0, _helpers.isIndexNotFoundError)(err)) {
          return res.ok({
            body: {
              ok: false,
              resp: {
                totalMonitors: 0,
                monitors: []
              }
            }
          });
        }
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "acknowledgeAlerts", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const acknowledgeResponse = await client('alerting.acknowledgeAlerts', params);
        return res.ok({
          body: {
            ok: !acknowledgeResponse.failed.length,
            resp: acknowledgeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - acknowledgeAlerts:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "acknowledgeChainedAlerts", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          workflowId: id,
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, this.dataSourceEnabled, req, this.esDriver);
        const acknowledgeResponse = await client('alerting.acknowledgeChainedAlerts', params);
        return res.ok({
          body: {
            ok: !acknowledgeResponse.failed.length,
            resp: acknowledgeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - acknowledgeChainedAlerts:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "executeMonitor", async (context, req, res) => {
      try {
        const {
          dryrun = 'true'
        } = req.query;
        const params = {
          body: req.body,
          dryrun
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const executeResponse = await client('alerting.executeMonitor', params);
        return res.ok({
          body: {
            ok: true,
            resp: executeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - executeMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    //TODO: This is temporarily a pass through call which needs to be deprecated
    _defineProperty(this, "searchMonitors", async (context, req, res) => {
      try {
        const {
          query,
          index,
          size
        } = req.body;
        const params = {
          index,
          size,
          body: query
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const results = await client('alerting.getMonitors', params);
        return res.ok({
          body: {
            ok: true,
            resp: results
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - searchMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
  }
}
exports.default = MonitorService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfbG9kYXNoIiwiX2ludGVyb3BSZXF1aXJlRGVmYXVsdCIsInJlcXVpcmUiLCJfY29uc3RhbnRzIiwiX2hlbHBlcnMiLCJfTURTRW5hYmxlZENsaWVudFNlcnZpY2UiLCJvYmoiLCJfX2VzTW9kdWxlIiwiZGVmYXVsdCIsIl9kZWZpbmVQcm9wZXJ0eSIsImtleSIsInZhbHVlIiwiX3RvUHJvcGVydHlLZXkiLCJPYmplY3QiLCJkZWZpbmVQcm9wZXJ0eSIsImVudW1lcmFibGUiLCJjb25maWd1cmFibGUiLCJ3cml0YWJsZSIsImFyZyIsIl90b1ByaW1pdGl2ZSIsIlN0cmluZyIsImlucHV0IiwiaGludCIsInByaW0iLCJTeW1ib2wiLCJ0b1ByaW1pdGl2ZSIsInVuZGVmaW5lZCIsInJlcyIsImNhbGwiLCJUeXBlRXJyb3IiLCJOdW1iZXIiLCJNb25pdG9yU2VydmljZSIsIk1EU0VuYWJsZWRDbGllbnRTZXJ2aWNlIiwiY29uc3RydWN0b3IiLCJhcmdzIiwiY29udGV4dCIsInJlcSIsInBhcmFtcyIsImJvZHkiLCJjbGllbnQiLCJnZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZSIsImNyZWF0ZVJlc3BvbnNlIiwib2siLCJyZXNwIiwiZXJyIiwiY29uc29sZSIsImVycm9yIiwibWVzc2FnZSIsImlkIiwibW9uaXRvcklkIiwicmVzcG9uc2UiLCJyZXN1bHQiLCJ3b3JrZmxvd0lkIiwiZ2V0UmVzcG9uc2UiLCJtb25pdG9yIiwiXyIsImdldCIsInZlcnNpb24iLCJpZlNlcU5vIiwiaWZQcmltYXJ5VGVybSIsImFzc29jaWF0ZWRfd29ya2Zsb3dzIiwiYWdnc1BhcmFtcyIsImluZGV4IiwiSU5ERVgiLCJBTExfQUxFUlRTIiwic2l6ZSIsInF1ZXJ5IiwiYm9vbCIsIm11c3QiLCJ0ZXJtIiwibW9uaXRvcl9pZCIsImFnZ3MiLCJhY3RpdmVfY291bnQiLCJ0ZXJtcyIsImZpZWxkIiwiZGF0ZV9yYW5nZSIsInJhbmdlcyIsImZyb20iLCJzZWFyY2hSZXNwb25zZSIsImRheUNvdW50IiwiYWN0aXZlQnVja2V0cyIsImFjdGl2ZUNvdW50IiwicmVkdWNlIiwiYWNjIiwiY3VyciIsImRvY19jb3VudCIsImFzc29jaWF0ZWRDb21wb3NpdGVNb25pdG9yQ250IiwibGVuZ3RoIiwiaXRlbV90eXBlIiwid29ya2Zsb3dfdHlwZSIsIm1vbml0b3JfdHlwZSIsIndvcmtmbG93IiwicmVmcmVzaCIsInR5cGUiLCJpZl9zZXFfbm8iLCJpZl9wcmltYXJ5X3Rlcm0iLCJ1cGRhdGVSZXNwb25zZSIsIl92ZXJzaW9uIiwiX2lkIiwic2VhcmNoIiwic29ydERpcmVjdGlvbiIsInNvcnRGaWVsZCIsInN0YXRlIiwibW9uaXRvcklkcyIsIm1hdGNoX2FsbCIsInRyaW0iLCJxdWVyeV9zdHJpbmciLCJkZWZhdWx0X2ZpZWxkIiwiZGVmYXVsdF9vcGVyYXRvciIsInNwbGl0Iiwiam9pbiIsInNob3VsZCIsIm11c3RMaXN0IiwicHVzaCIsIkFycmF5IiwiaXNBcnJheSIsImVuYWJsZWQiLCJtb25pdG9yU29ydHMiLCJuYW1lIiwibW9uaXRvclNvcnRQYWdlRGF0YSIsInNvcnQiLCJkZWZhdWx0VG8iLCJzZXFfbm9fcHJpbWFyeV90ZXJtIiwibWluaW11bV9zaG91bGRfbWF0Y2giLCJhZ2dyZWdhdGlvbnMiLCJhc3NvY2lhdGVkX2NvbXBvc2l0ZV9tb25pdG9ycyIsIm5lc3RlZCIsInBhdGgiLCJtb25pdG9yX2lkcyIsInRvdGFsTW9uaXRvcnMiLCJtb25pdG9yS2V5VmFsdWVUdXBsZXMiLCJtYXAiLCJfc2VxX25vIiwiX3ByaW1hcnlfdGVybSIsIl9zb3VyY2UiLCJtb25pdG9yTWFwIiwiTWFwIiwiYXNzb2NpYXRlZENvbXBvc2l0ZU1vbml0b3JDb3VudE1hcCIsImZvckVhY2giLCJtb25pdG9ySWRzT3V0cHV0Iiwia2V5cyIsImFnZ3NPcmRlckRhdGEiLCJhZ2dzU29ydHMiLCJhY3RpdmUiLCJhY2tub3dsZWRnZWQiLCJlcnJvcnMiLCJpZ25vcmVkIiwibGFzdE5vdGlmaWNhdGlvblRpbWUiLCJvcmRlciIsInVuaXFfbW9uaXRvcl9pZHMiLCJmaWx0ZXIiLCJtdXN0X25vdCIsImV4aXN0cyIsImxhc3Rfbm90aWZpY2F0aW9uX3RpbWUiLCJtYXgiLCJsYXRlc3RfYWxlcnQiLCJ0b3BfaGl0cyIsInN0YXJ0X3RpbWUiLCJpbmNsdWRlcyIsImVzQWdnc1Jlc3BvbnNlIiwiYnVja2V0cyIsImJ1Y2tldCIsImhpdHMiLCJ0cmlnZ2VyX25hbWUiLCJsYXRlc3RBbGVydCIsImRlbGV0ZSIsImN1cnJlbnRUaW1lIiwiRGF0ZSIsIm5vdyIsInVudXNlZE1vbml0b3JzIiwidmFsdWVzIiwicmVzdWx0cyIsIm9yZGVyQnkiLCJjb25jYXQiLCJzbGljZSIsIm1vbml0b3JzIiwiaXNJbmRleE5vdEZvdW5kRXJyb3IiLCJhY2tub3dsZWRnZVJlc3BvbnNlIiwiZmFpbGVkIiwiZGF0YVNvdXJjZUVuYWJsZWQiLCJlc0RyaXZlciIsImRyeXJ1biIsImV4ZWN1dGVSZXNwb25zZSIsImV4cG9ydHMiLCJtb2R1bGUiXSwic291cmNlcyI6WyJNb25pdG9yU2VydmljZS5qcyJdLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuICogQ29weXJpZ2h0IE9wZW5TZWFyY2ggQ29udHJpYnV0b3JzXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG5cbmltcG9ydCBfIGZyb20gJ2xvZGFzaCc7XG5cbmltcG9ydCB7IElOREVYIH0gZnJvbSAnLi4vLi4vdXRpbHMvY29uc3RhbnRzJztcbmltcG9ydCB7IGlzSW5kZXhOb3RGb3VuZEVycm9yIH0gZnJvbSAnLi91dGlscy9oZWxwZXJzJztcbmltcG9ydCB7IE1EU0VuYWJsZWRDbGllbnRTZXJ2aWNlIH0gZnJvbSAnLi9NRFNFbmFibGVkQ2xpZW50U2VydmljZSc7XG5cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIE1vbml0b3JTZXJ2aWNlIGV4dGVuZHMgTURTRW5hYmxlZENsaWVudFNlcnZpY2Uge1xuICBjcmVhdGVNb25pdG9yID0gYXN5bmMgKGNvbnRleHQsIHJlcSwgcmVzKSA9PiB7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IHBhcmFtcyA9IHsgYm9keTogcmVxLmJvZHkgfTtcbiAgICAgIGNvbnN0IGNsaWVudCA9IHRoaXMuZ2V0Q2xpZW50QmFzZWRPbkRhdGFTb3VyY2UoY29udGV4dCwgcmVxKTtcbiAgICAgIGNvbnN0IGNyZWF0ZVJlc3BvbnNlID0gYXdhaXQgY2xpZW50KCdhbGVydGluZy5jcmVhdGVNb25pdG9yJywgcGFyYW1zKTtcbiAgICAgIHJldHVybiByZXMub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IHRydWUsXG4gICAgICAgICAgcmVzcDogY3JlYXRlUmVzcG9uc2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGNvbnNvbGUuZXJyb3IoJ0FsZXJ0aW5nIC0gTW9uaXRvclNlcnZpY2UgLSBjcmVhdGVNb25pdG9yOicsIGVycik7XG4gICAgICByZXR1cm4gcmVzLm9rKHtcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIG9rOiBmYWxzZSxcbiAgICAgICAgICByZXNwOiBlcnIubWVzc2FnZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICBjcmVhdGVXb3JrZmxvdyA9IGFzeW5jIChjb250ZXh0LCByZXEsIHJlcykgPT4ge1xuICAgIHRyeSB7XG4gICAgICBjb25zdCBwYXJhbXMgPSB7IGJvZHk6IHJlcS5ib2R5IH07XG4gICAgICBjb25zdCBjbGllbnQgPSB0aGlzLmdldENsaWVudEJhc2VkT25EYXRhU291cmNlKGNvbnRleHQsIHJlcSk7XG4gICAgICBjb25zdCBjcmVhdGVSZXNwb25zZSA9IGF3YWl0IGNsaWVudCgnYWxlcnRpbmcuY3JlYXRlV29ya2Zsb3cnLCBwYXJhbXMpO1xuICAgICAgcmV0dXJuIHJlcy5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICByZXNwOiBjcmVhdGVSZXNwb25zZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgY29uc29sZS5lcnJvcignQWxlcnRpbmcgLSBNb25pdG9yU2VydmljZSAtIGNyZWF0ZVdvcmtmbG93OicsIGVycik7XG4gICAgICByZXR1cm4gcmVzLm9rKHtcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIG9rOiBmYWxzZSxcbiAgICAgICAgICByZXNwOiBlcnIubWVzc2FnZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICBkZWxldGVNb25pdG9yID0gYXN5bmMgKGNvbnRleHQsIHJlcSwgcmVzKSA9PiB7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IHsgaWQgfSA9IHJlcS5wYXJhbXM7XG4gICAgICBjb25zdCBwYXJhbXMgPSB7IG1vbml0b3JJZDogaWQgfTtcbiAgICAgIGNvbnN0IGNsaWVudCA9IHRoaXMuZ2V0Q2xpZW50QmFzZWRPbkRhdGFTb3VyY2UoY29udGV4dCwgcmVxKTtcbiAgICAgIGNvbnN0IHJlc3BvbnNlID0gYXdhaXQgY2xpZW50KCdhbGVydGluZy5kZWxldGVNb25pdG9yJywgcGFyYW1zKTtcblxuICAgICAgcmV0dXJuIHJlcy5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogcmVzcG9uc2UucmVzdWx0ID09PSAnZGVsZXRlZCcgfHwgcmVzcG9uc2UucmVzdWx0ID09PSB1bmRlZmluZWQsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGNvbnNvbGUuZXJyb3IoJ0FsZXJ0aW5nIC0gTW9uaXRvclNlcnZpY2UgLSBkZWxldGVNb25pdG9yOicsIGVycik7XG4gICAgICByZXR1cm4gcmVzLm9rKHtcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIG9rOiBmYWxzZSxcbiAgICAgICAgICByZXNwOiBlcnIubWVzc2FnZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICBkZWxldGVXb3JrZmxvdyA9IGFzeW5jIChjb250ZXh0LCByZXEsIHJlcykgPT4ge1xuICAgIHRyeSB7XG4gICAgICBjb25zdCB7IGlkIH0gPSByZXEucGFyYW1zO1xuICAgICAgY29uc3QgcGFyYW1zID0geyB3b3JrZmxvd0lkOiBpZCB9O1xuICAgICAgY29uc3QgY2xpZW50ID0gdGhpcy5nZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZShjb250ZXh0LCByZXEpO1xuICAgICAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCBjbGllbnQoJ2FsZXJ0aW5nLmRlbGV0ZVdvcmtmbG93JywgcGFyYW1zKTtcblxuICAgICAgcmV0dXJuIHJlcy5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogcmVzcG9uc2UucmVzdWx0ID09PSAnZGVsZXRlZCcgfHwgcmVzcG9uc2UucmVzdWx0ID09PSB1bmRlZmluZWQsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGNvbnNvbGUuZXJyb3IoJ0FsZXJ0aW5nIC0gTW9uaXRvclNlcnZpY2UgLSBkZWxldGVXb3JrZmxvdzonLCBlcnIpO1xuICAgICAgcmV0dXJuIHJlcy5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogZmFsc2UsXG4gICAgICAgICAgcmVzcDogZXJyLm1lc3NhZ2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG4gIH07XG5cbiAgZ2V0TW9uaXRvciA9IGFzeW5jIChjb250ZXh0LCByZXEsIHJlcykgPT4ge1xuICAgIHRyeSB7XG4gICAgICBjb25zdCB7IGlkIH0gPSByZXEucGFyYW1zO1xuICAgICAgY29uc3QgcGFyYW1zID0geyBtb25pdG9ySWQ6IGlkIH07XG4gICAgICBjb25zdCBjbGllbnQgPSB0aGlzLmdldENsaWVudEJhc2VkT25EYXRhU291cmNlKGNvbnRleHQsIHJlcSk7XG4gICAgICBjb25zdCBnZXRSZXNwb25zZSA9IGF3YWl0IGNsaWVudCgnYWxlcnRpbmcuZ2V0TW9uaXRvcicsIHBhcmFtcyk7XG4gICAgICBsZXQgbW9uaXRvciA9IF8uZ2V0KGdldFJlc3BvbnNlLCAnbW9uaXRvcicsIG51bGwpO1xuICAgICAgY29uc3QgdmVyc2lvbiA9IF8uZ2V0KGdldFJlc3BvbnNlLCAnX3ZlcnNpb24nLCBudWxsKTtcbiAgICAgIGNvbnN0IGlmU2VxTm8gPSBfLmdldChnZXRSZXNwb25zZSwgJ19zZXFfbm8nLCBudWxsKTtcbiAgICAgIGNvbnN0IGlmUHJpbWFyeVRlcm0gPSBfLmdldChnZXRSZXNwb25zZSwgJ19wcmltYXJ5X3Rlcm0nLCBudWxsKTtcbiAgICAgIGNvbnN0IGFzc29jaWF0ZWRfd29ya2Zsb3dzID0gXy5nZXQoZ2V0UmVzcG9uc2UsICdhc3NvY2lhdGVkX3dvcmtmbG93cycsIG51bGwpO1xuICAgICAgaWYgKG1vbml0b3IpIHtcbiAgICAgICAgY29uc3QgY2xpZW50ID0gdGhpcy5nZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZShjb250ZXh0LCByZXEpO1xuICAgICAgICBjb25zdCBhZ2dzUGFyYW1zID0ge1xuICAgICAgICAgIGluZGV4OiBJTkRFWC5BTExfQUxFUlRTLFxuICAgICAgICAgIGJvZHk6IHtcbiAgICAgICAgICAgIHNpemU6IDAsXG4gICAgICAgICAgICBxdWVyeToge1xuICAgICAgICAgICAgICBib29sOiB7XG4gICAgICAgICAgICAgICAgbXVzdDoge1xuICAgICAgICAgICAgICAgICAgdGVybToge1xuICAgICAgICAgICAgICAgICAgICBtb25pdG9yX2lkOiBpZCxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBhZ2dzOiB7XG4gICAgICAgICAgICAgIGFjdGl2ZV9jb3VudDoge1xuICAgICAgICAgICAgICAgIHRlcm1zOiB7XG4gICAgICAgICAgICAgICAgICBmaWVsZDogJ3N0YXRlJyxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAnMjRfaG91cl9jb3VudCc6IHtcbiAgICAgICAgICAgICAgICBkYXRlX3JhbmdlOiB7XG4gICAgICAgICAgICAgICAgICBmaWVsZDogJ3N0YXJ0X3RpbWUnLFxuICAgICAgICAgICAgICAgICAgcmFuZ2VzOiBbeyBmcm9tOiAnbm93LTI0aC9oJyB9XSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9O1xuICAgICAgICBjb25zdCBzZWFyY2hSZXNwb25zZSA9IGF3YWl0IGNsaWVudCgnYWxlcnRpbmcuZ2V0TW9uaXRvcnMnLCBhZ2dzUGFyYW1zKTtcbiAgICAgICAgY29uc3QgZGF5Q291bnQgPSBfLmdldChzZWFyY2hSZXNwb25zZSwgJ2FnZ3JlZ2F0aW9ucy4yNF9ob3VyX2NvdW50LmJ1Y2tldHMuMC5kb2NfY291bnQnLCAwKTtcbiAgICAgICAgY29uc3QgYWN0aXZlQnVja2V0cyA9IF8uZ2V0KHNlYXJjaFJlc3BvbnNlLCAnYWdncmVnYXRpb25zLmFjdGl2ZV9jb3VudC5idWNrZXRzJywgW10pO1xuICAgICAgICBjb25zdCBhY3RpdmVDb3VudCA9IGFjdGl2ZUJ1Y2tldHMucmVkdWNlKFxuICAgICAgICAgIChhY2MsIGN1cnIpID0+IChjdXJyLmtleSA9PT0gJ0FDVElWRScgPyBjdXJyLmRvY19jb3VudCA6IGFjYyksXG4gICAgICAgICAgMFxuICAgICAgICApO1xuICAgICAgICBpZiAoYXNzb2NpYXRlZF93b3JrZmxvd3MpIHtcbiAgICAgICAgICBtb25pdG9yID0ge1xuICAgICAgICAgICAgLi4ubW9uaXRvcixcbiAgICAgICAgICAgIGFzc29jaWF0ZWRfd29ya2Zsb3dzLFxuICAgICAgICAgICAgYXNzb2NpYXRlZENvbXBvc2l0ZU1vbml0b3JDbnQ6IGFzc29jaWF0ZWRfd29ya2Zsb3dzLmxlbmd0aCxcbiAgICAgICAgICB9O1xuICAgICAgICB9XG4gICAgICAgIG1vbml0b3IgPSB7XG4gICAgICAgICAgLi4ubW9uaXRvcixcbiAgICAgICAgICBpdGVtX3R5cGU6IG1vbml0b3Iud29ya2Zsb3dfdHlwZSB8fCBtb25pdG9yLm1vbml0b3JfdHlwZSxcbiAgICAgICAgICBpZCxcbiAgICAgICAgICB2ZXJzaW9uLFxuICAgICAgICB9O1xuICAgICAgICByZXR1cm4gcmVzLm9rKHtcbiAgICAgICAgICBib2R5OiB7IG9rOiB0cnVlLCByZXNwOiBtb25pdG9yLCBhY3RpdmVDb3VudCwgZGF5Q291bnQsIHZlcnNpb24sIGlmU2VxTm8sIGlmUHJpbWFyeVRlcm0gfSxcbiAgICAgICAgfSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICByZXR1cm4gcmVzLm9rKHtcbiAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICBvazogZmFsc2UsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICBjb25zb2xlLmVycm9yKCdBbGVydGluZyAtIE1vbml0b3JTZXJ2aWNlIC0gZ2V0TW9uaXRvcjonLCBlcnIpO1xuICAgICAgcmV0dXJuIHJlcy5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogZmFsc2UsXG4gICAgICAgICAgcmVzcDogZXJyLm1lc3NhZ2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG4gIH07XG5cbiAgZ2V0V29ya2Zsb3cgPSBhc3luYyAoY29udGV4dCwgcmVxLCByZXMpID0+IHtcbiAgICB0cnkge1xuICAgICAgY29uc3QgeyBpZCB9ID0gcmVxLnBhcmFtcztcbiAgICAgIGNvbnN0IHBhcmFtcyA9IHsgbW9uaXRvcklkOiBpZCB9O1xuICAgICAgY29uc3QgY2xpZW50ID0gdGhpcy5nZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZShjb250ZXh0LCByZXEpO1xuICAgICAgY29uc3QgZ2V0UmVzcG9uc2UgPSBhd2FpdCBjbGllbnQoJ2FsZXJ0aW5nLmdldFdvcmtmbG93JywgcGFyYW1zKTtcbiAgICAgIGxldCB3b3JrZmxvdyA9IF8uZ2V0KGdldFJlc3BvbnNlLCAnd29ya2Zsb3cnLCBudWxsKTtcbiAgICAgIGNvbnN0IHZlcnNpb24gPSBfLmdldChnZXRSZXNwb25zZSwgJ192ZXJzaW9uJywgbnVsbCk7XG4gICAgICBjb25zdCBpZlNlcU5vID0gXy5nZXQoZ2V0UmVzcG9uc2UsICdfc2VxX25vJywgbnVsbCk7XG4gICAgICBjb25zdCBpZlByaW1hcnlUZXJtID0gXy5nZXQoZ2V0UmVzcG9uc2UsICdfcHJpbWFyeV90ZXJtJywgbnVsbCk7XG4gICAgICB3b3JrZmxvdy5tb25pdG9yX3R5cGUgPSB3b3JrZmxvdy53b3JrZmxvd190eXBlO1xuICAgICAgd29ya2Zsb3cgPSB7XG4gICAgICAgIC4uLndvcmtmbG93LFxuICAgICAgICBpdGVtX3R5cGU6IHdvcmtmbG93LndvcmtmbG93X3R5cGUsXG4gICAgICAgIGlkLFxuICAgICAgICB2ZXJzaW9uLFxuICAgICAgfTtcblxuICAgICAgcmV0dXJuIHJlcy5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICByZXNwOiB3b3JrZmxvdyxcbiAgICAgICAgICBhY3RpdmVDb3VudDogMCxcbiAgICAgICAgICBkYXlDb3VudDogMCxcbiAgICAgICAgICB2ZXJzaW9uLFxuICAgICAgICAgIGlmU2VxTm8sXG4gICAgICAgICAgaWZQcmltYXJ5VGVybSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgY29uc29sZS5lcnJvcignQWxlcnRpbmcgLSBNb25pdG9yU2VydmljZSAtIGdldFdvcmtmbG93OicsIGVycik7XG4gICAgICByZXR1cm4gcmVzLm9rKHtcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIG9rOiBmYWxzZSxcbiAgICAgICAgICByZXNwOiBlcnIubWVzc2FnZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICB1cGRhdGVNb25pdG9yID0gYXN5bmMgKGNvbnRleHQsIHJlcSwgcmVzKSA9PiB7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IHsgaWQgfSA9IHJlcS5wYXJhbXM7XG4gICAgICBjb25zdCBwYXJhbXMgPSB7IG1vbml0b3JJZDogaWQsIGJvZHk6IHJlcS5ib2R5LCByZWZyZXNoOiAnd2FpdF9mb3InIH07XG4gICAgICBjb25zdCB7IHR5cGUgfSA9IHJlcS5ib2R5O1xuXG4gICAgICAvLyBUT0RPIERSQUZUOiBBcmUgd2Ugc3VyZSB3ZSBuZWVkIHRvIGluY2x1ZGUgaWZTZXFObyBhbmQgaWZQcmltYXJ5VGVybSBmcm9tIHRoZSBVSSBzaWRlIHdoZW4gdXBkYXRpbmcgbW9uaXRvcnM/XG4gICAgICBjb25zdCB7IGlmU2VxTm8sIGlmUHJpbWFyeVRlcm0gfSA9IHJlcS5xdWVyeTtcbiAgICAgIGlmIChpZlNlcU5vICYmIGlmUHJpbWFyeVRlcm0pIHtcbiAgICAgICAgcGFyYW1zLmlmX3NlcV9ubyA9IGlmU2VxTm87XG4gICAgICAgIHBhcmFtcy5pZl9wcmltYXJ5X3Rlcm0gPSBpZlByaW1hcnlUZXJtO1xuICAgICAgfVxuXG4gICAgICBjb25zdCBjbGllbnQgPSB0aGlzLmdldENsaWVudEJhc2VkT25EYXRhU291cmNlKGNvbnRleHQsIHJlcSk7XG4gICAgICBjb25zdCB1cGRhdGVSZXNwb25zZSA9IGF3YWl0IGNsaWVudChcbiAgICAgICAgYGFsZXJ0aW5nLiR7dHlwZSA9PT0gJ3dvcmtmbG93JyA/ICd1cGRhdGVXb3JrZmxvdycgOiAndXBkYXRlTW9uaXRvcid9YCxcbiAgICAgICAgcGFyYW1zXG4gICAgICApO1xuICAgICAgY29uc3QgeyBfdmVyc2lvbiwgX2lkIH0gPSB1cGRhdGVSZXNwb25zZTtcbiAgICAgIHJldHVybiByZXMub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IHRydWUsXG4gICAgICAgICAgdmVyc2lvbjogX3ZlcnNpb24sXG4gICAgICAgICAgaWQ6IF9pZCxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgY29uc29sZS5lcnJvcignQWxlcnRpbmcgLSBNb25pdG9yU2VydmljZSAtIHVwZGF0ZU1vbml0b3I6JywgZXJyKTtcbiAgICAgIHJldHVybiByZXMub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IGZhbHNlLFxuICAgICAgICAgIHJlc3A6IGVyci5tZXNzYWdlLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfVxuICB9O1xuXG4gIGdldE1vbml0b3JzID0gYXN5bmMgKGNvbnRleHQsIHJlcSwgcmVzKSA9PiB7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IHsgZnJvbSwgc2l6ZSwgc2VhcmNoLCBzb3J0RGlyZWN0aW9uLCBzb3J0RmllbGQsIHN0YXRlLCBtb25pdG9ySWRzIH0gPSByZXEucXVlcnk7XG5cbiAgICAgIGxldCBtdXN0ID0geyBtYXRjaF9hbGw6IHt9IH07XG4gICAgICBpZiAoc2VhcmNoLnRyaW0oKSkge1xuICAgICAgICAvLyBUaGlzIGlzIGFuIGV4cGVuc2l2ZSB3aWxkY2FyZCBxdWVyeSB0byBtYXRjaCBtb25pdG9yIG5hbWVzIHN1Y2ggYXM6IFwiVGhpcyBpcyBhIGxvbmcgbW9uaXRvciBuYW1lXCJcbiAgICAgICAgLy8gc2VhcmNoIHF1ZXJ5ID0+IFwibG9uZyBtb25pdFwiXG4gICAgICAgIC8vIFRoaXMgaXMgYWNjZXB0YWJsZSBiZWNhdXNlIHdlIHdpbGwgbmV2ZXIgYWxsb3cgbW9yZSB0aGFuIDEsMDAwIG1vbml0b3JzXG4gICAgICAgIG11c3QgPSB7XG4gICAgICAgICAgcXVlcnlfc3RyaW5nOiB7XG4gICAgICAgICAgICBkZWZhdWx0X2ZpZWxkOiAnbW9uaXRvci5uYW1lJyxcbiAgICAgICAgICAgIGRlZmF1bHRfb3BlcmF0b3I6ICdBTkQnLFxuICAgICAgICAgICAgcXVlcnk6IGAqJHtzZWFyY2gudHJpbSgpLnNwbGl0KCcgJykuam9pbignKiAqJyl9KmAsXG4gICAgICAgICAgfSxcbiAgICAgICAgfTtcbiAgICAgIH1cblxuICAgICAgY29uc3Qgc2hvdWxkID0gW107XG4gICAgICBjb25zdCBtdXN0TGlzdCA9IFttdXN0XTtcbiAgICAgIGlmIChtb25pdG9ySWRzICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgbXVzdExpc3QucHVzaCh7XG4gICAgICAgICAgdGVybXM6IHtcbiAgICAgICAgICAgIF9pZDogQXJyYXkuaXNBcnJheShtb25pdG9ySWRzKSA/IG1vbml0b3JJZHMgOiBbbW9uaXRvcklkc10sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9IGVsc2UgaWYgKG1vbml0b3JJZHMgPT09ICdlbXB0eScpIHtcbiAgICAgICAgbXVzdExpc3QucHVzaCh7XG4gICAgICAgICAgdGVybXM6IHtcbiAgICAgICAgICAgIF9pZDogW10sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9XG5cbiAgICAgIGlmIChzdGF0ZSAhPT0gJ2FsbCcpIHtcbiAgICAgICAgY29uc3QgZW5hYmxlZCA9IHN0YXRlID09PSAnZW5hYmxlZCc7XG4gICAgICAgIHNob3VsZC5wdXNoKHsgdGVybTogeyAnbW9uaXRvci5lbmFibGVkJzogZW5hYmxlZCB9IH0pO1xuICAgICAgICBzaG91bGQucHVzaCh7IHRlcm06IHsgJ3dvcmtmbG93LmVuYWJsZWQnOiBlbmFibGVkIH0gfSk7XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IG1vbml0b3JTb3J0cyA9IHsgbmFtZTogJ21vbml0b3IubmFtZS5rZXl3b3JkJyB9O1xuICAgICAgY29uc3QgbW9uaXRvclNvcnRQYWdlRGF0YSA9IHsgc2l6ZTogMTAwMCB9O1xuICAgICAgaWYgKG1vbml0b3JTb3J0c1tzb3J0RmllbGRdKSB7XG4gICAgICAgIG1vbml0b3JTb3J0UGFnZURhdGEuc29ydCA9IFt7IFttb25pdG9yU29ydHNbc29ydEZpZWxkXV06IHNvcnREaXJlY3Rpb24gfV07XG4gICAgICAgIG1vbml0b3JTb3J0UGFnZURhdGEuc2l6ZSA9IF8uZGVmYXVsdFRvKHNpemUsIDEwMDApO1xuICAgICAgICBtb25pdG9yU29ydFBhZ2VEYXRhLmZyb20gPSBfLmRlZmF1bHRUbyhmcm9tLCAwKTtcbiAgICAgIH1cblxuICAgICAgY29uc3QgcGFyYW1zID0ge1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgc2VxX25vX3ByaW1hcnlfdGVybTogdHJ1ZSxcbiAgICAgICAgICB2ZXJzaW9uOiB0cnVlLFxuICAgICAgICAgIC4uLm1vbml0b3JTb3J0UGFnZURhdGEsXG4gICAgICAgICAgcXVlcnk6IHtcbiAgICAgICAgICAgIGJvb2w6IHtcbiAgICAgICAgICAgICAgc2hvdWxkLFxuICAgICAgICAgICAgICBtaW5pbXVtX3Nob3VsZF9tYXRjaDogc3RhdGUgIT09ICdhbGwnID8gMSA6IDAsXG4gICAgICAgICAgICAgIG11c3Q6IG11c3RMaXN0LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGFnZ3JlZ2F0aW9uczoge1xuICAgICAgICAgICAgYXNzb2NpYXRlZF9jb21wb3NpdGVfbW9uaXRvcnM6IHtcbiAgICAgICAgICAgICAgbmVzdGVkOiB7XG4gICAgICAgICAgICAgICAgcGF0aDogJ3dvcmtmbG93LmlucHV0cy5jb21wb3NpdGVfaW5wdXQuc2VxdWVuY2UuZGVsZWdhdGVzJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgYWdnczoge1xuICAgICAgICAgICAgICAgIG1vbml0b3JfaWRzOiB7XG4gICAgICAgICAgICAgICAgICB0ZXJtczoge1xuICAgICAgICAgICAgICAgICAgICBmaWVsZDogJ3dvcmtmbG93LmlucHV0cy5jb21wb3NpdGVfaW5wdXQuc2VxdWVuY2UuZGVsZWdhdGVzLm1vbml0b3JfaWQnLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfTtcblxuICAgICAgY29uc3QgY2xpZW50ID0gdGhpcy5nZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZShjb250ZXh0LCByZXEpO1xuICAgICAgY29uc3QgZ2V0UmVzcG9uc2UgPSBhd2FpdCBjbGllbnQoJ2FsZXJ0aW5nLmdldE1vbml0b3JzJywgcGFyYW1zKTtcblxuICAgICAgY29uc3QgdG90YWxNb25pdG9ycyA9IF8uZ2V0KGdldFJlc3BvbnNlLCAnaGl0cy50b3RhbC52YWx1ZScsIDApO1xuICAgICAgY29uc3QgbW9uaXRvcktleVZhbHVlVHVwbGVzID0gXy5nZXQoZ2V0UmVzcG9uc2UsICdoaXRzLmhpdHMnLCBbXSkubWFwKChyZXN1bHQpID0+IHtcbiAgICAgICAgY29uc3Qge1xuICAgICAgICAgIF9pZDogaWQsXG4gICAgICAgICAgX3ZlcnNpb246IHZlcnNpb24sXG4gICAgICAgICAgX3NlcV9ubzogaWZTZXFObyxcbiAgICAgICAgICBfcHJpbWFyeV90ZXJtOiBpZlByaW1hcnlUZXJtLFxuICAgICAgICAgIF9zb3VyY2UsXG4gICAgICAgIH0gPSByZXN1bHQ7XG4gICAgICAgIGNvbnN0IG1vbml0b3IgPSBfc291cmNlLm1vbml0b3IgPyBfc291cmNlLm1vbml0b3IgOiBfc291cmNlO1xuICAgICAgICBtb25pdG9yWydpdGVtX3R5cGUnXSA9IG1vbml0b3Iud29ya2Zsb3dfdHlwZSB8fCBtb25pdG9yLm1vbml0b3JfdHlwZTtcbiAgICAgICAgY29uc3QgeyBuYW1lLCBlbmFibGVkLCBpdGVtX3R5cGUgfSA9IG1vbml0b3I7XG4gICAgICAgIHJldHVybiBbaWQsIHsgaWQsIHZlcnNpb24sIGlmU2VxTm8sIGlmUHJpbWFyeVRlcm0sIG5hbWUsIGVuYWJsZWQsIGl0ZW1fdHlwZSwgbW9uaXRvciB9XTtcbiAgICAgIH0sIHt9KTtcbiAgICAgIGNvbnN0IG1vbml0b3JNYXAgPSBuZXcgTWFwKG1vbml0b3JLZXlWYWx1ZVR1cGxlcyk7XG4gICAgICBjb25zdCBhc3NvY2lhdGVkQ29tcG9zaXRlTW9uaXRvckNvdW50TWFwID0ge307XG4gICAgICBfLmdldChcbiAgICAgICAgZ2V0UmVzcG9uc2UsXG4gICAgICAgICdhZ2dyZWdhdGlvbnMuYXNzb2NpYXRlZF9jb21wb3NpdGVfbW9uaXRvcnMubW9uaXRvcl9pZHMuYnVja2V0cycsXG4gICAgICAgIFtdXG4gICAgICApLmZvckVhY2goKHsga2V5LCBkb2NfY291bnQgfSkgPT4ge1xuICAgICAgICBhc3NvY2lhdGVkQ29tcG9zaXRlTW9uaXRvckNvdW50TWFwW2tleV0gPSBkb2NfY291bnQ7XG4gICAgICB9KTtcbiAgICAgIGNvbnN0IG1vbml0b3JJZHNPdXRwdXQgPSBbLi4ubW9uaXRvck1hcC5rZXlzKCldO1xuXG4gICAgICBjb25zdCBhZ2dzT3JkZXJEYXRhID0ge307XG4gICAgICBjb25zdCBhZ2dzU29ydHMgPSB7XG4gICAgICAgIGFjdGl2ZTogJ2FjdGl2ZScsXG4gICAgICAgIGFja25vd2xlZGdlZDogJ2Fja25vd2xlZGdlZCcsXG4gICAgICAgIGVycm9yczogJ2Vycm9ycycsXG4gICAgICAgIGlnbm9yZWQ6ICdpZ25vcmVkJyxcbiAgICAgICAgbGFzdE5vdGlmaWNhdGlvblRpbWU6ICdsYXN0X25vdGlmaWNhdGlvbl90aW1lJyxcbiAgICAgIH07XG4gICAgICBpZiAoYWdnc1NvcnRzW3NvcnRGaWVsZF0pIHtcbiAgICAgICAgYWdnc09yZGVyRGF0YS5vcmRlciA9IHsgW2FnZ3NTb3J0c1tzb3J0RmllbGRdXTogc29ydERpcmVjdGlvbiB9O1xuICAgICAgfVxuICAgICAgY29uc3QgYWdnc1BhcmFtcyA9IHtcbiAgICAgICAgaW5kZXg6IElOREVYLkFMTF9BTEVSVFMsXG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBzaXplOiAwLFxuICAgICAgICAgIHF1ZXJ5OiB7IHRlcm1zOiB7IG1vbml0b3JfaWQ6IG1vbml0b3JJZHNPdXRwdXQgfSB9LFxuICAgICAgICAgIGFnZ3JlZ2F0aW9uczoge1xuICAgICAgICAgICAgdW5pcV9tb25pdG9yX2lkczoge1xuICAgICAgICAgICAgICB0ZXJtczoge1xuICAgICAgICAgICAgICAgIGZpZWxkOiAnbW9uaXRvcl9pZCcsXG4gICAgICAgICAgICAgICAgLi4uYWdnc09yZGVyRGF0YSxcbiAgICAgICAgICAgICAgICBzaXplOiBmcm9tICsgc2l6ZSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgYWdncmVnYXRpb25zOiB7XG4gICAgICAgICAgICAgICAgYWN0aXZlOiB7IGZpbHRlcjogeyB0ZXJtOiB7IHN0YXRlOiAnQUNUSVZFJyB9IH0gfSxcbiAgICAgICAgICAgICAgICBhY2tub3dsZWRnZWQ6IHsgZmlsdGVyOiB7IHRlcm06IHsgc3RhdGU6ICdBQ0tOT1dMRURHRUQnIH0gfSB9LFxuICAgICAgICAgICAgICAgIGVycm9yczogeyBmaWx0ZXI6IHsgdGVybTogeyBzdGF0ZTogJ0VSUk9SJyB9IH0gfSxcbiAgICAgICAgICAgICAgICBpZ25vcmVkOiB7XG4gICAgICAgICAgICAgICAgICBmaWx0ZXI6IHtcbiAgICAgICAgICAgICAgICAgICAgYm9vbDoge1xuICAgICAgICAgICAgICAgICAgICAgIGZpbHRlcjogeyB0ZXJtOiB7IHN0YXRlOiAnQ09NUExFVEVEJyB9IH0sXG4gICAgICAgICAgICAgICAgICAgICAgbXVzdF9ub3Q6IHsgZXhpc3RzOiB7IGZpZWxkOiAnYWNrbm93bGVkZ2VkX3RpbWUnIH0gfSxcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBsYXN0X25vdGlmaWNhdGlvbl90aW1lOiB7IG1heDogeyBmaWVsZDogJ2xhc3Rfbm90aWZpY2F0aW9uX3RpbWUnIH0gfSxcbiAgICAgICAgICAgICAgICBsYXRlc3RfYWxlcnQ6IHtcbiAgICAgICAgICAgICAgICAgIHRvcF9oaXRzOiB7XG4gICAgICAgICAgICAgICAgICAgIHNpemU6IDEsXG4gICAgICAgICAgICAgICAgICAgIHNvcnQ6IFt7IHN0YXJ0X3RpbWU6IHsgb3JkZXI6ICdkZXNjJyB9IH1dLFxuICAgICAgICAgICAgICAgICAgICBfc291cmNlOiB7XG4gICAgICAgICAgICAgICAgICAgICAgaW5jbHVkZXM6IFsnbGFzdF9ub3RpZmljYXRpb25fdGltZScsICd0cmlnZ2VyX25hbWUnXSxcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH07XG5cbiAgICAgIGNvbnN0IGVzQWdnc1Jlc3BvbnNlID0gYXdhaXQgY2xpZW50KCdhbGVydGluZy5nZXRNb25pdG9ycycsIGFnZ3NQYXJhbXMpO1xuICAgICAgY29uc3QgYnVja2V0cyA9IF8uZ2V0KGVzQWdnc1Jlc3BvbnNlLCAnYWdncmVnYXRpb25zLnVuaXFfbW9uaXRvcl9pZHMuYnVja2V0cycsIFtdKS5tYXAoXG4gICAgICAgIChidWNrZXQpID0+IHtcbiAgICAgICAgICBjb25zdCB7XG4gICAgICAgICAgICBrZXk6IGlkLFxuICAgICAgICAgICAgbGFzdF9ub3RpZmljYXRpb25fdGltZTogeyB2YWx1ZTogbGFzdE5vdGlmaWNhdGlvblRpbWUgfSxcbiAgICAgICAgICAgIGlnbm9yZWQ6IHsgZG9jX2NvdW50OiBpZ25vcmVkIH0sXG4gICAgICAgICAgICBhY2tub3dsZWRnZWQ6IHsgZG9jX2NvdW50OiBhY2tub3dsZWRnZWQgfSxcbiAgICAgICAgICAgIGFjdGl2ZTogeyBkb2NfY291bnQ6IGFjdGl2ZSB9LFxuICAgICAgICAgICAgZXJyb3JzOiB7IGRvY19jb3VudDogZXJyb3JzIH0sXG4gICAgICAgICAgICBsYXRlc3RfYWxlcnQ6IHtcbiAgICAgICAgICAgICAgaGl0czoge1xuICAgICAgICAgICAgICAgIGhpdHM6IFtcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgX3NvdXJjZTogeyB0cmlnZ2VyX25hbWU6IGxhdGVzdEFsZXJ0IH0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0gPSBidWNrZXQ7XG4gICAgICAgICAgY29uc3QgbW9uaXRvciA9IG1vbml0b3JNYXAuZ2V0KGlkKTtcbiAgICAgICAgICBtb25pdG9yTWFwLmRlbGV0ZShpZCk7XG4gICAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIC4uLm1vbml0b3IsXG4gICAgICAgICAgICBpZCxcbiAgICAgICAgICAgIGxhc3ROb3RpZmljYXRpb25UaW1lLFxuICAgICAgICAgICAgaWdub3JlZCxcbiAgICAgICAgICAgIGxhdGVzdEFsZXJ0LFxuICAgICAgICAgICAgYWNrbm93bGVkZ2VkLFxuICAgICAgICAgICAgYWN0aXZlLFxuICAgICAgICAgICAgZXJyb3JzLFxuICAgICAgICAgICAgY3VycmVudFRpbWU6IERhdGUubm93KCksXG4gICAgICAgICAgICBhc3NvY2lhdGVkQ29tcG9zaXRlTW9uaXRvckNudDogYXNzb2NpYXRlZENvbXBvc2l0ZU1vbml0b3JDb3VudE1hcFtpZF0gfHwgMCxcbiAgICAgICAgICB9O1xuICAgICAgICB9XG4gICAgICApO1xuXG4gICAgICBjb25zdCB1bnVzZWRNb25pdG9ycyA9IFsuLi5tb25pdG9yTWFwLnZhbHVlcygpXS5tYXAoKG1vbml0b3IpID0+ICh7XG4gICAgICAgIC4uLm1vbml0b3IsXG4gICAgICAgIGxhc3ROb3RpZmljYXRpb25UaW1lOiBudWxsLFxuICAgICAgICBpZ25vcmVkOiAwLFxuICAgICAgICBhY3RpdmU6IDAsXG4gICAgICAgIGFja25vd2xlZGdlZDogMCxcbiAgICAgICAgZXJyb3JzOiAwLFxuICAgICAgICBsYXRlc3RBbGVydDogJy0tJyxcbiAgICAgICAgY3VycmVudFRpbWU6IERhdGUubm93KCksXG4gICAgICAgIGFzc29jaWF0ZWRDb21wb3NpdGVNb25pdG9yQ250OiBhc3NvY2lhdGVkQ29tcG9zaXRlTW9uaXRvckNvdW50TWFwW21vbml0b3IuaWRdIHx8IDAsXG4gICAgICB9KSk7XG5cbiAgICAgIGxldCByZXN1bHRzID0gXy5vcmRlckJ5KGJ1Y2tldHMuY29uY2F0KHVudXNlZE1vbml0b3JzKSwgW3NvcnRGaWVsZF0sIFtzb3J0RGlyZWN0aW9uXSk7XG4gICAgICAvLyBJZiB3ZSBzb3J0ZWQgb24gbW9uaXRvciBuYW1lIHRoZW4gd2UgYWxyZWFkeSBhcHBsaWVkIGZyb20vc2l6ZSB0byB0aGUgZmlyc3QgcXVlcnkgdG8gbGltaXQgd2hhdCB3ZSdyZSBhZ2dyZWdhdGluZyBvdmVyXG4gICAgICAvLyBUaGVyZWZvcmUgd2UgZG8gbm90IG5lZWQgdG8gYXBwbHkgZnJvbS9zaXplIHRvIHRoaXMgcmVzdWx0IHNldFxuICAgICAgLy8gSWYgd2Ugc29ydGVkIG9uIGFnZ3JlZ2F0aW9ucywgdGhlbiB0aGlzIGlzIG91ciBpbiBtZW1vcnkgcGFnaW5hdGlvblxuICAgICAgaWYgKCFtb25pdG9yU29ydHNbc29ydEZpZWxkXSkge1xuICAgICAgICByZXN1bHRzID0gcmVzdWx0cy5zbGljZShmcm9tLCBmcm9tICsgc2l6ZSk7XG4gICAgICB9XG5cbiAgICAgIHJldHVybiByZXMub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IHRydWUsXG4gICAgICAgICAgbW9uaXRvcnM6IHJlc3VsdHMsXG4gICAgICAgICAgdG90YWxNb25pdG9ycyxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgY29uc29sZS5lcnJvcignQWxlcnRpbmcgLSBNb25pdG9yU2VydmljZSAtIGdldE1vbml0b3JzJywgZXJyKTtcbiAgICAgIGlmIChpc0luZGV4Tm90Rm91bmRFcnJvcihlcnIpKSB7XG4gICAgICAgIHJldHVybiByZXMub2soe1xuICAgICAgICAgIGJvZHk6IHsgb2s6IGZhbHNlLCByZXNwOiB7IHRvdGFsTW9uaXRvcnM6IDAsIG1vbml0b3JzOiBbXSB9IH0sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgICAgcmV0dXJuIHJlcy5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogZmFsc2UsXG4gICAgICAgICAgcmVzcDogZXJyLm1lc3NhZ2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG4gIH07XG5cbiAgYWNrbm93bGVkZ2VBbGVydHMgPSBhc3luYyAoY29udGV4dCwgcmVxLCByZXMpID0+IHtcbiAgICB0cnkge1xuICAgICAgY29uc3QgeyBpZCB9ID0gcmVxLnBhcmFtcztcbiAgICAgIGNvbnN0IHBhcmFtcyA9IHtcbiAgICAgICAgbW9uaXRvcklkOiBpZCxcbiAgICAgICAgYm9keTogcmVxLmJvZHksXG4gICAgICB9O1xuICAgICAgY29uc3QgY2xpZW50ID0gdGhpcy5nZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZShjb250ZXh0LCByZXEpO1xuICAgICAgY29uc3QgYWNrbm93bGVkZ2VSZXNwb25zZSA9IGF3YWl0IGNsaWVudCgnYWxlcnRpbmcuYWNrbm93bGVkZ2VBbGVydHMnLCBwYXJhbXMpO1xuICAgICAgcmV0dXJuIHJlcy5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogIWFja25vd2xlZGdlUmVzcG9uc2UuZmFpbGVkLmxlbmd0aCxcbiAgICAgICAgICByZXNwOiBhY2tub3dsZWRnZVJlc3BvbnNlLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICBjb25zb2xlLmVycm9yKCdBbGVydGluZyAtIE1vbml0b3JTZXJ2aWNlIC0gYWNrbm93bGVkZ2VBbGVydHM6JywgZXJyKTtcbiAgICAgIHJldHVybiByZXMub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IGZhbHNlLFxuICAgICAgICAgIHJlc3A6IGVyci5tZXNzYWdlLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfVxuICB9O1xuXG4gIGFja25vd2xlZGdlQ2hhaW5lZEFsZXJ0cyA9IGFzeW5jIChjb250ZXh0LCByZXEsIHJlcykgPT4ge1xuICAgIHRyeSB7XG4gICAgICBjb25zdCB7IGlkIH0gPSByZXEucGFyYW1zO1xuICAgICAgY29uc3QgcGFyYW1zID0ge1xuICAgICAgICB3b3JrZmxvd0lkOiBpZCxcbiAgICAgICAgYm9keTogcmVxLmJvZHksXG4gICAgICB9O1xuICAgICAgY29uc3QgY2xpZW50ID0gdGhpcy5nZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZShcbiAgICAgICAgY29udGV4dCxcbiAgICAgICAgdGhpcy5kYXRhU291cmNlRW5hYmxlZCxcbiAgICAgICAgcmVxLFxuICAgICAgICB0aGlzLmVzRHJpdmVyXG4gICAgICApO1xuICAgICAgY29uc3QgYWNrbm93bGVkZ2VSZXNwb25zZSA9IGF3YWl0IGNsaWVudCgnYWxlcnRpbmcuYWNrbm93bGVkZ2VDaGFpbmVkQWxlcnRzJywgcGFyYW1zKTtcbiAgICAgIHJldHVybiByZXMub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6ICFhY2tub3dsZWRnZVJlc3BvbnNlLmZhaWxlZC5sZW5ndGgsXG4gICAgICAgICAgcmVzcDogYWNrbm93bGVkZ2VSZXNwb25zZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgY29uc29sZS5lcnJvcignQWxlcnRpbmcgLSBNb25pdG9yU2VydmljZSAtIGFja25vd2xlZGdlQ2hhaW5lZEFsZXJ0czonLCBlcnIpO1xuICAgICAgcmV0dXJuIHJlcy5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogZmFsc2UsXG4gICAgICAgICAgcmVzcDogZXJyLm1lc3NhZ2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG4gIH07XG5cbiAgZXhlY3V0ZU1vbml0b3IgPSBhc3luYyAoY29udGV4dCwgcmVxLCByZXMpID0+IHtcbiAgICB0cnkge1xuICAgICAgY29uc3QgeyBkcnlydW4gPSAndHJ1ZScgfSA9IHJlcS5xdWVyeTtcbiAgICAgIGNvbnN0IHBhcmFtcyA9IHtcbiAgICAgICAgYm9keTogcmVxLmJvZHksXG4gICAgICAgIGRyeXJ1bixcbiAgICAgIH07XG4gICAgICBjb25zdCBjbGllbnQgPSB0aGlzLmdldENsaWVudEJhc2VkT25EYXRhU291cmNlKGNvbnRleHQsIHJlcSk7XG4gICAgICBjb25zdCBleGVjdXRlUmVzcG9uc2UgPSBhd2FpdCBjbGllbnQoJ2FsZXJ0aW5nLmV4ZWN1dGVNb25pdG9yJywgcGFyYW1zKTtcbiAgICAgIHJldHVybiByZXMub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IHRydWUsXG4gICAgICAgICAgcmVzcDogZXhlY3V0ZVJlc3BvbnNlLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICBjb25zb2xlLmVycm9yKCdBbGVydGluZyAtIE1vbml0b3JTZXJ2aWNlIC0gZXhlY3V0ZU1vbml0b3I6JywgZXJyKTtcbiAgICAgIHJldHVybiByZXMub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IGZhbHNlLFxuICAgICAgICAgIHJlc3A6IGVyci5tZXNzYWdlLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfVxuICB9O1xuXG4gIC8vVE9ETzogVGhpcyBpcyB0ZW1wb3JhcmlseSBhIHBhc3MgdGhyb3VnaCBjYWxsIHdoaWNoIG5lZWRzIHRvIGJlIGRlcHJlY2F0ZWRcbiAgc2VhcmNoTW9uaXRvcnMgPSBhc3luYyAoY29udGV4dCwgcmVxLCByZXMpID0+IHtcbiAgICB0cnkge1xuICAgICAgY29uc3QgeyBxdWVyeSwgaW5kZXgsIHNpemUgfSA9IHJlcS5ib2R5O1xuICAgICAgY29uc3QgcGFyYW1zID0geyBpbmRleCwgc2l6ZSwgYm9keTogcXVlcnkgfTtcblxuICAgICAgY29uc3QgY2xpZW50ID0gdGhpcy5nZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZShjb250ZXh0LCByZXEpO1xuICAgICAgY29uc3QgcmVzdWx0cyA9IGF3YWl0IGNsaWVudCgnYWxlcnRpbmcuZ2V0TW9uaXRvcnMnLCBwYXJhbXMpO1xuICAgICAgcmV0dXJuIHJlcy5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICByZXNwOiByZXN1bHRzLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICBjb25zb2xlLmVycm9yKCdBbGVydGluZyAtIE1vbml0b3JTZXJ2aWNlIC0gc2VhcmNoTW9uaXRvcjonLCBlcnIpO1xuICAgICAgcmV0dXJuIHJlcy5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogZmFsc2UsXG4gICAgICAgICAgcmVzcDogZXJyLm1lc3NhZ2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG4gIH07XG59XG4iXSwibWFwcGluZ3MiOiI7Ozs7OztBQUtBLElBQUFBLE9BQUEsR0FBQUMsc0JBQUEsQ0FBQUMsT0FBQTtBQUVBLElBQUFDLFVBQUEsR0FBQUQsT0FBQTtBQUNBLElBQUFFLFFBQUEsR0FBQUYsT0FBQTtBQUNBLElBQUFHLHdCQUFBLEdBQUFILE9BQUE7QUFBb0UsU0FBQUQsdUJBQUFLLEdBQUEsV0FBQUEsR0FBQSxJQUFBQSxHQUFBLENBQUFDLFVBQUEsR0FBQUQsR0FBQSxLQUFBRSxPQUFBLEVBQUFGLEdBQUE7QUFBQSxTQUFBRyxnQkFBQUgsR0FBQSxFQUFBSSxHQUFBLEVBQUFDLEtBQUEsSUFBQUQsR0FBQSxHQUFBRSxjQUFBLENBQUFGLEdBQUEsT0FBQUEsR0FBQSxJQUFBSixHQUFBLElBQUFPLE1BQUEsQ0FBQUMsY0FBQSxDQUFBUixHQUFBLEVBQUFJLEdBQUEsSUFBQUMsS0FBQSxFQUFBQSxLQUFBLEVBQUFJLFVBQUEsUUFBQUMsWUFBQSxRQUFBQyxRQUFBLG9CQUFBWCxHQUFBLENBQUFJLEdBQUEsSUFBQUMsS0FBQSxXQUFBTCxHQUFBO0FBQUEsU0FBQU0sZUFBQU0sR0FBQSxRQUFBUixHQUFBLEdBQUFTLFlBQUEsQ0FBQUQsR0FBQSwyQkFBQVIsR0FBQSxnQkFBQUEsR0FBQSxHQUFBVSxNQUFBLENBQUFWLEdBQUE7QUFBQSxTQUFBUyxhQUFBRSxLQUFBLEVBQUFDLElBQUEsZUFBQUQsS0FBQSxpQkFBQUEsS0FBQSxrQkFBQUEsS0FBQSxNQUFBRSxJQUFBLEdBQUFGLEtBQUEsQ0FBQUcsTUFBQSxDQUFBQyxXQUFBLE9BQUFGLElBQUEsS0FBQUcsU0FBQSxRQUFBQyxHQUFBLEdBQUFKLElBQUEsQ0FBQUssSUFBQSxDQUFBUCxLQUFBLEVBQUFDLElBQUEsMkJBQUFLLEdBQUEsc0JBQUFBLEdBQUEsWUFBQUUsU0FBQSw0REFBQVAsSUFBQSxnQkFBQUYsTUFBQSxHQUFBVSxNQUFBLEVBQUFULEtBQUEsS0FUcEU7QUFDQTtBQUNBO0FBQ0E7QUFRZSxNQUFNVSxjQUFjLFNBQVNDLGdEQUF1QixDQUFDO0VBQUFDLFlBQUEsR0FBQUMsSUFBQTtJQUFBLFNBQUFBLElBQUE7SUFBQXpCLGVBQUEsd0JBQ2xELE9BQU8wQixPQUFPLEVBQUVDLEdBQUcsRUFBRVQsR0FBRyxLQUFLO01BQzNDLElBQUk7UUFDRixNQUFNVSxNQUFNLEdBQUc7VUFBRUMsSUFBSSxFQUFFRixHQUFHLENBQUNFO1FBQUssQ0FBQztRQUNqQyxNQUFNQyxNQUFNLEdBQUcsSUFBSSxDQUFDQywwQkFBMEIsQ0FBQ0wsT0FBTyxFQUFFQyxHQUFHLENBQUM7UUFDNUQsTUFBTUssY0FBYyxHQUFHLE1BQU1GLE1BQU0sQ0FBQyx3QkFBd0IsRUFBRUYsTUFBTSxDQUFDO1FBQ3JFLE9BQU9WLEdBQUcsQ0FBQ2UsRUFBRSxDQUFDO1VBQ1pKLElBQUksRUFBRTtZQUNKSSxFQUFFLEVBQUUsSUFBSTtZQUNSQyxJQUFJLEVBQUVGO1VBQ1I7UUFDRixDQUFDLENBQUM7TUFDSixDQUFDLENBQUMsT0FBT0csR0FBRyxFQUFFO1FBQ1pDLE9BQU8sQ0FBQ0MsS0FBSyxDQUFDLDRDQUE0QyxFQUFFRixHQUFHLENBQUM7UUFDaEUsT0FBT2pCLEdBQUcsQ0FBQ2UsRUFBRSxDQUFDO1VBQ1pKLElBQUksRUFBRTtZQUNKSSxFQUFFLEVBQUUsS0FBSztZQUNUQyxJQUFJLEVBQUVDLEdBQUcsQ0FBQ0c7VUFDWjtRQUNGLENBQUMsQ0FBQztNQUNKO0lBQ0YsQ0FBQztJQUFBdEMsZUFBQSx5QkFFZ0IsT0FBTzBCLE9BQU8sRUFBRUMsR0FBRyxFQUFFVCxHQUFHLEtBQUs7TUFDNUMsSUFBSTtRQUNGLE1BQU1VLE1BQU0sR0FBRztVQUFFQyxJQUFJLEVBQUVGLEdBQUcsQ0FBQ0U7UUFBSyxDQUFDO1FBQ2pDLE1BQU1DLE1BQU0sR0FBRyxJQUFJLENBQUNDLDBCQUEwQixDQUFDTCxPQUFPLEVBQUVDLEdBQUcsQ0FBQztRQUM1RCxNQUFNSyxjQUFjLEdBQUcsTUFBTUYsTUFBTSxDQUFDLHlCQUF5QixFQUFFRixNQUFNLENBQUM7UUFDdEUsT0FBT1YsR0FBRyxDQUFDZSxFQUFFLENBQUM7VUFDWkosSUFBSSxFQUFFO1lBQ0pJLEVBQUUsRUFBRSxJQUFJO1lBQ1JDLElBQUksRUFBRUY7VUFDUjtRQUNGLENBQUMsQ0FBQztNQUNKLENBQUMsQ0FBQyxPQUFPRyxHQUFHLEVBQUU7UUFDWkMsT0FBTyxDQUFDQyxLQUFLLENBQUMsNkNBQTZDLEVBQUVGLEdBQUcsQ0FBQztRQUNqRSxPQUFPakIsR0FBRyxDQUFDZSxFQUFFLENBQUM7VUFDWkosSUFBSSxFQUFFO1lBQ0pJLEVBQUUsRUFBRSxLQUFLO1lBQ1RDLElBQUksRUFBRUMsR0FBRyxDQUFDRztVQUNaO1FBQ0YsQ0FBQyxDQUFDO01BQ0o7SUFDRixDQUFDO0lBQUF0QyxlQUFBLHdCQUVlLE9BQU8wQixPQUFPLEVBQUVDLEdBQUcsRUFBRVQsR0FBRyxLQUFLO01BQzNDLElBQUk7UUFDRixNQUFNO1VBQUVxQjtRQUFHLENBQUMsR0FBR1osR0FBRyxDQUFDQyxNQUFNO1FBQ3pCLE1BQU1BLE1BQU0sR0FBRztVQUFFWSxTQUFTLEVBQUVEO1FBQUcsQ0FBQztRQUNoQyxNQUFNVCxNQUFNLEdBQUcsSUFBSSxDQUFDQywwQkFBMEIsQ0FBQ0wsT0FBTyxFQUFFQyxHQUFHLENBQUM7UUFDNUQsTUFBTWMsUUFBUSxHQUFHLE1BQU1YLE1BQU0sQ0FBQyx3QkFBd0IsRUFBRUYsTUFBTSxDQUFDO1FBRS9ELE9BQU9WLEdBQUcsQ0FBQ2UsRUFBRSxDQUFDO1VBQ1pKLElBQUksRUFBRTtZQUNKSSxFQUFFLEVBQUVRLFFBQVEsQ0FBQ0MsTUFBTSxLQUFLLFNBQVMsSUFBSUQsUUFBUSxDQUFDQyxNQUFNLEtBQUt6QjtVQUMzRDtRQUNGLENBQUMsQ0FBQztNQUNKLENBQUMsQ0FBQyxPQUFPa0IsR0FBRyxFQUFFO1FBQ1pDLE9BQU8sQ0FBQ0MsS0FBSyxDQUFDLDRDQUE0QyxFQUFFRixHQUFHLENBQUM7UUFDaEUsT0FBT2pCLEdBQUcsQ0FBQ2UsRUFBRSxDQUFDO1VBQ1pKLElBQUksRUFBRTtZQUNKSSxFQUFFLEVBQUUsS0FBSztZQUNUQyxJQUFJLEVBQUVDLEdBQUcsQ0FBQ0c7VUFDWjtRQUNGLENBQUMsQ0FBQztNQUNKO0lBQ0YsQ0FBQztJQUFBdEMsZUFBQSx5QkFFZ0IsT0FBTzBCLE9BQU8sRUFBRUMsR0FBRyxFQUFFVCxHQUFHLEtBQUs7TUFDNUMsSUFBSTtRQUNGLE1BQU07VUFBRXFCO1FBQUcsQ0FBQyxHQUFHWixHQUFHLENBQUNDLE1BQU07UUFDekIsTUFBTUEsTUFBTSxHQUFHO1VBQUVlLFVBQVUsRUFBRUo7UUFBRyxDQUFDO1FBQ2pDLE1BQU1ULE1BQU0sR0FBRyxJQUFJLENBQUNDLDBCQUEwQixDQUFDTCxPQUFPLEVBQUVDLEdBQUcsQ0FBQztRQUM1RCxNQUFNYyxRQUFRLEdBQUcsTUFBTVgsTUFBTSxDQUFDLHlCQUF5QixFQUFFRixNQUFNLENBQUM7UUFFaEUsT0FBT1YsR0FBRyxDQUFDZSxFQUFFLENBQUM7VUFDWkosSUFBSSxFQUFFO1lBQ0pJLEVBQUUsRUFBRVEsUUFBUSxDQUFDQyxNQUFNLEtBQUssU0FBUyxJQUFJRCxRQUFRLENBQUNDLE1BQU0sS0FBS3pCO1VBQzNEO1FBQ0YsQ0FBQyxDQUFDO01BQ0osQ0FBQyxDQUFDLE9BQU9rQixHQUFHLEVBQUU7UUFDWkMsT0FBTyxDQUFDQyxLQUFLLENBQUMsNkNBQTZDLEVBQUVGLEdBQUcsQ0FBQztRQUNqRSxPQUFPakIsR0FBRyxDQUFDZSxFQUFFLENBQUM7VUFDWkosSUFBSSxFQUFFO1lBQ0pJLEVBQUUsRUFBRSxLQUFLO1lBQ1RDLElBQUksRUFBRUMsR0FBRyxDQUFDRztVQUNaO1FBQ0YsQ0FBQyxDQUFDO01BQ0o7SUFDRixDQUFDO0lBQUF0QyxlQUFBLHFCQUVZLE9BQU8wQixPQUFPLEVBQUVDLEdBQUcsRUFBRVQsR0FBRyxLQUFLO01BQ3hDLElBQUk7UUFDRixNQUFNO1VBQUVxQjtRQUFHLENBQUMsR0FBR1osR0FBRyxDQUFDQyxNQUFNO1FBQ3pCLE1BQU1BLE1BQU0sR0FBRztVQUFFWSxTQUFTLEVBQUVEO1FBQUcsQ0FBQztRQUNoQyxNQUFNVCxNQUFNLEdBQUcsSUFBSSxDQUFDQywwQkFBMEIsQ0FBQ0wsT0FBTyxFQUFFQyxHQUFHLENBQUM7UUFDNUQsTUFBTWlCLFdBQVcsR0FBRyxNQUFNZCxNQUFNLENBQUMscUJBQXFCLEVBQUVGLE1BQU0sQ0FBQztRQUMvRCxJQUFJaUIsT0FBTyxHQUFHQyxlQUFDLENBQUNDLEdBQUcsQ0FBQ0gsV0FBVyxFQUFFLFNBQVMsRUFBRSxJQUFJLENBQUM7UUFDakQsTUFBTUksT0FBTyxHQUFHRixlQUFDLENBQUNDLEdBQUcsQ0FBQ0gsV0FBVyxFQUFFLFVBQVUsRUFBRSxJQUFJLENBQUM7UUFDcEQsTUFBTUssT0FBTyxHQUFHSCxlQUFDLENBQUNDLEdBQUcsQ0FBQ0gsV0FBVyxFQUFFLFNBQVMsRUFBRSxJQUFJLENBQUM7UUFDbkQsTUFBTU0sYUFBYSxHQUFHSixlQUFDLENBQUNDLEdBQUcsQ0FBQ0gsV0FBVyxFQUFFLGVBQWUsRUFBRSxJQUFJLENBQUM7UUFDL0QsTUFBTU8sb0JBQW9CLEdBQUdMLGVBQUMsQ0FBQ0MsR0FBRyxDQUFDSCxXQUFXLEVBQUUsc0JBQXNCLEVBQUUsSUFBSSxDQUFDO1FBQzdFLElBQUlDLE9BQU8sRUFBRTtVQUNYLE1BQU1mLE1BQU0sR0FBRyxJQUFJLENBQUNDLDBCQUEwQixDQUFDTCxPQUFPLEVBQUVDLEdBQUcsQ0FBQztVQUM1RCxNQUFNeUIsVUFBVSxHQUFHO1lBQ2pCQyxLQUFLLEVBQUVDLGdCQUFLLENBQUNDLFVBQVU7WUFDdkIxQixJQUFJLEVBQUU7Y0FDSjJCLElBQUksRUFBRSxDQUFDO2NBQ1BDLEtBQUssRUFBRTtnQkFDTEMsSUFBSSxFQUFFO2tCQUNKQyxJQUFJLEVBQUU7b0JBQ0pDLElBQUksRUFBRTtzQkFDSkMsVUFBVSxFQUFFdEI7b0JBQ2Q7a0JBQ0Y7Z0JBQ0Y7Y0FDRixDQUFDO2NBQ0R1QixJQUFJLEVBQUU7Z0JBQ0pDLFlBQVksRUFBRTtrQkFDWkMsS0FBSyxFQUFFO29CQUNMQyxLQUFLLEVBQUU7a0JBQ1Q7Z0JBQ0YsQ0FBQztnQkFDRCxlQUFlLEVBQUU7a0JBQ2ZDLFVBQVUsRUFBRTtvQkFDVkQsS0FBSyxFQUFFLFlBQVk7b0JBQ25CRSxNQUFNLEVBQUUsQ0FBQztzQkFBRUMsSUFBSSxFQUFFO29CQUFZLENBQUM7a0JBQ2hDO2dCQUNGO2NBQ0Y7WUFDRjtVQUNGLENBQUM7VUFDRCxNQUFNQyxjQUFjLEdBQUcsTUFBTXZDLE1BQU0sQ0FBQyxzQkFBc0IsRUFBRXNCLFVBQVUsQ0FBQztVQUN2RSxNQUFNa0IsUUFBUSxHQUFHeEIsZUFBQyxDQUFDQyxHQUFHLENBQUNzQixjQUFjLEVBQUUsZ0RBQWdELEVBQUUsQ0FBQyxDQUFDO1VBQzNGLE1BQU1FLGFBQWEsR0FBR3pCLGVBQUMsQ0FBQ0MsR0FBRyxDQUFDc0IsY0FBYyxFQUFFLG1DQUFtQyxFQUFFLEVBQUUsQ0FBQztVQUNwRixNQUFNRyxXQUFXLEdBQUdELGFBQWEsQ0FBQ0UsTUFBTSxDQUN0QyxDQUFDQyxHQUFHLEVBQUVDLElBQUksS0FBTUEsSUFBSSxDQUFDMUUsR0FBRyxLQUFLLFFBQVEsR0FBRzBFLElBQUksQ0FBQ0MsU0FBUyxHQUFHRixHQUFJLEVBQzdELENBQ0YsQ0FBQztVQUNELElBQUl2QixvQkFBb0IsRUFBRTtZQUN4Qk4sT0FBTyxHQUFHO2NBQ1IsR0FBR0EsT0FBTztjQUNWTSxvQkFBb0I7Y0FDcEIwQiw2QkFBNkIsRUFBRTFCLG9CQUFvQixDQUFDMkI7WUFDdEQsQ0FBQztVQUNIO1VBQ0FqQyxPQUFPLEdBQUc7WUFDUixHQUFHQSxPQUFPO1lBQ1ZrQyxTQUFTLEVBQUVsQyxPQUFPLENBQUNtQyxhQUFhLElBQUluQyxPQUFPLENBQUNvQyxZQUFZO1lBQ3hEMUMsRUFBRTtZQUNGUztVQUNGLENBQUM7VUFDRCxPQUFPOUIsR0FBRyxDQUFDZSxFQUFFLENBQUM7WUFDWkosSUFBSSxFQUFFO2NBQUVJLEVBQUUsRUFBRSxJQUFJO2NBQUVDLElBQUksRUFBRVcsT0FBTztjQUFFMkIsV0FBVztjQUFFRixRQUFRO2NBQUV0QixPQUFPO2NBQUVDLE9BQU87Y0FBRUM7WUFBYztVQUMxRixDQUFDLENBQUM7UUFDSixDQUFDLE1BQU07VUFDTCxPQUFPaEMsR0FBRyxDQUFDZSxFQUFFLENBQUM7WUFDWkosSUFBSSxFQUFFO2NBQ0pJLEVBQUUsRUFBRTtZQUNOO1VBQ0YsQ0FBQyxDQUFDO1FBQ0o7TUFDRixDQUFDLENBQUMsT0FBT0UsR0FBRyxFQUFFO1FBQ1pDLE9BQU8sQ0FBQ0MsS0FBSyxDQUFDLHlDQUF5QyxFQUFFRixHQUFHLENBQUM7UUFDN0QsT0FBT2pCLEdBQUcsQ0FBQ2UsRUFBRSxDQUFDO1VBQ1pKLElBQUksRUFBRTtZQUNKSSxFQUFFLEVBQUUsS0FBSztZQUNUQyxJQUFJLEVBQUVDLEdBQUcsQ0FBQ0c7VUFDWjtRQUNGLENBQUMsQ0FBQztNQUNKO0lBQ0YsQ0FBQztJQUFBdEMsZUFBQSxzQkFFYSxPQUFPMEIsT0FBTyxFQUFFQyxHQUFHLEVBQUVULEdBQUcsS0FBSztNQUN6QyxJQUFJO1FBQ0YsTUFBTTtVQUFFcUI7UUFBRyxDQUFDLEdBQUdaLEdBQUcsQ0FBQ0MsTUFBTTtRQUN6QixNQUFNQSxNQUFNLEdBQUc7VUFBRVksU0FBUyxFQUFFRDtRQUFHLENBQUM7UUFDaEMsTUFBTVQsTUFBTSxHQUFHLElBQUksQ0FBQ0MsMEJBQTBCLENBQUNMLE9BQU8sRUFBRUMsR0FBRyxDQUFDO1FBQzVELE1BQU1pQixXQUFXLEdBQUcsTUFBTWQsTUFBTSxDQUFDLHNCQUFzQixFQUFFRixNQUFNLENBQUM7UUFDaEUsSUFBSXNELFFBQVEsR0FBR3BDLGVBQUMsQ0FBQ0MsR0FBRyxDQUFDSCxXQUFXLEVBQUUsVUFBVSxFQUFFLElBQUksQ0FBQztRQUNuRCxNQUFNSSxPQUFPLEdBQUdGLGVBQUMsQ0FBQ0MsR0FBRyxDQUFDSCxXQUFXLEVBQUUsVUFBVSxFQUFFLElBQUksQ0FBQztRQUNwRCxNQUFNSyxPQUFPLEdBQUdILGVBQUMsQ0FBQ0MsR0FBRyxDQUFDSCxXQUFXLEVBQUUsU0FBUyxFQUFFLElBQUksQ0FBQztRQUNuRCxNQUFNTSxhQUFhLEdBQUdKLGVBQUMsQ0FBQ0MsR0FBRyxDQUFDSCxXQUFXLEVBQUUsZUFBZSxFQUFFLElBQUksQ0FBQztRQUMvRHNDLFFBQVEsQ0FBQ0QsWUFBWSxHQUFHQyxRQUFRLENBQUNGLGFBQWE7UUFDOUNFLFFBQVEsR0FBRztVQUNULEdBQUdBLFFBQVE7VUFDWEgsU0FBUyxFQUFFRyxRQUFRLENBQUNGLGFBQWE7VUFDakN6QyxFQUFFO1VBQ0ZTO1FBQ0YsQ0FBQztRQUVELE9BQU85QixHQUFHLENBQUNlLEVBQUUsQ0FBQztVQUNaSixJQUFJLEVBQUU7WUFDSkksRUFBRSxFQUFFLElBQUk7WUFDUkMsSUFBSSxFQUFFZ0QsUUFBUTtZQUNkVixXQUFXLEVBQUUsQ0FBQztZQUNkRixRQUFRLEVBQUUsQ0FBQztZQUNYdEIsT0FBTztZQUNQQyxPQUFPO1lBQ1BDO1VBQ0Y7UUFDRixDQUFDLENBQUM7TUFDSixDQUFDLENBQUMsT0FBT2YsR0FBRyxFQUFFO1FBQ1pDLE9BQU8sQ0FBQ0MsS0FBSyxDQUFDLDBDQUEwQyxFQUFFRixHQUFHLENBQUM7UUFDOUQsT0FBT2pCLEdBQUcsQ0FBQ2UsRUFBRSxDQUFDO1VBQ1pKLElBQUksRUFBRTtZQUNKSSxFQUFFLEVBQUUsS0FBSztZQUNUQyxJQUFJLEVBQUVDLEdBQUcsQ0FBQ0c7VUFDWjtRQUNGLENBQUMsQ0FBQztNQUNKO0lBQ0YsQ0FBQztJQUFBdEMsZUFBQSx3QkFFZSxPQUFPMEIsT0FBTyxFQUFFQyxHQUFHLEVBQUVULEdBQUcsS0FBSztNQUMzQyxJQUFJO1FBQ0YsTUFBTTtVQUFFcUI7UUFBRyxDQUFDLEdBQUdaLEdBQUcsQ0FBQ0MsTUFBTTtRQUN6QixNQUFNQSxNQUFNLEdBQUc7VUFBRVksU0FBUyxFQUFFRCxFQUFFO1VBQUVWLElBQUksRUFBRUYsR0FBRyxDQUFDRSxJQUFJO1VBQUVzRCxPQUFPLEVBQUU7UUFBVyxDQUFDO1FBQ3JFLE1BQU07VUFBRUM7UUFBSyxDQUFDLEdBQUd6RCxHQUFHLENBQUNFLElBQUk7O1FBRXpCO1FBQ0EsTUFBTTtVQUFFb0IsT0FBTztVQUFFQztRQUFjLENBQUMsR0FBR3ZCLEdBQUcsQ0FBQzhCLEtBQUs7UUFDNUMsSUFBSVIsT0FBTyxJQUFJQyxhQUFhLEVBQUU7VUFDNUJ0QixNQUFNLENBQUN5RCxTQUFTLEdBQUdwQyxPQUFPO1VBQzFCckIsTUFBTSxDQUFDMEQsZUFBZSxHQUFHcEMsYUFBYTtRQUN4QztRQUVBLE1BQU1wQixNQUFNLEdBQUcsSUFBSSxDQUFDQywwQkFBMEIsQ0FBQ0wsT0FBTyxFQUFFQyxHQUFHLENBQUM7UUFDNUQsTUFBTTRELGNBQWMsR0FBRyxNQUFNekQsTUFBTSxDQUNoQyxZQUFXc0QsSUFBSSxLQUFLLFVBQVUsR0FBRyxnQkFBZ0IsR0FBRyxlQUFnQixFQUFDLEVBQ3RFeEQsTUFDRixDQUFDO1FBQ0QsTUFBTTtVQUFFNEQsUUFBUTtVQUFFQztRQUFJLENBQUMsR0FBR0YsY0FBYztRQUN4QyxPQUFPckUsR0FBRyxDQUFDZSxFQUFFLENBQUM7VUFDWkosSUFBSSxFQUFFO1lBQ0pJLEVBQUUsRUFBRSxJQUFJO1lBQ1JlLE9BQU8sRUFBRXdDLFFBQVE7WUFDakJqRCxFQUFFLEVBQUVrRDtVQUNOO1FBQ0YsQ0FBQyxDQUFDO01BQ0osQ0FBQyxDQUFDLE9BQU90RCxHQUFHLEVBQUU7UUFDWkMsT0FBTyxDQUFDQyxLQUFLLENBQUMsNENBQTRDLEVBQUVGLEdBQUcsQ0FBQztRQUNoRSxPQUFPakIsR0FBRyxDQUFDZSxFQUFFLENBQUM7VUFDWkosSUFBSSxFQUFFO1lBQ0pJLEVBQUUsRUFBRSxLQUFLO1lBQ1RDLElBQUksRUFBRUMsR0FBRyxDQUFDRztVQUNaO1FBQ0YsQ0FBQyxDQUFDO01BQ0o7SUFDRixDQUFDO0lBQUF0QyxlQUFBLHNCQUVhLE9BQU8wQixPQUFPLEVBQUVDLEdBQUcsRUFBRVQsR0FBRyxLQUFLO01BQ3pDLElBQUk7UUFDRixNQUFNO1VBQUVrRCxJQUFJO1VBQUVaLElBQUk7VUFBRWtDLE1BQU07VUFBRUMsYUFBYTtVQUFFQyxTQUFTO1VBQUVDLEtBQUs7VUFBRUM7UUFBVyxDQUFDLEdBQUduRSxHQUFHLENBQUM4QixLQUFLO1FBRXJGLElBQUlFLElBQUksR0FBRztVQUFFb0MsU0FBUyxFQUFFLENBQUM7UUFBRSxDQUFDO1FBQzVCLElBQUlMLE1BQU0sQ0FBQ00sSUFBSSxDQUFDLENBQUMsRUFBRTtVQUNqQjtVQUNBO1VBQ0E7VUFDQXJDLElBQUksR0FBRztZQUNMc0MsWUFBWSxFQUFFO2NBQ1pDLGFBQWEsRUFBRSxjQUFjO2NBQzdCQyxnQkFBZ0IsRUFBRSxLQUFLO2NBQ3ZCMUMsS0FBSyxFQUFHLElBQUdpQyxNQUFNLENBQUNNLElBQUksQ0FBQyxDQUFDLENBQUNJLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQ0MsSUFBSSxDQUFDLEtBQUssQ0FBRTtZQUNsRDtVQUNGLENBQUM7UUFDSDtRQUVBLE1BQU1DLE1BQU0sR0FBRyxFQUFFO1FBQ2pCLE1BQU1DLFFBQVEsR0FBRyxDQUFDNUMsSUFBSSxDQUFDO1FBQ3ZCLElBQUltQyxVQUFVLEtBQUs3RSxTQUFTLEVBQUU7VUFDNUJzRixRQUFRLENBQUNDLElBQUksQ0FBQztZQUNaeEMsS0FBSyxFQUFFO2NBQ0x5QixHQUFHLEVBQUVnQixLQUFLLENBQUNDLE9BQU8sQ0FBQ1osVUFBVSxDQUFDLEdBQUdBLFVBQVUsR0FBRyxDQUFDQSxVQUFVO1lBQzNEO1VBQ0YsQ0FBQyxDQUFDO1FBQ0osQ0FBQyxNQUFNLElBQUlBLFVBQVUsS0FBSyxPQUFPLEVBQUU7VUFDakNTLFFBQVEsQ0FBQ0MsSUFBSSxDQUFDO1lBQ1p4QyxLQUFLLEVBQUU7Y0FDTHlCLEdBQUcsRUFBRTtZQUNQO1VBQ0YsQ0FBQyxDQUFDO1FBQ0o7UUFFQSxJQUFJSSxLQUFLLEtBQUssS0FBSyxFQUFFO1VBQ25CLE1BQU1jLE9BQU8sR0FBR2QsS0FBSyxLQUFLLFNBQVM7VUFDbkNTLE1BQU0sQ0FBQ0UsSUFBSSxDQUFDO1lBQUU1QyxJQUFJLEVBQUU7Y0FBRSxpQkFBaUIsRUFBRStDO1lBQVE7VUFBRSxDQUFDLENBQUM7VUFDckRMLE1BQU0sQ0FBQ0UsSUFBSSxDQUFDO1lBQUU1QyxJQUFJLEVBQUU7Y0FBRSxrQkFBa0IsRUFBRStDO1lBQVE7VUFBRSxDQUFDLENBQUM7UUFDeEQ7UUFFQSxNQUFNQyxZQUFZLEdBQUc7VUFBRUMsSUFBSSxFQUFFO1FBQXVCLENBQUM7UUFDckQsTUFBTUMsbUJBQW1CLEdBQUc7VUFBRXRELElBQUksRUFBRTtRQUFLLENBQUM7UUFDMUMsSUFBSW9ELFlBQVksQ0FBQ2hCLFNBQVMsQ0FBQyxFQUFFO1VBQzNCa0IsbUJBQW1CLENBQUNDLElBQUksR0FBRyxDQUFDO1lBQUUsQ0FBQ0gsWUFBWSxDQUFDaEIsU0FBUyxDQUFDLEdBQUdEO1VBQWMsQ0FBQyxDQUFDO1VBQ3pFbUIsbUJBQW1CLENBQUN0RCxJQUFJLEdBQUdWLGVBQUMsQ0FBQ2tFLFNBQVMsQ0FBQ3hELElBQUksRUFBRSxJQUFJLENBQUM7VUFDbERzRCxtQkFBbUIsQ0FBQzFDLElBQUksR0FBR3RCLGVBQUMsQ0FBQ2tFLFNBQVMsQ0FBQzVDLElBQUksRUFBRSxDQUFDLENBQUM7UUFDakQ7UUFFQSxNQUFNeEMsTUFBTSxHQUFHO1VBQ2JDLElBQUksRUFBRTtZQUNKb0YsbUJBQW1CLEVBQUUsSUFBSTtZQUN6QmpFLE9BQU8sRUFBRSxJQUFJO1lBQ2IsR0FBRzhELG1CQUFtQjtZQUN0QnJELEtBQUssRUFBRTtjQUNMQyxJQUFJLEVBQUU7Z0JBQ0o0QyxNQUFNO2dCQUNOWSxvQkFBb0IsRUFBRXJCLEtBQUssS0FBSyxLQUFLLEdBQUcsQ0FBQyxHQUFHLENBQUM7Z0JBQzdDbEMsSUFBSSxFQUFFNEM7Y0FDUjtZQUNGLENBQUM7WUFDRFksWUFBWSxFQUFFO2NBQ1pDLDZCQUE2QixFQUFFO2dCQUM3QkMsTUFBTSxFQUFFO2tCQUNOQyxJQUFJLEVBQUU7Z0JBQ1IsQ0FBQztnQkFDRHhELElBQUksRUFBRTtrQkFDSnlELFdBQVcsRUFBRTtvQkFDWHZELEtBQUssRUFBRTtzQkFDTEMsS0FBSyxFQUFFO29CQUNUO2tCQUNGO2dCQUNGO2NBQ0Y7WUFDRjtVQUNGO1FBQ0YsQ0FBQztRQUVELE1BQU1uQyxNQUFNLEdBQUcsSUFBSSxDQUFDQywwQkFBMEIsQ0FBQ0wsT0FBTyxFQUFFQyxHQUFHLENBQUM7UUFDNUQsTUFBTWlCLFdBQVcsR0FBRyxNQUFNZCxNQUFNLENBQUMsc0JBQXNCLEVBQUVGLE1BQU0sQ0FBQztRQUVoRSxNQUFNNEYsYUFBYSxHQUFHMUUsZUFBQyxDQUFDQyxHQUFHLENBQUNILFdBQVcsRUFBRSxrQkFBa0IsRUFBRSxDQUFDLENBQUM7UUFDL0QsTUFBTTZFLHFCQUFxQixHQUFHM0UsZUFBQyxDQUFDQyxHQUFHLENBQUNILFdBQVcsRUFBRSxXQUFXLEVBQUUsRUFBRSxDQUFDLENBQUM4RSxHQUFHLENBQUVoRixNQUFNLElBQUs7VUFDaEYsTUFBTTtZQUNKK0MsR0FBRyxFQUFFbEQsRUFBRTtZQUNQaUQsUUFBUSxFQUFFeEMsT0FBTztZQUNqQjJFLE9BQU8sRUFBRTFFLE9BQU87WUFDaEIyRSxhQUFhLEVBQUUxRSxhQUFhO1lBQzVCMkU7VUFDRixDQUFDLEdBQUduRixNQUFNO1VBQ1YsTUFBTUcsT0FBTyxHQUFHZ0YsT0FBTyxDQUFDaEYsT0FBTyxHQUFHZ0YsT0FBTyxDQUFDaEYsT0FBTyxHQUFHZ0YsT0FBTztVQUMzRGhGLE9BQU8sQ0FBQyxXQUFXLENBQUMsR0FBR0EsT0FBTyxDQUFDbUMsYUFBYSxJQUFJbkMsT0FBTyxDQUFDb0MsWUFBWTtVQUNwRSxNQUFNO1lBQUU0QixJQUFJO1lBQUVGLE9BQU87WUFBRTVCO1VBQVUsQ0FBQyxHQUFHbEMsT0FBTztVQUM1QyxPQUFPLENBQUNOLEVBQUUsRUFBRTtZQUFFQSxFQUFFO1lBQUVTLE9BQU87WUFBRUMsT0FBTztZQUFFQyxhQUFhO1lBQUUyRCxJQUFJO1lBQUVGLE9BQU87WUFBRTVCLFNBQVM7WUFBRWxDO1VBQVEsQ0FBQyxDQUFDO1FBQ3pGLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUNOLE1BQU1pRixVQUFVLEdBQUcsSUFBSUMsR0FBRyxDQUFDTixxQkFBcUIsQ0FBQztRQUNqRCxNQUFNTyxrQ0FBa0MsR0FBRyxDQUFDLENBQUM7UUFDN0NsRixlQUFDLENBQUNDLEdBQUcsQ0FDSEgsV0FBVyxFQUNYLGdFQUFnRSxFQUNoRSxFQUNGLENBQUMsQ0FBQ3FGLE9BQU8sQ0FBQyxDQUFDO1VBQUVoSSxHQUFHO1VBQUUyRTtRQUFVLENBQUMsS0FBSztVQUNoQ29ELGtDQUFrQyxDQUFDL0gsR0FBRyxDQUFDLEdBQUcyRSxTQUFTO1FBQ3JELENBQUMsQ0FBQztRQUNGLE1BQU1zRCxnQkFBZ0IsR0FBRyxDQUFDLEdBQUdKLFVBQVUsQ0FBQ0ssSUFBSSxDQUFDLENBQUMsQ0FBQztRQUUvQyxNQUFNQyxhQUFhLEdBQUcsQ0FBQyxDQUFDO1FBQ3hCLE1BQU1DLFNBQVMsR0FBRztVQUNoQkMsTUFBTSxFQUFFLFFBQVE7VUFDaEJDLFlBQVksRUFBRSxjQUFjO1VBQzVCQyxNQUFNLEVBQUUsUUFBUTtVQUNoQkMsT0FBTyxFQUFFLFNBQVM7VUFDbEJDLG9CQUFvQixFQUFFO1FBQ3hCLENBQUM7UUFDRCxJQUFJTCxTQUFTLENBQUN6QyxTQUFTLENBQUMsRUFBRTtVQUN4QndDLGFBQWEsQ0FBQ08sS0FBSyxHQUFHO1lBQUUsQ0FBQ04sU0FBUyxDQUFDekMsU0FBUyxDQUFDLEdBQUdEO1VBQWMsQ0FBQztRQUNqRTtRQUNBLE1BQU12QyxVQUFVLEdBQUc7VUFDakJDLEtBQUssRUFBRUMsZ0JBQUssQ0FBQ0MsVUFBVTtVQUN2QjFCLElBQUksRUFBRTtZQUNKMkIsSUFBSSxFQUFFLENBQUM7WUFDUEMsS0FBSyxFQUFFO2NBQUVPLEtBQUssRUFBRTtnQkFBRUgsVUFBVSxFQUFFcUU7Y0FBaUI7WUFBRSxDQUFDO1lBQ2xEZixZQUFZLEVBQUU7Y0FDWnlCLGdCQUFnQixFQUFFO2dCQUNoQjVFLEtBQUssRUFBRTtrQkFDTEMsS0FBSyxFQUFFLFlBQVk7a0JBQ25CLEdBQUdtRSxhQUFhO2tCQUNoQjVFLElBQUksRUFBRVksSUFBSSxHQUFHWjtnQkFDZixDQUFDO2dCQUNEMkQsWUFBWSxFQUFFO2tCQUNabUIsTUFBTSxFQUFFO29CQUFFTyxNQUFNLEVBQUU7c0JBQUVqRixJQUFJLEVBQUU7d0JBQUVpQyxLQUFLLEVBQUU7c0JBQVM7b0JBQUU7a0JBQUUsQ0FBQztrQkFDakQwQyxZQUFZLEVBQUU7b0JBQUVNLE1BQU0sRUFBRTtzQkFBRWpGLElBQUksRUFBRTt3QkFBRWlDLEtBQUssRUFBRTtzQkFBZTtvQkFBRTtrQkFBRSxDQUFDO2tCQUM3RDJDLE1BQU0sRUFBRTtvQkFBRUssTUFBTSxFQUFFO3NCQUFFakYsSUFBSSxFQUFFO3dCQUFFaUMsS0FBSyxFQUFFO3NCQUFRO29CQUFFO2tCQUFFLENBQUM7a0JBQ2hENEMsT0FBTyxFQUFFO29CQUNQSSxNQUFNLEVBQUU7c0JBQ05uRixJQUFJLEVBQUU7d0JBQ0ptRixNQUFNLEVBQUU7MEJBQUVqRixJQUFJLEVBQUU7NEJBQUVpQyxLQUFLLEVBQUU7MEJBQVk7d0JBQUUsQ0FBQzt3QkFDeENpRCxRQUFRLEVBQUU7MEJBQUVDLE1BQU0sRUFBRTs0QkFBRTlFLEtBQUssRUFBRTswQkFBb0I7d0JBQUU7c0JBQ3JEO29CQUNGO2tCQUNGLENBQUM7a0JBQ0QrRSxzQkFBc0IsRUFBRTtvQkFBRUMsR0FBRyxFQUFFO3NCQUFFaEYsS0FBSyxFQUFFO29CQUF5QjtrQkFBRSxDQUFDO2tCQUNwRWlGLFlBQVksRUFBRTtvQkFDWkMsUUFBUSxFQUFFO3NCQUNSM0YsSUFBSSxFQUFFLENBQUM7c0JBQ1B1RCxJQUFJLEVBQUUsQ0FBQzt3QkFBRXFDLFVBQVUsRUFBRTswQkFBRVQsS0FBSyxFQUFFO3dCQUFPO3NCQUFFLENBQUMsQ0FBQztzQkFDekNkLE9BQU8sRUFBRTt3QkFDUHdCLFFBQVEsRUFBRSxDQUFDLHdCQUF3QixFQUFFLGNBQWM7c0JBQ3JEO29CQUNGO2tCQUNGO2dCQUNGO2NBQ0Y7WUFDRjtVQUNGO1FBQ0YsQ0FBQztRQUVELE1BQU1DLGNBQWMsR0FBRyxNQUFNeEgsTUFBTSxDQUFDLHNCQUFzQixFQUFFc0IsVUFBVSxDQUFDO1FBQ3ZFLE1BQU1tRyxPQUFPLEdBQUd6RyxlQUFDLENBQUNDLEdBQUcsQ0FBQ3VHLGNBQWMsRUFBRSx1Q0FBdUMsRUFBRSxFQUFFLENBQUMsQ0FBQzVCLEdBQUcsQ0FDbkY4QixNQUFNLElBQUs7VUFDVixNQUFNO1lBQ0p2SixHQUFHLEVBQUVzQyxFQUFFO1lBQ1B5RyxzQkFBc0IsRUFBRTtjQUFFOUksS0FBSyxFQUFFd0k7WUFBcUIsQ0FBQztZQUN2REQsT0FBTyxFQUFFO2NBQUU3RCxTQUFTLEVBQUU2RDtZQUFRLENBQUM7WUFDL0JGLFlBQVksRUFBRTtjQUFFM0QsU0FBUyxFQUFFMkQ7WUFBYSxDQUFDO1lBQ3pDRCxNQUFNLEVBQUU7Y0FBRTFELFNBQVMsRUFBRTBEO1lBQU8sQ0FBQztZQUM3QkUsTUFBTSxFQUFFO2NBQUU1RCxTQUFTLEVBQUU0RDtZQUFPLENBQUM7WUFDN0JVLFlBQVksRUFBRTtjQUNaTyxJQUFJLEVBQUU7Z0JBQ0pBLElBQUksRUFBRSxDQUNKO2tCQUNFNUIsT0FBTyxFQUFFO29CQUFFNkIsWUFBWSxFQUFFQztrQkFBWTtnQkFDdkMsQ0FBQztjQUVMO1lBQ0Y7VUFDRixDQUFDLEdBQUdILE1BQU07VUFDVixNQUFNM0csT0FBTyxHQUFHaUYsVUFBVSxDQUFDL0UsR0FBRyxDQUFDUixFQUFFLENBQUM7VUFDbEN1RixVQUFVLENBQUM4QixNQUFNLENBQUNySCxFQUFFLENBQUM7VUFDckIsT0FBTztZQUNMLEdBQUdNLE9BQU87WUFDVk4sRUFBRTtZQUNGbUcsb0JBQW9CO1lBQ3BCRCxPQUFPO1lBQ1BrQixXQUFXO1lBQ1hwQixZQUFZO1lBQ1pELE1BQU07WUFDTkUsTUFBTTtZQUNOcUIsV0FBVyxFQUFFQyxJQUFJLENBQUNDLEdBQUcsQ0FBQyxDQUFDO1lBQ3ZCbEYsNkJBQTZCLEVBQUVtRCxrQ0FBa0MsQ0FBQ3pGLEVBQUUsQ0FBQyxJQUFJO1VBQzNFLENBQUM7UUFDSCxDQUNGLENBQUM7UUFFRCxNQUFNeUgsY0FBYyxHQUFHLENBQUMsR0FBR2xDLFVBQVUsQ0FBQ21DLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQ3ZDLEdBQUcsQ0FBRTdFLE9BQU8sS0FBTTtVQUNoRSxHQUFHQSxPQUFPO1VBQ1Y2RixvQkFBb0IsRUFBRSxJQUFJO1VBQzFCRCxPQUFPLEVBQUUsQ0FBQztVQUNWSCxNQUFNLEVBQUUsQ0FBQztVQUNUQyxZQUFZLEVBQUUsQ0FBQztVQUNmQyxNQUFNLEVBQUUsQ0FBQztVQUNUbUIsV0FBVyxFQUFFLElBQUk7VUFDakJFLFdBQVcsRUFBRUMsSUFBSSxDQUFDQyxHQUFHLENBQUMsQ0FBQztVQUN2QmxGLDZCQUE2QixFQUFFbUQsa0NBQWtDLENBQUNuRixPQUFPLENBQUNOLEVBQUUsQ0FBQyxJQUFJO1FBQ25GLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSTJILE9BQU8sR0FBR3BILGVBQUMsQ0FBQ3FILE9BQU8sQ0FBQ1osT0FBTyxDQUFDYSxNQUFNLENBQUNKLGNBQWMsQ0FBQyxFQUFFLENBQUNwRSxTQUFTLENBQUMsRUFBRSxDQUFDRCxhQUFhLENBQUMsQ0FBQztRQUNyRjtRQUNBO1FBQ0E7UUFDQSxJQUFJLENBQUNpQixZQUFZLENBQUNoQixTQUFTLENBQUMsRUFBRTtVQUM1QnNFLE9BQU8sR0FBR0EsT0FBTyxDQUFDRyxLQUFLLENBQUNqRyxJQUFJLEVBQUVBLElBQUksR0FBR1osSUFBSSxDQUFDO1FBQzVDO1FBRUEsT0FBT3RDLEdBQUcsQ0FBQ2UsRUFBRSxDQUFDO1VBQ1pKLElBQUksRUFBRTtZQUNKSSxFQUFFLEVBQUUsSUFBSTtZQUNScUksUUFBUSxFQUFFSixPQUFPO1lBQ2pCMUM7VUFDRjtRQUNGLENBQUMsQ0FBQztNQUNKLENBQUMsQ0FBQyxPQUFPckYsR0FBRyxFQUFFO1FBQ1pDLE9BQU8sQ0FBQ0MsS0FBSyxDQUFDLHlDQUF5QyxFQUFFRixHQUFHLENBQUM7UUFDN0QsSUFBSSxJQUFBb0ksNkJBQW9CLEVBQUNwSSxHQUFHLENBQUMsRUFBRTtVQUM3QixPQUFPakIsR0FBRyxDQUFDZSxFQUFFLENBQUM7WUFDWkosSUFBSSxFQUFFO2NBQUVJLEVBQUUsRUFBRSxLQUFLO2NBQUVDLElBQUksRUFBRTtnQkFBRXNGLGFBQWEsRUFBRSxDQUFDO2dCQUFFOEMsUUFBUSxFQUFFO2NBQUc7WUFBRTtVQUM5RCxDQUFDLENBQUM7UUFDSjtRQUNBLE9BQU9wSixHQUFHLENBQUNlLEVBQUUsQ0FBQztVQUNaSixJQUFJLEVBQUU7WUFDSkksRUFBRSxFQUFFLEtBQUs7WUFDVEMsSUFBSSxFQUFFQyxHQUFHLENBQUNHO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSjtJQUNGLENBQUM7SUFBQXRDLGVBQUEsNEJBRW1CLE9BQU8wQixPQUFPLEVBQUVDLEdBQUcsRUFBRVQsR0FBRyxLQUFLO01BQy9DLElBQUk7UUFDRixNQUFNO1VBQUVxQjtRQUFHLENBQUMsR0FBR1osR0FBRyxDQUFDQyxNQUFNO1FBQ3pCLE1BQU1BLE1BQU0sR0FBRztVQUNiWSxTQUFTLEVBQUVELEVBQUU7VUFDYlYsSUFBSSxFQUFFRixHQUFHLENBQUNFO1FBQ1osQ0FBQztRQUNELE1BQU1DLE1BQU0sR0FBRyxJQUFJLENBQUNDLDBCQUEwQixDQUFDTCxPQUFPLEVBQUVDLEdBQUcsQ0FBQztRQUM1RCxNQUFNNkksbUJBQW1CLEdBQUcsTUFBTTFJLE1BQU0sQ0FBQyw0QkFBNEIsRUFBRUYsTUFBTSxDQUFDO1FBQzlFLE9BQU9WLEdBQUcsQ0FBQ2UsRUFBRSxDQUFDO1VBQ1pKLElBQUksRUFBRTtZQUNKSSxFQUFFLEVBQUUsQ0FBQ3VJLG1CQUFtQixDQUFDQyxNQUFNLENBQUMzRixNQUFNO1lBQ3RDNUMsSUFBSSxFQUFFc0k7VUFDUjtRQUNGLENBQUMsQ0FBQztNQUNKLENBQUMsQ0FBQyxPQUFPckksR0FBRyxFQUFFO1FBQ1pDLE9BQU8sQ0FBQ0MsS0FBSyxDQUFDLGdEQUFnRCxFQUFFRixHQUFHLENBQUM7UUFDcEUsT0FBT2pCLEdBQUcsQ0FBQ2UsRUFBRSxDQUFDO1VBQ1pKLElBQUksRUFBRTtZQUNKSSxFQUFFLEVBQUUsS0FBSztZQUNUQyxJQUFJLEVBQUVDLEdBQUcsQ0FBQ0c7VUFDWjtRQUNGLENBQUMsQ0FBQztNQUNKO0lBQ0YsQ0FBQztJQUFBdEMsZUFBQSxtQ0FFMEIsT0FBTzBCLE9BQU8sRUFBRUMsR0FBRyxFQUFFVCxHQUFHLEtBQUs7TUFDdEQsSUFBSTtRQUNGLE1BQU07VUFBRXFCO1FBQUcsQ0FBQyxHQUFHWixHQUFHLENBQUNDLE1BQU07UUFDekIsTUFBTUEsTUFBTSxHQUFHO1VBQ2JlLFVBQVUsRUFBRUosRUFBRTtVQUNkVixJQUFJLEVBQUVGLEdBQUcsQ0FBQ0U7UUFDWixDQUFDO1FBQ0QsTUFBTUMsTUFBTSxHQUFHLElBQUksQ0FBQ0MsMEJBQTBCLENBQzVDTCxPQUFPLEVBQ1AsSUFBSSxDQUFDZ0osaUJBQWlCLEVBQ3RCL0ksR0FBRyxFQUNILElBQUksQ0FBQ2dKLFFBQ1AsQ0FBQztRQUNELE1BQU1ILG1CQUFtQixHQUFHLE1BQU0xSSxNQUFNLENBQUMsbUNBQW1DLEVBQUVGLE1BQU0sQ0FBQztRQUNyRixPQUFPVixHQUFHLENBQUNlLEVBQUUsQ0FBQztVQUNaSixJQUFJLEVBQUU7WUFDSkksRUFBRSxFQUFFLENBQUN1SSxtQkFBbUIsQ0FBQ0MsTUFBTSxDQUFDM0YsTUFBTTtZQUN0QzVDLElBQUksRUFBRXNJO1VBQ1I7UUFDRixDQUFDLENBQUM7TUFDSixDQUFDLENBQUMsT0FBT3JJLEdBQUcsRUFBRTtRQUNaQyxPQUFPLENBQUNDLEtBQUssQ0FBQyx1REFBdUQsRUFBRUYsR0FBRyxDQUFDO1FBQzNFLE9BQU9qQixHQUFHLENBQUNlLEVBQUUsQ0FBQztVQUNaSixJQUFJLEVBQUU7WUFDSkksRUFBRSxFQUFFLEtBQUs7WUFDVEMsSUFBSSxFQUFFQyxHQUFHLENBQUNHO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSjtJQUNGLENBQUM7SUFBQXRDLGVBQUEseUJBRWdCLE9BQU8wQixPQUFPLEVBQUVDLEdBQUcsRUFBRVQsR0FBRyxLQUFLO01BQzVDLElBQUk7UUFDRixNQUFNO1VBQUUwSixNQUFNLEdBQUc7UUFBTyxDQUFDLEdBQUdqSixHQUFHLENBQUM4QixLQUFLO1FBQ3JDLE1BQU03QixNQUFNLEdBQUc7VUFDYkMsSUFBSSxFQUFFRixHQUFHLENBQUNFLElBQUk7VUFDZCtJO1FBQ0YsQ0FBQztRQUNELE1BQU05SSxNQUFNLEdBQUcsSUFBSSxDQUFDQywwQkFBMEIsQ0FBQ0wsT0FBTyxFQUFFQyxHQUFHLENBQUM7UUFDNUQsTUFBTWtKLGVBQWUsR0FBRyxNQUFNL0ksTUFBTSxDQUFDLHlCQUF5QixFQUFFRixNQUFNLENBQUM7UUFDdkUsT0FBT1YsR0FBRyxDQUFDZSxFQUFFLENBQUM7VUFDWkosSUFBSSxFQUFFO1lBQ0pJLEVBQUUsRUFBRSxJQUFJO1lBQ1JDLElBQUksRUFBRTJJO1VBQ1I7UUFDRixDQUFDLENBQUM7TUFDSixDQUFDLENBQUMsT0FBTzFJLEdBQUcsRUFBRTtRQUNaQyxPQUFPLENBQUNDLEtBQUssQ0FBQyw2Q0FBNkMsRUFBRUYsR0FBRyxDQUFDO1FBQ2pFLE9BQU9qQixHQUFHLENBQUNlLEVBQUUsQ0FBQztVQUNaSixJQUFJLEVBQUU7WUFDSkksRUFBRSxFQUFFLEtBQUs7WUFDVEMsSUFBSSxFQUFFQyxHQUFHLENBQUNHO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSjtJQUNGLENBQUM7SUFFRDtJQUFBdEMsZUFBQSx5QkFDaUIsT0FBTzBCLE9BQU8sRUFBRUMsR0FBRyxFQUFFVCxHQUFHLEtBQUs7TUFDNUMsSUFBSTtRQUNGLE1BQU07VUFBRXVDLEtBQUs7VUFBRUosS0FBSztVQUFFRztRQUFLLENBQUMsR0FBRzdCLEdBQUcsQ0FBQ0UsSUFBSTtRQUN2QyxNQUFNRCxNQUFNLEdBQUc7VUFBRXlCLEtBQUs7VUFBRUcsSUFBSTtVQUFFM0IsSUFBSSxFQUFFNEI7UUFBTSxDQUFDO1FBRTNDLE1BQU0zQixNQUFNLEdBQUcsSUFBSSxDQUFDQywwQkFBMEIsQ0FBQ0wsT0FBTyxFQUFFQyxHQUFHLENBQUM7UUFDNUQsTUFBTXVJLE9BQU8sR0FBRyxNQUFNcEksTUFBTSxDQUFDLHNCQUFzQixFQUFFRixNQUFNLENBQUM7UUFDNUQsT0FBT1YsR0FBRyxDQUFDZSxFQUFFLENBQUM7VUFDWkosSUFBSSxFQUFFO1lBQ0pJLEVBQUUsRUFBRSxJQUFJO1lBQ1JDLElBQUksRUFBRWdJO1VBQ1I7UUFDRixDQUFDLENBQUM7TUFDSixDQUFDLENBQUMsT0FBTy9ILEdBQUcsRUFBRTtRQUNaQyxPQUFPLENBQUNDLEtBQUssQ0FBQyw0Q0FBNEMsRUFBRUYsR0FBRyxDQUFDO1FBQ2hFLE9BQU9qQixHQUFHLENBQUNlLEVBQUUsQ0FBQztVQUNaSixJQUFJLEVBQUU7WUFDSkksRUFBRSxFQUFFLEtBQUs7WUFDVEMsSUFBSSxFQUFFQyxHQUFHLENBQUNHO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSjtJQUNGLENBQUM7RUFBQTtBQUNIO0FBQUN3SSxPQUFBLENBQUEvSyxPQUFBLEdBQUF1QixjQUFBO0FBQUF5SixNQUFBLENBQUFELE9BQUEsR0FBQUEsT0FBQSxDQUFBL0ssT0FBQSJ9