"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _constants = require("../utils/constants");

var _jsYaml = require("js-yaml");

var _moment = _interopRequireDefault(require("moment"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

class RulesService {
  constructor(osDriver) {
    _defineProperty(this, "osDriver", void 0);

    _defineProperty(this, "createRule", async (_context, request, response) => {
      try {
        const {
          id,
          title,
          category,
          description,
          detection,
          status,
          author,
          references,
          tags,
          level,
          false_positives
        } = request.body;
        const today = (0, _moment.default)(_moment.default.now()).format('YYYY/MM/DD');
        const jsonPayload = {
          id,
          title,
          description: description || title,
          status,
          author,
          date: today,
          modified: today,
          logsource: {
            category
          },
          level,
          detection: (0, _jsYaml.load)(detection)
        };

        if (tags.length > 0) {
          jsonPayload['tags'] = tags.map(tag => tag.value);
        }

        if (references.length > 0) {
          jsonPayload['references'] = references.map(ref => ref.value);
        }

        if (false_positives.length > 0) {
          jsonPayload['falsepositives'] = false_positives.map(falsePos => falsePos.value);
        }

        const ruleYamlPayload = (0, _jsYaml.safeDump)(jsonPayload);
        const params = {
          body: ruleYamlPayload,
          category: encodeURIComponent(category)
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const createRuleResponse = await callWithRequest(_constants.CLIENT_RULE_METHODS.CREATE_RULE, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: createRuleResponse
          }
        });
      } catch (error) {
        console.error('Security Analytics - RulesService - createRule:', error);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: error.message
          }
        });
      }
    });

    _defineProperty(this, "getRules", async (_context, request, response) => {
      try {
        const {
          prePackaged
        } = request.query;
        const params = {
          prePackaged,
          body: request.body
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const getRuleResponse = await callWithRequest(_constants.CLIENT_RULE_METHODS.GET_RULES, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: getRuleResponse
          }
        });
      } catch (error) {
        console.error('Security Analytics - RulesService - getRules:', error);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: error.message
          }
        });
      }
    });

    _defineProperty(this, "deleteRule", async (_context, request, response) => {
      try {
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const deleteRuleResponse = await callWithRequest(_constants.CLIENT_RULE_METHODS.DELETE_RULE, request.params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: deleteRuleResponse
          }
        });
      } catch (error) {
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: error.message
          }
        });
      }
    });

    _defineProperty(this, "updateRule", async (_context, request, response) => {
      try {
        const {
          id,
          title,
          category,
          description,
          detection,
          status,
          author,
          references,
          tags,
          level,
          false_positives
        } = request.body;
        const today = (0, _moment.default)(_moment.default.now()).format('YYYY/MM/DD');
        const {
          ruleId
        } = request.params;
        const jsonPayload = {
          id,
          title,
          description: description || title,
          status,
          author,
          date: today,
          modified: today,
          logsource: {
            category
          },
          level,
          detection: (0, _jsYaml.load)(detection)
        };

        if (tags.length > 0) {
          jsonPayload['tags'] = tags.map(tag => tag.value);
        }

        if (references.length > 0) {
          jsonPayload['references'] = references.map(ref => ref.value);
        }

        if (false_positives.length > 0) {
          jsonPayload['falsepositives'] = false_positives.map(falsePos => falsePos.value);
        }

        const ruleYamlPayload = (0, _jsYaml.safeDump)(jsonPayload);
        const params = {
          body: ruleYamlPayload,
          category,
          ruleId
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const createRuleResponse = await callWithRequest(_constants.CLIENT_RULE_METHODS.UPDATE_RULE, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: createRuleResponse
          }
        });
      } catch (error) {
        console.error('Security Analytics - RulesService - updateRule:', error);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: error.message
          }
        });
      }
    });

    this.osDriver = osDriver;
  }
  /**
   * Calls backend POST Rules API.
   */

  /**
   * Calls backend PUT Rules API.
   */


}

exports.default = RulesService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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