/* $Id: world.c,v 30000.21 1993/05/18 09:02:02 kkeys Exp $ */
/******************************************************************
 * Copyright 1993 by Ken Keys.
 * Permission is granted to obtain and redistribute this code freely.
 * All redistributions must contain this header.  You may modify this
 * software, but any redistributions of modified code must be clearly
 * marked as having been modified.
 ******************************************************************/


/********************************************************
 * Fugue world routines.                                *
 ********************************************************/

#include <ctype.h>
#include "port.h"
#include "dstring.h"
#include "tf.h"
#include "util.h"
#include "history.h"
#include "world.h"
#include "output.h"
#include "process.h"
#include "macro.h"
#include "search.h"

static void FDECL(free_fields,(World *w));
static void FDECL(replace_world,(World *old, World *new));
static World *FDECL(insertworld,(World *world));

int    FDECL(addworld,(char *args));
World *FDECL(new_world,(char *name, char *character, char *pass,
             char *address, char *port, char *mfile));
int    FDECL(remove_world,(char *args));
int    FDECL(purge_world,(char *args));
int    FDECL(write_worlds,(char *args));
void   FDECL(list_worlds,(int full, char *pattern, TFILE *file));
void   FDECL(free_world,(World *w));
void   FDECL(nuke_world,(World *w));
World *NDECL(get_default_world);
World *NDECL(get_world_header);
World *FDECL(find_world,(char *name));
void   FDECL(flush_world,(World *w));

static World *hworld = NULL, *defaultworld = NULL;

static void free_fields(w)
    World *w;
{
    FREE(w->name);
    FREE(w->character);
    FREE(w->pass);
    FREE(w->address);
    FREE(w->port);
    FREE(w->mfile);
}

void free_world(w)
    World *w;
{
    free_fields(w);
    if (w->history) {
        free_history(w->history);
        FREE(w->history);
    }
    if (w->queue) {
        free_queue(w->queue);
        FREE(w->queue);
    }
    FREE(w);
}

static void replace_world(old, new)
    World *old, *new;
{
    free_fields(old);
    old->name      = new->name;
    old->character = new->character;
    old->pass      = new->pass;
    old->address   = new->address;
    old->port      = new->port;
    old->mfile     = new->mfile;
    FREE(new);
    do_hook(H_REDEF, "%% Redefined %s %s", "%s %s", "world", old->name);
}

World *new_world(name, character, pass, address, port, mfile)
    char *name, *character, *pass, *address, *port, *mfile;
{
    World *result;
    static int unnamed = 1;
    char buffer[16];               /* big enough for "(unnamedNNNNNN)" */
 
    result = (World *) MALLOC(sizeof(World));
    if (name) {
        result->name      = STRDUP(name);
    } else {
        sprintf(buffer, "(unnamed%d)", unnamed++);
        result->name      = STRDUP(buffer);
    }
    result->character = STRDUP(character);
    result->pass      = STRDUP(pass);
    result->address   = STRDUP(address);
    result->port      = STRDUP(port);
    result->mfile     = STRDUP(mfile);
    result->flags = 0;
    result->socket = NULL;
    result->history = (History *)MALLOC(sizeof(History));
    result->history->alines = NULL;
    result->queue = (Queue *)MALLOC(sizeof(Queue));
    init_queue(result->queue);
    return insertworld(result);
}

int addworld(args)
    char *args;
{
    int count, i;
    static Stringp fields[6];
    static int fields_inited = FALSE;
    World *new;

    if (!fields_inited) {
        for (i = 0; i < 6; i++) Stringinit(fields[i]);
        fields_inited = TRUE;
    }
    for (count = 0; count < 6; count++) {
        if (!*args) break;
        args = getword(fields[count], args);
    }
    if (count < 3 || count > 6) {
        tfputs("% Illegal world format", tferr);
        return 0;
    } else if (count >= 5) {
        new = new_world(fields[0]->s, fields[1]->s, fields[2]->s, fields[3]->s,
            fields[4]->s, count == 6 ? fields[5]->s : "");
    } else if (cstrcmp(fields[0]->s, "default") == 0) {
        new = new_world(fields[0]->s, fields[1]->s, fields[2]->s, "", "",
            count == 4 ? fields[3]->s : "");
    } else {
        new = new_world(fields[0]->s, "", "", fields[1]->s, fields[2]->s,
            count == 4 ? fields[3]->s : "");
    }
    new->flags &= ~WORLD_TEMP;
    return 1;
}

static World *insertworld(world)
    World *world;
{
    World *w;

    if (cstrcmp(world->name, "default") == 0) {
        if (defaultworld) replace_world(defaultworld, world);
        else defaultworld = world;
    } else if ((w = find_world(world->name)) != NULL) {
        replace_world(w, world);
        return w;
    } else if (hworld == NULL) {
        hworld = world;
        world->next = NULL;
    } else {
        for (w = hworld; w->next != NULL; w = w->next);
        w->next = world;
        world->next = NULL;
    }
    return world;
}

void nuke_world(w)
    World *w;
{
    World *t;

    if (w->socket) {
        tfprintf(tferr, "%% %s: Cannot nuke world currently in use.", w->name);
    } else {
        if (w == hworld) hworld = w->next;
        else {
            for (t = hworld; t->next != w; t = t->next);
            t->next = w->next;
        }
        remove_world_macros(w);
        kill_procs_by_world(w);
        free_world(w);
    }
}

int remove_world(args)
    char *args;
{
    World *w;

    w = find_world(args);
    if (w == NULL) tfprintf(tferr, "%% No world %s", args);
    else nuke_world(w);
    return w ? 1 : 0;
}

int purge_world(args)
    char *args;
{
    World *world, *next;

    if (!smatch_check(args)) return 0;
    for (world = hworld; world; world = next) {
        next = world->next;
        if (equalstr(args, world->name)) nuke_world(world);
    }
    return 1;
}

void list_worlds(full, pattern, file)
    int full;
    char *pattern;
    TFILE *file;
{
    World *p;
    int first = 1;
    STATIC_BUFFER(buf)

    for (p = defaultworld; p || first; p = first ? hworld : p->next, first=0) {
        if (!p || (pattern && !equalstr(pattern, p->name))) continue;
        Sprintf(buf, file ? "/addworld %s " : "%% %s ", p->name);
        if (*p->character) {
            Stringadd(Stringcat(buf, p->character), ' ');
            if (full) Stringadd(Stringcat(buf, p->pass), ' ');
        }
        if (*p->address) Sprintf(buf, "\200%s %s ", p->address, p->port);
        if (*p->mfile) Stringcat(buf, p->mfile);
        if (file) tfputs(buf->s, file);
        else oputs(buf->s);
    }
}

int write_worlds(args)
    char *args;
{
    TFILE *file;

    if ((file = tfopen(tfname(args, "WORLDFILE"), "w")) == NULL) {
        operror(args);
        return 0;
    }
    list_worlds(TRUE, NULL, file);
    tfclose(file);
    return 1;
}

World *get_default_world()
{
    return defaultworld;
}

World *get_world_header()
{
    return hworld;
}

World *find_world(name)
    char *name;
{
    World *p;

    for (p=hworld; p && (!p->name || cstrcmp(name, p->name) != 0); p = p->next);
    return p;
}

void flush_world(w)
    World *w;
{
    extern TFILE *tfscreen;
    ListEntry *node;

    for (node = w->queue->head; node; node = node->next)
        record_global((Aline *)node->data);
    queuequeue(w->queue, tfscreen->u.queue);
    w->history->index = w->history->pos;
    oflush();
}

