/* $Id: keyboard.c,v 30000.22 1993/05/18 08:56:20 kkeys Exp $ */
/******************************************************************
 * Copyright 1993 by Ken Keys.
 * Permission is granted to obtain and redistribute this code freely.
 * All redistributions must contain this header.  You may modify this
 * software, but any redistributions of modified code must be clearly
 * marked as having been modified.
 ******************************************************************/


/**************************************************
 * Fugue keyboard handling.                       *
 * Handles all keyboard input and keybindings.    *
 * Also deals with tty driver.                    *
 **************************************************/
#include "port.h"

#ifdef USE_TERMIO
# ifdef hpux              /* hpux's termio is slightly different than SysV's. */
#  undef USE_TERMIO
#  define USE_HPUX_TERMIO
#  include <sys/ioctl.h>
#  include <termio.h>
#  include <bsdtty.h>     /* ...and needs a few extra #includes. */
# endif
#endif

#ifdef USE_TERMIOS                    /* POSIX is the way to go. */
# include <termios.h>
# ifndef TIOCGWINSZ
#  include <sys/ioctl.h>              /* BSD needs this for TIOCGWINSZ */
# endif
/* These defs allow termio and termios users to share code. */
# define insetattr(fd, buf) (tcsetattr((fd), TCSAFLUSH, (buf)))
# define ingetattr(fd, buf) (tcgetattr((fd), (buf)))
#endif

#ifdef USE_TERMIO
# ifdef WINS_UNNEEDED      /* I really don't think any of this is needed. */
/* I'm going to nuke this block in the next release if nobody complains. */
#  include <sys/types.h>        /* Probably needed for inet.h, which isn't. */
#  include <sys/inet.h>         /* Huh?  inet.h in terminal code??? */
#  include <sys/stream.h>       /* We don't need no steenking streams. */
#  include <sys/ptem.h>         /* And we surely don't need pseudo-ttys. */
# endif
# include <termio.h>
/* The next few defs allow termio and termios users to share code. */
# define ingetattr(fd, buf) (ioctl((fd), TCGETA, (buf)))
# define insetattr(fd, buf) (ioctl((fd), TCSETAF, (buf)))
# define termios termio
# define USE_TERMIOS
#endif

#ifdef USE_SGTTY
# include <sys/ioctl.h>
# include <sgtty.h>                  /* BSD's old "new" terminal driver. */
#endif

#ifdef USE_TERMIOS
static struct termios old_tty;
#endif
#ifdef USE_HPUX_TERMIO
static struct termio old_tty;
#endif
#ifdef USE_SGTTY
static struct sgttyb old_tty;
#endif

static int iscbreak = 0;             /* is tty in cbreak/noecho mode? */
static int literal_next = 0;

#include <ctype.h>
#include <errno.h>
#include "dstring.h"
#include "tf.h"
#include "util.h"
#include "macro.h"
#include "output.h"
#include "history.h"
#include "socket.h"
#include "expand.h"

#undef CTRL
#define CTRL(x) ctrl_values[x]
#define DEFAULT_COLUMNS 80

typedef struct KeyNode {
    int children;
    union {
        struct KeyNode **child;
        Macro *macro;
    } u;
} KeyNode;

static void     NDECL(use_stty);
static void     NDECL(init_ctrl_table);
static Macro   *FDECL(trie_insert,(KeyNode **root, Macro *macro, char *key));
static KeyNode *FDECL(untrie_key,(KeyNode **root, char *s));
static void     FDECL(set_ekey,(char *key, char *cmd));
static void     FDECL(process_buffer,(Stringp inbuf));

/* The /dokey functions. */
void NDECL(do_newline);
void NDECL(do_recallb);
void NDECL(do_recallf);
void NDECL(do_socketb);
void NDECL(do_socketf);
void NDECL(do_bspc);
void NDECL(do_bword);
void NDECL(do_dline);
void NDECL(do_dch);
void NDECL(do_redraw);
void NDECL(do_up);
void NDECL(do_down);
void NDECL(do_left);
void NDECL(do_right);
void NDECL(do_home);
void NDECL(do_end);
void NDECL(do_wleft);
void NDECL(do_wright);
void NDECL(do_deol);
void NDECL(do_lnext);

void   NDECL(init_keyboard);
void   NDECL(get_window_size);
Macro *FDECL(bind_key,(Macro *macro));
void   FDECL(unbind_key,(Macro *macro));
Macro *FDECL(find_key,(char *key));
char  *FDECL(print_to_ascii,(char *src));
char  *FDECL(ascii_to_print,(char *key));
void   NDECL(handle_keyboard_input);
void   FDECL(handle_input_string,(char *input, unsigned int len));
void   NDECL(cbreak_noecho_mode);
void   NDECL(reset_tty);
void   NDECL((*FDECL(find_efunc,(char *name))));

static Stringp scratch;                 /* buffer for manipulating text */
static char ctrl_values[128];           /* map of '^X' form to ascii */
static Stringp cat_keybuf;              /* Total buffer for /cat */
static Stringp current_input;           /* unprocessed keystrokes */
static KeyNode *keytrie = NULL;         /* root of keybinding trie */

Stringp keybuf;                         /* input buffer */
unsigned int keyboard_pos;              /* current position in buffer */

typedef struct EditFunc {               /* Editing function */
    char *name;
    NFunc *func;
} EditFunc;

static EditFunc efunc[] = {
    { "RECALLB",  do_recallb  },
    { "RECALLF",  do_recallf  },
    { "SOCKETB",  do_socketb  },
    { "SOCKETF",  do_socketf  },
    { "BSPC"   ,  do_bspc     },
    { "BWORD"  ,  do_bword    },
    { "DLINE"  ,  do_dline    },
    { "DCH"    ,  do_dch      },
    { "REFRESH",  do_refresh  },
    { "REDRAW" ,  do_redraw   },
    { "UP"     ,  do_up       },
    { "DOWN"   ,  do_down     },
    { "RIGHT"  ,  do_right    },
    { "LEFT"   ,  do_left     },
    { "HOME"   ,  do_home     },
    { "END"    ,  do_end      },
    { "WLEFT"  ,  do_wleft    },
    { "WRIGHT" ,  do_wright   },
    { "DEOL"   ,  do_deol     },
    { "PAGE"   ,  do_page     },
    { "HPAGE"  ,  do_hpage    },
    { "LINE"   ,  do_line     },
    { "FLUSH"  ,  do_flush    },
    { "LNEXT"  ,  do_lnext    },
    { "NEWLINE",  do_newline  }
};

#define NFUNCS (sizeof(efunc) / sizeof(struct EditFunc))

NFunc *find_efunc(name)
    char *name;
{
    int i;

    for (i = 0; i < NFUNCS; i++)
        if (cstrcmp(name, efunc[i].name) == 0) break;
    return (i < NFUNCS) ? efunc[i].func : NULL;
}

void init_keyboard()
{
    init_ctrl_table();
    keytrie = NULL;
    use_stty();
    Stringinit(scratch);
    Stringinit(keybuf);
    Stringinit(cat_keybuf);
    Stringinit(current_input);
    cbreak_noecho_mode();
    keyboard_pos = 0;
}

#ifdef DMALLOC
void free_keyboard()
{
    Stringfree(scratch);
    Stringfree(keybuf);
    Stringfree(cat_keybuf);
    Stringfree(current_input);
}
#endif

/* create map of '^X' forms to ascii characters */
static void init_ctrl_table()
{
    int i, j;

    for (i = 0; i < 128; i++) {
        j = ucase(i) - 'A' + 1;
        ctrl_values[i] = (j < 0) ? (j + 128) : j;
    }
}

/* Bind <cmd> to <key> */
static void set_ekey(key, cmd)
    char *key, *cmd;
{
    Macro *macro;

    macro = new_macro("", "", key, 0, "", cmd, NULL, 0, 0, 0, 0, TRUE);
    if (install_bind(macro)) add_macro(macro);
}

/* Convert ascii string to printable string with '^X' forms. */
/* Returns pointer to static area; copy if needed. */
char *ascii_to_print(key)
    char *key;
{
    STATIC_BUFFER(buffer)

    for (Stringterm(buffer, 0); *key; key++) {
        if (*key == '^' || *key == '\\') {
            Stringadd(Stringadd(buffer, '\\'), *key);
        } else if (!isprint(*key) || iscntrl(*key)) {
            Stringadd(Stringadd(buffer, '^'), (*key + '@') % 128);
        } else Stringadd(buffer, *key);
    }
    return buffer->s;
}

/* Get a few editing key assignments from the terminal driver. */
static void use_stty()
{
#ifdef USE_TERMIOS
    struct termios tty;
#endif
#ifdef USE_HPUX_TERMIO
    struct termio tty;
    struct ltchars chars;
#endif
#ifdef USE_SGTTY
    struct sgttyb tty;
    struct ltchars chars;
#endif

    char bs[2], dline[2], bword[2], refresh[2], lnext[2];
    bs[0] = dline[0] = bword[0] = refresh[0] = lnext[0] = '\0';

#ifdef USE_TERMIOS
    if (ingetattr(0, &tty) < 0) perror("tcgetattr");
    *bs = tty.c_cc[VERASE];
    *dline = tty.c_cc[VKILL];
# ifdef VWERASE /* Not POSIX */
    *bword = tty.c_cc[VWERASE];
# endif
# ifdef VREPRINT /* Not POSIX */
    *refresh = tty.c_cc[VREPRINT];
# endif
# ifdef VLNEXT /* Not POSIX */
    *lnext = tty.c_cc[VLNEXT];
# endif
#endif

#ifdef USE_HPUX_TERMIO
    if (ioctl(0, TCGETA, &tty) < 0) perror("TCGETA ioctl");
    if (ioctl(0, TIOCGLTC, &chars) < 0) perror("TIOCGLTC ioctl");
    *bs = tty.c_cc[VERASE];
    *dline = tty.c_cc[VKILL];
    *bword = chars.t_werasc;
    *refresh = chars.t_rprntc;
#endif

#ifdef USE_SGTTY
    if (ioctl(0, TIOCGETP, &tty) < 0) perror("TIOCGETP ioctl");
    if (ioctl(0, TIOCGLTC, &chars) < 0) perror("TIOCGLTC ioctl");
    *bs = tty.sg_erase;
    *dline = tty.sg_kill;
    *bword = chars.t_werasc;
    *refresh = chars.t_rprntc;
    *lnext = chars.t_lnextc;
#endif

    bs[1] = dline[1] = bword[1] = refresh[1] = lnext[1] = '\0';
    /* Note that some systems use \0 to disable, others use \377. */
    if (isascii(*bs) && *bs) set_ekey(bs, "/DOKEY BSPC");
    if (isascii(*bword) && *bword) set_ekey(bword, "/DOKEY BWORD");
    if (isascii(*dline) && *dline) set_ekey(dline, "/DOKEY DLINE");
    if (isascii(*refresh) && *refresh) set_ekey(refresh, "/DOKEY REFRESH");
    if (isascii(*lnext) && *lnext) set_ekey(lnext, "/DOKEY LNEXT");
    get_window_size();
}

void get_window_size()
{
#ifdef TIOCGWINSZ
    extern int columns, lines;
    struct winsize size;
    int ocol, oline;

    ocol = columns;
    oline = lines;
    if (ioctl(0, TIOCGWINSZ, &size) < 0) {
        perror("TIOCGWINSZ ioctl");
        return;
    }
    if (!size.ws_col || !size.ws_row) return;
    columns = size.ws_col;
    lines = size.ws_row;
    if (columns <= 20) columns = DEFAULT_COLUMNS;
    if (columns == ocol && lines == oline) return;
    do_redraw();
    setivar("wrapsize", columns - 1, FALSE);
    do_hook(H_RESIZE, NULL, "%d %d", columns, lines);
#endif
}

/* Find the macro assosiated with <key> sequence.  Pretty damn fast. */
Macro *find_key(key)
    char *key;
{
    KeyNode *n;

    for (n = keytrie; n && n->children && *key; n = n->u.child[*key++]);
    return (n && !n->children && !*key) ? n->u.macro : NULL;
}

/* Insert a macro into the keybinding trie */
static Macro *trie_insert(root, macro, key)
    KeyNode **root;
    Macro *macro;
    char *key;
{
    int i;

    if (!*root) {
        *root = (KeyNode *) MALLOC(sizeof(KeyNode));
        if (*key) {
            (*root)->children = 1;
            (*root)->u.child = (KeyNode **) MALLOC(128 * sizeof(KeyNode *));
            for (i = 0; i < 128; i++) (*root)->u.child[i] = NULL;
            return trie_insert(&(*root)->u.child[*key], macro, key + 1);
        } else {
            (*root)->children = 0;
            return (*root)->u.macro = macro;
        }
    } else {
        if (*key) {
            if ((*root)->children) {
                if (!(*root)->u.child[*key]) (*root)->children++;
                return trie_insert(&(*root)->u.child[*key], macro, key + 1);
            } else {
                tfprintf(tferr, "%% %s is prefixed by an existing sequence.",
                    ascii_to_print(macro->bind));
                return NULL;
            }
        } else {
            if ((*root)->children) {
                tfprintf(tferr, "%% %s is prefix of an existing sequence.",
                  ascii_to_print(macro->bind));
                return NULL;
            } else if (redef) {
                return (*root)->u.macro = macro;
            } else {
                tfprintf(tferr, "%% Binding %s already exists.",
                    ascii_to_print(macro->bind));
                return NULL;
            }
        }
    }
}

Macro *bind_key(macro)
    Macro *macro;
{
    return trie_insert(&keytrie, macro, macro->bind);
}

static KeyNode *untrie_key(root, s)
    KeyNode **root;
    char *s;
{
    if (!*s) {
        FREE(*root);
        return *root = NULL;
    }
    if (untrie_key(&((*root)->u.child[*s]), s + 1)) return *root;
    if (--(*root)->children) return *root;
    FREE((*root)->u.child);
    FREE(*root);
    return *root = NULL;
}

void unbind_key(macro)
    Macro *macro;
{
    untrie_key(&keytrie, macro->bind);
}

/* Convert a printable string containing '^X' and '\nnn' to real ascii. */
/* Returns pointer to static area; copy if needed. */
char *print_to_ascii(src)
    char *src;
{
    STATIC_BUFFER(dest);

    Stringterm(dest, 0);
    while (*src) {
        if (*src == '^') {
            Stringadd(dest, (*++src) ? CTRL(*src++) : '^');
        } else if (*src == '\\' && isdigit(*++src)) {
            Stringadd(dest, strtochr(&src));
        } else Stringadd(dest, *src++);
    }
    return dest->s;
}

void handle_keyboard_input()
{
    char *s, buf[64];
    int i, count;
    static KeyNode *n;
    static int key_start = 0;
    static int input_start = 0;
    static int place = 0;

    /* read a block of text */
    if ((count = read(0, buf, 64)) < 0) {
        if (errno == EINTR) return;
        else die("%% Couldn't read keyboard.\n");
    }
    if (!count) return;

    for (i = 0; i < count; i++) {
        /* make sure input is palatable */
        buf[i] &= 0x7f;
        if (buf[i] == '\r') Stringadd(current_input, '\n');
        else if (buf[i] != '\0') Stringadd(current_input, buf[i]);
    }

    s = current_input->s;
    if (!n) n = keytrie;
    while (s[place]) {
        if (literal_next) {
            place++;
            key_start++;
            literal_next = FALSE;
            continue;
        }
        while (s[place] && n && n->children) n = n->u.child[s[place++]];
        if (!n || !keytrie->children) {
            /* No match.  Try a suffix. */
            place = ++key_start;
            n = keytrie;
        } else if (n->children) {
            /* Partial match.  Just hold on to it for now */
        } else {
            /* Total match.  Process everything up to this point, */
            /* and call the macro. */
            handle_input_string(s + input_start, key_start - input_start);
            key_start = input_start = place;
            do_macro(n->u.macro, "");
            n = keytrie;
        }
    }

    /* Process everything up to a possible match. */
    handle_input_string(s + input_start, key_start - input_start);

    /* Shift the window if there's no pending partial match. */
    if (!s[key_start]) {
        Stringterm(current_input, 0);
        place = key_start = 0;
    }
    input_start = key_start;
}

/* Update the input window and keyboard buffer. */
void handle_input_string(input, len)
    char *input;
    unsigned int len;
{
    extern int input_cursor;
    extern Sock *fsock;
    int i, j;
    char save;

    if (len == 0) return;
    if (!input_cursor) ipos();
    for (i = j = 0; i < len; i++) {
        if (isprint(input[i])) input[j++] = input[i];
    }
    save = input[len = j];
    input[len] = '\0';
    if (!fsock || fsock->flags & SOCKECHO) iputs(input);
    input[len] = save;

    if (keyboard_pos == keybuf->len) {                    /* add to end */
        Stringncat(keybuf, input, len);
    } else if (insert) {                                  /* insert in middle */
        Stringcpy(scratch, keybuf->s + keyboard_pos);
        Stringterm(keybuf, keyboard_pos);
        Stringncat(keybuf, input, len);
        SStringcat(keybuf, scratch);
    } else if (keyboard_pos + len < keybuf->len) {        /* overwrite */
        for (i = 0, j = keyboard_pos; i < len; keybuf->s[j++] = input[i++]);
    } else {                                              /* write past end */
        Stringterm(keybuf, keyboard_pos);
        Stringncat(keybuf, input, len);
    }                      
    keyboard_pos += len;
}


/*
 *  Builtin key functions.
 */

void do_newline()
{
    clear_refresh_pending();
    reset_outcount();
    inewline();
    SStringcpy(scratch, keybuf);
    Stringterm(keybuf, keyboard_pos = 0);
    process_buffer(scratch);
    if (visual) ipos();
}

void do_recallb()
{
    recall_input(keybuf, -1);
    keyboard_pos = keybuf->len;
    do_replace();
}

void do_recallf()
{
    recall_input(keybuf, 1);
    keyboard_pos = keybuf->len;
    do_replace();
}

void do_socketb()
{
    movesock(-1);
}

void do_socketf()
{
    movesock(1);
}

void do_bspc()
{
    if (!keyboard_pos) return;
    Stringcpy(scratch, keybuf->s + keyboard_pos);
    SStringcat(Stringterm(keybuf, --keyboard_pos), scratch);
    ibs();
}

void do_bword()
{
    int place;

    if (!keyboard_pos) return;
    place = keyboard_pos - 1;
    while(isspace(keybuf->s[place])) place--;
    while(!isspace(keybuf->s[place])) place--;
    place++;
    if (place < 0) place = 0;
    Stringcpy(scratch, keybuf->s + keyboard_pos);
    SStringcat(Stringterm(keybuf, place), scratch);
    ibackword(place);
    keyboard_pos = place;
}

void do_dline()
{
    Stringterm(keybuf, keyboard_pos = 0);
    do_replace();
}

void do_dch()
{
    if (keyboard_pos == keybuf->len) return;
    newpos(keyboard_pos + 1);
    do_bspc();
}

void do_redraw()
{
    extern int screen_setup;

    if (!screen_setup || !visual) return;
    clr();
    setup_screen();
    do_replace();
}

void do_up()
{
    newpos(keyboard_pos - getwrap());
}

void do_down()
{
    newpos(keyboard_pos + getwrap());
}

void do_left()
{
    newpos(keyboard_pos - 1);
}

void do_right()
{
    newpos(keyboard_pos + 1);
}

void do_home()
{
    newpos(0);
}

void do_end()
{
    newpos(keybuf->len);
}

void do_wleft()
{
    int place;

    place = keyboard_pos - 1;
    while(isspace(keybuf->s[place])) place--;
    while(!isspace(keybuf->s[place])) if (--place < 0) break;
    place++;
    newpos(place);
}

void do_wright()
{
    int place;

    place = keyboard_pos;
    while (!isspace(keybuf->s[place]))
        if (++place > keybuf->len) break;
    while (isspace(keybuf->s[place])) place++;
    newpos(place);
}

void do_deol()
{
    dEOL();
    Stringterm(keybuf, keyboard_pos);
}

void do_lnext()
{
    literal_next = TRUE;
}


static void process_buffer(inbuf)
    Stringp inbuf;
{
    extern int sub, concat;
    char *ptr, *start;
    STATIC_BUFFER(buffer)

    if (concat) {
        if (inbuf->s[0] == '.' && inbuf->len == 1) {
            SStringcpy(inbuf, cat_keybuf);
            Stringterm(cat_keybuf, 0);
            concat = 0;
        } else {
            SStringcat(cat_keybuf, inbuf);
            if (concat == 2) Stringcat(cat_keybuf, "%;");
            return;
        }
    }

    if (kecho) tfprintf(tferr, "%s%S", getvar("kprefix"), inbuf);
    else if (inbuf->len == 0 && visual) oputs("");

    if (*inbuf->s == '^') {
        history_sub(inbuf->s + 1);
        return;
    }

    record_input(inbuf->s);

    if (inbuf->len == 0 && snarf) return;

    if (sub == 0) {
        check_command(TRUE, inbuf);
    } else if (sub == 2) {
        process_macro(inbuf->s, "");
    } else {                                               /* sub == 1 */
        Stringterm(buffer, 0);
        ptr = inbuf->s;
        while (*ptr) {
            if (*ptr == '%' && sub) {
                if (*++ptr == '%')
                    while (*ptr == '%') Stringadd(buffer, *ptr++);
                else if (*ptr == '\\' || *ptr == ';') {
                    Stringadd(buffer, '\n');
                    ptr++;
                } else Stringadd(buffer, '%');
            } else if (*ptr == '\\') {
                ptr++;
                if (isdigit(*ptr)) Stringadd(buffer, strtochr(&ptr));
                else Stringadd(buffer, '\\');
            } else {
                for (start = ptr++; *ptr && *ptr != '%' && *ptr != '\\'; ptr++);
                Stringncat(buffer, start, ptr - start);
            }
        }
        check_command(TRUE, buffer);
    }
}

void cbreak_noecho_mode()
{
#ifdef USE_TERMIOS
    struct termios tty;

    if (ingetattr(0, &tty) < 0) {
        perror("tcgetattr");
        die("Can't get terminal mode.");
    }
    memcpy(&old_tty, &tty, sizeof(struct termios));  /* structure copy */
    tty.c_lflag &= ~(ECHO | ICANON);
    tty.c_iflag = IGNBRK | IGNPAR | ICRNL;
    tty.c_oflag = OPOST;
# ifdef ONLCR
    tty.c_oflag |= ONLCR;
# endif
    tty.c_lflag = ISIG;
    tty.c_cc[VMIN] = 0;
    tty.c_cc[VTIME] = 0;
    if (insetattr(0, &tty) < 0) {
        perror("tcsetattr");
        die("Can't set terminal mode.");
    }
#endif

#ifdef USE_HPUX_TERMIO
    struct termio tty;

    if (ioctl(0, TCGETA, &tty) < 0) {
        perror("TCGETA ioctl");
        die("Can't get terminal mode.");
    }
    memcpy(&old_tty, &tty, sizeof(struct termio));  /* structure copy */
    tty.c_lflag &= ~(ECHO | ECHOE | ICANON);
    tty.c_cc[VMIN] = 0;
    tty.c_cc[VTIME] = 0;
    if (ioctl(0, TCSETAF, &tty) < 0) {
        perror("TCSETAF ioctl");
        die("Can't set terminal mode.");
    }
#endif

#ifdef USE_SGTTY
    struct sgttyb tty;

    if (ioctl(0, TIOCGETP, &tty) < 0) {
        perror("TIOCGETP ioctl");
        die("Can't get terminal mode.");
    }
    memcpy(&old_tty, &tty, sizeof(struct sgttyb));  /* structure copy */
    tty.sg_flags |= CBREAK;
    tty.sg_flags &= ~ECHO;
    if (ioctl(0, TIOCSETP, &tty) < 0) {
        perror("TIOCSETP ioctl");
        die("Can't set terminal mode.");
    }
#endif

    iscbreak = 1;
}

void reset_tty()
{
    if (iscbreak)
#ifdef USE_TERMIOS
        if (insetattr(0, &old_tty) < 0) perror("tcsetattr");
#endif
#ifdef USE_HPUX_TERMIO
        if (ioctl(0, TCSETAF, &old_tty) < 0) perror("TCSETAF ioctl");
#endif
#ifdef USE_SGTTY
        if (ioctl(0, TIOCSETP, &old_tty) < 0) perror("TIOCSETP ioctl");
#endif

    iscbreak = 0;
}
