/* $Id: dstring.c,v 30000.21 1993/05/18 08:55:05 kkeys Exp $ */
/******************************************************************
 * Copyright 1993 by Ken Keys.
 * Permission is granted to obtain and redistribute this code freely.
 * All redistributions must contain this header.  You may modify this
 * software, but any redistributions of modified code must be clearly
 * marked as having been modified.
 ******************************************************************/


/*********************************************************************
 * Fugue dynamically allocated string handling                       *
 *                                                                   *
 * Stringinit() must be used to initialize a dynamically allocated   *
 * string, and Stringfree() to free up the contents so as to prevent *
 * memory leaks.                                                     *
 *                                                                   *
 *********************************************************************/

#include <ctype.h>
#include "port.h"
#include "tf.h"
#include "malloc.h"
#include "dstring.h"
#include "util.h"

#define ALLOCSIZE 32L

#define lcheck(str) do { if ((str)->len > (str)->maxlen) resize(str); } while(0)

static void  FDECL(resize,(Stringp str));

#ifdef DMALLOC
String *FDECL(dStringinit,(Stringp str, char *file, int line));
#else
String *FDECL(Stringinit,(Stringp str));
#endif
void   FDECL(Stringfree,(Stringp str));
String *FDECL(Stringadd,(Stringp str, int c));
String *FDECL(Stringnadd,(Stringp str, int c, unsigned int n));
String *FDECL(Stringterm,(Stringp str, unsigned int len));
String *FDECL(Stringcpy,(Stringp dest, char *src));
String *FDECL(SStringcpy,(Stringp dest, Stringp src));
String *FDECL(Stringncpy,(Stringp dest, char *src, unsigned int len));
String *FDECL(Stringcat,(Stringp dest, char *src));
String *FDECL(SStringcat,(Stringp dest, Stringp src));
String *FDECL(Stringncat,(Stringp dest, char *src, unsigned int len));
char   *FDECL(getword,(Stringp str, char *line));
String *FDECL(stripString,(Stringp str));
String *FDECL(newline_package,(Stringp buffer, unsigned int n));

#ifdef DMALLOC
String *dStringinit(str, file, line)
    Stringp str;
    char *file;
    int line;
{
    str->s = (char *) dmalloc(ALLOCSIZE * sizeof(char), file, line);
    str->s[0] = '\0';
    str->len = 0;
    str->maxlen = ALLOCSIZE - 1;
    return str;
}
#else
String *Stringinit(str)
    Stringp str;
{
    str->s = (char *) MALLOC(ALLOCSIZE * sizeof(char));
    str->s[0] = '\0';
    str->len = 0;
    str->maxlen = ALLOCSIZE - 1;
    return str;
}
#endif

void Stringfree(str)
    Stringp str;
{
    FREE(str->s);
    str->maxlen = str->len = -1;
}

static void resize(str)
    Stringp str;
{
    int size;

    size = (str->len + 1) / ALLOCSIZE + 1;
    str->maxlen = size * ALLOCSIZE - 1;
    str->s = (char *) REALLOC(str->s, ((str->maxlen + 1) * sizeof(char)));
}

String *Stringadd(str, c)
    Stringp str;
    char c;
{
    str->len++;
    lcheck(str);
    str->s[str->len - 1] = c;
    str->s[str->len] = '\0';
    return str;
}

String *Stringnadd(str, c, n)
    Stringp str;
    char c;
    unsigned int n;
{
    if (n <= 0) return str;
    str->len += n;
    lcheck(str);
    while (n) str->s[str->len - n--] = c;
    str->s[str->len] = '\0';
    return str;
}

String *Stringterm(str, len)
    Stringp str;
    unsigned int len;
{
    if (str->len <= len) return str;
    str->len = len;
    str->s[len] = '\0';
    return str;
}

String *Stringcpy(dest, src)
    Stringp dest;
    char *src;
{
    dest->len = strlen(src);
    lcheck(dest);
    strcpy(dest->s, src);
    return dest;
}

String *SStringcpy(dest, src)
    Stringp dest, src;
{
    dest->len = src->len;
    lcheck(dest);
    strcpy(dest->s, src->s);
    return dest;
}

String *Stringncpy(dest, src, len)
    Stringp dest;
    char *src;
    unsigned int len;
{
    dest->len = len;
    lcheck(dest);
    strncpy(dest->s, src, len);
    dest->s[dest->len] = '\0';
    return dest;
}

String *Stringcat(dest, src)
    Stringp dest;
    char *src;
{
    unsigned int len = dest->len;

    dest->len += strlen(src);
    lcheck(dest);
    strcpy(dest->s + len, src);
    return dest;
}

String *SStringcat(dest, src)
    Stringp dest, src;
{
    unsigned int len = dest->len;

    dest->len += src->len;
    lcheck(dest);
    strcpy(dest->s + len, src->s);
    return dest;
}

String *Stringncat(dest, src, len)
    Stringp dest;
    char *src;
    unsigned int len;
{
    unsigned int oldlen = dest->len;

    dest->len += len;
    lcheck(dest);
    strncpy(dest->s + oldlen, src, len);
    dest->s[dest->len] = '\0';
    return dest;
}

char *getword(str, line)
    Stringp str;
    char *line;
{
#ifdef QUOTED_ARGS
    char qchar;

    Stringterm(str, 0);
    while (isspace(*line)) line++;
    if (*line == '\'' || *line == '\"') qchar = *line++;
    else qchar = '\0';
    while (*line) {
        if (*line == qchar) {
            line++;
            break;
        }
        if (qchar == '\0' && isspace(*line)) break;
        if (*line == '\\') if (*++line == '\0') --line;
        Stringadd(str, *line++);
    }
    return line;
#else
    char *ptr;

    while (isspace(*line)) line++;
    for (ptr = line; *ptr && !isspace(*ptr); ptr++);
    Stringncpy(str, line, ptr - line);
    return ptr;
#endif
}

String *stripString(str)
    Stringp str;
{
    stripstr(str->s);
    str->len = strlen(str->s);
    return str;
}

/* make sure buffer ends in <n> newlines */
String *newline_package(buffer, n)
    Stringp buffer;
    unsigned int n;
{
    while (buffer->len > 0 && buffer->s[buffer->len - 1] == '\n')
        buffer->len--;
    buffer->s[buffer->len] = '\0';
    Stringnadd(buffer, '\n', n);
    return buffer;
}

