/* $Id: command1.c,v 30000.26 1993/05/25 00:50:49 kkeys Exp $ */
/******************************************************************
 * Copyright 1993 by Ken Keys.
 * Permission is granted to obtain and redistribute this code freely.
 * All redistributions must contain this header.  You may modify this
 * software, but any redistributions of modified code must be clearly
 * marked as having been modified.
 ******************************************************************/


/*****************************************************************
 * Fugue command handlers, part 1                                *
 *                                                               *
 * Contents:                                                     *
 * 1. Command table and declarations                             *
 * 2. Macro handlers                                             *
 * 3. Command handlers:                                          *
 *     A. Help subsystem                                         *
 *     B. Worlds                                                 *
 *     C. Processes (/quote and /repeat and related commands)    *
 *     D. Watchdog routines                                      *
 *     E. Miscellaneous routines                                 *
 *                                                               *
 * The other module contains flag and numeric runtime option     *
 * handers, and all string-processing subsystem command (macros, *
 * triggers, hilites, and gags).                                 *
 *                                                               *
 *****************************************************************/

#include <signal.h>
#include <ctype.h>
#include "port.h"
#include "dstring.h"
#include "tf.h"
#include "util.h"
#include "history.h"
#include "world.h"
#include "socket.h"
#include "output.h"
#include "macro.h"
#include "help.h"
#include "process.h"
#include "keyboard.h"
#include "expand.h"
#include "search.h"

#ifdef HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif

/* POSIX.1 systems should have these in sys/wait.h, but apparently some don't.*/
#ifndef WIFEXITED
# define WIFEXITED(w)  (((*(int *)&(w)) & 0177) == 0)   /* works most places */
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(w)  ((*(int *)&(w)) >> 8)          /* works most places */
#endif

Handler *FDECL(find_command,(char *cmd));

int FDECL(handle_command,(char *cmdline));

static HANDLER (handle_control);
static int FDECL(do_set,(char *args, int exportflag, int localflag));


static HANDLER (handle_addworld_command);
static HANDLER (handle_dc_command);
static HANDLER (handle_echo_command);
static HANDLER (handle_export_command);
static HANDLER (handle_eval_command);
static HANDLER (handle_grab_command);
static HANDLER (handle_help_command);
static HANDLER (handle_input_command);
static HANDLER (handle_kill_command);
static HANDLER (handle_lcd_command);
static HANDLER (handle_let_command);
static HANDLER (handle_listsockets_command);
static HANDLER (handle_listworlds_command);
static HANDLER (handle_ps_command);
static HANDLER (handle_purgeworld_command);
static HANDLER (handle_quit_command);
static HANDLER (handle_quote_command);
static HANDLER (handle_recall_command);
static HANDLER (handle_repeat_command);
static HANDLER (handle_saveline_command);
static HANDLER (handle_saveworld_command);
static HANDLER (handle_send_command);
static HANDLER (handle_set_command);
static HANDLER (handle_setenv_command);
static HANDLER (handle_sh_command);
static HANDLER (handle_test_command);
static HANDLER (handle_trigger_command);
static HANDLER (handle_unset_command);
static HANDLER (handle_unworld_command);
static HANDLER (handle_version_command);
static HANDLER (handle_world_command);

extern HANDLER (handle_bamf_command);
extern HANDLER (handle_beep_command);
extern HANDLER (handle_bind_command);
extern HANDLER (handle_cat_command);
extern HANDLER (handle_def_command);
extern HANDLER (handle_dokey_command);
extern HANDLER (handle_edit_command);
extern HANDLER (handle_gag_command);
extern HANDLER (handle_hilite_command);
extern HANDLER (handle_hook_command);
extern HANDLER (handle_kecho_command);
extern HANDLER (handle_list_command);
extern HANDLER (handle_load_command);
extern HANDLER (handle_log_command);
extern HANDLER (handle_mecho_command);
extern HANDLER (handle_nogag_command);
extern HANDLER (handle_nohilite_command);
extern HANDLER (handle_purge_command);
extern HANDLER (handle_qecho_command);
extern HANDLER (handle_save_command);
extern HANDLER (handle_sub_command);
extern HANDLER (handle_suspend_command);
extern HANDLER (handle_trig_command);
extern HANDLER (handle_trigc_command);
extern HANDLER (handle_trigp_command);
extern HANDLER (handle_trigpc_command);
extern HANDLER (handle_unbind_command);
extern HANDLER (handle_undef_command);
extern HANDLER (handle_undefn_command);
extern HANDLER (handle_undeft_command);
extern HANDLER (handle_unhook_command);
extern HANDLER (handle_untrig_command);
extern HANDLER (handle_watchdog_command);
extern HANDLER (handle_watchname_command);
extern HANDLER (handle_wrap_command);

typedef struct Command {
    char *name;
    Handler *func;
} Command;

  /* It is IMPORTANT that the commands be in alphabetical order! */

static Command cmd_table[] =
{
  { "ADDWORLD"    , handle_addworld_command    },
  { "BAMF"        , handle_bamf_command        },
  { "BEEP"        , handle_beep_command        },
  { "BIND"        , handle_bind_command        },
  { "BREAK"       , handle_control             },
  { "CAT"         , handle_cat_command         },
  { "DC"          , handle_dc_command          },
  { "DEF"         , handle_def_command         },
  { "DO"          , handle_control             },
  { "DOKEY"       , handle_dokey_command       },
  { "DONE"        , handle_control             },
  { "ECHO"        , handle_echo_command        },
  { "EDIT"        , handle_edit_command        },
  { "ELSE"        , handle_control             },
  { "ELSEIF"      , handle_control             },
  { "ENDIF"       , handle_control             },
  { "EVAL"        , handle_eval_command        },
  { "EXPORT"      , handle_export_command      },
  { "GAG"         , handle_gag_command         },
  { "GRAB"        , handle_grab_command        },
  { "HELP"        , handle_help_command        },
  { "HILITE"      , handle_hilite_command      },
  { "HOOK"        , handle_hook_command        },
  { "IF"          , handle_control             },
  { "INPUT"       , handle_input_command       },
  { "KECHO"       , handle_kecho_command       },
  { "KILL"        , handle_kill_command        },
  { "LCD"         , handle_lcd_command         },
  { "LET"         , handle_let_command         },
  { "LIST"        , handle_list_command        },
  { "LISTSOCKETS" , handle_listsockets_command },
  { "LISTWORLDS"  , handle_listworlds_command  },
  { "LOAD"        , handle_load_command        },
  { "LOG"         , handle_log_command         },
  { "MECHO"       , handle_mecho_command       },
  { "NOGAG"       , handle_nogag_command       },
  { "NOHILITE"    , handle_nohilite_command    },
  { "PS"          , handle_ps_command          },
  { "PURGE"       , handle_purge_command       },
  { "PURGEWORLD"  , handle_purgeworld_command  },
  { "QECHO"       , handle_qecho_command       },
  { "QUIT"        , handle_quit_command        },
  { "QUOTE"       , handle_quote_command       },
  { "RECALL"      , handle_recall_command      },
  { "REPEAT"      , handle_repeat_command      },
  { "SAVE"        , handle_save_command        },
  { "SAVELINE"    , handle_saveline_command    },
  { "SAVEWORLD"   , handle_saveworld_command   },
  { "SEND"        , handle_send_command        },
  { "SET"         , handle_set_command         },
  { "SETENV"      , handle_setenv_command      },
  { "SH"          , handle_sh_command          },
  { "SUB"         , handle_sub_command         },
  { "SUSPEND"     , handle_suspend_command     },
  { "TEST"        , handle_test_command        },  
  { "THEN"        , handle_control             },
  { "TRIG"        , handle_trig_command        },
  { "TRIGC"       , handle_trigc_command       },
  { "TRIGGER"     , handle_trigger_command     },  
  { "TRIGP"       , handle_trigp_command       },
  { "TRIGPC"      , handle_trigpc_command      },
  { "UNBIND"      , handle_unbind_command      },
  { "UNDEF"       , handle_undef_command       },
  { "UNDEFN"      , handle_undefn_command      },
  { "UNDEFT"      , handle_undeft_command      },
  { "UNHOOK"      , handle_unhook_command      },
  { "UNSET"       , handle_unset_command       },
  { "UNTRIG"      , handle_untrig_command      },
  { "UNWORLD"     , handle_unworld_command     },
  { "VERSION"     , handle_version_command     },
  { "WATCHDOG"    , handle_watchdog_command    },
  { "WATCHNAME"   , handle_watchname_command   },
  { "WHILE"       , handle_control             },
  { "WORLD"       , handle_world_command       },
  { "WRAP"        , handle_wrap_command        },
};

#define NUM_CMDS (sizeof(cmd_table) / sizeof(Command))

/*****************************************
 * Find, process and run commands/macros *
 *****************************************/

int handle_command(cmd_line)
    char *cmd_line;
{
    char *cmd, *args;
    Handler *handler;
    Macro *macro;
    int result;

    while (*cmd_line == '/') cmd_line++;
    if (!*cmd_line || isspace(*cmd_line)) return 0;
    cmd = STRDUP(cmd_line);
    for (args = cmd; *args && !isspace(*args); args++);
    if (*args) *args++ = '\0';
    stripstr(cmd);
    stripstr(args);
    if (*cmd == '@') {
        if ((handler = find_command(cmd + 1))) {
            result = (*handler)(args);
        } else {
            tfprintf(tferr, "%% %s: not a builtin command", cmd + 1);
            result = 0;
        }
    } else if ((macro = find_macro(cmd))) {
        result = do_macro(macro, args);
    } else if ((handler = find_command(cmd))) {
        result = (*handler)(args);
    } else {
        tfprintf(tferr, "%% %s: no such command or macro", cmd);
        result = 0;
    }
    FREE(cmd);
    return result;
}

Handler *find_command(cmd)
    char *cmd;
{
    int i;

    i = binsearch(cmd, (GENERIC*)cmd_table, NUM_CMDS, sizeof(Command), cstrcmp);
    return (i < 0) ? NULL : cmd_table[i].func;
}

static int handle_trigger_command(args)
    char *args;
{
    if (background) background++;  /* Nesting count. Prevents BACKGROUND hook */
    if (borg) check_trigger(args);
    if (background) background--;
    return 1;
}

static int handle_control(args)
    char *args;
{
    tfputs("% control statement illegal at top level.", tferr);
    return 0;
}

static int handle_help_command(args)
    char *args;
{
    return do_help(args);
}

/**********
 * Worlds *
 **********/

static int handle_listworlds_command(args)
    char *args;
{
    int full = FALSE;
    char c;

    startopt(args, "c");
    while ((c = nextopt(&args, NULL))) {
        switch (c) {
            case 'c':  full = TRUE; break;
            default:   return 0;
        }
    }
    if (*args && !smatch_check(args)) return 0;
    list_worlds(full, *args ? args : NULL, NULL);
    return 1;
}

static int handle_listsockets_command(args)
    char *args;
{
    if (*args) tfputs("% Arguments disregarded.", tferr);
    listsockets();
    return 1;
}

static int handle_world_command(args)
    char *args;
{
    World *where = NULL;
    int autologin;
    char *port;

    autologin = login;
    if (*args == '-') {
        autologin = FALSE;
        args++;
    }

    if (!*args) {
        if ((where = get_world_header()) == NULL)
            tfputs("% No default world is set.", tferr);
    } else if (cstrcmp(args, "NULL") == 0) {
        no_sock();
        return 1;
    } else if ((port = strchr(args, ' ')) == NULL) {
        if ((where = find_world(args)) == NULL)
            do_hook(H_CONFAIL, "%% Connection to %s failed: %s", "%s %s",
                args, "world unknown");
    } else {
        for (*port++ = '\0'; isspace(*port); port++);
        if (strchr(port, ' ')) {
            tfputs("% Too many arguments.", tferr);
        } else {
#ifdef RESTRICT_WORLD
            tfputs("% \"/world <host> <port>\" restricted", tferr);
#else
            where = new_world(NULL, "", "", args, port, "");
            where->flags |= WORLD_TEMP;
#endif
        }
    }

    if (!where) return 0;
    connect_to(where, autologin);
    return 1;
}

static int handle_addworld_command(args)
    char *args;
{
#ifdef RESTRICT_WORLD
    extern int configured;
    if (configured) {
        tfputs("% /addworld restricted", tferr);
        return 0;
    }
#endif
    return *args && addworld(args);
}

static int handle_unworld_command(args)
    char *args;
{
    return *args && remove_world(args);
}

static int handle_purgeworld_command(args)
    char *args;
{
    return *args && purge_world(args);
}

static int handle_saveworld_command(args)
    char *args;
{
#ifdef RESTRICT_FILE
    tfputs("% /saveworld: restricted", tferr);
    return 0;
#else
    return write_worlds(args);
#endif
}

static int handle_dc_command(args)
    char *args;
{
    return disconnect(args);
}

/*************
 * Processes *
 *************/

static int handle_quote_command(args)  /* quote file/command/history */
    char *args;
{
    return *args && start_quote(args);
}

static int handle_repeat_command(args) /* repeat action n times */
    char *args;
{
    return *args && start_repeat(args);
}

static int handle_kill_command(args)
    char *args;
{
    return *args && do_kill(atoi(args));
}

static int handle_ps_command(args)
    char *args;
{
    do_ps();
    return 1;
}

/*************
 * Variables *
 *************/

static int do_set(args, exportflag, localflag)
    char *args;
    int exportflag, localflag;
{
    char *value = "";

    if (!*args) {
        if (localflag) return 0;
        listvar(exportflag);
        return 1;
    } else if ((value = strchr(args, '='))) {
        *value++ = '\0';
        if (!*args) {
            tfputs("% missing variable name", tferr);
            return 0;
        }
    } else if ((value = strchr(args, ' '))) {
        for (*value++ = '\0'; *value && isspace(*value); value++);
    } else {
        value = localflag ? getlocalvar(args) : getvar(args);
        if (value) {
            oprintf("%% %s=%s", args, value);
            return 1;
        } else {
            oprintf("%% %s not defined at %s level", args,
                localflag ? "local" : "global");
            return 0;
        }
    }
    if (localflag) return do_let(args, value);
    else return setvar(args, value, exportflag);
}

static int handle_set_command(args)
    char *args;
{
    return do_set(args, FALSE, FALSE);
}

static int handle_setenv_command(args)
    char *args;
{
    return do_set(args, TRUE, FALSE);
}

static int handle_unset_command(args)
    char *args;
{
    unsetvar(args);
    return 1;
}

static int handle_let_command(args)
    char *args;
{
    return do_set(args, FALSE, TRUE);
}

static int handle_export_command(args)
    char *args;
{
    return export(args);
}

/********
 * Misc *
 ********/

static int handle_eval_command(args)
    char *args;
{
    return process_macro(args, "");
}

static int handle_quit_command(args)
    char *args;
{
    set_done();
    return 1;
}

static int handle_recall_command(args)
    char *args;
{
    return recall_history(args, tfout);
}

static int handle_sh_command(args)
    char *args;
{
#ifdef RESTRICT_SHELL
    tfputs("% /sh:  restricted", tferr);
    return 0;
#else
    char *cmd;
    int result;

    if (*args) {
        cmd = args;
        do_hook(H_SHELL, "%% Executing %s: \"%s\"", "%s %s", "command", cmd);
    } else {
        if ((cmd = getvar("SHELL")) == NULL) cmd = "/bin/sh";
        do_hook(H_SHELL, "%% Executing %s: %s", "%s %s", "shell", cmd);
    }
    if (visual) fix_screen();
    reset_tty();
    result = system(cmd);
    result = (WIFEXITED(result)) ? WEXITSTATUS(result) : -1;
    cbreak_noecho_mode();
    if (shpause) {
        oputs("% Done-- press a key to return.");
        igetchar();
    }
    get_window_size();
    setup_screen();
    do_hook(H_RESUME, "%% Resuming TinyFugue", "");
    if (maildelay > 0) check_mail();
    return result;
#endif
}

static int handle_version_command(args)
    char *args;
{
    extern char version[];
    oprintf("%% %s.", version);
    return 1;
}

#ifndef PATH_MAX
# ifdef MAXPATHLEN
#  define PATH_MAX MAXPATHLEN
# else
#  define PATH_MAX 1024
# endif
#endif

static int handle_lcd_command(args)
    char *args;
{
    char buffer[PATH_MAX + 1];
    STATIC_BUFFER(dirname)

    expand_filename(Stringcpy(dirname, args));
    if (dirname->len && chdir(dirname->s) < 0) {
        operror(dirname->s);
        return 0;
    }

#ifdef HAVE_GETCWD
    oprintf("%% Current directory is %s", getcwd(buffer, PATH_MAX));
#else
# ifdef HAVE_GETWD
    oprintf("%% Current directory is %s", getwd(buffer));
# endif
#endif
    return 1;
}

static int handle_send_command(args)
    char *args;
{
    return *args && do_send(args);
}

static int handle_echo_command(args)
    char *args;
{
    char c;
    short attrs = 0, wflag = 0;
    World *world = NULL;

    startopt(args, "a:w:");
    while ((c = nextopt(&args, NULL))) {
        switch (c) {
        case 'a': case 'f':
            if ((attrs |= parse_attrs(args)) < 0) return 0;
            break;
        case 'w':
            wflag = 1;
            if (!*args) world = xworld();
            else if ((world = find_world(args)) == NULL) {
                tfprintf(tferr, "%% World %s not found.", args);
                return 0;
            }
            break;
        default:
            return 0;
        }
    }
    if (!wflag) oputa(new_aline(args, attrs | F_NEWLINE));
    else if (world) world_output(world, args, attrs);
    else {
        tfputs("% No current world.", tferr);
        return 0;
    }
    return 1;
}

static int handle_saveline_command(args)
    char *args;
{
    return do_saveline(args);
}

static int handle_input_command(args)
    char *args;
{
    handle_input_string(args, strlen(args));
    return 1;
}

static int handle_grab_command(args)
    char *args;
{
    extern int keyboard_pos;
    extern Stringp keybuf;

    Stringcpy(keybuf, args);
    keyboard_pos = keybuf->len;
    if (visual && is_refresh_pending()) do_line_refresh();
    do_replace();
    return 1;
}

static int handle_test_command(args)
    char *args;
{
    return test(args);
}

