/* player_list.c */

#include <ctype.h>
#include "copyright.h"

#include "config.h"
#include "db.h"
#include "interface.h"
#include "globals.h"
#include "externs.h"


#define PLAYER_LIST_SIZE (1 << 12)	/* must be a power of 2 */

static dbref hash_function_table[256];
static int hft_initialized = 0;

struct pl_elt {
  dbref player;			/* pointer to player */
				/* key is db[player].name or alias */
  struct pl_elt *next;
};

static void init_hft()
{
  int i;

  for (i = 0; i < 256; i++) {
    hash_function_table[i] = random() & (PLAYER_LIST_SIZE - 1);
  }
  hft_initialized = 1;
}

static dbref hash_function(string)
    const char *string;
{
  dbref hash;
  if (!hft_initialized)
    init_hft();
  hash = 0;
  for (; *string; string++) {
    hash ^= ((hash >> 1) ^ hash_function_table[DOWNCASE(*string)]);
  }
  return (hash);
}

static struct pl_elt *player_list[PLAYER_LIST_SIZE];
static int pl_used = 0;
static struct pl_elt *palias_list[PLAYER_LIST_SIZE];
static int al_used = 0;

void clear_players()
{
    int i;
    struct pl_elt *e, *next;

    for (i = 0; i < PLAYER_LIST_SIZE; i++) {
	if (pl_used) {
	    for (e = player_list[i]; e; e = next) {
		next = e->next;
		free((char *) e);
	    }
	}
	if (al_used) {
	    for (e = palias_list[i]; e; e = next) {
		next = e->next;
		free(e);
	    }
	}
	player_list[i] = palias_list[i] = 0;
    }
    pl_used = al_used = 1;
}

void add_player(player, alias)
     dbref player;
     char *alias;
{
    dbref hash;
    struct pl_elt *e;

    e = (struct pl_elt *) malloc(sizeof(struct pl_elt));
    e->player = player;
    if (alias == NULL) {
	hash = hash_function(Name(player));
	e->next = player_list[hash];
	player_list[hash] = e;
    } else {
	hash = hash_function(alias);
	e->next = palias_list[hash];
	palias_list[hash] = e;
    }
}

dbref lookup_player(name)
    const char *name;
{
    struct pl_elt *e;
    dbref hash;
    ATTR *a;
    char alias[PLAYER_NAME_LIMIT];
 
    hash = hash_function(name);

    for (e = player_list[hash]; e; e = e->next) {
	if (!strcasecmp(Name(e->player), name))
	    return e->player;
    }
    for (e = palias_list[hash]; e; e = e->next) {
	if ((a = atr_get_noparent(e->player, "ALIAS")) != NULL) {
	    strcpy(alias, uncompress(a->value));
	    if (!strcasecmp(alias, name))
		return e->player;
	}
    }

    return NOTHING;
}

void delete_player(player, alias)
     dbref player;
     char *alias;
{
    dbref hash;
    struct pl_elt *prev, *e;

    if (alias == NULL) {
	hash = hash_function(Name(player));
	if ((e = player_list[hash]) == 0) {
	    return;
	} else if (e->player == player) {
	    /* it's the first one */
	    player_list[hash] = e->next;
	    free((char *) e);
	} else {
	    for (prev = e, e = e->next; e; prev = e, e = e->next) {
		if (e->player == player) {
		    /* got it */
		    prev->next = e->next;
		    free((char *) e);
		    break;
		}
	    }
	}
    } else {
	hash = hash_function(alias);
	if ((e = palias_list[hash]) == 0) {
	    return;
	} else if (e->player == player) {
	    /* it's the first one */
	    palias_list[hash] = e->next;
	    free((char *) e);
	} else {
	    for (prev = e, e = e->next; e; prev = e, e = e->next) {
		if (e->player == player) {
		    /* got it */
		    prev->next = e->next;
		    free((char *) e);
		    break;
		}
	    }
	}
    }
}
