/* flags.c */

/* Functions to cope with flags */

#include <sys/time.h>

#include "db.h"
#include "externs.h"
#include "interface.h"

#define FLAG_HASH_SIZE 256
#define FLAG_HASH_MASK 255

typedef struct flag_info FLAG;

struct flag_info {
  const char *name;
  char letter;
  int type;
  int flag;
  int perms;
  FLAG *next;
};

typedef struct flag_entry FLAGENT;

struct flag_entry {
  const char *name;
  FLAG *entry;
  FLAGENT *next;
};

typedef struct flag_alias FLAG_ALIAS;

struct flag_alias {
  const char *alias;
  const char *realname;
};

static FLAGENT *flag_hashtab[FLAG_HASH_SIZE];
  
FLAG flag_table[] =
{

  { "CHOWN_OK", 'C',	NOTYPE,		CHOWN_OK,	F_ANY },
  { "DARK", 	'D', 	NOTYPE, 	DARK,		F_ANY },
  { "GOING",	'G',	NOTYPE,		GOING,		F_ANY },
  { "HAVEN", 	'H', 	NOTYPE, 	HAVEN,		F_ANY },
#ifdef INHERIT_FLAG
  { "INHERIT",	'I',	NOTYPE,		INHERIT,	F_INHERIT },
#endif
  { "LINK_OK", 	'L', 	NOTYPE, 	LINK_OK,	F_ANY },
  { "OPAQUE",	'O',	NOTYPE,		OPAQUE,		F_ANY },
  { "QUIET",    'Q',	NOTYPE,		QUIET,		F_ANY },
  { "STICKY",	'S',	NOTYPE,		STICKY,		F_ANY },
  { "UNFINDABLE", 'U',	NOTYPE,		UNFIND,		F_ANY },
  { "VISUAL",	'V',	NOTYPE,		VISUAL,		F_ANY },
  { "WIZARD",	'W', 	NOTYPE, 	WIZARD,		F_INHERIT | F_WIZARD },
  { "SAFE",	'X',	NOTYPE,		SAFE,		F_ANY },
  { "AUDIBLE",  'a',	NOTYPE,		AUDIBLE,	F_ANY },
  { "DEBUG",	'b',	NOTYPE,		DEBUGGING,     	F_INHERIT | F_ROYAL },
  { "ENTER_OK", 'e',	NOTYPE,		ENTER_OK,	F_ANY },
  { "HALT", 	'h',    NOTYPE,		HALT,		F_ANY },
  { "NO_COMMAND", 'n',	NOTYPE,		NO_COMMAND,	F_ANY },
#ifdef ROYALTY_FLAG
  { "ROYALTY",	'r',	NOTYPE,		ROYALTY,	F_INHERIT | F_ROYAL },
#endif
  { "TRANSPARENT", 't', NOTYPE,		TRANSPARENT,	F_ANY },
  { "VERBOSE",	'v',	NOTYPE,		VERBOSE,	F_ANY },
  { "STARTUP",  'z',	NOTYPE,		STARTUP,	F_INTERNAL },

  { "ANSI",	'A', 	TYPE_PLAYER,	PLAYER_ANSI,	F_ANY },
  { "MONITOR",	'M',	TYPE_PLAYER,	PLAYER_MONITOR, F_ROYAL },
  { "NOSPOOF",	'N',	TYPE_PLAYER,	PLAYER_NOSPOOF,	F_ANY },
  { "ZONE",     'Z',    TYPE_PLAYER,    PLAYER_ZONE,    F_ANY },
  { "CONNECTED", 'c',	TYPE_PLAYER,	PLAYER_CONNECT,	F_INTERNAL },
  { "GAGGED",	'g',	TYPE_PLAYER,	PLAYER_GAGGED,	F_WIZARD },
  { "MYOPIC",	'm',	TYPE_PLAYER,	PLAYER_MYOPIC,  F_ANY },
  { "TERSE",	'x',	TYPE_PLAYER,	PLAYER_TERSE,	F_ANY },
  { "SUSPECT",	'u',	TYPE_PLAYER,	PLAYER_SUSPECT,	F_WIZARD },

  { "DESTROY_OK", 'd',	TYPE_THING,	THING_DEST_OK,	F_ANY },
  { "LISTENER",	'l',	TYPE_THING,	THING_LISTEN,	F_ANY },
  { "PUPPET",	'p',	TYPE_THING,	THING_PUPPET,	F_ANY },

  { "ABODE",	'A',	TYPE_ROOM,	ROOM_ABODE,	F_ANY },
  { "FLOATING",	'F',	TYPE_ROOM,	ROOM_FLOATING,	F_ANY },
  { "JUMP_OK",	'J',	TYPE_ROOM,	ROOM_JUMP_OK,	F_ANY },
  { "NO_TEL",	'N',	TYPE_ROOM,	ROOM_NO_TEL,	F_ANY },
  { "TEMPLE",	'T',	TYPE_ROOM,	ROOM_TEMPLE,	F_ANY },

  { "ACCESSED", '\0',	NOTYPE,		ACCESSED,	F_INTERNAL },
  { "MARKED",	'\0',	NOTYPE,		MARKED,		F_INTERNAL },
  { NULL, 	'\0', 	0,		0,		0 }
};

FLAG type_table[] =
{
  { "PLAYER",	'P',	TYPE_PLAYER,	TYPE_PLAYER,	F_INTERNAL },
  { "ROOM",	'R',	TYPE_ROOM,	TYPE_ROOM,	F_INTERNAL },
  { "EXIT",	'E',	TYPE_EXIT,	TYPE_EXIT,	F_INTERNAL },
  { "THING",	'T',	TYPE_THING,	TYPE_THING,	F_INTERNAL },
  { NULL,	'\0',	0,		0,		0 }
};

static FLAG_ALIAS flag_alias_tab[] =
{
	{ "CHOWN_O", 		"CHOWN_OK" },
	{ "CHOWN_",		"CHOWN_OK" },
 	{ "CHOWN", 		"CHOWN_OK" },
	{ "CHOW",		"CHOWN_OK" },
	{ "CHO", 		"CHOWN_OK" },
	{ "CH",		 	"CHOWN_OK" },
	{ "DAR",		"DARK" },
	{ "DA",			"DARK" },
	{ "GOIN",		"GOING" },
	{ "GOI",		"GOING" },
	{ "GO",			"GOING" },
	{ "HAVE",		"HAVEN" },
	{ "HAV",		"HAVEN" },
#ifdef INHERIT_FLAG
	{ "INHERI",		"INHERIT" },
	{ "INHER",		"INHERIT" },
	{ "INHE",		"INHERIT" },
	{ "INH",		"INHERIT" },
	{ "IN",			"INHERIT" },
	{ "I",			"INHERIT" },
#endif	
	{ "LINK_O",		"LINK_OK" },
	{ "LINK_",		"LINK_OK" },
	{ "LINK",		"LINK_OK" },
	{ "LIN",		"LINK_OK" },
	{ "OPAQU",		"OPAQUE" },
	{ "OPAQ",		"OPAQUE" },
	{ "OPA",		"OPAQUE" },
	{ "OP",			"OPAQUE" },
	{ "O",			"OPAQUE" },
	{ "QUIE",		"QUIET" },
	{ "QUI",		"QUIET" },
	{ "QU",			"QUIET" },
	{ "Q",			"QUIET" },
	{ "STICK",		"STICKY" },
	{ "STIC",		"STICKY" },
	{ "STI",		"STICKY" },
	{ "ST",			"STICKY" },
	{ "UNFINDABL",		"UNFINDABLE" },
	{ "UNFINDABL",		"UNFINDABLE" },
	{ "UNFINDAB",		"UNFINDABLE" },
	{ "UNFINDA",		"UNFINDABLE" },
	{ "UNFIND",		"UNFINDABLE" },
	{ "UNFIN",		"UNFINDABLE" },
	{ "UNFI",		"UNFINDABLE" },
	{ "UNF",		"UNFINDABLE" },
	{ "UN",			"UNFINDABLE" },
	{ "U",			"UNFINDABLE" },
	{ "VISUA",		"VISUAL" },
	{ "VISU",		"VISUAL" },
	{ "VIS",		"VISUAL" },
	{ "VI",			"VISUAL" },
	{ "WIZAR",		"WIZARD" },
	{ "WIZA",		"WIZARD" },
	{ "WIZ",		"WIZARD" },
	{ "WI",			"WIZARD" },
	{ "W",			"WIZARD" },
	{ "SAF",		"SAFE" },
	{ "SA",			"SAFE" },
	{ "AUDIBL",		"AUDIBLE" },
	{ "AUDIB",		"AUDIBLE" },
	{ "AUDI",		"AUDIBLE" },
	{ "AUD",		"AUDIBLE" },
	{ "AU",			"AUDIBLE" },
	{ "DEBU",		"DEBUG" },
	{ "DEB",		"DEBUG" },
	{ "ENTER_O",		"ENTER_OK" },
	{ "ENTER_",		"ENTER_OK" },
	{ "ENTER",		"ENTER_OK" },
	{ "ENTE",		"ENTER_OK" },
	{ "ENT",		"ENTER_OK" },
	{ "EN",			"ENTER_OK" },
	{ "HAL",		"HALT" },
	{ "NO_COMMAN",		"NO_COMMAND" },
	{ "NO_COMMA",		"NO_COMMAND" },
	{ "NO_COMM",		"NO_COMMAND" },
	{ "NO_COM",		"NO_COMMAND" },
	{ "NO_CO",		"NO_COMMAND" },
	{ "NO_C",		"NO_COMMAND" },
#ifdef ROYALTY_FLAG
	{ "ROYALT",		"ROYALTY" },
	{ "ROYAL",		"ROYALTY" },
	{ "ROYA",		"ROYALTY" },
	{ "ROY",		"ROYALTY" },
#endif
	{ "TRANSPAREN",		"TRANSPARENT" },
	{ "TRANSPARE",		"TRANSPARENT" },
	{ "TRANSPAR",		"TRANSPARENT" },
	{ "TRANSPA",		"TRANSPARENT" },
	{ "TRANSP",		"TRANSPARENT" },
	{ "TRANS",		"TRANSPARENT" },
	{ "TRAN",		"TRANSPARENT" },
	{ "TRA",		"TRANSPARENT" },
	{ "TR",			"TRANSPARENT" },
	{ "VERBOS",		"VERBOSE" },
	{ "VERBO",		"VERBOSE" },
	{ "VERB",		"VERBOSE" },
	{ "VER",		"VERBOSE" },
	{ "VE",			"VERBOSE" },
	{ "ANS",		"ANSI" },
	{ "AN",			"ANSI" },
	{ "MONITO",		"MONITOR" },
	{ "MONIT",		"MONITOR" },
	{ "MONI",		"MONITOR" },
	{ "MON",		"MONITOR" },
	{ "MO",			"MONITOR" },
	{ "NOSPOO",		"NOSPOOF" },
	{ "NOSPO",		"NOSPOOF" },
	{ "NOSP",		"NOSPOOF" },
	{ "NOS",		"NOSPOOF" },
	{ "ZON",		"ZONE" },
	{ "ZO",			"ZONE" },
	{ "Z",			"ZONE" },
	{ "CONNECTE",		"CONNECTED" },
	{ "CONNECT",		"CONNECTED" },
	{ "CONNEC",		"CONNECTED" },
	{ "CONNE",		"CONNECTED" },
	{ "CONN",		"CONNECTED" },
	{ "CON",		"CONNECTED" },
	{ "CO",			"CONNECTED" },
	{ "GAG",		"GAGGED" },
	{ "MYOPI",		"MYOPIC" },
	{ "MYOP",		"MYOPIC" },
	{ "MYO",		"MYOPIC" },
	{ "MY",			"MYOPIC" },
	{ "TERS",		"TERSE" },
	{ "TER",		"TERSE" },
	{ "SUSPEC",		"SUSPECT" },
	{ "SUSPE",		"SUSPECT" },
	{ "SUSP",		"SUSPECT" },
	{ "SUS",		"SUSPECT" },
	{ "SU",			"SUSPECT" },
	{ "DEST_OK",		"DESTROY_OK" },
	{ "DESTROY_O",		"DESTROY_OK" },
	{ "DESTROY_",		"DESTROY_OK" },
	{ "DESTROY",		"DESTROY_OK" },
	{ "DESTRO",		"DESTROY_OK" },
	{ "DESTR",		"DESTROY_OK" },
	{ "DEST",		"DESTROY_OK" },
	{ "DES",		"DESTROY_OK" },
	{ "DE",			"DESTROY_OK" },
	{ "LISTENE",		"LISTENER" },
	{ "LISTEN",		"LISTENER" },
	{ "LISTE",		"LISTENER" },
	{ "LIST",		"LISTENER" },
	{ "LIS",		"LISTENER" },
	{ "PUPPE",		"PUPPET" },
	{ "PUPP",		"PUPPET" },
	{ "PUP",		"PUPPET" },
	{ "PU",			"PUPPET" },
	{ "P",			"PUPPET" },
	{ "ABOD",		"ABODE" },
	{ "ABO",		"ABODE" },
	{ "AB",			"ABODE" },
	{ "FLOATIN",		"FLOATING" },
	{ "FLOATI",		"FLOATING" },
	{ "FLOAT",		"FLOATING" },
	{ "FLOA",		"FLOATING" },
	{ "FLO",		"FLOATING" },
	{ "FL",			"FLOATING" },
	{ "F",			"FLOATING" },
	{ "JUMP_O",		"JUMP_OK" },
	{ "JUMP_",		"JUMP_OK" },
	{ "JUMP", 		"JUMP_OK" },
	{ "JUM",		"JUMP_OK" },
	{ "JU",			"JUMP_OK" },
	{ "J",			"JUMP_OK" },
	{ "NO_TE",		"NO_TEL" },
	{ "NO_T",		"NO_TEL" },
	{ "TEMPL",		"TEMPLE" },
	{ "TEMP",		"TEMPLE" },
	{ "TEM",		"TEMPLE" },
	{ "TE",			"TEMPLE" },
	{ NULL,			NULL }
};

/*---------------------------------------------------------------------------
 * Flag hash table handlers 
 */

static FLAG *flag_hash_lookup(name)
     const char *name;
{
    FLAGENT *f;
    FLAG *t;

    for (f = flag_hashtab[hash_fn(name, FLAG_HASH_MASK)];
	 f != NULL; f = f->next) {
	if (!strcmp(name, f->name))
	    return f->entry;
    }

    /* provided for backwards compatibility: type flag checking */
    for (t = type_table; t->name != NULL; t++)
	if (string_prefix(name, t->name))
	    return t;

    return NULL;
}

static void flag_hash_insert(name, entry)
     const char *name;
     FLAG *entry;
{
    FLAGENT *new;
    unsigned hashval;

    /* insert flag into hash table. Assume no duplicates. */

    new = (FLAGENT *) malloc(sizeof(FLAGENT));
    new->name = (char *) strdup(name);
    new->entry = entry;
    hashval = hash_fn(name, FLAG_HASH_MASK);
    new->next = flag_hashtab[hashval];
    flag_hashtab[hashval] = new;
}

void init_flag_hashtab()
{
    FLAG *f;
    FLAG_ALIAS *a;

    /* do regular flags first */
    for (f = flag_table; f->name; f++)
	flag_hash_insert(f->name, f);

    /* now add in the aliases */
    for (a = flag_alias_tab; a->alias; a++) {
	if ((f = flag_hash_lookup(a->realname)) != NULL)
	    flag_hash_insert(a->alias, (FLAG *)f);
	else
	    fprintf(stderr,
		    "FLAG INIT: flag alias %s matches no known flag.\n",
		    a->alias);
    }
}

/*---------------------------------------------------------------------------
 * Other functions dealing with flags
 */

int can_set_flag(player, thing, flagp, negate)
     dbref player;
     dbref thing;
     FLAG *flagp;
     int negate;
{
    /* returns 1 if player can set a flag on thing. */

    if (!flagp || !GoodObject(player) || !GoodObject(thing))
	return 0;

    if (flagp->perms & F_INTERNAL)
	return 0;

    if ((flagp->type != NOTYPE) && (flagp->type != Typeof(thing)))
	return 0;

#ifdef INHERIT_FLAG
    if ((flagp->perms & F_INHERIT) && !Wizard(player) &&
	(!Inherit(player) || !Owns(player, thing)))
	return 0;
#endif

    if ((flagp->flag == GOING) && (flagp->type == NOTYPE) &&
	((Typeof(thing) != TYPE_ROOM) || !negate))
	return 0;

    if (flagp->perms & F_ANY)
	return 1;
    if ((flagp->perms & F_WIZARD) && !Wizard(player))
	return 0;
    else if ((flagp->perms & F_ROYAL) && !Hasprivs(player))
	return 0;
    else if ((flagp->perms & F_GOD) && !God(player))
	return 0;

    /* special case for the privileged flags. We do need to check
     * for royalty setting flags on objects they don't own, because
     * the controls check will not stop the flag set if the royalty
     * player is in a zone. Also, only God can set the wizbit on
     * players.
     */
    if (God(player))		/* God can do (almost) anything */
	return 1;

    /* Make sure we don't accidentally permission-check toggles when
     * checking priv bits.
     */
    if (flagp->type == NOTYPE) {

	/* A wiz can set things he owns WIZ, but can't reset his own bit. */
	if (flagp->flag == WIZARD) {
	    return (Wizard(player) && Owns(player, thing) &&
		    (Typeof(thing) != TYPE_PLAYER));
	}

	/* Wizards can set or unset anything royalty. Royalty can set anything
	 * they own royalty, but cannot reset their own bits. */
#ifdef ROYALTY_FLAG
	if (flagp->flag == ROYALTY) {
	    return (Wizard(player) ||
		    (Royalty(player) && Owns(player, thing) &&
		     (Typeof(thing) != TYPE_PLAYER)));
	}
#endif
    } else if (Wizard(thing) && (flagp->flag == PLAYER_GAGGED) &&
	       (flagp->type == TYPE_PLAYER)) {
	return 0;		/* can't gag wizards */
    }

    return 1;
}

const char *unparse_flags(thing, player)
	dbref thing;
	dbref player;
{
  /* print out the flag symbols (letters) */

  static char buf[BUFFER_LEN];
  char *p;
  const char *type_codes = "R-EP";
  FLAG *f;
  int t;

  p = buf;
  t = Typeof(thing);
  if (t != TYPE_THING)
    *p++ = type_codes[t];

  /* get generic flags */
  for (f = flag_table; f->type == NOTYPE; f++) {
    if (Flags(thing) & f->flag)
      *p++ = f->letter;
  }

  /* exits have no special flags. This may change in the future */
  if (t == TYPE_EXIT) {
    *p = '\0';
    return buf;
  }

  /* go to beginning of list of flags for the type and print flags for
   * that type.
   */
  while (f->type != t)   
    f++;
  for ( ; f->type == t; f++) {
    if (Toggles(thing) & f->flag)
      *p++ = f->letter;
  }

  /* we don't want the player to see the SUSPECT flag if he's not priv'ed.
   * As long as it's the last in the list, we can get rid of it simply by
   * putting our terminating null one character early.
   */
  if ((t == TYPE_PLAYER) && (Toggles(thing) & PLAYER_SUSPECT) &&
      !See_All(player))
    *(--p) = '\0';
  else
    *p = '\0';

  return buf;
}

#ifdef FLAGS_ON_EXAMINE
const char *flag_description(player, thing)
     dbref player;
     dbref thing;
{
  static char buf[BUFFER_LEN];
  char fbuf[BUFFER_LEN];
  char *bp;
  FLAG *f;
  int t;

  t = Typeof(thing);

  strcpy(buf, "Type: ");
  switch (t) {
  case TYPE_ROOM:
    strcat(buf, "Room");
    break;
  case TYPE_EXIT:
    strcat(buf, "Exit");
    break;
  case TYPE_THING:
    strcat(buf, "Thing");
    break;
  case TYPE_PLAYER:
    strcat(buf, "Player");
    break;
  default:
    strcat(buf, "***UNKNOWN TYPE***");
    break;
  }
  strcat(buf, " Flags:");

  bp = fbuf;

  /* get generic flags */
  for (f = flag_table; f->type == NOTYPE; f++) {
    if (Flags(thing) & f->flag) {
      safe_chr(' ', fbuf, &bp);
      safe_str(f->name, fbuf, &bp);
    }
  }

  /* go to beginning of list of flags for the type and print flags for
   * that type.
   */
  while ((f->type != t) && (f->type != NOTYPE))
    f++;
  for ( ; f->type == t; f++) {
    if (Toggles(thing) & f->flag) {
      safe_chr(' ', fbuf, &bp);
      safe_str(f->name, fbuf, &bp);
    }
  }

  /* we don't want the player to see the SUSPECT flag if he's not priv'ed.
   * As long as it's the last in the list, we can get rid of it simply by
   * putting our terminating null one word early.
   */
  if ((t == TYPE_PLAYER) && (Toggles(thing) & PLAYER_SUSPECT) &&
      !See_All(player)) {
    do {
      bp--;
    } while (!isspace(*bp));
  }
  *bp = '\0';

  /* no length checking needed. We're never going to have an overflow. */
  sprintf(buf, "%s%s", buf, fbuf); 
  return buf;
}
#endif				/* FLAGS_ON_EXAMINE */

object_flag_type letter_to_flag(c, type, toggle)
     char c;
     int type;
     int *toggle;
{
    /* convert letter to flag */

    FLAG *f;

    *toggle = 0;

    /* try generic flags */
    for (f = flag_table; f->type == NOTYPE; f++) {
	if (c == f->letter)
	    return (f->flag);
    }

    /* try type-specific flags */

    *toggle = 1;
    while ((f->type != type) && (f->type != NOTYPE))
	f++;

    for ( ; f->type != NOTYPE; f++) {
	if (c == f->letter)
	    return (f->flag);
    }
    return -1;
}

object_flag_type find_flag(name, type, toggle, is_conf)
     const char *name;
     int type;
     int *toggle;
     int is_conf;
{
    /* given a flag name and an object type, return a flag, and set the
     * value of toggle to 0 if flag, 1 if toggle. We also check if it's
     * a legal flag to set, if it's a conf flag set.
     */

    FLAG *f;
    
    *toggle = 0;

    if ((f = flag_hash_lookup(upcasestr(name))) == NULL)
	return -1;

    if (is_conf && ((f->perms == F_INTERNAL) || (f->flag == GOING)))
	return -1;

    if (f->type == NOTYPE) {
	return f->flag;
    } else {
	*toggle = 1;
	if (type != f->type) {
	    if (is_conf)
		return -2;
	    else
		return -1;
	} else {
	    return f->flag;
	}
    }
    return f->flag;		/* NOTREACHED */
}

void decompile_flags(player, thing, name)
     dbref player;
     dbref thing;
     const char *name;
{
  /* print out the flags for a decompile */

  FLAG *f;
  int t, ok;

  t = Typeof(thing);
  ok = !Suspect(thing);

  /* do generic flags */
  for (f = flag_table; f->type == NOTYPE; f++)
    if ((Flags(thing) & f->flag) && (f->perms & F_INTERNAL))
      notify(player, tprintf("@set %s = %s", name, f->name));

  /* do normal flags */
  while ((f->type != t) && (f->type != NOTYPE))
    f++;
  for ( ; f->type == t; f++)
    if ((Toggles(thing) & f->flag) && (ok || (f->flag != PLAYER_SUSPECT)))
      notify(player, tprintf("@set %s = %s", name, f->name));
}

int convert_flags(player, s, p_mask, p_toggle, p_type)
     dbref player;
     char *s;
     object_flag_type *p_mask;
     object_flag_type *p_toggle;
     object_flag_type *p_type;
{
  /* convert flags for search */

  FLAG *f;
  object_flag_type mask, toggle, type;
  int done;
  mask = toggle = 0;
  type = NOTYPE;

  while (s && *s) {

    done = 0;

    switch (*s) {
    case 'P':
      type = TYPE_PLAYER;
      break;
    case 'R':
      type = TYPE_ROOM;
      break;
    case 'E':
      type = TYPE_EXIT;
      break;
    default:

      /* check generic flags first */
      for (f = flag_table; (f->type == NOTYPE) && !done; f++) {
	if (*s == f->letter) {
	  mask |= f->flag;
	  done = 1;
	}
      }

      /* try type-specific flags */

      /* if we have a type, start searching from that type */
      if (!done && (type != NOTYPE))
	while ((f->type != type) && (f->type != NOTYPE))
	  f++;

      for ( ; !done && (f->type != NOTYPE); f++) {
	if (*s == f->letter) {
	  /* if we don't have a type yet, we do now. If we do, we need
	   * to check for a conflict.
	   */
	  if (type == NOTYPE) {
	    type = f->type;
	    toggle |= f->flag;
	    done = 1;
	  } else if (type != f->type) {
	    notify(player, tprintf("Type conflict with flag '%c'.", *s));
	    return 0;
	  }
	}
      }

      /* if we get this far and still haven't found anything, error. */
      if (!done) {
	notify(player, tprintf("%c: unknown flag.", *s));
	return 0;
      }
    }
    s++;
  }

  *p_mask = mask;
  *p_toggle = toggle;
  *p_type = type;
  return 1;
}

void set_flag(player, thing, flag, negate, hear, listen)
     dbref player;
     dbref thing;
     const char *flag;
     int negate;
     int hear;
     int listen;
{
  /* attempt to set a flag on an object */

  FLAG *f;
  char tbuf1[BUFFER_LEN];

  if ((f = flag_hash_lookup(upcasestr(flag))) == NULL) {
    notify(player, "I don't recognize that flag.");
    return;
  }

  if (!can_set_flag(player, thing, f, negate)) {
    notify(player, "Permission denied.");
    return;
  }

  if (negate) {
    
    /* reomve the flag */
    if (f->type == NOTYPE)
      Flags(thing) &= ~(f->flag);
    else
      Toggles(thing) &= ~(f->flag);

    /* log if necessary */
    if ((f->flag == WIZARD) && (f->type == NOTYPE))
      do_log(LT_WIZ, player, thing, "WIZFLAG RESET");
#ifdef ROYALTY_FLAG
    else if ((f->flag == ROYALTY) && (f->type == NOTYPE))
      do_log(LT_WIZ, player, thing, "ROYAL FLAG RESET");
#endif

    /* those who unDARK return to the RWHO */
    if ((f->flag == DARK) && (f->type == NOTYPE) &&
	(Typeof(thing) == TYPE_PLAYER))
	hide_player(thing, 0);
    
    /* notify the area if something stops listening */
    if ((Typeof(thing) == TYPE_THING) && (f->type == TYPE_THING) &&
	(((f->flag == THING_PUPPET) && !listen && !Hearer(thing)) ||
	 ((f->flag == THING_LISTEN) && !hear && !Listener(thing)))) {
      sprintf(tbuf1, "%s loses its ears and becomes deaf.", Name(thing));
      notify_except(Contents(Location(thing)), 0, tbuf1);
    }
    if ((f->flag == AUDIBLE) && (f->type == NOTYPE)) {
      switch (Typeof(thing)) {
      case TYPE_EXIT:
	if (Flags(Exits(thing)) & AUDIBLE) {
	  sprintf(tbuf1, "Exit %s is no longer broadcasting.", Name(thing));
	  notify_except(Contents(Exits(thing)), 0, tbuf1);
	}
	break;
      case TYPE_ROOM:
	notify_except(Contents(thing), 0, 
		      "Audible exits in this room have been deactivated.");
	break;
      case TYPE_THING:
      case TYPE_PLAYER:
	notify_except(Contents(thing), 0,
		      "This room is no longer broadcasting.");
	break;
      }
    }

    if (((f->flag == QUIET) && (f->type == NOTYPE)) ||
	(!Quiet(player) && !Quiet(thing)))
      notify(player, "Flag reset.");

  } else {

    /* set the flag */
    if (f->type == NOTYPE)
      Flags(thing) |= f->flag;
    else
      Toggles(thing) |= f->flag;

    /* log if necessary */
    if ((f->flag == WIZARD) && (f->type == NOTYPE))
      do_log(LT_WIZ, player, thing, "WIZFLAG SET");
#ifdef ROYALTY_FLAG
    if ((f->flag == ROYALTY) && (f->type == NOTYPE))
      do_log(LT_WIZ, player, thing, "ROYAL FLAG SET");
#endif

    /* DARK players should be treated as logged out */
    if ((f->flag == DARK) && (f->type == NOTYPE) &&
	(Typeof(thing) == TYPE_PLAYER))
	hide_player(thing, 1);

    /* notify area if something starts listening */
    if ((Typeof(thing) == TYPE_THING) && (f->type == TYPE_THING) &&
	((f->flag == THING_PUPPET) || (f->flag == THING_LISTEN)) &&
	!hear && !listen) {
      sprintf(tbuf1,"%s grows ears and can now hear.", Name(thing));
      notify_except(Contents(Location(thing)), NOTHING, tbuf1);
    }
    
    /* notify for audible exits */
    if ((f->flag == AUDIBLE) && (f->type == NOTYPE)) {
      switch (Typeof(thing)) {
      case TYPE_EXIT:
	if (db[db[thing].exits].flags & AUDIBLE) {
	  sprintf(tbuf1, "Exit %s is now broadcasting.", Name(thing));
	  notify_except(Contents(Exits(thing)), NOTHING, tbuf1);
	}
	break;
      case TYPE_ROOM:
	notify_except(Contents(thing), NOTHING,
		      "Audible exits in this room have been activated.");
	break;
      case TYPE_PLAYER:
      case TYPE_THING:
	notify_except(Contents(thing), NOTHING,
		      "This room is now broadcasting.");
	break;
      }
    }

    if (((f->flag == QUIET) && (f->flag == NOTYPE)) ||
	(!Quiet(player) && !Quiet(thing)))
      notify(player, "Flag set.");
  }
}
