/* db.c */

#include "copyright.h"

#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include "config.h"
#include "interface.h"
#include "db.h"
#include "attrib.h"
#include "externs.h"

#ifdef MEM_CHECK
#include "mem_check.h"
#endif

#ifdef ADD_POWERS
#include "convdb.c"
#endif

extern int paranoid_checkpt;	/* from game.c */

struct object *db = 0;
dbref db_top = 0;

dbref errobj;

#ifndef DB_INITIAL_SIZE
#define DB_INITIAL_SIZE 10000
#endif				/* DB_INITIAL_SIZE */

dbref db_size = DB_INITIAL_SIZE;

extern char ccom[];

/* manage boolean expression free list */
struct boolexp *alloc_bool()
{
#ifdef MEM_CHECK
  add_check("boolexp");
#endif
  return ((struct boolexp *) malloc(sizeof(struct boolexp)));
}

void free_bool(b)
    struct boolexp *b;
{
  if (b != TRUE_BOOLEXP && b) {
    free((char *) b);
#ifdef MEM_CHECK
    del_check("boolexp");
#endif
  }
}

char *set_string(ptr, new)
    char **ptr;
    char *new;
{
  /* if pointer not null unalloc it */
  if (*ptr) {
    free((char *) *ptr);
#ifdef MEM_CHECK
    del_check("object_name");
#endif
  }
  if (!new || !*new)
    return (*ptr = NULL);
  *ptr = (char *) malloc(strlen(new) +1);
#ifdef MEM_CHECK
  add_check("object_name");
#endif
  strcpy(*ptr, new);
  return (*ptr);
}

int db_init = 0;

static void db_grow(newtop)
    dbref newtop;
{
  struct object *newdb;

  if (newtop > db_top) {
    db_top = newtop;
    if (!db) {
      /* make the initial one */
      db_size = (db_init) ? db_init : DB_INITIAL_SIZE;
      if((db = (struct object *)
             malloc(db_size * sizeof(struct object))) == NULL) {
	fprintf(stderr, "ERROR: out of memory!\n");
	fflush(stderr);
	abort();
      }
    }
    /* maybe grow it */
    if (db_top > db_size) {
      /* make sure it's big enough */
      while (db_top > db_size)
	db_size *= 2;
      if((newdb = (struct object *)
                realloc(db, db_size * sizeof(struct object))) == NULL) {
	fprintf(stderr, "ERROR: out of memory!\n");
	fflush(stderr);
	abort();
      }
      db = newdb;
    }
  }
}

dbref new_object()
{
  dbref newobj;
  struct object *o;
  /* if stuff in free list use it */
  if ((newobj = free_get()) == NOTHING) {
    /* allocate more space */
    newobj = db_top;
    db_grow(db_top + 1);
  }
  /* clear it out */
  o = db + newobj;
  o->name = 0;
  o->list = 0;
  o->location = NOTHING;
  o->contents = NOTHING;
  o->exits = NOTHING;
  o->next = NOTHING;
  o->parent = NOTHING;
  o->key = TRUE_BOOLEXP;
  o->usekey = TRUE_BOOLEXP;
  o->enterkey = TRUE_BOOLEXP;
  o->owner = NOTHING;
  o->zone = NOTHING;
  o->penn = 0;
  o->flags = 0;
  o->toggles = 0;
  o->powers = 0;
#if (CHAT_SYSTEM != 0)
  /* initialize channels here, since it's not going to get done otherwise */
  o->channels = 0;
#endif
  /* flags you must initialize yourself */
  return newobj;
}

void putref(f, ref)
    FILE *f;
    dbref ref;
{
  fprintf(f, "%d\n", ref);
}

void putstring(f, s)
    FILE *f;
    const char *s;
{
  if (s) {
    fputs(s, f);
  }
  putc('\n', f);
}

static void putbool_subexp(f, b)
    FILE *f;
    struct boolexp *b;
{
  switch (b->type) {
    case BOOLEXP_IS:
      putc('(', f);
      putc(IS_TOKEN,f);
      putbool_subexp(f,b->sub1);
      putc(')', f);
      break;
    case BOOLEXP_CARRY:
      putc('(', f);
      putc(IN_TOKEN,f);
      putbool_subexp(f,b->sub1);
      putc(')', f);
      break;
    case BOOLEXP_OWNER:
      putc('(', f);
      putc(OWNER_TOKEN, f);
      putbool_subexp(f, b->sub1);
      putc(')', f);
      break;
    case BOOLEXP_IND:
      putc('(', f);
      putc(AT_TOKEN, f);
      putbool_subexp(f, b->sub1);
      putc(')', f);
      break;
    case BOOLEXP_AND:
      putc('(', f);
      putbool_subexp(f, b->sub1);
      putc(AND_TOKEN, f);
      putbool_subexp(f, b->sub2);
      putc(')', f);
      break;
    case BOOLEXP_OR:
      putc('(', f);
      putbool_subexp(f, b->sub1);
      putc(OR_TOKEN, f);
      putbool_subexp(f, b->sub2);
      putc(')', f);
      break;
    case BOOLEXP_NOT:
      putc('(', f);
      putc(NOT_TOKEN, f);
      putbool_subexp(f, b->sub1);
      putc(')', f);
      break;
    case BOOLEXP_CONST:
      fprintf(f, "%d", b->thing);
      break;
    case BOOLEXP_ATR:
      fprintf(f, "%s:%s", b->atr_lock->name, uncompress(b->atr_lock->text));
      break;
    case BOOLEXP_EVAL:
      fprintf(f, "%s/%s", b->atr_lock->name, uncompress(b->atr_lock->text));
      break;
    default:
      break;
  }
}

void putboolexp(f, b)
    FILE *f;
    struct boolexp *b;
{
  if (b != TRUE_BOOLEXP) {
    putbool_subexp(f, b);
  }
  putc('\n', f);
}

void db_write_obj_basic(f, i, o)
     FILE *f;
     dbref i;
     struct object *o;
{
  putstring(f, o->name);
  putref(f, o->location);
  putref(f, o->contents);
  putref(f, o->exits);
  putref(f, o->next);
  putref(f, o->parent);
  putboolexp(f, o->key);
  putboolexp(f, o->usekey);
  putboolexp(f, o->enterkey);
  putref(f, o->owner);
  putref(f, o->zone);
  putref(f, Pennies(i));
  putref(f, o->flags);
  putref(f, o->toggles);
  putref(f, o->powers);

#if (CHAT_SYSTEM != 0)
  putref(f, o->channels);
#endif
}

int db_write_object(f, i)
    FILE *f;
    dbref i;
{
  struct object *o;
  ALIST *list;
  o = db + i;
  db_write_obj_basic(f, i, o);

  /* write the attribute list */
  for (list = o->list; list; list = AL_NEXT(list)) {
    if (!AL_BAD(list)) {
      fprintf(f, "]%s^%d^%d\n", AL_NAME(list), db[AL_CREATOR(list)].owner,
	      AL_FLAGS(list));
      fprintf(f, "%s\n", uncompress(AL_STR(list)));
    }
  }
  fprintf(f, "<\n");
  return 0;
}

dbref db_write(f)
    FILE *f;
{
  dbref i;

/* print a header line to make a later conversion to 2.0 easier to do.
 * the odd choice of numbers is based on 256*x + 2 offset
 */
#if (CHAT_SYSTEM != 0)
  fprintf(f, "+V1282\n");
#else
  fprintf(f, "+V1538\n");
#endif				/* CHAT_SYSTEM */

  fprintf(f, "~%d\n", db_top);
  for (i = 0; i < db_top; i++) {
    fprintf(f, "!%d\n", i);
    db_write_object(f, i);
  }
  fputs("***END OF DUMP***\n", f);
/*  fflush(f); */
  return (db_top);
}

int db_paranoid_write_object(f, i)
    FILE *f;
    dbref i;
{
  struct object *o;
  ALIST *list;
  char name[BUFFER_LEN];
  char tbuf1[BUFFER_LEN];
  int err = 0;
  char *p;
  dbref owner;

  o = db + i;
  db_write_obj_basic(f, i, o);
/*  fflush(f); */

  /* write the attribute list, scanning */
  for (list = o->list; list; list = AL_NEXT(list)) {
    if (!AL_BAD(list)) {
      /* smash unprintable characters in the name, replace with ! */
      strcpy(name, AL_NAME(list));
      for (p = name; *p; p++) {
	if (!isprint(*p) || isspace(*p)) {
	  *p = '!';
	  err = 1;
	}
      }
      if (err) {
	fprintf(checklog_fp, 
		" * Bad attribute name on #%d. Changing name to %s.\n",
		i, name);
	err = 0;
      }

      /* check the owner */
      owner = AL_CREATOR(list);
      if (!GoodObject(owner)) {
	fprintf(checklog_fp,
		" * Bad owner on attribute %s on #%d.\n", name, i);
	owner = GOD;
      } else {
	owner = db[owner].owner;
      }

      /* write that info out */
      fprintf(f, "]%s^%d^%d\n", name, (int) owner, AL_FLAGS(list));

      /* now check the attribute */
      strcpy(tbuf1, uncompress(AL_STR(list)));
      /* get rid of unprintables and hard newlines */
      for (p = tbuf1; *p; p++) {
	if (!isprint(*p) || (*p == '\r') || (*p == '\n')) {
	  *p = '!';
	  err = 1;
	}
      }
      if (err) {
	fprintf(checklog_fp, 
		" * Bad text in attribute %s on #%d. Changed to:\n", name, i);
	fprintf(checklog_fp, "%s\n", tbuf1);
      }
      fprintf(f, "%s\n", tbuf1);
/*      fflush(f); */
    }
  }
  fprintf(f, "<\n");
  return 0;
}

dbref db_paranoid_write(f)
    FILE *f;
{
  dbref i;

/* print a header line to make a later conversion to 2.0 easier to do.
 * the odd choice of numbers is based on 256*x + 2 offset
 */
#if (CHAT_SYSTEM != 0)
  fprintf(f, "+V1282\n");
#else
  fprintf(f, "+V1538\n");
#endif				/* CHAT_SYSTEM */

  fprintf(checklog_fp, "PARANOID WRITE BEGINNING...\n");
  fflush(checklog_fp);

  /* print total number of objects at top of file */
  fprintf(f, "~%d\n", db_top);
  
  /* write out each object */
  for (i = 0; i < db_top; i++) {
    fprintf(f, "!%d\n", i);
    db_paranoid_write_object(f, i);
    /* print out a message every so many objects */
    if (i % paranoid_checkpt == 0) {
      fprintf(checklog_fp, "\t...wrote up to object #%d\n", i);
      fflush(checklog_fp);
    }
  }

  fputs("***END OF DUMP***\n", f);
/*  fflush(f); */
  fprintf(checklog_fp, "END OF PARANOID WRITE.\n");
  fflush(checklog_fp);
  return (db_top);
}

dbref parse_dbref(s)
     const char *s;
{
    /* We must make sure that everything in the string is a number.
     * Otherwise, something like "#123's foo" will get returned as 123,
     * since atoi() ignores trailing garbage.
     */

    const char *p;

    if (!s || !*s)
	return NOTHING;

    for (p = s; p && *p; p++)
	if (!isdigit(*p))
	    return NOTHING;

    return (atoi(s));
}

dbref getref(f)
    FILE *f;
{
  static char buf[BUFFER_LEN];
  fgets(buf, sizeof(buf), f);
  return (atol(buf));
}

#ifndef OLD_NEWLINES
const char *getstring_noalloc(f)
    FILE *f;
{
  static char buf[BUFFER_LEN];
  char *p;
  char c, lastc;

  p = buf;
  c = '\0';

  for (;;) {

    lastc = c;
    c = fgetc(f);

    /* if EOF or null, return */
    if (!c || (c == EOF)) {
      *p = '\0';
      return buf;
    }

    /* if it's a newline, return if prior char is not a carriage return.
     * "\r\n" sequences are created by using the %r substitution, and
     * thus may occur in the middle of a buffer.
     */
    if ((c == '\n') && (lastc != '\r')) {
      *p = '\0';
      return buf;
    }
    /* copy it in */
    safe_chr(c, buf, &p);
  }
}

#else
const char *getstring_noalloc(f)
     FILE *f;
{
  static char buf[2*BUFFER_LEN];
  char *p;
  fgets(buf, sizeof(buf), f);
  for (p= buf; *p; p++) {
    if (*p == '\n') {
      *p = '\0';
      break;
    }
  }
  return buf;
}
#endif				/* OLD_NEWLINES */

#define getstring(x,p) {p=NULL; SET(p,getstring_noalloc(x));}
#define getstring_compress(x,p)  {p=NULL; SETC(p,getstring_noalloc(x));}

static struct boolexp *getboolexp1(f)
    FILE *f;
{
  struct boolexp *b;
  int c;
  char tbuf1[BUFFER_LEN], tbuf2[BUFFER_LEN];

  c = getc(f);
  switch (c) {
    case '\n':
      ungetc(c, f);
      return TRUE_BOOLEXP;
      /* break; */
    case EOF:
      fprintf(stderr, "ERROR: Unexpected EOF in boolexp.  Object #%d\n", 
	      errobj);
      return TRUE_BOOLEXP;
      /*NOTREACHED*/
      break;
    case '(':
      b = alloc_bool();
      if ((c = getc(f)) == NOT_TOKEN) {
	b->type = BOOLEXP_NOT;
	b->sub1 = getboolexp1(f);
	if (getc(f) != ')')
	  goto error;
	return b;
      } else if (c == IS_TOKEN) {
	b->type = BOOLEXP_IS;
	b->sub1 = getboolexp1(f);
	if (getc(f) != ')')
	  goto error;
	return b;
      } else if (c == IN_TOKEN) {
	b->type = BOOLEXP_CARRY;
	b->sub1 = getboolexp1(f);
	if (getc(f) != ')')
	  goto error;
	return b;
      } else if (c == OWNER_TOKEN) {
	b->type = BOOLEXP_OWNER;
	b->sub1 = getboolexp1(f);
	if (getc(f) != ')')
	  goto error;
	return b;
      } else if (c == AT_TOKEN) {
	b->type = BOOLEXP_IND;
	b->sub1 = getboolexp1(f);
	if (getc(f) != ')')
	  goto error;
        return b;
      } else {
	ungetc(c, f);
	b->sub1 = getboolexp1(f);
	switch (c = getc(f)) {
	  case AND_TOKEN:
	    b->type = BOOLEXP_AND;
	    break;
	  case OR_TOKEN:
	    b->type = BOOLEXP_OR;
	    break;
	  default:
	    goto error;
	    /* break */
	}
	b->sub2 = getboolexp1(f);
	if (getc(f) != ')')
	  goto error;
	return b;
      }
      /* break; */
    case '-':
      /* obsolete NOTHING key */
      /* eat it */
      while ((c = getc(f)) != '\n')
	if (c == EOF) {
	  fprintf(stderr, "ERROR: Unexpected EOF in boolexp. Object #%d\n",
		  errobj);
	  return TRUE_BOOLEXP;
        }
      ungetc(c, f);
      return TRUE_BOOLEXP;
      /* break */
    default:
      /* can be either a dbref or : seperated string */ 
      ungetc(c, f);
      b = alloc_bool();
      b->type = BOOLEXP_CONST;
      b->thing = 0;

      /* constant dbref */
      if(isdigit(c = getc(f))) {
        while(isdigit(c)) {
	  b->thing = b->thing * 10 + c - '0';
          c = getc(f);
        }
	switch(c) {
	  case ':':	/* old style boolexp lock */
	    {
	    char *p;

	    for(p = tbuf1;
                ((c = getc(f)) != EOF) && (c != '\n') && (c != ')');
		*p++ = c);
	    *p = '\0';
	    if(c == EOF)
	      goto error;
	    b->atr_lock = alloc_atr(convert_atr(b->thing), tbuf1);
	    b->thing = 0;
            b->type = BOOLEXP_ATR;
	    /* this is only needed because of the braindeath of the previous
	     * version of atrlocks.. bleah.
	     */
	    if(c == '\n')
	      return(b);
	    }
	  default:
	    ungetc(c, f);
	    break;
	}
        return(b);
      } else {
	/* we indulge in a bit of a kludge. We either have a colon
         * separated string (attribute lock) or a slash-separate string
         * (eval lock).
	 */
        char *p = tbuf1, *s;
	char savec;
	
        *p++ = c;
        for ( ; 
	     ((c = getc(f)) != EOF) && (c != '\n') && (c != ':') && (c != '/');
	     *p++ = c)
	  ;
        *p = '\0';
        if ((c == EOF) || (c == '\n'))
          goto error;
        if ((c != ':') && (c != '/'))
          goto error;
	savec = c;
	for (s = tbuf2;
             ((c = getc(f)) != EOF) && (c != '\n') && (c != ')' &&
              (c != OR_TOKEN) && (c != AND_TOKEN));
	     *s++ = c) ;
	if (c == EOF)
	  goto error;
	*s++ = 0;
	ungetc(c, f);
	b->atr_lock = alloc_atr(tbuf1, tbuf2);
	if (savec == ':')
	  b->type = BOOLEXP_ATR;
	else
	  b->type = BOOLEXP_EVAL;
	return (b);
      }
  }
error:
  fprintf(stderr, "ERROR: Unknown error in boolexp. Object #%d\n", errobj);
  return TRUE_BOOLEXP;
}

struct boolexp *getboolexp(f)
    FILE *f;
{
  struct boolexp *b;
  b = getboolexp1(f);
  if (getc(f) != '\n') {
    fprintf(stderr, "ERROR: Invalid boolexp format on object #%d\n", errobj);
    return TRUE_BOOLEXP;
  }
  return b;
}

void free_boolexp(b)
    struct boolexp *b;
{
  if (b != TRUE_BOOLEXP && b) {
    switch (b->type) {
      case BOOLEXP_AND:
      case BOOLEXP_OR:
	free_boolexp(b->sub1);
	free_boolexp(b->sub2);
	free_bool(b);
	break;
      case BOOLEXP_NOT:
      case BOOLEXP_CARRY:
      case BOOLEXP_IND:
      case BOOLEXP_IS:
	free_boolexp(b->sub1);
	free_bool(b);
	break;
      case BOOLEXP_CONST:
	free_bool(b);
	break;
      case BOOLEXP_ATR:
      case BOOLEXP_EVAL:
	if (b->atr_lock->name)
		free((char *)b->atr_lock->name);
	if (b->atr_lock->text)
		free((char *)b->atr_lock->text);
	if (b->atr_lock)
		free((char *)b->atr_lock);
#ifdef MEM_CHECK
	del_check("bool_atr");
	del_check("bool_atr_name");
	del_check("bool_atr_val");
#endif
	free_bool(b);
	break;
    }
  }
}

struct boolexp *dup_bool(b)
    struct boolexp *b;
{
  struct boolexp *r;
  if (b == TRUE_BOOLEXP)
    return (TRUE_BOOLEXP);
  r = alloc_bool();
  switch (r->type = b->type) {
    case BOOLEXP_AND:
    case BOOLEXP_OR:
      r->sub2 = dup_bool(b->sub2);
    case BOOLEXP_NOT:
    case BOOLEXP_IND:
    case BOOLEXP_IS:
    case BOOLEXP_CARRY:
    case BOOLEXP_OWNER:
      r->sub1 = dup_bool(b->sub1);
    case BOOLEXP_CONST:
      r->thing = b->thing;
      break;
    case BOOLEXP_ATR:
    case BOOLEXP_EVAL:
      r->atr_lock = alloc_atr(b->atr_lock->name, b->atr_lock->text);
      break;
    default:
      fprintf(stderr, "ERROR: bad bool type in dup_bool!\n");
      return (TRUE_BOOLEXP);
  }
  return (r);
}

void db_free()
{
    dbref i;
    struct object *o;

    if (db) {

	for (i = 0; i < db_top; i++) {
	    o = &db[i];
	    SET(o->name, NULL);
	    atr_free(i);
	    if (o->key)
		free_boolexp(o->key);
	    if (o->enterkey)
		free_boolexp(o->enterkey);
	    if (o->usekey)
		free_boolexp(o->usekey);
	}

	free((char *)db);
	db = NULL;
	db_init = db_top = '\0';
    }
}

/* read attribute list */
int get_list(f, i)
    FILE *f;
    dbref i;
{
  int c;
  char *p, *q;
  char tbuf1[BUFFER_LEN];

#ifdef OLD_NEWLINES
  char nextc;
#endif				/* OLD_NEWLINES */

  db[i].list = NULL;
  while (1)
    switch (c = getc(f)) {
      case ']':		/* new style attribs, read name then value */
        strcpy(tbuf1, getstring_noalloc(f));
	if(!(p = (char *) index(tbuf1, '^'))) {
	  fprintf(stderr, "ERROR: Bad format on new attributes. object #%d\n",
		  i);
	  return 0;
	}
	*p++ = '\0';
	if(!(q = (char *) index(p, '^'))) {
	  fprintf(stderr, "ERROR: Bad format on new attributes. object #%d\n",
		  i);
	  return 0;
	}
	*q++ = '\0';
	atr_new_add(i, tbuf1, getstring_noalloc(f), atoi(p), atoi(q));
	/* Check removed for atoi(q) == 0  (which results in NOTHING for that
	 * parameter, and thus no flags), since this eliminates 'visual'
	 * attributes (which, if not built-in attrs, have a flag val of 0.)
	 */
#ifdef OLD_NEWLINES
	/* hack to check if there's an unexpected '\n' that got in */
	nextc = getc(f);
	if ((nextc != ']') && (nextc != '>') && (nextc != '<') &&
	    (nextc != '!')) {
	  /* we have a problem. print error */ 
	  fprintf(stderr, 
	    "** WARNING **  Hard newline in attribute %s on object #%d\n",
	    tbuf1, i);
	  /* throw out everything until next good attrib or object.
	   * we can tell a good attrib or object by looking for a \n
	   * followed by a ], >, <, or !. Otherwise, we can be "spoofed"
	   */
	  do {
	    while ((nextc = getc(f)) != '\n')
	      ;
	    nextc = getc(f);	/* go past EOL */
	  } while ((nextc != ']') && (nextc != '>') && (nextc != '<') &&
		   (nextc != '!'));
	}
	/* we've eaten a character, so go back. */
	ungetc(nextc, f);
#endif				/* OLD_NEWLINES */
	break;
      case '>':		/* old style attribs, read # then value*/
	atr_new_add(i, convert_atr(getref(f)), getstring_noalloc(f),
		       db[i].owner, NOTHING);
	break;
      case '<':		/* end of list */
	if ('\n' != getc(f)) {
	  fprintf(stderr, "ERROR: no line feed on object %d\n", i);
	  return (0);
	}
	return (1);
      default:
	if(c == EOF) {
	  fprintf(stderr, "ERROR: Unexpected EOF on file.\n" );
	  return (0);
	}
	fprintf(stderr, "ERROR: Bad character %c on object %d\n", c, i);
	return (0);
    }
}

dbref db_read(f)
    FILE *f;
{
  int c;
  dbref i;
  const char *dummy;
  struct object *o;
  const char *end;
#ifdef ADD_POWERS
  int flags, toggles, powers, temp;
#endif

  clear_players();
  db_free();
  for (i = 0;; i++) {
    errobj = i;
    c = getc(f);
    switch (c) {
	/* make sure database is at least this big *1.5 */
      case '~':
	db_init = (getref(f) * 3) / 2;
	break;
	/* skip over MUSH 2.0 header stuff so we can move up eventually */
      case '+':
	dummy = getstring_noalloc(f);
	break;
	/* old fashioned database */
      case '#':
	fprintf(stderr, "ERROR: old style database.\n");
	/* if you want to read in an old-style database, use an earlier
	 * patchlevel to upgrade.
	 */
	return -1;
	break;
	/* new database */
      case '!':		/* non-zone oriented database */
      case '&':		/* zone oriented database */
	/* make space */
	i = getref(f);
	db_grow(i + 1);
	/* read it in */
	o = db + i;
	getstring(f, o->name);
	o->location = getref(f);
	/* --- get zone number --- */
	(c == '&') ? (int) getref(f) : 0;
	/* ----------------------- */
	o->contents = getref(f);
	o->exits = getref(f);
	o->next = getref(f);
	o->parent = getref(f);
	o->key = getboolexp(f);
	o->usekey = getboolexp(f);
	o->enterkey = getboolexp(f);
	o->owner = getref(f);
	o->zone = getref(f);
	s_Pennies(i, getref(f));
#ifdef ADD_POWERS
	temp = getref(f);
	conv_newflags(temp, &flags, &toggles, &powers);
	o->flags = flags;
	o->toggles = toggles;
	o->powers = powers;
#else
	o->flags = getref(f);
	o->toggles = getref(f);
	o->powers = getref(f);
#endif

#if (CHAT_SYSTEM == 2)
	o->channels = 0;
#endif
#if ((CHAT_SYSTEM == 1) || (CHAT_SYSTEM == 3))
	o->channels = getref(f);
#endif				/* CHAT_SYSTEM */

	/* read attribute list for item */
	if (!get_list(f, i)) {
	  fprintf(stderr, "ERROR: bad attribute list object %d\n", i);
	  return -1;
	}
	/* check to see if it's a player */
	if (Typeof(i) == TYPE_PLAYER) {
	  add_player(i, NULL);
	  Toggles(i) &= ~PLAYER_CONNECT;
#ifdef ADD_POWERS
	  Flags(i) &= ~0x200;	/* zap old connect flag, too */
#endif
	}
	break;

      case '*':
	end = getstring_noalloc(f);
	if (!strcmp(end, "***END OF DUMP***")) {
	  fprintf(stderr, "ERROR: No end of dump %d\n", i);
	  return -1;
	} else {
	  {
	    fprintf(stderr, "READING: done\n");
	    FIX;
	    return db_top;
	  }
	}
      default:
	fprintf(stderr, "ERROR: failed object %d\n", i);
	return -1;
    }
  }
}
