# building.tcl --
#
#	The Building Window and related commands.
#

namespace eval NSBuilding {

variable Priv

# NSBuilding::InitModule --
#
#	One-time-only-ever initialization.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc InitModule {} {

	global Windows

	NSObject::New NSBuilding
}

# NSBuilding::NSBuilding --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc NSBuilding oop {

	global NSBuilding
	global Windows

	InitWindow $oop

	NSWindowManager::RegisterWindow building [Info $oop win] \
		"NSBuilding::GeometryCmd $oop" "" "NSBuilding::DisplayCmd $oop"

	#
	# Global list of application windows
	#

	set Windows(building,win) [Info $oop win]
	set Windows(building,class) NSBuilding
	set Windows(building,oop) $oop
}

# NSBuilding::Info --
#
#	Query and modify info.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc Info {oop info args} {

	global NSBuilding

	# Set info
	if {[llength $args]} {
		switch -- $info {
			default {
				set NSBuilding($oop,$info) [lindex $args 0]
			}
		}

	# Get info
	} else {
		switch -- $info {
			default {
				return $NSBuilding($oop,$info)
			}
		}
	}
}

# NSBuilding::InitWindow --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc InitWindow oop {

	global Windows

	set win .building$oop
	toplevel $win
	wm title $win Building

	wm transient $win $Windows(main,win)

	# Do stuff when window closes
	wm protocol $win WM_DELETE_WINDOW "NSBuilding::Close $oop"

	# This window plays sound
	SoundWindow $win

	# Start out withdrawn (hidden)
	wm withdraw $win

	Info $oop win $win

	InitMenus $oop

	#
	# Divider + Building Info
	#

	set frame $win.info
	frame $frame \
		-borderwidth 0
	frame $frame.divider1 \
		-borderwidth 1 -height 2 -relief groove
	label $frame.playerGold \
		-font {Courier 9} -text "Gold Remaining:"
	label $frame.gold \
		-font {Courier 9} -text [angband player gold]

	# Update the display when the character's gold changes
	qebind $frame.gold <Py-gold> {%W configure -text %c}
	bind $frame.gold <Destroy> {+qebind %W <Py-gold> ""}

	#
	# Canvas
	#

	canvas $win.canvas \
		-width 400 -height 300 -background Black

	#
	# Statusbar
	#

	frame $win.statusBar -relief flat -borderwidth 0
	label $win.statusBar.label -anchor w -relief sunken -padx 2
	label $win.statusBar.label2 -anchor w -relief sunken -padx 2 -width 12
	pack $win.statusBar.label -side left -expand yes -fill both
	pack $win.statusBar.label2 -side right -expand no

	#
	# Geometry
	#

	grid rowconfig $win 0 -weight 0
	grid rowconfig $win 1 -weight 1
	grid rowconfig $win 2 -weight 0
	grid columnconfig $win 0 -weight 1

	grid $win.info \
		-row 0 -column 0 -rowspan 1 -columnspan 1 -sticky ew
	grid $win.canvas \
		-row 1 -column 0 -rowspan 1 -columnspan 1 -sticky news
	grid $win.statusBar \
		-row 2 -column 0 -rowspan 1 -columnspan 2 -sticky ew

	#
	# Feed Term when keys pressed
	#

	Term_KeyPress_Bind $win
	Term_KeyPress_Bind $win.canvas
}

# NSBuilding::InitMenus --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc InitMenus oop {

	global NSBuilding

	set win $NSBuilding($oop,win)

	#
	# Menu bar
	#

	set menuDef "-tearoff 0 -postcommand \"NSBuilding::SetupMenus $oop\" -identifier MENUBAR"
	set mbar [NSObject::New NSMenu $win $menuDef]
	set NSBuilding($oop,mbar) $mbar

	#
	# Store Menu
	#

	NSObject::New NSMenu $mbar {-tearoff 0 -identifier MENU_BUILDING}
	NSMenu::MenuInsertEntry $mbar -end MENUBAR {-type cascade \
		-menu MENU_BUILDING -label "Building" -underline 0 \
		-identifier M_BUILDING}

	set entries {}
	lappend entries "-type command -label \"Leave\" \
		-command \"DoCommandIfAllowed \033\" -identifier E_EXIT"
	
	NSMenu::MenuInsertEntries $mbar -end MENU_BUILDING $entries
}

# NSBuilding::SetupMenus --
#
#	Description
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc SetupMenus {oop mbarID} {

	lappend identList E_EXIT

	NSMenu::MenuEnable $mbarID $identList
}

# NSBuilding::DisplayCmd --
#
#	Called by NSWindowManager::Display().
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc DisplayCmd {oop message first} {

	switch -- $message {
		preDisplay {
			set win [Info $oop win]
			set buildingname [angband building buildingname]
			set ownername [angband building ownername]
			set ownerrace [angband building ownerrace]
			wm title $win "The $buildingname run by $ownername the $ownerrace"

			ConfigureWindow $oop
			DisplayCommands $oop
		}
		reDisplay {
			DisplayCommands $oop
		}
		postDisplay {
		}
	}
}

# NSBuilding::GeometryCmd --
#
#	Called by NSWindowManager::Setup(). Returns the desired (default)
#	geometry for the window.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc GeometryCmd oop {

	global Windows

	set win [Info $oop win]
	set winMain $Windows(main,win)
	set winMicro $Windows(micromap,win)
	set x [winfo x $winMain]
	set y [winfo y $winMain]
	set width [NSToplevel::ContentWidth $win [expr [NSToplevel::EdgeRight $winMicro] - $x]]
	set height [NSToplevel::ContentHeight $win [expr [NSToplevel::EdgeBottom $winMain] - $y]]
	return ${width}x$height+$x+$y
}

# NSBuilding::Close --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc Close oop {

	angband keypress \033
}

# NSBuilding::ConfigureWindow --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc ConfigureWindow oop {

	set win [Info $oop win]
	set frame $win.info

	eval pack forget [winfo children $frame]

	pack $frame.divider1 \
		-side top -expand yes -fill x

	# Character's Gold
	pack $frame.gold \
		-side right -expand no -padx 2
	pack $frame.playerGold \
		-side right -expand no -padx 2
}

# NSBuilding::DisplayCommands --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc DisplayCommands oop {

	set canvas [Info $oop win].canvas

	$canvas delete all

	set x 10
	set y 10
	set font [Global font,fixed,normal]
	set lineHeight [font metrics $font -linespace]

	foreach action [concat [angband building actions] [list "ESC {Exit building}"]] {
		set cmdChar [lindex $action 0]
		set desc [lindex $action 1]
		set closed 0
		if {[string match "*gp)" $desc]} {
			set fill Gold
		} elseif {[string match "*(closed)" $desc]} {
			set fill gray60
			set closed 1
		} else {
			set fill White
		}
		set itemId [$canvas create text	$x $y -fill $fill -text "$cmdChar) \
			$desc" -font $font -anchor w -tags "action:$cmdChar"]

		# Create bindings to highlight the command when the poiner is
		# over it, unless the commands is "closed".
		if !$closed {
			$canvas bind $itemId <Enter> \
				"%W itemconfigure action:$cmdChar -fill [NSColorPreferences::Get listHilite]"
			$canvas bind $itemId <Leave> \
				"%W itemconfigure action:$cmdChar -fill $fill"
			if {$cmdChar == "ESC"} {
				$canvas bind $itemId <ButtonPress-1> "
					%W itemconfigure action:$cmdChar -fill $fill
					DoCommandIfAllowed \033
				"
			} else {
				$canvas bind $itemId <ButtonPress-1> "
					%W itemconfigure action:$cmdChar -fill $fill
					DoCommandIfAllowed $cmdChar
				"
			}
		}
		incr y $lineHeight
	}
}

# NSBuilding::CreateDieDot --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc CreateDieDot {canvas x y size pos} {

	set center [expr $size / 2]
	set offset [expr $size / 6]

	switch -- $pos {
		1 {
			incr x $offset
			incr y $offset
		}
		2 {
			incr x [expr $size - $offset]
			incr y $offset
		}
		3 {
			incr x $offset
			incr y [expr $size - $offset]
		}
		4 {
			incr x [expr $size - $offset]
			incr y [expr $size - $offset]
		}
		5 {
			incr x $center
			incr y $center
		}
		6 {
			incr x $offset
			incr y $center
		}
		7 {
			incr x [expr $size - $offset]
			incr y $center
		}
	}

	set radius 3
	$canvas create oval [expr $x - $radius] [expr $y - $radius] \
		[expr $x + $radius] [expr $y + $radius] -fill White -outline White \
		-tags temp
}

# NSBuilding::CreateDie --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc CreateDie {canvas x y size fill face} {

	$canvas create rectangle $x $y [expr $x + $size] [expr $y + $size] \
		-fill $fill -outline ""
	switch -- $face {
		1 {
			CreateDieDot $canvas $x $y $size 5
		}
		2 {
			CreateDieDot $canvas $x $y $size 1
			CreateDieDot $canvas $x $y $size 4
		}
		3 {
			CreateDieDot $canvas $x $y $size 1
			CreateDieDot $canvas $x $y $size 5
			CreateDieDot $canvas $x $y $size 4
		}
		4 {
			CreateDieDot $canvas $x $y $size 1
			CreateDieDot $canvas $x $y $size 2
			CreateDieDot $canvas $x $y $size 3
			CreateDieDot $canvas $x $y $size 4
		}
		5 {
			CreateDieDot $canvas $x $y $size 1
			CreateDieDot $canvas $x $y $size 2
			CreateDieDot $canvas $x $y $size 3
			CreateDieDot $canvas $x $y $size 4
			CreateDieDot $canvas $x $y $size 5
		}
		6 {
			CreateDieDot $canvas $x $y $size 1
			CreateDieDot $canvas $x $y $size 2
			CreateDieDot $canvas $x $y $size 6
			CreateDieDot $canvas $x $y $size 7
			CreateDieDot $canvas $x $y $size 3
			CreateDieDot $canvas $x $y $size 4
		}
	}
}

# NSBuilding::FruitText --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc FruitText slot {
	switch -- $slot {
		1 {
			append text "   ####.\n"
			append text "  #    #\n"
			append text " #     #\n"
			append text "#      #\n"
			append text "#      #\n"
			append text "#     # \n"
			append text "#    #  \n"
			append text ".####   \n"
			append text " Lemon"
		}
		2 {
			append text "   ##   \n"
			append text "  #..#  \n"
			append text " #....# \n"
			append text "#......#\n"
			append text "#......#\n"
			append text " #....# \n"
			append text "  #..#  \n"
			append text "   ##   \n"
			append text " Orange"
		}
		3 {
			append text "   /\\   \n"
			append text "   ##   \n"
			append text "   ##   \n"
			append text "   ##   \n"
			append text "   ##   \n"
			append text "   ##   \n"
			append text " ###### \n"
			append text "   ##   \n"
			append text " Sword"
		}
		4 {
			append text " ###### \n"
			append text "#      #\n"
			append text "# ++++ #\n"
			append text "# +==+ #\n"
			append text "#  ++  #\n"
			append text " #    # \n"
			append text "  #  #  \n"
			append text "   ##   \n"
			append text " Shield"
		}
		5 {
			append text "   ##   \n"
			append text " ###### \n"
			append text "########\n"
			append text "########\n"
			append text "########\n"
			append text " ###### \n"
			append text "  ####  \n"
			append text "   ##   \n"
			append text " Plum"
		}
		6 {
			append text "      ##\n"
			append text "   ###  \n"
			append text "  #..#  \n"
			append text "  #..#  \n"
			append text " ###### \n"
			append text "#..##..#\n"
			append text "#..##..#\n"
			append text " ##  ## \n"
			append text " Cherry"
		}
	}
	return $text
}

# NSBuilding::FruitColor --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc FruitColor slot {

	set colors [list "" Yellow Orange Gray Gray Violet Red]
	return [lindex $colors $slot]
}

# NSBuilding::HandleGamblingHouse --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

# What a big, bleeping mess!
proc HandleGamblingHouse {oop message args} {

	set canvas [Info $oop win].canvas

	switch -- $message {

		craps {
			$canvas delete temp
			set font [Global font,fixed,normal]
			set fh [font metrics $font -linespace]
			set fw [font measure $font W]
			NSMiscWindow::CanvasAddTextItem $canvas $font 2 1 -1 Green "Craps" left temp
			set size 48
			set x [expr $fw * 25 - (($size + 8) * 3 + 8) / 2]
			set y [expr $fh * 6 - $size / 2]
			$canvas create rectangle $x $y [expr $x + ($size + 8) * 3 + 8] [expr $y + $size + 16] \
				-fill Green -outline "" -tags temp
			incr x 8 ; incr y 8
			set faces [lindex $args 0]
			CreateDie $canvas $x $y $size Black [lindex $faces 0]
			CreateDie $canvas [incr x [expr $size + 8]] $y $size Red [lindex $faces 1]
			set font {Times 18}
			$canvas create text [incr x [expr $size + 8 + $size / 2]] [expr $y + $size / 2] \
				-text [lindex $faces 2] -font $font -fill Black -tags temp -anchor center
		}

		diceslots {
			$canvas delete temp
			set font [Global font,fixed,normal]
			set fh [font metrics $font -linespace]
			set fw [font measure $font W]
			NSMiscWindow::CanvasAddTextItem $canvas $font 2 1 -1 Green "Dice Slots" left temp
			set x [expr $fw * 15]
			set y [expr $fh * 3]
			scan [lindex $args 0] "%d %d %d" slot1 slot2 slot3
			$canvas create text $x $y \
				-font $font -fill [FruitColor $slot1] -text [FruitText $slot1] -tags temp -anchor nw
			$canvas create text [incr x [expr $fw * 10]] $y \
				-font $font -fill [FruitColor $slot2] -text [FruitText $slot2] -tags temp -anchor nw
			$canvas create text [incr x [expr $fw * 10]] $y \
				-font $font -fill [FruitColor $slot3] -text [FruitText $slot3] -tags temp -anchor nw
		}

		inbetween {
			$canvas delete temp
			set font [Global font,fixed,normal]
			set fh [font metrics $font -linespace]
			set fw [font measure $font W]
			NSMiscWindow::CanvasAddTextItem $canvas $font 2 1 -1 Green "In Between" left temp
			set size 48
			set x [expr $fw * 25 - (($size + 8) * 3 + 8) / 2]
			set y [expr $fh * 6 - $size / 2]
			$canvas create rectangle $x $y [expr $x + ($size + 8) * 3 + 8] [expr $y + $size + 16] \
				-fill Green -outline "" -tags temp
			incr x 8 ; incr y 8
			set faces [lindex $args 0]
			CreateDie $canvas $x $y $size Black [lindex $faces 0]
			CreateDie $canvas [incr x [expr $size + 8]] $y $size Red [lindex $faces 1]
			CreateDie $canvas [incr x [expr $size + 8]] $y $size Black [lindex $faces 2]
		}

		wheel {
			$canvas delete temp
			set font [Global font,fixed,normal]
			set fh [font metrics $font -linespace]
			set fw [font measure $font W]
			NSMiscWindow::CanvasAddTextItem $canvas $font 2 1 -1 Green "Wheel" left temp
			set radius 48
			set x [expr $fw * 25]
			set y [expr $fh * 6]
			for {set i 1} {$i <= 10} {incr i} {
				if {$i & 1} {
					set fill Blue
					set tag pieTag1
				} else {
					set fill Green
					set tag pieTag2
				}
				$canvas create arc [expr $x - $radius] [expr $y - $radius] \
					[expr $x + $radius] [expr $y + $radius] -extent 36 -start [expr ($i - 1) * 36] \
					-fill $fill -tags "temp $tag"
			}
			scan [lindex $args 0] "%d %d" choice wheel
			set font2 {Times 18}

			set x2 [expr $x - ($radius + 48)]
			$canvas create text $x2 $y \
				-text $choice -font $font2 -fill White -anchor center -tags "temp choice"
			set y2 [lindex [$canvas bbox choice] 1]
			$canvas create text $x2 [expr $y2 - 4] \
				-text "You bet on:" -font $font -fill White -anchor s -tags temp

			set x2 [expr $x + ($radius + 48)]
			$canvas create text $x2 $y \
				-text $wheel -font $font2 -fill White -anchor center -tags "temp wheel"
			$canvas create text $x2 [expr $y2 - 4] \
				-text "This spin:" -font $font -fill White -anchor s -tags temp

			# Animation
			set fill1 Green ; set fill2 Blue
			for {set i 1} {$i <= 4} {incr i} {
				$canvas itemconfigure pieTag1 -fill $fill1
				$canvas itemconfigure pieTag2 -fill $fill2
				set temp $fill1 ; set fill1 $fill2 ; set fill2 $temp
				$canvas itemconfigure wheel -text [string index " 4792" $i]
				update idletasks ; after 20
			}
			$canvas itemconfigure wheel -text $wheel
		}

		loser {
			set wager [lindex $args 0]
			set font [Global font,fixed,normal]
			NSMiscWindow::CanvasAddTextItem $canvas $font 37 13 -1 Red "You lost $wager gp" left temp
			NSMiscWindow::CanvasAddTextItem $canvas $font 37 15 -1 White "Play again? (y/n)" left temp
		}

		wager {

			# This is the first thing called for any gambling screen
			# Clear display completely here
			$canvas delete all

			set wager [lindex $args 0]
			set font [Global font,fixed,normal]
			NSMiscWindow::CanvasAddTextItem $canvas $font 2 15 -1 White [format "Current Wager:    %9ld" $wager] left {}

			set gold [angband player gold]
			NSMiscWindow::CanvasAddTextItem $canvas $font 2 14 -1 White [format "Gold before game: %9ld" $gold] left {}
		}

		winner {
			scan [lindex $args 0] "%d %d" wager odds
			set font [Global font,fixed,normal]
			NSMiscWindow::CanvasAddTextItem $canvas $font 37 13 -1 Gold "YOU WON $wager gp" left temp
			NSMiscWindow::CanvasAddTextItem $canvas $font 37 14 -1 White "Payoff $odds-to-1" left temp
			NSMiscWindow::CanvasAddTextItem $canvas $font 37 15 -1 White "Play again? (y/n)" left temp
		}
	}
}

# NSBuilding::FixItem --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc FixItem {oop message args} {

	variable Priv
	set canvas [Info $oop win].canvas

	set font [Global font,fixed,normal]
	set fh [font metrics $font -linespace]
	set fw [font measure $font W]

	switch -- $message {
		init {
			$canvas delete all
			NSMiscWindow::CanvasAddTextItem $canvas $font 2 1 -1 Green "Enchant Item" left temp
			set prompt [lindex $args 0]
			NSMiscWindow::CanvasAddTextItem $canvas $font 2 3 -1 White $prompt left temp
			set Priv(fixitem,count) 0
		}
		display_one {
			set row [expr $Priv(fixitem,count) * 3 + 5]
			set desc [lindex $args 0]
			set result [lindex $args 1]
			NSMiscWindow::CanvasAddTextItem $canvas $font 4 $row -1 White $desc left temp
			NSMiscWindow::CanvasAddTextItem $canvas $font 8 [incr row] -1 White $result left temp
			incr Priv(fixitem,count)
		}
	}
}

# NSBuilding::Quest --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc Quest {oop message args} {

	set canvas [Info $oop win].canvas

	set font [Global font,fixed,normal]
	set fh [font metrics $font -linespace]
	set fw [font measure $font W]

	switch -- $message {
		info {
			$canvas delete all
			set qname [lindex $args 0]
			set qlevel [lindex $args 1]
			set qtext [lindex $args 2]
			NSMiscWindow::CanvasAddTextItem $canvas $font 2 1 -1 Green "Quest information (Danger level: $qlevel)" left temp
			NSMiscWindow::CanvasAddTextItem $canvas $font 2 3 -1 White $qname left temp
			NSMiscWindow::CanvasAddTextItem $canvas $font 4 5 -1 Yellow $qtext left "temp qtext"
			set y2 [lindex [$canvas bbox qtext] 3]
			set row [expr $y2 / $fh + 1]
			if {$y2 % $fh} {incr row}
#			set prompt "Complete this quest and return for a reward."
#			NSMiscWindow::CanvasAddTextItem $canvas $font 2 $row -1 White $prompt left temp
		}
	}
}

# NSBuilding::FlagText --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc FlagText {oop flag} {

	switch -- $flag {
		SLAY_ANIMAL {return "vs Animals"}
		SLAY_EVIL {return "vs Evil"}
		SLAY_UNDEAD {return "vs Undead"}
		SLAY_DEMON {return "vs Demons"}
		SLAY_ORC {return "vs Orcs"}
		SLAY_TROLL {return "vs Trolls"}
		SLAY_GIANT {return "vs Giants"}
		SLAY_DRAGON {return "vs Dragons"}
		KILL_DRAGON {return "vs Dragons"}
		BRAND_ACID {return "Acid brand"}
		BRAND_ELEC {return "Lightning brand"}
		BRAND_FIRE {return "Fire brand"}
		BRAND_COLD {return "Frost brand"}
		BRAND_POIS {return "Poison brand"}
		default {
			return $flag
		}
	}
}

# NSBuilding::CompareWeapons --
#
#	Description.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc CompareWeapons {oop message args} {

	variable Priv

	set canvas [Info $oop win].canvas

	set font [Global font,fixed,normal]
	set fh [font metrics $font -linespace]
	set fw [font measure $font W]

	switch -- $message {
		init {
			$canvas delete all
			set prompt [lindex $args 0]
			NSMiscWindow::CanvasAddTextItem $canvas $font 2 1 -1 Green $prompt left temp
			set Priv(compare,count) 0
		}
		list_weapon {
			set desc [lindex $args 0]
			scan [lindex $args 1] "%d %d %d %d %d" Priv(compare,to_h) \
				Priv(compare,to_d) Priv(compare,dd) Priv(compare,ds) \
				Priv(compare,num_blow)

			incr Priv(compare,count)
			if {$Priv(compare,count) == 1} {
				set col 2
			} else {
				set col 40
			}
			set row 2
			NSMiscWindow::CanvasAddTextItem $canvas $font $col [incr row] -1 Yellow $desc left temp

			set string "To Hit: $Priv(compare,to_h)   To Damage: $Priv(compare,to_d)"
			NSMiscWindow::CanvasAddTextItem $canvas $font $col [incr row] -1 White $string left temp

			set string "Dice: $Priv(compare,dd)   Sides: $Priv(compare,ds)"
			NSMiscWindow::CanvasAddTextItem $canvas $font $col [incr row] -1 White $string left temp

			set string "Number of Blows: $Priv(compare,num_blow)"
			NSMiscWindow::CanvasAddTextItem $canvas $font $col [incr row] -1 White $string left temp

			incr row
			set string "Base Damage:"
			NSMiscWindow::CanvasAddTextItem $canvas $font $col [incr row] -1 Yellow $string left temp

			incr col
			set dam1 [expr $Priv(compare,dd) + $Priv(compare,to_d)]
			set dam2 [expr $Priv(compare,dd) * $Priv(compare,ds) + $Priv(compare,to_d)]
			set string "One Strike: $dam1-$dam2 damage"
			NSMiscWindow::CanvasAddTextItem $canvas $font $col [incr row] -1 White $string left temp

			set dam1 [expr $Priv(compare,num_blow) * $dam1]
			set dam2 [expr $Priv(compare,num_blow) * $dam2]
			set string "One Attack: $dam1-$dam2 damage"
			NSMiscWindow::CanvasAddTextItem $canvas $font $col [incr row] -1 White $string left temp

			set Priv(compare,flagCount) 0
		}
		flag {
			set flag [lindex $args 0]
			scan [lindex $args 1] "%d" mult

			if {$Priv(compare,count) == 1} {
				set col 2
			} else {
				set col 40
			}
			set row [expr 11 + [incr Priv(compare,flagCount)]]

			if {$Priv(compare,flagCount) == 1} {
				set string "Damage Multipliers: "
				NSMiscWindow::CanvasAddTextItem $canvas $font $col 12 -1 Yellow $string left temp
			}

			incr col
			set string "[FlagText $oop $flag] (x$mult): "
			NSMiscWindow::CanvasAddTextItem $canvas $font $col [incr row] -1 White $string left temp

			incr col [string length $string]
			set dam1 [expr $mult * $Priv(compare,num_blow) * ($Priv(compare,dd) + $Priv(compare,to_d))]
			set dam2 [expr $mult * $Priv(compare,num_blow) * ($Priv(compare,dd) * $Priv(compare,ds) + $Priv(compare,to_d))]
			set string "$dam1-$dam2 damage"
			NSMiscWindow::CanvasAddTextItem $canvas $font $col $row -1 White $string left temp
		}
	}
}

# namespace eval NSBuilding
}

# angband_building --
#
#	A bit of ugliness to handle building irregularites.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc angband_building {message args} {

	global Windows

	set oop $Windows(building,oop)
	set win [NSBuilding::Info $oop win]
	set canvas $win.canvas

	set font [Global font,fixed,normal]

	switch -- $message {
		"compare_weapons" {
			eval NSBuilding::CompareWeapons $oop $args
			return
		}
		"fix_item" {
			eval NSBuilding::FixItem $oop $args
			return
		}
		"quest" {
			eval NSBuilding::Quest $oop $args
			return
		}
		"words" {
			$canvas delete all
			set text [lindex $args 0]
			NSMiscWindow::CanvasAddTextItem $canvas $font 2 1 -1 White $text left temp
			return
		}
		craps -
		diceslots -
		inbetween -
		wheel -
		loser -
		wager -
		winner	{
			eval NSBuilding::HandleGamblingHouse $oop $message $args
		}
	}
}
