
/* borg_goals.c  main goal routines */
#include "angband.h"

#ifdef AUTO_PLAY

/* this is not the main routine */
#define EXTERN extern
#include "borg.h"
#undef EXTERN
#define EXTERN /* normal */


/*
 * Extract the "NNN" from "(with NNN turns of lite)", or return (-1)
 */
EXTERN int extract_fuel(cptr tail)
{
    /* Require the prefix and suffix */
    if (!prefix(tail, "(with ")) return (-1);
    if (!suffix(tail, " of light)")) return (-1);

    /* Extract and return the "turns of lite" */
    return (atoi(tail + 6));
}


/*
 * Extract the "NNN" from "(NNN charges)", or return (-1)
 */
EXTERN int extract_charges(cptr tail)
{
    /* Require the prefix and suffix */
    if (!prefix(tail, "(")) return (-1); /* --(-- */
    if (!suffix(tail, " charge)") && !suffix(tail, " charges)")) return (-1);

    /* Extract and return the "charges" */
    return (atoi(tail + 1));
}



/*
 * Determine if an item kind is "easy" to identify
 * We can assume that the item kind is known.
 */
EXTERN bool obvious_kind(int kind)
{
    /* Analyze the "tval" */
    switch (k_list[kind].tval) {
	case TV_MAGIC_BOOK:
	case TV_PRAYER_BOOK:
	case TV_FLASK:
	case TV_FOOD:
	case TV_POTION:
	case TV_SCROLL:
	case TV_SPIKE:
	case TV_SKELETON:
	case TV_BOTTLE:
	case TV_JUNK:
	    return (TRUE);
    }

    /* Nope */
    return (FALSE);
}



/*
 * Given a weapon/armor, find the slot it will be wielded into
 *
 * Returns "-1" if the item cannot (or should not) be wielded
 *
 * Note that "Bows" really are not proper weapons at all...
 */
EXTERN int wield_slot(int tval)
{
    /* Slot for equipment */
    switch (tval) {

	case TV_DRAG_ARMOR:
	case TV_HARD_ARMOR:
	case TV_SOFT_ARMOR:
	    return (INVEN_BODY);

	case TV_CLOAK:
	    return (INVEN_OUTER);

	case TV_SHIELD:
	    return (INVEN_ARM);

	case TV_HELM:
	    return (INVEN_HEAD);

	case TV_GLOVES:
	    return (INVEN_HANDS);

	case TV_BOOTS:
	    return (INVEN_FEET);

	case TV_SWORD:
	case TV_POLEARM:
	case TV_HAFTED:
	case TV_DIGGING:
	    return (INVEN_WIELD);

	case TV_BOW:
	    return (INVEN_BOW);

	case TV_RING:
#ifndef FRITS
	    return (-1);
#else
	    return (INVEN_LEFT);
#endif /* FRITS */
	    
	case TV_AMULET:
#ifndef FRITS
	    return (-1);
#else
	    return (INVEN_NECK);
#endif /* FRITS */
	    
	case TV_LITE:
	    return (INVEN_LITE);
    }

    /* No slot available */
    return (-1);
}




/*
 * Analyze a (clean) auto_item based on a description
 *
 * Currently, "special artifacts", including the "Phial", are not
 * recognized or parsed in any way.  This really only matters for
 * the Phial, so the Borg will just have to use a lantern.
 *
 * The Borg must examine the "object names" in reverse order, so that
 * he will not think that a "mace of disruption" (etc) is a "mace" with
 * some form of special powers.  It may be possible to somehow handle
 * these "fake special objects" in a "better" way.  Be careful before
 * attempting to use "searching" on the object names.
 *
 * It may be a good idea for the "x_list" array to do "pre-processing"
 * on the "single"/"plural" forms of object names, since this is done
 * so often on the "objdes()" routine.  It might even be a good idea
 * to save these forms in a file *instead* of the "encoded" forms.
 * If this is done, the Borg can use these forms directly (?)
 */
EXTERN void borg_item_analyze(auto_item *item, cptr desc, cptr cost)
{
    int i;

    cptr base = NULL;
    
    char *scan;
    char *tail;
    
    char temp[128];

    char p1 = '(', p2 = ')';
    char b1 = '[', b2 = ']';

#ifdef FRITSDEBUG
   borg_note("In routine borg_item_analyze()");
#endif /* FRITSDEBUG */
    
    /* Wipe it */
    WIPE(item, auto_item);
	
    /* Empty items are empty */
    if (!desc[0]) return;


    /* Save the item description */
    strcpy(item->desc, desc);

    /* Extract the item cost */
    item->cost = atol(cost);


    /* Assume singular */
    item->iqty = 1;
    
    /* Notice various "prefixes" */
    if (prefix(desc, "The ")) desc += 4;
    else if (prefix(desc, "an ")) desc += 3;
    else if (prefix(desc, "a ")) desc += 2;
    else if (prefix(desc, "1 ")) desc += 2;

    /* Notice "numerical" prefixes */
    else if (isdigit(desc[0])) {
	cptr s = strchr(desc, ' ');
	if (!s) return;
	item->iqty = atoi(desc);
	desc = s + 1;
    }

    /* Extract and remove the inscription */
    strcpy(temp, desc);
    scan = strchr(temp, '{' /* --}-- */);
    if (scan) strcpy(item->note, scan);
    if (scan && (scan[-1] == ' ')) scan[-1] = '\0';


    /* Hack -- non-aware, singular, flavored items */
    if (item->iqty == 1) {
	if (prefix(temp, "Scroll titled ")) item->tval = TV_SCROLL;
	else if (suffix(temp, " Potion")) item->tval = TV_POTION;
	else if (suffix(temp, " Hairy Mold")) item->tval = TV_FOOD;
	else if (suffix(temp, " Mushroom")) item->tval = TV_FOOD;
	else if (suffix(temp, " Amulet")) item->tval = TV_AMULET;
	else if (suffix(temp, " Ring")) item->tval = TV_RING;
	else if (suffix(temp, " Staff")) item->tval = TV_STAFF;
	else if (suffix(temp, " Wand")) item->tval = TV_WAND;
	else if (suffix(temp, " Rod")) item->tval = TV_ROD;
    }

    /* Hack -- non-aware, plural, flavored items */
    else {
	if (prefix(temp, "Scrolls titled ")) item->tval = TV_SCROLL;
	else if (suffix(temp, " Potions")) item->tval = TV_POTION;
	else if (suffix(temp, " Hairy Molds")) item->tval = TV_FOOD;
	else if (suffix(temp, " Mushrooms")) item->tval = TV_FOOD;
	else if (suffix(temp, " Amulets")) item->tval = TV_AMULET;
	else if (suffix(temp, " Rings")) item->tval = TV_RING;
	else if (suffix(temp, " Staffs")) item->tval = TV_STAFF;
	else if (suffix(temp, " Wands")) item->tval = TV_WAND;
	else if (suffix(temp, " Rods")) item->tval = TV_ROD;
    }

    /* Accept non-aware flavored objects */
    if (item->tval) return;
    

    /* Check all the item templates */
    for (i = 0; i < i_size; i++) {

	/* Extract the "base" string */
	base = (item->iqty == 1) ? i_single[i] : i_plural[i];

	/* Check for the proper prefix */
	if (prefix(temp, base)) break;
    }

    /* Oops.  Bizarre item. */
    if (i >= i_size) return;
    
    /* Skip past the base name */
    tail = temp + strlen(base);

    
    /* Save the item kind */
    item->kind = i_kind[i];

    /* Extract some info */
    item->tval = k_list[item->kind].tval;
    item->sval = k_list[item->kind].sval;


    /* Hack -- check for ego-items and artifacts */
    if ((tail[0] == ' ') &&
        (item->tval >= TV_MIN_WEAR) && (item->tval <= TV_MAX_WEAR)) {
    
	/* Hack -- check for ego items */
	for (i = 0; i < EGO_MAX; i++) {

	    /* Check for an ego name */
	    if (ego_names[i] && prefix(tail+1, ego_names[i])) {

		/* Save the ego name */
		item->name2 = i;
		
		/* Skip the space and the ego name */
		tail += strlen(ego_names[i]) + 1;

		/* And stop looking */
		break;
	    }
	}

	/* Hack -- check for artifacts */
	for (i = 0; i < ART_MAX; i++) {

	    /* Check for an ego name */
	    if (v_list[i].name && prefix(tail+1, v_list[i].name)) {

		/* Save the artifact name */
		item->name1 = i;
		
		/* Hack -- forget any ego-name */
		item->name2 = 0;
		
		/* Skip the space and the artifact name */
		tail += strlen(v_list[i].name) + 1;

		/* And stop looking */
		break;
	    }
	}
    }
    
    
    /* Hack -- skip any final spaces */
    while (tail[0] == ' ') tail++;
    
    
    /* XXX XXX Hack -- Chests are too complicated */
    if (item->tval == TV_CHEST) {
	return;
    }
    
    /* Hack -- Some kinds of objects are always obvious */
    if (obvious_kind(item->kind)) {
	item->able = TRUE;
	return;
    }
    
    /* Hack -- Examine Wands/Staffs for charges */
    if ((item->tval == TV_WAND) || (item->tval == TV_STAFF)) {
	i = extract_charges(tail);
	if (i >= 0) item->able = TRUE;
	if (item->able) item->pval = i;
	return;
    }

    /* Hack -- Examine Rods for charging */
    if (item->tval == TV_ROD) {

	/* Rods must be known */
	item->able = TRUE;

	/* Notice "charging" */
	if (streq(tail, "(charging)")) item->pval = 999;
    }


    /* Hack -- Extract Lites for Light */
    if (item->tval == TV_LITE) {

	/* Fuels yields known (and fuel) */
	i = extract_fuel(tail);
	if (i >= 0) item->pval = i;
	if (i >= 0) item->able = TRUE;

	/* Hack -- Artifacts have infinite fuel */
	if (item->name1) item->pval = 29999;
	if (item->name1) item->able = TRUE;

	/* Done */
	return;
    }


    /* Wearable stuff */
    if ((item->tval >= TV_MIN_WEAR) && (item->tval <= TV_MAX_WEAR)) {

	bool done = FALSE;
	
	int d1 = 0, d2 = 0, ac = 0, th = 0, td = 0, ta = 0;
	

	/* Hack -- examine the "easy know" flag */
	if (k_list[item->kind].flags3 & TR3_EASY_KNOW) {
	    item->able = TRUE;
	}


	/* Must have a suffix */
	if (!tail[0]) return;
	
	/* Parse "weapon-style" damage strings */
	if ((tail[0] == p1) &&
	    ((item->tval == TV_HAFTED) ||	
	    (item->tval == TV_POLEARM) ||	
	    (item->tval == TV_SWORD) ||	
	    (item->tval == TV_DIGGING) ||	
	    (item->tval == TV_BOLT) ||	
	    (item->tval == TV_ARROW) ||	
	    (item->tval == TV_SHOT))) {
	
	    /* First extract the damage string */
	    for (scan = tail; *scan != p2; scan++); scan++;

	    /* Notice "end of string" */
	    if (scan[0] != ' ') done = TRUE;

	    /* Terminate the string and advance */
	    *scan++ = '\0';

	    /* Parse the damage string, or stop XXX */
	    if (sscanf(tail, "(%dd%d)", &d1, &d2) != 2) return;

	    /* Save the values */
	    item->dd = d1; item->ds = d2;
	    
	    /* No extra information means not identified */
	    if (done) return;

	    /* Start on the damage bonus string */
	    tail = scan;
	}
	
	/* Parse the "damage" string for bows */
	else if ((tail[0] == p1) &&
	    (item->tval == TV_BOW)) {
	
	    /* First extract the damage string */
	    for (scan = tail; *scan != p2; scan++); scan++;

	    /* Notice "end of string" */
	    if (scan[0] != ' ') done = TRUE;

	    /* Terminate the string and advance */
	    *scan++ = '\0';

	    /* Parse the multiplier string, or stop */
	    if (sscanf(tail, "(x%d)", &d1) != 1) return;

	    /* No extra information means not identified */
	    if (done) return;

	    /* Start on the damage bonus string */
	    tail = scan;
	}
	

	/* Parse the "bonus" string */
	if (tail[0] == p1) {
	
	    /* XXX Extract the extra info */
	    for (scan = tail; *scan != p2; scan++); scan++;

	    /* Notice "end of string" */
	    if (scan[0] != ' ') done = TRUE;

	    /* Terminate the damage, advance */
	    *scan++ = '\0';

	    /* XXX XXX XXX XXX This will "eat" some "pval" codes */
	    /* Especially on rings/amulets, and some artifact armor */
	    
	    /* Parse the "bonuses" -- XXX Guess at weird ones */
	    if ((sscanf(tail, "(%d,%d)", &th, &td) != 2) &&
	        (sscanf(tail, "(%d)", &th) != 1)) return;

	    /* Save the values */
	    item->to_h = th; item->to_d = td;

	    /* Known (bonuses) */
	    item->able = TRUE;

	    /* Nothing left */
	    if (done) return;

	    /* Then look for "armor" values */
	    tail = scan;
	}
	

	/* Parse the "bonus" string */
	if (tail[0] == b1) {
	
	    /* XXX Extract the extra info */
	    for (scan = tail; *scan != b2; scan++); scan++;

	    /* Notice "end of string" */
	    if (scan[0] != ' ') done = TRUE;

	    /* Terminate the armor string, advance */
	    *scan++ = '\0';

	    /* Parse the armor, and bonus */
	    if (sscanf(tail, "[%d,%d]", &ac, &ta) == 2) {
		item->ac = ac; item->to_a = ta;
		item->able = TRUE;
	    }

	    /* Just base armor */
	    else if (sscanf(tail, "[%d]", &ac) == 1) {
		item->ac = ac;
	        return;
	    }

	    /* Oops */
	    else {
		return;
	    }
	
	    /* Nothing left */
	    if (done) return;
	}
	

	/* Parse the final "pval" string, if any */
	if (tail[0] == p1) {

	    /* Assume identified */
	    item->able = TRUE;
	    
	    /* Grab it */
	    item->pval = atoi(tail + 1);
	}
    }
}




/*
 * Check the inventory for the quantity of objects of the given type
 */
EXTERN int borg_count_items(int tval, int sval)
{
    int i, k = 0;
    
    /* Check every item */
    for (i = 0; i < INVEN_PACK; i++) {
	auto_item *item = &auto_items[i];
	if ((tval >= 0) && (item->tval != tval)) continue;
	if ((sval >= 0) && (item->sval != sval)) continue;
	k += item->iqty;
    }

    /* Return the count */    
    return (k);
}

 

#if 0

/*
 * Send a command to inscribe item number "i" with the inscription "str".
 */
EXTERN void borg_send_inscribe(int i, cptr str)
{
    cptr s;
    
    /* The "inscribe" command */
    char draw = '{'; /* --}-- */

    /* Label it */
    borg_keypress(draw);

    /* Hack -- allow "equipment" labelling */
    if (i >= INVEN_WIELD) {
	borg_keypress('/');
	i -= INVEN_WIELD;
    }

    /* Choose the item */
    borg_keypress('a' + i);

    /* Send the label */
    for (s = str; *s; s++) borg_keypress(*s);

    /* End the inscription */
    borg_keypress('\n');
}

#endif



/*
 * Examine the inventory
 */
EXTERN void borg_notice(void)
{
    int i, slot;
    
#ifdef FRITSDEBUG
   borg_note("In routine borg_notice()");
#endif /* FRITSDEBUG */
    
    /* Analyze the inventory */
    for (i = 0; i < INVEN_PACK; i++) {

	auto_item *item = &auto_items[i];

	auto_item *worn = NULL;


	/* Clear the flags */
	item->wear = item->cash = item->junk = item->test = FALSE;

	/* Skip "empty" / "bizarre" items */
	if (!item->iqty || !item->tval) continue;

	
	/* Throw away total "junk" */
	if ((item->tval <= TV_SPIKE) ||
	    (item->kind && (k_list[item->kind].cost <= 0)) ||
	    (suffix(item->desc, " {cursed}")) ||
	    (suffix(item->desc, " {worthless}")) ||
	    (suffix(item->desc, " {junk}"))) {

	    /* This is junk */
	    item->junk = TRUE;

	    continue;
	}

	/* Identify "good" etc items */
	if ((suffix(item->desc, " {good}")) ||
	    (suffix(item->desc, " {blessed}")) ||
	    (suffix(item->desc, " {special}")) ||
	    (suffix(item->desc, " {excellent}"))) {

	    /* Test this */
	    item->test = TRUE;

	    continue;
	}

	/* Process books */
#ifndef FRITS
	if ((item->tval == TV_MAGIC_BOOK) ||
	    (item->tval == TV_PRAYER_BOOK)) 
        {
	    /* Sell it */
	    item->cash = TRUE;

	    continue;
	}
#else
	if (item->tval == TV_MAGIC_BOOK) {
           if ((sclass!=SCLASS_MAGE)|| (borg_count_items(item->tval,item->sval)>6))
	    item->cash = TRUE;
            continue;
        }
	if (item->tval == TV_PRAYER_BOOK) {
           if ((sclass!=SCLASS_PRIEST)|| (borg_count_items(item->tval,item->sval)>6))
	    item->cash = TRUE;
            continue;
        }
#endif /* FRITS */
	
	/* Process scrolls */
	if (item->tval == TV_SCROLL) {

	    /* Destroy known crap */
	    if (item->kind && (k_list[item->kind].cost <= 0L)) {
		item->junk = TRUE;
	    }

#ifdef FRITS
	    /* Identify stuff of higher levels */
            if ( (item->kind == FALSE) && (auto_depth > 5) ) {
                item->test = TRUE;
            }
#endif /* FRITS */

	    /* XXX Note useful stuff */
	    
	    /* Save scrolls of identify */
	    else if (item->kind == OBJ_SCROLL_IDENTIFY) {

		/* Sell "excess" items */
		if (borg_count_items(item->tval, item->sval) > 50) {
		    item->cash = TRUE;
		}
	    }
	    
	    /* Save scrolls of word of recall */
	    else if (item->kind == OBJ_SCROLL_RECALL) {

		/* Sell "excess" items */
		if (borg_count_items(item->tval, item->sval) > 10) {
		    item->cash = TRUE;
		}
	    }
	    
	    /* Sell the rest */
	    else {
#ifdef FRITS
                switch(item->kind) {

                   case OBJ_SCROLL_ENCHANT_HIT:
                   case OBJ_SCROLL_ENCHANT_DAM:
                   case OBJ_SCROLL_ENCHANT__WE:
                   case OBJ_SCROLL_ENCHANT_AC:
                   case OBJ_SCROLL_ENCHANT__AC:
                   item->cash = FALSE;
                   break;
                
                   default:
#endif /* FRITS */
		   item->cash = TRUE;
#ifdef FRITS
                   break;
                }
#endif /* FRITS */
	    }

	    continue;
	}

	/* Process potions */
	if (item->tval == TV_POTION) {

	    /* Destroy known crap */
	    if (item->kind && (k_list[item->kind].cost <= 0L)) {
		item->junk = TRUE;
	    }

#ifdef FRITS
	    /* Identify stuff of higher levels */
            if ( (item->kind == FALSE) && (auto_depth > 5) ) {
                item->test = TRUE;
            }
#endif /* FRITS */

	    /* XXX Note useful stuff */

	    /* Save potions of cure critical wounds */
	    else if (item->kind == OBJ_POTION_CRITICAL) {

		/* Sell "excess" items */
		if (borg_count_items(item->tval, item->sval) > 25) {
		    item->cash = TRUE;
		}
	    }
	    
	    /* Sell the rest */
	    else {
#ifdef FRITS
                switch(item->kind) {

                   case OBJ_POTION_STR:
                   case OBJ_POTION_INT:
                   case OBJ_POTION_WIS:
                   case OBJ_POTION_DEX:
                   case OBJ_POTION_CON:
                   case OBJ_POTION_CHR:
                   case OBJ_POTION_EXP:
                   case OBJ_POTION_AUGMENT:
                   case OBJ_POTION__ENLIGHT:
                   case OBJ_POTION_REST_STR:
                   case OBJ_POTION_REST_INT:
                   case OBJ_POTION_REST_WIS:
                   case OBJ_POTION_REST_DEX:
                   case OBJ_POTION_REST_CON:
                   case OBJ_POTION_REST_CHR:
                   case OBJ_POTION_REST_LIFE:
                     borg_note("Will not sell stat potions");
                     item->cash = FALSE;
                   break;

                   default:
#endif /* FRITS */
		     item->cash = TRUE;
#ifdef FRITS
                   break;
                }
#endif /* FRITS */
               
	    }

	    continue;
	}

	/* Process rods */
	if (item->tval == TV_ROD) {

	    /* Destroy known crap */
	    if (item->kind && (k_list[item->kind].cost <= 0L)) {
		item->junk = TRUE;
	    }

	    /* XXX Note useful stuff */
	    
	    /* Sell the rest */
	    else if (item->kind) {
		item->cash = TRUE;
	    }

	    /* Identify non-aware rods */
	    else {
		item->test = TRUE;
	    }

	    continue;
	}


	/* Process wands/staffs */
	if ((item->tval == TV_WAND) ||
	    (item->tval == TV_STAFF)) {

	    /* Destroy known crap */
	    if (item->kind && (k_list[item->kind].cost <= 0L)) {
		item->junk = TRUE;
	    }

	    /* XXX Note useful stuff */
	    
	    /* Sell known items */
	    else if (item->able) {
		item->cash = TRUE;
	    }

	    /* Identify charges */
	    else if (item->kind) {
		item->test = TRUE;
	    }
	    
	    /* Non-aware item */
	    else {
		item->test = TRUE;
		item->cash = TRUE;
	    }
	    
	    continue;
	}


	/* Process rings/amulets */
	if ((item->tval == TV_RING) ||
	    (item->tval == TV_AMULET)) {

	    /* XXX Note useful stuff */
	    
	    /* Destroy known "crap" */
	    if (item->kind && (k_list[item->kind].cost <= 0L)) {
		item->junk = TRUE;
	    }

	    /* XXX Attempt to analyze the "pval" code */
#ifdef FRITS
            if (item->kind) {
               switch(item->kind) {
                 case OBJ_RING_SEE_INVIS:
                   borg_note ("Found a ring of see invis!!");
                   item->wear=TRUE;
                 break;

                 case OBJ_RING_PROT:
                   if (strchr(item->desc,'['))
                   {
                      item->test=FALSE;
                   } else {
                      item->test=TRUE;
                      borg_note(format("not fully identified ring: %s",item->desc));
                   }
                   item->cash=TRUE;
                 break;

                 case OBJ_RING_SLAYING:
                 case OBJ_RING_SEARCH:
                 case OBJ_RING_HIT:
                 case OBJ_RING_DAM:
                 case OBJ_RING_STR:
                 case OBJ_RING_INT:
                 case OBJ_RING_DEX:
                 case OBJ_RING_CON:
                   if (strchr(item->desc,'(')) /* vi happy ) */
                   {
                      item->test=FALSE;
                   } else {
                      item->test=TRUE;
                      borg_note(format("not fully identified ring: %s",item->desc));
                   }
                   item->cash=TRUE;
                 break;

                 case OBJ_RING_SPEED:
                   if (strchr(item->desc,'(')) /* vi happy ) */
                   {
                      borg_oops(format("***RING OF SPEED: %s ***",item->desc));
                      item->test=FALSE;
                   } else {
                      item->test=TRUE;
                      borg_note(format("not fully identified ring: %s",item->desc));
                   }
                   item->cash=TRUE;
                 break;

                 default:
#ifdef FRITSDEBUG
                   borg_note(format("simple ring %s",item->desc));
#endif /* FRITSDEBUG */
                   item->able=TRUE;
                   item->cash=TRUE;
                 break;
               }
            } else { /* unknown ring/amulet */
              item->test=TRUE;
              item->cash=TRUE;
            }
#endif /* FRITS */

	    continue;
	}


	/* Mega-Hack -- Junk chests */
	if (item->tval == TV_CHEST) {

	    /* Throw it away */
#ifndef FRITS
	    item->junk = TRUE;
#else
	    do_chest=TRUE;
#endif /* FRITS */

	    continue;
	}


	/* Sell excess food */
	if (item->tval == TV_FOOD) {

	    /* Sell unknown or "dungeon" food */
	    if (item->sval < SV_FOOD_MIN_FOOD) {
		item->cash = TRUE;
	    }

#ifdef FRITS
	/* mages and priests can create food */
           if ( ( (sclass==SCLASS_PRIEST) && (borg_can_cast(sclass,'b','f')) ) ||
                ( (sclass==SCLASS_MAGE) && (borg_can_cast(sclass,'c','a'))) ) {
	      if (borg_count_items(TV_FOOD,-1) > 5)
              item->cash=TRUE;
           }
#endif /* FRITS */

	    continue;
	}


	/* Sell excess flasks */
	if (item->tval == TV_FLASK) {

	    /* Extract the current lite */
	    worn = &auto_items[INVEN_LITE];

	    /* Flasks require a lantern */
	    if (worn->sval != SV_LITE_LANTERN) {
		item->cash = TRUE;
	    }

	    continue;
	}


	/* Ignore unaware items */
	if (!item->kind) continue;


	/* Process Lite's */
	if (item->tval == TV_LITE) {

	    /* Extract the current lite */
	    worn = &auto_items[INVEN_LITE];
	   
	    /* Hack -- identify artifacts */
	    if (!item->able) {
		item->test = TRUE;
	    }
	    
	    /* Hack -- Replace missing lites */
	    else if (!worn->iqty) {
		item->wear = TRUE;
	    }
	    
	    /* Hack -- never wear "empty" Lites */
	    else if (!item->pval) {
		item->cash = TRUE;
	    }
	    
	    /* Hack -- replace empty lites */
	    else if (!worn->pval) {
		item->wear = TRUE;
	    }
	    
	    /* Hack -- Prefer phials to lanterns to torches */
	    else if (k_list[item->kind].cost > k_list[worn->kind].cost) {
		item->wear = TRUE;
	    }

	    /* Notice "junky" torches */
	    else if (item->sval == SV_LITE_TORCH) {

		/* Artifact lites are best */
		if (worn->name1) {
		    item->cash = TRUE;
		}

		/* Lantern plus fuel pre-empts torches */
		else if (worn->sval == SV_LITE_LANTERN) {
		    if (borg_count_items(TV_FLASK, -1) > 10) {
			item->cash = TRUE;
		    }
		}

		/* Sell "excess" torches */
		else if (borg_count_items(TV_LITE, SV_LITE_TORCH) > 30) {
		    item->cash = TRUE;
		}
	    }
	    	    
	    /* Notice "extra" lanterns */
	    else if (item->sval == SV_LITE_LANTERN) {

		/* Artifact lites are best */
		if (worn->name1) {
		    item->cash = TRUE;
		}

		/* Already wielding a lantern */
		else if (worn->sval == SV_LITE_LANTERN) {
		    item->cash = TRUE;
		}

		/* Already carrying a lantern */
		else if (borg_count_items(TV_LITE, SV_LITE_LANTERN) > 1) {
		    item->cash = TRUE;
		}
	    }
	    	    
	    continue;
	}



	/* XXX XXX No analysis done on missiles currently */
	
	
	/* See what slot that item could go in */
	slot = wield_slot(item->tval);
		
	/* Ignore unwearable items */
	if (slot < 0) continue;
	
	/* Extract the item currently in that slot */
	worn = &auto_items[slot];

	/* Known (or average) items can be worn */
	if (item->able || suffix(item->desc, " {average}")) {

	    s32b worth = item_worth(item);
	    s32b other = item_worth(worn);
	    
	    /* Throw away "junk" and damaged items */
	    if (worth <= 0) {
		item->junk = TRUE;
	    }

	    /* Always replace missing equipment */
	    else if (!worn->iqty) {
		item->wear = TRUE;
	    }
	    
	    /* Compare items -- Hack -- by "price" */
	    else if (worth > other) {
		item->wear = TRUE;
	    }

	    /* Sell all NORMAL items */
	    else if (!item->name1 && !item->name2) {
		item->cash = TRUE;
	    }
	    
	    continue;
	}
    }
}



    
/*
 * Use things in a useful manner
 */
EXTERN bool borg_use_things(void)
{
    int i;

#ifdef FRITSDEBUG
   borg_note("In routine borg_use_things()");
#endif /* FRITSDEBUG */

/* oops moved to after healup */


    /* XXX XXX XXX Hack -- heal */
    if ((auto_curhp < auto_maxhp / 2) ||
        do_blind || do_confused || do_poisoned) {

#ifdef FRITS
	/* don't use 'em all up! esp. for poison (2 in 3) */
	if ((do_poisoned) && (rand_int(3) > 0))
	{
	   /* Neutralize Poison */
	   if ((sclass==SCLASS_PRIEST) && (borg_can_cast(sclass,'c','a')))
	   {
              borg_cast_spell(sclass,'c','a');
	      return(TRUE);
           }
	   /* Cure Poison */
	   if ((sclass==SCLASS_MAGE) && (borg_can_cast(sclass,'b','f')))
	   {
              borg_cast_spell(sclass,'b','f');
	      return(TRUE);
           }
	}

	if ((do_blind) || (do_confused)) {
	   /* just sweat it out, 1 in 3 */
	   if (rand_int(3) == 0) return(TRUE);
	}

        /* priests are expert at curing */
        if (
            (sclass==SCLASS_PRIEST) && 
            (auto_curmana > auto_maxmana/3) &&
            (!(do_blind||do_confused)) && 
            (rand_int(2)>0) &&
            /* keep mana for Portal spell */
            (auto_curmana > 2*manacost[pclass]['b'-'a']['b'-'a'])
           ) {
             switch(rand_int(5)+auto_level/10) {
                case 0:
                case 1:
                  borg_cast_spell(sclass,'a','b'); /* Cure Light Wounds */
                break;

                case 2:
                case 3:
                  borg_cast_spell(sclass,'b','c'); /* Cure Medium Wounds */
                break;

                case 4:
                  borg_cast_spell(sclass,'c','c'); /* Cure Serious Wounds */
                break;

                case 5:
                case 6:
                  borg_cast_spell(sclass,'c','h'); /* Cure Critical Wounds */
                break;

                case 7:
                case 8:
                  borg_cast_spell(sclass,'d','c'); /* Heal */
                break;
 
                default:
                  borg_cast_spell(sclass,'d','c'); /* Purifications spell? */
                break;

             }
             return(TRUE); 
        }
#endif /* FRITS */

	/* Scan the inventory */
	for (i = 0; i < INVEN_PACK; i++) {

	    /* Quaff healing potions */
	    if (auto_items[i].kind == OBJ_POTION_CRITICAL) {
	    
		borg_keypress('q');
		borg_keypress('a' + i);

		return (TRUE);
	    }
	}
    }

    /* Heal or abort */
    if (do_heal) {
	borg_oops("near death");
	return (TRUE);
    }

#ifdef FRITS
    /* do_afraid */
    if (do_afraid) {
       if ((sclass==SCLASS_PRIEST) && (rand_int(5)==0) ) {
          borg_cast_spell(SCLASS_PRIEST,'a','d');
          return(TRUE);
       }
    }

    if (drained_stat != 0) {
       for (i=0;i<INVEN_PACK;i++) {
           switch(auto_items[i].kind) {
              case OBJ_POTION_REST_STR:
                 if ((drained_stat & (1<<0))) {
                    borg_note("Restoring Strength.");
                    borg_keypress('q');
                    borg_keypress('a'+i);
                    drained_stat = drained_stat ^ (1<<0); 
                    return(TRUE);
                 }
              break;

              case OBJ_POTION_REST_INT:
                 if ((drained_stat & (1<<1))) {
                    borg_note("Restoring Intelligence.");
                    borg_keypress('q');
                    borg_keypress('a'+i);
                    drained_stat = drained_stat ^ (1<<1); 
                    return(TRUE);
                 }
              break;

              case OBJ_POTION_REST_WIS:
                 if ((drained_stat & (1<<2))) {
                    borg_note("Restoring Wisdom.");
                    borg_keypress('q');
                    borg_keypress('a'+i);
                    drained_stat = drained_stat ^ (1<<2); 
                    return(TRUE);
                 }
              break;

              case OBJ_POTION_REST_DEX:
                 if ((drained_stat & (1<<3))) {
                    borg_note("Restoring Dexterity.");
                    borg_keypress('q');
                    borg_keypress('a'+i);
                    drained_stat = drained_stat ^ (1<<3); 
                    return(TRUE);
                 }
              break;

              case OBJ_POTION_REST_CON:
                 if ((drained_stat & (1<<4))) {
                    borg_note("Restoring Constitution.");
                    borg_keypress('q');
                    borg_keypress('a'+i);
                    drained_stat = drained_stat ^ (1<<4); 
                    return(TRUE);
                 }
              break;

              case OBJ_POTION_REST_CHR:
                 if ((drained_stat & (1<<5))) {
                    borg_note("Restoring Charisma.");
                    borg_keypress('q');
                    borg_keypress('a'+i);
                    drained_stat = drained_stat ^ (1<<5); 
                    return(TRUE);
                 }
              break;

              case OBJ_POTION_REST_LIFE:
                 if ((drained_stat & (1<<6))) {
                    borg_note("Restoring Life Levels.");
                    borg_keypress('q');
                    borg_keypress('a'+i);
                    drained_stat = drained_stat ^ (1<<6); 
                    return(TRUE);
                 }
              break;

              default:
              break;
          }
       }
    }
#endif

    /* XXX XXX XXX Hack -- attempt to escape */
    if ((auto_curhp < auto_maxhp / 2) && auto_depth && !auto_recall) {

#ifdef FRITS
        /* first, try to use magic! */
        if (sclass==SCLASS_MAGE)
        {
           if (auto_curhp < auto_maxhp / 4 )  
              if(borg_cast_spell(SCLASS_MAGE,'b','f')) /* TELEPORT */
              {
                 borg_note("escaped?");
                 state=STATE_START;
                 return(TRUE);
              }
           else
              if(borg_cast_spell(SCLASS_MAGE,'a','c')) /* phase door */
              {
                 borg_note("escaped?");
                 state=STATE_START;
                 return(TRUE);
              }
        }
        if (sclass==SCLASS_PRIEST)
        {
           if(borg_cast_spell(SCLASS_PRIEST,'b','b')) /* Portal */
           {
              borg_note("escaped?");
              state=STATE_START;
              return(TRUE);
           }
        }
#endif /* FRITS */

	/* Scan the inventory */
	for (i = 0; i < INVEN_PACK; i++) {

	    /* Read scrolls of Recall */
	    if (auto_items[i].kind == OBJ_SCROLL_RECALL) {

		auto_recall = c_t;
		    
		borg_keypress('r');
		borg_keypress('a' + i);

		return (TRUE);
	    }
	}
    }
        
    
    /* Eat or abort */
    if (do_food) {

	/* Scan the inventory */
	for (i = 0; i < INVEN_PACK; i++) {

	    /* Skip non-food */
	    if (auto_items[i].tval != TV_FOOD) continue;
	    
	    /* Eat food rations and such */
	    if (auto_items[i].sval >= SV_FOOD_MIN_FOOD) {

		borg_keypress('E');
		borg_keypress('a' + i);

		return (TRUE);
	    }
	}

#ifdef FRITS
        /* make it yourself */
        if ((pclass==PCLASS_PRIEST)||(pclass==PCLASS_PALADIN))
        {
           if (borg_cast_spell(SCLASS_PRIEST,'b','f')==TRUE) return(TRUE); /* Satisfy Hunger */
        } else {
           if ((pclass==PCLASS_MAGE)||(pclass==PCLASS_RANGER)||(pclass==PCLASS_ROGUE))
           {
              if(borg_cast_spell(SCLASS_MAGE,'c','a')==TRUE) return(TRUE); /* Satisfy Hunger */
           }
        }
#endif /* FRITS */
    
	/* Hack -- Require food! */
	if (do_starve) {
           borg_oops("starving");
	   return (TRUE);
        }
    }
    
#ifdef FRITS
        /* potions of gain stat */
	for (i = 0; i < INVEN_PACK; i++) {
	    if (auto_items[i].tval == TV_POTION) {
               switch (auto_items[i].kind) {
                  case OBJ_POTION_STR:
                  case OBJ_POTION_INT:
                  case OBJ_POTION_WIS:
                  case OBJ_POTION_DEX:
                  case OBJ_POTION_CON:
                  case OBJ_POTION_CHR:
                  case OBJ_POTION_EXP:
                  case OBJ_POTION_AUGMENT:
                  case OBJ_POTION__ENLIGHT:
                     borg_note("GAIN STAT potion!");
                     borg_keypress('q');
                     borg_keypress('a'+i);
                     return(TRUE);
                  break;
   
                  default:
                  break;
              }
           }

           /* enchant scrolls */
	    if (auto_items[i].tval == TV_SCROLL) {
               switch (auto_items[i].kind) {
                  case OBJ_SCROLL_ENCHANT_HIT:
                  case OBJ_SCROLL_ENCHANT_DAM:
                  case OBJ_SCROLL_ENCHANT__WE:
                  case OBJ_SCROLL_ENCHANT_AC:
                  case OBJ_SCROLL_ENCHANT__AC:

		   if ((!do_blind) && (!do_confused)) {
                     borg_note("Enchantment scroll!");
                     borg_keypress('r');
                     borg_keypress('a'+i);
                     return(TRUE);
		   } else {return(FALSE);}
                  break;
   
                  default:
                  break;
              }
           }
       }

/* chests */
    if (do_chest) {
       borg_note("Attempt to loot a chest");
       for(i=0;i<INVEN_PACK;i++) {
         if (auto_items[i].tval == TV_CHEST) {
            if (suffix(auto_items[i].desc, " (empty)")) {
               borg_note("junking it. empty.");
               auto_items[i].junk=TRUE;
               do_chest=FALSE;
               return(FALSE);
            } 
            if ( (suffix(auto_items[i].desc, " (locked)")) || 
                 (suffix(auto_items[i].desc, " (disarmed)")) ) {
              borg_note("Opening a chest");
              borg_keypress('d');
              borg_keypress('a'+i);
              switch (rand_int(4)) {
                case 0:
                 borg_keypress('4');
                 borg_keypress('o');
                 borg_keypress('6');
                break;
                case 1:
                 borg_keypress('6');
                 borg_keypress('o');
                 borg_keypress('4');
                break;
                case 2:
                 borg_keypress('8');
                 borg_keypress('o');
                 borg_keypress('2');
                break;
                case 3:
                 borg_keypress('2');
                 borg_keypress('o');
                 borg_keypress('8');
                break;
                default:
                break;
              }
              return(TRUE);
            }
            if ( (suffix(auto_items[i].desc, " (Poison Needle)")) ||
                 (suffix(auto_items[i].desc, " (Gas Trap)")) ||
                 (suffix(auto_items[i].desc, " (Explosion Device)")) ||
                 (suffix(auto_items[i].desc, " (Summoning Runes)")) ||
                 (suffix(auto_items[i].desc, " (Multiple Traps)")) ) {
              borg_note("Disarming a chest");
              borg_keypress('d');
              borg_keypress('a'+i);
              switch (rand_int(4)) {
                case 0:
                 borg_keypress('4');
                 borg_keypress('D');
                 borg_keypress('6');
                break;
                case 1:
                 borg_keypress('6');
                 borg_keypress('D');
                 borg_keypress('4');
                break;
                case 2:
                 borg_keypress('8');
                 borg_keypress('D');
                 borg_keypress('2');
                break;
                case 3:
                 borg_keypress('2');
                 borg_keypress('D');
                 borg_keypress('8');
                break;
                default:
                break;
              }
              return(TRUE);
            } else { /* no suffix, have to search */
              borg_note("Searching a chest.");
              borg_keypress('d');
              borg_keypress('a'+i);
              borg_keypress('s');
              return(TRUE);
            }
         }
       }
    }
#endif /* FRITS */

    /* Nothing to destroy */
    return (FALSE);
}


/*
 * Destroy everything we know we don't want
 */
EXTERN bool borg_throw_junk(void)
{
    int i;

    /* Throw away "junk" */
    for (i = 0; i < INVEN_PACK; i++) {

	/* Skip "unknown" items */
	if (auto_items[i].junk) {

	    /* Throw it at a monster or myself */
	    borg_keypress('f');
	    borg_keypress('a' + i);
	    borg_keypress('*');
	    borg_keypress('t');

	    /* Did something */
	    return (TRUE);
	}
    }


    /* Nothing to destroy */
    return (FALSE);
}

    

/*
 * Make sure we have at least one free inventory slot
 * Return TRUE if an action was performed.
 */
EXTERN bool borg_free_space(void)
{
    int i;
    
    u32b limit = 8L;


    /* Throw junk until done */
    while (auto_items[INVEN_PACK-1].iqty) {

	int k = 0;
	

	/* Increase the limit */
	limit *= 2;


	/* Try to throw away some junk */
	if (borg_throw_junk()) return (TRUE);


	/* Prevent infinite loops */
	for (i = 0; i < INVEN_PACK; i++) {
	    auto_item *item = &auto_items[i];
	    if (item->cash) k++;
	}
	
	/* Hack -- Crash if confused */
	if (!k) {
	    borg_oops("Too much stuff!");
	    return (TRUE);
	}


	/* Mark items as junk */
	for (i = 0; i < INVEN_PACK; i++) {

	    s32b cost;
	
	    auto_item *item = &auto_items[i];
	
	    /* Only examine "sell" items */
	    if (!item->cash) continue;

	    /* Extract the basic cost per item */
	    cost = item_worth(item) * item->iqty;

	    /* Save "expensive" items */
	    if (cost > limit) continue;

	    /* Debug */
	    borg_note(format("Junking %ld gold", cost));
	    
	    /* Hack -- Mark it as junk */
	    item->junk = TRUE;
	}
    }


    /* Success */
    return (FALSE);
}


/*
 * Count the number of items worth "selling"
 * This determines the choice of stairs.
 *
 * XXX XXX Base "cheap" on current cash, and use a similar
 * calculation in the "throw junk" routine.  Or take weight
 * into account.
 */
EXTERN int borg_count_sell(void)
{
    int i, k = 0;
    s32b greed, worth;

    /* Calculate "greed" factor */
    greed = auto_gold / 100L;
    
    /* Throw away "junk" */
    for (i = 0; i < INVEN_PACK; i++) {

	auto_item *item = &auto_items[i];
	
	/* Skip non-items */
	if (!item->iqty) continue;	

	/* Skip "junk" items */
	if (item->junk) continue;

	/* Skip known "cheap" items */
	if (item->kind && item->cash && item->able) {

	    /* Guess at the item value */
	    worth = item_worth(item) * item->iqty;

	    /* Do not bother with "cheap" items */
	    if (worth < greed) continue;
	}
	
	/* Skip "average" "cheap" items */
	if (item->kind && item->cash && suffix(item->desc, " {average}")) {

	    /* Guess at the item value */
	    worth = item_worth(item) * item->iqty;

	    /* Do not bother with "cheap" items */
	    if (worth < greed) continue;
	}
	
	/* Count remaining items */
	k++;
    }


    /* Count them */
    return (k);
}




/*
 * Identify items if possible
 */
EXTERN bool borg_test_stuff(void)
{
    int i, j;
        
#ifdef FRITSDEBUG
   borg_note("In routine borg_test_stuff()");
#endif /* FRITSDEBUG */

#ifdef FRITS
   /* don't try if blind/confused */
   if ((do_blind) || (do_confused)) return(FALSE);
#endif /* FRITS */

    /* Look for an item to identify */
    for (i = 0; i < INVEN_PACK; i++) {

	/* Skip tested items */
	if (!auto_items[i].test) continue;

#ifdef FRITS
        if((sclass==SCLASS_MAGE)&&(rand_int(3)==0))
        {
        if (borg_can_cast(SCLASS_MAGE,'c','e')){ 
           borg_cast_spell_oninv(SCLASS_MAGE,'c','e','a'+i);
	   return (TRUE);
        }
        } 
#endif /* FRITS */
	/* Look for a scroll of identify */
	for (j = 0; j < INVEN_PACK; j++) {

	    /* Is this a scroll of identify */
	    if (auto_items[j].kind == OBJ_SCROLL_IDENTIFY) {

		/* Read the scroll */
		borg_keypress('r');
		borg_keypress('a' + j);

		/* Identify the item */
		borg_keypress(' ');
		borg_keypress('a' + i);

#ifdef FRITS
/* shouldn't this item have the able flag? */
                auto_items[i].able=TRUE;
#endif /* FRITS */
		
		/* Success */
		return (TRUE);
	    }
	}
    }


    /* Nothing to do */
    return (FALSE);
}    

    


/*
 * Maintain a "useful" inventory
 */
EXTERN bool borg_wear_stuff(void)
{
    int i;
        
#ifdef FRITSDEBUG
   borg_note("In routine borg_wear_stuff()");
#endif /* FRITSDEBUG */

    /* Look at the "lite" slot */    
    i = INVEN_LITE;

    /* Check the current "light source" */
    if (!auto_items[i].iqty) {
	do_lite = TRUE;
    }
    else if (auto_items[i].sval == SV_LITE_TORCH) {
	if (auto_items[i].pval < 1000) do_torch = TRUE;
	if (auto_items[i].pval < 1) do_lite = TRUE;
    }
    else if (auto_items[i].sval == SV_LITE_LANTERN) {
	if (auto_items[i].pval < 5000) do_flask = TRUE;
	if (auto_items[i].pval < 1) do_lite = TRUE;
    }


    /* Refuel a torch */
    if (do_torch) {

	/* Try to wield some other lite */
	for (i = 0; i < INVEN_PACK; i++) {
	    if ((auto_items[i].tval == TV_LITE) &&
		(auto_items[i].sval == SV_LITE_LANTERN)) {
		borg_keypress('w');
		borg_keypress('a' + i);
		return (TRUE);
	    }
	}

	/* Scan the inventory -- note "empties" are gone */
	for (i = 0; i < INVEN_PACK; i++) {
	    if ((auto_items[i].tval == TV_LITE) &&
		(auto_items[i].sval == SV_LITE_TORCH)) {
		borg_keypress('F');
		return (TRUE);
	    }
	}
    }


    /* Refuel a lantern */
    if (do_flask) {

	/* Scan the inventory */
	for (i = 0; i < INVEN_PACK; i++) {
	    if (auto_items[i].tval == TV_FLASK) {
		borg_keypress('F');
		return (TRUE);
	    }
	}
    }


    /* Get a new lite */
    if (do_lite) {

	/* Scan the inventory looking for (non-empty) lanterns */
	for (i = 0; i < INVEN_PACK; i++) {
	    if ((auto_items[i].tval == TV_LITE) &&
		(auto_items[i].sval == SV_LITE_LANTERN)) {
		borg_keypress('w');
		borg_keypress('a' + i);
		return (TRUE);
	    }
	}

	/* Scan the inventory looking for (non-empty) torches */
	for (i = 0; i < INVEN_PACK; i++) {
	    if ((auto_items[i].tval == TV_LITE) &&
		(auto_items[i].sval == SV_LITE_TORCH)) {
		borg_keypress('w');
		borg_keypress('a' + i);
		return (TRUE);
	    }
	}
    }


    /* Wear stuff (top down) */
    for (i = 0; i < INVEN_PACK; i++) {

	/* Skip "unknown" items */
	if (auto_items[i].wear) {

	    /* Wear it */
	    borg_keypress('w');
	    borg_keypress('a' + i);

	    /* Did something */
	    return (TRUE);
	}
    }


    /* Nothing to do */
    return (FALSE);
}




/*
 * Analyze the "inventory" screen
 */
EXTERN bool borg_prepare_inven(void)
{
    int i;
    
    char buf[256];

    bool done = FALSE;
    
    
    /* Extract the inventory */
    for (i = 0; i < INVEN_PACK; i++) {

	/* Default to "nothing" */
	buf[0] = '\0';

	/* Cheat -- extract the inventory directly */
	if (cheat_inven) {

	    /* Extract a real item */
	    if (inventory[i].tval) {
		objdes(buf, &inventory[i], TRUE);
		buf[75] = '\0';
	    }
	}

	/* Actually parse the screen */
	else if (!done) {

	    /* XXX XXX XXX Not implemented -- see store parser */
	    /* Note -- Be sure to strip any trailing spaces */
	}

	/* Ignore "unchanged" items */
	if (streq(buf, auto_items[i].desc)) continue;
	
	/* Analyze the item (no price) */
	borg_item_analyze(&auto_items[i], buf, "");
    }

    
    /* Show the "equipment screen" */
    if (!cheat_equip) borg_keypress('e');

    /* Prepare to parse the equipment */
    state = STATE_EQUIP;

    /* Done */
    return (TRUE);
}


/*
 * Analyze the "equipment" screen
 */
EXTERN bool borg_prepare_equip(void)
{
    int i;
    
    char buf[160];

    /* Extract the inventory */
    for (i = INVEN_WIELD; i < INVEN_TOTAL; i++) {

	/* Default to "nothing" */
	buf[0] = '\0';

	/* Cheat -- extract the inventory directly */
	if (cheat_equip) {

	    /* Extract a real item */
	    if (inventory[i].tval) {
		objdes(buf, &inventory[i], TRUE);
		buf[75] = '\0';
	    }
	}

	/* Actually parse the screen */
	else {

	    /* XXX Not implemented */
	    /* XXX Be sure to strip trailing spaces */

	    /* Use the nice "show_empty_slots" flag */
	    if (streq(buf, "(nothing)")) strcpy(buf, "");
	}

	/* Ignore "unchanged" items */
	if (streq(buf, auto_items[i].desc)) continue;
	
	/* Analyze the item (no price) */
	borg_item_analyze(&auto_items[i], buf, "");
    }
    
    
    /* Leave the "equipment screen" */
    if (!cheat_inven || !cheat_equip) borg_keypress(ESCAPE);

    /* Enter the "think" state */
    state = STATE_THINK;

    /* Hack -- go to the "store" state instead */
    if (state_store) state = STATE_STORE;

    /* Success */
    return (TRUE);
}



#endif /* AUTO_PLAY */
