/*
** Roland mpu-401 interface driver
** @(#)mpu.c 1.16 89/01/09
** $Header: mpu.c,v 1.12 89/02/21 19:26:45 psl Exp $
** RCS source in spud:/u/psl/SYS/sunmidi
** Original by Rusty Wright, UCSD.
** Hacked at various times by Peter Langston, Mike Hawley, Daniel Steinberg.
*/

#ifdef	LINT
#define	NMPU	1
#define	KERNEL
#else
#include "mpu.h"
#endif

#if NMPU > 0

/*
** This driver currently attempts to support:
**	MULTIBUS	Andy Voelkel's MultiBus card (single MPU)
**	VMEBUS1		Dave Cumming's VMEBus card (four MPUs)
**	VMEBUS2		Don Jackson's VMEBus card (four MPUs)
**	ATBUS		Roland's MPU-IPC or equivalent (single MPU)
** Define one of these.
*/
#define	ATBUS	1	/* for Sun386 */

/*
 * The following is a cheap trick to try to determine which Release
 * of SunOS this is being compiled for.  It looks at /usr/include/signal.h
 * to figure out which features are present.  It will not work if you've
 * got a pre-4.0 /usr/include/signal.h and you're compiling a post-4.0 driver.
 * If that's the case, #define the damn release yourself.
 * NOTE:
 *  This version of the driver has not been tested against releases
 *  prior to 3.2, though there's special code to make it work with 3.0.
 *  You buys your ticket and you takes your chances.
 */
#include <sys/signal.h>

#ifdef SIGABRT		/* post-4.0 source reorg */
#define SunOS4_x
#define SunOS3_x	/* use PCATCH in sleep() */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/ioctl.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/proc.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/map.h>
#include <sys/uio.h>

#include <machine/psl.h>
#include <machine/mmu.h>
#include <machine/pte.h>
#include <sundev/mbvar.h>

#include <sunnmw/mpuvar.h>
#include <sunnmw/mpureg.h>

#else /*pre-4.0*/

#include "../h/param.h"
#ifdef PCATCH		/* 3.2 feature */
#define SunOS3_x
#else
#define SunOS3_0	/* pre-3.2 */
#endif

#include "../h/systm.h"
#include "../h/ioctl.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/proc.h"
#include "../h/buf.h"
#include "../h/file.h"
#include "../h/map.h"
#include "../h/uio.h"

#include "../machine/psl.h"
#include "../machine/mmu.h"
#include "../machine/pte.h"
#include "../sundev/mbvar.h"

#include "../sunnmw/mpuvar.h"
#include "../sunnmw/mpureg.h"

#endif pre-4.0


#ifdef SunOS3_x
#define mpusleep(a,b)	sleep(a,(b)|PCATCH)
#endif SunOS3_x


#define MPU_DELAYTIME	100	/* delay count for probing */
#define MPU_DELAYLOOPS	1000	/* retry count for probing (can be slow) */

/* bracket these debugging printfs with {} when used after an 'if' statement */
#define d1printf  if (MPUdebug & 1) printf	/* unusual errors */
#define d2printf  if (MPUdebug & 2) printf	/* special cases */
#define d3printf  if (MPUdebug & 4) printf	/* track/cond output trace */
#define d4printf  if (MPUdebug & 8) printf	/* input interrupt trace */
#define d5printf  if (MPUdebug & 16) printf	/* command output trace */
#define d6printf  if (MPUdebug & 32) printf	/* subr trace */
/****
int	MPUdebug	= 0;	/* mask of above bits */
/****/int	MPUdebug	= 1;	/* mask of above bits */
#if VMEBUS1|VMEBUS2
int	MPUreset	= 1;	/* send a hard reset on mpureset() */
#endif VMEBUS1|VMEBUS2

#define MPUS		(NMPU * MPUS_PER_BOARD)	/* max number of MPUs */
#define MPU_UNIT(dev)	(minor(dev))
#define MPU_CTLR(unit)	((unit) / MPUS_PER_BOARD)
#define MPU_UOFF(unit)	((unit) % MPUS_PER_BOARD)

#define MPU_DEV(u)	(mpudinfo[MPU_CTLR(u)])
#define MPU_CTOR(c,u)	(&(((struct mpu_ctlr *)(c))->mpur[MPU_UOFF(u)]))
#define MPU_CTLRADDR(u)	((struct mpu_ctlr *)(MPU_DEV(u)->md_addr))
#define MPU_REGADDR(u)	MPU_CTOR(MPU_CTLRADDR(u), (u))

#define MPU_PRI_LOW	(PZERO+1)	/* interruptable	*/

#define	PEEK		0		/* for mpugetbyte() */
#define	NO_MO_DATA	UCH(0xFD)	/* from mpugetbyte() */

#ifdef ATBUS
#define	PEEKC(ADDR)		inb((short)(ADDR))
#define	POKEC(ADDR,DATA)	outb((short)(ADDR),UCH(DATA))
#else MULTIBUS|VMEBUS1|VMEBUS2
#define	PEEKC(ADDR)		peekc((char*)(ADDR))
#define	POKEC(ADDR,DATA)	pokec((char*)(ADDR),UCH(DATA))
#endif MULTIBUS|VMEBUS1|VMEBUS2

#define	DRR_SET(R)		((PEEKC(&(R)) & MPU_STAT_DRR) == 0)
#define	DSR_SET(R)		((PEEKC(&(R)) & MPU_STAT_DSR) == 0)

unsigned char	mpugetbyte();
struct mpu_l	*mpugetl();

int		mpuprobe();
#if MULTIBUS|ATBUS
int		mpupoll();
#elif VMEBUS1|VMEBUS2
#define mpupoll	0
#endif

struct mpu_device	mpu_device[MPUS];
struct mb_device	*mpudinfo[NMPU];

extern	int	nodev();
#define	NOFCN	nodev
struct mb_driver	mpudriver = {
	mpuprobe, NOFCN, NOFCN, NOFCN, NOFCN, mpupoll,
	MPU_SIZE, "mpu", mpudinfo,
};

/*
** Mpuprobe() determines whether the interface card really exists.
** If so, it calls mpualive() for each possible mpu port to see
** if there is really an mpu plugged in.  If the controller does not
** respond for an individual mpu's csr&data addresses, then that mpu
** is marked unavailable (MPUD_NOCSR).  Otherwise, individual mpus
** may be connected and disconnected anytime, and mpuopen() and mpuclose()
** will attempt to reset and verify the existence of the mpu in question.
*/
mpuprobe(reg, ctlr)
caddr_t	reg;
int	ctlr;
{
	register struct mpu_ctlr *mc;
	register struct mpu_device *mp;
	register int i, cnum;

	mc = (struct mpu_ctlr *)reg;
	/* First, see if the controller's there at all */
	if (PEEKC(&mc->mpu_alive) == -1) {
	    d1printf("mpu%d: no mpu ctlr @ 0x%x\n", ctlr, &mc->mpu_alive);
	    return (0);
	}
	/* Check each MPU individually */
	cnum = ctlr * MPUS_PER_BOARD;
	mp = &mpu_device[cnum];
	for (i = 0; i < MPUS_PER_BOARD; mp++, i++) {
	    switch (mpualive(mc, i)) {
	    case -1:
		mp->mpud_state = MPUD_NOCSR;
		printf("/dev/mpu%x: no csr\n", (cnum + i));
		break;
	    case 0:
		mp->mpud_state = MPUD_NOMPU;
		d1printf("probe /dev/mpu%x: not responding\n", (cnum + i));
		break;
	    default:
		mp->mpud_state = MPUD_CLOSE;
		printf("/dev/mpu%x: alive\n", (cnum + i));
	    }
	}
	return (MPU_SIZE);	/* not necessarily == sizeof (struct mpu_reg) */
}

/*
** mpualive() sends a hard & soft reset to the mpu (VMEBUS only) and checks to
** see if an acknowledge comes back.  If not, the mpu is probably not connected
** (or the cable is bad).
** Return -1 if any controller registers are not present,
**	0 if device is there but doesn't respond (DRR, DSR, or ACK), and
**	1 for success.
*/
mpualive(mc, uoff)
struct	mpu_ctlr *mc;
int	uoff;
{
	register struct mpu_reg *mpur;
	register int i;

	mpur = &mc->mpur[uoff];
#if VMEBUS1|VMEBUS2
	/* send a hard reset to the mpu */
	if (MPUreset) {
#ifdef VMEBUS1
	    if (POKEC(&mc->mpu_reset, ~(1 << uoff))
	     || POKEC(&mc->mpu_reset, 0xFF)) { 
		printf("no MPU reset reg @0x%x??...", mc->mpu_reset);
		return (-1);
	    }
#elif VMEBUS2==1
	    if (POKEC(&mc->mpu_reset, ~(1 << uoff))) {
		printf("no MPU reset reg @0x%x??...", mc->mpu_reset);
		return (-1);
	    }
	    DELAY(MPU_DELAYTIME);
	    mc->mpu_reset = ~0;
#endif VMEBUS2
	}
#endif VMEBUS
	if (PEEKC(&mpur->mpur_csr) == -1) {
	    printf("no csr for mpu%d @0x%x?\n", uoff, &mpur->mpur_csr);
	    return (-1);
	}
	if (mpudelay(mpur)) {				/* wait for DRR */
	    d1printf("no DRR in mpu%d csr @0x%x\n",uoff,&mpur->mpur_csr);
	    return (0);
	}
#ifdef VMEBUS1
	/* send 0's to mpur_csr first, then a reset */
	if (POKEC(&mpur->mpur_csr, 0x00)) {
	    printf("no mpu%d csr @0x%x\n", uoff, &mpur->mpur_csr);
	    return (-1);
	}
#endif VMEBUS1
#ifdef ATBUS
	POKEC(&mpur->mpur_csr, MPU_COM_RESET);		/* reset mpu */
#else MULTIBUS|VMEBUS
	if (POKEC(&mpur->mpur_csr, MPU_COM_RESET)) {	/* reset mpu */
	    printf("no mpu%d csr @0x%x\n", uoff, &mpur->mpur_csr);
	    return (-1);
	}
#endif MULTIBUS|VMEBUS
	if (mpudsr(mpur)) {				/* wait for DSR */
	    d1printf("no DSR in mpu%d csr @0x%x\n",uoff,&mpur->mpur_csr);
	    return (0);
	}
	if ((i = PEEKC(&mpur->mpur_data)) == -1) {	/* read data */
	    printf("no mpu%d data reg @0x%x\n", uoff, &mpur->mpur_data);
	    return (-1);
	}
	if (UCH(i) != MPU_MESS_ACK) {
	    d1printf("mpu%d reset got 0x%x instead of ack (0x%x)\n",
	     uoff, UCH(i), MPU_MESS_ACK);
	    return (0);
	}
	return (1);
}

/*ARGSUSED*/
mpuopen(dev, flag)
dev_t	dev;
int	flag;
{
	register struct mpu_device *mp;
	register int unit, i;
	register struct buf *bp;
	register struct mb_device *md;

	d6printf("mpuopen(0x%x, %d)\n", dev, flag);
	/* check that unit is valid and interface card is there */
	if (((unit = MPU_UNIT(dev)) >= MPUS) || ((md = MPU_DEV(unit)) == 0))
	    return (ENXIO);
	mp = &mpu_device[unit];
	mp->mpud_spl = pritospl(md->md_intpri);	/* save spl priority */
	if (mp->mpud_state & MPUD_OPEN) {	/* already open */
	    if ((mp->mpud_state & MPUD_EXCL)	/* for exclusive use */
	     && u.u_uid)			/* (and not root) */
		    return (EBUSY);		/* sorry */
	    else if (((flag & FREAD) && !mp->mpud_ib.mr_buf))
		return (EBUSY);			/* can't add READ access */
	    mp->mpud_ocnt++;
	    return (0);	/* okay to reopen, without adding READ access */
	}
	/* check that MPU is hooked up...if not, probe it again */
	if (mp->mpud_state == MPUD_NOMPU) {
	    d1printf("probing /dev/mpu%x...", unit);
	    if (mpureset(unit)) {
		mp->mpud_state = MPUD_CLOSE;
		d1printf("ok\n");
	    } else
		return (ENODEV);
	} else if ((mp->mpud_state & MPUD_NOCSR) || !mpureset(unit))
	    return (ENODEV);
	else
	    mp->mpud_state = MPUD_OPEN;	/* mark it open */
	if (!(flag & FREAD))			/* no ring-buffer if not read */
	    mp->mpud_ib.mr_buf = (caddr_t) 0;
	else if ((mp->mpud_ib.mr_buf = kmem_alloc(MPUBUFSIZ)) == NULL) {
	    mp->mpud_state = MPUD_CLOSE;	/* not open after all */
	    return (ENOMEM);			/* no memory for buffer */
	}
	mp->mpud_ib.mr_ip = 0;			/* reset ring buffer offsets */
	mp->mpud_ib.mr_op = 0;
	mp->mpud_track = MPU_CTR_COM;		/* command track by default */
	mp->mpud_cnt = 0;			/* nothing sent yet */
	mp->mpud_blimit = MPUD_DEFBLIMIT;/* default per track buffer limit */
	for (i = MPU_TTR_MAX; --i >= 0; ) {	/* init track buffer lists */
	    bp = &mp->mpud_trackl[i].mpul_head;
	    bp->av_forw = bp;
	    bp->av_back = bp;
	    mp->mpud_trackl[i].mpul_bused = 0;
	}
	mp->mpud_ocnt = 1;
	return (0);
}

/*ARGSUSED*/
mpuclose(dev, flag)
dev_t	dev;
int	flag;
{
	register struct mpu_device *mp;
	register struct mpu_reg *mpur;
	register int unit, i;
	int err, opri;

	d6printf("mpuclose(0x%x, %d)\n", dev, flag);
	unit = MPU_UNIT(dev);
	mp = &mpu_device[unit];
	if (mp->mpud_state == MPUD_CLOSE) {
	    uprintf("mpuclose: /dev/mpu%d not open\n", unit);
	    return (EINVAL);
	}
	if (mp->mpud_state & MPUD_CLOSING) {
	    if (--mp->mpud_ocnt <= 0)	/* multiple opens are allowed, so... */
		uprintf("mpuclose: /dev/mpu%d phantom close\n", unit);
	    return (0);
	}
	mp->mpud_state |= MPUD_CLOSING;
	mpur = MPU_REGADDR(unit);
	opri = splx(mp->mpud_spl);
	/*
	** If an mpu dies or is disconnected, give up.
	** Otherwise, wait for the track buffers to clear.
	*/
	for (i = MPU_TTR_MAX; --i >= 0; ) {
	    while (mp->mpud_trackl[i].mpul_tcount > 0
	     && !(mp->mpud_state & MPUD_NOMPU)) {
		d2printf("mpuclose: waiting for track %d\n", i);
		if (mpusleep((caddr_t)&mp->mpud_track, MPU_PRI_LOW))
		    mpupurge(mp, unit);			/* ^C gets us here */
	    }
	}
	(void) splx(opri);
	POKEC(&mpur->mpur_csr, MPU_COM_CLRPLAYCNTRS);	/* hanging notes? */
	DELAY(400*MPU_DELAYTIME);		/* seems to need a long delay */
	err = mp->mpud_state & MPUD_ERROR;
	mp->mpud_state = MPUD_CLOSE;
	if (--mp->mpud_ocnt != 0)
	    uprintf("/dev/mpu%d closed with ocnt=%d\n", unit, mp->mpud_ocnt);
	if (mp->mpud_ib.mr_buf) {   /* don't do this before state=MPUD_CLOSE */
	    kmem_free(mp->mpud_ib.mr_buf, MPUBUFSIZ);
	    mp->mpud_ib.mr_buf = NULL;
	}
	if (!mpureset(unit)) {			/* check the mpu */
	    uprintf("mpuclose: /dev/mpu%d not responding\n", unit);
	    mp->mpud_state = MPUD_NOMPU;
	}
	return (err ? EINVAL : 0);
}

/*
** Record data is saved in a per-mpu ring buffer.
** Note that the uimove() can happen at spl(0) because mpuputc() doesn't
** wrap around.  The buffer offsets are only updated at device priority.
**
** Also, looping until uio_resid == 0 defeats any user buffering,
** so return as soon as all available characters ( >0 ) have been read.
** This is more like terminal devices and FIFOs.
**
** If the MPU is open for WRITE only and root tries to open it for READ,
** root's open will fail; so we should never have mpuread() called without
** an input buffer...  but we check anyway.
*/
mpuread(dev, uio)
dev_t	dev;
register struct uio *uio;
{
	register struct mpu_device *mp;
	register unsigned ip, op;
	int err, opri, mincnt;
#define mr	(&mp->mpud_ib)		/* mr -> ring buffer descriptor */

	d6printf("mpuread(0x%x, 0x%x)\n", dev, uio);
	mp = &mpu_device[MPU_UNIT(dev)];
	if (MPU_UNIT(dev) >= MPUS
	 || !(mp->mpud_state & MPUD_OPEN)
	 || mr->mr_buf == NULL)		/* not really open for reading */
	    return(EBADF);				/* "can't happen" */
	if (mp->mpud_state & MPUD_NOMPU)
	    return (ETIMEDOUT);	/* mpu went offline */
	/*
	** While the user needs more data and there haven't been any errors
	** get the data and send it to user.  Quit as soon as some data
	** has been transferred and no more is available.
	*/
	opri = splx(mp->mpud_spl);
	for (mincnt = err = 0; (uio->uio_resid > 0) && !err; ) {
	    /* Wait for ring buffer data unless we've already got some.  */
	    while ((ip = mr->mr_ip) == (op = mr->mr_op)) {
		if (mincnt > 0)
			goto rddone;
		(void) sleep((caddr_t)mr, MPU_PRI_LOW);
	    }
	    (void) splx(opri);
	    /*
	    ** if ip > op, transfer all the current data.
	    ** if op > ip, can only transfer to end of buffer
	    ** (pick up the rest the next time thru this loop).
	    */
	    mincnt = (ip > op) ? (ip - op) : (MPUBUFSIZ - op);
	    mincnt = MIN(mincnt, uio->uio_resid);
	    err = uiomove((mr->mr_buf + op), mincnt, UIO_READ, uio);
	    opri = splx(mp->mpud_spl);
	    mr->mr_op += mincnt;		/* drop data even if err (?) */
	    if (mr->mr_op == MPUBUFSIZ)
		mr->mr_op = 0;			/* wrap around */
	}
rddone:
	(void) splx(opri);
	return (err);
#undef mr
}

mpuwrite(dev, uio)
dev_t	dev;
register struct uio *uio;
{
	register struct mpu_device *mp;
	register struct mpu_l *mpul;
	register struct buf *bp, *hp;
	int err, cnt, opri, bsiz, unit, bufs;
	caddr_t addr;

	d6printf("mpuwrite(0x%x, 0x%x)\n", dev, uio);
	unit = MPU_UNIT(dev);
	mp = &mpu_device[unit];
	if (unit >= MPUS
	 || (mp->mpud_state & MPUD_OPEN) != MPUD_OPEN)
	    return (EBADF);
	if (mp->mpud_state & MPUD_NOMPU)
	    return (ETIMEDOUT);	/* mpu went offline */
	if ((cnt = uio->uio_iov->iov_len) <= 0)
	    return (0);		/* happens on 'play /dev/null' */
	mpul = mpugetl(mp);
	hp = &mpul->mpul_head;
	/*
	** Play data is handled as a linked list of buffers.  If this request
	** won't fit in the current buffer and we're below the buffer limit,
	** we get a new buffer and make it current; we copy the data and then
	** link the current buffer if it's new.
	*/
	opri = splx(mp->mpud_spl);
	if ((bp = hp->av_back) == hp || bp->b_bufsize - bp->b_bcount < cnt) {
	    bufs = (cnt + MAXBSIZE - 1) / MAXBSIZE;
	    bsiz = MAXBSIZE * bufs;			/* mult of MAXBSIZE */
	    if (bufs > mp->mpud_blimit) {	/* this request > limit */
		(void) splx(opri);
		return(EINVAL);			/* tell 'em where to go */
	    }
	    while (mpul->mpul_bused + bufs > mp->mpud_blimit) {
		d2printf("mpuwrite: waiting for track %d buffer\n",
		 mp->mpud_track);
		if (mpusleep((caddr_t)&mpul->mpul_bused, MPU_PRI_LOW)) {
		    (void) splx(opri);			/* ^C gets us here */
		    return(EINTR);
		}
	    }
	    mpul->mpul_bused  += bufs;
	    bp = geteblk(bsiz);
	    d2printf("mpuwrite: %d byte(s) -> new track %d buffer\n",
	     cnt, mp->mpud_track);
	    addr = bp->b_un.b_addr;
	    if (!(err = uiomove(bp->b_un.b_addr, cnt, UIO_WRITE, uio))) {
		bp->b_bcount = cnt;
		mpul->mpul_tcount += cnt;
		if (hp->av_forw == hp) {	/* the only (& first) buffer */
		    mpul->mpul_cp = (unsigned char *) addr;
		    mpul->mpul_bcount = bp->b_bcount;
		}
		hp->av_back->av_forw = bp;
		bp->av_back = hp->av_back;
		hp->av_back = bp;
		bp->av_forw = hp;
	    }
	} else {
	    d2printf("mpuwrite: %d byte(s) -> old track %d buffer\n",
	     cnt, mp->mpud_track);
	    addr = bp->b_un.b_addr + bp->b_bcount;
	    if (!(err = uiomove(addr, cnt, UIO_WRITE, uio))) {
		bp->b_bcount += cnt;
		mpul->mpul_tcount += cnt;
		if (hp->av_forw == bp)		/* adding to current buffer */
		    mpul->mpul_bcount += cnt;
	    }
	}
	/* Start sending mpu commands, if not in-progress.  */
	if ((mp->mpud_track == MPU_CTR_COM)
	 && !(mp->mpud_state & MPUD_CMDTRACK)) {
	    mp->mpud_state |= MPUD_CMDTRACK;
	    mpusendcom(mp, unit);
	}
	(void) splx(opri);
	return (err);
}

/*ARGSUSED*/
mpuioctl(dev, cmd, data, flag)
dev_t	dev;
int	cmd, flag;
caddr_t	data;
{
        register struct mpu_device *mp;
	register int track, i;
	register struct mpu_reg *mpur;

	d6printf("mpuioctl(0x%x, 0x%x, 0x%x, %d)\n", dev, cmd, data, flag);
	mp = &mpu_device[MPU_UNIT(dev)];
	if (cmd == MPU_IOC_TRACK) {		/* set current track */
	    if ((track = *(int *)data) < 0 || MPU_TTR_MAX <= track)
		return (EINVAL);
	    mp->mpud_track = track;
	} else if (cmd == MPU_IOC_PURGE) {	/* purge all track buffers */
	    mpupurge(mp, MPU_UNIT(dev));
	} else if (cmd == MPU_IOC_RESID) {	/* get resid count of tracks */
	    for (i = MPU_TTR_MAX; --i >= 0; )
		((int *)data)[i] = mp->mpud_trackl[i].mpul_tcount;
	} else if (cmd == MPU_IOC_LIMIT) {	/* set track buffer limit */
	    if ((i = *(int *)data) <= 0 || MPUD_MAXBLIMIT < i)
		return (EINVAL);
	    mp->mpud_blimit = i;
	} else if (cmd == FIONREAD) {		/* return # bytes unread */
	    if (mp->mpud_ib.mr_buf) {	/* yes, open for reading */
		i = splx(mp->mpud_spl);
		*(int *)data = Mpuinq(&mp->mpud_ib);
		(void) splx(i);
	    } else
		*(int *)data = 0;
	} else if (cmd == MPU_IOC_EXCL) {	/* set exclusive use */
	    if (mp->mpud_ocnt > 1)
		return (EBUSY);
	    mp->mpud_state |= MPUD_EXCL;
	} else if (cmd == MPU_IOC_NXCL) {	/* clear exclusive use */
	    mp->mpud_state &= ~MPUD_EXCL;
/**** DEBUGGING STUFF ****/
	} else if (cmd == MPU_IOC_PEEK) {	/* peek data|csr reg */
	    mpur = MPU_REGADDR(MPU_UNIT(dev));
	    i = *(int *)data;
	    *(int *)data = PEEKC(&mpur->mpur_data + i);
	} else if (cmd == MPU_IOC_POKE) {	/* poke data|csr reg */
	    mpur = MPU_REGADDR(MPU_UNIT(dev));
	    i = *(int *)data;
	    *(int *)data = POKEC(&mpur->mpur_data + (i >> 8), i & 0xFF);
/**** DEBUGGING STUFF ****/
	} else
	    return (EINVAL);
	return (0);
}

mpuselect(dev, flag)
dev_t	dev;
int	flag;
{
        register struct mpu_device *mp;
	int opri;

	d6printf("mpuselect(0x%x, %d)\n", dev, flag);
	mp = &mpu_device[MPU_UNIT(dev)];
	if (mp->mpud_state & MPUD_NOMPU) {
	    u.u_error = ETIMEDOUT;		/* mpu went offline */
	    return (0);
	}
	if (flag == FREAD) {
	    if (!mp->mpud_ib.mr_buf) {
		u.u_error = EBADF;		/* not open for read */
		return(0);
	    }
	    opri = splx(mp->mpud_spl);
	    if (Mpuinq(&mp->mpud_ib) > 0) {
		(void) splx(opri);
		return(1);
	    }
	    if ((mp->mpud_sel != 0)
	     && (mp->mpud_sel->p_wchan == (caddr_t)&selwait))
		mp->mpud_state |= MPUD_SELWAIT;
	    else
		mp->mpud_sel = u.u_procp;
	    (void) splx(opri);
	    return(0);
	} else if (flag == FWRITE) {
	    return(1);
	} else if (mp->mpud_state & MPUD_ERROR)
	    return(1);
	else
	    return(0);
}

#if MULTIBUS|ATBUS
/*
** mpupoll() is the interrupt polling routine for interfaces without
** an interrupt bit register to identify interrupting units.
*/
mpupoll()
{
	register struct mb_device *md;
	register struct mpu_reg *mpur;
	register int unit, me = 0;

	d4printf("mpupoll()...\n");
	for (unit = 0; unit < MPUS; unit++) {
	    if (((md = MPU_DEV(unit)) == 0) || (md->md_alive == 0))
		continue;
	    for (mpur = MPU_REGADDR(unit); DSR_SET(mpur->mpur_csr); me = 1)
		mpuiguts(unit, mpur);			/* do the real work */
	}
	return (me);
}

#elif VMEBUS1|VMEBUS2

/*
** mpuintr() is the interrupt polling routine for interfaces with
** an interrupt bit register to identify interrupting units.
*/
mpuintr(c)
register int c;
{
	register struct mpu_reg *mpur;
	register int unit;

	register struct mpu_ctlr *mc;
	register unsigned imask;

	d4printf("mpuintr(%d)...\n", c);
	mc = (struct mpu_ctlr *)mpudinfo[c]->md_addr;
	/*
	** Bits in the interrupt mask are 0 if interrupt pending.
	** We complement the mask so that bits are 1 if interrupt pending.
	*/
	while (imask = ((unsigned)~mc->mpuc_intmask) & MPU_INTMASK) {
	    unit = c * MPUS_PER_BOARD;
	    for (mpur = mc->mpur; imask; imask >>= 1, unit++, mpur++)
		if ((imask & 1))		/* interrupt on this unit? */
		    mpuiguts(unit, mpur);
	}
}
#endif VMEBUS

mpuiguts(unit, mpur)	/* unit and mpu_reg pointer for an interrupting mpu */
register int unit;
register struct mpu_reg *mpur;
{
	register struct mpu_device *mp;
	unsigned char data;

	mp = &mpu_device[unit];
	data = PEEKC(&mpur->mpur_data);
	d4printf("mpuiguts(mpu%x) data=0x%x ", unit, data);
	if (mp->mpud_state & (MPUD_NOMPU | MPUD_NOCSR)) {
	    printf("mpu%x: intermittent mpu...data:0x%x\n", unit, data);
	    if (!(mp->mpud_state & MPUD_OPEN) && mpureset(unit)) {
		mp->mpud_state = MPUD_CLOSE;
		d1printf("mpu%d: now ok\n", unit);
	    }
	    return;
	}
	if (!(mp->mpud_state & MPUD_OPEN)) {
	    printf("mpu%x: spurious interrupt...data:0x%x\n", unit, data);
	    mpureset(unit);
	    return;
	}
	if (mp->mpud_state & MPUD_TT) {		/* prev byte was time tag */
	    if (data < 0x80) {	/* running status */
		if (mp->mpud_contin)
		    mp->mpud_cnt = mp->mpud_contin - 1;
		else {			/* ERROR - running stat was cleared */
		    uprintf("mpu: no mode & no running stat: %x\n", data);
		    mp->mpud_cnt = 0;
		}
	    } else if (data < UCH(0xF0)) {	/* explicit, set runstat */
		mp->mpud_cnt = mpumidicnt(data);
		mp->mpud_contin = mp->mpud_cnt;
	    } else if (data < UCH(0xF8)) {	/* explicit (SYS) clr runstat */
		mp->mpud_cnt = mpumidicnt(data);
		mp->mpud_contin = 0;
	    } else				/* explicit (RT), no effect */
		mp->mpud_cnt = 1;
	    mp->mpud_state &= ~MPUD_TT;
	} else if (mp->mpud_state & MPUD_SMESS) {  /* prev byte was Sys Mess */
	    mp->mpud_cnt = mpumidicnt(data);	   /* find how many follow */
	    mp->mpud_state &= ~MPUD_SMESS;
	}
	d4printf("mpud_cnt=%d ", mp->mpud_cnt);
	/*
	** By now mpud_cnt has been set up.
	**  >0 ==> normal MIDI data
	**  -1 ==> sys excl, pass bytes until 0xF7 (or other non-RT >= 80) found
	**   0 ==> between commands, use mpuisubr to decode new one
	*/
	if (mp->mpud_cnt > 0) {		/* normal midi data */
	    if (mp->mpud_ib.mr_buf)
		(void) mpuputc(data, &mp->mpud_ib);
	    --mp->mpud_cnt;
	} else if (mp->mpud_cnt == -1) {	/* system exclusive midi data */
	    if (mp->mpud_ib.mr_buf)
		(void) mpuputc(data, &mp->mpud_ib);
	    if (MIDI_EOX(data))
		mp->mpud_cnt = 0;
	} else if (mpuisubr(mp, data, unit)) {	/* starting a new command */
	    if (mp->mpud_ib.mr_buf)
		(void) mpuputc(data, &mp->mpud_ib);
	}
	wakeup((caddr_t)&mp->mpud_ib);
	if (mp->mpud_sel) {
	    selwakeup(mp->mpud_sel, mp->mpud_state & MPUD_SELWAIT);
	    mp->mpud_state &= ~MPUD_SELWAIT;
	    mp->mpud_sel = 0;
	}
}

/*
** mpuputc(), called by mpuiguts() to put a byte on the input ringbuffer
** Should NEVER be called if (mp->mpud_ib.mr_buf == NULL).
*/
mpuputc(data, mr)
unsigned char data;
struct mpu_rbuf *mr;
{

	d4printf("mpuputc: mpu input data 0x%x\n", data);
	if (((mr->mr_op - mr->mr_ip) == 1)
	 || ((mr->mr_op == 0) && (mr->mr_ip == (MPUBUFSIZ - 1)))) {
	    uprintf("mpu: mpuputc(0x%x) buffer overflow\n", data);
	    return (-1);			/* ring buffer overflow */
	}
	((unsigned char *)(mr->mr_buf))[mr->mr_ip++] = data;
	if (mr->mr_ip == MPUBUFSIZ)
	    mr->mr_ip = 0;			/* wrap around */
	return (0);
}

/*
** mpuisubr(), called by mpuiguts() to decode a MIDI command byte.
*/
mpuisubr(mp, data, unit)
register struct mpu_device *mp;
unsigned char data;
int	unit;
{
	int touser = 0;

	d4printf("mpuisubr(0x%x, 0x%x, %d) ", mp, data, unit);
	/*
	** If the upper nibble is 0xF, it's an mpu message or a data request.
	** Otherwise it's a time tag.  Some things get sent back to the user,
	** some don't; we use 'touser' to keep track of what gets sent back.
	*/
	if ((data & MIDI_CMD_MASK) == UCH(0xF0)) {
	    if (data & bit(3)) {
		switch (data) {
		case MPU_MESS_CONDREQ:		/* 0xF9, conductor request */
		    mpusendcond(mp, unit);
		    break;
		case MPU_MESS_ACK:		/* 0xFE, command ack */
		    if (mp->mpud_state & MPUD_WACK) {
			mp->mpud_state &= ~MPUD_WACK;
			mpusendcom(mp, unit);
		    }
		    break;
		case MPU_MESS_SYS:		/* 0xFF, system message */
		    mp->mpud_state |= MPUD_SMESS;	/* fall thru */
		case MPU_MESS_TIMOFLO:		/* 0xF8 aka RT_TCIP */
		case MPU_MESS_CLK2HST:		/* 0xFD clock to host */
		    touser = 1;
			/* don't return these to host */
		default:		/* 0xFA, 0xFB, undef "never happens" */
		case MPU_MESS_DATAEND:		/* 0xFC aka "all end" */
		    break;
		}
	    } else				/* track data request */
		mpusendtrack(mp, unit, (int)(data & UCH(0x7)));
	} else {				/* time tag */
	    d4printf("time-tag ");
	    mp->mpud_state |= MPUD_TT;
	    touser = 1;
	}
	d4printf("state=%x touser=%d\n", mp->mpud_state, touser);
	return (touser);
}

/* recoverable wait-for-device-ready macro */
#define MPUDELAY()	if (!DRR_SET(mpur->mpur_csr) && mpudelay(mpur)) { \
			    mpuoffline(mp, unit);			\
			    return;					\
			}

/*
** mpusendcom(), used to send the next command data byte to the mpu.
** called at device interrupt priority, either by mpuwrite() or mpuisubr().
*/
mpusendcom(mp, unit)
register struct mpu_device *mp;
int unit;
{
	register struct mpu_l *mpul;
	register struct mpu_reg *mpur;
	unsigned char byte;

	mpur = MPU_REGADDR(unit);
	mpul = &mp->mpud_trackl[MPU_CTR_COM];
	while (mpul->mpul_tcount > 0) {
	    byte = mpugetbyte(mpul, mp->mpud_blimit);
	    MPUDELAY();
	    /*
	    ** If we're sending midi data we use mpul_contin to keep track
	    ** of how many bytes to send.
	    */
	    if (mp->mpud_state & MPUD_MIDI_DATA) {
		if ((mpul->mpul_contin = mpumidicnt(byte)) == 0) {
		    mp->mpud_state |= MPUD_ERROR;
/****/ uprintf("mpusendcom: 0x%x on the command track?\n", byte);
		}
		mp->mpud_state &= ~MPUD_MIDI_DATA;
	    }
	    if (mp->mpud_state & MPUD_MPU_DATA) {	/* data for mpu cmd */
		d5printf("mpusendcom: mpu data 0x%x\n", byte);
		POKEC(&mpur->mpur_data, byte);
		mp->mpud_state &= ~MPUD_MPU_DATA;
	    } else if (mpul->mpul_contin > 0) {		/* MIDI data */
		d5printf("mpusendcom: MIDI data 0x%x\n", byte);
		POKEC(&mpur->mpur_data, byte);
		--mpul->mpul_contin;
	    } else if (mpul->mpul_contin == -1) {	/* sys excl data */
		d5printf("mpusendcom: sys excl data 0x%x\n", byte);
		POKEC(&mpur->mpur_data, byte);
		if (MIDI_EOX(byte))
			mpul->mpul_contin = 0;
	    } else {					/* start of mpu cmd */
		/* Send out command byte...
		** Asynchronous acknowledge dispatches at mpuisubr()
		*/
		mp->mpud_state |= MPUD_WACK;
		if ((byte & UCH(0xF0)) == UCH(0xE0))	/* need mpu data */
		    mp->mpud_state |= MPUD_MPU_DATA;
		else if ((byte & UCH(0xF0)) == UCH(0xD0))	/* midi data */
		    mp->mpud_state |= MPUD_MIDI_DATA;
		d5printf("mpusendcom: mpu cmd 0x%x\n", byte);
		POKEC(&mpur->mpur_csr, byte);
		return;
	    }
	}
	mp->mpud_state &= ~MPUD_CMDTRACK;	/* all done with cmd track */
	wakeup((caddr_t)&mp->mpud_track);	/* for mpuclose() */
}

/*
** mpusendtrack() is used to send the next byte from a track of play data.
*/
mpusendtrack(mp, unit, track)
register struct mpu_device *mp;
int unit;
int track;
{
	register struct mpu_reg *mpur;
	register struct mpu_l *mpul;
	register int nbytes, i;
	unsigned char byte;

	if ((track < 0) || (track > 7))
	    panic("mpusendtrack");
	mpur = MPU_REGADDR(unit);
	mpul = &mp->mpud_trackl[track];
	if (mpul->mpul_tcount <= 0) {			/* nothing to send */
	    MPUDELAY();
	    POKEC(&mpur->mpur_data, 0x00);		/* timing byte */
	    d3printf("mpusendtrack(%d): sending DATA END (0xFC)\n", track);
	    MPUDELAY();
	    POKEC(&mpur->mpur_data, MPU_DAT_DATAEND);	/* send end of track */
	    wakeup((caddr_t)&mp->mpud_track);		/* for mpuclose() */
	    return;
	} else if ((byte = mpugetbyte(mpul, PEEK)) == MPU_TT_TCIP) {
	    d3printf("mpusendtrack(%d): sending TCIP (0xF8)\n", track);
	    MPUDELAY();					/* send 0xF8s as is */
	    POKEC(&mpur->mpur_data, byte);
	    mpugetbyte(mpul, mp->mpud_blimit);		/* flush the F8 */
	    return;
	} else if (byte > MPU_TT_MAX) {		/* oops!  NOT a timing byte */
	    mpugetbyte(mpul, mp->mpud_blimit);		/* flush this byte */
			    /* to be nice, we eat certain obvious mistakes */
	    if (byte == MPU_COM_RESET			/* skip reset */
	     || byte == MPU_DAT_DATAEND) {		/* or DATA END */
		uprintf("mpusendtrack(%d): bad time tag 0x%x flushed\n",
		 track, byte);
		mpusendtrack(mp, unit, track);		/* try again */
		return;
	    } else if (byte == MIDI_SX) {		/* skip sys excl */
		uprintf("mpusendtrack(%d): flushing sys excl\n");
		d3printf("\t0x%x flushed\n", byte);
		while (mpul->mpul_tcount > 0) {
		    byte = mpugetbyte(mpul, mp->mpud_blimit);
		    d3printf("\t0x%x flushed\n", byte);
		    if (MIDI_EOX(byte)) {
			mpusendtrack(mp, unit, track);
			return;
		    }
		}
	    }
	    uprintf("mpusendtrack(%d): bad time tag (%x)", track, byte);
	    uprintf(" flushing entire track...\n");
	    MPUDELAY();
	    POKEC(&mpur->mpur_data, MPU_TT_TCIP);	/* stall while we... */
	    while (mpul->mpul_tcount > 0)
		mpugetbyte(mpul, mp->mpud_blimit);	/* flush */
	    wakeup((caddr_t)&mp->mpud_track);		/* for mpuclose() */
	    return;
	}
	d3printf("mpusendtrack(%d): time tag 0x%x\n", track, byte);
	MPUDELAY();
	byte = mpugetbyte(mpul, mp->mpud_blimit);	 /* mpu time tag */
	POKEC(&mpur->mpur_data, byte);
	if (mpul->mpul_tcount <= 0) {
	    d3printf("mpusendtrack(%d): sending DATA END (0xFC)\n", track);
	    MPUDELAY();				/* time tag with no MIDI? */
	    POKEC(&mpur->mpur_data, MPU_DAT_DATAEND);	/* end of track */
	    wakeup((caddr_t)&mp->mpud_track);		/* for mpuclose() */
	    return;
	}
	/* Figure out how many bytes to send by examing the current byte in
	** the track.  Here again, we use mpul_contin to keep track of how
	** many bytes to send when the mpu uses running status.
	*/
	byte = mpugetbyte(mpul, PEEK);
	if (byte & bit(7)) {
	    if (byte == MPU_DAT_NOP			/* no op */
	     || byte == MPU_DAT_TCWME			/* measure end mark */
	     || byte == MPU_DAT_DATAEND)		/* data end mark */
		nbytes = 1;
	    else if (byte == MIDI_SX)			/* sys exclusive */
		nbytes = -1;				/* (not legal here) */
	    else if (nbytes = mpumidicnt(byte))
		mpul->mpul_contin = nbytes - 1;		/* midi command */
	    else
		mp->mpud_state |= MPUD_ERROR;		/* no good */
	} else						/* running status */
	    nbytes = mpul->mpul_contin;
	/* Now send the data */
	if (nbytes == -1) {				/* system exclusive */
	    uprintf("mpusendtrack(%d): sys.excl on data track!\n", track);
	    /* Can't send sys excl over a normal track, so skip over
	    ** entire message - if we run out of data we bail out.
	    */
	    while (mpul->mpul_tcount > 0) {	/* skip to end of sys excl */
		byte = mpugetbyte(mpul, mp->mpud_blimit);
		d3printf("mpusendtrack(%d): flushing 0x%x\n", track, byte);
		if (MIDI_EOX(byte))
		    break;
	    }
	    d3printf("mpusendtrack(%d): sending 0x%x\n", track,MPU_DAT_NOP);
	    MPUDELAY();			/* since we already sent a time tag */
	    POKEC(&mpur->mpur_data, MPU_DAT_NOP);	/* send a no-op */
	} else {	/* guard against expecting more bytes than there are. */
	    for (i = 0; i < nbytes && mpul->mpul_tcount > 0; i++) {
		d3printf("mpusendtrack(%d): sending 0x%x\n",
		 track, UCH(mpugetbyte(mpul, PEEK)));
		MPUDELAY();
		byte = mpugetbyte(mpul, mp->mpud_blimit);
		POKEC(&mpur->mpur_data, byte);
	    }
	}
	if (mpul->mpul_tcount <= 0)
	    wakeup((caddr_t)&mp->mpud_track);		/* for mpuclose() */
}

/*
** mpusendcond(), used to send conductor data.
** It's similar to, but not as complicated as, mpusendtrack().
*/
mpusendcond(mp, unit)
register struct mpu_device *mp;
int unit;
{
	register struct mpu_l *mpul;
	register struct mpu_reg *mpur;
	register int nbytes, i;
	unsigned char byte;

	d3printf("mpusendcond(0x%x, %d)\n", mp, unit);
	mpur = MPU_REGADDR(unit);
	mpul = &mp->mpud_trackl[MPU_CTR_CND];
	if (mpul->mpul_tcount <= 0) {			/* nothing to send */
	    MPUDELAY();
	    POKEC(&mpur->mpur_data, 0x00);		/* send a timing byte */
	    MPUDELAY();
	    POKEC(&mpur->mpur_data, MPU_DAT_DATAEND);	/* data end */
	    wakeup((caddr_t)&mp->mpud_track);		/* for mpuclose() */
	    return;
	}
	byte = mpugetbyte(mpul, mp->mpud_blimit);/* mpu time tag */
	MPUDELAY();
	POKEC(&mpur->mpur_data, byte);
	if (byte == MPU_TT_TCIP)			/* timing overflow */
	    return;
	if (mpul->mpul_tcount <= 0) {			/* just in case */
	    MPUDELAY();
	    POKEC(&mpur->mpur_data, MPU_DAT_DATAEND);
	    wakeup((caddr_t)&mp->mpud_track);		/* for mpuclose() */
	    return;
	}
	byte = mpugetbyte(mpul, PEEK);
	nbytes = ((UCH(0xE0) <= byte) && (byte <= UCH(0xEF)))? 2 : 1;
	for (i = nbytes; --i >= 0 && mpul->mpul_tcount > 0; ) {
	    MPUDELAY();
	    byte = mpugetbyte(mpul, mp->mpud_blimit);
	    POKEC(&mpur->mpur_data, byte);
	}
}

/*
** mpugetbyte() returns the next byte for a track.
** Track data is arranged as a linked list of system buffers.
** The limit arg is the number of buffers the track can allocate (needed
** so we can do a wakeup if freeing a buffer drops us below the limit)
** unless it's PEEK, in which case mpugetbyte won't advance to the next
** byte, thus can't free any buffers and can't bring us below the limit.
*/
unsigned char
mpugetbyte(mpul, limit)
register struct mpu_l *mpul;
int limit;
{
	register struct buf *bp;
	unsigned char byte;

	if (((bp = mpul->mpul_head.av_forw) == &mpul->mpul_head)
	 || (mpul->mpul_bcount <= 0))
/****/{
/****/ if (mpul->mpul_tcount >0) {
/****/  if (mpul->mpul_head.av_forw == &mpul->mpul_head)
/****/   uprintf("mpugetbyte() 0 buffers, but tcount=%d\n", mpul->mpul_tcount);
/****/  else
/****/   uprintf("mpugetbyte() > 0 buffers, but bcount=%d, while tcount=%d\n",
/****/   mpul->mpul_bcount, mpul->mpul_tcount);
/****/  mpul->mpul_tcount = 0;}
		return (NO_MO_DATA);		/* no more data */
/****/}
	byte = *mpul->mpul_cp;
	if (limit != PEEK) {
	    mpul->mpul_cp++;
	    --mpul->mpul_tcount;
	    if (--mpul->mpul_bcount <= 0) {		/* buf now empty */
		bp->av_back->av_forw = bp->av_forw;
		bp->av_forw->av_back = bp->av_back;
		brelse(bp);				/* free it */
		if (mpul->mpul_bused-- == limit)	/* blocked? */
		    wakeup((caddr_t)&mpul->mpul_bused);
		if ((bp=mpul->mpul_head.av_forw) != &mpul->mpul_head) {
		    mpul->mpul_cp = (unsigned char*)bp->b_un.b_addr;
		    mpul->mpul_bcount = bp->b_bcount;
/****/ if (bp->b_bcount <= 0)
/****/  uprintf("mpugetbyte() gets next buffer, b_bcount=%d?\n", bp->b_bcount);
		}
	    }
	}
	return (byte);
}

/*
** mpumidicnt() examines the midi command "byte" and
** returns the length of the whole command
** (number of bytes that should follow it + 1).
*/
mpumidicnt(byte)
unsigned char byte;
{
	switch (byte & MIDI_CMD_MASK) {
	case 0x80:			/* note off */
	case 0x90:			/* note on */
	case 0xA0:			/* poly keypress */
	case 0xB0:			/* control change */
	    return (3);
	case 0xC0:			/* program change */
	case 0xD0:			/* change after touch */
	    return (2);
	case 0xE0:			/* pitch wheel */
	    return (3);
	case 0xF0:			/* system messages */
	    switch (byte & UCH(0x0F)) {
	    case 0:			/* system exclusive	*/
		return (-1);		/* (watch for eox)	*/
	    case 2:			/* song position	*/
		return (3);
	    case 3:			/* song select		*/
		return (2);
	    case 6:			/* tune request		*/
	    case 7:			/* end of sys. excl.	*/
		return (1);
	    }
	    uprintf("mpumidicnt: unknown system msg (byte=0x%x)\n", byte);
	    return (0);
	default:			/* "can't happen" */
	    uprintf("mpumidicnt: byte=%x?\n", byte);
	    return (0);
	}
/*NOTREACHED*/
}

struct mpu_l *
mpugetl(mp)		/* return a pointer to a track buffer */
struct mpu_device *mp;
{

	if (0 <= mp->mpud_track && mp->mpud_track < MPU_TTR_MAX)
	    return (&mp->mpud_trackl[mp->mpud_track]);
	uprintf("mpugetl: track=%d\n", mp->mpud_track);
	panic("mpugetl");
/*NOTREACHED*/
}

mpupurge(mp, unit)		/* purge all track buffers */
register struct mpu_device *mp;
int unit;
{
	register struct mpu_l *tlp;
	register struct mpu_reg *mpur;
	int opri, bused;

	d2printf("mpupurge(0x%x, 0x%x)\n", mp, unit);
	mpur = MPU_REGADDR(unit);
	opri = splx(mp->mpud_spl);
	if (mp->mpud_state & (MPUD_MIDI_DATA | MPUD_MPU_DATA)) {
	    tlp = &mp->mpud_trackl[MPU_CTR_COM];
	    if (tlp->mpul_contin <= 0)
		tlp->mpul_contin = 1;
	    while (--tlp->mpul_contin >= 0) {
		MPUDELAY();
		POKEC(&mpur->mpur_data, MIDI_END_SX);	/* seem safe? */
	    }
	}
	for (tlp = &mp->mpud_trackl[MPU_TTR_MAX]; --tlp >= mp->mpud_trackl; ) {
	    mpufreeall(&tlp->mpul_head);
/****/  if (tlp->mpul_head.av_forw != &tlp->mpul_head)
/****/   uprintf("mpupurge() still has buffers after mpufreeall!\n");
	    tlp->mpul_tcount = 0;
	    tlp->mpul_bcount = 0;
	    tlp->mpul_contin = 0;
	    tlp->mpul_cp = 0;
	    bused = tlp->mpul_bused;		/* save for test */
	    tlp->mpul_bused = 0;
	    if (bused >= mp->mpud_blimit)
		wakeup((caddr_t)&tlp->mpul_bused);
	}
	wakeup((caddr_t)&mp->mpud_track);	/* for mpuclose(), natch */
	mp->mpud_state &= ~(MPUD_CMDTRACK | MPUD_WACK
	 | MPUD_MIDI_DATA | MPUD_MPU_DATA);
	(void) splx(opri);
	/* Send a 'Clear Play Map' command to the mpu to turn off all notes */
	MPUDELAY();
	POKEC(&mpur->mpur_csr, MPU_COM_CLRPLAYMAP);
	DELAY(250*MPU_DELAYTIME);
}

/*
** mpufreeall() returns all buffers of track dp to the system.
*/
mpufreeall(dp)
register struct buf *dp;
{
	register struct buf *bp;

	for (bp = dp->av_forw; bp != dp; bp = dp->av_forw) {
	    bp->av_back->av_forw = bp->av_forw;
	    bp->av_forw->av_back = bp->av_back;
	    brelse(bp);
	}
}

/*
** Mpureset() calls mpualive() to send a hard & soft reset to the mpu.
** This routine must be called (instead of mpualive) when interrupts
** are being fielded to prevent a race.
** Return 1 on success, 0 on failure.
**
** NB-Not called externally by SunOS.
*/
mpureset(unit)
int	unit;
{
	int opri, reset;

	opri = splx(mpu_device[unit].mpud_spl);
	reset = mpualive(MPU_CTLRADDR(unit), MPU_UOFF(unit));
	(void) splx(opri);
	if (reset != 1) {
	    uprintf("mpureset /dev/mpu%x: not responding\n", unit);
	    reset = 0;
	}
	return (reset);
}

/*
** hang (but not forever) waiting for mpu ready
** returns 0 on success, -1 if timeout
*/
mpudelay(mpur)
register struct mpu_reg *mpur;
{
	register int i;

	for (i = MPU_DELAYLOOPS; --i >= 0; ) {
	    if (DRR_SET(mpur->mpur_csr))
		return (0);
	    DELAY(MPU_DELAYTIME);
	}
	d1printf("mpudelay failed  ");
	return (-1);
}

/*
** hang (but not forever) waiting for mpu dsr
** returns 0 on success, -1 if timeout
*/
mpudsr(mpur)
	register struct mpu_reg *mpur;
{
	register int i;

	for (i = MPU_DELAYLOOPS; --i >= 0; ) {
	    if (DSR_SET(mpur->mpur_csr))
		return (0);
	    DELAY(MPU_DELAYTIME);
	}
	d1printf("mpudsr failed  ");
	return (-1);
}

/*
** mark mpu offline
*/
mpuoffline(mp, unit)
register struct mpu_device *mp;
int unit;
{
	mp->mpud_state |= MPUD_NOMPU;
	uprintf("/dev/mpu%x: offline\n", unit);
}

#ifndef SunOS3_x
/*
** Like tsleep() for the mpu driver (stolen from archive driver).
** Mpusleep() will return 1 if it was interrupted by the user typing ^C
** otherwise it returns 0 if it was awakened with wakeup().
**
** NB-from 3.2 onward, the PCATCH flag (from SystemV) does this for you.
*/
mpusleep(cp, pri)
caddr_t	cp;
int	pri;
{
	label_t q;

	q = u.u_qsave;
	if (setjmp(&u.u_qsave))
	    return (1);		/* interrupted  DOES THIS NEED u.u_qsave = q? */
	(void) sleep(cp, pri);
	u.u_qsave = q;
	return (0);
}
#endif pre-3.2/

#endif NMPU
