/*
**	MOZART -- Data from his Musical Dice Game & access routines
**	mozinit(key)
**		key  - key of piece (0=C, 2=D, etc.)
**	mozgen(buf, s, r, e, chan)
**		buf  - output buffer (needs ~100 bytes)
**		s    - segment # [0..15]
**		r    - random # [0..10] (rand() % 6 + rand() % 6)
**		e    - ending, [0..2] (2 => final)
**		chan - output MIDI chan [0..15]
**	psl 5/3/86
*/
#include	<stdio.h>
#include	<midi.h>
#include	<notedefs.h>

#define	NUMSEG	16
#define	NUMRND	11

#define	DUP	-1
#define	NXT	-2
#define	END	-3

typedef	struct	evstr	{		/* events */
	int	ev_tim;		/* how far into the measure it starts (16ths) */
	int	ev_key;		/* What note */
	int	ev_dur;		/* How long it lasts (16ths) */
} Event;

Event	ev[]	= {
	0,	F3,	2,	/* 1 */
	0,	F5,	2,
	2,	D3,	2,
	2,	D5,	2,
	4,	G3,	2,
	4,	G5,	2,
	0,	B2,	4,	/* 2 */
	0,	G3,	4,
	0,	A4,	2,
	2,	Gb4,	1,
	3,	G4,	1,
	4,	B4,	1,
	5,	G5,	1,
	0,	C3,	4,	/* 3 */
	0,	E3,	4,
	0,	G5,	2,
	2,	C5,	2,
	4,	E5,	2,
	0,	G5,	2,	/* 4 */
	0,	G2,	1,
	1,	B2,	1,
	2,	D5,	4,
	2,	G3,	2,
	4,	B2,	2,
	0,	G5,	4,	/* 5 (first ending) */
	0,	D5,	4,
	0,	B4,	4,
	0,	G4,	4,
	0,	G2,	2,
	2,	G3,	1,
	3,	F3,	1,
	4,	E3,	1,
	5,	D3,	1,
	0,	C3,	4,
	0,	E3,	4,
	0,	G4,	2,
	2,	C5,	2,
	4,	E5,	2,
	0,	C3,	4,
	0,	G3,	4,
	0,	E5,	1,
	1,	C5,	1,
	2,	E5,	1,
	3,	G5,	1,
	4,	C6,	1,
	5,	G5,	1,
	0,	C3,	2,
	0,	C5,	4,
	2,	G2,	2,
	4,	C2,	2,
	0,	G3,	4,
	0,	C5,	2,
	0,	E5,	2,
	2,	B5,	2,
	2,	D5,	2,
	4,	G2,	2,
	0,	G3,	4,	/* 10 */
	0,	B4,	1,
	1,	A4,	1,
	2,	B4,	1,
	3,	C5,	1,
	4,	D5,	1,
	5,	B4,	1,
	0,	C3,	2,
	0,	E5,	1,
	1,	C5,	1,
	2,	D3,	2,
	2,	B4,	1,
	3,	A4,	1,
	4,	C2,	2,
	4,	G4,	1,
	5,	Gb4,	1,
	0,	C3,	2,
	0,	E4,	2,
	0,	C5,	2,
	2,	C3,	2,
	2,	E4,	2,
	2,	C5,	2,
	4,	C3,	2,
	4,	E4,	2,
	4,	C5,	2,
	0,	E3,	4,
	0,	G3,	4,
	0,	C5,	2,
	2,	G4,	2,
	4,	E4,	2,
	DUP,	8,	0,
	0,	C3,	4,
	0,	G3,	4,
	0,	E5,	2,
	2,	G5,	1,
	3,	E5,	1,
	4,	C5,	2,
	4,	C3,	2,
	4,	E3,	2,
	0,	D3,	4,
	0,	Gb3,	4,
	0,	A5,	2,
	2,	Gb5,	2,
	4,	D5,	2,
	4,	C3,	2,
	4,	Gb3,	2,
	0,	E3,	4,
	0,	G3,	4,
	0,	C5,	1,
	1,	G4,	1,
	2,	C5,	1,
	3,	E5,	1,
	4,	G4,	1,
	5,	C5,	1,
	0,	C3,	4,
	0,	E3,	4,
	0,	G4,	2,
	2,	C5,	2,
	4,	E5,	2,
	4,	C3,	2,
	4,	E3,	2,
	0,	C3,	4,
	0,	E3,	4,
	0,	E5,	1,
	1,	C5,	1,
	2,	E5,	2,
	4,	G5,	2,
	4,	C3,	2,
	4,	E3,	2,
	0,	B2,	4,
	0,	G5,	2,
	2,	B5,	1,
	3,	D6,	1,
	4,	D5,	2,
	0,	C3,	2,
	0,	C5,	1,
	1,	E5,	1,
	2,	D3,	2,
	2,	G5,	1,
	3,	D5,	1,
	4,	D2,	2,
	4,	A4,	1,
	5,	Gb5,	1,
	0,	C3,	4,
	0,	E5,	2,
	2,	C5,	2,
	4,	G4,	2,
	0,	F3,	1,
	0,	F5,	1,
	1,	E3,	1,
	1,	E5,	1,
	2,	D3,	1,
	2,	D5,	1,
	3,	E3,	1,
	3,	E5,	1,
	4,	E3,	1,
	4,	F5,	1,
	5,	G3,	1,
	5,	G5,	1,
	DUP,	5,	0,
	0,	D3,	4,
	0,	D4,	1,
	1,	Gb4,	1,
	2,	A4,	1,
	3,	D5,	1,
	4,	Gb5,	1,
	5,	A5,	1,
	0,	C5,	2,
	0,	E5,	2,
	0,	C3,	1,
	1,	E3,	1,
	2,	C5,	2,
	2,	E5,	2,
	2,	G3,	1,
	3,	E3,	1,
	4,	C5,	2,
	4,	E5,	2,
	4,	C4,	1,
	5,	C3,	1,
	0,	G3,	4,
	0,	B3,	4,
	0,	F5,	1,
	1,	E5,	1,
	2,	F5,	1,
	3,	D5,	1,
	4,	C5,	1,
	5,	B4,	1,
	0,	C3,	4,
	0,	A3,	4,
	0,	Gb5,	1,
	1,	D5,	1,
	2,	A4,	1,
	3,	A5,	1,
	4,	Gb5,	1,
	5,	D5,	1,
	0,	G3,	4,
	0,	B4,	1,
	1,	D5,	1,
	2,	G5,	1,
	3,	D5,	1,
	4,	B4,	2,
	4,	G2,	2,
	DUP,	5,	0,
	0,	C3,	4,
	0,	G3,	4,
	0,	E5,	1,
	1,	C5,	1,
	2,	G4,	2,
	4,	E5,	2,
	4,	C3,	2,
	4,	G3,	2,
	DUP,	6,	0,
	DUP,	5,	0,
	0,	G3,	4,
	0,	E5,	1,
	1,	C5,	1,
	2,	D5,	1,
	3,	B4,	1,
	4,	G4,	2,
	0,	D3,	4,
	0,	Gb3,	4,
	0,	A4,	2,
	2,	D5,	2,
	4,	Gb5,	2,
	4,	C3,	2,
	4,	A3,	2,
	0,	C3,	2,
	0,	A4,	1,
	1,	E5,	1,
	2,	D3,	2,
	2,	D5,	1,
	3,	G5,	1,
	4,	D2,	2,
	4,	Gb5,	1,
	5,	A5,	1,
	0,	B2,	4,
	0,	D3,	4,
	0,	G5,	1,
	1,	B5,	1,
	2,	G5,	1,
	3,	D5,	1,
	4,	B4,	2,
	0,	C5,	2,
	0,	C3,	1,
	0,	E3,	1,
	1,	G3,	1,
	2,	G4,	2,
	2,	C3,	1,
	2,	E3,	1,
	3,	G3,	1,
	4,	E5,	2,
	4,	C3,	1,
	4,	E3,	1,
	5,	G3,	1,
	0,	G5,	2,
	0,	B2,	1,
	1,	D3,	1,
	2,	G4,	2,
	2,	G3,	1,
	3,	D3,	1,
	4,	G4,	2,
	4,	B2,	1,
	5,	G2,	1,
	0,	C3,	4,
	0,	E3,	4,
	0,	C5,	1,
	1,	B4,	1,
	2,	C5,	1,
	3,	E5,	1,
	4,	G4,	1,
	5,	C5,	1,
	0,	C3,	4,
	0,	E3,	4,
	0,	C5,	1,
	1,	B4,	1,
	2,	C5,	1,
	3,	E5,	1,
	4,	G4,	2,
	0,	G2,	4,
	0,	B4,	1,
	1,	C5,	1,
	2,	D5,	1,
	3,	B4,	1,
	4,	A4,	1,
	5,	G4,	1,
	0,	C3,	4,
	0,	E3,	4,
	0,	G5,	2,
	2,	F5,	1,
	3,	E5,	1,
	4,	D5,	1,
	5,	C5,	1,
	0,	F3,	4,
	0,	A4,	2,
	2,	F5,	1,
	3,	D5,	1,
	4,	G3,	2,
	4,	A4,	1,
	5,	B4,	1,
	0,	E3,	4,
	0,	G3,	4,
	0,	C5,	1,
	1,	B4,	1,
	2,	C5,	1,
	3,	G4,	1,
	4,	E4,	1,
	5,	C4,	1,
	0,	B2,	4,
	0,	D3,	4,
	0,	G5,	2,
	2,	B5,	1,
	3,	G5,	1,
	4,	D5,	1,
	5,	B4,	1,
	0,	B2,	4,
	0,	D3,	4,
	0,	G5,	2,
	2,	G5,	1,
	3,	D5,	1,
	4,	B5,	2,
	0,	C3,	4,
	0,	G3,	4,
	0,	E5,	2,
	2,	C5,	1,
	3,	E5,	1,
	4,	C3,	2,
	4,	E3,	2,
	4,	G5,	1,
	5,	C6,	1,
	0,	E5,	2,
	0,	C3,	1,
	0,	E3,	1,
	1,	G3,	1,
	2,	C5,	2,
	2,	C3,	1,
	2,	E3,	1,
	3,	G3,	1,
	4,	G4,	2,
	4,	C3,	1,
	4,	E3,	1,
	5,	G3,	1,
	0,	E3,	4,
	0,	G3,	4,
	0,	C5,	2,
	2,	E5,	1,
	3,	C5,	1,
	4,	G4,	2,
	0,	C3,	4,
	0,	E3,	4,
	0,	C5,	1,
	1,	G4,	1,
	2,	E5,	1,
	3,	C5,	1,
	4,	G5,	1,
	5,	E5,	1,
	0,	F3,	4,
	0,	D5,	1,
	1,	Db5,	1,
	2,	D5,	1,
	3,	F5,	1,
	4,	G4,	1,
	5,	B4,	1,
	0,	C3,	4,
	0,	C5,	2,
	0,	E5,	2,
	2,	C5,	1,
	2,	E5,	1,
	3,	D5,	1,
	3,	F5,	1,
	4,	E5,	2,
	4,	G5,	2,
	DUP,	12,	0,
	0,	B2,	4,
	0,	D3,	4,
	0,	G5,	2,
	2,	B5,	2,
	4,	D5,	2,
	0,	G2,	4,
	0,	G3,	4,
	0,	D5,	1,
	1,	B4,	1,
	2,	G4,	2,
	4,	G3,	2,
	DUP,	49,	0,
	0,	G5,	2,
	0,	C3,	1,
	0,	E3,	1,
	1,	G3,	1,
	2,	E5,	2,
	2,	C3,	1,
	2,	E3,	1,
	3,	G3,	1,
	4,	C5,	2,
	4,	C3,	1,
	4,	E3,	1,
	5,	G3,	1,
	0,	G5,	2,
	0,	C3,	1,
	0,	E3,	1,
	1,	G3,	1,
	2,	C5,	2,
	2,	C3,	1,
	2,	E3,	1,
	3,	G3,	1,
	4,	E5,	2,
	4,	C3,	1,
	4,	E3,	1,
	5,	G3,	1,
	DUP,	43,	0,
	0,	E3,	4,
	0,	G3,	4,
	0,	C5,	2,
	2,	E5,	1,
	3,	C5,	1,
	4,	G5,	2,
	0,	C3,	2,
	0,	E5,	1,
	1,	C5,	1,
	2,	D3,	2,
	2,	B4,	1,
	3,	G4,	1,
	4,	D2,	2,
	4,	A4,	1,
	5,	Gb4,	1,
	0,	C3,	4,
	0,	E5,	1,
	1,	C5,	1,
	2,	B4,	1,
	3,	C5,	1,
	4,	G4,	2,
	0,	C3,	4,
	0,	G3,	4,
	0,	E5,	1,
	1,	G5,	1,
	2,	C6,	1,
	3,	G5,	1,
	4,	C3,	2,
	4,	G3,	2,
	4,	E5,	1,
	5,	C5,	1,
	0,	D3,	4,
	0,	Gb3,	4,
	0,	D5,	1,
	1,	A4,	1,
	2,	D5,	2,
	4,	Gb5,	2,
	0,	D3,	2,
	0,	A3,	2,
	0,	Gb5,	2,
	2,	D3,	2,
	2,	Gb3,	2,
	2,	A5,	2,
	4,	C3,	2,
	4,	D3,	2,
	4,	Gb5,	2,
	0,	C3,	4,
	0,	E3,	4,
	0,	C5,	1,
	1,	B4,	1,
	2,	C5,	1,
	3,	E5,	1,
	4,	E3,	2,
	4,	G3,	2,
	4,	G4,	1,
	5,	C5,	1,
	0,	B2,	4,
	0,	G5,	2,
	2,	B5,	1,
	3,	G5,	1,
	4,	D5,	1,
	5,	G5,	1,
	0,	C3,	4,
	0,	E3,	4,
	0,	G5,	2,
	2,	E5,	2,
	4,	C5,	2,
	0,	D3,	4,
	0,	Gb5,	2,
	2,	A5,	1,
	3,	Gb5,	1,
	4,	C3,	2,
	4,	D5,	1,
	5,	Gb5,	1,
	0,	B2,	4,
	0,	D3,	4,
	0,	G5,	1,
	1,	B5,	1,
	2,	D6,	1,
	3,	B5,	1,
	4,	B2,	2,
	4,	D3,	2,
	4,	G5,	2,
	0,	F3,	4,
	0,	F5,	1,
	1,	E5,	1,
	2,	D5,	1,
	3,	C5,	1,
	4,	G3,	2,
	4,	B4,	1,
	5,	D5,	1,
	DUP,	58,	0,
	0,	C3,	4,
	0,	E3,	4,
	0,	C6,	1,
	1,	B5,	1,
	2,	C6,	1,
	3,	G5,	1,
	4,	E5,	1,
	5,	C5,	1,
	0,	C3,	2,
	0,	D5,	2,
	0,	Gb5,	2,
	2,	C3,	2,
	2,	D5,	2,
	2,	Gb5,	2,
	4,	C3,	2,
	4,	D5,	2,
	4,	Gb5,	2,
	0,	C3,	4,
	0,	E3,	4,
	0,	C6,	1,
	1,	B5,	1,
	2,	C6,	1,
	3,	G5,	1,
	4,	C3,	2,
	4,	G3,	2,
	4,	E5,	1,
	5,	C5,	1,
	0,	B2,	4,
	0,	D3,	4,
	0,	G5,	1,
	1,	B5,	1,
	2,	G5,	2,
	4,	B2,	2,
	4,	G3,	2,
	4,	D5,	2,
	0,	C3,	4,
	0,	C5,	2,
	2,	C4,	2,
	4,	C2,	2,
	DUP,	8,	0,
	0,	C3,	4,
	0,	D5,	2,
	2,	A4,	2,
	4,	Gb5,	2,
	DUP,	5,	0,
	0,	B2,	4,
	0,	G3,	4,
	0,	D5,	1,
	1,	B4,	1,
	2,	G4,	2,
	4,	B2,	2,
	4,	D3,	2,
	4,	G5,	2,
	DUP,	8,	0,
	DUP,	51,	0,
	0,	E3,	4,
	0,	G3,	4,
	0,	C5,	2,
	2,	E5,	2,
	4,	G4,	2,
	0,	B2,	4,
	0,	G3,	4,
	0,	D5,	2,
	2,	D5,	1,
	3,	G5,	1,
	4,	B5,	2,
	0,	C3,	4,
	0,	E3,	4,
	0,	G5,	2,
	2,	C5,	2,
	4,	C3,	2,
	4,	G3,	2,
	4,	E5,	2,
	0,	B2,	4,
	0,	D3,	4,
	0,	G5,	1,
	1,	D5,	1,
	2,	G5,	1,
	3,	B5,	1,
	4,	B2,	2,
	4,	D3,	2,
	4,	G5,	1,
	5,	D5,	1,
	0,	F3,	1,
	0,	F5,	1,
	1,	E3,	1,
	1,	E5,	1,
	2,	D3,	2,
	2,	D5,	2,
	4,	G3,	2,
	4,	G5,	2,
	0,	C3,	4,
	0,	A3,	4,
	0,	Gb5,	1,
	1,	A5,	1,
	2,	D6,	1,
	3,	A5,	1,
	4,	C3,	2,
	4,	A3,	2,
	4,	Gb5,	1,
	5,	A5,	1,
	DUP,	5,	0,
	0,	G2,	4,
	0,	G3,	4,
	0,	B4,	2,
	0,	D5,	2,
	2,	G5,	1,
	3,	B5,	1,
	4,	G3,	2,
	4,	D5,	2,
	DUP,	8,	0,
	DUP,	5,	0,
	DUP,	69,	0,
	DUP,	22,	0,
	0,	B2,	4,
	0,	D3,	4,
	0,	G5,	1,
	1,	Gb5,	1,
	2,	G5,	1,
	3,	D5,	1,
	4,	B2,	2,
	4,	G3,	2,
	4,	B4,	1,
	5,	G4,	1,
	DUP,	38,	0,
	0,	C3,	4,	/* 99 */
	0,	A3,	4,
	0,	Gb5,	2,
	2,	A5,	2,
	4,	C3,	2,
	4,	A3,	2,
	4,	D5,	2,
	DUP,	5,	0,
	0,	C3,	4,
	0,	G3,	4,
	0,	E5,	1,
	1,	D5,	1,
	2,	E5,	1,
	3,	G5,	1,
	4,	C3,	2,
	4,	E3,	2,
	4,	C6,	1,
	5,	G5,	1,
	0,	C3,	4,
	0,	A3,	4,
	0,	Gb5,	1,
	1,	D5,	1,
	2,	A4,	2,
	4,	C3,	2,
	4,	A3,	2,
	4,	Gb5,	2,
	0,	E3,	4,
	0,	G3,	4,
	0,	C5,	1,
	1,	E5,	1,
	2,	C5,	1,
	3,	G4,	1,
	4,	E4,	2,
	0,	C3,	4,
	0,	E5,	1,
	1,	D5,	1,
	2,	E5,	1,
	3,	G5,	1,
	4,	C6,	1,
	5,	G5,	1,
	0,	C3,	4,	/* 105 */
	0,	Gb5,	2,
	2,	A5,	1,
	3,	Gb5,	1,
	4,	D5,	1,
	5,	Gb5,	1,
	0,	C3,	2,
	0,	A4,	2,
	2,	D3,	2,
	2,	D5,	1,
	3,	C5,	1,
	4,	D2,	2,
	4,	B4,	1,
	5,	A4,	1,
	DUP,	5,	0,
	0,	C3,	4,
	0,	G3,	4,
	0,	E5,	2,
	2,	G5,	2,
	4,	C3,	2,
	4,	E3,	2,
	4,	C5,	2,
	0,	F3,	4,
	0,	A3,	4,
	0,	D5,	1,
	1,	F5,	1,
	2,	D5,	1,
	3,	F5,	1,
	4,	G3,	2,
	4,	D4,	2,
	4,	B4,	1,
	5,	D5,	1,
	0,	C3,	2,	/* 110 */
	0,	B4,	1,
	0,	D5,	1,
	1,	A4,	1,
	1,	C5,	1,
	2,	D3,	2,
	2,	A4,	1,
	2,	C5,	1,
	3,	G4,	1,
	3,	B4,	1,
	4,	D2,	2,
	4,	G4,	1,
	4,	B4,	1,
	5,	Gb4,	1,
	5,	A4,	1,
	DUP,	8,	0,
	DUP,	49,	0,
	0,	G3,	4,
	0,	B3,	4,
	0,	F5,	2,
	2,	D5,	2,
	4,	B4,	2,
	0,	G3,	2,
	0,	B4,	2,
	0,	D5,	2,
	2,	G3,	2,
	2,	B4,	2,
	2,	D5,	2,
	4,	G3,	2,
	4,	B4,	2,
	4,	D5,	2,
	0,	C3,	4,	/* 115 */
	0,	E3,	4,
	0,	C5,	1,
	1,	G4,	1,
	2,	E5,	1,
	3,	C5,	1,
	4,	G5,	1,
	5,	E5,	1,
	0,	F3,	4,
	0,	D5,	1,
	1,	F5,	1,
	2,	A5,	1,
	3,	F5,	1,
	4,	G3,	2,
	4,	D5,	1,
	5,	B4,	1,
	0,	D3,	4,
	0,	Gb3,	4,
	0,	D5,	1,
	1,	A4,	1,
	2,	D5,	1,
	3,	Gb5,	1,
	4,	A5,	1,
	5,	Gb5,	1,
	0,	C3,	2,
	0,	E5,	1,
	1,	A5,	1,
	2,	D3,	2,
	2,	G5,	1,
	3,	B5,	1,
	4,	D2,	2,
	4,	Gb5,	1,
	5,	A5,	1,
	0,	C3,	4,
	0,	E3,	4,
	0,	E5,	1,
	1,	C5,	1,
	2,	G5,	1,
	3,	E5,	1,
	4,	C6,	1,
	5,	G5,	1,
	0,	D3,	4,	/* 120 */
	0,	Gb3,	4,
	0,	D6,	2,
	2,	A5,	1,
	3,	Gb5,	1,
	4,	C3,	2,
	4,	Gb3,	2,
	4,	D5,	1,
	5,	A4,	1,
	0,	B2,	4,
	0,	G3,	4,
	0,	G5,	2,
	2,	B5,	1,
	3,	G5,	1,
	4,	D5,	2,
	0,	B2,	2,
	0,	D3,	2,
	0,	G5,	1,
	1,	Gb5,	1,
	2,	B2,	2,
	2,	D3,	2,
	2,	G5,	1,
	3,	B5,	1,
	4,	B2,	2,
	4,	G3,	2,
	4,	D5,	2,
	DUP,	5,	0,
	DUP,	12,	0,
	0,	G3,	2,	/* 125 */
	0,	G5,	1,
	1,	E5,	1,
	2,	G2,	2,
	2,	D5,	1,
	3,	B4,	1,
	4,	G4,	2,
	0,	E3,	4,
	0,	C5,	1,
	1,	G4,	1,
	2,	C5,	1,
	3,	E5,	1,
	4,	E3,	1,
	4,	G5,	1,
	5,	C3,	1,
	5,	C5,	1,
	5,	E5,	1,
	DUP,	5,	0,
	0,	G2,	4,
	0,	B4,	2,
	2,	D5,	2,
	4,	G5,	2,
	0,	B2,	2,
	0,	D3,	2,
	0,	A5,	1,
	1,	G5,	1,
	2,	B2,	2,
	2,	D3,	2,
	2,	Gb5,	1,
	3,	G5,	1,
	4,	B2,	2,
	4,	G3,	2,
	4,	D5,	2,
	DUP,	12,	0,	/* 130 */
	DUP,	8,	0,
	0,	G3,	2,
	0,	C5,	2,
	0,	E5,	2,
	2,	B4,	1,
	2,	D5,	1,
	3,	G4,	1,
	3,	B4,	1,
	4,	G4,	2,
	0,	B2,	4,
	0,	G3,	4,
	0,	D5,	2,
	2,	G5,	1,
	3,	D5,	1,
	4,	B4,	1,
	5,	D5,	1,
	0,	C3,	2,
	0,	A4,	1,
	1,	E5,	1,
	2,	D3,	2,
	2,	B4,	1,
	2,	D5,	1,
	3,	A4,	1,
	3,	C5,	1,
	4,	D2,	2,
	4,	G4,	1,
	4,	B4,	1,
	5,	Gb4,	1,
	5,	A4,	1,
	0,	C3,	2,	/* 135 */
	0,	D3,	2,
	0,	Gb5,	2,
	2,	C3,	2,
	2,	D3,	2,
	2,	Gb5,	1,
	3,	D5,	1,
	4,	C3,	2,
	4,	D3,	2,
	4,	A5,	2,
	DUP,	74,	0,
	DUP,	38,	0,
	0,	D2,	1,
	0,	A4,	2,
	0,	D5,	2,
	0,	Gb5,	2,
	1,	D3,	1,
	2,	Db3,	1,
	2,	Gb5,	4,
	3,	D3,	1,
	4,	Db3,	1,
	5,	D3,	1,
	0,	B2,	4,
	0,	G5,	1,
	1,	B5,	1,
	2,	G5,	1,
	3,	B5,	1,
	4,	D5,	2,
	0,	C3,	2,	/* 140 */
	0,	Gb3,	2,
	0,	A4,	2,
	2,	C3,	2,
	2,	Gb3,	2,
	2,	A4,	1,
	3,	D5,	1,
	4,	C3,	2,
	4,	A3,	2,
	4,	Gb5,	2,
	0,	B2,	4,
	0,	G3,	4,
	0,	D5,	1,
	1,	E5,	1,
	2,	F5,	1,
	3,	D5,	1,
	4,	G2,	2,
	4,	C5,	1,
	5,	B4,	1,
	0,	C3,	4,
	0,	E3,	4,
	0,	C5,	2,
	2,	G4,	2,
	4,	E5,	2,
	0,	B2,	4,
	0,	D3,	4,
	0,	G5,	2,
	2,	D5,	1,
	3,	B4,	1,
	4,	B2,	2,
	4,	D3,	2,
	4,	G4,	2,
	DUP,	59,	0,
	0,	F3,	4,	/* 145 */
	0,	D5,	1,
	1,	F5,	1,
	2,	A4,	1,
	3,	D5,	1,
	4,	G3,	2,
	4,	B4,	1,
	5,	D5,	1,
	0,	C3,	2,
	0,	Gb4,	2,
	0,	D5,	2,
	2,	C3,	2,
	2,	D5,	2,
	2,	Gb5,	2,
	4,	C3,	2,
	4,	Gb5,	2,
	4,	A5,	2,
	0,	C3,	2,
	0,	E5,	1,
	1,	C5,	1,
	2,	D3,	2,
	2,	B5,	1,
	3,	G5,	1,
	4,	D2,	2,
	4,	A5,	1,
	5,	Gb5,	1,
	DUP,	74,	0,
	0,	F3,	4,
	0,	F5,	1,
	1,	D5,	1,
	2,	A4,	2,
	4,	G3,	2,
	4,	B4,	2,
	0,	C3,	1,	/* 150 */
	0,	G4,	2,
	0,	C5,	2,
	0,	E5,	2,
	1,	B2,	1,
	2,	C3,	1,
	2,	E5,	4,
	3,	D3,	1,
	4,	E3,	1,
	5,	Gb3,	1,
	DUP,	8,	0,
	DUP,	43,	0,
	0,	C3,	4,
	0,	D5,	1,
	1,	A4,	1,
	2,	Gb5,	1,
	3,	D5,	1,
	4,	A5,	1,
	5,	Gb5,	1,
	0,	C3,	4,
	0,	D5,	1,
	1,	Db5,	1,
	2,	D5,	1,
	3,	Gb5,	1,
	4,	A5,	1,
	5,	Gb5,	1,
	0,	B2,	4,	/* 155 */
	0,	D3,	4,
	0,	G5,	1,
	1,	B5,	1,
	2,	G5,	1,
	3,	D5,	1,
	4,	B4,	1,
	5,	G4,	1,
	0,	E3,	4,
	0,	G3,	4,
	0,	C5,	1,
	1,	G4,	1,
	2,	E5,	1,
	3,	C5,	1,
	4,	G5,	2,
	DUP,	104,	0,
	0,	G2,	4,
	0,	B4,	2,
	2,	D5,	1,
	3,	B4,	1,
	4,	A4,	1,
	5,	G5,	1,
	0,	C3,	2,
	0,	E5,	1,
	1,	G5,	1,
	2,	D3,	2,
	2,	D5,	1,
	3,	C5,	1,
	4,	D2,	2,
	4,	B4,	1,
	5,	A4,	1,
	0,	C3,	4,	/* 160 */
	0,	E3,	4,
	0,	C5,	1,
	1,	B4,	1,
	2,	C5,	1,
	3,	E5,	1,
	4,	C3,	2,
	4,	E3,	2,
	4,	G4,	1,
	5,	C5,	1,
	0,	C3,	2,
	0,	Gb4,	2,
	0,	D5,	2,
	2,	C3,	2,
	2,	Gb4,	2,
	2,	D5,	2,
	4,	C3,	2,
	4,	Gb4,	2,
	4,	D5,	2,
	DUP,	101,	0,
	0,	B2,	4,
	0,	D3,	4,
	0,	G5,	1,
	1,	Gb5,	1,
	2,	G5,	1,
	3,	D5,	1,
	4,	B4,	1,
	5,	G4,	1,
	0,	G3,	1,
	0,	D5,	2,
	1,	Gb3,	1,
	2,	G4,	4,
	2,	G3,	1,
	3,	D3,	1,
	4,	B2,	1,
	5,	G2,	1,
	0,	B2,	4,	/* 165 */
	0,	D5,	2,
	2,	B4,	2,
	4,	G4,	2,
	0,	G3,	4,
	0,	B3,	4,
	0,	D5,	1,
	1,	B5,	1,
	2,	G5,	1,
	3,	D5,	1,
	4,	B4,	2,
	0,	C3,	4,
	0,	E3,	4,
	0,	C5,	2,
	2,	C5,	1,
	3,	D5,	1,
	4,	E5,	2,
	0,	C3,	4,
	0,	E3,	4,
	0,	G5,	2,
	2,	F5,	1,
	3,	E5,	1,
	4,	E3,	2,
	4,	G3,	2,
	4,	D5,	1,
	5,	C5,	1,
	0,	C3,	2,
	0,	E5,	1,
	1,	G5,	1,
	2,	D3,	2,
	2,	D5,	1,
	3,	G5,	1,
	4,	D2,	2,
	4,	A4,	1,
	5,	Gb5,	1,
	DUP,	8,	0,	/* 170 */
	0,	G2,	4,
	0,	G3,	4,
	0,	B4,	1,
	1,	C5,	1,
	2,	D5,	1,
	3,	E5,	1,
	4,	B2,	2,
	4,	G3,	2,
	4,	F5,	1,
	5,	D5,	1,
	DUP,	8,	0,
	0,	F3,	4,
	0,	F5,	1,
	1,	A5,	1,
	2,	A4,	2,
	4,	G3,	2,
	4,	B4,	1,
	5,	D5,	1,
	0,	C3,	1,
	0,	E3,	1,
	0,	G4,	2,
	1,	G3,	1,
	2,	C3,	1,
	2,	E3,	1,
	2,	C5,	2,
	3,	G3,	1,
	4,	C3,	1,
	4,	E3,	1,
	4,	E5,	2,
	5,	G3,	1,
	0,	G3,	2,
	0,	E5,	1,
	1,	C5,	1,
	2,	G2,	2,
	2,	B4,	1,
	3,	D5,	1,
	4,	G5,	2,
	0,	B2,	4,
	0,	D3,	4,
	0,	A5,	1,
	1,	G5,	1,
	2,	B5,	1,
	3,	G5,	1,
	4,	B2,	2,
	4,	D3,	2,
	4,	D5,	1,
	5,	G5,	1,
	0,	G5,	4,	/* 177, actually second ending for 5 */
	0,	D5,	4,
	0,	B4,	4,
	0,	G4,	4,
	0,	G2,	2,
	2,	B3,	1,
	3,	G3,	1,
	4,	Gb3,	1,
	5,	E3,	1,
	END,
};


int	rifftab[NUMSEG][NUMRND]	= {
	 22,   6,  69,  40,  74, 104,  43, 119,  38,   3,  12,
	 22,   6,  69,  17,  74, 104,  43,  51, 142,  87,  12,
	141, 128, 158, 113, 163,  27, 171, 114,  42, 165,  10,
	 41,  63,  13,  85,  45, 167,  53,  50, 156,  61, 103,
	105, 146, 153, 161,  80, 154,  99, 140,  75, 135,  28,
	122,  46,  55,   2,  97,  68, 133,  86, 129,  47,  37,
	 11, 134, 110, 159,  36, 118,  21, 169,  62, 147, 106,
	  5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,

	 70, 117,  66,  90,  25, 138,  16, 120,  65, 102,  35,
	121,  39, 139, 176, 143,  71, 155,  88,  77,   4,  20,
	 26, 126,  15,   7,  64, 150,  49,  48,  19,  31, 108,
	  9,  56, 132,  34, 125,  29, 175, 166,  82, 164,  92,
	 49, 174,  58,  67,  76, 101,  43,  51,  38,  59,  12,
	 49,  18,  58, 160,  74, 101, 168, 115,  38,  59,  12,
	109, 116, 145,  52,   1,  23,  89,  72, 149, 173,  44,
	  8,   8,   8,   8,   8,   8,   8,   8,   8,  78,   8,
};

int	Key;
int	Loc[NUMSEG * NUMRND + 1];

mozinit(key)
{
	register int t, i, e, l;

	Key = key;
	Loc[0] = l = 0;
	for (i = e = 1; (t = ev[e].ev_tim) != END; e++) {
	    if (t < l) {
		if (t == DUP) {
		    Loc[i++] = ev[e].ev_key - 1;
		    t = 99;
		} else if (t == NXT)
		    t = 99;
		else
		    Loc[i++] = e;
	    }
	    l = t;
	}
	for (t = NUMSEG; --t >= 0; )
	    for (i = NUMRND; --i >= 0; )
		rifftab[t][i] = Loc[rifftab[t][i] - 1];
}

#define	CPS	30		/* MIDI clocks per sixteenth note */
#define	KVEL	64		/* default key velocity */

mozgen(buf, s, r, e, chan)
char	*buf;
{
	register int onum, lt, t, i;
	char *cp;
	Event *ep;
	struct { int when, key; } off[32];

	ep = &ev[rifftab[s][r]];
	if (s == 7 && e == 2)		/* second ending */
	    ep = &ev[Loc[177 - 1]];
	onum = 0;
	lt = 0;
	cp = buf;
	for (t = 0; t <= 6; t++) {
	    for (i = 0; i < onum; i++) {
		if (off[i].when == t) {
		    *cp++ = (t - lt) * CPS;
		    lt = t;
		    *cp++ = CH_KEY_ON | chan;
		    *cp++ = off[i].key;
		    *cp++ = 0;
		}
	    }
	    while (ep->ev_tim == t) {
		off[onum].when = t + ep->ev_dur;
		off[onum++].key = Key + ep->ev_key;
		*cp++ = (t - lt) * CPS;
		lt = t;
		*cp++ = CH_KEY_ON | chan;
		*cp++ = Key + ep->ev_key;
		*cp++ = KVEL;
		ep++;
	    }
	}
	*cp++ = (6 - lt) * CPS;
	*cp++ = RT_TCWME;
	return(cp - buf);
}
