/*
**	MIXPLAY -- Mix recorded & "live" MIDI data & play it.
**	P. Langston, 10/87
*/
#include	<stdio.h>
#include	<signal.h>
#include	<midi.h>
#include	<mpuvar.h>
#include	<sys/time.h>
#include	<sys/ioctl.h>

#define	MAXMPUS	4		/* how many MPUs we can handle */
#define	MAXTPM	8		/* how many tracks each MPU can have */
#define	MAXINP	64		/* how many simultaneous inputs we can have */

char	Mbuf[512], *Mbp = Mbuf;	/* misc commands */
int	Sync = 'i';		/* clock sync spec */
int	Record	= -1;		/* MPU #, record (overdub) while playing */
int	Debug	= 0;		/* if set, output ascii (to a tty) */
int	Midifh;			/* file descriptor for /dev/mpu */
int	Fpipe	= 0;		/* is at least one source a pipe? */
int	Ninp	= 0;		/* number of sources */

struct	mpustr	{
	int	fd;		/* file descriptor for the MPU device */
	int	trax;		/* how many tracks we've assigned */
	int	mode;		/* mode in which to start MPU */
	char	name[32];	/* device file name */
} Mpu[MAXMPUS];

struct	inpstr	{		/* track info */
	FILE	*fp;		/* input file pointer */
	int	fd;		/* file descriptors for Fp[] */
	int	lm;		/* 0 ==> time-tagged MPU, 1 ==> "live" MIDI */
	int	dorc;		/* 0 ==> data, 1 ==> MPU commands */
	int	mpu;		/* which MPU this track uses */
	int	trak;		/* which track on this MPU (0-9) */
} Inp[MAXINP];

#define	TBITSLOC	1
#define	SYNCLOC		5
#define	MODELOC		6
char	Minit[]	= {		/* default tape sync initialization */
	MPU_ACTIVE_TRACKS,
	0,			/* track mask goes here */
	MPU_CLEAR_PLAY_COUNTERS,
	MPU_MIDI_THRU_OFF, MPU_BENDER_ON,
	MPU_INT_CLOCK,		/* sync type goes here */
	0,			/* mode command (START, OVERDUB) goes here */
};

main(argc, argv)
char	*argv[];
{
	char *init_file = NULL, *arg;
	int i, Wait=0;
	int inraw, dorc, mpu;
	void interrupt();

/****/setbuf(stderr, 0);	/* THOSE ASSHOLES! */
	av0 = argv[0];
	signal(SIGINT, interrupt);
	for (i = MAXMPUS; --i >= 0; ) {
	    Mpu[i].fd = -1;
	    sprintf(Mpu[i].name, "/dev/mpu%d", i);
	}
	for (i = MAXINP; --i >= 0; Inp[i].fd = -1);
	inraw = dorc = 0;
	openmpu(mpu = 0);
	for (i = 1; i < argc; i++) {
	    arg = argv[i];
	    if (arg[0] == '-') {
		switch (arg[1]) {
		case '-': in(&arg[2], mpu, inraw, dorc); break;
		case 'a': Debug = 1; break;
		case 'C': dorc = 1; break;
		case 'c': init_file = &arg[2]; break;
		case 'D': dorc = 0; break;
		case 'd': openmpu(mpu = atoi(&arg[2])); break;
		case 'M': if (setmeter(1, &arg[2])) syntax(2); break;
		case 'm': if (setmeter(0, &arg[2])) syntax(2); break;
		case 'R': inraw = 1; break;
		case 'r': if (Record >= 0) syntax(1);
			Mpu[Record = mpu].mode = MPU_START_OVERDUB; break;
		case 'S': Sync = arg[2]; break;
		case 'T': inraw = 0; break;
		case 't': *Mbp++ = MPU_TEMPO, *Mbp++ = atoi(&arg[2]); break;
		case 'w': Wait = atoi(&arg[2]); break;
		case 'x': *Mbp++ = MPU_EXCLUSIVE_TO_HOST_ON;
			*Mbp++ = MPU_SEND_MEASURE_END_OFF; break;
		default : syntax(2);
		}
	    } else
		in(arg, mpu, inraw, dorc);
	}
	if (Wait || Fpipe)
	    sleep(Wait? Wait : 2); /* let input pipes fill up a little */
	if (midiinit(init_file))
	    Error(init_file);
	splay();
	if (Record >= 0)
	    for(;;)
		record(fileno(stdout));
	done();
}

setmeter(acc, cp)
char	*cp;
{
	register int i;

	if (acc)
	    *Mbp++ = MPU_METRO_ACC;
	else
	    *Mbp++ = MPU_METRO_NO_ACC;
	if (!cp || !*cp || (i = atoi(cp)) == 0)
	    return(0);
	if (i < 0 || i > 127)
	    return(1);
	*Mbp++ = MPU_METRO_MEAS;
	*Mbp++ = i;
	for (; *cp && *cp != '/'; cp++);
	if (*cp++ == '/') {
	    i = atoi(cp);
	    if (i < 0 || i > 96)
		return(1);
	    *Mbp++ = MPU_MIDI_METRO;
	    *Mbp++ = 96 / i;
	}
	return(0);
}

openmpu(mpu)			/* open appropriate /dev/mpu? */
{
	if (mpu < 0 || mpu >= MAXMPUS) {
	    fprintf(stderr,
	     "mpu number, %d, out of range (0-%d).\n", mpu, MAXMPUS - 1);
	    syntax(1);
	}
	if ((Mpu[mpu].fd = open(Mpu[mpu].name, 2)) < 0) {
/****/fprintf(stderr, "openmpu()  ");
	    perror(Mpu[mpu].name);
	    syntax(1);
	}
	Mpu[mpu].mode = MPU_START_PLAY;
}

in(file, mpu, inraw, dorc)	/* associate input data with mpu track */
char	*file;
{
	struct mpustr *mp;
	struct inpstr *ip;

	if (mpu < 0 || mpu > MAXMPUS) {
	    fprintf(stderr,
	     "mpu number, %d, out of range (0-%d).\n", mpu, MAXMPUS - 1);
	    syntax(1);
	}
	if (Ninp >= MAXINP) {
	    fprintf(stderr, "Too many input sources, max is %d.\n", MAXINP);
	    syntax(1);
	}
	mp = &Mpu[mpu];
	if (mp->trax >= MAXTPM) {
	    fprintf(stderr,
	     "Too many tracks for mpu%d, max is %d.\n", mpu, MAXTPM);
	    syntax(1);
	}
	ip = &Inp[Ninp++];
	if (file && *file)
	    ip->fp = sopen(file, "r");
	else
	    ip->fp = stdin;
	if (!ip->fp) {
	    perror(file);
	    ip->fd = -1;		/* ip->fp = (FILE *) 0; */
	} else {
	    ip->fd = fileno(ip->fp);
	    Fpipe += isapipe(ip->fd);
	    ip->lm = inraw;
#ifdef	UNDEF
	    if (inraw)
		fseek(ip->fp, 0, 2);	/* seek to end of live files */
#endif
	    ip->dorc = dorc;
	    ip->mpu = mpu;
	    ip->trak = dorc? (inraw? MPU_CTR_COM : MPU_CTR_CND) : mp->trax++;
	}
}

done()
{
	int i;

	for (i = 0; i < MAXMPUS; i++)
	    if (Mpu[i].fd >= 0)
		close(Mpu[i].fd);
	exit(0);
}

Error(s)
{
	MidiError("%s: ", av0);
/****/fprintf(stderr, "Error   ");
	perror(s);
	exit(1);
}

midiinit(file)
char *file;
{
	register int i, n;
	char buf[1024], *iseq;
	int d, trackbits[MAXMPUS], islen;
	struct inpstr *ip;
	struct mpustr *mp;
	FILE *ifp;

	for (i = MAXMPUS; --i >= 0; trackbits[i] = 0);
	for (i = 0; i < Ninp; i++) {
	    ip = &Inp[i];
	    mp = &Mpu[ip->mpu];
	    if (!ip->lm && ip->trak < MPU_DTR_NUM) {
		trackbits[ip->mpu] |= (1 << ip->trak);
		if ((n = read(ip->fd, buf, sizeof buf)) < sizeof buf) {
		    sclose(ip->fp);
		    ip->fp = (FILE *) 0;
		    ip->fd = -1;
		}
		if (n > 0
		 && (MpuSetTrack(mp->fd, ip->trak) == -1
		  || mwrite(mp->fd, buf, n) != n)) {
/****/fprintf(stderr, "midiinit() MpuSetTrack() || write() failed   ");
		    perror(mp->name);
		    return(-1);
		}
/****/fprintf(stderr, "track=%d\n", ip->trak);
	    }
	}
	/* send initialization sequence */
	if (file) {
	    if (!(ifp = sopen(file,"r")))
		Error(file);
	    for (iseq = buf; fscanf(ifp, "%x", &d) == 1; *iseq++ = d);
	    islen = iseq - buf;
	    iseq = buf;
	    sclose(ifp);
	} else {
	    iseq = Minit;
	    if (Sync == 'f')
		iseq[SYNCLOC] = MPU_FSK_CLOCK;
	    else if (Sync == 'm')
		iseq[SYNCLOC] = MPU_MIDI_CLOCK;
	    else
		iseq[SYNCLOC] = MPU_INT_CLOCK;
	    islen = sizeof Minit;
	}
	for (i = 0; i < MAXMPUS; i++) {
	    if (Mpu[i].fd < 0)
		continue;
	    if (!file) {
		iseq[TBITSLOC] = trackbits[i];
		iseq[MODELOC] = Mpu[i].mode;
	    }
	    MpuSetTrack(Mpu[i].fd, MPU_CTR_COM);
/****/fprintf(stderr, "track=%d\n", MPU_CTR_COM);
	    mwrite(Mpu[i].fd, Mbuf, Mbp - Mbuf);
	    mwrite(Mpu[i].fd, iseq, islen);
	    if (MpuSetTrack(Mpu[i].fd, 0) == -1) {
/****/fprintf(stderr, "midiinit() MpuSetTrack() failed   ");
		perror(Mpu[i].name);
		return(-1);
	    }
/****/fprintf(stderr, "track=0\n");
	}
	return(0);
}

#define	P	fprintf
#define	E	stderr

syntax(i)
{
	P(E,"Usage: %s [files or stdin]\n", av0);
	P(E,"flags:\n");
	P(E,"--	  read stdin for data\n");
	P(E,"-a	  output ascii hex instead of binary (for debug)\n");
	P(E,"-C	  following sources contain MPU commands, not data.\n");
	P(E,"-cfile	  use contents of \"file\" to initialize MPU\n");
	P(E,"-D	  following sources contain data, not MPU commands.\n");
	P(E,"-d#	  following data sources are for /dev/mpu#\n");
	P(E,"-M[#[/#]] metronome on, with secondary accent\n");
	P(E,"-m[#[/#]] metronome on, no secondary accent\n");
	P(E,"-R	  following data sources are \"raw data\".\n");
	P(E,"-r	  record (writing the recording to stdout)\n");
	P(E,"-Si	  synchronize with internal clock (default)\n");
	P(E,"-Sf	  synchronize with FSK signal at TAPE IN jack\n");
	P(E,"-Sm	  synchronize with external MIDI clock\n");
	P(E,"-T	  following data sources are time-tagged.\n");
	P(E,"-t#	  set tempo to # (beats/minute, default=100)\n");
	P(E,"-w#	  wait for # seconds before starting\n");
	P(E,"-x	  enable recording system exclusive data\n");
	exit(i);
}

record(ofh)		/* copy any data from Midifh to ofh */
{
	char c;
	int n, fh;

	if (Record < 0)
	    return;
	if (iwait(0, 0)) {	/* input arrived from keyboard; quit */
	    getchar();
	    done();
	    /*NOTREACHED*/
	}
	fh = Mpu[Record].fd;
	if (iwait(fh, 0) && (n = read(fh, &c, 1)) > 0) {
	    if (Debug)
		printf("0x%02x\n", c);
	    else if (mwrite(ofh, &c, n) != n)
		perror("write");
	}
}

splay()		/* simultaneously play all files in "Fp", "Fd" & "Lm" */
{
	int n, in, dataleft;
	char s[512];
	struct inpstr *ip;
	struct mpustr *mp;

	do {
	    dataleft = 0;
	    for (in = 0; in < Ninp; in++) {
		ip = &Inp[in];
		if (ip->fd < 0)
		    continue;
		dataleft++;
		if (ip->lm) {
		    if (iwait(ip->fd, 0) > 0)
			livedata(ip);
		} else {
		    mp = &Mpu[ip->mpu];
		    MpuSetTrack(mp->fd, ip->trak);
/****/fprintf(stderr, "track=%d\n", ip->trak);
		    if ((n = read(ip->fd, s, sizeof s)) > 0) {
			if (mwrite(mp->fd, s, n) != n)
/****/{
/****/fprintf(stderr, "splay() write(%d, %x, %d) failed   ", mp->fd, s, n);
			    perror(mp->name);
/****/}
		    } else {
			sclose(ip->fp);
			ip->fp = (FILE *)0;
			ip->fd = -1;
			putTCIP(mp->fd);
		    }
		}
	    }
	    if (Record >= 0)
		record(fileno(stdout));
	} while (dataleft);
}

livedata(ip)		/* read & write out "live" data */
struct	inpstr *ip;
{
	register int c, n, ifh;
	u_char buf[64], *cp, *bep;		/* not big enough for SysEx */
	static int status;
	struct mpustr *mp;

	ifh = ip->fd;
	mp = &Mpu[ip->mpu];
	while (iwait(ifh, 0) && (n = read(ifh, buf, 1)) == 1) {
	    cp = buf;
	    c = *cp;
	    if (ip->dorc) {				/* raw commands */
		MpuSetTrack(mp->fd, ip->trak);
		cp++;
		if (iwait(ifh, 0))
		    cp += read(ifh, cp, sizeof buf - 1);
	    } else {					/* raw data */
		MpuSetTrack(mp->fd, MPU_CTR_COM);
		n = statproc(&status, c);
		if (n < 0) {				/* Sys Excl */
		    *cp++ = MPU_SEND_SYSTEM_MESSAGE;
		    *cp++ = c;				/* SX_CMD */
		    mwrite(mp->fd, buf, cp - buf);
		    for (cp = buf; read(ifh, cp, 1) >= 0; ) {
			c = *cp++;
			if (cp - buf >= sizeof buf) {
			    mwrite(mp->fd, buf, cp - buf);
			    cp = buf;
			}
			if (c == SX_EOB)
			    break;
		    }
		} else {
		    *cp++ = MPU_WANT_TO_SEND_DATA + ip->trak;
		    *cp++ = c;
		    for (bep = &buf[n+1]; cp < bep; cp += read(ifh,cp,bep-cp));
		}
	    }
	    if (cp > buf)
		mwrite(mp->fd, buf, cp - buf);
	}
	if (n == 0) {				/* EOF */
	    sclose(ip->fp);
	    ip->fp = (FILE *)0;
	    ip->fd = -1;
/****	    ioctl(mp->fd, MPU_IOC_PURGE, &ip->trak);	/****/
	}
}

mwrite(fh, buf, len)		/* write out midi data to fh */
u_char	*buf;
{
	register int i;

	if (Debug) {
	    fprintf(stderr, "mwrite(%d,0x%x,%d)", fh, buf, len);
	    for (i = 0; i < len; i++)
		fprintf(stderr, " %02x", buf[i] & 0xFF);
	    fprintf(stderr, "\n");
	} else
	    if ((i = write(fh, buf, len)) != len)
		perror("mixplay:mwrite");
	return(i);
}

void
interrupt()	/* user must ^C again to really quit, I think */
{
	int i;

	for (i = 0; i < MAXMPUS; i++)
	    if (Mpu[i].fd >= 0)
		ioctl(Mpu[i].fd, MPU_IOC_PURGE, 0);
	done();
}

iwait(f, timeout)
unsigned long timeout;  /* in seconds */
/*
 * Wait until 'f' is ready for reading, or 'timeout'.
 * Return '>=0' when 'f' is readable, '0' if timeout, '-1' on error.
 * Example: 'iwait(f,0)' polls a file descriptor
 * without blocking and returns true if it's readable;
 * 'iwait(0,0)' returns true when standard input is available.
 */
{
	int readfd;
	struct timeval t;

	readfd = (1 << f);
	t.tv_sec = timeout;
	t.tv_usec = 0;
	return(select(f + 1,  &readfd, (int *)0, (int *)0, &t));
}
