/*
**	ACCL -- Automated Chord Chart Lead Player
**	psl 1/89
*/
#include	<stdio.h>
#include	"midi.h"
#undef	d

#define	MAXCH		512
#define	NCHANS		16
#define	NKEYS		128
#define	NAMELEN		20
#define	GRPLEN		8
#define	BSDIR		"/u/psl/MIDI/BELLSCORE"
		/* nearest() flags */
#define	UPOK	1
#define	DNOK	2
#define	OLDOK	4
#define	UPONLY	UPOK
#define	DNONLY	DNOK
#define	NEW	(UPOK|DNOK)
#define	ANY	(UPOK|DNOK|OLDOK)
		/* time values (in MPU clocks) */
#define	BREVE		(MPU_CLOCK_PERIOD<<2)
#define	WHOLE		(MPU_CLOCK_PERIOD<<1)
#define	HALF		(MPU_CLOCK_PERIOD)
#define	QUARTER		(MPU_CLOCK_PERIOD>>1)
#define	EIGHTH		(MPU_CLOCK_PERIOD>>2)
#define	SIXTEENTH	(MPU_CLOCK_PERIOD>>3)
#define	THIRTYSECOND	(MPU_CLOCK_PERIOD>>4)
#define	PR(PCNT)	((rand()%100)<(PCNT))	/* "random" probability fcn */
#define	PR50		(rand()&010)		/* 50-50 fcn */
#define	PC(N)		(((N)+120)%12)		/* Pitch class (mod 12) */
#define	DBG		if(Debug)fprintf
#define	DBG2		if(Debug>1)fprintf

void	bebop(), grass(), boogi(), class(), march(), mozar(), seque(), swing();
void	toner();
char	*speel(), *strcopy(), oneof();

#define	DEFSTYLE	"class"
struct	stylstr	{		/* definitions of styles */
	char	*name;		/* style names (for arguments) */
	int	key;		/* standard chord key offset */
	void	(*comp)();	/* composition routine */
} S[]	= {
	{ "bebop", 0, bebop, },
	{ "grass", 7, grass, },
	{ "boogi", 0, boogi, },
	{ "class", 0, class, },
	{ "march", 0, march, },
	{ "mozar", 0, mozar, },
	{ "samba", 0, swing, },
	{ "seque", 0, seque, },
	{ "swing", 0, swing, },
	{ "toner", 0, toner, },
#ifndef	LINT
	{ 0, },
#endif
};

struct	chstr	{
	char	name[NAMELEN];	/* chord name */
	int	trans[12];	/* transpositions from standard chord */
	int	group;		/* standard chord group */
	int	ctone[12];	/* chord tones */
} Ch[MAXCH]	= {
	"-", { 0, }, 0, { 0, },	/* dummy to allow "-" to be used as a rest */
};

#define	T_VAMP	0
#define	T_TINY	1
#define	T_I_II	2
#define	T_I_IV	3
#define	T_I_V	4
#define	T_I_VI	5
#define	T_END	6

struct	typstr	{
	int	bars;
} Tpat[]	= {		/* transition pattern */
	2,			/* vamp */
	1,			/* tiny */
	1,			/* I-II */
	1,			/* I-IV */
	1,			/* I-V */
	1,			/* I-VI */
	2,			/* end */
};

int	Motion[12]	= {
	T_VAMP,	T_VAMP,	T_I_II,	T_VAMP,	T_I_II,	T_I_IV,
	T_VAMP,	T_I_V,	T_VAMP,	T_I_VI,	T_VAMP,	T_VAMP,
};
char	Part[128];			/* an arbitrary name? */
int	Cpq;				/* MPU clocks per quantum */
int	Ctones[4][10]	= {		/* chord notes in each chord group */
	{ 0, 4, 7, -1, },		/* (based on C) */
	{ 0, 4, 7, 10, -1, },
	{ 0, 4, 8, -1, },
	{ 0, 3, 6, 9, -1, },
};
int	Sctones[4][10]	= {		/* special chord notes for bebop */
	{ 0, 4, 7, -1, },		/* (based on C) */
	{ 0, 2, 4, 7, 9, 10, -1, },
	{ 0, 2, 4, 6, 8, 10, -1, },
	{ 0, 3, 6, 9, -1, },
};
int	Numchords	= 1;		/* "-" already defined */
int	Stones[12]	= {		/* derived from Key */
	1,0,1,0,1,1,0,1,0,1,0,1,
};
/* Data available to all styles */
int	Bars		= 0;		/* from args, to help plan */
int	Chan[16];			/* lead instrument channels */
int	Debug		= 0;		/* debugging output */
int	Energy		= 50;		/* how frenzied */
int	Key		= 0;		/* from args, to help plan */
long	Lstart		= 0;		/* time of current input line start */
int	Nchans		= 0;		/* how many output channels defined */
int	Predict		= 50;		/* how much repeating */
long	Pstart		= 0;		/* time of most recent "Part" line */
int	Seed		= 0;		/* random number seed */
int	Style;				/* index into S[] */
int	Vol[16]		= { 80, };	/* key velocities for output */

/* This struct contains all historical/continuity info */
/* it should contain enough to reproduce the output... */
typedef	struct	histr {
	int	k;			/* last note (pitch) played */
	int	d;			/* last pitch delta */
	int	s;			/* starting rand() seed */
} HIST;

HIST	Hist	= { 60, 0, 0, };

extern	MCMD	Mpu_nop;	/* MPU_NO_OP command */
extern	char	*key2name();

main(argc, argv)
char	*argv[];
{
	register int i, j, files;
	char *cp;
	long now;
	FILE *ifp;

	files = 0;
	now = 0L;
	srand(Seed = (int) time(0));
	for (Style = 0; S[Style].name; Style++)
	    if (strcmp(DEFSTYLE, S[Style].name) == 0)
		break;
	for (i = 1; i < argc; i++) {
	    if (argv[i][0] == '-') {
		switch (argv[i][1]) {
		case 'b':
		    Bars = atoi(&argv[i][2]);
		    break;
		case 'c':
		    Chan[Nchans] = atoi(&argv[i][2]) - 1;
		    for (cp = &argv[i][2]; *cp && *cp != '='; cp++);
		    if (*cp++ == '=') {
			j = atoi(cp);
			Vol[Nchans] = (j < 1? 1 : (j > 127? 127 : j));
		    } else
			Vol[Nchans] = 80;
		    Nchans++;
		    break;
		case 'd':
		    Debug++;
		    break;
		case 'e':
		    Energy = atoi(&argv[i][2]);
		    break;
		case 'k':
		    Key = pc2key(&argv[i][2]);
		    for (j = 12; --j >= 0; )
			Stones[PC(Key+j)] = (j!=1&&j!=3&&j!=6&&j!=8&&j!=10);
		    break;
		case 'p':
		    Predict = atoi(&argv[i][2]);
		    break;
		case 's':
		    if (j = atoi(&argv[i][2]))
			srand(Seed = j);
		    break;
		default:
		    fprintf(stderr, "Usage: %s [options] [files or stdin]\n",
		     argv[0]);
		    fprintf(stderr, "-b# predicts bar count (for planning)\n");
		    fprintf(stderr, "-c# adds an output channel [1..16]\n");
		    fprintf(stderr, "-c#=# also sets key velocity [1..127]\n");
		    fprintf(stderr, "-d turns on debugging info\n");
		    fprintf(stderr, "-e# sets solo \"energy\" [0..100]\n");
		    fprintf(stderr, "-kKEY sets solo key\n");
		    fprintf(stderr,
		     "-p# sets solo \"predictability\" [0..100]\n");
		    fprintf(stderr, "-s# sets the random number seed (#!=0)\n");
		    exit(2);
		}
	    } else {
		if ((ifp = fopen(argv[i], "r")) == NULL)
		    perror(argv[i]);
		else {
		    now = process(ifp, now);
		    fclose(ifp);
		}
		files++;
	    }
	}
	if (!files)
	    now = process(stdin, now);
	exit(0);
}

process(ifp, now)
FILE	*ifp;
long	now;
{
	char *bp, *cp;
	int ncn, ocn;		/* new & old chord numbers */
	long nlstart;		/* next line start time (MPU clocks) */
	long start;		/* when ocn started */
	char buf[512];
	HIST hsave;		/* saved to repeat a section */

DBG(stderr, "process(ifp, %d) seed=%d\n", now, Seed);
	srand(Hist.s = rand());
	hsave = Hist;
	ocn = -1;
	start = Lstart = Pstart = 0L;
	for (; fgets(buf, sizeof buf, ifp) != NULL; ) {
	    for (bp = buf; *bp && *bp <= ' '; bp++);
	    cp = speel(&bp, 0);
	    if (*cp == '\n' || (cp[0] == '#' && cp[1] == '\0'))
		;
	    else if (*cp == '#')
		procntl(cp, bp, now);
	    else {
		DBG(stderr, "input: %s %s", cp, bp);
		for (nlstart = now; *cp; now += Cpq) {
		    if (*cp == '/') {
			if ((ncn = ocn) == -1)
			    syntax("First line begins with '/'");
		    } else {
			if ((ncn = find(cp)) < 0)
			    syntax(cp);
		    }
		    if (ncn != ocn || nlstart == now) {
			if (now > start)
			    start = snip(start, now, ocn, ncn);
			ocn = ncn;
			Lstart = nlstart;
		    }
		    if (now == nlstart) {	/* first chord in line next */
			if (now && PR(Predict)) { /* repeat last solo line? */
DBG(stderr, "repeat last line of solo?\n");
			    Hist = hsave;
			    srand(Hist.s);
			} else {
DBG(stderr, "save params to allow repeat.\n");
			    srand(Hist.s = rand());
			    hsave = Hist;
			}
		    }
		    cp = speel(&bp, 0);
		}
	    }
	}
	snip(start, now, ocn, -1);
	return(now);
}

procntl(cp, bp, now)
char	*cp, *bp;
long	now;
{
	int cn, i, j, *ctp;
	FILE *fp;

	if (wrdcmp(cp, "#CHORD") == 0) {
	    cp = speel(&bp, 0);
	    if ((cn = find(cp)) == -1 && (cn = Numchords++) >= MAXCH)
		syntax("Too many chords defined");
	    strcopy(Ch[cn].name, cp);
	    cp = speel(&bp, 0);
	    Ch[cn].group = myatoi(cp);
	    for (i = 0; i < 12; i++) {
		Ch[cn].trans[i] = myatoi(bp);
		while (*bp && *bp++ != ',');
		Ch[cn].ctone[i] = 0;
	    }
	    for (ctp = Ctones[Ch[cn].group - 1]; *ctp >= 0; ctp++) {
		i = PC(*ctp + S[Style].key);
		Ch[cn].ctone[PC(i + Ch[cn].trans[i])]++;
	    }
	} else if (wrdcmp(cp, "#INCLUDE") == 0) {
	    if (*bp++ != '"')
		syntax("#INCLUDE must use quotes (\")");
	    for (cp = bp; *cp && *cp != '"'; cp++);
	    if (*cp != '"')
		syntax("#INCLUDE must use quotes (\")");
	    *cp = '\0';
	    if ((fp = fopen(bp, "r")) == NULL) {
		perror(bp);
		exit(1);
	    }
	    now = process(fp, now);
	    fclose(fp);
	} else if (wrdcmp(cp, "#PART") == 0) {
	    Pstart = now;
	    strcopy(Part, speel(&bp, 0));
	} else if (wrdcmp(cp, "#QUANTUM") == 0) {
	    if (wrdcmp(bp, "whole") == 0)
		i = 1;
	    else if (wrdcmp(bp, "half") == 0)
		i = 2;
	    else if (wrdcmp(bp, "quarter") == 0)
		i = 4;
	    else if (wrdcmp(bp, "eighth") == 0)
		i = 8;
	    else if (wrdcmp(bp, "sixteenth") == 0)
		i = 16;
	    else
		syntax("Unrecognized quantum arg");
	    Cpq = WHOLE / i;
	} else if (wrdcmp(cp, "#STYLE") == 0) {
	    cp = speel(&bp, 0);
	    for (i = 0; S[i].name && strcmp(S[i].name, cp); i++);
	    if (!S[i].name)
		syntax("Unrecognized Style");
	    Style = i;
	    if (strcmp(cp, "bebop") == 0)
		for (i = 4; --i >= 0; )
		    for (j = 0; Sctones[i][j] >= 0; j++)
			Ctones[i][j] = Sctones[i][j];
	} else {
DBG(stderr, "Unrecognized control: %s %s\n", cp, bp);
	    ;
	}
}

snip(start, stop, ocn, ncn)
long	start, stop;
{
	int nct, tlen;
	long ctime, vstop;

DBG(stderr, "snip(Lstart=%d Pstart=%d start=%d stop=%d ocn=%d %s ncn=%d %s)\n",
/****/  Lstart, Pstart, start, stop,
/****/  ocn, ocn>=0? Ch[ocn].name : "", ncn, ncn>=0? Ch[ncn].name : "");
	if (start >= stop)
	    return(stop);
	if (ncn == -1)
	    nct = T_END;
	else if (stop - start <= QUARTER)	/* tiny piece */
	    nct = T_TINY;
	else
	    nct = Motion[PC(Ch[ncn].trans[0] - Ch[ocn].trans[0])];
	if (Ch[ocn].name[0] == '-') {		/* rests are special */
	    if (nct == T_END) {
		Mpu_nop.when = stop;
		putmcmd(stdout, &Mpu_nop);
	    }
	    return(stop);
	}
	tlen = Tpat[nct].bars * WHOLE;
	vstop = Lstart + tlen * ((stop - Lstart - 1) / tlen);
DBG(stderr, " start=%d vstop=%d stop=%d\n", start, vstop, stop);
	ctime = start;
	if (ctime < vstop) {
/****/if(Debug) { int i;
fprintf(stderr,"%s(%d, %d, ocn(%d), ocn(%d))  ",
 S[Style].name, start, stop, ocn, ocn);
fprintf(stderr,"o:");for(i=0;i<12;fprintf(stderr,"%d",Ch[ocn].ctone[i++]));
fprintf(stderr,"  S:");for(i=0;i<12;fprintf(stderr,"%d",Stones[i++]));
fprintf(stderr, "\n");
/****/}
	    (*S[Style].comp)(ctime, vstop, ocn, ocn);
	    ctime = vstop;
	}
	if (ctime < stop) {
/****/if(Debug) { int i;
fprintf(stderr,"%s(%d, %d, ocn(%d), ncn(%d))  ",
 S[Style].name, start, stop, ocn, ncn);
fprintf(stderr,"o:");for(i=0;i<12;fprintf(stderr,"%d",Ch[ocn].ctone[i++]));
if (ncn >= 0)
 fprintf(stderr,"  n:");for(i=0;i<12;fprintf(stderr,"%d",Ch[ncn].ctone[i++]));
fprintf(stderr,"  S:");for(i=0;i<12;fprintf(stderr,"%d",Stones[i++]));
fprintf(stderr, "\n");
/****/}
	    (*S[Style].comp)(ctime, stop, ocn, ncn);
	}
	return(stop);
}

void
bebop(start, stop, ocn, ncn)
long	start, stop;
{
	register int i, dk, dur, dir;
	int prob, nk, nk2, nk3, needforce, *tp, *otones, *ntones;
	long lt, lt2;
	static int force;	/* key that needs to be played (not a rest) */

	needforce = 0;
	otones = Ch[ocn].ctone;
	ntones = Ch[ncn].ctone;
	if (!force && (Hist.k <= 36 || Hist.k >= 84)) {
DBG(stderr, "== Hist.k=%d Hist.d=%d\n", Hist.k, Hist.d);
	    Hist.k = 60;
	    Hist.d = 0;
	    if (stop - start > QUARTER)
		start = (start + stop) / 2;
	}
	if (ncn < 0) {					/* ending */
DBG(stderr, " ending note\n");
	    if (force) {
		nk = force;
DBG(stderr, " forced; nk=%d\n", nk);
	    } else
		nk = nearest(Hist.k + Hist.d / 2, otones, ANY);
	    lt = (stop - start) >= WHOLE? stop - HALF : stop;
	    noteonoff(nk, start, lt);			/* chord */
	    notehist(&Hist.k, &Hist.d);
	    return;
	}
	while ((dur = stop - start) > HALF && PR(Energy)) {
DBG(stderr, " recurse (>half)\n");
	    swing(start, start + HALF, ocn, ocn);
	    start += HALF;
	}
	if (dur > QUARTER && PR(Energy)) {
DBG(stderr, " recurse (>quarter)\n");
	    swing(start, start + QUARTER, ocn, ocn);
	    start += QUARTER;
	}
	dir = Hist.d? (Hist.d > 0 ? UPONLY : DNONLY) : NEW;
	dir = Hist.k <= 48? UPONLY : (Hist.k >= 72? DNONLY : dir);
	dur = stop - start;
	prob = rand() % (50 + Energy);
	if ((prob -= 20) < 0 || (force && PR(50))) {	/* 20, one long note */
DBG(stderr, " long note\n");
	    if (force) {
		nk = force;
DBG(stderr, " forced; nk=%d\n", nk);
	    } else {
		nk = nearest(Hist.k, otones, NEW);	/* look for pivot */
		if (ntones[PC(nk)]) {
DBG(stderr, "  pivot nk=%d\n", nk);
		} else {
		    nk2 = nearest(nk, ntones, ANY);
		    if (otones[PC(nk2)] > 0) {
			i = nk2 - Hist.k;
			if (Hist.d * i >= 0
			 && -2 < (i - Hist.d) && (i - Hist.d) < 2) {
DBG(stderr, "  pivot? nk=%d, nk2=%d\n", nk, nk2);
			    nk = nk2;			/* found one */
			}
		    }
		}
	    }
	    lt = stop - (dur >= HALF? EIGHTH : 0);
	    noteonoff(nk, start, lt);			/* chord */
	} else if ((prob -= 25) < 0) {		/* 45, two note split */
	    if (dur > QUARTER && PR(67)) {
		lt = (start + stop + stop) / 3;		/* 67% swung */
DBG(stderr, " swung two-way\n");
	    } else {
		lt = (start + stop) / 2;		/* 33% straight */
DBG(stderr, " straight two-way\n");
	    }
	    nk = force? force : nearest(Hist.k, Stones, dir);
	    nk2 = nearest(nk, ntones, dir);
	    dk = nk2 - nk;
DBG(stderr, "   nk=%d, nk2=%d\n", nk, nk2);
	    if (dk == 1 || dk == -1) {		/* nk2 is 1 step from nk */
		noteonoff(nk, start, lt);		/* scale */
		nk3 = nearest(nk2, Stones, dir);	/* one step beyond */
		noteonoff(Hist.k = nk3, lt, stop);	/* scale */
		needforce = nk2;			/* not critical */
	    } else if (dk == 2 || dk == -2) {	/* nk2 is 2 steps from nk */
		noteonoff(nk, start, lt);		/* scale */
		if (stop - lt > EIGHTH) {
		    nk3 = nearest(nk2, Stones, dir);	/* one step beyond */
		    noteonoff(Hist.k = nk3, lt, stop);	/* scale */
		} else
		    noteonoff(nk2 - dk / 2, lt, stop);	/* chromatic */
		needforce = nk2;
	    } else if (dk == 3 || dk == -3) {	/* nk2 is 3|4|5 steps */
		noteonoff(nk, start, lt);		/* scale */
		nk3 = nk2 - dk / 3;
		noteonoff(nk3, lt, stop);		/* whole step */
		if (nk3 != nk2)
		    needforce = nk2;			/* not critical */
	    } else if (dk && -5 <= dk && dk <= 5) {	/* nk2 is 4|5 steps */
		noteonoff(nk, start, lt);		/* scale */
		nk3 = nearest(nk, Stones, dir);
		noteonoff(nk3, lt, stop);		/* scale */
		if (nk3 != nk2)
		    needforce = nk2;			/* not critical */
	    } else {					/* who knows? */
		noteonoff(nk, start, lt);		/* scale */
		nk = nearest(nk, Stones, dir);
		noteonoff(nk, lt, stop);		/* scale */
	    }
	} else if ((prob -= 20) < 0) {		/* 65, three note split */
	    if (dur > QUARTER && PR(50)) {	/* 50% triplets */
		lt = (start + start + stop) / 3;
		lt2 = (start + stop + stop) / 3;
DBG(stderr, " triplet 3-way\n");
	    } else {				/* 50% half & quarters */
		lt = (start + stop) / 2;
		lt2 = (start + stop + stop + stop) / 4;
DBG(stderr, " 1/2, 1/4, 1/4 3-way\n");
	    }
	    nk = force? force : nearest(Hist.k, otones, dir);
	    noteonoff(nk, start, lt);			/* chord */
	    dk = (nk < (54 + rand() % 13))? UPONLY : DNONLY;
DBG(stderr, "   Hist.k=%d, Hist.d=%d, nk=%d, dk=%d\n", Hist.k, Hist.d, nk, dk);
	    if (PR((100 - Energy) / 3)) {		/* possible rest */
		nk = nearest(nk, Stones, dk);
		noteonoff(nk, lt, lt2);			/* scale */
	    }
	    nk2 = nearest(nk, ntones, dk);
	    dk = nk2 - nk;
	    if ((dk == -2 || dk == 2) && stop - lt2 < QUARTER)
		nk3 = nk2 - dk / 2;			/* chromatic */
	    else if ((dk == -3 || dk == 3) && stop - lt2 < QUARTER)
		nk3 = nk2 - dk / 3;			/* whole step */
	    else
		nk3 = nearest(nk, Stones, dk);		/* scale step */
	    noteonoff(nk3, lt2, stop);
	    if (nk3 != nk2)
		needforce = nk2;
	} else if ((prob -= 25) < 0 && dur >= EIGHTH) {	/* 90, 4 note split */
DBG(stderr, " even 4-way\n");
	    lt = (start + start + start + stop) / 4;	/* even split */
	    dk = (Hist.k < (56 + rand() % 9))? UPONLY : DNONLY;
	    tp = (48 < Hist.k && Hist.k < 72)? Stones : otones;
DBG(stderr, "   Hist.k=%d, Hist.d=%d, dk=%d\n", Hist.k, Hist.d, dk);
	    if (!force && PR(50 - Energy))		/* possible rest */
		nk = Hist.k;
	    else {
		nk = force? force : nearest(Hist.k, tp, dk);
		noteonoff(nk, start, lt);		/* scale or chord */
	    }
	    lt2 = (start + stop) / 2;
	    nk = nearest(nk, tp, dk);
	    noteonoff(nk, lt, lt2);			/* scale or chord */
	    lt = (start + stop + stop + stop) / 4;
	    if (!PR(50 - Energy)) {			/* possible rest */
		nk = nearest(nk, tp, dk);
		noteonoff(nk, lt2, lt);			/* scale or chord */
	    }
	    nk3 = nearest(nk, otones, NEW);
	    noteonoff(nk3, lt, stop);			/* chord */
	} else if ((prob -= 20) < 0 && dur > EIGHTH) {	/* 110, recurse */
	    lt = (start + stop) / 2;
	    if (force || PR(Energy)) {
DBG(stderr, " recurse (>eighth)\n");
		swing(start, lt, ocn, ocn);
	    } else {
DBG(stderr, " rest & recurse (>eighth)\n");
	    }
	    swing(lt, stop, ocn, ncn);
	} else if ((prob -= 20) < 0 && dur >= EIGHTH) {	/* 130, recurse */
	    lt = (start + stop) / 2;
	    if (force || PR(Energy)) {
DBG(stderr, " recurse (>=eighth)\n");
		swing(start, lt, ocn, ocn);
	    } else {
DBG(stderr, " rest & recurse (>=eighth)\n");
	    }
	    swing(lt, stop, ocn, ncn);
	} else if (force) {				/* play forced note */
DBG(stderr, " forced note\n");
	    noteonoff(force, start, stop);		/* forced (chord) */
	} else if (dur <= QUARTER) {			/* move toward C3 */
DBG(stderr, " move toward center\n");
	    dir = (Hist.k < 60)? UPONLY : DNONLY;
	    nk = nearest(Hist.k, Stones, dir);
	    noteonoff(nk, start, stop);			/* scale */
	} else {					/* try again */
DBG(stderr, " try again\n");
	    swing(start, stop, ocn, ncn);
	}
	force = needforce;
	notehist(&Hist.k, &Hist.d);
}

void
grass(start, stop, ocn, ncn)
long	start, stop;
{
	int *ntones;
	static int init;
	static int etones[12] = { -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, };

	if (!init++)
	    banjotune(Key, Stones, Chan[0]);
	ntones = (ncn == -1)? etones : Ch[ncn].ctone;
	banjoplay(start, stop, Ch[ocn].ctone, ntones, Vol[0], stdout);
}

/* BOOGI data */
#define	RST	0,999
#define	END	999,0,999
#define	CNT	0
#define	REL	999
int	Bw0[]	= {
	72,64,40, RST,80, RST,80, 64,64,CNT, 67,64,CNT, 69,64,CNT, 72,64,40,
	RST,80, 63,64,40, 64,64,40, 67,64,40, 69,64,40, END, };
int	Bw1[]	= {
	63,64,CNT, 66,64,80, 64,64,CNT, 67,64,40, 60,64,80,
	63,64,CNT, 66,64,40, 64,64,CNT, 67,64,80, 60,64,40, END, };
int	Bw2[]	= {
	72,80,CNT, 76,80,40, RST,80, 69,64,80, 67,64,40, 74,80,CNT, 77,80,40,
	RST,80, 69,64,80, 67,64,40, 72,80,CNT, 76,80,80, 67,64,40, 69,64,80,
	72,64,40, 75,80,CNT, 79,80,80, 74,64,CNT, 77,64,40, 72,64,CNT, 75,64,80,
	69,64,CNT, 72,64,40, END, };
int	Bw3[]	= {
	64,64,80, 72,80,40, RST,120, 72,72,CNT, 69,69,CNT, 67,67,CNT, 64,64,40,
	RST,200, 72,72,CNT, 69,69,CNT, 67,67,CNT, 64,64,40, RST,160, 72,72,CNT,
	69,69,CNT, 67,67,CNT, 64,64,40, RST,60, 72,72,20, 69,69,20, 67,67,20,
	63,80,120, END, };
int	Bw4[]	= {
	RST,80, 60,64,40, 63,64,CNT, 67,64,120, 65,64,CNT, 69,64,120,
	63,64,CNT, 67,64,120, END, };
int	Bw5[]	= {
	67,80,CNT, 75,80,240, RST,240, 64,64,CNT, 72,64,40, RST,200, RST,240,
	67,80,CNT, 76,80,80, 65,80,CNT, 74,80,40, 64,80,CNT, 72,80,80,
	60,80,CNT, 69,80,40, 58,80,CNT, 67,80,120, 57,80,CNT, 65,80,80,
	54,80,CNT, 63,80,40, 55,80,CNT, 64,80,240, RST,240, END, };
int	Bw6[]	= {
	82,64,40, 83,64,40, 84,64,40, END, };
int	Bw7[]	= {
	76,64,CNT, 79,64,CNT, 82,64,40, 72,64,40,
	76,64,CNT, 79,64,CNT, 82,64,40, 72,64,40, RST,80, END, };
int	Bw8[]	= {
	RST,80, 72,64,30, 75,48,CNT, 78,48,10, 76,64,CNT, 79,64,70,
	75,48,CNT, 78,48,10, 76,64,CNT, 79,64,40, RST,240, END, };
int	Bw9[]	= {
	60,64,CNT, 67,64,CNT, 70,64,240, RST,80, 60,64,40, 65,64,CNT, 69,64,80,
	60,64,CNT, 64,64,CNT, 67,64,40, 60,64,CNT, 64,64,CNT, 67,64,120, RST,80,
	60,64,40, 64,64,CNT, 67,64,120, 65,64,CNT, 69,64,120, END, };
typedef	struct	bwsstr	{
	int	*seq;		/* (key,vel,dur) tuples */
	int	len;		/* pattern length (MPU clocks) */
	int	tpr;		/* % chance that trn is used */
	int	sip;		/* % chance the sequence gets interrupted */
	int	trn;		/* REL => relative trans, else absolute */
} BWSSTR;
BWSSTR Bws[]	= {
	Bw0,	480,	0, 0, 0,
	Bw1,	360,	0, 10, 0,
	Bw2,	960,	0, 0, 0,
	Bw3,	960,	0, 0, 0,
	Bw4,	480,	0, 10, 0,
	Bw5,	1920,	0, 0, 0,
	Bw6,	120,	60, 10, REL,
	Bw7,	240,	50, 20, REL,
	Bw8,	480,	99, 20, 3,
	Bw9,	960,	67, 0, REL,
};
#define	NBWS	((sizeof Bws) / (sizeof Bws[0]))
int	Bwle[]	= {		/* 2-bar ending */
	63,64,10, 64,64,CNT, 67,64,CNT, 70,64,110,
	63,64,CNT, 65,64,CNT, 69,64,40, RST,80, 63,64,10,
	64,64,CNT, 67,64,CNT, 70,64,110, 63,64,CNT, 65,64,CNT, 69,64,40,
	RST,70, 63,64,CNT, 66,64,10,
	60,64,CNT, 64,64,CNT, 67,64,CNT, 70,64,40, RST,40,
	60,64,CNT, 64,64,CNT, 67,64,CNT, 70,64,40, RST,40,
	60,64,CNT, 64,64,CNT, 67,64,CNT, 70,64,40, RST,40,
	60,64,CNT, 64,64,CNT, 67,64,CNT, 70,64,120, RST,120, END, };
int	Bwse[]	= {		/* 1-bar ending */
	60,64,CNT, 64,64,CNT, 67,64,CNT, 70,64,40, RST,40,
	60,64,CNT, 64,64,CNT, 67,64,CNT, 70,64,40, RST,40,
	60,64,CNT, 64,64,CNT, 67,64,CNT, 70,64,40, RST,40,
	60,64,CNT, 64,64,CNT, 67,64,CNT, 70,64,120, RST,120, END, };
int	Bwte[]	= {		/* tiny? ending */
	60,64,CNT, 64,64,CNT, 67,64,CNT, 70,64,40, 48,80,40,
	60,64,CNT, 64,64,CNT, 67,64,CNT, 70,64,40, 48,80,40,
	60,64,CNT, 64,64,CNT, 67,64,CNT, 70,64,40, 48,80,40,
	60,64,CNT, 64,64,CNT, 67,64,CNT, 70,64,40, 48,80,40,
	60,64,CNT, 64,64,CNT, 67,64,CNT, 70,64,40, 48,80,40,
	60,64,CNT, 64,64,CNT, 67,64,CNT, 70,64,40, 48,80,40, END, };
BWSSTR Bwe[]	= {
	Bwle,	960,	100, 0, REL,
	Bwse,	480,	100, 0, REL,
	Bwte,	480,	100, 0, REL,
};

void
boogi(start, stop, ocn, ncn)
long	start, stop;
{
	u_char ubuf[4];
	int otones[12], keyson[16], nkeyson, k, v, d, trn;
	MCMD m;
	static int tones7[12] = {1,0,1,0,1,1,0,1,0,1,1,0,};
	static int *dp, lastseq = 1, curseq = 1; /* (won't start with 1) */
	static long lsstart = -1L;
	static BWSSTR *sp;

	m.cmd = ubuf;
	m.len = 3;
	if (ncn == -1) {		/* ending */
	    m.cmd[0] = CH_KEY_ON | Chan[0];
	    m.when = start;
	    m.cmd[2] = Vol[0];
	    if (stop - start == BREVE)
		sp = &Bwe[0];
	    else if (stop - start == WHOLE)
		sp = &Bwe[1];
	    else
		sp = &Bwe[2];
	    dp = sp->seq;
	    lsstart = Pstart;
	}
	if (PC(Ch[ocn].trans[0]) == Key)		/* root == key */
	    for (k = 12; --k >= 0; otones[k] = 1);	/* anything goes */
	else {
	    for (k = 12; --k >= 0; otones[k] = 0);
	    for (k = 12; --k >= 0; )
		otones[PC(k + Ch[ocn].trans[k])] = tones7[k];
	}
	if (sp) {
	    trn = PR(sp->tpr)? sp->trn : 0;
	    if (trn == REL)
		trn = Ch[ocn].trans[0];
	}
	nkeyson = 0;
	if (dp && start + WHOLE == Pstart)	/* short turnaround */
	    dp = (int *) 0;			/* force new riff */
	while (start < stop) {
	    if (dp) {				/* a sequence exists */
		if (dp[1] == 0) {		/* end of riff sequence */
		    dp = sp->seq;			/* loop back */
		    if ((start % BREVE) == 0 && PR(sp->sip))	/* or switch */
			dp = (int *) 0;		/* force new riff */
		}
		if (start + BREVE == Pstart)	/* turnaround */
		    dp = (int *) 0;			/* force new riff */
		else if (start == Pstart && lsstart < Pstart)	/* short part */
		    dp = (int *) 0;			/* force new riff */
	    }
	    while (!dp) {			/* pick a new riff sequence */
		k = rand() % NBWS;
DBG(stderr, "rand() returns %d\n", k);
		if (k == curseq || k == lastseq)
		    continue;
		sp = &Bws[k];
		if ((start - Pstart) % sp->len)
		    continue;
		lastseq = curseq;
		curseq = k;
		dp = sp->seq;
		lsstart = start;
		trn = PR(sp->tpr)? sp->trn : 0;
		if (trn == REL)
		    trn = Ch[ocn].trans[0];
DBG(stderr, "Use sequence %d\n", curseq);
	    }
DBG(stderr, "dp=%x,", dp);
	    k = *dp++;
	    v = *dp++;
	    d = *dp++;
DBG(stderr, " k=%d(%s), v=%d, d=%d\n", k, key2name(k), v, d);
	    if (!k)		/* rest */
		start += d;
	    else {
		k = nearest(k + trn, otones, ANY);
		m.when = start;
		m.cmd[0] = CH_KEY_ON | Chan[0];
		m.cmd[1] = k;
		m.cmd[2] = Vol[0];
		putmcmd(stdout, &m);
		keyson[nkeyson++] = k;
DBG(stderr, " %4d: %d key-on (%s)\n", start, k, key2name(k));
		if (d) {
		    start += d;
		    m.when = start;
		    m.cmd[2] = 0;
		    do {
			m.cmd[1] = keyson[--nkeyson];
			putmcmd(stdout, &m);
DBG(stderr, " %4d: %d key-off (%s)\n",
 start, m.cmd[1], key2name((int)m.cmd[1]));
		    } while (nkeyson > 0);
		}
	    }
	}
}

/* CLASS data */
char	Rp[][64]	= {	/* melodic rhythm patterns (in 32nds) */
	/*		 |   &   |   &   |   &   |   &   | */
	/* A */		"h...............q.......q.......",
	/* B */		"q.......E.....s.q.......q.......",
	/* C */		"s.s.s.s.e...e...q.......e...e...",
	/* D */		"q.......E.....s.q.......r.......",
	/* E */		"Q...........e...h...............",
	/* F */		"h...............r.......q.......",
	/* G */		"w...............................",
};
char	Arp[64]	= {	/* alternate rhythm pattern for march endings */
	/* G */		"h...............r.......s.r.....",
};

void
class(start, stop, ocn, ncn)
long	start, stop;
{
	register int i, dk, dur;
	char *pp, p, m;
	int nn, nk, begbar, endbar, beg32, udopt, *otones;
	long st;
	static char plan[512];
	static int planlen, mcnt[7], bbegan = 0, bnomove = 0, bdir, corn;

	otones = Ch[ocn].ctone;
	if (planlen == 0) {			/* make a rhythmic plan */
	    planlen = Bars? Bars : 32;		/* fake it if no bars arg */
	    for (pp = plan; dk = planlen - (pp - plan); ) {
		if (dk >= 12) {
		    *pp++ = p = oneof("ABC", 'G');
		    *pp++ = p = oneof("BCE", p);
		    *pp++ = p = oneof("ABC", p);
		    *pp++ = p = oneof("EF", p);
		    *pp++ = p = oneof("ABCF", p);
		    *pp++ = p = oneof("ACEF", p);
		    *pp++ = p = oneof("ABC", p);
		    *pp++ = p = oneof("AF", p);
		    *pp++ = p = oneof("ABC", p);
		    *pp++ = p = oneof("ABC", p);
		    *pp++ = p = oneof("BCC", p);
		    *pp++ = 'G';
		} else if (dk >= 8) {
		    *pp++ = p = oneof("AB", 'G');
		    *pp++ = p = oneof("ABE", p);
		    *pp++ = p = oneof("BCE", p);
		    *pp++ = p = oneof("ADEF", p);
		    *pp++ = p = oneof("BCE", p);
		    *pp++ = p = oneof("ABC", p);
		    *pp++ = p = oneof("ABC", p);
		    *pp++ = 'G';
		} else if (dk >= 4) {
		    *pp++ = p = 'A';
		    *pp++ = p = oneof("ABE", p);
		    *pp++ = p = oneof("ABC", p);
		    *pp++ = 'G';
		} else if (dk >= 2) {
		    *pp++ = 'B';
		    *pp++ = 'D';
		} else {
		    *pp++ = 'D';
		}
	    }
	    *pp = '\0';
DBG(stderr, " plan: %s\n", plan);
	}
	begbar = start / WHOLE;
	for (endbar = (stop - 1) / WHOLE; begbar < endbar; start = st)
	    class(start, st = (++begbar) * WHOLE, ocn, ocn);
	while (begbar >= planlen)		/* ran past end? */
	    begbar -= planlen;
	m = plan[begbar];
	mcnt[m - 'A']++;
	if (ncn < 0 && begbar != planlen - 1) {		/* ending */
DBG(stderr, " unforeseen ending, begbar=%d, planlen=%d\n", begbar, planlen);
	    m = 'G';
	}
	for (i = planlen; i > 16; i >>= 1);	/* determine cycle length */
	udopt = (begbar % i) < (i >> 1)? UPOK : DNOK;
	beg32 = (start % WHOLE) / THIRTYSECOND;
	nn = 0;
DBG(stderr, " PLAN CHAR: %c, udopt=%d, beg32=%d\n", m, udopt, beg32);
	for (pp = &Rp[m - 'A'][beg32]; start < stop; start += THIRTYSECOND) {
	    if ((p = *pp++) == 'w')
		dur = WHOLE;
	    else if (p == 'h')
		dur = HALF;
	    else if (p == 'Q')
		dur = QUARTER + EIGHTH;
	    else if (p == 'q')
		dur = QUARTER;
	    else if (p == 'E')
		dur = EIGHTH + SIXTEENTH;
	    else if (p == 'e')
		dur = EIGHTH;
	    else if (p == 's')
		dur = SIXTEENTH;
	    else {
		if (p == 'r' && Hist.k >= 72)
		    Hist.k -= 12;
		continue;
	    }
DBG(stderr, "  PATTERN CHAR: %c\n", *pp);
	    dur = (dur > stop - start? (stop - start) : dur);
	    nn++;
	    if (Hist.k <= 55) {
DBG(stderr, "== Hist.k=%d\n", Hist.k);
		udopt = UPONLY;
		while (Hist.k <= 48)
		    Hist.k += 12;
	    }
	    if (Hist.k >= 84) {
DBG(stderr, "== Hist.k=%d\n", Hist.k);
		udopt = DNONLY;
		while (Hist.k >= 90)
		    Hist.k -= 12;
	    }
	    if (m == 'A') {
		if (mcnt[m - 'A'] & 1) {
		    i = PR(33)? UPOK | OLDOK : UPOK;
		    nk = nearest(Hist.k, otones, i);
		} else {
		    if (nn == 1)
			nk = nearest(Hist.k, otones, ANY);
		    else if (nn == 2) {
			i = PR(33)? UPOK | udopt : udopt;
			nk = nearest(Hist.k, Stones, i);
		    } else
			nk = nearest(Hist.k, Stones, udopt);
		}
	    } else if (m == 'B') {
		if (start == 0)		/* beginning of piece */
		    bbegan = 1;
		if (nn == 1) {
		    nk = nearest(Hist.k, otones, ANY);
		    if (bbegan || PR50)
			bnomove = 1;
		    else
			bdir = PR50? UPONLY : udopt;
		} else if (bbegan || bnomove || nn < 4)
		    nk = Hist.k;
		else
		    nk = nearest(Hist.k, otones, bdir);
	    } else if (m == 'C') {
		if (nn == 1)
		    corn = PR(25);
		if (corn) {
		    if (nn == 1)	/* make ornament turn on chord tone */
			Hist.k = nearest(Hist.k, otones, ANY);
		    if (nn == 1 || nn == 4)
			nk = nearest(Hist.k, Stones, UPONLY);
		    else if (nn == 2 || nn == 3)
			nk = nearest(Hist.k, Stones, DNONLY);
		    else if (nn == 5) {
			nk = nearest(Hist.k, otones, udopt);
			if (PR50)
			    nk = nearest(nk, otones, udopt);
			else if (PR50) {
			    nk = nearest(nk, otones, udopt);
			    nk = nearest(nk, otones, udopt);
			}
		    } else if (nn == 7) {
			if (PR50)
			    nk = Hist.k;
			else
			    nk = nearest(Hist.k, otones, udopt ^ (UPOK|DNOK));
		    } else if (nn == 6 || nn == 8 || nn == 9)
			nk = Hist.k;
		} else {
		    if (nn == 1)
			nk = nearest(Hist.k, otones, ANY);
		    else if (nn == 2 || nn == 5 || nn == 8)
			nk = nearest(Hist.k, Stones, UPONLY);
		    else if (nn == 3 || nn == 4)
			nk = nearest(Hist.k, Stones, DNONLY);
		    else if (nn == 6)
			nk = Hist.k;
		    else if (nn == 7) {
			if (PR(75)) {
			    nk = nearest(Hist.k, otones, DNONLY);
			    nk = nearest(nk, otones, DNONLY);
			} else
			    nk = Hist.k - 12;
		    } else if (nn == 9)
			nk = nearest(Hist.k, otones, UPONLY);
		}
	    } else if (m == 'D') {
		if (begbar == planlen - 1)
		    nk = 60 + Key;
		else {
		    if (nn == 1)
			nk = nearest(Hist.k, otones, ANY);
		    if (nn == 2)
			nk = Hist.k;
		    else
			nk = nearest(Hist.k, otones, udopt);
		}
	    } else if (m == 'E') {
		if (begbar == planlen - 1)
		    nk = 60 + Key;
		else if (mcnt[m - 'A'] & 1) {
		    if (begbar == 0)
			nk = nearest(Hist.k, otones, UPONLY);
		    else
			nk = nearest(Hist.k, otones, udopt);
		} else {
		    if (nn == 2)
			nk = nearest(Hist.k, Stones, udopt);
		    else
			nk = nearest(Hist.k, otones, udopt);
		}
	    } else if (m == 'F') {
		if (begbar == 0)
		    nk = nearest(Hist.k, otones, UPONLY);
		else
		    nk = nearest(Hist.k, otones, udopt);
	    } else if (m == 'G') {
		if (PC(Hist.k - Key - 3) < 6)
		    nk = Hist.k + 6 - PC(Hist.k - Key + 2);
		else
		    nk = Hist.k + 6 - PC(Hist.k - Key - 6);
		if (otones[PC(nk)] <= 0) {		/* not a I chord */
		    nk = nearest(Hist.k, otones, udopt);
		    if (otones[PC(nk)] <= 0)
			nk = nearest(Hist.k, otones, ANY);
		}
	    } else
		fprintf(stderr, "motive rhythm character is '%c'?\n", m);
/****/if (nk < 20 || nk > 120) fprintf(stderr, "nk=%d!\n", nk), exit(1);
	    if (PC(Hist.k + 1) == PC(Key)
	     && nk != Hist.k + 1
	     && otones[PC(Key)] > 0)
		nk = Hist.k + 1;
	    noteonoff(Hist.k = nk, start, start + dur);
	}
}

char
oneof(str, avoid)
char	*str, avoid;
{
	char c;
	int n;

	n = strlen(str);
	c = str[rand() % n];
	if (c == avoid)
	    c = str[rand() % n];
	return(c);
}

void
march(start, stop, ocn, ncn)
long	start, stop;
{
	register int i;

	if (ncn < 0)				/* ending coming up */
	    for (i = 64; --i >= 0; Rp['G'-'A'][i] = Arp[i]);
	class(start, stop, ocn, ncn);
}

void
mozar(start, stop, ocn, ncn)
long	start, stop;
{
	register int i, bars, s;
	char buf[1024], *bp;

	if (ncn >= 0)		/* wait until last call */
	    return;
	mozinit(Key);
	start = 0L;
	bars = (stop - start) / 360;	/* waltz time */
	i = (stop - start) % 360;	/* partial bar */
	if (i % 120) {			/* partial beat */
	    noteonoff(60 + Key, start, start + i % 120);
	    i -= (i % 120);
	}
	if (i)
	    noteonoff(60 + Key, start, start + i);
	if (bars <= 0)
	    return;
	if (i = (bars % 4)) {
	    if (bars < 4)
		for (bp = buf, s = 16 - 2 * i; s < 16; s++)
		    bp += mozgen(bp, s, rand() % 6 + rand() % 6, 1, Chan[0]);
	    else
		for (bp = buf, s = 0; s < 2 * i; s++)
		    bp += mozgen(bp, s, rand() % 6 + rand() % 6, 1, Chan[0]);
	    write(1, buf, bp - buf);
	    bars -= i;
	}
	if (bars <= 0)
	    return;
	for (bars -= 4; bars > 0; bars -= 4) {
	    for (bp = buf, s = 0; s < 8; s++)
		bp += mozgen(bp, s, rand() % 6 + rand() % 6, 1, Chan[0]);
	    write(1, buf, bp - buf);
	    if ((bars -= 4) <= 0)
		break;
	    write(1, buf, bp - buf);
	    if ((bars -= 4) <= 0)
		break;
	    for (bp = buf, s = 8; s < 16; s++)
		bp += mozgen(bp, s, rand() % 6 + rand() % 6, 1, Chan[0]);
	    write(1, buf, bp - buf);
	}
	for (bp = buf, s = 8; s < 16; s++)
	    bp += mozgen(bp, s, rand() % 6 + rand() % 6, 1, Chan[0]);
	write(1, buf, bp - buf);
}

void
seque(start, stop, ocn, ncn)
long	start, stop;
{
	char buf[1024], seqfile[64];
	int sixteenths, plen, reps, extra, ifh, i, seed;
	long end;

	if (ncn >= 0)		/* wait until last call */
	    return;
	start = 0L;
	end = WHOLE * ((stop - HALF) / WHOLE);	/* plan for chord at end */
	sixteenths = (end - start) / SIXTEENTH;
DBG(stderr, "start=%d end=%d stop=%d, sixteenths=%d\n",
 start, end, stop, sixteenths);
	plen = 32;
	reps = (sixteenths + plen - 1) / plen;
	extra = sixteenths - reps * plen;
DBG(stderr, "plen=%d, reps=%d, extra=%d\n", plen, reps, extra);
	sprintf(seqfile, "/tmp/accl.seq%d", getpid());
	sprintf(buf, "pseq -iC3 -cE4 -l%d -s%d >%s.b", plen, Seed, seqfile);
	sys(buf);
	sprintf(buf,
	 "%s %d=1-16 <%s.b | %s %d | %s %d | %s -f%g | %s -h%g >%s",
	 "chmap", Chan[0] + 1, seqfile,
	 "transpose", Key - 12,
	 "rpt", reps,
	 "notedur", 2.0,
	 "bars", (float) (end - start) / WHOLE,
	 seqfile);
	sys(buf);
	sprintf(buf,
	 "%s -h%g <%s.b | %s %d=1-16 | %s %d | %s -f%g | %s -h%g >>%s",
	 "bars", 0.5, seqfile,
	 "chmap", Chan[0] + 1,
	 "transpose", Key - 12,
	 "notedur", 16.0,
	 "bars", (float) (stop - end) / WHOLE,
	 seqfile);
	sys(buf);
	if ((ifh = open(seqfile, 0)) < 0) {
	    perror(seqfile);
	    return;
	}
	while ((i = read(ifh, buf, sizeof buf)) > 0)
	    write(1, buf, i);
	unlink(seqfile);
}

void
swing(start, stop, ocn, ncn)
long	start, stop;
{
	register int i, dk, dur, dir;
	int nk, nk2, nk3, needforce, *tp, *otones, *ntones;
	long lt, lt2;
	static int force;	/* key that needs to be played (not a rest) */

	needforce = 0;
	otones = Ch[ocn].ctone;
	ntones = Ch[ncn].ctone;
	if (!force && (Hist.k <= 36 || Hist.k >= 84)) {
DBG(stderr, "== Hist.k=%d Hist.d=%d\n", Hist.k, Hist.d);
	    Hist.k = 60;
	    Hist.d = 0;
	    if (stop - start > QUARTER)
		start = (start + stop) / 2;
	}
	if (ncn < 0) {					/* ending */
DBG(stderr, " ending note\n");
	    if (force) {
		nk = force;
DBG(stderr, " forced; nk=%d\n", nk);
	    } else
		nk = nearest(Hist.k + Hist.d / 2, otones, ANY);
	    lt = (stop - start) >= WHOLE? stop - HALF : stop;
	    noteonoff(nk, start, lt);			/* chord */
	    notehist(&Hist.k, &Hist.d);
	    return;
	}
	while ((dur = stop - start) > HALF && PR(Energy)) {
DBG(stderr, " recurse (>half)\n");
	    swing(start, start + HALF, ocn, ocn);
	    start += HALF;
	}
	if (dur > QUARTER && PR(Energy)) {
DBG(stderr, " recurse (>quarter)\n");
	    swing(start, start + QUARTER, ocn, ocn);
	    start += QUARTER;
	}
	dir = Hist.d? (Hist.d > 0 ? UPONLY : DNONLY) : NEW;
	dir = Hist.k <= 48? UPONLY : (Hist.k >= 72? DNONLY : dir);
	dur = stop - start;
	i = rand() % (50 + Energy);
	if ((i -= 25) < 0 || (force && PR(50))) {	/* one long note */
DBG(stderr, " long note\n");
	    if (force) {
		nk = force;
DBG(stderr, " forced; nk=%d\n", nk);
	    } else {
		nk = nearest(Hist.k, otones, NEW);	/* look for pivot */
		if (ntones[PC(nk)]) {
DBG(stderr, "  pivot nk=%d\n", nk);
		} else {
		    nk2 = nearest(nk, ntones, ANY);
		    if (otones[PC(nk2)] > 0) {
			i = nk2 - Hist.k;
			if (Hist.d * i >= 0
			 && -2 < (i - Hist.d) && (i - Hist.d) < 2) {
DBG(stderr, "  pivot? nk=%d, nk2=%d\n", nk, nk2);
			    nk = nk2;			/* found one */
			}
		    }
		}
	    }
	    lt = stop - (dur >= HALF? EIGHTH : 0);
	    noteonoff(nk, start, lt);			/* chord */
	} else if ((i -= 25) < 0) {		/* two note split */
	    if (PR(67)) {
		lt = (start + stop + stop) / 3;		/* 67% swung */
DBG(stderr, " swung two-way\n");
	    } else {
		lt = (start + stop) / 2;		/* 33% straight */
DBG(stderr, " straight two-way\n");
	    }
	    nk = force? force : nearest(Hist.k, Stones, dir);
	    nk2 = nearest(nk, ntones, dir);
	    dk = nk2 - nk;
DBG(stderr, "   nk=%d, nk2=%d\n", nk, nk2);
	    if (dk == 1 || dk == -1) {		/* nk2 is 1 step from nk */
		noteonoff(nk, start, lt);		/* scale */
		nk3 = nearest(nk2, Stones, dir);	/* one step beyond */
		noteonoff(Hist.k = nk3, lt, stop);	/* scale */
		needforce = nk2;			/* not critical */
	    } else if (dk == 2 || dk == -2) {	/* nk2 is 2 steps from nk */
		noteonoff(nk, start, lt);		/* scale */
		if (stop - lt > EIGHTH) {
		    nk3 = nearest(nk2, Stones, dir);	/* one step beyond */
		    noteonoff(Hist.k = nk3, lt, stop);	/* scale */
		} else
		    noteonoff(nk2 - dk / 2, lt, stop);	/* chromatic */
		needforce = nk2;
	    } else if (dk && -5 <= dk && dk <= 5) {	/* nk2 is 3|4|5 steps */
		noteonoff(nk, start, lt);		/* scale */
		nk3 = nearest(nk, Stones, dir);
		noteonoff(nk3, lt, stop);		/* scale */
		if (nk3 != nk2)
		    needforce = nk2;			/* not critical */
	    } else {					/* who knows? */
		noteonoff(nk, start, lt);		/* scale */
		nk = nearest(nk, otones, dir);
		noteonoff(nk, lt, stop);		/* chord */
	    }
	} else if ((i -= 25) < 0) {		/* three note split */
	    if (PR(50)) {			/* 50% triplets */
		lt = (start + start + stop) / 3;
		lt2 = (start + stop + stop) / 3;
DBG(stderr, " triplet 3-way\n");
	    } else {				/* 50% half & quarters */
		lt = (start + stop) / 2;
		lt2 = (start + stop + stop + stop) / 4;
DBG(stderr, " 1/2, 1/4, 1/4 3-way\n");
	    }
	    nk = force? force : nearest(Hist.k, otones, dir);
	    noteonoff(nk, start, lt);			/* chord */
	    dk = (nk < (54 + rand() % 13))? UPONLY : DNONLY;
DBG(stderr, "   Hist.k=%d, Hist.d=%d, nk=%d, dk=%d\n", Hist.k, Hist.d, nk, dk);
	    if (PR((100 - Energy) / 3)) {		/* possible rest */
		nk = nearest(nk, Stones, dk);
		noteonoff(nk, lt, lt2);			/* scale */
	    }
	    nk2 = nearest(nk, ntones, dk);
	    dk = nk2 - nk;
	    if ((dk == -2 || dk == 2) && stop - lt2 < QUARTER)
		nk3 = nk2 - dk / 2;			/* chromatic */
	    else
		nk3 = nearest(nk, Stones, dk);		/* scale step */
	    noteonoff(nk3, lt2, stop);
	    if (nk3 != nk2)
		needforce = nk2;
	} else if ((i -= 25) < 0 && dur >= EIGHTH) {	/* four note split */
DBG(stderr, " even 4-way\n");
	    lt = (start + start + start + stop) / 4;	/* even split */
	    dk = (Hist.k < (56 + rand() % 9))? UPONLY : DNONLY;
	    tp = (53 < Hist.k && Hist.k < 67)? Stones : otones;
DBG(stderr, "   Hist.k=%d, Hist.d=%d, dk=%d\n", Hist.k, Hist.d, dk);
	    if (!force && PR(50 - Energy))		/* possible rest */
		nk = Hist.k;
	    else {
		nk = force? force : nearest(Hist.k, tp, dk);
		noteonoff(nk, start, lt);		/* scale or chord */
	    }
	    lt2 = (start + stop) / 2;
	    nk = nearest(nk, tp, dk);
	    noteonoff(nk, lt, lt2);			/* scale or chord */
	    lt = (start + stop + stop + stop) / 4;
	    if (!PR(50 - Energy)) {			/* possible rest */
		nk = nearest(nk, tp, dk);
		noteonoff(nk, lt2, lt);			/* scale or chord */
	    }
	    nk3 = nearest(nk, otones, NEW);
	    noteonoff(nk3, lt, stop);			/* chord */
	} else if ((i -= 10) < 0 && dur >= EIGHTH) {	/* recurse */
	    lt = (start + stop) / 2;
	    if (force || PR(Energy)) {
DBG(stderr, " recurse (>=eighth)\n");
		swing(start, lt, ocn, ocn);
	    } else {
DBG(stderr, " rest & recurse (>eighth)\n");
	    }
	    swing(lt, stop, ocn, ncn);
	} else if (force) {				/* play forced note */
DBG(stderr, " forced note\n");
	    noteonoff(force, start, stop);		/* forced (chord) */
	} else {					/* move toward C3 */
DBG(stderr, " move toward center\n");
	    dir = (Hist.k < 60)? UPONLY : DNONLY;
	    nk = nearest(Hist.k, otones, dir);
	    noteonoff(nk, start, stop);			/* chord */
	}
	force = needforce;
	notehist(&Hist.k, &Hist.d);
}

void
toner(start, stop, ocn, ncn)
long	start, stop;
{
	char buf[1024], seqfile[64];
	int sixteenths, plen, reps, extra, ifh, i, seed;
	long end;

	if (ncn >= 0)		/* wait until last call */
	    return;
	start = 0L;
	end = WHOLE * ((stop - HALF) / WHOLE);	/* plan for chord at end */
	sixteenths = (end - start) / SIXTEENTH;
DBG(stderr, "start=%d end=%d stop=%d, sixteenths=%d\n",
 start, end, stop, sixteenths);
	plen = 12;
	reps = (sixteenths + plen - 1) / plen;
	extra = sixteenths - reps * plen;
DBG(stderr, "plen=%d, reps=%d, extra=%d\n", plen, reps, extra);
	sprintf(seqfile, "/tmp/accl.seq%d", getpid());
	sprintf(buf, "tonerow -kC3 -s%d >%s.b", Seed, seqfile);
	sys(buf);
	sprintf(buf,
	 "%s %d=1-16 <%s.b | %s %d | %s %d | %s -f%g | %s -h%g >%s",
	 "chmap", Chan[0] + 1, seqfile,
	 "transpose", Key - 12,
	 "rpt", reps,
	 "notedur", 2.0,
	 "bars", (float) (end - start) / WHOLE,
	 seqfile);
	sys(buf);
	sprintf(buf,
	 "%s -h%g <%s.b | %s %d=1-16 | %s %d | %s -f%g | %s -h%g >>%s",
	 "bars", 0.5, seqfile,
	 "chmap", Chan[0] + 1,
	 "transpose", Key - 12,
	 "notedur", 16.0,
	 "bars", (float) (stop - end) / WHOLE,
	 seqfile);
	sys(buf);
	if ((ifh = open(seqfile, 0)) < 0) {
	    perror(seqfile);
	    return;
	}
	while ((i = read(ifh, buf, sizeof buf)) > 0)
	    write(1, buf, i);
	unlink(seqfile);
}

nearest(k, tones, flg)		/* find note nearest k in tones */
int	tones[12];
{
	register int i, j;

	for (i = (flg & OLDOK)? 0 : 1; i < 7; i++)
	    if (((flg & UPOK) && tones[PC(j = k + i)])
	     || ((flg & DNOK) && tones[PC(j = k - i)]))
		break;
	if (i < 7)
	    k = j;
	if (k < 36) {
	    if (k < 30 || (flg & UPOK))
		k += 12;
	} else if (k > 84) {
	    if (k > 90 || (flg & DNOK))
		k -= 12;
	}
	return(k);
}

int	Lastnotek, Lastnoted;	/* used by noteonoff & notehist() */

noteonoff(k, start, stop)
long	start, stop;
{
	u_char ubuf[4];
	MCMD m;

DBG(stderr, "  %s (%d) @%d until %d\n", key2name(k), k, start, stop);
	m.cmd = ubuf;
	m.len = 3;
	m.when = start;
	m.cmd[0] = CH_KEY_ON | Chan[0];
	m.cmd[1] = k;
	m.cmd[2] = Vol[0];
	putmcmd(stdout, &m);
	m.when = stop;
	m.cmd[2] = 0;
	putmcmd(stdout, &m);
	Lastnoted = k - Lastnotek;
	Lastnotek = k;
}

notehist(kp, dp)
int	*kp, *dp;
{
	*kp = Lastnotek;
	*dp = Lastnoted;
}

wrdcmp(ap, bp)
char	*ap, *bp;
{
	while (*bp && *ap == *bp) {
	    ap++;
	    bp++;
	}
	return ((*ap <= ' ' && *bp == '\0')? 0 : *ap - *bp);
}

syntax(msg)
char	*msg;
{
	fprintf(stderr, "Chord chart syntax error: %s\n", msg);
	exit(1);
}

find(name)
char	*name;
{
	register int i;

	for (i = 0; i < Numchords; i++)
	    if (wrdcmp(name, Ch[i].name) == 0)
		return(i);
	return(-1);
}

sys(buf)
char	*buf;
{
DBG(stderr, "%s\n", buf);
	system(buf);
}
