/*
**  wt -- a 3d game engine
**
**  Copyright (C) 1994 by Chris Laurel
**  email:  claurel@mr.net
**  snail mail:  Chris Laurel, 5700 W Lake St #208,  St. Louis Park, MN  55416
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>

#include "wt.h"
#include "error.h"
#include "wtmem.h"
#include "framebuf.h"
#include "graphics.h"


static void create_window(void);
static void load_palette(void);
static void map_shared_mem(void);

Display *display;
int screen;
Window wtwin;
static GC gc;
static XShmSegmentInfo shminfo;
static XImage *shmimage;
static unsigned char *image_mem;
static int graphics_initialized = 0;


void init_graphics(void)
{
     if (graphics_initialized == 1)
	  return;

     image_mem = wtmalloc(SCREEN_HEIGHT * SCREEN_WIDTH);
     create_window();
     load_palette();
     map_shared_mem();

     graphics_initialized = 1;
}


void create_window(void)
{
     XGCValues gc_values;

     display = XOpenDisplay(NULL);
     if (display == NULL)
	  fatal_error("Unable to open display.\n");

     screen = DefaultScreen(display);
     wtwin = XCreateSimpleWindow(display, DefaultRootWindow(display),
				 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT,
				 2, BlackPixel(display, screen),
				 BlackPixel(display, screen));
     XStoreName(display, wtwin, "xwt");
     XMapWindow(display, wtwin);
     XSelectInput(display, wtwin,
		  ButtonPressMask | ButtonReleaseMask |
		  ButtonMotionMask | KeyPressMask | ExposureMask);
     gc_values.graphics_exposures = False;
     gc = XCreateGC(display, wtwin, GCGraphicsExposures, &gc_values);
}


void load_palette(void)
{
     FILE *fp;
     int i, r, g, b;
     XColor palette[PALETTE_ENTRIES];
     Colormap wtcolormap;


     fp = fopen(DEFAULT_PALETTE_FILE, "rb");
     if (fp == NULL)
	  fatal_error("unable to open palette file");

     for (i = 0; i < PALETTE_ENTRIES; i++) {
	  r = getc(fp);
	  g = getc(fp);
	  b = getc(fp);
	  if (b == EOF)
	       fatal_error("error reading palette file");

	  palette[i].pixel = i;
	  palette[i].red = r << 10;
	  palette[i].green = g << 10;
	  palette[i].blue = b << 10;
	  palette[i].flags = DoRed | DoGreen | DoBlue;
     }
     
     fclose(fp);

     wtcolormap = XCreateColormap(display, DefaultRootWindow(display),
				  DefaultVisual(display, screen), AllocAll);
     if (!XStoreColors(display, wtcolormap, palette, PALETTE_ENTRIES))
	  fatal_error("Unable to set up private colormap");
     XSetWindowColormap(display, wtwin, wtcolormap);
}


void map_shared_mem(void)
{
     int depth;
     XWindowAttributes win_attributes;

     XGetWindowAttributes(display, DefaultRootWindow(display),
			  &win_attributes);
     depth = win_attributes.depth;
     shmimage = XShmCreateImage(display, DefaultVisual(display, screen), depth,
				ZPixmap, image_mem, &shminfo,
				SCREEN_WIDTH, SCREEN_HEIGHT);
     shminfo.shmid = shmget(IPC_PRIVATE,
			    shmimage->bytes_per_line * shmimage->height,
			    IPC_CREAT | 0777);
     if (shminfo.shmid < 0)
	  fatal_error("shmget failed");
     
     shmimage->data = (unsigned char *) shmat(shminfo.shmid, 0, 0);
     if (shmimage->data == ((char *) -1))
	  fatal_error("shmat failed");
     
     image_mem = shmimage->data;
     shminfo.shmaddr = shmimage->data;
     shminfo.readOnly = False;
     if (!XShmAttach(display, &shminfo))
	  fatal_error("Unable to map shared memory.\n");
}


void end_graphics(void)
{
     wtfree(image_mem);
     graphics_initialized = 0;
}


void update_screen(Framebuffer *fb)
{
     XShmPutImage(display, wtwin, gc, shmimage, 0, 0, 0, 0,
		  fb->fb_width, fb->fb_height, 0);
     XSync(display, 0);
}


Pixel *get_framebuffer_memory(int width, int height)
{
     return (Pixel *) image_mem;
}
