/* stringutil.c */

/* String utilities */

#include <ctype.h>
#include <string.h>

#include "copyright.h"
#include "config.h"
#include "interface.h"
#include "globals.h"

#ifdef NEVER
int strcasecmp(s1, s2)
    const char *s1;
    const char *s2;
{
  while (*s1 && *s2 && DOWNCASE(*s1) == DOWNCASE(*s2))
    s1++, s2++;

  return (DOWNCASE(*s1) - DOWNCASE(*s2));
}
#endif				/* NEVER */

int string_prefix(string, prefix)
    const char *string;
    const char *prefix;
{
  if (!string || !prefix) return 0;
  while (*string && *prefix && DOWNCASE(*string) == DOWNCASE(*prefix))
    string++, prefix++;
  return *prefix == '\0';
}

/* accepts only nonempty matches starting at the beginning of a word */
const char *string_match(src, sub)
    const char *src;
    const char *sub;
{
    if (!src || !sub)
	return 0;

    if (*sub != '\0') {
	while (*src) {
	    if (string_prefix(src, sub))
		return src;
	    /* else scan to beginning of next word */
	    while (*src && isalnum(*src))
		src++;
	    while (*src && !isalnum(*src))
		src++;
	}
    }
    return 0;
}

char *strupper(s)
    const char *s;
{
    static char buf1[BUFFER_LEN];
    char *p;

    strcpy(buf1, s);
    for(p = buf1; *p; p++)
      *p = UPCASE(*p);
    return buf1;
}

char *upcasestr(s)
     char *s;
{
  /* modifies a string in-place to be upper-case */

  char *p;
  for (p = s; p && *p; p++)
    *p = UPCASE(*p);
  return s;
}

/* safe_chr and safe_str are essentially straight out of the 2.0 code */

int safe_chr(c, buf, bufp)
     char c;
     char *buf;
     char **bufp;
{
  /* adds a character to a string, being careful not to overflow buffer */

  char *tp;
  int val;

  tp = *bufp;
  val = 0;

  if ((tp - buf) < BUFFER_LEN)
    *tp++ = c;
  else 
    val = 1;

  *bufp = tp;
  return val;
}

int safe_copy_str(c, buff, bp, maxlen)
     char *c;
     char *buff;
     char **bp;
     int maxlen;
{
  /* copies a string into a buffer, making sure there's no overflow. */

  char *tp;

  tp = *bp;
  if (c == NULL)
    return 0;
  while (*c && ((tp - buff) < maxlen))
    *tp++ = *c++;
  *bp = tp;
  return strlen(c);
}

/* skip_space and seek_char are essentially right out of the 2.0 code */

char *skip_space(s)
     const char *s;
{
  /* returns pointer to the next non-space char in s, or NULL if s == NULL
   * or *s == NULL or s has only spaces.
   */

  char *c = (char *) s;
  while (c && *c && isspace(*c))
    c++;
  return c;
}

char *seek_char(s, c)
     const char *s;
     char c;
{
  /* similar to strchr(). returns a pointer to the next char in s which
   * matches c, or a pointer to the terminating null at the end of s.
   */

  char *p = (char *) s;
  while (p && *p && (*p != c))
    p++;
  return p;
}

#ifdef NEED_STRDUP

#ifdef NEVER
char *strdup(s)
     const char *s;
{
  char *dup;
  
  dup = (char *) malloc(strlen(s) + 1);
  strcpy(dup, s);
  return dup;
}

#else

char *strdup(s)
     const char *s;
{
  int len;
  int dup = 0;

  len = strlen(s) + 1;
  if ((dup = (char *)malloc(len)) != NULL)
    bcopy(s, dup, len);
  return (dup);
}
#endif

#endif				/* NEED_STRDUP */

char *replace_string(old, new, string)
     const char *old;
     const char *new;
     const char *string;
{
  /* another 2.0 function: replaces string "old" with string "new".
   * The result returned by this must be freed.
   */

  char *result, *r, *s;
  int len;

  if (!string)
    return NULL;

  s = (char *) string;
  len = strlen(old);
  r = result = (char *) malloc(BUFFER_LEN + 1);
#ifdef MEM_CHECK
  add_check("replace_string.buff");
#endif

  while (*s) {

    /* copy up to the next occurence of first char of old */
    while (*s && *s != *old) {
      safe_chr(*s, result, &r);
      s++;
    }

    /* if we've really found  old, append new to the result and
     * move past the occurence of old. Else, copy the char and
     * continue.
     */
    if (*s) {
      if (!strncmp(old, s, len)) {
	safe_copy_str((char *) new, result, &r, BUFFER_LEN);
	s += len;
      } else {
	safe_chr(*s, result, &r);
	s++;
      }
    }
  }

  *r = '\0';
  return result;
}
