/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * This file is part of the libe-book project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#ifndef LRFTYPES_H_INCLUDED
#define LRFTYPES_H_INCLUDED

#include <deque>
#include <map>
#include <string>

#include <boost/optional.hpp>

namespace libebook
{

enum LRFImageType
{
  LRF_IMAGE_TYPE_UNKNOWN = 0,
  LRF_IMAGE_TYPE_JPEG = 0x11,
  LRF_IMAGE_TYPE_PNG = 0x12,
  LRF_IMAGE_TYPE_BMP = 0x13,
  LRF_IMAGE_TYPE_GIF = 0x14
};

enum LRFEmptyLinePosition
{
  LRF_EMPTY_LINE_POSITION_UNKNOWN,
  LRF_EMPTY_LINE_POSITION_BEFORE,
  LRF_EMPTY_LINE_POSITION_AFTER
};

enum LRFEmptyLineMode
{
  LRF_EMPTY_LINE_MODE_NONE,
  LRF_EMPTY_LINE_MODE_SOLID,
  LRF_EMPTY_LINE_MODE_DASHED,
  LRF_EMPTY_LINE_MODE_DOUBLE,
  LRF_EMPTY_LINE_MODE_DOTTED
};

enum LRFAlign
{
  LRF_ALIGN_START,
  LRF_ALIGN_END,
  LRF_ALIGN_CENTER
};

struct LRFBookInfo
{
  LRFBookInfo();

  std::string author;
  std::string title;
  std::string bookID;
  std::string publisher;
  boost::optional<std::string> label;
  std::deque<std::string> categories;
  std::string classification;
  boost::optional<std::string> freeText;
};

struct LRFDocInfo
{
  LRFDocInfo();

  std::string language;
  std::string creator;
  std::string creationDate;
  std::string producer;
  unsigned page;
};

struct LRFMetadata
{
  LRFMetadata();

  LRFBookInfo bookInfo;
  LRFDocInfo docInfo;
  boost::optional<std::string> keyword;
};

struct LRFColor
{
  LRFColor();
  explicit LRFColor(unsigned color);
  LRFColor(unsigned char red, unsigned char green, unsigned char blue, unsigned char alpha = 0);

  unsigned char r;
  unsigned char g;
  unsigned char b;
  unsigned char a;
};

struct LRFEmptyLine
{
  LRFEmptyLine();

  LRFEmptyLinePosition m_position;
  LRFEmptyLineMode m_mode;
};

struct LRFAttributes
{
  LRFAttributes();

  boost::optional<unsigned> fontSize;
  boost::optional<unsigned> fontWidth;
  boost::optional<unsigned> fontEscapement;
  boost::optional<unsigned> fontOrientation;
  boost::optional<unsigned> fontWeight;
  boost::optional<std::string> fontFacename;
  boost::optional<LRFColor> textColor;
  boost::optional<LRFColor> textBgColor;
  boost::optional<unsigned> wordSpace;
  boost::optional<unsigned> letterSpace;
  boost::optional<unsigned> baseLineSkip;
  boost::optional<unsigned> lineSpace;
  boost::optional<unsigned> parIndent;
  boost::optional<unsigned> parSkip;
  boost::optional<unsigned> height;
  boost::optional<unsigned> width;
  boost::optional<unsigned> locationX;
  boost::optional<unsigned> locationY;
  bool italic;
  bool sup;
  bool sub;
  bool heading;
  boost::optional<LRFEmptyLine> emptyLine;
  boost::optional<LRFAlign> align;
  boost::optional<unsigned> topSkip;
  boost::optional<unsigned> topMargin;
  boost::optional<unsigned> oddSideMargin;
  boost::optional<unsigned> evenSideMargin;
};

typedef std::map<unsigned, LRFAttributes> LRFAttributeMap_t;

}

#endif // LRFTYPES_H_INCLUDED

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
